 {
 ***********************************************************************
 Daq Pascal application program FdbSrv.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Awake         - Awake to save data immediately (ahead of schedule).
| @AssertMode m  - Mask m=1/2/4=Use(Trouble)/Use(tagSaveErr)/Verbose.
| @TimeQuantum t - Time quantum t, ms. It`s datapoints saving period.
| @FastTransfer f - flag f=0/1 for fast data transfer (addnew/update).
| @BigTimeout t  - Watchdog big timeout uses for long operations, ms.
|********************************************************
[]
 ***************************************************************************
 Универсальная программа сохранения данных в суточные файлы *.fdb.
 Выполняет периодическое сохранение списка кривых в файл СУБД Firebird.
 Предполагается, что сохранение идет достаточно редко, скажем, раз в минуту.
 При сохранении данные записываются все, без потерь, то есть записываются
 исторические данные из списка кривых, а не только текущее значение.
 При этом предполагается, что кривые содержат данные (x,y), где x-время
 по часам DAQ, функция time (поэтому данные должны быть упорядочены по x).
 Утилиту НЕ СЛЕДУЕТ применять для данных других типов (когда x - не время).
 За счет округления и упаковки объем сохраняемых данных сильно сокращается.
 При сохранении контролируется время CPU, чтобы не блокировать ресурсы на
 длительное время. Этим управляет параметр TimeQuota. Несохраненные данные
 будут сохранены после пробуждения (@Awake) в следующем кванте времени.
 ***************************************************************************
 Список кривых для сохранения задается в виде выражений
      CurveList = Curve1, Curve2, ...
      CurveList = Section1, Section2, ...
 где CurveN-имя кривой, SectionN - имя секции, содержащей список кривых.
 Имена секций анализируются рекурсивно, включаются все кривые из секции.
 Выражение CurveList = ... повторяется столько раз, сколько потребуется.
 ***************************************************************************
 Параметр WriteEnableTag задает имя тега который разрешает запись файла.
 Параметр WriteErrorsTag задает имя тега который считает ошибки записи.
 Параметр DataPath       задает каталог сохранения данных.
 Параметр SavePeriod     задает период сохранения в секундах.
 Параметр FilePrefix     задает префикс имени файла.
 ***************************************************************************
 Данные сохраняются в суточный файл типа PPPPYYYYMMDD.FDB
 где  PPPP       - префикс имени, заданный параметром FilePrefix
      YYYY,MM,DD - год, месяц и день, например: QMS_20050819.FDB
 ***************************************************************************
 Пример конфигурации:
      [TagList]
      FDBSRV.GATE     = integer 1
      FDBSRV.BUGS     = real    0
      [DeviceList]
      &FdbSrv = device software program
      [&FdbSrv]
      Comment         = DAT server to save data to *.dat files
      InquiryPeriod   = 1
      DevicePolling   = 1000, tpNormal
      ProgramSource   = ~~\Resource\DaqSite\FdbServer\FdbSrv.pas
      OpenConsole     = 1
      DebugFlags      = 15
      StdInFifo       = 32
      StdOutFifo      = 128
      SavePeriod      = 300
      FilePrefix      = CRW-DAQ_
      DataPath        = ..\..\CRW-DAQ_DATA
      WriteEnableTag  = FDBSRV.GATE
      WriteErrorsTag  = FDBSRV.BUGS
      TimeQuota       = 2000
      []
      [&FdbSrv]
      CurveList       = QMS_CH0, QMS_CH1
      CurveList       = [QMS_Curves]     
      []
 ***************************************************************************
 }
{
********************************************************
[Compiler.Options]
Compiler.dtabmax = 1024*16 ; Data segment
Compiler.stabmax = 1024*8  ; String table
Compiler.dtabmin = 1024*1  ; Min stack  space
Compiler.stabmin = 1024*1  ; Min string space
[]
********************************************************
}
program FdbSrv;                  { DAT Server                       }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 {$I _con_DbLibrary}             { DbLibrary constants              }
 {------------------------------------------------------------------}
 BufferSize    = 256;            { DataPoint buffer size, max=256   }
 TimeQuantMin  = 1;              { Minimal TimeQuantum, ms          }
 TimeQuantDef  = 1000;           { Default TimeQuantum, ms          } 
 TimeQuantMax  = 60000;          { Maximal TimeQuantum, ms          }
 TimeGapFactor = 0.999999;       { To calc TimeGap by TimeQuantum   }  
 idFileSign    = '[CRW-DAQ DATA FILE]'; { DAT file signature        }
 tab_Curves    = '(ID integer generated by default as identity primary key, X_TIME timestamp not null, Y_VALUE double precision not null)';
 tab_Params    = '(ID integer generated by default as identity primary key, PAR_NAME varchar(40) not null, PAR_VALUE varchar(255) not null)';
 WdtBigTimeout = 30000;          { Watchdog big timeout for long op }
 
type
 {------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}            { Include all Standard types,      }
 {------------------------------}{ And add User defined types:      }
 TDataXY = record x,y:Real; end; { Data point (x,y) of curve        }
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 {$I _var_DbLibrary}             { DbLibrary variables              }
 {------------------------------------------------------------------}
 FdbSrvRec     : record          { FdbSrv internal data             }
  CurveList    : Integer;        { Curve list to save               }
  DataPath     : String;         { Directory where database located }
  FilePref     : String;         { File prefix for database files   }
  TimeQuota    : Integer;        { Max. time slice duration         }
  SavePeriod   : Real;           { Data save period, sec            }
  Account_uid  : String;         { Account User Id                  }
  Account_pwd  : String;         { Account Password                 }
  Account_mode : Integer;        { Account encoding mode            }
  tagSave      : Integer;        { Integer Button to save data      }
  tagSaveErr   : Integer;        { Real    Error counter            }
  DataCurve    : Integer;        { Current curve for data transfer  }
  DataCount    : Integer;        { Count data points to transfer    }
  DataPoint    : array[1..BufferSize] of TDataXY; { Buffer for data }
  Incompleted  : Integer;        { Counter of incompleted transfers }
  AwakeFlag    : Boolean;        { Flag to awake: enforce data save }
  LastSave     : Real;           { Time of last save                }
  ErrorCount   : Integer;        { For internal use                 }
  FirstStart   : Boolean;        { For internal use                 }
  AssertMode   : Integer;        { FdbAssert mode                   }
  BigTimeout   : Integer;        { BigTimeout, ms                   }
  TimeQuantum  : Real;           { Time quantum, i.e.resolution, ms }
  DaqTimeGap   : Real;           { Time scale gap in DAQ Time units }
  FastTransfer : Boolean;        { Use recordset addnew/update      }
  cmd_Awake        : Integer;    { @Awake                           }
  cmd_AssertMode   : Integer;    { @AssertMode                      }
  cmd_BigTimeout   : Integer;    { @BigTimeout                      }
  cmd_TimeQuantum  : Integer;    { @TimeQuantum                     }
  cmd_FastTransfer : Integer;    { @FastTransfer                    }
 end;                            {                                  }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_DbLibrary}             { DbLibrary functions              }
 {------------------------------------------------------------------}

 {
 Format time as Firebird TimeStamp like 2008-08-13 14:20:23.1200.
 Timestamp assignment has precision 1 millisecond.
 }
 function GetTimeAsFirebirdTimestamp(ms:Real):String;
 begin
  GetTimeAsFirebirdTimestamp:=StrTimeFmt('yyyy-mm-dd hh:nn:ss.zzz0',ms);
 end;
 {
 Validate table name: uppercase and "quotes" if needed and mode=1.
 }
 function ValidateTableName(tab:String; mode:Integer):String;
 begin
  tab:=Trim(tab);
  if (tab<>'') then begin
   tab:=UpcaseStr(tab);
   if HasFlags(mode,1) then
   if not IsLexeme(tab,lex_SqlName)
   then tab:=AnsiQuotedStr(tab,QuoteMark);
  end;
  ValidateTableName:=tab;
 end;
 {
 FdbSrv Assertion.
 }
 procedure FdbAssert(cond:Boolean; msg:string);
 begin
  if cond then begin
   if (msg<>'') then if HasFlags(FdbSrvRec.AssertMode,4) then Success(msg) else Details(msg);
  end else begin
   FdbSrvRec.ErrorCount:=(FdbSrvRec.ErrorCount+1);
   if (msg<>'') then if HasFlags(FdbSrvRec.AssertMode,1) then Trouble(msg) else Problem(msg);
   if (FdbSrvRec.tagSaveErr<>0) and HasFlags(FdbSrvRec.AssertMode,2) then bNul(rAtomicTagOp(FdbSrvRec.tagSaveErr,'+',1));
  end;
 end;
 {
 Make directory (dir) accessible for all users.
 }
 procedure MakeDirAccessible(dir:String);
 var pm,cmd,ans:String; ec:Integer;
  procedure Cleanup;
  begin
   pm:=''; cmd:=''; ans:='';
  end;
 begin
  Cleanup;
  if IsNonEmptyStr(dir) then
  if DirExists(dir) then begin
   if IsUnix then begin
    pm:=Trim(CookieScan(GetFileProperties(dir,'Permissions'),'Permissions',0));
    if (pm<>'drwxrwxrwx') then begin
     Success('Access Permissions '+pm+' of Directory '+dir);
     cmd:='chmod -c ugo+rwx '+dir;
     ec:=RunSysCommand(cmd,'','',ans,5000);
     if (ec=0) then begin
      Success(cmd);
      Success(Trim(ans));
     end else begin
      Problem(cmd);
     end;
     pm:=Trim(CookieScan(GetFileProperties(dir,'Permissions'),'Permissions',0));
    end;
    Success('Access Permissions '+pm+' of Directory '+dir);
   end;
  end;
  Cleanup;
 end;
 {
 FdbSrv Calculate DB file name corresponded to time (ms).
 }
 function FdbSrv_CalcFileName(ms:Real):String;
 var num:Integer;
 begin
  num:=GetDateAsNumber(ms);
  FdbSrv_CalcFileName:=AddPathDelim(FdbSrvRec.DataPath)+FdbSrvRec.FilePref+Str(num)+'.fdb';
 end;
 {
 FdbSrv Return CurveList Count.
 }
 function FdbSrv_CurveListCount:Integer;
 begin
  FdbSrv_CurveListCount:=hashlist_count(FdbSrvRec.CurveList);
 end;
 {
 FdbSrv Add curve (crv) to CurveList.
 }
 procedure FdbSrv_CurveListAdd(crv:Integer);
 var done:Boolean;
 begin
  if (crv<>0) then
  if IsSameText(RefInfo(crv,'Type'),'Curve') then begin
   done:=hashlist_setlink(FdbSrvRec.CurveList,CrvName(crv),crv);
   Assertion(done,'CurveList = '+CrvName(crv));
  end else Trouble('Bad Curve Ref: '+Str(crv));
 end;
 {
 FdbSrv Return CurveList Item by index (i=1..Count).
 }
 function FdbSrv_CurveListItem(i:Integer):Integer;
 var curve:Integer;
 begin
  curve:=0;
  if (i>=1) and (i<=hashlist_count(FdbSrvRec.CurveList))
  then curve:=hashlist_getlink(FdbSrvRec.CurveList,hashlist_getkey(FdbSrvRec.CurveList,i-1));
  FdbSrv_CurveListItem:=curve;
 end;
 {
 FdbSrv Get TimeStamp of curve (crv) from CurveList.
 }
 function FdbSrv_GetCurveTimeStamp(crv:Integer):Real;
 var stamp:Real;
 begin
  if (crv=0)
  then stamp:=0
  else stamp:=hashlist_getdata(FdbSrvRec.CurveList,CrvName(crv));
  FdbSrv_GetCurveTimeStamp:=stamp;
 end;
 {
 FdbSrv Set TimeStamp (stamp) of curve (crv) from CurveList.
 }
 procedure FdbSrv_SetCurveTimeStamp(crv:Integer; stamp:Real);
 begin
  if (crv<>0) then
  if (hashlist_indexof(FdbSrvRec.CurveList,CrvName(crv))>=0)
  then bNul(hashlist_setdata(FdbSrvRec.CurveList,CrvName(crv),stamp));
 end;
 {
 FdbSrv Get Account (uid,pwd,mode) from data record.
 }
 procedure FdbSrv_GetAccount(var uid,pwd:String; var mode:Integer);
 begin
  uid:=FdbSrvRec.Account_uid;
  pwd:=FdbSrvRec.Account_pwd;
  mode:=FdbSrvRec.Account_mode;
 end;
 {
 FdbSrv Read CurveList=C1,C2,...,[Sec1],[Sec2]
 Use recursion to read sections. 
 }
 procedure FdbSrv_ReadCurveList(Section:String);
 var line,item:String; i,j,crv,txt,mode:Integer;
  procedure Cleanup;
  begin
   line:=''; item:='';
  end;
 begin
  Cleanup;
  mode:=risModeTrimL+risModeTrimR+risModeRemComm;
  txt:=ReadIniSection(text_new,mode,ParamStr('DaqConfigFile'),Section);
  for i:=0 to text_numln(txt)-1 do begin
   line:=Trim(text_getln(txt,i));
   if IsSameText(ExtractWord(1,line),'CurveList') then
   for j:=2 to WordCount(line) do begin
    item:=ExtractWord(j,line);
    crv:=RefFind('Curve '+item);
    if (crv<>0) then begin
     FdbSrv_CurveListAdd(crv);
    end else begin
     if IsSectionName(item)
     then FdbSrv_ReadCurveList(item)
     else Trouble('Could not find curve '+item);
    end;
   end;
  end;
  FreeAndZero(txt);
  Cleanup;
 end;
 {
 FdbSrv Create Database corresponded to given time (when).
 }
 function FdbSrv_CheckCreateDatabase(when:Real):Boolean;
 var fdb,dir,uid,pwd,opt,constr,sqlcmd,table:String;
     okfdb,okdir,okcon,okcom,okrbk:Boolean;
     con,trn,crv,rst,mode,ic,wdttmt:Integer;
  procedure Cleanup;
  begin
   fdb:=''; dir:=''; uid:=''; pwd:=''; opt:=''; constr:=''; sqlcmd:=''; table:='';
   con:=0; trn:=0; crv:=0; mode:=0; okfdb:=false; wdttmt:=0;
  end;
  procedure CreateTable(table,decl:String);
  begin
   sqlcmd:='create table '+table+' '+Trim(decl);
   rst:=db_execute(con,sqlcmd,adCmdText);
   FdbAssert(rst<>0,'Execute: '+sqlcmd);
   FreeAndZero(rst);
  end;
  procedure InsertParNameValue(table,pn,pv:String);
  begin
   pn:=AnsiQuotedStr(pn,Apostrophe); pv:=AnsiQuotedStr(pv,Apostrophe);
   sqlcmd:='insert into '+table+' (PAR_NAME,PAR_VALUE) values ('+pn+','+pv+')';
   rst:=db_execute(con,sqlcmd,adCmdText);
   FdbAssert(rst<>0,'Execute: '+sqlcmd);
   FreeAndZero(rst);
  end;
 begin
  Cleanup;
  if not IsNan(when) then
  if not IsInf(when) then
  if (when>0) then begin
   if (wdt_reset(true)>=0) then wdttmt:=wdt_timeout(FdbSrvRec.BigTimeout);
   fdb:=FdbSrv_CalcFileName(when);
   okfdb:=FileExists(fdb);
   if not okfdb then begin
    dir:=ExtractFilePath(fdb);
    okdir:=DirExists(dir);
    if not okdir then okdir:=MkDir(dir);
    FdbAssert(okdir,'Call MkDir: '+dir);
    if okdir then begin
     MakeDirAccessible(dir);
     FdbSrv_GetAccount(uid,pwd,mode);
     opt:='PAGE_SIZE = 4096 DEFAULT CHARACTER SET UTF8';
     okfdb:=db_create_local_fdb(fdb,uid,pwd,opt,constr,mode);
     FdbAssert(okfdb,'Create Database: '+fdb);
     if okfdb then okfdb:=FileExists(fdb);
     if okfdb then begin
      constr:=db_build_connectionstring(db_sym_Firebird,db_sym_localhost,fdb,uid,pwd,'',mode);
      constr:=db_validate_known_providers(constr);
      //
      // Create tables
      //
      con:=db_connection(db_engine_uses,constr);
      db_bugreport_mode(con,db_brm_uses);
      okcon:=db_open(con,adConnectUnspecified); 
      FdbAssert(okcon,'Open Connection: '+fdb);
      if okcon then begin
       Success('Create tables...');
       trn:=db_begintrans(con);
       FdbAssert(trn>0,'Transaction: '+Str(trn));
       if (trn>0) then begin
        table:=ValidateTableName(idFileSign,1);
        CreateTable(table,tab_Params);
        for ic:=1 to FdbSrv_CurveListCount do begin
         crv:=FdbSrv_CurveListItem(ic);
         table:=ValidateTableName(CrvName(crv),1);
         CreateTable(table,tab_Curves);
        end;
        okcom:=db_committrans(con);
        FdbAssert(okcom,'Commit: '+fdb);
        if not okcom then begin
         okrbk:=db_rollbacktrans(con);
         FdbAssert(okrbk,'Rollback: '+fdb);
        end;
       end;
      end; // okcon
      FreeAndZero(con);
      //
      // Fill tables
      //
      con:=db_connection(db_engine_uses,constr);
      db_bugreport_mode(con,db_brm_uses);
      okcon:=db_open(con,adConnectUnspecified); 
      FdbAssert(okcon,'Open Connection: '+fdb);
      if okcon then begin
       Success('Fill tables...');
       trn:=db_begintrans(con);
       FdbAssert(trn>0,'Transaction: '+Str(trn));
       if (trn>0) then begin
        table:=ValidateTableName(idFileSign,1);
        InsertParNameValue(table,'DateTime',GetDateTime(when));
        InsertParNameValue(table,'TimeStamp',Str(when));
        InsertParNameValue(table,'OriginalHostName',ParamStr('HostName'));
        InsertParNameValue(table,'OriginalFileName',fdb);
        InsertParNameValue(table,'OriginalConfigFile',ParamStr('DaqConfigFile'));
        for ic:=1 to FdbSrv_CurveListCount do begin
         crv:=FdbSrv_CurveListItem(ic);
         InsertParNameValue(table,'CurveList',CrvName(crv));
        end;
        okcom:=db_committrans(con);
        FdbAssert(okcom,'Commit: '+fdb);
        if not okcom then begin
         okrbk:=db_rollbacktrans(con);
         FdbAssert(okrbk,'Rollback: '+fdb);
        end;
       end;
      end; // okcon
      FreeAndZero(con);
     end; // okfdb
    end; // okdir
   end; // okfdb
   if (wdt_reset(true)>=0) then bNul(wdt_timeout(wdttmt)>0);
  end; // when
  FdbSrv_CheckCreateDatabase:=okfdb;
  FreeAndZero(con);
  Cleanup;
 end;
 {
 FdbSrv init Database corresponded to given time (when).
 Create missed tables, readout timestamp of curves.
 }
 function FdbSrv_CheckInitDatabase(when:Real):Boolean;
 var fdb,dir,uid,pwd,opt,constr,sqlcmd,table,id,sv:String;
     con,tabs,trn,crv,rst,mode,ic,tp,nr:Integer; tm,ms:Real;
     okfdb,okdir,okcon,okcom,okrbk:Boolean;    
  procedure Cleanup;
  begin
   fdb:=''; dir:=''; uid:=''; pwd:=''; opt:=''; constr:=''; sqlcmd:=''; table:=''; id:=''; sv:='';
   con:=0; tabs:=0; okfdb:=false;
  end;
  procedure CreateTable(table,decl:String);
  begin
   sqlcmd:='create table '+table+' '+Trim(decl);
   rst:=db_execute(con,sqlcmd,adCmdText);
   FdbAssert(rst<>0,'Execute: '+sqlcmd);
   FreeAndZero(rst);
  end;
  procedure InsertParNameValue(table,pn,pv:String);
  begin
   pn:=AnsiQuotedStr(pn,Apostrophe); pv:=AnsiQuotedStr(pv,Apostrophe);
   sqlcmd:='insert into '+table+' (PAR_NAME,PAR_VALUE) values ('+pn+','+pv+')';
   rst:=db_execute(con,sqlcmd,adCmdText);
   FdbAssert(rst<>0,'Execute '+sqlcmd);
   FreeAndZero(rst);
  end;
 begin
  Cleanup;
  if (when>0) then begin
   fdb:=FdbSrv_CalcFileName(when);
   dir:=ExtractFilePath(fdb);
   okfdb:=FileExists(fdb);
   if okfdb then begin
    MakeDirAccessible(dir);
    FdbSrv_GetAccount(uid,pwd,mode);
    constr:=db_build_connectionstring(db_sym_Firebird,db_sym_localhost,fdb,uid,pwd,'',mode);
    constr:=db_validate_known_providers(constr);
    //
    // Create tables which is not exist yet.
    // Get last timestamp from existing tables.
    //
    con:=db_connection(db_engine_uses,constr);
    db_bugreport_mode(con,db_brm_uses);
    okcon:=db_open(con,adConnectUnspecified); 
    FdbAssert(okcon,'Open Connection: '+fdb);
    if okcon then begin
     //
     // Read list of tables...
     //
     Success('Read Tables...');
     tabs:=Text_new;
     sqlcmd:=db_build_selectalltables(db_sym_Firebird,'');
     FdbAssert(sqlcmd<>'','ShowTables: '+sqlcmd);
     if (sqlcmd<>'') then begin
      trn:=db_begintrans(con);
      FdbAssert(trn>0,'Transaction: '+Str(trn));
      if (trn>0) then begin
       rst:=db_execute(con,sqlcmd,adCmdText);
       FdbAssert(rst<>0,'Execute: '+sqlcmd);
       if (rst<>0) then begin
        if (db_fieldscount(rst)=1) then
        while not db_eof(rst) do begin
         id:=db_fieldsNames(rst,0);
         tp:=db_fieldsTypes(rst,id);
         if (db_FieldTypeToTagType(tp)=tag_type_string) then begin
          sv:=UpcaseStr(Trim(db_fieldsAsString(rst,id,'r','')));
          if (sv<>'') then bNul(text_addln(tabs,sv));
         end;
         bNul(db_movenext(rst));
        end;
       end;
       okcom:=db_committrans(con);
       FdbAssert(okcom,'Commit: '+fdb);
       if not okcom then begin
        okrbk:=db_rollbacktrans(con);
        FdbAssert(okrbk,'Rollback: '+fdb);
       end;
      end;
     end;
     FdbAssert(text_numln(tabs)>0,StrFmt('%d table(s) found',text_numln(tabs)));
     //
     // Check list of tables...
     //
     Success('Check tables...');
     trn:=db_begintrans(con);
     FdbAssert(trn>0,'Transaction: '+Str(trn));
     if (trn>0) then begin
      for ic:=1 to FdbSrv_CurveListCount do begin
       crv:=FdbSrv_CurveListItem(ic);
       if (Text_IndexOf(tabs,CrvName(crv))<0) then begin
        table:=ValidateTableName(idFileSign,1);
        InsertParNameValue(table,'CurveList',CrvName(crv));
        table:=ValidateTableName(CrvName(crv),1);
        CreateTable(table,tab_Curves);
        FdbSrv_SetCurveTimeStamp(crv,0);
       end else begin
        table:=ValidateTableName(CrvName(crv),1);
        sqlcmd:='select first 5 * from '+table+' order by X_TIME desc';
        rst:=db_execute(con,sqlcmd,adCmdText);
        FdbAssert(rst<>0,'Execute: '+sqlcmd);
        ms:=0; nr:=0;
        if db_fieldsCount(rst)>0 then
        while not db_eof(rst) do begin
         tm:=db_fieldsAsFloat(rst,'X_TIME','r',0); nr:=nr+1;
         if (tm>0) then ms:=max(ms,OleTimeToMs(tm));
         bNul(db_movenext(rst));
        end;
        if (nr>0) and (ms>0) then FdbSrv_SetCurveTimeStamp(crv,MsToDaqTime(ms));
        FdbAssert((ms>0) or (nr=0),'TimeStart['+CrvName(crv)+']='+GetDateTime(ms));
       end;
      end;
      okcom:=db_committrans(con);
      FdbAssert(okcom,'Commit: '+fdb);
      if not okcom then begin
       okrbk:=db_rollbacktrans(con);
       FdbAssert(okrbk,'Rollback: '+fdb);
      end;
     end;
     FreeAndZero(tabs);
    end; // okcon
    FreeAndZero(con);
   end; // okfdb
  end; // when
  FdbSrv_CheckInitDatabase:=okfdb;
  FreeAndZero(tabs);
  FreeAndZero(con);
  Cleanup;
 end;
 {
 FdbSrv Extract curve (crv) data to buffer after timestamp (stamp+gap). 
 If data extraction is not complete, increment Incomplete transfer counter.
 Return number of points extracted. Timestamp (stamp) is set in DaqTime units.
 }
 function FdbSrv_CurveToBuffer(crv:Integer; stamp,gap:Real; var Incomplete:Integer):Integer;
 var num,ind,len:Integer; px,py,edge:Real; over:Boolean;
 begin
  num:=0;
  over:=false;
  if (crv<>0) then
  if crvlock(crv) then begin
   len:=Round(crvlen(crv));
   if (len>0) then begin
    if (crvx(crv,len)>stamp+gap) then begin
     ind:=Round(crvwhere(crv,stamp)); edge:=stamp;
     while (ind<=len) and not over do begin
      px:=crvx(crv,ind);
      if (px>edge+gap) then begin
       if (num<BufferSize) then begin
        num:=num+1; py:=crvy(crv,ind);
        FdbSrvRec.DataPoint[num].x:=px;
        FdbSrvRec.DataPoint[num].y:=py;
        edge:=px;
       end else over:=true;
      end;
      ind:=ind+1;
     end;
    end;
   end;
   bNul(crvunlock(crv));
  end;
  if over then Incomplete:=Incomplete+1;
  FdbSrvRec.DataCurve:=crv;
  FdbSrvRec.DataCount:=num;
  FdbSrv_CurveToBuffer:=num;
 end;
 {
 FdbSrv Uses FastTransfer via recordset addnew/update only if FastTransfer flag is ON
 and TimeQuantum is greater or equals to second (1000 ms). The reason is because ADO`s
 recordset rounds timestamp to second precision (cut milliseconds). To avoid time data
 loss we should use direct SQL insert instructions to ftansfer data with milliseconds.
 }
 function FdbSrv_UsesFastTransfer:Boolean;
 begin
  if (db_engine_uses=db_engine_sqldb) or (db_engine_uses=db_engine_zeos)
  then FdbSrv_UsesFastTransfer:=FdbSrvRec.FastTransfer
  else
  if (FdbSrvRec.TimeQuantum<1000)
  then FdbSrv_UsesFastTransfer:=false
  else FdbSrv_UsesFastTransfer:=FdbSrvRec.FastTransfer;
 end;
 {
 FdbSrv save curve (crv) to database connection(con).
 Uses next SQL query (sample): (see https://firebirdfaq.org/faq336/)
  EXECUTE BLOCK AS BEGIN
  insert into "DEMO.DIM_PING.CPU_LOAD" (X_TIME,Y_VALUE) values ('2022-12-25 23:56:50.1230','1.5');
  insert into "DEMO.DIM_PING.CPU_LOAD" (X_TIME,Y_VALUE) values ('2022-12-25 23:56:51.4560','1.2');
  ...
  END
 As alternative, use recordset (with fast addnew/update calls) if FastTransfer flag is ON.
 FastTransfer can be used only if TimeQuantum>=1000 because timestamp rounding to seconds.
 }
 procedure FdbSrv_SaveCurveToDatabase(con,crv:Integer);
 var table,sqlcmd,sv,sl,sx,sm,sy,sr,s:String; sqlbuf,rst,ern,np,sp,ip,tn,bc:Integer;
     stamp,mks,px,py,pt,ms:Real; ct,ru,rb:Boolean; cc:Char;
  procedure Cleanup;
  begin
   table:=''; sqlcmd:=''; sv:=''; sl:=''; sx:=''; sm:=''; sy:=''; sr:=''; s:='';
   sqlbuf:=0; rst:=0; ern:=0; np:=0; sp:=0; ip:=0; tn:=0; bc:=0;
   ct:=false; ru:=false; rb:=false;
  end;
 begin
  Cleanup;
  ern:=FdbSrvRec.ErrorCount;
  if (con<>0) and (crv<>0) then
  if IsSameText(RefInfo(con,'Type'),'DB') then
  if IsSameText(RefInfo(crv,'Type'),'Curve') then begin
   if db_active(con) then begin
    Success('Save Curve: '+CrvName(crv));
    stamp:=FdbSrv_GetCurveTimeStamp(crv);
    mks:=mksecnow;
    np:=FdbSrv_CurveToBuffer(crv,stamp,FdbSrvRec.DaqTimeGap,FdbSrvRec.Incompleted);
    mks:=mksecnow-mks;
    if (np>0) then begin
     tn:=db_begintrans(con);
     FdbAssert(tn>0,'Transaction: '+CrvName(crv));
     if (tn>0) then begin 
      table:=ValidateTableName(CrvName(crv),1);
      if FdbSrv_UsesFastTransfer then begin
       sqlcmd:='select first 1 * from '+table;
       rst:=db_recordset(con,sqlcmd);
       if (rst<>0) then begin
        sNul(db_ctrl(rst,StrFmt('CursorType=%d',adOpenStatic)));
        sNul(db_ctrl(rst,StrFmt('LockType=%d',adLockOptimistic)));
        if db_open(rst,adConnectUnspecified) then begin
         for ip:=1 to np do begin
          px:=FdbSrvRec.DataPoint[ip].x;
          py:=FdbSrvRec.DataPoint[ip].y;
          if not IsNan(px) and not IsNan(py) then
          if not IsInf(px) and not IsInf(py) then
          if (px>stamp) then begin
           stamp:=px;
           ms:=DaqTimeToMs(px);
           pt:=MsToOleTime(ms);
           if db_addnew(rst,'') then begin
            rNul(db_fieldsAsFloat(rst,'X_TIME','w',pt));
            rNul(db_fieldsAsFloat(rst,'Y_VALUE','w',py));
            sp:=sp+1;
           end else FdbAssert(false,'');
          end;
         end;
         ru:=db_update(rst);
         FdbAssert(ru,'Update: '+CrvName(crv));
         if not ru then begin sp:=0; stamp:=FdbSrv_GetCurveTimeStamp(crv); end; // Data update failed
        end else FdbAssert(false,'Could not open Recordset '+CrvName(crv));
       end else FdbAssert(false,'Could not create Recordset '+CrvName(crv));
      end else begin
       sqlbuf:=text_new;
       bNul(text_addln(sqlbuf,'EXECUTE BLOCK AS BEGIN'));
       sl:='insert into '+table+' (X_TIME,Y_VALUE) values ('''; sm:=''','''; sr:=''');';
       for ip:=1 to np do begin
        px:=FdbSrvRec.DataPoint[ip].x;
        py:=FdbSrvRec.DataPoint[ip].y;
        if not IsNan(px) and not IsNan(py) then
        if not IsInf(px) and not IsInf(py) then
        if (px>stamp) then begin
         stamp:=px;
         ms:=DaqTimeToMs(px);
         pt:=MsToOleTime(ms);
         sx:=StrTimeFmt('yyyy-mm-dd hh:nn:ss.zzz0',ms);
         sy:=StrFmt('%g',py); sv:=sl+((sx+sm)+(sy+sr));
         bNul(text_addln(sqlbuf,sv));
         sp:=sp+1;
        end;
       end;
       bNul(text_addln(sqlbuf,'END'));
       sqlcmd:=text_tostring(sqlbuf);
       rst:=db_execute(con,sqlcmd,adCmdText); ru:=(rst<>0);
       FdbAssert(rst<>0,StrFmt('Execute: BLOCK[%d]',Length(sqlcmd)));
       if not ru then FdbSrvRec.ErrorCount:=FdbSrvRec.ErrorCount+sp; // Failed points count
       if not ru then begin sp:=0; stamp:=FdbSrv_GetCurveTimeStamp(crv); end; // Data update failed
       FreeAndZero(sqlbuf);
      end;
      if (sp>0) and ru then begin
       ct:=db_committrans(con);
       FdbAssert(ct,'Commit: '+CrvName(crv)); 
      end else ct:=false;   
      if not ct then begin
       rb:=db_rollbacktrans(con);
       FdbAssert(rb,'Rollback: '+CrvName(crv));
      end;
      FreeAndZero(rst);
     end;
     bc:=Val(db_ctrl(con,'BugsClear'));
     FdbAssert(bc=0,'Bug(s) count: '+Str(bc));
     ern:=FdbSrvRec.ErrorCount-ern;
     FdbSrv_SetCurveTimeStamp(crv,stamp);
     if DebugFlagEnabled(dfViewExp) then begin
      s:=CrvName(crv);
      s:=s+StrFmt(': saved %d point(s)',sp);
      s:=s+StrFmt(', %d error(s) found',ern);
      s:=s+StrFmt(', dead time %4.0f mks',mks);
      ViewExp(s);
     end;
    end;
   end else FdbAssert(false,'Connection is not opened.');
  end else FdbAssert(false,'Invalid Curve reference.');
  FreeAndZero(sqlbuf);
  Cleanup;
 end;
 {
 FdbSrv Save all curves to database calculated by time (When).
 }
 procedure FdbSrv_SaveAllCurvesToDatabase(When:Real);
 var fdb,dir,uid,pwd,opt,constr,sqlcmd,table:String;
     con,crv,ic,mode,ern:Integer; okcon:Boolean;
  procedure Cleanup;
  begin
   fdb:=''; dir:=''; uid:=''; pwd:=''; opt:=''; constr:=''; sqlcmd:=''; table:='';
   con:=0; crv:=0;
  end;
 begin
  Cleanup;
  ern:=FdbSrvRec.ErrorCount;
  if (iGetTag(FdbSrvRec.tagSave)<>0) then
  if (FdbSrv_CurveListCount>0) then begin
   FdbSrvRec.Incompleted:=0;
   FdbSrvRec.AwakeFlag:=false;
   if FdbSrv_CheckCreateDatabase(When) then begin
    fdb:=FdbSrv_CalcFileName(When);
    dir:=ExtractFilePath(fdb);
    MakeDirAccessible(dir);
    FdbSrv_GetAccount(uid,pwd,mode);
    constr:=db_build_connectionstring(db_sym_Firebird,db_sym_localhost,fdb,uid,pwd,opt,mode);
    constr:=db_validate_known_providers(constr);
    con:=db_connection(db_engine_uses,constr);
    db_bugreport_mode(con,db_brm_uses);
    okcon:=db_open(con,adConnectUnspecified);
    FdbAssert(okcon,'Open Database: '+fdb);
    if okcon then begin
     for ic:=1 to FdbSrv_CurveListCount do begin
      crv:=FdbSrv_CurveListItem(ic);
      if (msecnow-When<FdbSrvRec.TimeQuota)
      then FdbSrv_SaveCurveToDatabase(con,crv)
      else FdbSrvRec.Incompleted:=FdbSrvRec.Incompleted+1;
      rNul(wdt_reset(true));
     end;
    end;
   end;
   if (FdbSrvRec.Incompleted>0) then DevPostCmdLocal('@Awake');
  end;
  FreeAndZero(con);
  Cleanup;
 end;
 {
 FdbSrv Assign TimeQuantum value and update DaqTimeGap.
 }
 procedure FdbSrv_AssignTimeQuantum(TimeQuantum:Real);
 begin
  FdbSrvRec.TimeQuantum:=max(TimeQuantMin,min(TimeQuantMax,TimeQuantum));
  FdbSrvRec.DaqTimeGap:=(Max(0,MsToDaqTime(FdbSrvRec.TimeQuantum)-MsToDaqTime(0)))*TimeGapFactor;
 end;
 {
 FdbSrv Cleaning.
 }
 procedure FdbSrv_Clear;
 begin
  FdbSrvRec.CurveList:=0;
  FdbSrvRec.DataPath:='';
  FdbSrvRec.FilePref:='';
  FdbSrvRec.TimeQuota:=0;
  FdbSrvRec.SavePeriod:=0;
  FdbSrvRec.Account_uid:='';
  FdbSrvRec.Account_pwd:='';
  FdbSrvRec.Account_mode:=0;
  FdbSrvRec.DataCurve:=0;
  FdbSrvRec.DataCount:=0;
  FdbSrvRec.Incompleted:=0;
  FdbSrvRec.AwakeFlag:=false;
  FdbSrvRec.cmd_Awake:=0;
  FdbSrvRec.cmd_AssertMode:=0;
  FdbSrvRec.cmd_BigTimeout:=0;
  FdbSrvRec.cmd_TimeQuantum:=0;
  FdbSrvRec.cmd_FastTransfer:=0;
  FdbSrvRec.ErrorCount:=0;
  FdbSrvRec.AssertMode:=0;
  FdbSrvRec.BigTimeout:=0;
  FdbSrvRec.FirstStart:=false;
  FdbSrvRec.TimeQuantum:=0;
  FdbSrvRec.DaqTimeGap:=0;
 end;
 {
 FdbSrv Initialization.
 }
 procedure FdbSrv_Init;
 var s:String; i:Integer;
  procedure Cleanup;
  begin
   s:='';
  end;
 begin
  Cleanup;
  {
  Set uses DB Engine: ADO,SQLDB.
  }
  db_engine_uses_assign(db_engine_ado);
  db_engine_uses_assign(db_engine_zeos);
  //db_engine_uses_assign(db_engine_sqldb);
  {
  Initialize commands.
  }
  FdbSrvRec.cmd_Awake:=RegisterStdInCmd('@Awake','');
  FdbSrvRec.cmd_AssertMode:=RegisterStdInCmd('@AssertMode','');
  FdbSrvRec.cmd_BigTimeout:=RegisterStdInCmd('@BigTimeout','');
  FdbSrvRec.cmd_TimeQuantum:=RegisterStdInCmd('@TimeQuantum','');
  FdbSrvRec.cmd_FastTransfer:=RegisterStdInCmd('@FastTransfer','');
  {
  Initialize tags.
  }
  InitTag( FdbSrvRec.tagSave,    ReadIni('WriteEnableTag'), 1);
  InitTag( FdbSrvRec.tagSaveErr, ReadIni('WriteErrorsTag'), 2);
  {
  Initialize curve list.
  }
  FdbSrvRec.CurveList:=hashlist_init(0);
  FdbSrv_ReadCurveList('['+DevName+']');
  if (FdbSrv_CurveListCount<=0)
  then Trouble('CurveList not found!');
  {
  Read DataPath - directory where database located.
  }
  FdbSrvRec.DataPath:=AdaptFileName(ReadIni('DataPath'));
  if (Length(FdbSrvRec.DataPath)>0) then FdbSrvRec.DataPath:=DaqFileRef(FdbSrvRec.DataPath,'');
  Assertion(MkDir(FdbSrvRec.DataPath),'DataPath='+FdbSrvRec.DataPath);
  MakeDirAccessible(FdbSrvRec.DataPath);
  {
  Read Database file prefix.
  }
  FdbSrvRec.FilePref:=AdaptFileName(ReadIni('FilePrefix'));
  Assertion(FdbSrvRec.FilePref<>'','FilePrefix='+FdbSrvRec.FilePref);
  {
  Read SavePeriod.
  }
  FdbSrvRec.SavePeriod:=rValDef(ReadIni('SavePeriod'),60);
  Assertion(FdbSrvRec.SavePeriod>0,'SavePeriod='+Str(FdbSrvRec.SavePeriod));
  {
  Read TimeQuota.
  }
  FdbSrvRec.TimeQuota:=iValDef(ReadIni('TimeQuota'),1000);
  Assertion(FdbSrvRec.TimeQuota>0,'TimeQuota='+Str(FdbSrvRec.TimeQuota));
  {
  Read Account_uid, Account_pwd, Account_mode.
  }
  FdbSrvRec.Account_uid:=ReadIniStr(4+8+16,ParamStr('DaqConfigFile'),'['+DevName+']','Account_uid');
  Assertion(FdbSrvRec.Account_uid<>'','Account_uid='+FdbSrvRec.Account_uid);
  FdbSrvRec.Account_pwd:=ReadIniStr(4+8+16,ParamStr('DaqConfigFile'),'['+DevName+']','Account_pwd');
  Assertion(FdbSrvRec.Account_pwd<>'','Account_pwd='+FdbSrvRec.Account_pwd);
  FdbSrvRec.Account_mode:=iValDef(ReadIni('Account_mode'),-1);
  Assertion(FdbSrvRec.Account_mode>=0,'Account_mode='+Str(FdbSrvRec.Account_mode));
  {
  Read TimeQuantum and DaqTimeGap for time resolution.
  }
  FdbSrvRec.TimeQuantum:=rValDef(ReadIni('TimeQuantum'),TimeQuantDef);
  FdbSrv_AssignTimeQuantum(FdbSrvRec.TimeQuantum);
  Assertion(FdbSrvRec.TimeQuantum>0,StrFmt('TimeQuantum = %1.5g',FdbSrvRec.TimeQuantum));
  Assertion(FdbSrvRec.DaqTimeGap>0,StrFmt('DaqTimeGap = %1.5g',FdbSrvRec.DaqTimeGap));
  {
  Read FastTransfer to use recordset addnew/update.
  }
  FdbSrvRec.FastTransfer:=iValDef(ReadIni('FastTransfer'),0)<>0;
  Assertion(true,StrFmt('FastTransfer = %d',Ord(FdbSrvRec.FastTransfer)));
  {
  Read AssertMode to set method of success/error notifications.
  Bit 0 Mask 1 - Use Trouble (otherwise Problem) on error(s).
  Bit 1 Mask 2 - Use WriteErrorsTag to count errors.
  Bit 2 Mask 4 - Use Success (otherwise Details) on success.
  }
  FdbSrvRec.AssertMode:=iValDef(ReadIni('AssertMode'),3);
  Assertion(true,'AssertMode = '+Str(FdbSrvRec.AssertMode));
  {
  Read BigTimeout
  }
  FdbSrvRec.BigTimeout:=iValDef(ReadIni('BigTimeout'),WdtBigTimeout);
  Assertion(FdbSrvRec.BigTimeout>=0,'BigTimeout = '+Str(FdbSrvRec.BigTimeout));
  {
  Initialize internal variables.
  }
  FdbSrvRec.LastSave:=mSecNow;
  FdbSrvRec.FirstStart:=true;
  Cleanup;
 end;
 {
 FdbSrv Finalization.
 }
 procedure FdbSrv_Free;
 begin
  FreeAndZero(FdbSrvRec.CurveList);
 end;
 {
 FdbSrv Polling.
 }
 procedure FdbSrv_Poll;
 const SecFactor=1e3;
 var When:Real;
 begin
  {
  Awake on external event.
  }
  if AwakeFlag then DevPostCmdLocal('@Awake');
  {
  Save all curves do database.
  }
  if iGetTag(FdbSrvRec.tagSave)>0 then begin
   When:=msecnow;
   if (When-FdbSrvRec.LastSave>FdbSrvRec.SavePeriod*SecFactor) or FdbSrvRec.AwakeFlag then begin
    if FdbSrvRec.FirstStart then begin
     if FdbSrv_CheckCreateDatabase(When) then
     if FdbSrv_CheckInitDataBase(When) then
     FdbSrvRec.FirstStart:=false;
    end;    
    FdbSrv_SaveAllCurvesToDatabase(When);
    FdbSrvRec.LastSave:=When;
   end;
  end;
  {
  Handle button click.
  }
  if (ClickWhat=cw_MouseDown) then begin
   if (ClickButton=VK_LBUTTON) then begin
    if (ClickTag=FdbSrvRec.tagSave) then begin
     bNul(iSetTag(ClickTag,Ord(iGetTag(ClickTag)=0)));
     bNul(Voice(snd_Click));
    end;
    if (ClickTag=FdbSrvRec.tagSaveErr) then begin
     bNul(iSetTag(ClickTag,0));
     bNul(Voice(snd_Click));
    end;
   end;
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearDbLibrary;
  FdbSrv_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('@StartupScript','@FinallyScript');
  StdIn_SetTimeouts(0,MaxInt,MaxInt,0);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  InitDbLibrary;
  FdbSrv_Init;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  FdbSrv_Free;
  FreeDbLibrary;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  if ShouldPollDbLibrary then PollDbLibrary;
  FdbSrv_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid:Integer; ms,rv:Real;
  procedure Cleanup;
  begin
   cmd:=''; arg:='';
  end;
 begin
  Cleanup;
  Data:=Trim(Data);
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Awake
   }
   if (cmdid = FdbSrvRec.cmd_Awake) then begin
    ms:=msecnow-wdt_reset(false);
    Success('Awoke, wdt='+Str(ms)+' ms');
    FdbSrvRec.AwakeFlag:=true;
    Data:='';
   end else
   {
   @AssertMode 3
   }
   if (cmdid = FdbSrvRec.cmd_AssertMode) then begin
    if not IsEmptyStr(arg) then FdbSrvRec.AssertMode:=iValDef(arg,FdbSrvRec.AssertMode);
    Success(cmd+'='+Str(FdbSrvRec.AssertMode));
    Data:='';
   end else
   {
   @BigTimeout 30000
   }
   if (cmdid = FdbSrvRec.cmd_BigTimeout) then begin
    if not IsEmptyStr(arg) then FdbSrvRec.BigTimeout:=iValDef(arg,FdbSrvRec.BigTimeout);
    Success(cmd+'='+Str(FdbSrvRec.BigTimeout));
    Data:='';
   end else
   {
   @TimeQuantum 10
   }
   if (cmdid = FdbSrvRec.cmd_TimeQuantum) then begin
    if not IsEmptyStr(arg) then begin
     rv:=rValDef(arg,FdbSrvRec.TimeQuantum);
     FdbSrv_AssignTimeQuantum(rv);
    end;
    Success(cmd+'='+StrFmt('%1.5g',FdbSrvRec.TimeQuantum));
    Success('DaqTimeGap = '+StrFmt('%1.5g',FdbSrvRec.DaqTimeGap));
    Data:='';
   end else
   {
   @FastTransfer 1
   }
   if (cmdid = FdbSrvRec.cmd_FastTransfer) then begin
    if not IsEmptyStr(arg) then FdbSrvRec.FastTransfer:=(iValDef(arg,Ord(FdbSrvRec.FastTransfer))<>0);
    Success(cmd+'='+Str(Ord(FdbSrvRec.FastTransfer)));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
