 {
 ***********************************************************************
 Daq Pascal application program DimSrv.
 DimSrv is DIM Server for CRW-DAQ, see http://dim.web.cern.ch about DIM.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
 [@Help]
 |StdIn Command list: "@cmd=arg" or "@cmd arg"
 |********************************************************
 | ##i            - update service which contains tag i
 |                  i is integer tag reference as decimal string
 |                  data composed from tags included in service
 | ##i=m          - update service which contains tag i
 |                  i is integer tag reference as decimal string
 |                  data given as mime-encoded string m
 | ##s            - update service with nickname s
 |                  s is service name identifier string
 |                  data composed from tags included in service
 | ##s=m          - update service with nickname s
 |                  s is service name identifier string
 |                  data given as mime-encoded string m
 |                  Example:
 |                   bNul(DevPostMsg('&DimSrv ##'+str(tagButton)));
 |                   bNul(DevPostMsg('&DimSrv ##dic_cmnd_test'));
 | @dns.exe       - Run DNS.EXE - DIM Name Server in hidden window.
 | @dns.exe Period p - Set check period (p,ms) to restart DNS.EXE
 | @did.exe       - Run DID.EXE - DIM Information Display viewer.
 | @dimtree.exe   - Run DIMTree - DIM Tree data viewer.
 | @exit          - Exit DIMSRV.EXE, restart it.
 | @StatPeriod p  - Statistics period, ms
 | @ReportFlags f - DIM error(s) report flags: 1=Writeln,2=Echo; Default=3.
 | @IfHostName n c         - Run command c if host name (FQHN) equal to n.
 | @IfNotHostName n c      - Run command c if host name (FQHN) is not = n.
 | @IfComputerName n c     - Run command c if computer name equal to n.
 | @IfNotComputerName n c  - Run command c if computer name is not = n.
 | @IfProcessExists e c    - Run command c if process e exists.
 | @IfNotProcessExists e c - Run command c if process e not exists.
 | @FallBackMode n         - Set FallBack Mode no n (0/1).
 | @DimPorts add enable
 | @DimPorts delete
 | @DimPorts show
 | @DimBridge source.domain target.domain EXPORT/LIST/*
 | @DimBridge Period 60000
 | @DimBridge Restart
 | @DimBridge Start
 | @DimBridge Stop
 | @DimBridge Kill
 | @DimBridge Init
 | @DimBridge View
 |********************************************************
 []
 }
 {
[]
;******************************************************************************
;*** Data segment dtabmax should be >= 1024*12+MaxServ*19+MaxTags*3+MaxMails*3
;*** String table stabmax should be >= 1024*12+MaxServ*7+MaxMails
;******************************************************************************
[Compiler.Options]
Compiler.dtabmax = 1024*12+4096*19+16384*3+8192*3
Compiler.stabmax = 1024*12+4096*7+8192
Compiler.dtabmin = 1024*4
Compiler.stabmin = 1024*4
[]
;******************************************************************************
 }
program DimSrv;                  { DIM server for CRW-DAQ           }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 timed       = 1;                { Update service by timer          }
 monitored   = 2;                { Update service by monitor        }
 dis_info    = 1;                { information server               }
 dis_cmnd    = 2;                { command server                   }
 dic_info    = 3;                { information client               }
 dic_cmnd    = 4;                { command client                   }
 PollList    = 'timed,monitored';{ service update type list         }
 KindList    = 'dis_info,dis_cmnd,dic_info,dic_cmnd'; { serv. tipes }
 MaxServ     = 4096;             { max. number of services          }
 MaxTags     = 16384;            { max. number of tags              }
 MaxMails    = 8192;             { max. number of mails             }
 MaxBrid     = 32;               { max. number of bridges           }
 MaxLeng     = 16384;            { max. length of I/O string        }
 SendTimeOut = 100;              { timeout on message send          }
 TimerPeriod = 1000;             { poll period to check DIM         }
 StartDelay  = 7000;             { Delay on start                   }
 IniFlags    = 28;               { Delete comment, Trim             }
 ThreshRefs  = 16;               { Threshold for references         }
 ThreshNick  = 8;                { Threshold for nicknames          }
 MimeDump_1  = '/////w==';       { mime_encode(dump(-1))            }
 DimSiteBin  = 'Resource\DimSite\Dim\bin\'; { DIM binaries location }
 PPrioList   = 'Idle,Lower,Normal,Higher,High,RealTime'; { Process priorities }
 TPrioList   = 'tpIdle,tpLowest,tpLower,tpNormal,tpHigher,tpHighest,tpTimeCritical';
 opt_Post    = 1;                { use devPost with Mails           }
 DefDBPeriod = 60000;            { Default DimBridge Period, ms     }
 sevINFO     = 0;                { Severity INFO                    }
 sevWARNING  = 1;                { Severity WARNING                 }
 sevERROR    = 2;                { Severity ERROR                   }
 sevFATAL    = 3;                { Severity FATAL                   }
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 errorserv   : Integer;          { Error code for service           }
 errorterm   : Integer;          { Error code for host death        }
 DimLogMode  : Integer;          { 1:INFO,2:WARNING,4:ERROR,8:FATAL }
 TheDIM      : record            { DIM server data                  }
  DnsNode    : String;           { DIM_DNS_NODE                     }
  Section    : String;           { DIM_SECTION                      }
  Server     : String;           { DIM server path                  }
  NoLinkFill : String;           { Data filler on NoLink            }
  NoLinkMark : String;           { Data marker on NoLink            }
  NoLinkHand : Integer;          { Handler device on NoLink         }
  TStart     : Real;             { Time of DIM server start         }
  Count      : Integer;          { Number of known services         }
  Task       : String;           { Name of DIM task                 }
  Buff       : String;           { DIM task input buffer            }
  Line       : String;           { Temporary variable               }
  Tid        : Integer;          { DIM task identifier              }
  Prio       : array[1..4] of Integer;          {Priorities         }
  Padding    : Integer;          { Use data fields padding?         }
  ClnList    : Integer;          { List of connected clients        }
  IPipeSize  : Integer;          { StdInPipeSize                    }
  OPipeSize  : Integer;          { StdOutPipeSize                   }
  Serv       : array[1..MaxServ] of record          { Services      }
   Kind      : Integer;          { one of dis_info..dic_cmnd        }
   Name      : String;           { DIM service name                 }
   Nick      : String;           { DIM service nickname             }
   Sign      : String;           { Signature #n=                    }
   Frmt      : String;           { DIM format string                }
   Fill      : String;           { Error (nolink) data filler       }
   Mark      : String;           { Error (nolink) data marker       }
   Hand      : Integer;          { Error (nolink) handler device    }
   IsUp      : Integer;          { Service is UP (connected) flag   }
   Poll      : Integer;          { timed,monitored                  }
   Tout      : Integer;          { timeout, milliseconds            }
   Dead      : Integer;          { dead time on start, msec         }
   Last      : real;             { time of last update, msec        }
   TagIndex  : Integer;          { Index in Tags table              }
   TagCount  : Integer;          { Count of uses tags               }
   MailIndex : Integer;          { Index in Mails table             }
   MailCount : Integer;          { Count of uses mails              }
  end;                           {                                  }
  Tags       : record            { list of tag references           }
   Count     : Integer;          { counter of uses tags             }
   Ref       : array[1..MaxTags] of Integer; { tag reference        }
  end;                           {                                  }
  Mails      : record            { list of mails to send            }
   Count     : Integer;          { counter of uses mails            }
   Dev       : array[1..MaxMails] of Integer; { device reference    }
   Msg       : array[1..MaxMails] of String;  { message to send     }
   Opt       : array[1..MaxMails] of Integer; { message options     }
  end;                           {                                  }
  Stat       : record            { statistic information            }
   Period    : Integer;          { period to update stat.info       }
   LastTime  : Real;             { time of last update              }
   Errors    : Real;             { DimSrv.exe errors                }
   Memory    : Real;             { DimSrv.exe memory                }
   OpCount   : Real;             { vdpm_opcount value               }
   ServCount : array[dis_info..dic_cmnd] of Integer; {              }
   MsgPerSec : array[dis_info..dic_cmnd] of Integer; {              }
   TagPerSec : array[dis_info..dic_cmnd] of Integer; {              }
   ChrPerSec : array[dis_info..dic_cmnd] of Integer; {              }
  end;                           {                                  }
  rIndex     : record            { for fast reference search        }
   Count     : Integer;          { counter of entries               }
   Serv      : array[1..MaxTags] of Integer; {                      }
   Refs      : array[1..MaxTags] of Integer; {                      }
  end;                           {                                  }
  nIndex     : record            { for fast nickname search         }
   Count     : Integer;          { counter of entries               }
   Serv      : array[1..MaxServ] of Integer; {                      }
   Nick      : array[1..MaxServ] of String;  {                      }
  end;                           {                                  }
  UseSorting : Boolean;          { Use sorted index list            }
  CheckSort  : Boolean;          { Check list is sorted for tests   }
  CheckFind  : Boolean;          { Check list search for tests      }
  DnsPeriod  : Integer;          { Period to check dns.exe          }
  DnsTimer   : Real;             { Timer  to check dns.exe          }
  FallBackModeTag : Integer;     { FallBack Mode Tag reference      }
 end;                            {                                  }
 DimBridge   : record            { DimBridge.exe service            }
  Period     : Integer;          { Period to check state & restart  }
  Count      : Integer;          { Counter                          } 
  exe        : String;           { DimBridge.exe executable path    }
  ms         : Real;             { Time of last poll                }
  tid        : array[1..MaxBrid] of Integer; { Task Id              }
  arg        : array[1..MaxBrid] of String;  { Arguments of cmdline }
  buf        : array[1..MaxBrid] of String;  { Buffer for I/O       }
 end;                            {                                  }
 ReportFlags          : Integer; { Uses to control print/echo       }
 cmd_DnsExe           : Integer; { @dns.exe                         }
 cmd_DidExe           : Integer; { @did.exe                         }
 cmd_DimTreeExe       : Integer; { @dimtree.exe                     }
 cmd_Exit             : Integer; { @exit                            }
 cmd_StatPeriod       : Integer; { @StatPeriod                      }
 cmd_DimBridge        : Integer; { @DimBridge                       }
 cmd_DimPorts         : Integer; { @DimPorts                        }
 cmd_IfHostName       : Integer; { @IfHostName                      }
 cmd_IfNotHostName    : Integer; { @IfNotHostName                   }
 cmd_IfComputerName   : Integer; { @IfComputerName                  }
 cmd_IfNotComputerName: Integer; { @IfNotComputerName               }
 cmd_IfProcessExists  : Integer; { @IfProcessExists                 }
 cmd_IfNotProcessExists:Integer; { @IfNotProcessExists              }
 cmd_Errors           : Integer; { @Errors                          }
 cmd_Memory           : Integer; { @Memory                          }
 cmd_ClientEnter      : Integer; { @ClientEnter                     }
 cmd_ClientKill       : Integer; { @ClientKill                      }
 cmd_ClientExit       : Integer; { @ClientExit                      }
 cmd_ClientError      : Integer; { @ClientError                     }
 cmd_ServerError      : Integer; { @ServerError                     }
 cmd_ReportFlags      : Integer; { @ReportFlags                     }
 cmd_FallBackMode     : Integer; { @FallBackMode                    }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 {
 Default NoLink Filler/Marker.
 }
 function DefNoLinkFill:String;
 begin
  DefNoLinkFill:=Copy(dump(-1),1,3);
 end;
 function DefNoLinkMark:String;
 begin
  DefNoLinkMark:=Copy(dump(-1),1,3);
 end;
 {
 Post @Integrity @DIM_INFO/WARNING/ERROR/FATAL: message.
 }
 procedure PostAsyncIntegrityMsg(msg:String);
 begin
  rNul(Eval('@System '+DupeString('@Async @Silent ',4)+'@Integrity '+msg));
 end;
 procedure PostIntegrityDimEvent(msg:String; severity:Integer);
 begin
  if IsBit(DimLogMode,severity) then
  case severity of
   0: PostAsyncIntegrityMsg('@DIM_INFO: '+msg);    // INFO
   1: PostAsyncIntegrityMsg('@DIM_WARNING: '+msg); // WARNING
   2: PostAsyncIntegrityMsg('@DIM_ERROR: '+msg);   // ERROR
   3: PostAsyncIntegrityMsg('@DIM_FATAL: '+msg);   // FATAL
  end;
 end;
 procedure DimSysLogNote(msg:String; severity:Integer);
 begin
  if (msg<>'') then
  case severity of
   0: iNul(SysLogNote(sev_Success,msg)); // INFO
   1: iNul(SysLogNote(sev_Problem,msg)); // WARNING
   2: iNul(SysLogNote(sev_Trouble,msg)); // ERROR
   3: iNul(SysLogNote(sev_Failure,msg)); // FATAL
  end;
 end;
 procedure DimSuccess(msg:String);
 begin
  Success(msg);
  PostIntegrityDimEvent(msg,sevINFO);
 end;
 procedure DimTrouble(msg:String);
 begin
  Trouble(msg);
  PostIntegrityDimEvent(msg,sevERROR);
 end;
 {
 Report on host terminated.
 }
 procedure HostTerm(msg:String);
 begin
  Failure(errorterm,msg);
  PostIntegrityDimEvent(msg,sevWARNING);
 end;
 {
 Increment
 }
 procedure Inc(var n:Integer);
 begin
  n:=n+1;
 end;
 {
 Decrement
 }
 procedure Dec(var n:Integer);
 begin
  n:=n-1;
 end;
 {
 Compare strings, return -1,0,+1 if s1 <,=,> s2.
 }
 function StrComp(s1,s2:String):Integer;
 begin
  if s1>s2 then StrComp:=+1 else
  if s1<s2 then StrComp:=-1 else StrComp:=0;
 end;
 {
 Compare strings, ignore char case.
 }
 function StrIComp(s1,s2:String):Integer;
 begin
  StrIComp:=StrComp(UpCaseStr(s1),UpCaseStr(s2));
 end;
 {
 Quick search tag reference Ref in Dim.rIndex table.
 Return 0 if not found or index>0 in Dim.rIndex table.
 }
 function TheDIM_FindRef(Ref:Integer):Integer;
 var Left,Right,Middle,Comparison,Res:Integer;
 begin
  Res:=0;
  Left:=0;
  Right:=TheDIM.rIndex.Count-1;
  while Left<=Right do begin
   Middle:=(Left+Right) div 2;
   Comparison:=TheDIM.rIndex.Refs[1+Middle]-Ref;
   if Comparison<0 then Left:=Middle+1 else begin
    if Comparison=0 then Res:=1+Middle;
    Right:=Middle-1;
   end;
  end;
  TheDIM_FindRef:=Res;
 end;
 {
 Quick search nickname Ref in TheDIM.nIndex table.
 Return 0 if not found or index>0 in TheDIM.nIndex table.
 }
 function TheDIM_FindNick(Nick:String):Integer;
 var Left,Right,Middle,Comparison,Res:Integer;
 begin
  Res:=0;
  Left:=0;
  Right:=TheDIM.nIndex.Count-1;
  while Left<=Right do begin
   Middle:=(Left+Right) div 2;
   Comparison:=StrIComp(TheDIM.nIndex.Nick[1+Middle],Nick);
   if Comparison<0 then Left:=Middle+1 else begin
    if Comparison=0 then Res:=1+Middle;
    Right:=Middle-1;
   end;
  end;
  TheDIM_FindNick:=Res;
 end;
 {
 Clear Dim table.
 TheDIM_Clear(0) clears all fields.
 TheDIM_Clear(n>0) clear only Serv[n] fields.
 }
 procedure TheDIM_Clear(n:integer;MakeFree:boolean);
 var i,j:integer;
 begin
  if n=0 then begin
   if MakeFree then begin
    if TheDIM.ClnList<>0 then bNul(text_free(TheDIM.ClnList));
   end;
   TheDIM.Tid:=0;
   TheDIM.Buff:='';
   TheDIM.Line:='';
   TheDIM.Task:='';
   TheDIM.Count:=0;
   TheDIM.TStart:=0;
   TheDIM.Prio[1]:=0;
   TheDIM.Prio[2]:=0;
   TheDIM.Prio[3]:=0;
   TheDIM.Prio[4]:=0;
   TheDIM.Padding:=0;
   TheDIM.ClnList:=0;
   TheDIM.Server:='';
   TheDIM.DnsNode:='';
   TheDIM.Section:='';
   TheDIM.NoLinkHand:=0;
   TheDim.NoLinkFill:='';
   TheDim.NoLinkMark:='';
   TheDIM.Stat.Period:=0;
   TheDIM.Stat.Errors:=0;
   TheDIM.Stat.Memory:=0;
   TheDIM.Stat.LastTime:=0;
   TheDIM.Stat.OpCount:=0;
   for i:=dis_info to dic_cmnd do TheDIM.Stat.ServCount[i]:=0;
   for i:=dis_info to dic_cmnd do TheDIM.Stat.MsgPerSec[i]:=0;
   for i:=dis_info to dic_cmnd do TheDIM.Stat.TagPerSec[i]:=0;
   for i:=dis_info to dic_cmnd do TheDIM.Stat.ChrPerSec[i]:=0;
   TheDIM.Tags.Count:=0;
   for i:=1 to MaxTags do TheDIM.Tags.Ref[i]:=0;
   TheDIM.Mails.Count:=0;
   for i:=1 to MaxMails do TheDIM.Mails.Dev[i]:=0;
   for i:=1 to MaxMails do TheDIM.Mails.Msg[i]:='';
   for i:=1 to MaxMails do TheDIM.Mails.Opt[i]:=0;
   TheDIM.rIndex.Count:=0;
   for i:=1 to MaxTags do TheDIM.rIndex.Serv[i]:=0;
   for i:=1 to MaxTags do TheDIM.rIndex.Refs[i]:=0;
   TheDIM.nIndex.Count:=0;
   for i:=1 to MaxServ do TheDIM.nIndex.Serv[i]:=0;
   for i:=1 to MaxServ do TheDIM.nIndex.Nick[i]:='';
   TheDIM.DnsPeriod:=0;
   TheDIM.DnsTimer:=0;
  end;
  for i:=1 to MaxServ do
  if (n=0) or (n=i) then begin
   TheDIM.Serv[i].Kind:=0;
   TheDIM.Serv[i].Name:='';
   TheDIM.Serv[i].Nick:='';
   TheDIM.Serv[i].Sign:='';
   TheDIM.Serv[i].Frmt:='';
   TheDIM.Serv[i].Fill:='';
   TheDIM.Serv[i].Mark:='';
   TheDIM.Serv[i].Hand:=0;
   TheDIM.Serv[i].IsUp:=0;
   TheDIM.Serv[i].Poll:=0;
   TheDIM.Serv[i].Tout:=0;
   TheDIM.Serv[i].Dead:=0;
   TheDIM.Serv[i].Last:=0;
   TheDIM.Serv[i].TagIndex:=0;
   TheDIM.Serv[i].TagCount:=0;
   TheDIM.Serv[i].MailIndex:=0;
   TheDIM.Serv[i].MailCount:=0;
  end;
  TheDIM.FallBackModeTag:=0;
 end;
 {
 Initialize Dim table.
 }
 procedure TheDIM_Init;
 var i,j,k,t1,t2,ref,dev,gap:Integer; msg,fmt,line:String;
  procedure ReadIniStr(var s:string; name:string);
  var i,t:Integer;
  begin
   s:='';
   t:=ReadIniSection(text_new,IniFlags,'','');
   for i:=0 to text_numln(t)-1 do
   if WordCount(text_getln(t,i))>1 then
   if IsSameText(name,ExtractWord(1,text_getln(t,i))) then
   if s='' then s:=ExtractWord(2,text_getln(t,i));
   bNul(text_free(t));
  end;
  procedure RememberIndex(var Index:Integer; Value:Integer);
  begin
   if Index=0 then Index:=Value;
  end;
 begin
  msg:='';
  fmt:='';
  line:='';
  {---Clear Dim---}
  TheDIM_Clear(0,false);
  {---Read StatPeriod---}
  TheDIM.Stat.Period:=val(ReadIni('StatPeriod'));
  {---Read priorities---}
  TheDIM.Prio[1]:=val(ReadIni('PriorityClass'));
  TheDIM.Prio[2]:=val(ReadIni('PriorityMain'));
  TheDIM.Prio[3]:=val(ReadIni('PriorityIO'));
  TheDIM.Prio[4]:=val(ReadIni('PriorityTimer'));
  {---Read FIFO size---}
  TheDIM.IPipeSize:=val(ReadIni('StdInPipe'));
  TheDIM.OPipeSize:=val(ReadIni('StdOutPipe'));
  if (TheDIM.IPipeSize<=0) or (TheDIM.IPipeSize>64*1024) then TheDIM.IPipeSize:=64;
  if (TheDIM.OPipeSize<=0) or (TheDIM.OPipeSize>64*1024) then TheDIM.OPipeSize:=64;
  TheDIM.IPipeSize:=TheDIM.IPipeSize*1024;
  TheDIM.OPipeSize:=TheDIM.OPipeSize*1024;
  {---Read DIM_DNS_NODE---}
  ReadIniStr(TheDIM.DnsNode,'DIM_DNS_NODE');
  if Length(TheDIM.DnsNode)>0
  then Success('DIM_DNS_NODE='+TheDIM.DnsNode)
  else Trouble('DIM_DNS_NODE=?');
  {---Find DIM server executable---}
  TheDIM.Server:=DaqFileRef(AdaptExeFileName(ReadIni('DIM_SERVER')),'');
  if FileExists(TheDIM.Server)
  then Success('DIM_SERVER='+TheDIM.Server)
  else Trouble('Could not find DIM_SERVER: '+TheDIM.Server);
  {---Read NoLinkFill/Mark/Handler---}
  TheDIM.NoLinkFill:=mime_decode(ReadIniVar('NoLinkFill',28));
  TheDIM.NoLinkMark:=mime_decode(ReadIniVar('NoLinkMark',28));
  TheDIM.NoLinkHand:=RefFind('Device '+ReadIni('NoLinkHandler'));
  if (TheDIM.NoLinkFill='') then TheDim.NoLinkFill:=DefNoLinkFill;
  if (TheDIM.NoLinkMark='') then TheDim.NoLinkMark:=DefNoLinkMark;
  Success('NoLinkFill = '+mime_encode(TheDim.NoLinkFill)+' = '+hex_encode(TheDim.NoLinkFill));
  Success('NoLinkMark = '+mime_encode(TheDim.NoLinkMark)+' = '+hex_encode(TheDim.NoLinkMark));
  Success('NoLinkHandler = '+RefInfo(TheDIM.NoLinkHand,'Name'));
  DimLogMode:=iValDef(ReadIni('DimLogMode'),15);
  Success('DimLogMode = '+Str(DimLogMode));
  {---Read DIM task name---}
  ReadIniStr(TheDIM.Task,'DIM_TASK');
  if Length(TheDIM.Task)>0
  then Success('DIM_TASK='+TheDIM.Task)
  else Trouble('DIM_TASK=?');
  {---Read DIM_SECTION---}
  TheDIM.Section:=ReadIni('DIM_SECTION');
  if Length(TheDIM.Section)>0
  then t1:=ReadIniSection(text_new,IniFlags,'',TheDIM.Section)
  else t1:=text_new;
  if text_numln(t1)=0 then Trouble('Could not find DIM_SECTION!');
  for i:=0 to text_numln(t1)-1 do
  if WordCount(text_getln(t1,i))=3 then
  if TheDIM.Count<MaxServ then begin
   Inc(TheDIM.Count);
   TheDIM.Serv[TheDIM.Count].TagIndex:=0;
   TheDIM.Serv[TheDIM.Count].TagCount:=0;
   TheDIM.Serv[TheDIM.Count].MailIndex:=0;
   TheDIM.Serv[TheDIM.Count].MailCount:=0;
   if IsSameText('dis_info',ExtractWord(2,text_getln(t1,i))) then TheDIM.Serv[TheDIM.Count].Kind:=dis_info;
   if IsSameText('dis_cmnd',ExtractWord(2,text_getln(t1,i))) then TheDIM.Serv[TheDIM.Count].Kind:=dis_cmnd;
   if IsSameText('dic_info',ExtractWord(2,text_getln(t1,i))) then TheDIM.Serv[TheDIM.Count].Kind:=dic_info;
   if IsSameText('dic_cmnd',ExtractWord(2,text_getln(t1,i))) then TheDIM.Serv[TheDIM.Count].Kind:=dic_cmnd;
   TheDIM.Serv[TheDIM.Count].Name:=ExtractWord(3,text_getln(t1,i));
   TheDIM.Serv[TheDIM.Count].Nick:=ExtractWord(1,text_getln(t1,i));
   TheDIM.Serv[TheDIM.Count].Sign:='#'+str(TheDIM.Count)+'=';
   {---Read service section---}
   if TheDIM.Serv[TheDIM.Count].Kind>0 then
   if Length(TheDIM.Serv[TheDIM.Count].Name)>0 then
   if Length(TheDIM.Serv[TheDIM.Count].Nick)>0 then begin
    t2:=readinisection(text_new,IniFlags,'',TheDIM.Serv[TheDIM.Count].Nick);
    for j:=0 to text_numln(t2)-1 do begin
     line:=text_getln(t2,j);
     if WordCount(line)>1 then begin
      if IsSameText('polling',ExtractWord(1,line)) then begin
       if IsSameText('timed',ExtractWord(2,line))
       then TheDIM.Serv[TheDIM.Count].Poll:=timed;
       if IsSameText('monitored',ExtractWord(2,line))
       then TheDIM.Serv[TheDIM.Count].Poll:=monitored;
       TheDIM.Serv[TheDIM.Count].Tout:=val(ExtractWord(3,line));
       TheDIM.Serv[TheDIM.Count].Dead:=val(ExtractWord(4,line));
      end;
      if IsSameText('filling',ExtractWord(1,line))
      or IsSameText('NoLinkFillMarkHandler',ExtractWord(1,line)) then begin
       TheDIM.Serv[TheDIM.Count].Fill:=mime_decode(ExtractWord(2,line));
       TheDIM.Serv[TheDIM.Count].Mark:=mime_decode(ExtractWord(3,line));
       if (WordCount(line)>=4)
       then TheDIM.Serv[TheDIM.Count].Hand:=RefFind('Device '+ExtractWord(4,line))
       else TheDIM.Serv[TheDIM.Count].Hand:=0;
      end;
      if IsSameText('devmsg',ExtractWord(1,line))
      or IsSameText('devsend',ExtractWord(1,line))
      or IsSameText('devpost',ExtractWord(1,line))
      or IsSameText('devsendmsg',ExtractWord(1,line))
      or IsSameText('devpostmsg',ExtractWord(1,line)) then begin
       msg:=Trim(SkipWords(1,line));
       dev:=RefFind('Device '+ExtractWord(1,msg));
       if IsSameText(RefInfo(dev,'Type'),'Device') then begin
        msg:=Trim(SkipWords(1,msg));
        if Length(msg)>0 then msg:=msg+EOL;
        if TheDIM.Mails.Count<MaxMails then begin
         Inc(TheDIM.Mails.Count);
         Inc(TheDIM.Serv[TheDIM.Count].MailCount);
         TheDIM.Mails.Dev[TheDIM.Mails.Count]:=dev;
         TheDIM.Mails.Msg[TheDIM.Mails.Count]:=msg;
         if Pos('devpost',LoCaseStr(ExtractWord(1,line)))>0 then
         TheDIM.Mails.Opt[TheDIM.Mails.Count]:=iOr(TheDIM.Mails.Opt[TheDIM.Mails.Count],opt_Post);
         RememberIndex(TheDIM.Serv[TheDIM.Count].MailIndex,TheDIM.Mails.Count);
        end else Trouble('Could not add mail:'+line);
       end else Trouble('Invalid expression: '+line);
      end;
      if IsSameText('Tag',ExtractWord(1,line)) then
      for k:=2 to WordCount(line) do begin
       ref:=reffind('Tag '+ExtractWord(k,line));
       if IsSameText('Tag',refinfo(ref,'Type')) then begin
        if TheDIM.Tags.Count<MaxTags then begin
         Inc(TheDIM.Tags.Count);
         Inc(TheDIM.Serv[TheDIM.Count].TagCount);
         TheDIM.Tags.Ref[TheDIM.Tags.Count]:=ref;
         RememberIndex(TheDIM.Serv[TheDIM.Count].TagIndex,TheDIM.Tags.Count);
        end else Trouble('Could not add tag '+refinfo(ref,'Name'));
        TheDIM.Serv[TheDIM.Count].Frmt:='C';
       end;
      end;
     end;
    end;
    bNul(text_free(t2));
   end;
   if (TheDIM.Serv[TheDIM.Count].Kind=0)
   or (TheDIM.Serv[TheDIM.Count].Tout<0)
   or (TheDIM.Serv[TheDIM.Count].Dead<0)
   or (TheDIM.Serv[TheDIM.Count].TagCount=0)
   or (Length(TheDIM.Serv[TheDIM.Count].Name)=0)
   or (Length(TheDIM.Serv[TheDIM.Count].Nick)=0)
   or (Length(TheDIM.Serv[TheDIM.Count].Sign)=0)
   or (Length(TheDIM.Serv[TheDIM.Count].Frmt)=0)
   or (TheDIM.Tags.Count=MaxTags) or (TheDIM.Mails.Count=MaxMails)
   then begin
    while TheDIM.Serv[TheDIM.Count].TagCount>0 do begin
     Dec(TheDIM.Serv[TheDIM.Count].TagCount);
     TheDIM.Tags.Ref[TheDIM.Tags.Count]:=0;
     Dec(TheDIM.Tags.Count);
    end;
    TheDIM.Serv[TheDIM.Count].TagIndex:=0;
    while TheDIM.Serv[TheDIM.Count].MailCount>0 do begin
     Dec(TheDIM.Serv[TheDIM.Count].MailCount);
     TheDIM.Mails.Msg[TheDIM.Mails.Count]:='';
     TheDIM.Mails.Dev[TheDIM.Mails.Count]:=0;
     TheDIM.Mails.Opt[TheDIM.Mails.Count]:=0;
     Dec(TheDIM.Mails.Count);
    end;
    TheDIM.Serv[TheDIM.Count].MailIndex:=0;
    TheDIM_Clear(TheDIM.Count,true);
    Dec(TheDIM.Count);
   end else begin
    if TheDIM.Serv[TheDIM.Count].Poll=0
    then TheDIM.Serv[TheDIM.Count].Poll:=monitored;
    if Length(TheDIM.Serv[TheDIM.Count].Fill)=0
    then TheDIM.Serv[TheDIM.Count].Fill:=TheDIM.NoLinkFill;
    if Length(TheDIM.Serv[TheDIM.Count].Fill)=0
    then TheDIM.Serv[TheDIM.Count].Fill:=DefNoLinkFill;
    if Length(TheDIM.Serv[TheDIM.Count].Mark)=0
    then TheDIM.Serv[TheDIM.Count].Mark:=TheDIM.NoLinkMark;
    if Length(TheDIM.Serv[TheDIM.Count].Mark)=0
    then TheDIM.Serv[TheDIM.Count].Mark:=DefNoLinkMark;
    if (TheDIM.Serv[TheDIM.Count].Hand=0)
    then TheDIM.Serv[TheDIM.Count].Hand:=TheDIM.NoLinkHand;
    Inc(TheDIM.Stat.ServCount[TheDIM.Serv[TheDIM.Count].Kind]);
   end;
  end;
  bNul(text_free(t1));
  {---New client list---}
  TheDIM.ClnList:=text_new;
  {---Read UsePadding---}
  TheDIM.Padding:=Val(ReadIni('UsePadding'));
  {
  Calculate DIM format
  }
  if Val(ReadIni('UseFormat'))>0 then
  for i:=1 to TheDIM.Count do begin
   fmt:='';
   msg:='';
   for j:=1 to TheDIM.Serv[i].TagCount do begin
    ref:=TheDIM.Tags.Ref[TheDIM.Serv[i].TagIndex+j-1];
    if TypeTag(ref)=1 then msg:=msg+'L' else
    if TypeTag(ref)=2 then msg:=msg+'D' else
    if TypeTag(ref)=3 then msg:=msg+'C' else Trouble('Bad tag type.');
   end;
   k:=1;
   for j:=1 to Length(msg) do begin
    if j>1 then begin
     if msg[j-1]=msg[j] then k:=k+1 else begin
      if Length(fmt)>0 then fmt:=fmt+';';
      fmt:=fmt+msg[j-1]+':'+str(k);
      k:=1;
     end;
    end;
    if j=Length(msg) then begin
     if Length(fmt)>0 then fmt:=fmt+';';
     fmt:=fmt+msg[j];
    end;
   end;
   TheDIM.Serv[i].Frmt:=fmt;
  end;
  {
  Create sorted index for quick search to improve performance
  }
  TheDIM.UseSorting:=(Val(ReadIni('NoUseSorting'))<>1);
  TheDIM.CheckSort:=(Val(ReadIni('NoCheckSort'))<>1);
  TheDIM.CheckFind:=(Val(ReadIni('NoCheckFind'))<>1);
  if TheDIM.UseSorting then begin
   {
   Create reference index table
   }
   k:=1;
   for i:=1 to TheDIM.Count do begin
    for j:=1 to TheDIM.Serv[i].TagCount do begin
     TheDIM.rIndex.Count:=k;
     TheDIM.rIndex.Serv[k]:=i;
     TheDIM.rIndex.Refs[k]:=TheDIM.Tags.Ref[TheDIM.Serv[i].TagIndex+j-1];
     k:=k+1;
    end;
   end;
   {
   Sort reference index table with Shell method
   }
   t1:=0;
   gap:=TheDIM.rIndex.Count div 2;
   while gap>0 do begin
    i:=gap;
    while i<TheDIM.rIndex.Count do begin
     j:=i-gap;
     while j>=0 do begin
      if TheDIM.rIndex.Refs[1+j+gap]<TheDIM.rIndex.Refs[1+j] then begin
       k:=TheDIM.rIndex.Refs[1+j];
       TheDIM.rIndex.Refs[1+j]:=TheDIM.rIndex.Refs[1+j+gap];
       TheDIM.rIndex.Refs[1+j+gap]:=k;
       k:=TheDIM.rIndex.Serv[1+j];
       TheDIM.rIndex.Serv[1+j]:=TheDIM.rIndex.Serv[1+j+gap];
       TheDIM.rIndex.Serv[1+j+gap]:=k;
       t1:=t1+1;
      end;
      j:=j-gap;
     end;
     i:=i+1;
    end;
    gap:=gap div 2;
   end;
   {
   Create nickname index table
   }
   k:=1;
   for i:=1 to TheDIM.Count do begin
    TheDIM.nIndex.Count:=k;
    TheDIM.nIndex.Serv[k]:=i;
    TheDIM.nIndex.Nick[k]:=TheDIM.Serv[i].Nick;
    k:=k+1;
   end;
   {
   Sort nickname index table with Shell method
   }
   t2:=0;
   gap:=TheDIM.nIndex.Count div 2;
   while gap>0 do begin
    i:=gap;
    while i<TheDIM.nIndex.Count do begin
     j:=i-gap;
     while j>=0 do begin
      if StrIComp(TheDIM.nIndex.Nick[1+j+gap],TheDIM.nIndex.Nick[1+j])<0 then begin
       k:=TheDIM.nIndex.Serv[1+j];
       TheDIM.nIndex.Serv[1+j]:=TheDIM.nIndex.Serv[1+j+gap];
       TheDIM.nIndex.Serv[1+j+gap]:=k;
       fmt:=TheDIM.nIndex.Nick[1+j];
       TheDIM.nIndex.Nick[1+j]:=TheDIM.nIndex.Nick[1+j+gap];
       TheDIM.nIndex.Nick[1+j+gap]:=fmt;
       t2:=t2+1;
      end;
      j:=j-gap;
     end;
     i:=i+1;
    end;
    gap:=gap div 2;
   end;
   {
   Check sorting and searching for test purposes
   }
   if TheDIM.CheckSort then begin
    k:=0;
    for i:=2 to TheDIM.rIndex.Count do k:=k+Ord(TheDIM.rIndex.Refs[i]<TheDIM.rIndex.Refs[i-1]);
    if k=0
    then Success('Ref Sort Ok '+Str(t1))
    else Trouble('Ref Sort Fails '+Str(k));
    k:=0;
    for i:=2 to TheDIM.nIndex.Count do k:=k+Ord(StrIComp(TheDIM.nIndex.Nick[i],TheDIM.nIndex.Nick[i-1])<0);
    if k=0
    then Success('Nick Sort Ok '+Str(t2))
    else Trouble('Nick Sort Fails '+Str(k));
   end;
   if TheDIM.CheckFind then begin
    k:=0;
    for i:=1 to TheDIM.Tags.Count do begin
     ref:=TheDIM.Tags.Ref[i];
     j:=TheDIM_FindRef(ref);
     if j<=0 then k:=k+1 else
     if TheDIM.rIndex.Refs[j]<>ref then k:=k+1;
    end;
    if k=0
    then Success('Ref Search Ok')
    else Trouble('Ref Search Fails '+Str(k));
    k:=0;
    for i:=1 to TheDIM.Count do begin
     fmt:=TheDIM.Serv[i].Nick;
     j:=TheDIM_FindNick(fmt);
     if j<=0 then k:=k+1 else
     if StrIComp(TheDIM.nIndex.Nick[j],fmt)<>0 then k:=k+1;
    end;
    if k=0
    then Success('Nick Search Ok')
    else Trouble('Nick Search fails '+Str(k));
   end;
  end;
  TheDIM.FallBackModeTag:=FindTag(ReadIni('FallBackModeTag'));
  if (TypeTag(TheDIM.FallBackModeTag)<>1) then TheDIM.FallBackModeTag:=0;
  if TheDIM.FallBackModeTag=0 then TheDIM.FallBackModeTag:=FindTag(DevName+'.FallBackMode');
  if (TypeTag(TheDIM.FallBackModeTag)<>1) then TheDIM.FallBackModeTag:=0;
  fmt:='';
  msg:='';
  line:='';
 end;
 {
 Send message to DIM task.
 Wait for some time if transmitter FIFO is over.
 }
 procedure TheDIM_Send(msg:string);
 var ms:real;
 begin
  if TheDIM.Tid<>0 then
  if Length(msg)>0 then begin
   if task_txspace(TheDIM.Tid)<Length(msg) then begin
    ms:=msecnow;
    while(msecnow-ms<SendTimeOut) and (task_txspace(TheDIM.Tid)<Length(msg)) do bNul(Sleep(1));
   end;
   if task_send(TheDIM.Tid,msg+EOL)=0
   then Trouble('Send error!')
   else ViewImp(msg);
  end;
 end;
 {
 Update DIM service.
 }
 procedure TheDIM_Update(n:integer;s:string);
 var nt,kind:Integer; b:Boolean;
  {---Write data from dump to tags---}
  procedure DumpToTags;
  var i,j,k,ref,ofs:Integer;
  begin
   nt:=0;
   ofs:=1;
   for i:=1 to TheDIM.Serv[n].TagCount do begin
    ref:=TheDIM.Tags.Ref[TheDIM.Serv[n].TagIndex+i-1];
    case typetag(ref) of
     0: Trouble('Invalid tag!');
     1: begin
         if Length(s)-ofs+1>=SizeOfInteger then
         nt:=nt+ord(iSetTag(ref,dump2i(Copy(s,ofs,SizeOfInteger))));
         ofs:=ofs+SizeOfInteger;
        end;
     2: begin
         if Length(s)-ofs+1>=SizeOfReal then
         nt:=nt+ord(rSetTag(ref,dump2r(Copy(s,ofs,SizeOfReal))));
         ofs:=ofs+SizeOfReal;
        end;
     3: begin
         if Length(s)-ofs+1>=SizeOfChar then begin
          k:=0;
          j:=ofs;
          while (k=0) and (j<=Length(s)) do begin
           if ord(s[j])=0 then k:=j;
           j:=j+1;
          end;
          if k=0 then k:=Length(s)+1;
          nt:=nt+ord(sSetTag(ref,Copy(s,ofs,k-ofs)));
          ofs:=ofs+(k-ofs+1);
         end;
        end;
    end;
   end;
  end;
  {---Write data from tags to dump---}
  procedure TagsToDump;
  var i,ref:Integer;
  begin
   nt:=0;
   for i:=1 to TheDIM.Serv[n].TagCount do begin
    ref:=TheDIM.Tags.Ref[TheDIM.Serv[n].TagIndex+i-1];
    case typetag(ref) of
     0: Trouble('Invalid tag!');
     1: begin s:=s+dump(iGetTag(ref));  nt:=nt+1; end;
     2: begin s:=s+dump(rGetTag(ref));  nt:=nt+1; end;
     3: begin s:=s+sGetTag(ref)+chr(0); nt:=nt+1; end;
    end;
   end;
  end;
  {---Report update done---}
  procedure UpdateDone;
  var i,p,k,dev,ref,opt:Integer; msg,dat:string;
  begin
   for i:=1 to TheDIM.Serv[n].MailCount do begin
    dev:=TheDIM.Mails.Dev[TheDIM.Serv[n].MailIndex+i-1];
    msg:=TheDIM.Mails.Msg[TheDIM.Serv[n].MailIndex+i-1];
    opt:=TheDIM.Mails.Opt[TheDIM.Serv[n].MailIndex+i-1];
    p:=1;
    if Pos('%',msg)>0 then
    while p<Length(msg) do begin
     if msg[p]='%' then begin
      k:=0;
      dat:='';
      if Copy(msg,p+1,2)='**'
      then dat:=mime_encode(s)
      else k:=val(Copy(msg,p+1,2));
      if k>=1 then
      if k<=TheDIM.Serv[n].TagCount then begin
       ref:=TheDIM.Tags.Ref[TheDIM.Serv[n].TagIndex+k-1];
       case typetag(ref) of
        0 : dat:='';
        1 : dat:=mime_encode(dump(iGetTag(ref)));
        2 : dat:=mime_encode(dump(rGetTag(ref)));
        3 : dat:=mime_encode(sGetTag(ref));
       end;
      end;
      if Length(dat)>0 then begin
       msg:=Copy(msg,1,p-1)+dat+Copy(msg,p+3,MaxLeng);
       p:=p+Length(dat)-1;
      end;
     end;
     p:=p+1;
    end;
    if iAnd(opt,opt_Post)<>0 then begin
     if devPost(dev,msg)<Length(msg)
     then Trouble('Failed devPost '+RefInfo(dev,'Name')+' '+Trim(msg));
    end else begin
     if devSend(dev,msg)<Length(msg)
     then Trouble('Failed devSend '+RefInfo(dev,'Name')+' '+Trim(msg));
    end;
   end;
   TheDIM.Stat.ChrPerSec[kind]:=TheDIM.Stat.ChrPerSec[kind]+Length(s);
   TheDIM.Stat.TagPerSec[kind]:=TheDIM.Stat.TagPerSec[kind]+nt;
   TheDIM.Stat.MsgPerSec[kind]:=TheDIM.Stat.MsgPerSec[kind]+1;
   TheDIM.Serv[n].Last:=msecnow;
   msg:='';
   dat:='';
   s:='';
  end;
  procedure OnServiceUp;
  var dev:Integer; msg:String;
  begin
   msg:='';
   if (TheDIM.Serv[n].IsUp=0) then begin
    msg:='@DimServiceUp '+TheDIM.Serv[n].Name;
    DevPostCmdLocal(msg); dev:=TheDIM.Serv[n].Hand;
    if (dev<>0) and (dev<>devMySelf) then DevPostCmd(dev,msg);
    PostIntegrityDimEvent(msg,sevINFO);
   end;
   TheDIM.Serv[n].IsUp:=1;
   msg:='';
  end;
  procedure OnServiceDie;
  var dev:Integer; msg:String;
  begin
   msg:='';
   bNul(fixerror(errorserv));
   if (TheDIM.Serv[n].Mark<>TheDIM.Serv[n].Fill) then begin
    s:=TheDIM.Serv[n].Fill;
    DumpToTags;
    UpdateDone;
   end;
   if (TheDIM.Serv[n].IsUp<>0) then begin
    msg:='@DimServiceDie '+TheDIM.Serv[n].Name;
    DevPostCmdLocal(msg); dev:=TheDIM.Serv[n].Hand;
    if (dev<>0) and (dev<>devMySelf) then DevPostCmd(dev,msg);
    PostIntegrityDimEvent(msg,sevWARNING);
   end;
   TheDIM.Serv[n].IsUp:=0;
   msg:=''
  end;
 begin
  if (n>=1) and (n<=TheDIM.Count) then begin
   kind:=TheDIM.Serv[n].Kind;
   if (kind=dis_info) or (kind=dic_cmnd) then begin
    if Length(s)>0 then DumpToTags else TagsToDump;
    if Length(s)>0 then begin
     TheDIM_Send(TheDIM.Serv[n].Sign+mime_encode(s));
     UpdateDone;
    end;
   end;
   if Length(s)>0 then
   if (kind=dic_info) or (kind=dis_cmnd) then begin
    if msecnow-TheDIM.TStart>=TheDIM.Serv[n].Dead then
    if s=TheDIM.Serv[n].Mark then begin
     if msecnow-TheDIM.TStart>=StartDelay
     then OnServiceDie;
    end else begin
     if (TheDIM.Serv[n].IsUp=0) then OnServiceUp;
     DumpToTags;
     UpdateDone;
    end;
   end;
  end;
 end;
 {
 Analyse data coming from DIM task stdout.
 }
 procedure TheDIM_Process(var Data:string);
 var cmd,arg,buf:String; cmdid,i,p,n:integer;
  function ClientIndex(Client:String):Integer;
  var i,n:Integer;
  begin
   i:=0;
   n:=-1;
   while (i<text_numln(TheDIM.ClnList)) and (n<0) do begin
    if IsSameText(Client,text_getln(TheDIM.ClnList,i)) then n:=i;
    i:=i+1;
   end;
   ClientIndex:=n;
  end;
  procedure Report(msg:String; severity:Integer);
  begin
   if Length(msg)>0 then begin
    PostIntegrityDimEvent(msg,severity);
    if (iAnd(ReportFlags,2)>0) then DimSysLogNote(msg,severity); 
    msg:=GetDateTime(msecnow)+' -> '+devname+' : '+msg;
    if (iAnd(ReportFlags,2)>0) then bNul(echo(msg));
    if (iAnd(ReportFlags,1)>0) then writeln(msg);
   end;
  end;
  procedure ReportErr(msg:String);
  var severity:Integer;
  begin
   if Length(msg)>0 then begin
    severity:=WordIndex(ExtractWord(1,msg),'INFO,WARNING,ERROR,FATAL')-1;
    Report(msg,severity);
   end;
  end;
 begin
  cmd:='';
  arg:='';
  buf:='';
  if length(Data)>0 then begin
   if DebugFlagEnabled(dfViewExp) then ViewExp(Data);
   if strFetch(Data,1)='#' then begin
    p:=pos('=',Data);
    if p>0 then begin
     cmd:=Copy(Data,2,p-2);
     arg:=Copy(Data,p+1,Length(Data)-p);
    end else begin
     cmd:=Copy(Data,2,Length(Data)-1);
     arg:='';
    end;
    n:=val(cmd);
    if n<>0 then TheDIM_Update(n,mime_decode(arg));
   end else
   if strFetch(Data,1)='@' then
   if GotCommandId(Data,cmd,arg,cmdid) then begin
    if (cmdid=cmd_Exit) then begin
     Success('Exit with code '+Trim(arg));
    end else
    if (cmdid=cmd_Errors) then begin
     n:=val(arg);
     if n>0 then begin
      bNul(fixerror(errorcode));
      TheDIM.Stat.Errors:=TheDIM.Stat.Errors+n;
     end;
    end else
    if (cmdid=cmd_Memory) then begin
     n:=val(arg);
     if n>0 then TheDIM.Stat.Memory:=n;
    end else
    if (cmdid=cmd_ClientEnter) then begin
     buf:=ExtractWord(2,arg);
     if Length(buf)>0 then
     if ClientIndex(buf)<0 then begin
      bNul(text_addln(TheDIM.ClnList,buf));
      Report(ExtractWord(2,arg)+' connected.',sevINFO);
     end;
    end else
    if (cmdid=cmd_ClientKill) then begin
     Report(ExtractWord(2,arg)+' kill request.',sevINFO);
    end else
    if (cmdid=cmd_ClientExit) then begin
     buf:=ExtractWord(2,arg);
     if Length(buf)>0 then begin
      bNul(text_delln(TheDIM.ClnList,ClientIndex(buf)));
      Report(ExtractWord(2,arg)+' disconnected.',sevINFO);
     end;
    end else
    if (cmdid=cmd_DnsExe) then begin
     Success(Data);
    end else
    if (cmdid=cmd_ClientError) then begin
     ReportErr(arg+'.');
    end else
    if (cmdid=cmd_ServerError) then begin
     ReportErr(arg+'.');
    end;
   end;
  end;
  buf:='';
  cmd:='';
  arg:='';
 end;
 {
 Stop DIM server task if one started.
 }
 procedure TheDIM_Stop;
 begin
  if TheDIM.Tid>0 then begin
   if task_wait(TheDIM.Tid,0) then begin
    TheDIM_Send('@stop');
    TheDIM_Send('@exit');
    if task_wait(TheDIM.Tid,1000) then bNul(task_kill(TheDIM.Tid,0,1,1000));
    TheDIM.TStart:=_plusinf;
    if task_rxcount(TheDIM.Tid)>0 then
    while Task_Readln(TheDIM.Tid,TheDIM.Line,TheDIM.Buff) do TheDIM_Process(TheDIM.Line);
    DimSuccess('DimSrv PID '+str(task_pid(TheDIM.Tid))+' stopped with exit code '+str(task_result(TheDIM.Tid)));
   end;
   bNul(task_free(TheDIM.Tid));
  end;
  TheDIM.Tid:=0;
  TheDIM.Buff:='';
  TheDIM.Line:='';
 end;
 {
 Finalize Dim table.
 }
 procedure TheDIM_Free;
 begin
  TheDIM_Stop;
  TheDIM_Clear(0,true);
 end;
 {
 Start DIM server if one not started.
 }
 procedure TheDIM_Start;
 var i,j:Integer;
 begin
  if TheDIM.Tid=0 then begin
   {
   Print header
   }
   Success('Start DIM Server '+TheDIM.Server);
   {
   Initialize separate user task, run it invisible...
   }
   TheDIM.Tid:=task_init(TheDIM.Server);
   if pos('?',task_ctrl(TheDIM.Tid,'HomeDir='+ExtractFilePath(TheDIM.Server))
             +task_ctrl(TheDIM.Tid,'StdInPipeSize='+str(TheDIM.IPipeSize))
             +task_ctrl(TheDIM.Tid,'StdOutPipeSize='+str(TheDIM.OPipeSize))
             +task_ctrl(TheDIM.Tid,'Display=0')
          )>0
   then begin
    DimTrouble('User task setup error!');
    TheDIM_Stop;
   end;
   {
   Setup priorities...
   }
   Success('Set StdInPriority   = '+task_ctrl(TheDIM.Tid,'StdInPriority='+
           extractword(TheDIM.Prio[2]+4,TPrioList)));
   Success('Set StdOutPriority  = '+task_ctrl(TheDIM.Tid,'StdOutPriority='+
           extractword(TheDIM.Prio[2]+4,TPrioList)));
   Success('Set ThreadPriority  = '+task_ctrl(TheDIM.Tid,'ThreadPriority='+
           extractword(TheDIM.Prio[2]+4,TPrioList)));
   Success('Set ProcessPriority = '+task_ctrl(TheDIM.Tid,'ProcessPriority='+
           extractword(TheDIM.Prio[1]+3,PPrioList)));
   {
   Run task if one was created...
   }
   if TheDIM.Tid>0 then
   if task_run(TheDIM.Tid) then begin
    Success('TaskId  = '+str(TheDIM.Tid));
    Success('TaskPid = '+str(task_pid(TheDIM.Tid)));
    Success('TaskRef = '+str(task_ref(TheDIM.Tid)));
    Success('CmdLine = '+task_ctrl(TheDIM.Tid,'CmdLine'));
    Success('HomeDir = '+task_ctrl(TheDIM.Tid,'HomeDir'));
    Success('PipeIn  = '+task_ctrl(TheDIM.Tid,'StdInPipeSize'));
    Success('PipeOut = '+task_ctrl(TheDIM.Tid,'StdOutPipeSize'));
    Success('IPrior. = '+task_ctrl(TheDIM.Tid,'StdInPriority'));
    Success('OPrior. = '+task_ctrl(TheDIM.Tid,'StdOutPriority'));
    Success('TPrior. = '+task_ctrl(TheDIM.Tid,'ThreadPriority'));
    Success('PPrior. = '+task_ctrl(TheDIM.Tid,'ProcessPriority'));
    Success('Display = '+task_ctrl(TheDIM.Tid,'Display'));
    DimSuccess('DimSrv PID '+str(task_pid(TheDIM.Tid))+' started.');
   end else begin
    DimTrouble('Could not start DIM Server!');
    TheDIM_Stop;
   end;
   {
   Is it Ok with user task? Send preset parameters.
   }
   if TheDIM.Tid>0 then
   if task_wait(TheDIM.Tid,0) then begin
    TheDIM_Send('@padding='+Str(TheDIM.Padding));
    TheDIM_Send('@dnsnode='+TheDIM.DnsNode);
    TheDIM_Send('@dns.exe');
    TheDIM.Line:='';
    for i:=1 to TheDIM.Count do begin
     TheDIM.Line:='+'+str(i)+'=';
     TheDIM.Line:=TheDIM.Line+ExtractWord(TheDIM.Serv[i].Kind,KindList);
     TheDIM.Line:=TheDIM.Line+','+TheDIM.Serv[i].Name;
     TheDIM.Line:=TheDIM.Line+','+TheDIM.Serv[i].Frmt;
     TheDIM.Line:=TheDIM.Line+','+ExtractWord(TheDIM.Serv[i].Poll,PollList);
     TheDIM.Line:=TheDIM.Line+','+str(TheDIM.Serv[i].Tout div 1000);
     TheDIM.Line:=TheDIM.Line+','+mime_encode(TheDIM.Serv[i].Mark);
     TheDIM_Send(TheDIM.Line);
    end;
    TheDIM.Line:='';
    if (TheDIM.Stat.ServCount[dis_info]>0)
    or (TheDIM.Stat.ServCount[dis_cmnd]>0)
    then TheDIM_Send('@start='+TheDIM.Task);
    TheDIM.Line:='@priority='+str(TheDIM.Prio[1]);
    TheDIM.Line:=TheDIM.Line+','+str(TheDIM.Prio[2]);
    TheDIM.Line:=TheDIM.Line+','+str(TheDIM.Prio[3]);
    TheDIM.Line:=TheDIM.Line+','+str(TheDIM.Prio[4]);
    TheDIM_Send(TheDIM.Line);
    TheDIM.Line:='';
    TheDIM.TStart:=msecnow;
   end else bNul(fixerror(errorcode));
  end;
 end;
 {
 DIM service polling: check last polling time, update if needed.
 }
 procedure TheDIM_Poll;
 var n,kind:Integer; ms:real;
 begin
  ms:=msecnow;
  for n:=1 to TheDIM.Count do begin
   kind:=TheDIM.Serv[n].Kind;
   if (kind=dis_info) or (kind=dic_cmnd) then begin
    if TheDIM.Serv[n].Poll=Timed then begin
     if TheDIM.Serv[n].Tout>0 then
     if ms-TheDIM.Serv[n].Last>=TheDIM.Serv[n].Tout then TheDIM_Update(n,'');
    end else
    if TheDIM.Serv[n].Poll=Monitored then begin
     if TheDIM.Serv[n].Tout>0 then
     if ms-TheDIM.Serv[n].Last>=TheDIM.Serv[n].Tout then TheDIM_Update(n,'');
    end;
   end;
  end;
 end;
 {
 DIM stat. information report.
 }
 procedure TheDIM_Stat;
 const fw=11; fd=2;
 var i:Integer; dt,factor,opcount,cpclock:Real;
 begin
  if TheDIM.Stat.Period>0 then begin
   dt:=mSecNow-TheDIM.Stat.LastTime;
   if dt>=TheDIM.Stat.Period then begin
    factor:=1e3/dt;
    if DebugFlagEnabled(dfDetails) then begin
     writeln(DevName,' ~ ',GetDateTime(mSecNow),' - Stat.Report:');
     writeln(DevName,' ~ Stat.Info ','dis_info':fw,'dis_cmnd':fw,'dic_info':fw,'dic_cmnd':fw);
     write(DevName,'   ServCount ');
     for i:=dis_info to dic_cmnd do write(TheDIM.Stat.ServCount[i]:fw);
     writeln;
     write(DevName,'   MsgPerSec ');
     for i:=dis_info to dic_cmnd do write(TheDIM.Stat.MsgPerSec[i]*factor:fw:fd);
     writeln;
     write(DevName,'   TagPerSec ');
     for i:=dis_info to dic_cmnd do write(TheDIM.Stat.TagPerSec[i]*factor:fw:fd);
     writeln;
     write(DevName,'   ChrPerSec ');
     for i:=dis_info to dic_cmnd do write(TheDIM.Stat.ChrPerSec[i]*factor:fw:fd);
     writeln;
     opcount:=vdpm_opcount-TheDIM.Stat.OpCount;
     TheDIM.Stat.OpCount:=vdpm_opcount;
     writeln(DevName,'   Memory = ',TheDIM.Stat.Memory:1:0,
                     '   Errors = ',TheDIM.Stat.Errors:1:0,
                     '   Clients = ',text_numln(TheDIM.ClnList):1,
                     '   OpCount = ',opcount:1:0);
    end;
    if aomap(0,12)<>0 then begin
     for i:=dis_info to dic_cmnd do UpdateAo(i-1,time,TheDIM.Stat.MsgPerSec[i]*factor);
     for i:=dis_info to dic_cmnd do UpdateAo(i+3,time,TheDIM.Stat.TagPerSec[i]*factor);
     for i:=dis_info to dic_cmnd do UpdateAo(i+7,time,TheDIM.Stat.ChrPerSec[i]*factor);
    end;
    for i:=dis_info to dic_cmnd do TheDIM.Stat.MsgPerSec[i]:=0;
    for i:=dis_info to dic_cmnd do TheDIM.Stat.TagPerSec[i]:=0;
    for i:=dis_info to dic_cmnd do TheDIM.Stat.ChrPerSec[i]:=0;
    TheDIM.Stat.LastTime:=msecnow;
   end;
  end;
 end;
 {
 Clear DimBridge variables
 }
 procedure DimBridge_Clear;
 var i:Integer;
 begin
  DimBridge.ms:=0;
  DimBridge.exe:='';
  DimBridge.Count:=0;
  DimBridge.Period:=0;
  for i:=1 to MaxBrid do DimBridge.tid[i]:=0;
  for i:=1 to MaxBrid do DimBridge.arg[i]:='';
  for i:=1 to MaxBrid do DimBridge.buf[i]:='';
 end;
 {
 Initialize DimBridge variables
 }
 procedure DimBridge_Init;
 var code:Integer;
 begin
  DimBridge_Clear;
  DimBridge.Period:=DefDBPeriod;
  if IsWindows then DimBridge.exe:=ParamStr('FileSearch DimBridge.exe');
  if IsUnix then DimBridge.exe:=Trim(RunSysCommandAsText('unix which DimBridge','','',code,3000));
  if FileExists(DimBridge.exe)
  then Success('DimBridge.exe path '+DimBridge.exe)
  else Trouble('Could not find DimBridge.exe '+DimBridge.exe);
 end;
 {
 Read DimBridge[i] input; i=0 mean all i=(1..DimBridge.Count)
 }
 procedure DimBridge_Read(i:Integer);
 var line:String;
 begin
  line:='';
  if (i=0) then for i:=1 to DimBridge.Count do DimBridge_Read(i) else
  if (i>=1) and (i<=DimBridge.Count) and (i<=MaxBrid) then begin
   if DimBridge.tid[i]<>0 then
   while Pipe_Readln(DimBridge.tid[i],line,DimBridge.buf[i]) do
   Success('DimBridge['+Str(i)+'] -> '+line);
  end;
  line:='';
 end;
 {
 Kill DimBridge[i] process; i=0 mean all i=(1..DimBridge.Count)
 }
 procedure DimBridge_Kill(i:Integer);
 var tid,pid:Integer;
  procedure TryTerminate(tid,pid,i,met,timeout:Integer);
  begin
   if tid<>0 then if task_Wait(tid,0) then if task_Kill(tid,met,0,timeout)
   then Success('DimBridge['+Str(i)+'] PID '+Str(pid)+' CODE '+Str(task_Result(tid))+' killed('+Str(met)+')');
  end;
 begin
  if (i=0) then for i:=1 to DimBridge.Count do DimBridge_Kill(i) else
  if (i>=1) and (i<=DimBridge.Count) and (i<=MaxBrid) then begin
   tid:=DimBridge.tid[i];
   if tid<>0 then begin
    DimBridge_Read(i);
    pid:=task_Pid(tid);
    TryTerminate(tid,pid,i,1,100);  rNul(wdt_reset(true));
    TryTerminate(tid,pid,i,3,1000); rNul(wdt_reset(true));
    TryTerminate(tid,pid,i,0,1000); rNul(wdt_reset(true));
    DimBridge_Read(i);
    if task_Wait(tid,0) then Trouble('Could not kill DimBridge['+Str(i)+'] PID '+Str(pid)
                                    +' CODE '+Str(task_Result(tid)));
    bNul(task_Free(tid));
   end;
   DimBridge.tid[i]:=0;
   DimBridge.buf[i]:='';
  end;
 end;
 {
 Run DimBridge[i] process; i=0 mean all (1..DimBridge.Count)
 [&DimSrv.StartupPath]
 @DimBridge egp-mzvs.dep0404.ru egp-main.dep0404.ru EGP/MZVS/*
 }
 procedure DimBridge_Run(i:Integer);
 var pid,cod:Integer;
 begin
  if (i=0) then for i:=1 to DimBridge.Count do DimBridge_Run(i) else
  if (i>=1) and (i<=DimBridge.Count) and (i<=MaxBrid) then begin
   if DimBridge.tid[i]<>0 then begin
    if not task_Wait(DimBridge.tid[i],0) then begin
     pid:=task_Pid(DimBridge.tid[i]);
     cod:=task_Result(DimBridge.tid[i]);
     Trouble('DimBridge['+Str(i)+'] PID '+Str(pid)+' CODE '+Str(cod)+' is DEAD');
     bNul(task_Free(DimBridge.tid[i]));
     DimBridge.tid[i]:=0;
     DimBridge.buf[i]:='';
    end;
   end;
   if DimBridge.tid[i]=0 then begin
    if iMin(Length(DimBridge.exe),Length(DimBridge.arg[i]))>0 then begin
     DimBridge.tid[i]:=task_Init(DimBridge.exe+' '+DimBridge.arg[i]);
     sNul(task_Ctrl(DimBridge.tid[i],'Display=0'));
     sNul(task_Ctrl(DimBridge.tid[i],'StdInPipeSize='+Str(16*1024)));
     sNul(task_Ctrl(DimBridge.tid[i],'StdOutPipeSize='+Str(32*1024)));
     sNul(task_Ctrl(DimBridge.tid[i],'HomeDir='+ExtractFilePath(DimBridge.exe)));
     sNul(task_Ctrl(DimBridge.tid[i],'ProcessPriority='+ExtractWord(TheDIM.Prio[1]+3,PPrioList)));
     if task_Run(DimBridge.tid[i]) then begin
      pid:=task_Pid(DimBridge.tid[i]);
      cod:=task_Result(DimBridge.tid[i]);
      Success('Start DimBridge['+Str(i)+'], PID '+Str(pid)+' CODE '+Str(cod));
      Success('CmdLine         = '+task_Ctrl(DimBridge.tid[i],'CmdLine'));
      Success('HomeDir         = '+task_Ctrl(DimBridge.tid[i],'HomeDir'));
      Success('Display         = '+task_Ctrl(DimBridge.tid[i],'Display'));
      Success('StdInPipeSize   = '+task_Ctrl(DimBridge.tid[i],'StdInPipeSize'));
      Success('StdOutPipeSize  = '+task_Ctrl(DimBridge.tid[i],'StdOutPipeSize'));
      Success('ProcessPriority = '+task_Ctrl(DimBridge.tid[i],'ProcessPriority'));
     end else begin
      Trouble('DimBridge['+Str(i)+'] start failed, cmd: '+task_Ctrl(DimBridge.tid[i],'CmdLine'));
      bNul(task_Free(DimBridge.tid[i]));
      DimBridge.tid[i]:=0;
      DimBridge.buf[i]:='';
     end;
    end;
    rNul(wdt_reset(true));
   end;
  end;
 end;
 {
 Finalize DimBridge processes and variables (except exe,period)
 }
 procedure DimBridge_Free;
 var i:Integer;
 begin
  DimBridge_Kill(0);
  for i:=1 to DimBridge.Count do DimBridge.tid[i]:=0;
  for i:=1 to DimBridge.Count do DimBridge.arg[i]:='';
  for i:=1 to DimBridge.Count do DimBridge.buf[i]:='';
  DimBridge.Count:=0;
  DimBridge.ms:=0;
 end;
 {
 View DimBridge processes and variables
 }
 procedure DimBridge_View;
 var i,tid,pid,cod:Integer;
 begin
  Success('@DimBridge Period '+Str(DimBridge.Period));
  if Length(DimBridge.exe)>0 then Success('@DimBridge Binary '+DimBridge.exe);
  for i:=1 to DimBridge.Count do begin
   tid:=DimBridge.tid[i];
   pid:=task_Pid(tid);
   cod:=task_Result(tid);
   if pid=0
   then Success('@DimBridge['+Str(i)+'] '+DimBridge.arg[i])
   else Success('@DimBridge['+Str(i)+'] '+DimBridge.arg[i]+' PID '+Str(pid)+' CODE '+Str(cod));
  end;
 end;
 {
 Execute DimBridge polling actions
 }
 procedure DimBridge_Polling;
 var i:Integer;
 begin
  if SysTimer_Pulse(100)>0 then begin
   for i:=1 to DimBridge.Count do
   if DimBridge.tid[i]<>0 then DimBridge_Read(i);
  end;
  if DimBridge.Period>0 then
  if mSecNow-DimBridge.ms>DimBridge.Period then begin
   for i:=1 to DimBridge.Count do begin
    if not task_Wait(DimBridge.tid[i],0) then DimBridge_Run(i);
   end;
   DimBridge.ms:=msecnow;
  end;
 end;
 {
 @DimBridge source.dep0404.ru target.dep0404.ru EXPORT/LIST/*
 @DimBridge Period 60000
 @DimBridge Restart
 @DimBridge Start
 @DimBridge Stop
 @DimBridge Init
 @DimBridge View
 }
 procedure DimBridge_Cmd(cmd,arg:String);
 const AppendDomain=true;
 var s,t,e,h,c,d,r,f:String;
 begin
  s:=''; t:=''; e:=''; h:=''; c:=''; d:=''; r:=''; f:='';
  if IsSameText(cmd,'@DimBridge') then begin
   if (WordCount(arg)=3) then begin
    s:=ExtractWord(1,arg);                      // source
    t:=ExtractWord(2,arg);                      // target
    e:=ExtractWord(3,arg);                      // export list
    h:=ParamStr('HostName 1');                  // host name
    c:=ParamStr('ComputerName');                // computer name
    d:=Copy(h,Pos('.',h)+1); if d=h then d:=''; // domain
    r:='0';                                     // rate, sec
    f:='-copy';                                 // copy flag
    if (s='.') or IsSameText(s,'localhost') then s:=c;
    if (t='.') or IsSameText(t,'localhost') then t:=c;
    if Pos('%',s)>0 then s:=StrReplace(s,'%HostName%',h,3);
    if Pos('%',t)>0 then t:=StrReplace(t,'%HostName%',h,3);
    if Pos('%',s)>0 then s:=StrReplace(s,'%ComputerName%',c,3);
    if Pos('%',t)>0 then t:=StrReplace(t,'%ComputerName%',c,3);
    if AppendDomain then begin
     if (Pos('.',s)=0) and (d<>'') then s:=s+'.'+d;
     if (Pos('.',t)=0) and (d<>'') then t:=t+'.'+d;
     if (Pos('.',s)=0) then s:=s+'.';
    end;
    s:=LoCaseStr(s);
    t:=LoCaseStr(t);
    if DimBridge.Count<MaxBrid then begin
     DimBridge.Count:=DimBridge.Count+1;
     DimBridge.arg[DimBridge.Count]:=s+' '+t+' '+e+' '+r+' '+f;
     Success('@DimBridge '+DimBridge.arg[DimBridge.Count]);
     if DimBridge.ms>0 then DimBridge_Run(DimBridge.Count);
    end else Trouble('DimBridge table overflow ('+Str(DimBridge.Count)+')');
   end else
   if (WordCount(arg)=2) and IsSameText(ExtractWord(1,arg),'Period') then begin
    DimBridge.Period:=iValDef(ExtractWord(2,arg),DefDBPeriod);
    Success('@DimBridge Period '+Str(DimBridge.Period));
   end else
   if (WordCount(arg)=1) and IsSameText(ExtractWord(1,arg),'Init') then begin
    DimBridge_Free;
    DimBridge_Init;
   end else
   if (WordCount(arg)=1) and IsSameText(ExtractWord(1,arg),'View') then begin
    DimBridge_View;
   end else
   if (WordCount(arg)=1) and IsSameText(ExtractWord(1,arg),'Kill') then begin
    DimBridge_Kill(0);
    DimBridge.ms:=0;
   end else
   if (WordCount(arg)=1) and IsSameText(ExtractWord(1,arg),'Stop') then begin
    DimBridge_Kill(0);
    DimBridge.ms:=0;
   end else
   if (WordCount(arg)=1) and IsSameText(ExtractWord(1,arg),'Start') then begin
    DimBridge_Run(0);
    DimBridge.ms:=0;
   end else
   if (WordCount(arg)=1) and IsSameText(ExtractWord(1,arg),'Restart') then begin
    DimBridge_Kill(0);
    DimBridge_Run(0);
    DimBridge.ms:=0;
   end else
   Trouble('Invalid command syntax: '+cmd+' '+arg);
  end;
  s:=''; t:=''; e:=''; h:=''; c:=''; d:=''; r:=''; f:='';
 end;
 {
 @DimPorts add enable
 @DimPorts delete
 @DimPorts show
 }
 procedure DimPorts(arg:String);
 begin
  if not IsEmptyStr(arg) then
  rNul(Eval('@System @Async @Silent @Run cmd /c '+AddBackSlash(ProgramSourceDir)+'DimPorts.cmd'+' '+arg
           +' & pause'));
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  TheDIM_Clear(0,false);
  DimBridge_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('@StartupScript','');
  StdIn_SetTimeouts(0,0,MaxInt,0);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  errorserv:=registererr(devname+': DIM service die.');
  errorterm:=registererr(devname+': process terminated.');
  cmd_DnsExe             := RegisterStdInCmd('@dns.exe',     '');
  cmd_DidExe             := RegisterStdInCmd('@did.exe',     '');
  cmd_DimTreeExe         := RegisterStdInCmd('@dimtree.exe', '');
  cmd_Exit               := RegisterStdInCmd('@exit',        '');
  cmd_StatPeriod         := RegisterStdInCmd('@StatPeriod',  '');
  cmd_DimBridge          := RegisterStdInCmd('@DimBridge',   '');
  cmd_DimPorts           := RegisterStdInCmd('@DimPorts',    '');
  cmd_IfHostName         := RegisterStdInCmd('@IfHostName',         '');
  cmd_IfNotHostName      := RegisterStdInCmd('@IfNotHostName',      '');
  cmd_IfComputerName     := RegisterStdInCmd('@IfComputerName',     '');
  cmd_IfNotComputerName  := RegisterStdInCmd('@IfNotComputerName',  '');
  cmd_IfProcessExists    := RegisterStdInCmd('@IfProcessExists',    '');
  cmd_IfNotProcessExists := RegisterStdInCmd('@IfNotProcessExists', '');
  cmd_Errors             := RegisterStdInCmd('@Errors',             '');
  cmd_Memory             := RegisterStdInCmd('@Memory',             '');
  cmd_ClientEnter        := RegisterStdInCmd('@ClientEnter',        '');
  cmd_ClientKill         := RegisterStdInCmd('@ClientKill',         '');
  cmd_ClientExit         := RegisterStdInCmd('@ClientExit',         '');
  cmd_ClientError        := RegisterStdInCmd('@ClientError',        '');
  cmd_ServerError        := RegisterStdInCmd('@ServerError',        '');
  cmd_ReportFlags        := RegisterStdInCmd('@ReportFlags',        '');
  cmd_FallBackMode       := RegisterStdInCmd('@FallBackMode',       '');
  TheDIM_Init;
  DimBridge_Init;
  ReportFlags:=3;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  TheDIM_Free;
  DimBridge_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  {
  If DIM server is not still running,
  try to start DIM server periodically.
  }
  if TheDIM.Tid=0 then
  if SysTimer_Pulse(TimerPeriod)>0 then TheDIM_Start;
  {
  Communicate with DIM server if one still running...
  }
  if TheDIM.Tid>0 then
  if task_wait(TheDIM.Tid,0) then begin
   {
   If has data coming from Task StdOut, analyse it...
   }
   if task_rxcount(TheDIM.Tid)>0 then
   while Task_Readln(TheDIM.Tid,TheDIM.Line,TheDIM.Buff) do TheDIM_Process(TheDIM.Line);
   {
   DIM timer actions...
   }
   if SysTimer_Pulse(TimerPeriod)>0 then begin
    TheDIM_Send('@memory');
    TheDIM_Send('@errors=0');
   end;
   {
   DNS timer actions...
   }
   if TheDIM.DnsPeriod>0 then
   if mSecNow>TheDIM.DnsTimer+TheDIM.DnsPeriod then begin
    TheDIM.DnsTimer:=mSecNow;
    TheDIM_Send('@dns.exe');
   end;
   {
   Update services, if needed.
   }
   TheDIM_Poll;
   TheDIM_Stat;
  end else begin
   HostTerm('DimSrv PID '+str(task_pid(TheDIM.Tid))+' terminated with exit code '+str(task_result(TheDIM.Tid)));
   TheDIM_Stop;
  end;
  {
  DimBridge polling
  }
  if DimBridge.Count>0 then DimBridge_Polling;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid,i,j,p,ref,kind:Integer;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Messages:
   ##i   - update service which contains tag i (integer reference as decimal string) with data composed from tags
   ##i=m - update service which contains tag i (integer reference as decimal string) with mime-encoded data m
   ##s   - update service with nickname s (string) with data composed from tags
   ##s=m - update service with nickname s (string) with mime-encoded data m
  Example:
   bNul(DevPostMsg('&DimSrv ##'+str(tagButton)));
   bNul(DevPostMsg('&DimSrv ##dic_cmnd_test'));
  }
  cmd:='';
  arg:='';
  if (strFetch(Data,1)='#') and (strFetch(Data,2)='#') then begin
   p:=Pos('=',Data);
   if p>0 then begin
    cmd:=Copy(Data,3,p-3);
    arg:=mime_decode(Copy(Data,p+1,MaxLeng));
   end else begin
    cmd:=Copy(Data,3,MaxLeng);
    arg:='';
   end;
   ref:=val(cmd);
   if typetag(ref)>0 then begin
    if TheDIM.rIndex.Count>ThreshRefs then begin
     p:=TheDIM_FindRef(ref);
     while (p>0) and (p<=TheDIM.rIndex.Count) do begin
      if TheDIM.rIndex.Refs[p]=ref then begin
       i:=TheDIM.rIndex.Serv[p];
       if (i>=1) and (i<=TheDIM.Count) then begin
        kind:=TheDIM.Serv[i].Kind;
        if (kind=dis_info) or (kind=dic_cmnd) then begin
         if Length(arg)>0 then
         case typetag(ref) of
          0: ;
          1: if Length(arg)>=SizeOfInteger then bNul(iSetTag(ref,dump2i(arg)));
          2: if Length(arg)>=SizeOfReal    then bNul(rSetTag(ref,dump2r(arg)));
          3: if Length(arg)>=SizeOfChar    then bNul(sSetTag(ref,arg));
         end;
         TheDIM_Update(i,'');
        end;
       end;
       p:=p+1; 
      end else p:=0;
     end;
    end else begin
     for i:=1 to TheDIM.Count do begin
      kind:=TheDIM.Serv[i].Kind;
      if (kind=dis_info) or (kind=dic_cmnd) then
      for j:=1 to TheDIM.Serv[i].TagCount do
      if ref=TheDIM.Tags.Ref[TheDIM.Serv[i].TagIndex+j-1] then begin
       if Length(arg)>0 then
       case typetag(ref) of
        0: ;
        1: if Length(arg)>=SizeOfInteger then bNul(iSetTag(ref,dump2i(arg)));
        2: if Length(arg)>=SizeOfReal    then bNul(rSetTag(ref,dump2r(arg)));
        3: if Length(arg)>=SizeOfChar    then bNul(sSetTag(ref,arg));
       end;
       TheDIM_Update(i,'');
      end;
     end;
    end;
   end else begin
    if TheDIM.nIndex.Count>ThreshNick then begin
     p:=TheDIM_FindNick(cmd);
     while (p>0) and (p<=TheDIM.nIndex.Count) do begin
      if StrIComp(TheDIM.nIndex.Nick[p],cmd)=0 then begin
       i:=TheDIM.nIndex.Serv[p];
       if (i>=1) and (i<=TheDIM.Count) then begin
        kind:=TheDIM.Serv[i].Kind;
        if (kind=dis_info) or (kind=dic_cmnd) then TheDIM_Update(i,arg);
       end;
       p:=p+1; 
      end else p:=0;
     end;
    end else begin
     for i:=1 to TheDIM.Count do begin
      kind:=TheDIM.Serv[i].Kind;
      if (kind=dis_info) or (kind=dic_cmnd) then
      if IsSameText(cmd,TheDIM.Serv[i].Nick)
      then TheDIM_Update(i,arg);
     end;
    end;
   end;
   Data:='';
  end;
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if Length(Data)>0 then
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @dns.exe
   @dns.exe Period
   @dns.exe Period 60
   }
   if (cmdid=cmd_DnsExe) then begin
    if length(arg)>0 then begin
     if IsSameText(ExtractWord(1,arg),'Period') then begin
      if WordCount(arg)>1 then begin
       TheDIM.DnsPeriod:=iValDef(ExtractWord(2,arg),TheDIM.DnsPeriod);
       if TheDIM.DnsPeriod<0 then TheDIM.DnsPeriod:=0;
       if TheDIM.DnsPeriod>0 then TheDIM.DnsPeriod:=iMax(TheDIM.DnsPeriod,1000);
       TheDim.DnsTimer:=mSecNow*ord(TheDIM.DnsPeriod>0);
      end;
      Success(cmd+' Period '+Str(TheDIM.DnsPeriod));
     end;
    end;
    TheDIM_Send(cmd);
    Data:='';
   end else
   {
   @did.exe
   }
   if (cmdid=cmd_DidExe) then begin
    TheDIM_Send(Data);
    Data:='';
   end else
   {
   @dimtree.exe
   }
   if (cmdid=cmd_DimTreeExe) then begin
    TheDIM_Send(Data);
    Data:='';
   end else
   {
   @exit
   }
   if (cmdid=cmd_Exit) then begin
    TheDIM_Send(Data);
    Data:='';
   end else
   {
   @StatPeriod 60000
   }
   if (cmdid=cmd_StatPeriod) then begin
    TheDIM.Stat.Period:=iValDef(arg,TheDIM.Stat.Period);
    Success(cmd+' '+Str(TheDIM.Stat.Period));
    Data:='';
   end else
   {
   @DimBridge source.domain target.domain EXPORT/LIST/*
   }
   if (cmdid=cmd_DimBridge) then begin
    DimBridge_Cmd(cmd,arg);
    Data:='';
   end else
   {
   @DimPorts add enable
   }
   if (cmdid=cmd_DimPorts) then begin
    DimPorts(arg);
    Data:='';
   end else
   {
   @IfHostName alidcscom252.cern.ch @Warning It's PHOS cooling plant server!
   }
   if (cmdid = cmd_IfHostName) then begin
    if WordCount(arg)>1 then
    if IsSameText(ExtractWord(1,arg),ParamStr('HostName 1')) then begin
     Data:=SkipWords(1,arg);
     StdIn_Processor(Data);
    end;
    Data:='';
   end else
   {
   @IfNotHostName alidcscom252.cern.ch @Warning It's not PHOS cooling plant server!
   }
   if (cmdid = cmd_IfNotHostName) then begin
    if WordCount(arg)>1 then
    if not IsSameText(ExtractWord(1,arg),ParamStr('HostName 1')) then begin
     Data:=SkipWords(1,arg);
     StdIn_Processor(Data);
    end;
    Data:='';
   end else
   {
   @IfComputerName alidcscom252 @Warning It's PHOS cooling plant server!
   }
   if (cmdid = cmd_IfComputerName) then begin
    if WordCount(arg)>1 then
    if IsSameText(ExtractWord(1,arg),ParamStr('ComputerName')) then begin
     Data:=SkipWords(1,arg);
     StdIn_Processor(Data);
    end;
    Data:='';
   end else
   {
   @IfNotComputerName alidcscom252 @Warning It's not PHOS cooling plant server!
   }
   if (cmdid = cmd_IfNotComputerName) then begin
    if WordCount(arg)>1 then
    if not IsSameText(ExtractWord(1,arg),ParamStr('ComputerName')) then begin
     Data:=SkipWords(1,arg);
     StdIn_Processor(Data);
    end;
    Data:='';
   end else
   {
   @IfProcessExists Crw32.exe @Warning Process Crw32.exe exists!
   }
   if (cmdid = cmd_IfProcessExists) then begin
    if WordCount(arg)>1 then
    if PidCounter(ExtractWord(1,arg))>0 then begin
     Data:=SkipWords(1,arg);
     StdIn_Processor(Data);
    end;
    Data:='';
   end else
   {
   @IfNotProcessExists cmd.exe @Warning Process cmd.exe not exists!
   }
   if (cmdid = cmd_IfNotProcessExists) then begin
    if WordCount(arg)>1 then
    if PidCounter(ExtractWord(1,arg))=0 then begin
     Data:=SkipWords(1,arg);
     StdIn_Processor(Data);
    end;
    Data:='';
   end else
   {
   @ReportFlags 3
   }
   if (cmdid = cmd_ReportFlags) then begin
    ReportFlags:=iValDef(Trim(arg),ReportFlags);
    Success(cmd+' '+Str(ReportFlags));
    Data:='';
   end else
   {
   @FallBackMode 1
   }
   if (cmdid = cmd_FallBackMode) then begin
    bNul(iSetTag(TheDIM.FallBackModeTag,iValDef(Trim(arg),iGetTag(TheDIM.FallBackModeTag))));
    Success(cmd+' '+Str(iGetTag(TheDIM.FallBackModeTag)));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
