#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CRW-DAQ script to call on file integrity Verify.
##########################################################

function print_comment(){
 echo "****************************************************";
 echo "CRW-DAQ Verification script.  Run to verify CRW-DAQ.";
 echo "****************************************************";
};

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

readonly logfile="$CRW_DAQ_VAR_TMP_DIR/checksum.log";
readonly lang="$(echo "$CRW_DAQ_SYS_LANG" | cut -d ',' -f 1)";
readonly delay_failure=300000; #d:86400000,h:3600000,m:60000
readonly delay_success=15000;
declare -i verify_status=0;

readonly secure='secure.md5';

function fatal(){
 1>&2 echo "$2";
 exit $1;
};

function check_environ(){
 if [ -z "$CRW_DAQ_VAR_TMP_DIR" ]; then
  fatal 1 "Error: script should be called from CRW-DAQ system only.";
 fi;
};

function show_tooltip(){
 unix tooltip-notifier "$@";
};

function cmd_pid_running(){
 pgrep -lx "$1" | grep "$2\s\s*$1";
};

function syslog_success_cmnd(){ echo "@silent @syslog INFO/SUCCESS System - $1"; };
function syslog_failure_cmnd(){ echo "@silent @syslog ERROR/ALARM  System - $1"; };

function syslog_success_check(){
 syslog_success_cmnd "______________";
 syslog_success_cmnd "Session Check.";
 syslog_success_cmnd "^^^^^^^^^^^^^^";
 syslog_success_cmnd "$1";
};

function syslog_failure_check(){
 syslog_failure_cmnd "______________";
 syslog_failure_cmnd "Session Check.";
 syslog_failure_cmnd "^^^^^^^^^^^^^^";
 syslog_failure_cmnd "$1";
};

function report_success(){
 show_tooltip text "$1" preset stdSuccess delay delay_success btn1 "$2" cmd1 "unix grun lister $logfile";
 if cmd_pid_running crwdaq $CRW_DAQ_SYS_EXE_PID; then syslog_success_check "$1" | unix send2crwdaq; fi;
};

function report_failure(){
 show_tooltip text "$1" preset stdSiren   delay delay_failure btn1 "$2" cmd1 "unix grun lister $logfile";
 if cmd_pid_running crwdaq $CRW_DAQ_SYS_EXE_PID; then syslog_failure_check "$1" | unix send2crwdaq; fi;
};

function report_status(){
 if [ "$lang" = "RUSSIAN" ]; then
  if [ $verify_status -eq 0 ]; then report_success "$CRW_DAQ_SYS_TITLE: Целостность в норме."  "см. Журнал"; fi;
  if [ $verify_status -eq 1 ]; then report_failure "$CRW_DAQ_SYS_TITLE: Целостность НАРУШЕНА." "см. Журнал"; fi;
 else
  if [ $verify_status -eq 0 ]; then report_success "$CRW_DAQ_SYS_TITLE: Integrity Check Succeeded." "view Log"; fi;
  if [ $verify_status -eq 1 ]; then report_failure "$CRW_DAQ_SYS_TITLE: Integrity Check is FAILED." "view Log"; fi;
 fi;
};

function verify_checksum(){
 if pushd $scriptHOME/.. >/dev/null 2>&1; then
  if unix checksum -c$secure > $logfile; then
   let verify_status=0;
   report_status;
   cat $logfile;
  else
   let verify_status=1;
   report_status;
   cat $logfile;
  fi;
 fi;
 popd >/dev/null 2>&1;
};

function main(){
 print_comment;
 check_environ;
 verify_checksum;
};

main "$@";

#############
# END OF FILE
#############
