#!/bin/bash

##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CRW-DAQ script to call on Logout.
##########################################################

function print_comment(){
 echo "***************************************************";
 echo "CRW-DAQ Finalization script. Run on CRW-DAQ logout.";
 echo "***************************************************";
};

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

readonly logfile="$CRW_DAQ_VAR_TMP_DIR/$scriptbase.log";

function fatal(){
 1>&2 echo "$2";
 exit $1;
};

function check_environ(){
 if [ -z "$CRW_DAQ_VAR_TMP_DIR" ]; then
  fatal 1 "Error: script should be called from CRW-DAQ system only.";
 fi;
};

function enter_logfile(){
 echo ""                                              >> $logfile;
 echo "$(date +%Y.%m.%d-%H:%M:%S): Enter $scriptfile" >> $logfile;
};

function leave_logfile(){
 echo "$(date +%Y.%m.%d-%H:%M:%S): Leave $scriptfile" >> $logfile;
 echo ""                                              >> $logfile;
};

function get_file_size(){
 if [ -e "$1" ]; then
  stat -c %s "$1";
 else
  echo "0";
 fi;
};

function cleanup_dirs(){
 let KB=1024; let MB=$KB*$KB; let maxsize=32*$MB;
 echo "$(date +%Y.%m.%d-%H:%M:%S): Cleanup dirs: $maxsize $*";
 for dir in $*; do
  if [ -d "$dir" ]; then
   for fn in $(ls -1 "$dir"); do
    local file="$dir/$fn";
    if [ -f "$file" ]; then
     if [ "$file" = "$logfile" ]; then continue; fi;
     local size="$(get_file_size $file)";
     if [ $size -gt $maxsize ]; then
      old="$dir/obsolete";
      mkdir -p "$old";
      mv -fv "$file" "$old/$fn";
     fi;
    fi;
   done;
  fi;
 done;
};

function cleanup_files(){
 $scriptHOME/crwdaqlogroll.sh; return; # skip obsolete log files cleaner
 cleanup_dirs "$CRW_DAQ_VAR_TMP_DIR" "$CRW_DAQ_SYS_TMP_DIR" >> $logfile;
};

function chk_integrity(){
 $scriptHOME/crwdaq.verify.sh;
};

##########################
# perform required actions
##########################
function perform_actions(){
 print_comment;
 cleanup_files;
 chk_integrity;
};

################
# main function.
################
function main(){
 check_environ;
 enter_logfile;
 perform_actions | tee -a $logfile;
 leave_logfile;
};

main "$@";

#############
# END OF FILE
#############
