 //////////////////////////////////////////////////////////////////////////////
 // CRW32 project: L-CARD E14-140 driver for CRW-DAQ
 // Copyright (C) by Kuryakin Alexey, Sarov, Russia, 2010, <kouriakine@mail.ru>
 // Last release: 20101112 by Kuruakin Alexey <kouriakine@mail.ru>
 //////////////////////////////////////////////////////////////////////////////
program E140_DRV;
{$APPTYPE CONSOLE} // ! Declare application type as CONSOLE.
{$I _sysdef}       // ! By CRW conventions, include _SYSDEF,
uses               // ! ShareMem must to be FIRST USES UNIT,
 ShareMem,         // ! borlndmm.dll should present in path.
 SysUtils, Windows, Math, Classes, Forms,
 _alloc, _str, _dynar, _fpu, _fio, _rtc, _polling, _task,
 _mime, _fifo, _sharm, _ascio, _az, Lusbapi, E140_SHM, E140_API;

 //////////////////////////////////
 // General variables and constants
 //////////////////////////////////
const
 E140       : TE140   = nil;    // E140 driver instance
 Terminated : Boolean = False;  // Program should be terminated ?
 Simulation : Boolean = False;  // Simulation mode ?
 SimTimer   : Double  = 0;      // Simulation timer
 SimBlocks  : Int64   = 0;      // Blocks counter
 SimSamples : Int64   = 0;      // Sample counter
 SimAdcZero : array[0..31] of Integer  = (0,     250,  500,  750, 1000, 1250, 1500, 1750,
                                          2000, 2250, 2500, 2750, 3000, 3250, 3500, 3750,
                                          4000, 4250, 4500, 4750, 5000, 5250, 5500, 5750,
                                          6000, 6250, 6500, 6750, 7000, 7250, 7500, 7750);
 SimAdcAmpl : array[0..31] of Integer  = ( 200,  200,  200,  200,  200,  200,  200,  200,
                                           200,  200,  200,  200,  200,  200,  200,  200,
                                           200,  200,  200,  200,  200,  200,  200,  200,
                                           200,  200,  200,  200,  200,  200,  200,  200);
 SimAdcNois : array[0..31] of Integer  = (  20,   20,   20,   20,   20,   20,   20,   20,
                                            20,   20,   20,   20,   20,   20,   20,   20,
                                            20,   20,   20,   20,   20,   20,   20,   20,
                                            20,   20,   20,   20,   20,   20,   20,   20);

 ////////////////////
 // Utility functions
 ////////////////////
 
 //
 // Send message to StdOut stream.
 //
procedure E140_Reporter(const s:LongString);
begin
 StdOut.Put:=s;
end;

 //
 // Construct unique name for shared memory.
 // Uses program name and process ID.
 //
function GetShareMemName:LongString;
begin
 Result:=Format('%s/%d',[ExtractFileNameExt(ProgName),GetCurrentProcessId]);
end;

 //
 // Report on exceptions.
 //
procedure BugReport(const E:Exception; const ErrorMsg:LongString='');
begin
 if Assigned(E)
 then StdOut.Put:=Format('@E140.OnException=%s,%s',[E.ClassName,E.Message])
 else StdOut.Put:=Format('@E140.OnError=%s',[ErrorMsg]);
 LockedInc(StdIoErrorCount);
end;

 //
 // Return string with first word skipped.
 //
function SkipWord(const S:LongString; const ScanSpaces:TCharSet):LongString;
var i,k:Integer;
begin
 k:=1;
 for i:=k to Length(S) do if S[i] in ScanSpaces then Inc(k) else Break;
 for i:=k to Length(S) do if S[i] in ScanSpaces then Break  else Inc(k);
 for i:=k to Length(S) do if S[i] in ScanSpaces then Inc(k) else Break;
 Result:=Copy(S,k,Length(S)-k+1);  
end;

 //
 // Convert String to Double or return default value.
 //
function StrToDblDef(const s:String; def:Double):Double;
begin
 if not Str2Real(s,Result) then Result:=def;
end;

 //
 // Extract E140 ADC parameters from string list.
 // Return -1:Failure, invalid parameters;
 //         0:No parameters specified, use defaults;
 //         1:Ok, correct parameters specified.
 // arg may be one of two next modes (Test,Work):
 // arg = Test,cq,gn,ar                         - Test mode
 // arg = cs,ec,im,st,sm,sc,sp,ar,ik,cq,ct[cq]  - Work mode
 // where cs - ClkSource
 //       ec - EnableClkOutput
 //       im - InputMode
 //       st - SynchroAdType
 //       sm - SynchroAdMode
 //       sc - SynchroAdChannel
 //       sp - SynchroAdPorog
 //       ar - AdcRate
 //       ik - InterKadrDelay
 //       cq - ChannelsQuantity
 //       ct - list of ControlTable, ChannelsQuantity length
 //            each ct[i] = (Channel and 31) or (Gain shl 6)
 //       gn - ADC gain for Test mode 
 // In Test mode uses 0..cq-1 channels with same gain gn.
 // See E14-140 manual for details.
 //
function ExtractAdcParams(var Adc:ADC_PARS_E140; args:String):Integer;
var i:Integer;
begin
 Result:=0;
 if WordCount(args,ScanSpaces)=0 then Exit;
 if IsSameText(ExtractWord(1,args,ScanSpaces),'Test') then begin
  Adc:=TestE140AdcPars(StrToIntDef(ExtractWord(2,args,ScanSpaces),1),
                       StrToIntDef(ExtractWord(3,args,ScanSpaces),ADC_INPUT_RANGE_2500mV_E140),
                       StrToDblDef(ExtractWord(4,args,ScanSpaces),100))^;
  Result:=1;
  Exit;
 end;
 Adc.ClkSource        := StrToIntDef( ExtractWord(1,args,ScanSpaces), INT_ADC_CLOCK_E140 );
 Adc.EnableClkOutput  := StrToIntDef( ExtractWord(2,args,ScanSpaces), ADC_CLOCK_TRANS_DISABLED_E140 );
 Adc.InputMode        := StrToIntDef( ExtractWord(3,args,ScanSpaces), NO_SYNC_E140 );
 Adc.SynchroAdType    := StrToIntDef( ExtractWord(4,args,ScanSpaces), 0 );
 Adc.SynchroAdMode    := StrToIntDef( ExtractWord(5,args,ScanSpaces), 0 );
 Adc.SynchroAdChannel := StrToIntDef( ExtractWord(6,args,ScanSpaces), 0 );
 Adc.SynchroAdPorog   := StrToIntDef( ExtractWord(7,args,ScanSpaces), 0 );
 Adc.AdcRate          := StrToDblDef( ExtractWord(8,args,ScanSpaces), 1 );	  			  						// тактовая частота АЦП в кГц
 Adc.InterKadrDelay   := StrToDblDef( ExtractWord(9,args,ScanSpaces), 0 );
 Adc.ChannelsQuantity := StrToIntDef( ExtractWord(10,args,ScanSpaces), 1 );
 for i:=0 to Min(Max(0,Adc.ChannelsQuantity),MAX_CONTROL_TABLE_LENGTH_E140)-1 do
 Adc.ControlTable[i]  := StrToIntDef( ExtractWord(11+i,args,ScanSpaces), i and 31 );
 Result:=-1;
 if not (Adc.EnableClkOutput in [ADC_CLOCK_TRANS_DISABLED_E140..ADC_CLOCK_TRANS_ENABLED_E140]) then Exit;
 if not (Adc.ChannelsQuantity in [1..MAX_CONTROL_TABLE_LENGTH_E140]) then Exit;
 if not (Adc.ClkSource in [INT_ADC_CLOCK_E140..EXT_ADC_CLOCK_E140]) then Exit;
 if not (Adc.InputMode in [NO_SYNC_E140..ANALOG_SYNC_E140]) then Exit;
 if WordCount(args,ScanSpaces)<10+Adc.ChannelsQuantity then Exit;
 if (Adc.AdcRate<=0) or (Adc.AdcRate>400) then Exit;
 Result:=1;  
end;

 //
 // Simulate E140 data acquisition without real hardware.
 // Uses for test purposes.
 //
procedure SimulationPolling;
const BlockPeriod=50; WavePeriod=20;
var ms,data:Double; i,imod,ctab,chan,gain,code,nWords,nRequest,Index,Offset,Wave:Integer;
begin
 if Simulation then
 if not Terminated then
 if SimTimer>0 then begin
  ms:=mSecNow;
  if ms-SimTimer>BlockPeriod then begin
   nWords:=AdjustBufferSize(Round(BlockPeriod*E140.SharedBuff.AdcActual.AdcRate),E140BuffAdjust);
   nRequest:=SimBlocks mod E140NumBuffers;
   Index:=nRequest*nWords;
   Offset:=SizeOf(E140.SharedBuff.Header)+Index*SizeOf(E140.SharedBuff.Buffer[0]);
   for i:=0 to nWords-1 do begin
    imod:=i mod E140.SharedBuff.AdcActual.ChannelsQuantity;
    ctab:=E140.SharedBuff.AdcActual.ControlTable[imod];
    chan:=ctab and 31;
    gain:=ctab shr 6;
    wave:=Max(1,Round(WavePeriod*E140.SharedBuff.AdcActual.AdcRate));
    data:=SimAdcZero[chan];
    if SimAdcAmpl[chan]>0 then data:=data+SimAdcAmpl[Chan]*((SimSamples+i) mod Wave)/Wave;
    if SimAdcNois[chan]>0 then data:=data+Random(SimAdcNois[chan])-0.5*SimAdcNois[chan];
    code:=Round(Max(-8192,Min(8191,data*(1 shl (gain*2)))));  
    E140.SharedBuff.Buffer[Index+i]:=code;
   end;
   StdOut.Put:=Format('@E140.OnReadout=%s,%d,%d,%d,%d,%d,%d',
                [E140.SharedName, nWords, Offset, Index, nRequest, SimBlocks, SimSamples]);
   Inc(SimSamples,nWords);
   Inc(SimBlocks);
   SimTimer:=ms;
  end;
 end;
end;

 ///////////////////////////////////////////
 // Incoming request handling implementation
 ///////////////////////////////////////////
 //
 // Request: @exit
 // Request: @exit=n
 // Reply:   @exit=n
 // Comment: Terminate program with exit code n.
 //
 procedure DoExit(const cmnd,args:LongString);
 begin
  try
   Terminated:=True;
   System.ExitCode:=StrToIntDef(args,0);
   StdOut.Put:=Format('%s=%d',[cmnd,System.ExitCode]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @errors
 // Request: @errors=n
 // Reply:   @errors=n
 // Comment: Return value n of error counter.
 //
 procedure DoErrors(const cmnd,args:LongString);
 var
  n : LongInt;
 begin
  try
   if Str2Long(args,n)
   then n:=LockedExchange(StdIoErrorCount,n)
   else n:=StdIoErrorCount;
   StdOut.Put:=Format('%s=%d',[cmnd,n]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @memory
 // Reply:   @memory=n
 // Comment: Return AllocMemSize, i.e. memory usage counter.
 //
 procedure DoMemory(const cmnd,args:LongString);
 begin
  try
   StdOut.Put:=Format('%s=%d',[cmnd,GetAllocMemSize]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @ProcessPriority
 // Request: @ProcessPriority=n
 // Reply:   @ProcessPriority=n
 // Comment: Set process priority class.
 //
 procedure DoProcessPriority(const cmnd,args:LongString);
 var p:DWORD;
 begin
  try
   if not IsEmptyStr(args) then begin
    p:=GetPriorityClassByName(args);
    if p>0 then SetPriorityClass(GetCurrentProcess,p);
   end;
   StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityClassName(GetPriorityClass(GetCurrentProcess))]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @ThreadPriority
 // Request: @ThreadPriority=n
 // Reply:   @ThreadPriority=n
 // Comment: Set thread priority class.
 //
 procedure DoThreadPriority(const cmnd,args:LongString);
 var p:TThreadPriority;
 begin
  try
   if not IsEmptyStr(args) then begin
    p:=GetPriorityByName(args);
    StdIn.Priority:=p;
    StdOut.Priority:=p;
    case p of
     tpIdle         : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_IDLE);
     tpLowest       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_LOWEST);
     tpLower        : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_BELOW_NORMAL);
     tpNormal       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
     tpHigher       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_ABOVE_NORMAL);
     tpHighest      : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_HIGHEST);
     tpTimeCritical : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
    end;
   end;
   StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityName(StdIn.Priority)]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.Simulation
 // Request: @E140.Simulation=m
 // Reply:   @E140.Simulation=n
 // Comment: Open device and prepare for data acquisition.
 //          Return n=0/1 - not/opened.
 //
 procedure DoE140Simulation(const cmnd,args:LongString);
 var m:Integer;
 begin
  try
   if Str2Int(ExtractWord(1,args,ScanSpaces),m) then begin
    Simulation:=(m>0);
    if Simulation then begin
     if E140.Started>0 then E140.Stop;
     if E140.Opened then E140.Close;
     SimSamples:=0;
     SimBlocks:=0;
     SimTimer:=0;
    end;
   end;
   StdOut.Put:=Format('%s=%d',[cmnd,Ord(Simulation)]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.SimulateAdc=i,z,a,n
 // Reply:   @E140.SimulateAdc=i,z,a,n
 // Comment: Set simulation zero z, amplitude a, noise n for channel i.
 //
 procedure DoE140SimulateAdc(const cmnd,args:LongString);
 var i,z,a,n:Integer;
 begin
  try
   i:=StrToIntDef(ExtractWord(1,args,ScanSpaces),-1);
   if i in [Low(SimAdcZero)..High(SimAdcZero)] then begin
    z:=StrToIntDef(ExtractWord(2,args,ScanSpaces),SimAdcZero[i]);
    a:=StrToIntDef(ExtractWord(3,args,ScanSpaces),SimAdcAmpl[i]);
    n:=StrToIntDef(ExtractWord(4,args,ScanSpaces),SimAdcNois[i]);
    SimAdcZero[i]:=z; SimAdcAmpl[i]:=a; SimAdcNois[i]:=n;
    StdOut.Put:=Format('%s=%d,%d,%d,%d',[cmnd,i,z,a,n]);
   end;
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.SerNum=sernum
 // Reply:   @E140.SerNum=sernum
 // Comment: Set/get device serial number to open.
 //
 procedure DoE140SerNum(const cmnd,args:LongString);
 begin
  try
   if Length(args)>0 then E140.SerNum:=Trim(args);
   StdOut.Put:=Format('%s=%s',[cmnd,E140.SerNum]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.Open=cs,ec,im,st,sm,sc,sp,ar,ik,cq,ct[cq]
 // Reply:   @E140.Open=n
 // Comment: Open device and prepare for data acquisition.
 //          Return n=0/1 - not/opened.
 //
 procedure DoE140Open(const cmnd,args:LongString);
 var Adc:ADC_PARS_E140;
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    case ExtractAdcParams(Adc,args) of
     -1: RAISE E140_FAILURE.Create('Invalid ADC parameters in '+args);
      0: Adc:=TestE140AdcPars(4,ADC_INPUT_RANGE_2500mV_E140,100.0)^;
     +1: ;
    end;
    E140.SharedBuff.AdcInited:=Adc;
    E140.SharedBuff.AdcWanted:=Adc;
    E140.SharedBuff.AdcActual:=Adc;
    SimSamples:=0;
    SimBlocks:=0;
    SimTimer:=0;
    StdOut.Put:=Format('%s=%d',[cmnd,1]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   case ExtractAdcParams(Adc,args) of
    -1: BugReport(NIL,'Invalid ADC parameters in '+args);
     0: E140.Open;
    +1: E140.Open(@Adc);
   end;
   StdOut.Put:=Format('%s=%d',[cmnd,Ord(E140.Opened)]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.Start=cs,ec,im,st,sm,sc,sp,ar,ik,cq,ct[cq]
 // Reply:   @E140.Start=t
 // Comment: Start data acquisition.
 //          Return t=0 - not started,
 //                 t>0 - base time when started, ms.
 //
 procedure DoE140Start(const cmnd,args:LongString);
 var Adc:ADC_PARS_E140;
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    case ExtractAdcParams(Adc,args) of
     -1: RAISE E140_FAILURE.Create('Invalid ADC parameters in '+args);
      0: Adc:=TestE140AdcPars(4,ADC_INPUT_RANGE_2500mV_E140,100.0)^;
     +1: ;
    end;
    Adc.KadrRate:=Adc.AdcRate/Adc.ChannelsQuantity;
    E140.SharedBuff.AdcInited:=Adc;
    E140.SharedBuff.AdcWanted:=Adc;
    E140.SharedBuff.AdcActual:=Adc;
    SimTimer:=mSecNow;
    SimSamples:=0;
    SimBlocks:=0;
    StdOut.Put:=Format('%s=%d',[cmnd,1]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   case ExtractAdcParams(Adc,args) of
    -1: BugReport(NIL,'Invalid ADC parameters in '+args);
     0: E140.Start;
    +1: E140.Start(@Adc);
   end;
   StdOut.Put:=Format('%s=%g',[cmnd,E140.Started]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.Stop
 // Reply:   @E140.Stop=t
 // Comment: Stop data acquisition.
 //          Return t=0 - was not started, t>0 - base time when last time started.
 //
 procedure DoE140Stop(const cmnd,args:LongString);
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    SimSamples:=0;
    SimBlocks:=0;
    SimTimer:=0;
    StdOut.Put:=Format('%s=%d',[cmnd,1]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   StdOut.Put:=Format('%s=%g',[cmnd,E140.Started]);
   if E140.Started>0 then E140.Stop;
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.Close
 // Reply:   @E140.Close=n
 // Comment: Close device.
 //          Return n=0/1 - not/closed device.
 //
 procedure DoE140Close(const cmnd,args:LongString);
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    SimSamples:=0;
    SimBlocks:=0;
    SimTimer:=0;
    StdOut.Put:=Format('%s=%d',[cmnd,1]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   StdOut.Put:=Format('%s=%d',[cmnd,Ord(E140.Opened)]);
   if E140.Opened then E140.Close;
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.DI
 // Reply:   @E140.DI=n,di
 // Comment: Read digital input.
 //          Return n=0/1 - not/read data di.
 //
 procedure DoE140DI(const cmnd,args:LongString);
 var n,di:Word;
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    StdOut.Put:=Format('%s=%d,$%4.4x',[cmnd,1,$00FF]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   n:=Ord(E140.TTL_IN(@di));
   StdOut.Put:=Format('%s=%d,$%4.4x',[cmnd,n,di]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.EDO=m
 // Reply:   @E140.EDO=n
 // Comment: m=1/0 - Enabe/disable digital output.
 //          Return n=0/1 - Error/Ok status.
 //
 procedure DoE140EDO(const cmnd,args:LongString);
 var n,m:Integer;
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    StdOut.Put:=Format('%s=%d',[cmnd,1]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   if Str2Int(ExtractWord(1,args,ScanSpaces),m)
   then n:=Ord(E140.ENABLE_TTL_OUT(m<>0))
   else n:=0;
   StdOut.Put:=Format('%s=%d',[cmnd,n]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.DO=m
 // Reply:   @E140.DO=n
 // Comment: Write m to digital output.
 //          Return n=0/1 - not/write data m.
 //
 procedure DoE140DO(const cmnd,args:LongString);
 var n,m:Integer;
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    StdOut.Put:=Format('%s=%d',[cmnd,1]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   if Str2Int(ExtractWord(1,args,ScanSpaces),m)
   then n:=Ord(E140.TTL_OUT(m))
   else n:=0;
   StdOut.Put:=Format('%s=%d',[cmnd,n]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @E140.AO=i,m
 // Reply:   @E140.AO=n
 // Comment: Write m to analog output number i.
 //          Return n=0/1 - not/write data m.
 //
 procedure DoE140AO(const cmnd,args:LongString);
 var i,n,m:Integer; sm:SHORT;
 begin
  try
   //
   // Simulation mode...
   //
   if Simulation then begin
    StdOut.Put:=Format('%s=%d',[cmnd,1]);
    Exit;
   end;
   //
   // Real measurement mode...
   //
   if Str2Int(ExtractWord(1,args,ScanSpaces),i) and
      Str2Int(ExtractWord(2,args,ScanSpaces),m)
   then begin sm:=m; n:=Ord(E140.DAC_SAMPLE(@sm,i)); end
   else n:=0;
   StdOut.Put:=Format('%s=%d',[cmnd,n]);
  except
   on E:Exception do BugReport(E);
  end;
 end;
 //
 // Request: @help
 // Reply:   help info
 // Comment: Show help.
 //
 procedure DoHelp(const cmnd,args:LongString);
 begin
  StdOut.Put:='>> Command list:';
  StdOut.Put:='>> @Help                       This help.';
  StdOut.Put:='>> @Exit=n                     Exit program with code n.';
  StdOut.Put:='>> @Errors                     Return error counter.';
  StdOut.Put:='>> @Memory                     Return memory status.';
  StdOut.Put:='>> @ProcessPriority=n          Set process priority:Idle/Low/Normal/High/RealTime.';
  StdOut.Put:='>> @ThreadPriority=n           Set thread priority:tpIdle/tpLow/tpNormal/tpHigh/tpTimeCritical.';
  StdOut.Put:='>> @E140.SimulateAdc=i,z,a,n   Simulate ADC chan i, zero z, ampl a, noise n.';
  StdOut.Put:='>> @E140.Simulation=n          1/0 = Start/stop simulation mode.';
  StdOut.Put:='>> @E140.SerNum=sernum         Set device serial number to identify.';
  StdOut.Put:='>> @E140.Open=arglist          Open device and prepare parameters.';
  StdOut.Put:='>> @E140.Start=arglist         Start measure.';
  StdOut.Put:='>> @E140.Stop                  Stop measure.';
  StdOut.Put:='>> @E140.Close                 Close device.';
  StdOut.Put:='>> @E140.DI                    Read DI.';
  StdOut.Put:='>> @E140.EDO=1/0               Enable/disable DO.';
  StdOut.Put:='>> @E140.DO=m                  Write m to DO.';
  StdOut.Put:='>> @E140.AO=i,m                Write code m to AO[i],i=0/1.';
 end;
 //
 // This callback handles unrecognized commands.
 //
 procedure DoSpecificCommands(const args:LongString);
 begin
  if Length(args)>0 then BugReport(NIL,'Unrecognized command '+args);
 end;
 //
 // Application specific initialization.
 //
 procedure SpecificInitialization;
 begin
  Application.Initialize;
  SystemEchoProcedure:=StdOutEcho;
  StdOut.Put:='E140 driver for CRW-DAQ:';
  StdOut.Put:='© by Alexey Kuryakin,2010.11.12';
  StdOut.Put:='Russia,Sarov,kouriakine@mail.ru';
  StdOut.Put:='Type @help to get Help.';
  // 
  // Register user commands coming from StdIn.
  //
  StdIn.SpecHandler:=DoSpecificCommands;
  StdIn.AddCommand('@Help',                          DoHelp);
  StdIn.AddCommand('@Exit',                          DoExit);
  StdIn.AddCommand('@Errors',                        DoErrors);
  StdIn.AddCommand('@Memory',                        DoMemory);
  StdIn.AddCommand('@ProcessPriority',               DoProcessPriority);
  StdIn.AddCommand('@ThreadPriority',                DoThreadPriority);
  StdIn.AddCommand('@E140.Help',                     DoHelp);
  StdIn.AddCommand('@E140.Exit',                     DoExit);
  StdIn.AddCommand('@E140.Errors',                   DoErrors);
  StdIn.AddCommand('@E140.Memory',                   DoMemory);
  StdIn.AddCommand('@E140.ProcessPriority',          DoProcessPriority);
  StdIn.AddCommand('@E140.ThreadPriority',           DoThreadPriority);
  StdIn.AddCommand('@E140.SimulateAdc',              DoE140SimulateAdc);
  StdIn.AddCommand('@E140.Simulation',               DoE140Simulation);
  StdIn.AddCommand('@E140.SerNum',                   DoE140SerNum);
  StdIn.AddCommand('@E140.Open',                     DoE140Open);
  StdIn.AddCommand('@E140.Close',                    DoE140Close);
  StdIn.AddCommand('@E140.Start',                    DoE140Start);
  StdIn.AddCommand('@E140.Stop',                     DoE140Stop);
  StdIn.AddCommand('@E140.EDO',                      DoE140EDO);
  StdIn.AddCommand('@E140.DI',                       DoE140DI);
  StdIn.AddCommand('@E140.DO',                       DoE140DO);
  StdIn.AddCommand('@E140.AO',                       DoE140AO);
  //
  // Create instance of E140 device
  //
  E140:=NewE140(GetShareMemName,E140_Reporter,nil,False,False);
  E140.Master:=E140;
 end;
 //
 // Application specific finalization.
 //
 procedure SpecificFinalization;
 begin
  if E140.Ok then begin
   if E140.Started>0 then E140.Stop;
   if E140.Opened then E140.Close;
   E140.Free;
  end;
  Kill(E140);
  Application.Terminate;
 end;
 //
 // Application specific polling.
 //
 procedure SpecificPolling;
 begin
  if Simulation
  then SimulationPolling
  else E140.Polling;
  Application.ProcessMessages;
  if BecameZombie(FILE_TYPE_PIPE,1000) then StdIn.Put:='@Exit';
 end;
 //
 // Main program
 //
begin
 try
  try
   SpecificInitialization;
   while not Terminated do begin
    while StdIn.Count>0 do StdIn.Process(StdIn.Get);
    SpecificPolling;
    Sleep(1);
   end;
  finally
   SpecificFinalization;
  end;
 except
  on E:Exception do begin
   StdOut.Put:=E.Message;
   ExitCode:=2;
  end;
 end;
 Sleep(100);
 if BecameZombie(FILE_TYPE_PIPE) then ExitProcess(1);
end.
