#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

function fatal(){
 local date="$(date +%Y.%m.%d-%H:%M:%S)"; local script="$(basename ${BASH_SOURCE[0]})";
 if which notify-send; then notify-send -t 30000 -i dialog-error "$date: $script" "$1"; fi;
 1>&2 echo "-ne ";"\n$1\n\n";
 exit 1;
};

# Script identification.
########################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

function uniheat_main(){
 if [ -z "$1" ]; then return 1; fi;
 do_uniheat_sim  $1 > "$scriptHOME/duh_simulator.cfg";
 do_uniheat_tag  $1 > "$scriptHOME/duh_tags.cfg";
 do_uniheat_sel  $1 > "$scriptHOME/uniheat.cfg";
 do_duh_main_gui $1 > "$scriptHOME/duh_main_gui.cfg";
 do_duh_datsrv   $1 > "$scriptHOME/duh_datsrv.cfg";
};

function do_uniheat_sim(){
 do_generalmap $1;
 do_link_ai $1;
 do_link_ao $1;
 do_link_di $1;
 do_sim_text $1;
 do_end_sim_text;
};

function do_generalmap(){
 echo ";********************************************";
 echo "; Симулятор нагревателей для проверки UniHeat";
 echo ";********************************************";
 echo "[DeviceList]";
 echo "&DUH.SIM = device software script";
 echo "[&DUH.SIM]";
 echo "Comment       = UniHeat simulation";
 echo "InquiryPeriod = 1";
 echo "DevicePolling = 100, tpNormal";
 echo "ScriptSection = &DUH.SIM.TEXT";
};
function do_link_ai(){
 echo "AnalogInputs  = $1";
 for n in $(seq 1 $1); do do_link_ais $n; done;
};
function do_link_ais(){
 let "ai=$1-1";
 echo "Link AnalogInput   $ai with curve DUH.SIM.T$1";
};
function do_link_ao(){
 let "numao=$1*2";
 echo "AnalogOutputs = $numao";
 for n in $(seq 1 $1); do do_link_aos_t $n; done;
 for n in $(seq 1 $1); do do_link_aos_p $n $1; done;
};
function do_link_aos_t(){
 let "ao=$1-1";
 echo "Link AnalogOutput $ao with curve DUH.SIM.T$1  history 10000";
};
function do_link_aos_p(){
 let "ao=$1-1+$2";
 echo "Link AnalogOutput $ao with curve DUH.SIM.P$1  history 10000";
};
function do_link_di(){
 let "numdi=$1*2";
 echo "DigitalInputs = $numdi";
 for n in $(seq 1 $1); do do_link_dis_blk $n; done;
 for n in $(seq 1 $1); do do_link_dis_gat $n $1; done;
};
function do_link_dis_blk(){
 let "di=$1-1";
 echo "Link DigitalInput $di with curve UH_BLK$1  inverted bit 0 ; BS$1  POWER";
};
function do_link_dis_gat(){
 let "di=$1-1+$2";
 echo "Link DigitalInput $di with curve UH_GATE$1          bit 0 ; BS$1  SSR";
};
function do_sim_text(){
 echo "[]";
 echo "[&DUH.SIM.TEXT]";
 echo "var runcount";
 echo "runcount=runcount+1";
 echo "Noise=0.0";
 echo "RoomTemp=20";
 echo "HeatRate=1e-1";
 echo "CoolRate=1e-4";
 echo "n=NumAis()";
 echo "i=0";
 echo "mas=vnew(n)";
 do_sim_text_var $1;
 for n in $(seq 1 $1); do do_sim_text_tags_read $n; done;
};
function do_sim_text_var(){
 case $1 in
  8)  do_vars8; ;;
  10) do_vars10; ;;
  16) do_vars16; ;;
  20) do_vars20; ;;
  24) do_vars24; ;;
  30) do_vars30; ;;
  32) do_vars32; ;;
  *)  ;;
 esac;
};
function do_vars8(){
 echo "var tagMode1,  tagMode2,  tagMode3,  tagMode4";
 echo "var tagMode5,  tagMode6,  tagMode7,  tagMode8";
};
function do_vars10(){
 echo "var tagMode1,  tagMode2,  tagMode3,  tagMode4,  tagMode5";
 echo "var tagMode6,  tagMode7,  tagMode8,  tagMode9,  tagMode10";
};
function do_vars16(){
 do_vars8;
 echo "var tagMode9,  tagMode10, tagMode11, tagMode12";
 echo "var tagMode13, tagMode14, tagMode15, tagMode16";
};
function do_vars20(){
 do_vars10;
 echo "var tagMode11, tagMode12, tagMode13, tagMode14, tagMode15";
 echo "var tagMode16, tagMode17, tagMode18, tagMode19, tagMode20";
};
function do_vars24(){
 do_vars16;
 echo "var tagMode17, tagMode18, tagMode19, tagMode20";
 echo "var tagMode21, tagMode22, tagMode23, tagMode24";
};
function do_vars30(){
 do_vars20;
 echo "var tagMode21, tagMode22, tagMode23, tagMode24, tagMode25";
 echo "var tagMode26, tagMode27, tagMode28, tagMode29, tagMode30";
};
function do_vars32(){
 do_vars24;
 echo "var tagMode25, tagMode26, tagMode27, tagMode28";
 echo "var tagMode29, tagMode30, tagMode31, tagMode32";
};

function do_sim_text_tags_read(){
 let "num=$1-1";
 echo "if not(tagMode$1) then tagMode$1=@findtag UH_MODE$1";
 echo "mode=iGetTag(tagMode$1)";
 echo "vput(mas,$num,mode)";
};

function do_end_sim_text(){
echo "Loop:";
echo " j=i+n";
echo " pow=eq(getdi_yn(i),0)";
echo " ssr=ne(getdi_yn(j),0)";
echo " temp=getai_yn(i)+RoomTemp*eq(runcount,1)";
echo " if eq(vget(mas,i),0) then temp=temp+pow*ssr*HeatRate-(temp-RoomTemp)*CoolRate";
echo " if eq(vget(mas,i),1) then temp=temp-pow*ssr*HeatRate-(temp-RoomTemp)*CoolRate";
echo " temp=temp+Noise*random(-1,1)";
echo " pres=temp/2";
echo " putao(i, time(), temp)";
echo " putao(j, time(), pres)";
echo " i=i+1";
echo "if lt(i,n) then goto Loop";
echo "vfree(mas)";
echo "[]";
echo "";
echo ";****************************************************************";
echo ";*** Включить речевой синтезатор для UniHeat                  ***";
echo ";****************************************************************";
echo "[&UH_CTRL]";
echo "UsesSpeakSrv = 1";
echo "[]";
echo "";
echo ";****************************************************************";
echo ";*** Поместить основные параметры печей на главную мнемосхему ***";
echo ";****************************************************************";
echo "[DUH.MAIN.GUI]";
echo "Link sensor   UH_BT1     with   tag   UH_ST1    device &UH_CTRL";
echo "Link sensor   UH_TGOAL1  with   tag   UH_TGOAL1 device &UH_CTRL";
echo "Link sensor   UH_TREAL1  with   curve UH_TREAL1 device &UH_CTRL";
echo "Link sensor   UH_GATE1   with   curve UH_GATE1  device &UH_CTRL";
echo "Link sensor   UH_PROP1   with                   device &UH_CTRL";
echo "Link sensor   UH_TNAME1  with   tag   UH_TNAME1 device &UH_CTRL";
echo "Link sensor   UH_PNAME1  with   tag   UH_PNAME1 device &UH_CTRL";
echo "Link sensor   UH_BT2     with   tag   UH_ST2    device &UH_CTRL";
echo "Link sensor   UH_TGOAL2  with   tag   UH_TGOAL2 device &UH_CTRL";
echo "Link sensor   UH_TREAL2  with   curve UH_TREAL2 device &UH_CTRL";
echo "Link sensor   UH_GATE2   with   curve UH_GATE2  device &UH_CTRL";
echo "Link sensor   UH_PROP2   with                   device &UH_CTRL";
echo "Link sensor   UH_TNAME2  with   tag   UH_TNAME2 device &UH_CTRL";
echo "Link sensor   UH_PNAME2  with   tag   UH_PNAME2 device &UH_CTRL";
echo "Link sensor   UH_BT3     with   tag   UH_ST3    device &UH_CTRL";
echo "Link sensor   UH_TGOAL3  with   tag   UH_TGOAL3 device &UH_CTRL";
echo "Link sensor   UH_TREAL3  with   curve UH_TREAL3 device &UH_CTRL";
echo "Link sensor   UH_GATE3   with   curve UH_GATE3  device &UH_CTRL";
echo "Link sensor   UH_PROP3   with                   device &UH_CTRL";
echo "Link sensor   UH_TNAME3  with   tag   UH_TNAME3 device &UH_CTRL";
echo "Link sensor   UH_PNAME3  with   tag   UH_PNAME3 device &UH_CTRL";
echo "[]";
};

function do_uniheat_tag(){
 echo ";***************************";
 echo "; Demo_UniHeat Curves & tags";
 echo ";***************************";
 echo "";
 echo "[TagList] ; ToolBar commands";
 echo "DUH.CMD.HELP     = integer 0";
 echo "DUH.CMD.HOME     = integer 0";
 echo "DUH.CMD.OPEN     = integer 0";
 echo "DUH.CMD.SAVE     = integer 0";
 echo "DUH.CMD.LOADINI  = integer 0";
 echo "DUH.CMD.SAVEINI  = integer 0";
 echo "DUH.CMD.TOOLS    = integer 0";
 echo "DUH.CMD.SOUND    = integer 0";
 echo "DUH.CMD.SMILE    = integer 0";
 echo "DUH.CMD.CLOSE    = integer 0";
 echo "[]";
 echo "";
 echo "[DataStorage] ; UniHeat T,P simulation";
 for n in $(seq 1 $1); do do_uniheat_dst $n; done;
 for n in $(seq 1 $1); do do_uniheat_dsp $n; done;
 echo "[]";
};
function do_uniheat_dst(){
 echo "DUH.SIM.T$1  = Curve 0 10 Black  15 1";
};
function do_uniheat_dsp(){
 echo "DUH.SIM.P$1  = Curve 0 10 Black  15 1";
};

function do_uniheat_sel(){
 echo ";***********************************************************";
 echo ";*** КОНФИГУРАЦИОННЫЙ ФАЙЛ CRW-DAQ ДЛЯ СОЗДАНИЯ ПРОГРАММ  **";
 echo ";*** УПРАВЛЕНИЯ ПЕЧАМИ (ЗАГОТОВКА НА 8,16,24 или 32 ПЕЧИ) **";
 echo ";*** Подсистема селектора каналов для управления UniHeat  **";
 echo ";*** Этот файл надо настраивать индивидуально для каждой  **";
 echo ";*** системы, т.к. кривые источников данных меняются      **";
 echo ";*** Для задействования 8,16,24 или 32 печей используйте  **";
 echo ";*** конфигурации UH_8,UH_16,UH_24,UH_32 соответственно.  **";
 echo ";***********************************************************";
 echo "[ConfigFileList] ; Подключение конфигурации UniHeat";
 echo "ConfigFile =~~\resource\daqsite\uniheat\config\uh_$1.cfg";
 echo "[]";
 echo "";
 echo ";************************************";
 echo ";*** Селектор каналов температуры ***";
 echo ";*** Редактируются AnalogInputs и ***";
 echo ";*** UH_TNAMES - имена датчиков T ***";
 echo ";*** Имена датчиков без пробелов, ***";
 echo ";*** разделяются слэшем: T1/T2/.. ***";
 echo ";*** Имена датчиков должны строго ***";
 echo ";*** соответствовать AnalogInputs ***";
 echo ";************************************";
 echo "[&UH_REDIR_T]";
 echo "UsesSmoothing  = 1";
 let "numai=$1+1";
 echo "AnalogInputs   = $numai";
 for n in $(seq 1 $1); do do_uniheat_sel_ait $n $1; done;
 echo "[]";
 echo "[TagList]";
 do_uniheat_sel_t $1;
 echo "[]";
 echo "";
 echo ";************************************";
 echo ";*** Селектор каналов давлений    ***";
 echo ";*** Редактируются AnalogInputs и ***";
 echo ";*** UH_PNAMES - имена датчиков P ***";
 echo ";*** Имена датчиков без пробелов, ***";
 echo ";*** разделяются слэшем: P1/P2/.. ***";
 echo ";*** Имена датчиков должны строго ***";
 echo ";*** соответствовать AnalogInputs ***";
 echo ";************************************";
 echo "[&UH_REDIR_P]";
 echo "UsesSmoothing  = 1";
 let "numai=$1+1";
 echo "AnalogInputs   = $numai";
 for n in $(seq 1 $1); do do_uniheat_sel_aip $n $1; done;
 echo "[]";
 echo "[TagList]";
 do_uniheat_sel_p $1;
 echo "[]";
};
function do_uniheat_sel_ait(){
 echo "Link AnalogInput $1  with curve DUH.SIM.T$1  smoothing 1e-4 0 2 2";
};
function do_uniheat_sel_t(){
 local s="";
 for n in $(seq 1 $1); do s="$s/SIMT$n"; done;
 echo "UH_TNAMES = string ${s:1}";
};
function do_uniheat_sel_aip(){
 echo "Link AnalogInput $1  with curve DUH.SIM.P$1  smoothing 1e-4 0 2 2";
};
function do_uniheat_sel_p(){
 local s="";
 for n in $(seq 1 $1); do s="$s/SIMP$n"; done;
 echo "UH_PNAMES = string ${s:1}";
};

function do_duh_main_gui(){
 echo "[DeviceList]";
 echo "&DUH.MAIN.GUI = device software program";
 echo "[&DUH.MAIN.GUI]";
 echo "Comment        = DEMO UNIHEAT MAIN GUI.";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 10, tpTimeCritical";
 echo "ProgramSource  = ..\daqpas\duh_main_gui.pas";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "StdInFifo      = 128";
 echo "StdOutFifo     = 128";
 echo "AnalogFifo     = 10000";
 echo "DigitalFifo    = 10000";
 echo "tagDUH         = DUH";
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [CustomParameters.TagList]";
 echo "CustomIniFileRef  = ..\data\duh_custom.ini";
 echo "CustomIniSection  = [CustomParameters]";
 echo "CustomIniBackups  = ..\..\duh_data\custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "";
 echo "[ConfigFileList]";
 echo "ConfigFile = ..\data\duh_custom.ini";
 echo "ConfigFile = ..\config\duh_custom.cfg";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "DUH.MAIN.GUI = Circuit_Window ";
 echo "[DUH.MAIN.GUI]";
 echo "Circuit = ..\circuits\duh_main_gui.crc";
 echo "Link sensor DUH.CMD.HELP     with tag   DUH.CMD.HELP     device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.HOME     with tag   DUH.CMD.HOME     device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.OPEN     with tag   DUH.CMD.OPEN     device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.SAVE     with tag   DUH.CMD.SAVE     device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.LOADINI  with tag   DUH.CMD.LOADINI  device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.SAVEINI  with tag   DUH.CMD.SAVEINI  device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.SOUND    with tag   DUH.CMD.SOUND    device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.SMILE    with tag   DUH.CMD.SMILE    device &DUH.MAIN.GUI";
 echo "Link sensor DUH.CMD.CLOSE    with tag   DUH.CMD.CLOSE    device &DUH.MAIN.GUI";
 echo "Link sensor @cron.run+UH_CONTROL.SELECT             with device &DUH.MAIN.GUI";
 if [ $1 -eq 8 ]; then let num=8; else let num=10; fi;
 for n in $(seq 1 $num); do do_link_sens_t $n; done;
 for n in $(seq 1 $num); do do_link_sens_p $n; done;
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "DUH.T.PLOT = Curve_Window";
 echo "[DUH.T.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CT_Sensors_(Поджиг)^N^L___{C}, 0, 1000";
 do_duh_plot_t $1;
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "DUH.P.PLOT = Curve_Window";
 echo "[DUH.P.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CP_Sensors^N^L___{Bar}, 0, 2500";
 do_duh_plot_p $1;
 echo "[]";
};
function do_link_sens_t(){
 echo "Link sensor DUH.T$1           with curve DUH.SIM.T$1       device &DUH.MAIN.GUI";
};
function do_link_sens_p(){
 echo "Link sensor DUH.P$1           with curve DUH.SIM.P$1       device &DUH.MAIN.GUI";
};
function do_duh_plot_t(){
 case $1 in
  8)  do_duh_plot_t8; ;;
  10) do_duh_plot_t10; ;;
  16) do_duh_plot_t16; ;;
  20) do_duh_plot_t20; ;;
  24) do_duh_plot_t24; ;;
  30) do_duh_plot_t30; ;;
  32) do_duh_plot_t32; ;;
  *) ;;
 esac;
};
function do_duh_plot_t8(){
 echo "CurveList = DUH.SIM.T1,  DUH.SIM.T2,  DUH.SIM.T3,  DUH.SIM.T4";
 echo "CurveList = DUH.SIM.T5,  DUH.SIM.T6,  DUH.SIM.T7,  DUH.SIM.T8";
};
function do_duh_plot_t10(){
 echo "CurveList = DUH.SIM.T1,  DUH.SIM.T2,  DUH.SIM.T3,  DUH.SIM.T4,  DUH.SIM.T5";
 echo "CurveList = DUH.SIM.T6,  DUH.SIM.T7,  DUH.SIM.T8,  DUH.SIM.T9,  DUH.SIM.T10";
};
function do_duh_plot_t16(){
 do_duh_plot_t8;
 echo "CurveList = DUH.SIM.T9,  DUH.SIM.T10, DUH.SIM.T11, DUH.SIM.T12";
 echo "CurveList = DUH.SIM.T13, DUH.SIM.T14, DUH.SIM.T15, DUH.SIM.T16";
};
function do_duh_plot_t20(){
 do_duh_plot_t10;
 echo "CurveList = DUH.SIM.T11, DUH.SIM.T12, DUH.SIM.T13, DUH.SIM.T14, DUH.SIM.T15";
 echo "CurveList = DUH.SIM.T16, DUH.SIM.T17, DUH.SIM.T18, DUH.SIM.T19, DUH.SIM.T20";
};
function do_duh_plot_t24(){
 do_duh_plot_t16;
 echo "CurveList = DUH.SIM.T17, DUH.SIM.T18, DUH.SIM.T19, DUH.SIM.T20";
 echo "CurveList = DUH.SIM.T21, DUH.SIM.T22, DUH.SIM.T23, DUH.SIM.T24";
};
function do_duh_plot_t30(){
 do_duh_plot_t20;
 echo "CurveList = DUH.SIM.T21, DUH.SIM.T22, DUH.SIM.T23, DUH.SIM.T24, DUH.SIM.T25";
 echo "CurveList = DUH.SIM.T26, DUH.SIM.T27, DUH.SIM.T28, DUH.SIM.T29, DUH.SIM.T30";
};
function do_duh_plot_t32(){
 do_duh_plot_t24;
 echo "CurveList = DUH.SIM.T25, DUH.SIM.T26, DUH.SIM.T27, DUH.SIM.T28";
 echo "CurveList = DUH.SIM.T29, DUH.SIM.T30, DUH.SIM.T31, DUH.SIM.T32";
};
function do_duh_plot_p(){
 case $1 in
  8)  do_duh_plot_p8;  ;;
  10) do_duh_plot_p10; ;;
  16) do_duh_plot_p16; ;;
  20) do_duh_plot_p20; ;;
  24) do_duh_plot_p24; ;;
  30) do_duh_plot_p30; ;;
  32) do_duh_plot_p32; ;;
  *) ;;
 esac;
};
function do_duh_plot_p8(){
 echo "CurveList = DUH.SIM.P1,  DUH.SIM.P2,  DUH.SIM.P3,  DUH.SIM.P4";
 echo "CurveList = DUH.SIM.P5,  DUH.SIM.P6,  DUH.SIM.P7,  DUH.SIM.P8";
};
function do_duh_plot_p10(){
 echo "CurveList = DUH.SIM.P1,  DUH.SIM.P2,  DUH.SIM.P3,  DUH.SIM.P4,  DUH.SIM.P5";
 echo "CurveList = DUH.SIM.P6,  DUH.SIM.P7,  DUH.SIM.P8,  DUH.SIM.P9,  DUH.SIM.P10";
};
function do_duh_plot_p16(){
 do_duh_plot_p8;
 echo "CurveList = DUH.SIM.P9,  DUH.SIM.P10, DUH.SIM.P11, DUH.SIM.P12";
 echo "CurveList = DUH.SIM.P13, DUH.SIM.P14, DUH.SIM.P15, DUH.SIM.P16";
};
function do_duh_plot_p20(){
 do_duh_plot_p10;
 echo "CurveList = DUH.SIM.P11, DUH.SIM.P12, DUH.SIM.P13, DUH.SIM.P14, DUH.SIM.P15";
 echo "CurveList = DUH.SIM.P16, DUH.SIM.P17, DUH.SIM.P18, DUH.SIM.P19, DUH.SIM.P20";
};
function do_duh_plot_p24(){
 do_duh_plot_p16;
 echo "CurveList = DUH.SIM.P17, DUH.SIM.P18, DUH.SIM.P19, DUH.SIM.P20";
 echo "CurveList = DUH.SIM.P21, DUH.SIM.P22, DUH.SIM.P23, DUH.SIM.P24";
};
function do_duh_plot_p30(){
 do_duh_plot_p20;
 echo "CurveList = DUH.SIM.P21, DUH.SIM.P22, DUH.SIM.P23, DUH.SIM.P24, DUH.SIM.P25";
 echo "CurveList = DUH.SIM.P26, DUH.SIM.P27, DUH.SIM.P28, DUH.SIM.P29, DUH.SIM.P30";
};
function do_duh_plot_p32(){
 do_duh_plot_p24;
 echo "CurveList = DUH.SIM.P25, DUH.SIM.P26, DUH.SIM.P27, DUH.SIM.P28";
 echo "CurveList = DUH.SIM.P29, DUH.SIM.P30, DUH.SIM.P31, DUH.SIM.P32";
};

function do_duh_datsrv(){
 echo "[ConfigFileList]";
 echo "ConfigFile = ~~\resource\daqsite\default\datsrv.cfg";
 echo "[]";
 echo "[&DATSRV]";
 echo "IntegrityMode  = 1";
 echo "SavePeriod     = 300";
 echo "FilePrefix     = duh_";
 echo "DataPath       = ..\..\duh_data";
 echo "[]";
 echo "";
 echo "[&DATSRV]";
 do_duh_plot_t $1;
 echo "[]";
};

function main(){
 uniheat_main "$@";
};

main "$@";

##############
## END OF FILE
##############
