 {
 ***********************************************************************
 Daq Pascal application program I7083_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Status n      - Read encoder n=0..2 status.
| @OpMode n m    - Set  encoder n=0..2  operation mode m.
| @Preset n m    - Set  encoder n=0..2 to preset value m.
| @DevMsg  &d m  - Send message m to device &d.
| @DevSend &d m  - Send message m to device &d.
|********************************************************
| Curves:
|  DO[0..2] - Encoder 0..2 values (counter)
|  DO[3..5] - Encoder 0..2 status
|  DO[6..8] - Encoder 0..2 preset values
|  AO[0]    - Polling rate per second
|  AO[1]    - 1/0 Connection status
| Tags:
|  None
| INI params:
|  Adam_Slot_Device = Link to Adam_Slot device
|  StartupScript    = Section name of startup script
|  WatchDogTime     = Watchdog deadline time, sec
| Messages:
|  @Status - Read encoder status.
|  @OpMode - Set encoder operation mode. 
|  @Preset - Read or reset encoder to preset value.
[]
 }
program I7083_DRV;               { I7083 device driver              }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 MaxCmdNum         = 31;         { Maximal command number           }
 MaxEncNum         = 2;          { Maximal encoder 0-base index     }
 {-------------------------------- Mitsubishi FR-F740 command codes }
 cm_RdModuleId     = 0;          { Read Module Name                 }
 cm_RdFirmwareId   = 1;          { Read Firmware Version            }
 cm_RdEnc0Val      = 2;          { Read Encoder 0 Value             }
 cm_RdEnc1Val      = 3;          { Read Encoder 1 Value             }
 cm_RdEnc2Val      = 4;          { Read Encoder 2 Value             }
 cm_RdEnc0Stat     = 5;          { Read Encoder 0 Status            }
 cm_RdEnc1Stat     = 6;          { Read Encoder 1 Status            }
 cm_RdEnc2Stat     = 7;          { Read Encoder 2 Status            }
 cm_RdWdtStat      = 8;          { Read Watchdog Status             }
 cm_ResetWdt       = 9;          { Reset Watchdog Status            }
 cm_SetupWdt       = 10;         { Setup Watchdog Parameters        }
 cm_HostIsOk       = 11;         { Watchdog Host is Ok Notification }
 cm_WrEnc0OpMode   = 12;         { Write Encoder 0 Operation Mode   }
 cm_WrEnc1OpMode   = 13;         { Write Encoder 1 Operation Mode   }
 cm_WrEnc2OpMode   = 14;         { Write Encoder 2 Operation Mode   }
 cm_RdEnc0Preset   = 15;         { Read Encoder 0 Preset Value      }
 cm_RdEnc1Preset   = 16;         { Read Encoder 1 Preset Value      }
 cm_RdEnc2Preset   = 17;         { Read Encoder 2 Preset Value      }
 cm_WrEnc0Preset   = 18;         { Write Encoder 0 Preset Value     }
 cm_WrEnc1Preset   = 19;         { Write Encoder 1 Preset Value     }
 cm_WrEnc2Preset   = 20;         { Write Encoder 2 Preset Value     }
 cm_DoEnc0Preset   = 21;         { Reset Encoder 0 to Preset Value  }
 cm_DoEnc1Preset   = 22;         { Reset Encoder 0 to Preset Value  }
 cm_DoEnc2Preset   = 23;         { Reset Encoder 0 to Preset Value  }
 {-------------------------------- Digital Outputs                  }
 do_Enc0Val        = 0;          { Encoder 0 Value (Counter)        }
 do_Enc1Val        = 1;          { Encoder 1 Value (Counter)        }
 do_Enc2Val        = 2;          { Encoder 2 Value (Counter)        }
 do_Enc0Stat       = 3;          { Encoder 0 Status                 }
 do_Enc1Stat       = 4;          { Encoder 1 Status                 }
 do_Enc2Stat       = 5;          { Encoder 2 Status                 }
 do_Enc0Preset     = 6;          { Encoder 0 Preset Value           }
 do_Enc1Preset     = 7;          { Encoder 1 Preset Value           }
 do_Enc2Preset     = 8;          { Encoder 2 Preset Value           }
 {-------------------------------- Analog Outputs                   }
 ao_PolRate        = 0;          { PollingRate, Hz                  }
 ao_Connect        = 1;          { Connection status, 0/1           }
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 I7083             : record      { I7083 driver data                }
  RatePeriod       : Integer;    { Polling rate refresh period      }
  RateCounter      : Integer;    { Polling rate counter             }
  CmdTab           : record      { Command table                    }
   Count           : Integer;    { Command counter, 0..MaxCmdNum+1  }
   Current         : Integer;    { Current command index to send    }
   SentCmd         : Integer;    { Last sent command ID             }
   Enabled         : array[0..MaxCmdNum] of Boolean; {              }
   Comment         : array[0..MaxCmdNum] of String;  {              }
   CmdId           : array[0..MaxCmdNum] of Integer; {              }
  end;                           {                                  }
  Com              : record      { Com port data                    }
   Port            : Integer;    { Com port number                  }
   Status          : Integer;    { Com port status, see rs_XX const }
   Answer          : String;     { Com port received reply          }
   Request         : String;     { Com port transmitted request     }
   HexAddr         : String;     { Com port device address, hex     }
   Address         : Integer;    { Com port device address          }
  end;                           {                                  }
  ErrorCode        : record      { Special error codes              }
   NotAvail        : Integer;    { COM port is not available        }
   TimeOut         : Integer;    { Timeout detected                 }
   Format          : Integer;    { Format error                     }
  end;                           {                                  }
  WatchdogTime     : Real;       { Watchdog deadline time, sec      }
  HostIsOkPeriod   : Integer;    { Watchdog HostIsOk period, ms     }
  Encoder          : array[0..MaxEncNum] of record { Encoder data   }
   OpMode          : Integer;    { Operation mode                   }
   Preset          : Integer;    { Preset value                     }
  end;                           {                                  }
 end;                            {                                  }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 {
 Allowed Success/Trouble/ViewImp/ViewExp/Details.
 }
 function AllowedSuccess:Boolean; begin AllowedSuccess:=iAnd(dfSuccess,DebugFlags)>0; end;
 function AllowedTrouble:Boolean; begin AllowedTrouble:=iAnd(dfTrouble,DebugFlags)>0; end;
 function AllowedViewExp:Boolean; begin AllowedViewExp:=iAnd(dfViewExp,DebugFlags)>0; end;
 function AllowedViewImp:Boolean; begin AllowedViewImp:=iAnd(dfViewImp,DebugFlags)>0; end;
 function AllowedDetails:Boolean; begin AllowedDetails:=iAnd(dfDetails,DebugFlags)>0; end;
 {
 Add command to CmdId command table.
 }
 procedure I7083_AddCmdId(CmdId:Integer; Enabled:Boolean; Comment:String);
 begin
  if I7083.CmdTab.Count<=MaxCmdNum then begin
   I7083.CmdTab.Enabled[I7083.CmdTab.Count]:=Enabled;
   I7083.CmdTab.Comment[I7083.CmdTab.Count]:=Comment;
   I7083.CmdTab.CmdId[I7083.CmdTab.Count]:=CmdId;
   I7083.CmdTab.Count:=I7083.CmdTab.Count+1;
   if AllowedSuccess then Success('I7083.CmdTab['+StrFix(I7083.CmdTab.Count,2,0)+']: '
     +StrFix(CmdId,3,0)+'=$'+HexB(CmdId)+', '+Str(Ord(Enabled))+', "'+Comment+'".');
  end;
 end;
 {
 Enable command with given command identifier.
 }
 procedure I7083_EnableCmdId(CmdId:Integer; Enabled:Boolean);
 var i:Integer;
 begin
  for i:=0 to I7083.CmdTab.Count-1 do
  if I7083.CmdTab.CmdId[i]=CmdId then I7083.CmdTab.Enabled[i]:=Enabled;
 end;
 {
 Get comment string for given command CmdId.
 }
 function I7083_CommentCmdId(CmdId:Integer):String;
 var i,j:Integer;
 begin
  i:=0; j:=-1;
  while (i<I7083.CmdTab.Count) and (j<0) do begin
   if I7083.CmdTab.CmdId[i]=CmdId then j:=i;
   i:=i+1;
  end;
  if j<0 then I7083_CommentCmdId:='' else I7083_CommentCmdId:=I7083.CmdTab.Comment[j];
 end;
 {
 Reset command table.
 }
 procedure I7083_ResetCmdTable;
 begin
  I7083.CmdTab.Count:=0;
  I7083.CmdTab.Current:=0;
  I7083.CmdTab.SentCmd:=0;
  I7083_AddCmdId(cm_RdModuleId,   True,                      'Read Module ID');
  I7083_AddCmdId(cm_RdFirmwareId, True,                      'Read Firmware ID');
  I7083_AddCmdId(cm_RdWdtStat,    I7083.WatchdogTime>0,      'Read Watchdog Status');
  I7083_AddCmdId(cm_HostIsOk,     I7083.WatchdogTime>0,      'HostIsOk Notification');
  I7083_AddCmdId(cm_ResetWdt,     True,                      'Reset Watchdog Status');
  I7083_AddCmdId(cm_SetupWdt,     True,                      'Setup Watchdog Params');
  I7083_AddCmdId(cm_WrEnc0OpMode, False,                     'Write Encoder 0 Operation Mode');
  I7083_AddCmdId(cm_WrEnc1OpMode, False,                     'Write Encoder 1 Operation Mode');
  I7083_AddCmdId(cm_WrEnc2OpMode, False,                     'Write Encoder 2 Operation Mode');
  I7083_AddCmdId(cm_WrEnc0Preset, False,                     'Write Encoder 0 Preset Value');
  I7083_AddCmdId(cm_WrEnc1Preset, False,                     'Write Encoder 1 Preset Value');
  I7083_AddCmdId(cm_WrEnc2Preset, False,                     'Write Encoder 2 Preset Value');
  I7083_AddCmdId(cm_DoEnc0Preset, False,                     'Reset Encoder 0 to Preset Value');
  I7083_AddCmdId(cm_DoEnc1Preset, False,                     'Reset Encoder 1 to Preset Value');
  I7083_AddCmdId(cm_DoEnc2Preset, False,                     'Reset Encoder 2 to Preset Value');
  I7083_AddCmdId(cm_RdEnc0Preset, DoMap(do_Enc0Preset,1)>0,  'Read Encoder 0 Preset Value');
  I7083_AddCmdId(cm_RdEnc1Preset, DoMap(do_Enc1Preset,1)>0,  'Read Encoder 1 Preset Value');
  I7083_AddCmdId(cm_RdEnc2Preset, DoMap(do_Enc2Preset,1)>0,  'Read Encoder 2 Preset Value');
  I7083_AddCmdId(cm_RdEnc0Val,    DoMap(do_Enc0Val,1)>0,     'Read Encoder 0 Value');
  I7083_AddCmdId(cm_RdEnc1Val,    DoMap(do_Enc1Val,1)>0,     'Read Encoder 1 Value');
  I7083_AddCmdId(cm_RdEnc2Val,    DoMap(do_Enc2Val,1)>0,     'Read Encoder 2 Value');
  I7083_AddCmdId(cm_RdEnc0Stat,   DoMap(do_Enc0Stat,1)>0,    'Read Encoder 0 Status');
  I7083_AddCmdId(cm_RdEnc1Stat,   DoMap(do_Enc1Stat,1)>0,    'Read Encoder 1 Status');
  I7083_AddCmdId(cm_RdEnc2Stat,   DoMap(do_Enc2Stat,1)>0,    'Read Encoder 2 Status');
  I7083.HostIsOkPeriod:=Ord(I7083.WatchdogTime>0)*Round(I7083.WatchdogTime*1000*0.5);
  I7083.RateCounter:=0;
 end;
 {
 Prepare next command to send.
 Also skips disabled commands. 
 }
 procedure I7083_PrepareNextCmd(Step:Integer);
 begin
  I7083.CmdTab.Current:=I7083.CmdTab.Current+Step;
  if I7083.CmdTab.Current<0 then I7083.CmdTab.Current:=0;
  if I7083.CmdTab.Current>=I7083.CmdTab.Count then I7083.CmdTab.Current:=0;
  Step:=0;
  while (Step<=MaxCmdNum) and not I7083.CmdTab.Enabled[I7083.CmdTab.Current] do begin
   I7083.CmdTab.Current:=I7083.CmdTab.Current+1;
   if I7083.CmdTab.Current>=I7083.CmdTab.Count then I7083.CmdTab.Current:=0;
   Step:=Step+1;
  end;
 end;
 {
 Bad data format handler.
 }
 procedure I7083_BadFormat;
 begin
  Failure(I7083.ErrorCode.Format,'REQUEST "'
          +Trim(I7083.Com.Request)+'" ERROR: BAD FORMAT ANSWER "'
          +Trim(I7083.Com.Answer)+'"');
 end;
 {
 Update data curves & tags.
 }
 procedure UpdateModuleId(id:String);
 begin
  if AllowedSuccess then Success('ModuleId '+id);
  I7083_EnableCmdId(cm_RdModuleId,False);
 end;
 procedure UpdateFirmwareId(id:String);
 begin
  if AllowedSuccess then Success('FirmwareId '+id);
  I7083_EnableCmdId(cm_RdFirmwareId,False);
 end;
 procedure UpdateEnc0Val(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc0Val,t,v);
 end;
 procedure UpdateEnc1Val(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc1Val,t,v);
 end;
 procedure UpdateEnc2Val(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc2Val,t,v);
 end;
 procedure UpdateEnc0Stat(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc0Stat,t,v);
  I7083_EnableCmdId(cm_RdEnc0Stat,False);
 end;
 procedure UpdateEnc1Stat(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc1Stat,t,v);
  I7083_EnableCmdId(cm_RdEnc1Stat,False);
 end;
 procedure UpdateEnc2Stat(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc2Stat,t,v);
  I7083_EnableCmdId(cm_RdEnc2Stat,False);
 end;
 procedure UpdateEnc0Preset(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc0Preset,t,v);
  I7083_EnableCmdId(cm_RdEnc0Preset,False);
 end;
 procedure UpdateEnc1Preset(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc1Preset,t,v);
  I7083_EnableCmdId(cm_RdEnc1Preset,False);
 end;
 procedure UpdateEnc2Preset(t,v:Real);
 begin
  if IsNan(v) then I7083_BadFormat else UpdateDo(do_Enc2Preset,t,v);
  I7083_EnableCmdId(cm_RdEnc2Preset,False);
 end;
 procedure UpdatePolRate(t,v:Real);
 begin
  UpdateAo(ao_PolRate,t,v);
  UpdateAo(ao_Connect,t,Ord(v>0));
 end;
 {
 Handle "ComPort is not available"...
 }
 procedure I7083_OnNotAvail;
 begin
  if RunCount<1000
  then Problem(                         'COM'+Str(I7083.Com.Port)+' is not available.')
  else Failure(I7083.ErrorCode.NotAvail,'COM'+Str(I7083.Com.Port)+' is not available.');
 end;
 {
 Handle "RS-232 line is ready for new request"...
 If has command to send, try to send the command.
 }
 procedure I7083_OnNoRequest;
 var msg:String; TimeOut:Integer;
 begin  
  msg:='';
  TimeOut:=0;
  I7083.Com.Answer:='';     // Clear COM Rx buffer
  I7083.Com.Request:='';    // Clear COM Tx buffer
  I7083_PrepareNextCmd(0);  // Skip disabled commands
  if I7083.CmdTab.Enabled[I7083.CmdTab.Current] then begin
   I7083.CmdTab.SentCmd:=I7083.CmdTab.CmdId[I7083.CmdTab.Current];
   // Read Encoder 0 Value
   if (I7083.CmdTab.SentCmd=cm_RdEnc0Val) then begin
    msg:='#'+I7083.Com.HexAddr+'0';
   end else
   // Read Encoder 1 Value
   if (I7083.CmdTab.SentCmd=cm_RdEnc1Val) then begin
    msg:='#'+I7083.Com.HexAddr+'1';
   end else
   // Read Encoder 2 Value
   if (I7083.CmdTab.SentCmd=cm_RdEnc2Val) then begin
    msg:='#'+I7083.Com.HexAddr+'2';
   end else
   // Read Encoder 0 Status
   if (I7083.CmdTab.SentCmd=cm_RdEnc0Stat) then begin
    msg:='$'+I7083.Com.HexAddr+'S0';
   end else
   // Read Encoder 1 Status
   if (I7083.CmdTab.SentCmd=cm_RdEnc1Stat) then begin
    msg:='$'+I7083.Com.HexAddr+'S1';
   end else
   // Read Encoder 2 Status
   if (I7083.CmdTab.SentCmd=cm_RdEnc2Stat) then begin
    msg:='$'+I7083.Com.HexAddr+'S2';
   end else
   // Read Encoder 0 Preset Value
   if (I7083.CmdTab.SentCmd=cm_RdEnc0Preset) then begin
    msg:='@'+I7083.Com.HexAddr+'G0';
   end else
   // Read Encoder 1 Preset Value
   if (I7083.CmdTab.SentCmd=cm_RdEnc1Preset) then begin
    msg:='@'+I7083.Com.HexAddr+'G1';
   end else
   // Read Encoder 2 Preset Value
   if (I7083.CmdTab.SentCmd=cm_RdEnc2Preset) then begin
    msg:='@'+I7083.Com.HexAddr+'G2';
   end else
   // Write Encoder 0 Preset Value
   if (I7083.CmdTab.SentCmd=cm_WrEnc0Preset) then begin
    msg:='@'+I7083.Com.HexAddr+'P0'+HexL(I7083.Encoder[0].Preset);
   end else
   // Write Encoder 1 Preset Value
   if (I7083.CmdTab.SentCmd=cm_WrEnc1Preset) then begin
    msg:='@'+I7083.Com.HexAddr+'P1'+HexL(I7083.Encoder[1].Preset);
   end else
   // Write Encoder 2 Preset Value
   if (I7083.CmdTab.SentCmd=cm_WrEnc2Preset) then begin
    msg:='@'+I7083.Com.HexAddr+'P2'+HexL(I7083.Encoder[2].Preset);
   end else
   // Reset Encoder 0 to Preset Value
   if (I7083.CmdTab.SentCmd=cm_DoEnc0Preset) then begin
    msg:='$'+I7083.Com.HexAddr+'60';
   end else
   // Reset Encoder 1 to Preset Value
   if (I7083.CmdTab.SentCmd=cm_DoEnc1Preset) then begin
    msg:='$'+I7083.Com.HexAddr+'61';
   end else
   // Reset Encoder 2 to Preset Value
   if (I7083.CmdTab.SentCmd=cm_DoEnc2Preset) then begin
    msg:='$'+I7083.Com.HexAddr+'62';
   end else
   // Read WDT status
   if (I7083.CmdTab.SentCmd=cm_RdWdtStat) then begin
    msg:='~'+I7083.Com.HexAddr+'0';
   end else
   // Reset WDT status
   if (I7083.CmdTab.SentCmd=cm_ResetWdt)  then begin
    msg:='~'+I7083.Com.HexAddr+'1';
   end else
   // Setup WDT params
   if (I7083.CmdTab.SentCmd=cm_SetupWdt) then begin
    msg:='~'+I7083.Com.HexAddr+'3'+Str(Ord(I7083.WatchdogTime>0))
        +HexB(Round(Max(0,Min(255,I7083.WatchdogTime*10))));
   end else
   // Host is Ok notification
   if (I7083.CmdTab.SentCmd=cm_HostIsOk) then begin
    I7083_EnableCmdId(cm_HostIsOk,False);
    msg:='~**'; TimeOut:=-10;
   end else
   // Write operation mode encoder 0
   if (I7083.CmdTab.SentCmd=cm_WrEnc0OpMode) then begin
    msg:='$'+I7083.Com.HexAddr+'D0'+Copy(HexB(I7083.Encoder[0].OpMode),2,1);
   end else
   // Write operation mode encoder 1
   if (I7083.CmdTab.SentCmd=cm_WrEnc1OpMode) then begin
    msg:='$'+I7083.Com.HexAddr+'D1'+Copy(HexB(I7083.Encoder[1].OpMode),2,1);
   end else
   // Write operation mode encoder 2
   if (I7083.CmdTab.SentCmd=cm_WrEnc2OpMode) then begin
    msg:='$'+I7083.Com.HexAddr+'D2'+Copy(HexB(I7083.Encoder[2].OpMode),2,1);
   end else
   // Read Firmware ID
   if (I7083.CmdTab.SentCmd=cm_RdFirmwareId) then begin
    msg:='$'+I7083.Com.HexAddr+'F';
   end else
   // Read Module ID
   if (I7083.CmdTab.SentCmd=cm_RdModuleId) then begin
    msg:='$'+I7083.Com.HexAddr+'M';
   end else
   // Unknown
   begin
    if AllowedDetails then Details('Unknown command: $'+HexB(I7083.CmdTab.SentCmd));
    msg:='';
   end;
   if Length(msg)>0 then begin
    if Adam_Request(msg+Chr(_CR),TimeOut) then begin
     if AllowedViewExp then ViewExp('COM'+Str(I7083.Com.Port)+': '+msg);
     if AllowedDetails then Details('Request '+msg+' = '+I7083_CommentCmdId(I7083.CmdTab.SentCmd));
    end else begin
     Trouble('Failure request '+msg+' = '+I7083_CommentCmdId(I7083.CmdTab.SentCmd));
    end;
   end else begin
    I7083_PrepareNextCmd(1);
   end;
  end;
  msg:='';
 end;
 {
 Handle "WaitQueue detected" event
 }
 procedure I7083_OnWaitQueue;
 begin
  if AllowedDetails then Details('Waiting queue...')
 end;
 {
 Handle "Request sent, waiting answer" state.
 }
 procedure I7083_OnWaitReply;
 begin
  if AllowedDetails then Details('Waiting reply...')
 end;
 {
 Handle "Answer received"...
 }
 procedure I7083_OnGotAnswer;
 var v:Real;
  procedure UpdateWdtStat(w:Integer);
  begin
   if IsBit(w,2) then begin
    I7083_EnableCmdId(cm_ResetWdt,True);
    if AllowedDetails then Details('Watchdog raised: $'+HexB(w));
   end;
  end;
 begin
  I7083.Com.Answer:=Adam_Get('Answer');
  I7083.Com.Request:=Adam_Get('Request');
  I7083.RateCounter:=I7083.RateCounter+1;
  if AllowedViewImp then ViewImp('COM'+Str(I7083.Com.Port)+': '+I7083.Com.Answer);
  if AllowedDetails then Details('Got Answer='+I7083.Com.Answer+' on Request='
            +I7083.Com.Request+' = '+I7083_CommentCmdId(I7083.CmdTab.SentCmd));
  // Read Encoder 0 Value
  if I7083.CmdTab.SentCmd=cm_RdEnc0Val then begin
   if Pos('>',I7083.Com.Answer)=1
   then UpdateEnc0Val(time,rVal('$'+Copy(I7083.Com.Answer,2)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 1 Value
  if I7083.CmdTab.SentCmd=cm_RdEnc1Val then begin
   if Pos('>',I7083.Com.Answer)=1
   then UpdateEnc1Val(time,rVal('$'+Copy(I7083.Com.Answer,2)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 2 Value
  if I7083.CmdTab.SentCmd=cm_RdEnc2Val then begin
   if Pos('>',I7083.Com.Answer)=1
   then UpdateEnc2Val(time,rVal('$'+Copy(I7083.Com.Answer,2)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 0 Status
  if I7083.CmdTab.SentCmd=cm_RdEnc0Stat then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateEnc0Stat(time,rVal('$'+Copy(I7083.Com.Answer,4)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 1 Status
  if I7083.CmdTab.SentCmd=cm_RdEnc1Stat then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateEnc1Stat(time,rVal('$'+Copy(I7083.Com.Answer,4)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 2 Status
  if I7083.CmdTab.SentCmd=cm_RdEnc2Stat then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateEnc2Stat(time,rVal('$'+Copy(I7083.Com.Answer,4)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 0 Preset Value
  if I7083.CmdTab.SentCmd=cm_RdEnc0Preset then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateEnc0Preset(time,rVal('$'+Copy(I7083.Com.Answer,4)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 1 Preset Value
  if I7083.CmdTab.SentCmd=cm_RdEnc1Preset then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateEnc1Preset(time,rVal('$'+Copy(I7083.Com.Answer,4)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Encoder 2 Preset Value
  if I7083.CmdTab.SentCmd=cm_RdEnc2Preset then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateEnc2Preset(time,rVal('$'+Copy(I7083.Com.Answer,4)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Write Encoder 0 Preset Value
  if (I7083.CmdTab.SentCmd=cm_WrEnc0Preset) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Write Encoder 1 Preset Value
  if (I7083.CmdTab.SentCmd=cm_WrEnc1Preset) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Write Encoder 2 Preset Value
  if (I7083.CmdTab.SentCmd=cm_WrEnc2Preset) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Reset Encoder 0 to Preset Value
  if (I7083.CmdTab.SentCmd=cm_DoEnc0Preset) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Reset Encoder 1 to Preset Value
  if (I7083.CmdTab.SentCmd=cm_DoEnc1Preset) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Reset Encoder 2 to Preset Value
  if (I7083.CmdTab.SentCmd=cm_DoEnc2Preset) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read WDT Status
  if I7083.CmdTab.SentCmd=cm_RdWdtStat then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateWdtStat(Val('$'+Copy(I7083.Com.Answer,4)))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Reset WDT Status
  if I7083.CmdTab.SentCmd=cm_ResetWdt then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Setup WDT params
  if I7083.CmdTab.SentCmd=cm_SetupWdt then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Host is Ok notification
  if (I7083.CmdTab.SentCmd=cm_HostIsOk) then begin
   I7083_PrepareNextCmd(1);
  end else
  // Write Encoder 0 Operation Mode
  if (I7083.CmdTab.SentCmd=cm_WrEnc0OpMode) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Write Encoder 1 Operation Mode
  if (I7083.CmdTab.SentCmd=cm_WrEnc1OpMode) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Write Encoder 2 Operation Mode
  if (I7083.CmdTab.SentCmd=cm_WrEnc2OpMode) then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then I7083_EnableCmdId(I7083.CmdTab.SentCmd,False)
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Firmware ID
  if I7083.CmdTab.SentCmd=cm_RdFirmwareId then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateFirmwareId(Copy(I7083.Com.Answer,4))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Read Module ID
  if I7083.CmdTab.SentCmd=cm_RdModuleId then begin
   if Pos('!'+I7083.Com.HexAddr,I7083.Com.Answer)=1
   then UpdateModuleId(Copy(I7083.Com.Answer,4))
   else I7083_BadFormat;
   I7083_PrepareNextCmd(1);
  end else
  // Unknown
  begin
   Trouble('Unknown command $'+HexB(I7083.CmdTab.SentCmd)+', request '+I7083.Com.Request);
   I7083_PrepareNextCmd(1);
  end;
 end;
 {
 Handle "TimeOut detected" event
 }
 procedure I7083_OnTimeOut;
 var SkipEvent:Boolean;
 begin
  I7083.Com.Answer:='';
  I7083.Com.Request:='';
  SkipEvent:=(I7083.CmdTab.SentCmd=cm_HostIsOk);
  if not SkipEvent then begin
   Failure(I7083.ErrorCode.TimeOut,'TimeOut detected.');
   I7083_ResetCmdTable;
  end;
 end;
 {
 Clear all I7083 params.
 }
 procedure I7083_Clear;
 var i:Integer;
 begin
  I7083.Com.Port:=0;
  I7083.Com.Answer:='';
  I7083.Com.Request:='';
  I7083.Com.HexAddr:='';
  I7083.Com.Status:=0;
  I7083.CmdTab.Count:=0;
  I7083.CmdTab.Current:=0;
  I7083.CmdTab.SentCmd:=0;
  for i:=0 to MaxCmdNum do begin
   I7083.CmdTab.Enabled[i]:=False;
   I7083.CmdTab.Comment[i]:='';
   I7083.CmdTab.CmdId[i]:=0;
  end;
  I7083.WatchdogTime:=0;
  I7083.HostIsOkPeriod:=0;
  for i:=0 to MaxEncNum do begin
   I7083.Encoder[i].OpMode:=0;
   I7083.Encoder[i].Preset:=0;
  end;
 end;
 {
 Initialize I7083 params.
 }
 procedure I7083_Init;
 var i:Integer;
 begin
  I7083_Clear;
  I7083.Com.HexAddr:=Adam_Get('Address');
  I7083.Com.Address:=Val('$'+I7083.Com.HexAddr);
  Success('Address = $'+HexB(I7083.Com.Address));
  I7083.Com.Port:=Val(Adam_Get('Port'));
  if I7083.Com.Port<>0
  then Success('COM'+Str(I7083.Com.Port)+' opened')
  else Trouble('Could not open COM'+Str(I7083.Com.Port));
  I7083.WatchdogTime:=Max(0,Min(25.5,rValDef(ReadIni('WatchdogTime'),0)));
  Success('WatchdogTime = '+Str(I7083.WatchdogTime));
  I7083.RatePeriod:=iValDef(ReadIni('RatePeriod'),1000);
  Success('RatePeriod = '+Str(I7083.RatePeriod));
  I7083_ResetCmdTable;
  {
  Initialize special errors...
  }
  I7083.ErrorCode.NotAvail:=RegisterErr('ADAM: ComPort not avail');
  I7083.ErrorCode.Format:=RegisterErr('ADAM: Format is invalid');
  I7083.ErrorCode.TimeOut:=RegisterErr('ADAM: TimeOut detected');
 end;
 {
 Finalize I7083 device.
 }
 procedure I7083_Free;
 begin
 end;
 {
 Poll I7083 device.
 }
 procedure I7083_Poll;
 begin
  //
  // Polling rate calculator
  //
  if I7083.RatePeriod>0 then
  if SysTimer_Pulse(I7083.RatePeriod)>0 then begin
   UpdatePolRate(time,I7083.RateCounter*1e3/I7083.RatePeriod);
   I7083.RateCounter:=0;
  end;
  //
  // Watchdog Host Is Ok
  //
  if I7083.WatchdogTime>0 then
  if I7083.HostIsOkPeriod>0 then
  if SysTimer_Pulse(I7083.HostIsOkPeriod)>0
  then I7083_EnableCmdId(cm_HostIsOk,True);
  //
  // COM port polling
  //
  if I7083.Com.Port>0 then begin
   I7083.Com.Status:=Adam_Status;
   if I7083.Com.Status=rs_NoRequest  then I7083_OnNoRequest else
   if I7083.Com.Status=rs_WaitQueue  then I7083_OnWaitQueue else
   if I7083.Com.Status=rs_WaitAnswer then I7083_OnWaitReply else
   if I7083.Com.Status=rs_Answer     then I7083_OnGotAnswer else
   if I7083.Com.Status=rs_TimeOut    then I7083_OnTimeOut   else
   if I7083.Com.Status=rs_NotAvail   then I7083_OnNotAvail  else
   Trouble('Invalid status '+Str(I7083.Com.Status));
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  I7083_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  I7083_Init;
  RunStartupScript;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  RunFinallyScript;
  I7083_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  I7083_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; i,n:Integer; r:Real;
 begin
  ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   @Preset 1 0 - Reset Encoder 1 to preset value 0
   }
   if IsSameText(cmd,'@Preset') then begin
    i:=iValDef(ExtractWord(1,arg),-1);
    if (i>=0) and (i<=MaxEncNum) then begin
     I7083_EnableCmdId(cm_RdEnc0Preset,DoMap(do_Enc0Preset+i,1)>0);
     r:=rValDef(ExtractWord(2,arg),_NaN);
     if not IsNan(r) then begin
      I7083.Encoder[i].Preset:=Round(r);
      I7083_EnableCmdId(cm_WrEnc0Preset+i,True);
      I7083_EnableCmdId(cm_DoEnc0Preset+i,True);
     end;
    end else begin
     I7083_EnableCmdId(cm_RdEnc0Preset,DoMap(do_Enc0Preset,1)>0);
     I7083_EnableCmdId(cm_RdEnc1Preset,DoMap(do_Enc1Preset,1)>0);
     I7083_EnableCmdId(cm_RdEnc2Preset,DoMap(do_Enc2Preset,1)>0);
    end;
    Data:='';
   end else
   {
   @Status 0 - Read Encoder 0 status
   }
   if IsSameText(cmd,'@Status') then begin
    i:=iValDef(ExtractWord(1,arg),-1);
    if (i>=0) and (i<=MaxEncNum) then begin
     I7083_EnableCmdId(cm_RdEnc0Stat,DoMap(do_Enc0Stat+i,1)>0);
    end else begin
     I7083_EnableCmdId(cm_RdEnc0Stat,DoMap(do_Enc0Stat,1)>0);
     I7083_EnableCmdId(cm_RdEnc1Stat,DoMap(do_Enc1Stat,1)>0);
     I7083_EnableCmdId(cm_RdEnc2Stat,DoMap(do_Enc2Stat,1)>0);
    end;
    Data:='';
   end else
   {
   @OpMode 1 $B - Set Encoder 1 operation mode $B
   }
   if IsSameText(cmd,'@OpMode') then begin
    i:=iValDef(ExtractWord(1,arg),-1);
    if (i>=0) and (i<=MaxEncNum) then begin
     r:=rValDef(ExtractWord(2,arg),_NaN);
     if not IsNan(r) then begin
      I7083.Encoder[i].OpMode:=Round(r);
      I7083_EnableCmdId(cm_WrEnc0OpMode+i,True);
     end;
    end;
    Data:='';
   end else
   {
   Example: @DevMsg  &SpeakSrv @Speak=Hello
            @DevSend &SpeakSrv @Speak=Hello
   }
   if IsSameText(cmd,'@DevMsg')
   or IsSameText(cmd,'@DevSend') then begin
    i:=RefFind('Device '+ExtractWord(1,arg));
    if i=0 then n:=0 else n:=Round(DevSend(i,Trim(SkipWords(1,arg))+CRLF));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
