 {
 ***********************************************************************
 Daq Pascal application program F740_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
 Protocol example:
 URL > %0501FA1027B%0D               | ENQ 01FA1027B CR      run forward
 URL < %0601%0D                      | ACK 01 CR             acknowledge
 URL < %0501FA10079%0D               | ENQ 01FA10079 CR      stop motor
 URL < %0601%0D                      | ACK 01 CR             acknowledge
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @CheckSumm s   - Calculate checksumm of string s
| @CheckSummUrl s - Checksumm of string URL_DECODE(s)
| @CheckSummHex s - Checksumm of string HEX_DECODE(s)
| @ComSend s     - Send to COM string s
| @ComSendUrl s  - Send to COM string URL_DECODE(s)
| @ComSendHex s  - Send to COM string HEX_DECODE(s)
| @Polling n     - 0/1 enable polling
|********************************************************
[]
 }
program F740_DRV;                { Mitsubishi FR-F740 Driver        }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 TimeOutMin        = 10;         { Minimal timeout value            }
 TimeOutDef        = 150;        { Default timeout value            }
 TimeOutMax        = 60000;      { Maximal timeout value            }
 TimeGapDef        = 10;         { Default TimeGap                  }
 MaxStrLeng        = 16384;      { Maximal string length for safety }
 MaxComLeng        = 255;        { Maximal data length for ComRead  }
 MaxCmdNum         = 31;         { Maximal command number           }
 MinInpFreq        = 0;          { Minimal input frequency, Hz  0   }
 MaxInpFreq        = 60;         { Maximal input frequency, Hz  400 }
 MinColFreq        = 25;         { Minimal color frequency, Hz  0   }
 MaxColFreq        = 55;         { Maximal color frequency, Hz  400 }
 updEmpty          = -1e300;     { Marker for empty update          }
 {-------------------------------- Mitsubishi FR-F740 command codes }
 cm_RdParam00      = 0;          { $00 - read parameter $00      H4 }
 cm_RdParam63      = 99;         { $63 - read parameter $63      H4 }
 cm_RdFreqRAM      = 109;        { $6D - read frequency RAM      H4 }
 cm_RdFreqEEPROM   = 110;        { $6E - read frequency EEPROM   H4 }
 cm_RdOutFreq      = 111;        { $6F - read output frequency   H4 }
 cm_RdOutCurr      = 112;        { $70 - read output current     H4 }
 cm_RdOutVolt      = 113;        { $71 - read output voltage     H4 }
 cm_RdSpecMonVal   = 114;        { $72 - read special monitor    H4 }
 cm_RdSpecMonSel   = 115;        { $73 - read spec.mon.select.   H2 }
 cm_RdFaultDef1    = 116;        { $74 - read fault definition 1 H4 }
 cm_RdFaultDef2    = 117;        { $75 - read fault definition 2 H4 }
 cm_RdFaultDef3    = 118;        { $76 - read fault definition 3 H4 }
 cm_RdFaultDef4    = 119;        { $77 - read fault definition 4 H4 }
 cm_RdStatMonEx    = 121;        { $79 - read inv.status monitor H4 }
 cm_RdStatMon      = 122;        { $7A - read inv.status monitor H2 }
 cm_RdOperMode     = 123;        { $7B - read operation mode     H4 }
 cm_RdModelId      = 124;        { $7C - read inverter type      H20}
 cm_RdCapacity     = 125;        { $7D - read capacity           H6 }
 cm_WrParam00      = 128;        { $80 - write parameter $00     H4 }
 cm_WrParam63      = 227;        { $E3 - write parameter $63     H4 }
 cm_WrFreqRAM      = 237;        { $ED - write Frequency RAM     H4 }
 cm_WrFreqEEPROM   = 238;        { $EE - write Frequency EEPROM  H4 }
 cm_WrSpecMonSel   = 243;        { $F3 - write spec.mon.select   H2 }
 cm_ClearFaults    = 244;        { $F4 - clear faults history    H4 }
 cm_RunCommandEx   = 249;        { $F9 - run command extended    H4 }
 cm_RunCommand     = 250;        { $FA - run command             H2 }
 cm_WrOperMode     = 251;        { $FB - write operation mode    H4 }
 cm_Reset          = 253;        { $FD - inverter reset $9966    H4 }
 {-------------------------------- RunCommand flags                 }
 rc_AU             = 1;          { AU current input selection       }
 rc_Forward        = 2;          { Forward rotation                 }
 rc_Reverse        = 4;          { Reverse rotation                 }
 rc_RL             = 8;          { Rotate low speed                 }
 rc_RM             = 16;         { Rotate middle speed              }
 rc_RH             = 32;         { Rotate high speed                }
 rc_RT             = 64;         { second function selection        }
 rc_MRS            = 128;        { output stop                      }
 {-------------------------------- Digital Inputs                   }
 di_RevEnd         = 0;          { Reverse end                      }
 di_FwdEnd         = 1;          { Forward end                      }
 di_TRelay         = 2;          { Thermo relay state KT            }
 {-------------------------------- Digital Outputs                  }
 do_Connect        = 0;          { Connection state                 }
 do_StatMon        = 1;          { Status Monitor                   }
 do_Power          = 2;          { Power state                      }
 do_RevEnd         = 3;          { Reverse end                      }
 do_FwdEnd         = 4;          { Forward end                      }
 do_ErrDrv         = 5;          { Driver errors                    }
 do_ErrTmt         = 6;          { Timeout errors                   }
 do_ErrFmt         = 7;          { Format errors                    }
 do_ErrNak         = 8;          { NAK errors                       }
 do_TRelay         = 9;          { Thermo relay state KT            }
 {-------------------------------- Analog Outputs                   }
 ao_InpFreq        = 0;          { Input frequency                  }
 ao_OutFreq        = 1;          { Output frequency                 }
 ao_OutCurr        = 2;          { Output current                   }
 ao_OutVolt        = 3;          { Output voltage                   }
 ao_PolRate        = 4;          { PollingRate, Hz                  }
 {-------------------------------- CheckCommand flags               }
 cc_Success        = 0;          { Success, data is good to use     }
 cc_BadSumm        = 1;          { Bad status, wrong checksumm      }
 cc_BadHead        = 2;          { Bad format, no ENQ/ACK/NAK/STX   }
 cc_BadTail        = 3;          { Bad format, no ENQ/ACK/NAK/STX   }
 cc_BadLeng        = 4;          { Bad status, small data length    }
 cc_BadAddr        = 5;          { Bad address = station number     }

type
 TTagRef   = record tag,nai,nao,ndi,ndo:Integer; val:Real; end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 F740              : record      { FR-F740 driver data              }
  Simulator        : Boolean;    { T/F=Simulation/Driver mode       }
  Com              : record      { Com port data                    }
   Port            : Integer;    { Com port number                  }
   Reply           : String;     { Com port received reply          }
   Status          : Integer;    { Com port status, see rs_XX const }
   Address         : Integer;    { Com port device address          }
   TimeOut         : Integer;    { Com port TimeOut                 }
   TimeGap         : Integer;    { Com port TimeGap                 }
   WaitQueueStart  : Real;       { Time when rs_WaitQueue started   }
   WaitReplyStart  : Real;       { Time when rs_WaitReply started   }
  end;                           {                                  }
  ErrorCode        : record      { Special error codes              }
   Tmt             : Integer;    { TimeOut detected                 }
   Fmt             : Integer;    { Check summ & Bad format detected }
   Nak             : Integer;    { Hardware error - NAK             }
  end;                           {                                  }
  TimeWaitStr      : String;     { Should be '1'                    }
  Power            : TTagRef;    { Power button                     }
  ModelId          : TTagRef;    { Model Id                         }
  InpFreq          : TTagRef;    { Wanted frequency                 }
  Polling          : TTagRef;    { Enable polling                   }
  Reverse          : TTagRef;    { Reverse command                  }
  Forward          : TTagRef;    { Forward command                  }
  RevBlk,FwdBlk    : TTagRef;    { Reverse & Forward blocking       }
  StatB            : array[0..7] of TTagRef; { StatMon bits         }
  RunCommand       : Integer;    { Command FWD/REV/STOP etc         }
  RateCounter      : Integer;    { Polling rate counter             }
  CmdTab           : record      { Command table                    }
   Count           : Integer;    { Command counter, 0..MaxCmdNum+1  }
   Current         : Integer;    { Current command index to send    }
   SentCmd         : Integer;    { Last sent command ID             }
   Enabled         : array[0..MaxCmdNum] of Boolean; {              }
   Comment         : array[0..MaxCmdNum] of String;  {              }
   CmdId           : array[0..MaxCmdNum] of Integer; {              }
  end;                           {                                  }
  Sim              : record      { Simulator data                   }
   StatMon         : Integer;    { Status monitor                   }
   OutFreq         : Real;       { Output frequency                 }
  end;                           {                                  }
 end;                            {                                  }
 CmdSound          : TTagRef;    { Sound button                     }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 {
 Xor bit on click (local version)
 }
 procedure ClickBitXorLocal(tag,XorMask:Integer);
 var nv:Integer;
 begin
  if ClickTag=tag then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Speak, i.e. send message to speech server.
 }
 procedure Speaker(msg:String);
 begin
  if iGetTag(CmdSound.tag)>0 then Speak(msg);
 end;
 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  if Length(s)>0 then begin
   UnixTooltipNotifier('text "'+s+'" ico information.ico delay 15000');
   if Pos(':',s)>0 then s:=Trim(Copy(s,Pos(':',s)+1));
   Speaker(s);
  end;
 end;
 {
 Find comma separated window list where curve is inside.
 }
 function ListWinByCurve(crv:Integer):String;
 var list,win:String; n,m,i,j,t,c:Integer;
 begin
  list:=''; win:='';
  if not IsEmptyStr(CrvName(crv)) then for m:=1 to 2 do begin
   n:=0;
   repeat
    c:=0;
    win:=ParamStr(ExtractWord(m,'CURWINNAME,TABWINNAME')+' '+Str(n)); n:=n+1;
    if not IsEmptyStr(win) then begin
     t:=ReadIniSection(Text_New,16+8+4+1,'', StrAddBrackets(win,'[',']'));
     for i:=0 to Text_NumLn(t)-1 do
     if IsSameText(ExtractWord(1,Text_GetLn(t,i)),'CurveList') then
     for j:=2 to WordCount(Text_GetLn(t,i)) do
     if IsSameText(ExtractWord(j,Text_GetLn(t,i)),CrvName(crv)) then c:=c+1;
     bNul(Text_Free(t));
     if c>0 then
     if IsEmptyStr(list) then list:=win else list:=list+','+Trim(win);
    end;
   until IsEmptyStr(win);
  end;
  ListWinByCurve:=list;
  list:=''; win:='';
 end;
 {
 Select windows where curve is inside.
 }
 procedure SelectWinByCurve(crv:Integer);
 var list:String; i:Integer;
 begin
  list:='';
  list:=ListWinByCurve(crv);
  for i:=1 to WordCountDelims(list,Dump(',')) do begin
   bNul(WinSelect(ExtractWordDelims(i,list,Dump(','))));
   bNul(WinDraw(ExtractWordDelims(i,list,Dump(','))+'|SelectCurve='+CrvName(crv)));
  end;
  list:='';
 end;
 {
 Calculate Mitsubishi protocol control check summ of Data string.
 }
 function mCheckSummInt(Data:String):Integer;
 var i,cs:Integer;
 begin
  cs:=0;
  for i:=1 to Length(Data) do cs:=cs+Ord(Data[i]);
  mCheckSummInt:=Ord(Chr(cs));
 end;
 function mCheckSummHex(Data:String):String;
 begin
  mCheckSummHex:=HexB(mCheckSummInt(Data));
 end;
 {
 Compose Body with check summ added.
 }
 function ComposeBody(Body:String):String;
 begin
  ComposeBody:=Body+mCheckSummHex(Body);
 end;
 {
 Compose command (ENQuiry) with given Addr,Code,Data.
 Addr - address, i.e. station number
 Code - command instruction code
 Data - any data to send
 }
 function ComposeENQ(Addr,Code:Integer; Data:String):String;
 begin
  ComposeENQ:=Chr(_ENQ)+ComposeBody(HexB(Addr)+HexB(Code)+F740.TimeWaitStr+Data)+Chr(_CR);
 end;
 {
 Compose ACKnowledge (success) with given Addr.
 Addr - address, i.e. station number
 }
 function ComposeACK(Addr:Integer):String;
 begin
  ComposeACK:=Chr(_ACK)+HexB(Addr)+Chr(_CR);
 end;
 {
 Compose NAK (error) with given Addr,Err.
 Addr - address, i.e. station number
 Err  - error number 0..15 or -1
 }
 function ComposeNAK(Addr,Err:Integer):String;
 begin
  if Err<0
  then ComposeNAK:=Chr(_NAK)+HexB(Addr)+                    Chr(_CR)
  else ComposeNAK:=Chr(_NAK)+HexB(Addr)+Copy(HexB(Err),2,1)+Chr(_CR);
 end;
 {
 Compose device reply (STX..ETX) with given Addr,Data.
 Addr - address, i.e. station number
 Data - any data to send
 }
 function ComposeSTX(Addr:Integer; Data:String):String;
 var Body:String;
 begin
  Body:=HexB(Addr)+Data;
  ComposeSTX:=Chr(_STX)+Body+Chr(_ETX)+mCheckSummHex(Body)+Chr(_CR);
  Body:='';
 end;
 {
 Extract data from CR-delimited message.
 Data going after ENQ/STX/ACK/NAK and Addr/Cmd.
 }
 function ExtractData(Msg:String):String;
 var hc,len,ltw:Integer;
 begin
  len:=Pos(Chr(_CR),Msg);
  ltw:=Length(F740.TimeWaitStr);
  if len<4 then hc:=_NUL else hc:=Ord(Msg[1]);
  if (hc=_STX) then ExtractData:=Copy(Msg,4,len-7) else
  if (hc=_ENQ) then ExtractData:=Copy(Msg,6+ltw,len-8-ltw) else
  if (hc=_NAK) then ExtractData:=Copy(Msg,4,len-4) else
  if (hc=_ACK) then ExtractData:=Copy(Msg,4,len-4) else
  ExtractData:='';
 end;
 {
 Extract address from CR-delimited message.
 Address is 2 chars after ENQ/STX/ACK/NAK.
 }
 function ExtractAddr(Msg:String):String;
 var hc,len:Integer;
 begin
  len:=Pos(Chr(_CR),Msg);
  if len<4 then hc:=_NUL else hc:=Ord(Msg[1]);
  if (hc=_ENQ) then ExtractAddr:=Copy(Msg,2,2) else 
  if (hc=_STX) then ExtractAddr:=Copy(Msg,2,2) else
  if (hc=_ACK) then ExtractAddr:=Copy(Msg,2,2) else
  if (hc=_NAK) then ExtractAddr:=Copy(Msg,2,2) else
  ExtractAddr:='';
 end;
 {
 Extract command from CR-delimited message.
 Command is 2 chars after ENQ and Address.
 }
 function ExtractCmnd(Msg:String):String;
 var hc,len:Integer;
 begin
  len:=Pos(Chr(_CR),Msg);
  if len<4 then hc:=_NUL else hc:=Ord(Msg[1]);
  if (hc=_ENQ) then ExtractCmnd:=Copy(Msg,4,2) else 
  ExtractCmnd:='';
 end;
 {
 Extract check summ from CR-delimited message.
 Check summ is 2 chars after ENQ/STX and before CR.
 }
 function ExtractSumm(Msg:String):String;
 var hc,len:Integer;
 begin
  len:=Pos(Chr(_CR),Msg);
  if len<4 then hc:=_NUL else hc:=Ord(Msg[1]);
  if (hc=_ENQ) then ExtractSumm:=Copy(Msg,len-2,2) else 
  if (hc=_STX) then ExtractSumm:=Copy(Msg,len-2,2) else
  ExtractSumm:='';
 end;
 {
 Calculate check summ from CR-delimited message.
 }
 function CalcCheckSumm(Msg:String):String;
 var hc,len:Integer;
 begin
  len:=Pos(Chr(_CR),Msg);
  if len<4 then hc:=_NUL else hc:=Ord(Msg[1]);
  if (hc=_ENQ) then CalcCheckSumm:=mCheckSummHex(Copy(Msg,2,len-4)) else
  if (hc=_STX) then CalcCheckSumm:=mCheckSummHex(Copy(Msg,2,len-5)) else
  CalcCheckSumm:='';
 end;
 {
 Mitsubishi checksum tester:
 mCheckSumTester('011770','30');
 mCheckSumTester('01E1107AD','F4');
 }
 procedure mCheckSumTester(s,h:String);
 begin
  if mCheckSummHex(s)=h
  then Success('CheckSum('+s+')='+mCheckSummHex(s)+' - Ok')
  else Trouble('CheckSum('+s+')='+mCheckSummHex(s)+' - Bad, must be: '+h);
 end;
 procedure F740_CheckSumTest;
 begin
  Success('CheckSumTest:');
  // Extracted from manual:
  mCheckSumTester('011770','30');
  mCheckSumTester('01E1107AD','F4');
  // Test commands:
  mCheckSumTester('01FA102','7B'); // run forward
  mCheckSumTester('01FA100','79'); // stop motor
 end;
 {
 Purge (clear) COM port, special version.
 }
 procedure PurgeComPortSpec;
 var data:String;
 begin
  data:='';
  if F740.Com.Port>0 then begin
   if ComClear then Details('ComClear');
   while ComCount>0 do begin
    data:=ComRead(MaxComLeng);
    Details('Skip COM: '+Hex_Encode(data)+' '+data);
   end;
  end;
  data:='';
 end;
 {
 Send data message to COM port.
 }
 procedure ComSendMessage(Data:String);
 begin
  if Length(Data)>0 then begin
   ViewExp('COM: $'+Hex_Encode(Data)+', '+Data);
   if not ComWrite(Data) then Trouble('Could not send "'+Data+'"');
  end;
 end;
 {
 Add command to CmdId command table.
 }
 procedure F740_AddCmdId(CmdId:Integer; Enabled:Boolean; Comment:String);
 begin
  if F740.CmdTab.Count<=MaxCmdNum then begin
   F740.CmdTab.Enabled[F740.CmdTab.Count]:=Enabled;
   F740.CmdTab.Comment[F740.CmdTab.Count]:=Comment;
   F740.CmdTab.CmdId[F740.CmdTab.Count]:=CmdId;
   F740.CmdTab.Count:=F740.CmdTab.Count+1;
   Success('F740.CmdTab['+StrFix(F740.CmdTab.Count,2,0)+']: '
          +StrFix(CmdId,3,0)+'=$'+HexB(CmdId)+', '+Str(Ord(Enabled))+', "'+Comment+'".');
  end;
 end;
 {
 Update Connect,StatMon,OutFreq,OutCurr,OutVolt
 }
 procedure UpdateConnect(t,v:Real);
 begin
  UpdateDo(do_Connect,t,v);
 end;
 procedure UpdateStatMon(t,v:Real);
 var i:Integer;
 begin
  UpdateDo(do_StatMon,t,v);
  for i:=0 to 7 do bNul(iSetTag(F740.StatB[i].tag,Ord(IsBit(v,i))));
 end;
 procedure UpdatePower(t,v:Real);
 begin
  UpdateDo(do_Power,t,v);
 end;
 procedure UpdateRevEnd(t,v:Real);
 begin
  UpdateDo(do_RevEnd,t,v);
 end;
 procedure UpdateFwdEnd(t,v:Real);
 begin
  UpdateDo(do_FwdEnd,t,v);
 end;
 procedure UpdateErrDrv(t,v:Real);
 begin
  UpdateDo(do_ErrDrv,t,v);
 end;
 procedure UpdateErrTmt(t,v:Real);
 begin
  UpdateDo(do_ErrTmt,t,v);
 end;
 procedure UpdateErrFmt(t,v:Real);
 begin
  UpdateDo(do_ErrFmt,t,v);
 end;
 procedure UpdateErrNak(t,v:Real);
 begin
  UpdateDo(do_ErrNak,t,v);
 end;
 procedure UpdateTRelay(t,v:Real);
 begin
  UpdateDo(do_TRelay,t,v);
 end;
 procedure UpdateInpFreq(t,v:Real);
 begin
  UpdateAo(ao_InpFreq,t,v);
 end;
 procedure UpdateOutFreq(t,v:Real);
 begin
  UpdateAo(ao_OutFreq,t,v);
 end;
 procedure UpdateOutCurr(t,v:Real);
 begin
  UpdateAo(ao_OutCurr,t,v);
 end;
 procedure UpdateOutVolt(t,v:Real);
 begin
  UpdateAo(ao_OutVolt,t,v);
 end;
 procedure UpdatePolRate(t,v:Real);
 begin
  UpdateAo(ao_PolRate,t,v);
 end;
 procedure UpdateModelId(s:String);
 begin
  bNul(sSetTag(F740.ModelId.tag,Trim(s)));
 end;
 {
 Enable command with given command identifier.
 }
 procedure F740_EnableCmdId(CmdId:Integer; Enabled:Boolean);
 var i:Integer;
 begin
  for i:=0 to F740.CmdTab.Count-1 do
  if F740.CmdTab.CmdId[i]=CmdId then F740.CmdTab.Enabled[i]:=Enabled;
 end;
 {
 Get comment string for given command CmdId.
 }
 function F740_CommentCmdId(CmdId:Integer):String;
 var i,j:Integer;
 begin
  i:=0; j:=-1;
  while (i<F740.CmdTab.Count) and (j<0) do begin
   if F740.CmdTab.CmdId[i]=CmdId then j:=i;
   i:=i+1;
  end;
  if j<0 then F740_CommentCmdId:='' else F740_CommentCmdId:=F740.CmdTab.Comment[j];
 end;
 {
 Reset command table and set ComStatus.
 }
 procedure F740_ResetCmd(aComStatus:Integer);
 begin
  if F740.Com.Port=0 then aComStatus:=rs_NotAvail;
  F740.CmdTab.Count:=0;
  F740.CmdTab.Current:=0;
  F740.CmdTab.SentCmd:=0;
  F740.Com.Status:=aComStatus;
  F740_AddCmdId(cm_RdModelId,     True,  'Read Model Id');
  F740_AddCmdId(cm_RdCapacity,    False, 'Read Capacity');
  F740_AddCmdId(cm_RdOutFreq,     True,  'Read Out Freq');
  F740_AddCmdId(cm_RdOutCurr,     True,  'Read Out Curr');
  F740_AddCmdId(cm_RdOutVolt,     True,  'Read Out Volt');
  F740_AddCmdId(cm_RdStatMon,     True,  'Read Status Mon');
  F740_AddCmdId(cm_RdStatMonEx,   False, 'Read Status Mon Ex');
  F740_AddCmdId(cm_RunCommand,    True,  'Run Command');
  F740_AddCmdId(cm_WrFreqRAM,     True,  'Write Freq RAM');
  F740.RateCounter:=0;
  PurgeComPortSpec;
 end;
 {
 Prepare next command to send.
 Also skips disabled commands. 
 }
 procedure F740_PrepareNextCmd(Step:Integer);
 begin
  F740.CmdTab.Current:=F740.CmdTab.Current+Step;
  if F740.CmdTab.Current<0 then F740.CmdTab.Current:=0;
  if F740.CmdTab.Current>=F740.CmdTab.Count then F740.CmdTab.Current:=0;
  Step:=0;
  while (Step<=MaxCmdNum) and not F740.CmdTab.Enabled[F740.CmdTab.Current] do begin
   F740.CmdTab.Current:=F740.CmdTab.Current+1;
   if F740.CmdTab.Current>=F740.CmdTab.Count then F740.CmdTab.Current:=0;
   Step:=Step+1;
  end;
 end;
 {
 Check Com Data input:
 cc_Sussess - Command looks good to use.
 cc_BadSumm - Control check summ is bad.
 cc_BadHead - Bad head char, ENQ/STX/ACK/NAK expected.
 cc_BadTail - Bad tail char, ETX/CR expected.
 cc_BadAddr - Bad address = station number.
 cc_BadLeng - Bad data length, too small.
 }
 function CheckComInput(Data:String;Addr:Integer):Integer;
 var Status,hc,tc,dc,leng:Integer;
 begin
  Status:=cc_Success;
  Leng:=Length(Data);
  if Leng<4 then Status:=cc_BadLeng else begin
   hc:=Ord(Data[1]); // Head char code, expected ENQ,STX,ACK,NAK
   if (hc<>_ENQ) and (hc<>_STX) and (hc<>_ACK) and (hc<>_NAK) then Status:=cc_BadHead else begin
    dc:=Ord(Data[Leng]); // Delimeter char code, expected CR
    if (dc<>_CR) then Status:=cc_BadTail else begin
     if (hc=_STX) then tc:=Ord(Data[Leng-3]) else tc:=_ETX; // Tail char code
     if (tc<>_ETX) then Status:=cc_BadTail else begin
      if (ExtractAddr(Data)<>HexB(Addr)) then Status:=cc_BadAddr else begin
       if(hc=_ENQ) then if CalcCheckSumm(Data)<>ExtractSumm(Data) then Status:=cc_BadSumm;
       if(hc=_STX) then if CalcCheckSumm(Data)<>ExtractSumm(Data) then Status:=cc_BadSumm;
      end;
     end;
    end;
   end;
  end;
  CheckComInput:=Status;
 end;

 {
 Handle "ComPort is not available"...
 }
 procedure F740_OnNotAvail;
 begin
  Details('ComPort '+Str(F740.Com.Port)+' is not available!');
 end;
 {
 Handle "RS-232 line is ready for new request"...
 If has command to send, try to send the command.
 }
 procedure F740_OnNoRequest;
 var msg:String; code:Integer;
 begin  
  msg:='';
  F740.Com.Reply:=''; // Clear Com port input buffer
  F740_PrepareNextCmd(0);  // Skip disabled commands
  if F740.CmdTab.Enabled[F740.CmdTab.Current] then begin
   F740.CmdTab.SentCmd:=F740.CmdTab.CmdId[F740.CmdTab.Current];
   // Read Status Monitor
   if (F740.CmdTab.SentCmd=cm_RdStatMon) then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,'');
   end else
   // Read Status Monitor Ex
   if (F740.CmdTab.SentCmd=cm_RdStatMonEx) then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,'');
   end else
   // Read Output Frequency
   if (F740.CmdTab.SentCmd=cm_RdOutFreq) then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,'');
   end else
   // Read Output Current
   if (F740.CmdTab.SentCmd=cm_RdOutCurr) then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,'');
   end else
   // Read Output Voltage
   if (F740.CmdTab.SentCmd=cm_RdOutVolt) then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,'');
   end else
   // Read Capacity
   if F740.CmdTab.SentCmd=cm_RdCapacity then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,'');
   end else
   // Read ModelId
   if F740.CmdTab.SentCmd=cm_RdModelId then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,'');
   end else
   // RunCommand
   if F740.CmdTab.SentCmd=cm_RunCommand then begin
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,HexB(F740.RunCommand));
   end else
   // Write Frequency to RAM
   if F740.CmdTab.SentCmd=cm_WrFreqRAM then begin
    code:=Round(100*Max(MinInpFreq,Min(MaxInpFreq,F740.InpFreq.val)));
    msg:=ComposeENQ(F740.Com.Address,F740.CmdTab.SentCmd,HexW(code));
   end else
   // Unknown
   begin
    Details('Unknown command: $'+HexB(F740.CmdTab.SentCmd));
    msg:='';
   end;
   if Length(msg)>0 then begin
    PurgeComPortSpec;
    ComSendMessage(msg);
    F740.Com.Status:=rs_WaitAnswer;
    F740.Com.WaitReplyStart:=mSecNow;
    Details(F740_CommentCmdId(F740.CmdTab.SentCmd)+' = '+Hex_Encode(msg)+' = '+msg);
   end else begin
    F740.Com.Status:=rs_NoRequest;
    F740_PrepareNextCmd(1);
   end;
  end;
  msg:='';
 end;
 {
 Handle "WaitQueue detected" event
 }
 procedure F740_OnWaitQueue;
 begin
  if mSecNow<F740.Com.WaitQueueStart+F740.Com.TimeGap
  then Details('Waiting queue...')
  else F740.Com.Status:=rs_NoRequest;
 end;
 {
 Handle "Request sent, waiting answer" state.
 }
 procedure F740_OnWaitReply;
 var Check:Integer;
 begin
  if Length(F740.Com.Reply)>MaxStrLeng-MaxComLeng then F740.Com.Reply:='';
  F740.Com.Reply:=F740.Com.Reply+ComRead(MaxComLeng);
  Details('Waiting reply... '+Hex_Encode(F740.Com.Reply)+' '+F740.Com.Reply);
  Check:=CheckComInput(F740.Com.Reply,F740.Com.Address);
  if Check=cc_Success then begin
   F740.RateCounter:=F740.RateCounter+1;
   ViewImp('COM: $'+Hex_Encode(F740.Com.Reply)+' '+F740.Com.Reply);
   UpdateConnect(time,1);
   F740.Com.Status:=rs_Answer;
  end else
  if Check=cc_BadSumm then begin
   Failure(F740.ErrorCode.Fmt,'CheckSumm error detected: '+Hex_Encode(F740.Com.Reply)+' '+F740.Com.Reply);
   F740.Com.WaitQueueStart:=mSecNow;
   F740.Com.Status:=rs_WaitQueue;
  end else
  if mSecNow>F740.Com.WaitReplyStart+F740.Com.TimeOut
  then F740.Com.Status:=rs_Timeout;
 end;
 {
 Handle "Answer received"...
 }
 procedure F740_OnGotAnswer;
 var Leng:Integer; v:Real; Data:String;
  procedure BadFormat;
  begin
   Failure(F740.ErrorCode.Fmt,'COMMAND $'+HexB(F740.CmdTab.SentCmd)+' ERROR: BAD FORMAT');
  end;
 begin
  Leng:=Length(F740.Com.Reply);
  Data:=Trim(ExtractData(F740.Com.Reply));
  // NAK error
  if Pos(Chr(_NAK),F740.Com.Reply)=1 then begin
   Failure(F740.ErrorCode.Nak,'COMMAND $'+HexB(F740.CmdTab.SentCmd)+' ERROR: NAK $'+ExtractData(F740.Com.Reply));
   F740_PrepareNextCmd(1);
  end else
  // Read Status Monitor
  if F740.CmdTab.SentCmd=cm_RdStatMon then begin
   v:=rVal('$'+Data); Details('StatMon $'+Data);
   if IsNan(v) then BadFormat else UpdateStatMon(time,v);
   F740_PrepareNextCmd(1);
  end else
  // Read Status Monitor Ex
  if F740.CmdTab.SentCmd=cm_RdStatMonEx then begin
   v:=rVal('$'+Data); Details('StatMon $'+Data);
   if IsNan(v) then BadFormat else UpdateStatMon(time,v);
   F740_PrepareNextCmd(1);
  end else
  // Read Output Frequency
  if F740.CmdTab.SentCmd=cm_RdOutFreq then begin
   v:=0.01*rVal('$'+Data); Details('OutFreq $'+Data);
   if IsNan(v) then BadFormat else UpdateOutFreq(time,v);
   F740_PrepareNextCmd(1);
  end else
  // Read Output Current
  if F740.CmdTab.SentCmd=cm_RdOutCurr then begin
   v:=0.01*rVal('$'+Data); Details('OutCurr $'+Data);
   if IsNan(v) then BadFormat else UpdateOutCurr(time,v);
   F740_PrepareNextCmd(1);
  end else
  // Read Output Voltage
  if F740.CmdTab.SentCmd=cm_RdOutVolt then begin
   v:=0.1*rVal('$'+Data); Details('OutVolt $'+Data);
   if IsNan(v) then BadFormat else UpdateOutVolt(time,v);
   F740_PrepareNextCmd(1);
  end else
  // Read Capacity
  if F740.CmdTab.SentCmd=cm_RdCapacity then begin
   if IsEmptyStr(Data) then BadFormat else Details('Capacity '+Data);
   F740_PrepareNextCmd(1);
  end else
  // Read ModelId
  if F740.CmdTab.SentCmd=cm_RdModelId then begin
   UpdateModelId(Data);
   if IsEmptyStr(Data) then BadFormat else
   F740_EnableCmdId(F740.CmdTab.SentCmd,False);
   F740_PrepareNextCmd(1);
  end else
  // RunCommand
  if F740.CmdTab.SentCmd=cm_RunCommand then begin
   F740_EnableCmdId(F740.CmdTab.SentCmd,False);
   F740_PrepareNextCmd(1);
  end else
  // Write Frequency RAM
  if F740.CmdTab.SentCmd=cm_WrFreqRAM then begin
   F740_EnableCmdId(F740.CmdTab.SentCmd,False);
   F740_PrepareNextCmd(1);
  end else
  // Unknown
  begin
   Trouble('Unknown command $'+HexB(F740.CmdTab.SentCmd));
   F740_PrepareNextCmd(1);
  end;
  F740.Com.WaitQueueStart:=mSecNow;
  F740.Com.Status:=rs_WaitQueue;
  Data:='';
 end;
 {
 Handle "TimeOut detected" event
 }
 procedure F740_OnTimeOut;
 begin
  F740.Com.Reply:='';
  UpdateConnect(time,0);
  UpdateModelId('TIMEOUT');
  F740_ResetCmd(rs_NoRequest);
  Failure(F740.ErrorCode.Tmt,'TimeOut('+Str(F740.Com.TimeOut)+') detected!');
 end;
 {
 Enforce stop motor.
 }
 procedure F740_EnforceRunCommand(RunCommand,Delay,nTry:Integer);
 var i:Integer;
 begin
  bNul(Sleep(Delay));
  PurgeComPortSpec;
  for i:=1 to nTry do begin
   F740.RunCommand:=RunCommand;
   ComSendMessage(ComposeENQ(F740.Com.Address,cm_RunCommand,HexB(F740.RunCommand)));
   bNul(Sleep(Delay));
  end;
 end;
 {
 Handle command Cmd with argument Arg coming from Com port in simulation mode.
 }
 procedure Sim_Handle(Cmd,Arg:String);
 var CmdId:Integer; v:Real;
 begin
  if not IsEmptyStr(Cmd) then begin
   CmdId:=Val('$'+Cmd);
   if CmdId=cm_RdStatMon then begin
    v:=F740.Sim.StatMon;
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    ComSendMessage(ComposeSTX(F740.Com.Address,HexW(Round(v))));
   end else
   if CmdId=cm_RdStatMonEx then begin
    v:=F740.Sim.StatMon;
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    ComSendMessage(ComposeSTX(F740.Com.Address,HexW(Round(v))));
   end else
   if CmdId=cm_RdOutFreq then begin
    v:=Max(0,(F740.Sim.OutFreq+0.1*random(-1,1))/0.01);
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    ComSendMessage(ComposeSTX(F740.Com.Address,HexW(Round(v))));
   end else
   if CmdId=cm_RdOutCurr then begin
    v:=Max(0,(F740.Sim.OutFreq/MaxInpFreq*10+0.1*random(-1,1))/0.01);
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    ComSendMessage(ComposeSTX(F740.Com.Address,HexW(Round(v))));
   end else
   if CmdId=cm_RdOutVolt then begin
    v:=Max(0,(F740.Sim.OutFreq/MaxInpFreq*380+0.1*random(-1,1))/0.1);
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    ComSendMessage(ComposeSTX(F740.Com.Address,HexW(Round(v))));
   end else
   if CmdId=cm_RdCapacity then begin
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    ComSendMessage(ComposeSTX(F740.Com.Address,RightPad('17',20,' ')));
   end else
   if CmdId=cm_RdModelId then begin
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    ComSendMessage(ComposeSTX(F740.Com.Address,RightPad('SIMULATOR',20,' ')));
   end else
   if CmdId=cm_RunCommand then begin
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    F740.Sim.StatMon:=Val('$'+Trim(Arg));
    ComSendMessage(ComposeACK(F740.Com.Address));
   end else
   if CmdId=cm_WrFreqRAM then begin
    Details(F740_CommentCmdId(CmdId)+' = '+Arg);
    F740.Sim.OutFreq:=Val('$'+Trim(Arg))*0.01;
    ComSendMessage(ComposeACK(F740.Com.Address));
   end else
   Trouble('Unknown command: $'+Cmd+','+Arg);
  end;
 end;
 {
 Process commands coming from COM port in simulation mode.
 }
 procedure Sim_Process(Data:String);
 var Leng:Integer;
 begin
  Leng:=Length(Data);
  if Leng>0 then begin
   ViewImp('COM: $'+Hex_Encode(Data)+' '+Data);
   if CheckComInput(Data,F740.Com.Address)=cc_Success then begin
    Details('Received Command="'+Data+'"');
    if Pos(Chr(_ENQ),Data)=1 then Sim_Handle(ExtractCmnd(Data),ExtractData(Data));
   end;
  end;
 end;
 {
 Clear all F740 params.
 }
 procedure F740_Clear;
 var i:Integer;
 begin
  F740.Simulator:=False;
  F740.Com.Port:=0;
  F740.Com.Reply:='';
  F740.Com.Status:=0;
  F740.RunCommand:=0;
  F740.Com.TimeOut:=0;
  F740.TimeWaitStr:='';
  F740.Com.WaitQueueStart:=0;
  F740.Com.WaitReplyStart:=0;
  F740.CmdTab.Count:=0;
  F740.CmdTab.Current:=0;
  F740.CmdTab.SentCmd:=0;
  for i:=0 to MaxCmdNum do begin
   F740.CmdTab.Enabled[i]:=False;
   F740.CmdTab.Comment[i]:='';
   F740.CmdTab.CmdId[i]:=0;
  end;
 end;
 {
 Initialize F740 params.
 }
 procedure F740_Init;
 var i:Integer;
 begin
  F740_Clear;
  F740.TimeWaitStr:='1';
  F740_CheckSumTest;
  F740.Simulator:=Val(ReadIni('Simulator'))>0;
  Success('Simulator = '+Str(Ord(F740.Simulator)));
  F740.Com.Address:=Val(ReadIni('Address'));
  Success('Address = '+Str(F740.Com.Address));
  F740.Com.TimeOut:=Val(ReadIni('TimeOut'));
  if F740.Com.TimeOut<=0 then F740.Com.TimeOut:=TimeOutDef;
  F740.Com.TimeOut:=Round(Max(TimeOutMin,Min(TimeOutMax,F740.Com.TimeOut)));
  Success('TimeOut = '+Str(F740.Com.TimeOut));
  F740.Com.TimeGap:=Val(ReadIni('TimeGap'));
  if F740.Com.TimeGap<=0 then F740.Com.TimeGap:=TimeGapDef;
  Success('TimeGap = '+Str(F740.Com.TimeGap));
  F740.Com.Port:=Val(ReadIni('ComPort'));
  if F740.Com.Port>0 then begin
   if ComOpen('[SerialPort-COM'+Str(F740.Com.Port)+']')
   then Success('ComPort '+Str(F740.Com.Port)+' opened') else begin
    Trouble('Could not open ComPort '+Str(F740.Com.Port));
    F740.Com.Port:=0;
   end;
  end else begin
   Trouble('Invalid ComPort='+Str(F740.Com.Port));
   F740.Com.Port:=0;
  end;
  F740_ResetCmd(rs_NoRequest);
  {
  Initialize special errors...
  }
  F740.ErrorCode.Tmt:=RegisterErr(DevName+' TimeOut');
  F740.ErrorCode.Fmt:=RegisterErr(DevName+' BadFormat/CheckSum');
  F740.ErrorCode.Nak:=RegisterErr(DevName+' Hardware/NAK');
  {
  Initialize tags...
  Some tags MUST be co-named with curves!
  }
  InitTag(F740.Power.tag,      ReadIni('tagF740')+'.Power',   -1);
  InitTag(F740.ModelId.tag,    ReadIni('tagF740')+'.ModelId', -3);
  InitTag(F740.InpFreq.tag,    ReadIni('tagF740')+'.InpFreq', -2);
  InitTag(F740.Polling.tag,    ReadIni('tagF740')+'.Polling', -1);
  InitTag(F740.Reverse.tag,    ReadIni('tagF740')+'.Reverse', -1);
  InitTag(F740.Forward.tag,    ReadIni('tagF740')+'.Forward', -1);
  InitTag(F740.RevBlk.tag,     ReadIni('tagF740')+'.RevBlk',  -1);
  InitTag(F740.FwdBlk.tag,     ReadIni('tagF740')+'.FwdBlk',  -1);
  InitTag(CmdSound.tag,        ReadIni('tagCmdSound'),        -1);
  for i:=0 to 7 do InitTag(F740.StatB[i].tag, ReadIni('tagF740')+'.StatB'+Str(i), -1);
  bNul(sSetTag(F740.ModelId.tag,''));
  F740.Power.val:=updEmpty;
  F740.ModelId.val:=updEmpty;
  F740.InpFreq.val:=updEmpty;
  F740.Polling.val:=updEmpty;
  F740.Reverse.val:=updEmpty;
  F740.Forward.val:=updEmpty;
  F740.RevBlk.val:=updEmpty;
  F740.FwdBlk.val:=updEmpty;
  UpdateErrDrv(time,GetErrCount(-1));
  UpdateErrTmt(time,GetErrCount(F740.ErrorCode.Tmt));
  UpdateErrFmt(time,GetErrCount(F740.ErrorCode.Fmt));
  UpdateErrNak(time,GetErrCount(F740.ErrorCode.Nak));
 end;
 {
 Close F740 device.
 }
 procedure F740_Close;
 begin
  if F740.Com.Port>0 then begin
   if ComClose
   then Success('ComPort '+Str(F740.Com.Port)+' closed.')
   else Trouble('Could not close ComPort '+Str(F740.Com.Port)+'.');
   F740.Com.Port:=0;
  end;
 end;
 {
 Poll F740 device.
 }
 procedure F740_Poll;
 var Polling,Forward,Reverse,Power:Boolean; v:Real;
     WantedStatus,ClickCurve,RevEnd,FwdEnd,RevBlk,FwdBlk,TRelay:Integer;
 begin
  //
  // Driver/simulator separation
  //
  if F740.Simulator then begin
   if F740.Com.Port>0 then
   Sim_Process(ComRead(MaxComLeng));
  end else begin
   //
   // Update error counters
   //
   UpdateErrDrv(time,GetErrCount(-1));
   UpdateErrTmt(time,GetErrCount(F740.ErrorCode.Tmt));
   UpdateErrFmt(time,GetErrCount(F740.ErrorCode.Fmt));
   UpdateErrNak(time,GetErrCount(F740.ErrorCode.Nak));
   //
   // Update Power, RevEnd, FwdEnd. Apply blocking.
   //
   RevBlk:=iGetTag(F740.RevBlk.tag);
   FwdBlk:=iGetTag(F740.FwdBlk.tag);
   RevEnd:=Round(DiWord(di_RevEnd,1));
   FwdEnd:=Round(DiWord(di_FwdEnd,1));
   TRelay:=Round(DiWord(di_TRelay,1));
   if abs(RevBlk*RevEnd)>0 then begin
    bNul(iSetTag(F740.Reverse.tag,0));
   end;
   if abs(FwdBlk*FwdEnd)>0 then begin
    bNul(iSetTag(F740.Forward.tag,0));
   end;
   Power:=iGetTag(F740.Power.tag)<>0;
   UpdatePower(time,Ord(Power));
   UpdateRevEnd(time,RevEnd);
   UpdateFwdEnd(time,FwdEnd);
   UpdateTRelay(time,TRelay);
   if not Power then begin
    bNul(iSetTag(F740.Polling.tag,0));
    bNul(iSetTag(F740.Forward.tag,0));
    bNul(iSetTag(F740.Reverse.tag,0));
   end;
   //
   // Actions on polling on/off
   //
   Polling:= iGetTag(F740.Polling.tag)<>0;
   if F740.Polling.val<>Ord(Polling) then begin
    F740.Polling.val:=Ord(Polling);
    if Polling then begin
     F740_ResetCmd(rs_NoRequest);
     UpdateModelId('NONE');
     UpdateConnect(time,0);
     UpdateStatMon(time,0);
     UpdatePolRate(time,0);
     UpdateOutFreq(time,0);
     UpdateOutCurr(time,0);
     UpdateOutVolt(time,0);
    end else begin
     bNul(sSetTag(F740.ModelId.tag,'OFFLINE'));
     UpdateConnect(time,0);
     UpdateStatMon(time,0);
     UpdatePolRate(time,0);
     UpdateOutFreq(time,0);
     UpdateOutCurr(time,0);
     UpdateOutVolt(time,0);
     F740_EnforceRunCommand(0,50,3);
    end;
   end;
   //
   // Calculate & apply wanted status
   // Forward && Reverse is forbidden
   //
   WantedStatus:=0;
   Forward:=iGetTag(F740.Forward.tag)<>0;
   Reverse:=iGetTag(F740.Reverse.tag)<>0;
   if Forward and Reverse then begin
    bNul(iSetTag(F740.Forward.tag,0));
    bNul(iSetTag(F740.Reverse.tag,0));
    Forward:=False;
    Reverse:=False;
   end;
   if Forward then begin
    bNul(iSetTag(F740.Reverse.tag,0));
    WantedStatus:=WantedStatus+rc_Forward;
   end;
   if Reverse then begin
    bNul(iSetTag(F740.Forward.tag,0));
    WantedStatus:=WantedStatus+rc_Reverse;
   end;
   if F740.RunCommand<>WantedStatus then begin
    F740_EnableCmdId(cm_RunCommand,True);
    F740.RunCommand:=WantedStatus;
   end;
   //
   // Set inverter frequency if needed
   //
   if rGetTag(F740.InpFreq.tag)<MinInpFreq then bNul(rSetTag(F740.InpFreq.tag,MinInpFreq));
   if rGetTag(F740.InpFreq.tag)>MaxInpFreq then bNul(rSetTag(F740.InpFreq.tag,MaxInpFreq));
   if ShouldRefresh(F740.InpFreq.val,rGetTag(F740.InpFreq.tag))>0
   then F740_EnableCmdId(cm_WrFreqRAM,True);
   UpdateInpFreq(time,F740.InpFreq.val);
   //
   // COM port polling
   //
   if Polling then begin
    if F740.Com.Port>0 then begin
     if F740.Com.Status=rs_NoRequest  then F740_OnNoRequest else
     if F740.Com.Status=rs_WaitQueue  then F740_OnWaitQueue else
     if F740.Com.Status=rs_WaitAnswer then F740_OnWaitReply else
     if F740.Com.Status=rs_Answer     then F740_OnGotAnswer else
     if F740.Com.Status=rs_TimeOut    then F740_OnTimeOut   else F740_OnNotAvail;
    end;
   end else begin
    bNul(iSetTag(F740.Forward.tag,0));
    bNul(iSetTag(F740.Reverse.tag,0));
    UpdateConnect(time,0);
    UpdateStatMon(time,0);
    UpdatePolRate(time,0);
    UpdateOutFreq(time,0);
    UpdateOutCurr(time,0);
    UpdateOutVolt(time,0);
   end;
   //
   // Polling rate calculator
   //
   if SysTimer_Pulse(1000)>0 then begin
    UpdatePolRate(time,F740.RateCounter);
    F740.RateCounter:=0;
   end;
   //
   // Edit tags...
   //
   if EditState=ef_Done then begin
    {
    Warning.
    }
    if IsSameText(ExtractWord(1,edit('?ans 0')),'Warning') then begin
     sNul(Edit(''));
    end;
    {
    Information.
    }
    if IsSameText(ExtractWord(1,edit('?ans 0')),'Information') then begin
     sNul(Edit(''));
    end;
    {
    Accept tag edit
    }
    CheckEditTagUpdate(F740.InpFreq.tag,MinInpFreq,MaxInpFreq);
   end;
   if EditState=ef_Done then begin
    Problem('Unknown tag edition!');
    sNul(Edit(''));
   end;
   if iAnd(EditState,ef_ErrorFound)<>0 then begin
    Problem('Dialog error detected!');
    sNul(Edit(''));
   end;
   //
   // Handle left button clicks...
   //
   if ClickButton=1 then begin
    {
    Control buttons
    }
    ClickBitXorLocal(F740.Power.tag,1);
    ClickBitXorLocal(F740.Polling.tag,1);
    ClickBitXorLocal(F740.Reverse.tag,1);
    ClickBitXorLocal(F740.Forward.tag,1);
    ClickBitXorLocal(F740.RevBlk.tag,1);
    ClickBitXorLocal(F740.FwdBlk.tag,1);
    {
    Plot & Tab windows
    }
    ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
    if ClickCurve<>0 then SelectWinByCurve(ClickCurve);
    {
    Set frequency by click position
    }
    if IsSameText(ExtractFileExt(ClickSensor),'.COLOR_FREQ') then begin
     v:=rEvalDef(Str(MinColFreq)+'+'+Str(MaxColFreq-MinColFreq)+'*('
                +ExtractWord(1,ClickParams('Where'))+'-'
                +ExtractWord(1,ClickParams('Bounds'))+')/('
                +ExtractWord(3,ClickParams('Bounds'))+'-'
                +ExtractWord(1,ClickParams('Bounds'))+')'
                ,rGetTag(F740.InpFreq.tag));
     bNul(rSetTag(F740.InpFreq.tag,Round(v)));
     bNul(Voice(snd_Press));
    end;
    {
    Start edit tags
    }
    if ClickTag=F740.INPFREQ.tag then StartEditTag(ClickTag,URL_Decode(ClickParams('Hint')));
   end;
   //
   // Handle right button clicks...
   //
   if ClickButton=2 then begin
    SensorHelp(Url_Decode(ClickParams('Hint')));
   end;
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  F740_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  F740_Init;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  F740_Close;
  F740_Clear;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  F740_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String;
 begin
  ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   @CheckSumm 0123
   }
   if IsSameText(cmd,'@CheckSumm') then begin
    if not IsEmptyStr(arg) then Success('$'+mCheckSummHex(arg));
    Data:='';
   end else
   {
   @CheckSummUrl %01%23
   }
   if IsSameText(cmd,'@CheckSummUrl') then begin
    if not IsEmptyStr(arg) then Success('$'+mCheckSummHex(Url_Decode(arg)));
    Data:='';
   end else
   {
   @CheckSummHex 01AF
   }
   if IsSameText(cmd,'@CheckSummHex') then begin
    if not IsEmptyStr(arg) then Success('$'+mCheckSummHex(Hex_Decode(arg)));
    Data:='';
   end else
   {
   @ComSend 0123
   }
   if IsSameText(cmd,'@ComSend') then begin
    if not IsEmptyStr(arg) then ComSendMessage(arg);
    Data:='';
   end else
   {
   @ComSendUrl %01%23
   }
   if IsSameText(cmd,'@ComSendUrl') then begin
    if not IsEmptyStr(arg) then ComSendMessage(Url_Decode(arg));
    Data:='';
   end else
   {
   @ComSendHex 01AF
   }
   if IsSameText(cmd,'@ComSendHex') then begin
    if not IsEmptyStr(arg) then ComSendMessage(Hex_Decode(arg));
    Data:='';
   end else
   {
   @Polling 1
   }
   if IsSameText(cmd,'@Polling') then begin
    bNul(iSetTag(F740.Polling.tag,iValDef(arg,iGetTag(F740.Polling.tag))));
    Success(cmd+'='+Str(iGetTag(F740.Polling.tag)));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
