 {
 ***********************************************************************
 Daq Pascal application program I7018_drv.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Poll - Enable/Disable Polling
| @SetCfg NNTTCCFF - Set Configuration
|                    NN - New Address (0x01..0xFF); TT - Input Type (0x00..0x06, 0x0E..0x18);
|                    CC - Baudrate (0x03..0x0A); FF - Data Format. See to Manual for i-7018
| @ReadCfg - Read Module Configuration
| @ReadVer - Read Firmware Version
| @ReadName - Read Module Name
| @SetName name - Set Module Name (name = Name, Maximum Length = 6)
| @ReadAI - Read Analog Input Data
| @SetMesureAI VV  - Enable Measurement AI (VV - 0x00..0xFF)
| @ReadMesureAI - Read Enable Measurement AI
| @CalibrateAllow 0/1 - Allow Calibration (0-disallow/1-allow)
| @CalibrateRange - Execute Range Calibration
| @CalibrateZero - Execute Zero Calibration
| @ReadTCJ - Read Cold Junction Temperature
| @SetShiftTCJ data - Set Shift TCJ (data - -40.96°С..+40.96°С)
| @ReadWDStatus - Read WatchDog Status
| @ReadWDTimeout - Read WatchDog WatchDog Timeout
| @SetWDStatus T Enabled - Set WatchDog Status (T - Timeout 0.1 - 25.5 s, Enbaled - 0 or 1)
| @ResetWDStatus - Reset WatchDog Status
| @MenuTools - Open Menu Tools
|********************************************************
[]
 }
program I7018_drv;
const
 {------------------------------}{ Declare uses program constants:                }
 {$I _con_StdLibrary}            { Include all Standard constants,                }
 {------------------------------}{ And add User defined constants:                }
 {$I _con_NetRS485}              { Include RS485Proxy   constants.  }
 cm_SetModuleConf     = 1;       { ID CMD: Set Module Configuration               }
 cm_SetModuleName     = 2;       { ID CMD: Set Module Name                        }
 cm_SetMesureAI       = 3;       { ID CMD: Set Enable Mesure Channel              }
 cm_SetShiftTCJ       = 4;       { ID CMD: Set Shift TCJ                          }
 cm_CalAllow          = 5;       { ID CMD: Calibrate Allow                        }
 cm_CalRange          = 6;       { ID CMD: Perform Calibration Range              }
 cm_CalZero           = 7;       { ID CMD: Perform Calibration Zero               }
 cm_HostIsOk          = 8;       { ID CMD: HostIsOk                               }
 cm_SetWDTStatus      = 9;       { ID CMD: Set WATCHDOG Status and Timeout        }
 cm_RstWDTStatus      = 10;      { ID CMD: Reset WATCHDOG                         }
 cm_ReadModuleConf    = 11;      { ID CMD: Read Module Configuration              }
 cm_ReadFirmwarVer    = 12;      { ID CMD: Read Firmware module                   }
 cm_ReadModuleName    = 13;      { ID CMD: Read Module Name                       }
 cm_ReadAnalogInp     = 14;      { ID CMD: Read Analog Inputs                     }
 cm_ReadMesureAI      = 15;      { ID CMD: Read Enale Mesure Channel              }
 cm_ReadTCJ           = 16;      { ID CMD: Read Temperature Cold Junction         }
 cm_ReadWDTStatus     = 17;      { ID CMD: Read WATCHDOG Status                   }
 cm_ReadWDTimeOut     = 18;      { ID CMD: Read WATCHDOG Timeout                  }
 MaxCmdNum            = 18;      { Maximal count of commands                      }
 do_state             = 0;       { DO : State                                     }
 do_ais               = 1;       { DO : Active AI channel                         }
 do_rx                = 2;       { DO : Recieve rate                              }
 do_tx                = 3;       { DO : Transmite rate                            }
 ao_ai0               = 0;       { AO : AI0 Data                                  }
 ao_ai1               = 1;       { AO : AI1 Data                                  }
 ao_ai2               = 2;       { AO : AI2 Data                                  }
 ao_ai3               = 3;       { AO : AI3 Data                                  }
 ao_ai4               = 4;       { AO : AI4 Data                                  }
 ao_ai5               = 5;       { AO : AI5 Data                                  }
 ao_ai6               = 6;       { AO : AI6 Data                                  }
 ao_ai7               = 7;       { AO : AI7 Data                                  }
 ao_tcj               = 8;       { AO : TCJ Data                                  }
 ao_shift             = 9;       { AO : TCJ Shift Data                            }
 ao_cal_ai0           = 10;      { AO : Calibrated AI0 Data                       }
 ao_cal_ai1           = 11;      { AO : Calibrated AI1 Data                       }
 ao_cal_ai2           = 12;      { AO : Calibrated AI2 Data                       }
 ao_cal_ai3           = 13;      { AO : Calibrated AI3 Data                       }
 ao_cal_ai4           = 14;      { AO : Calibrated AI4 Data                       }
 ao_cal_ai5           = 15;      { AO : Calibrated AI5 Data                       }
 ao_cal_ai6           = 16;      { AO : Calibrated AI6 Data                       }
 ao_cal_ai7           = 17;      { AO : Calibrated AI7 Data                       }
 stt_Disable          = 0;       { Status flag: Disable                           }
 stt_Work             = 1;       { Status flag: Work                              }
 stt_TimeOut          = 2;       { Status flag: TimeOut                           }
 stt_Refused          = 3;       { Status flag: Refused                           }
 stt_WatchDog         = 4;       { Status flag: WatchDog event                    }
 stt_OpenDet          = 5;       { Status flag: Open Det                          }
 stt_Simulator        = 6;       { Status flag: Simulator                         }
 MaxChannelNum        = 7;       { Device Channels Count                          }
 defTypeCode          = 0;       { Default AI Type                                }
 defBaudRate          = 10;      { Default Baudrate                               }
 defDataFormat        = 0;       { Default Data Format                            }
 minLowValidTypeCode  = 0;       { Minimal Value Low Range Input Type             }
 maxLowValidTypeCode  = 6;       { Maximal Value Low Range Input Type             }
 minHighValidTypeCode = 14;      { Minimal Value High Range Input Type            }
 maxHighValidTypeCode = 24;      { Maximal Value High Range Input Type            }
 le_req = '0D';
 dfEngineering = 0;     // Use engineering units
 dfPercentFSR  = 1;     // Use % of Full scale range
 dfHexadecimal = 2;     // Use hexadecimal data
 bitCheckSum   = 6;     // Bit Number State CheckSum
 bitReject     = 7;     // Bit Number of Rejector
 NoCalibr      = 0;     // Do not calibrate
 PuCalibr      = 1;     // Calibrate in mode Phisical Unit
 TcCalibr      = 2;     // Calibrate in mode Termocouple
 DimDeadline        = 5000;       { Detect DIM server is dead           }

type
 {------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}            { Include all Standard types,      }
 {------------------------------}{ And add User defined types:      }

var
 {------------------------------}{ Declare uses program variables:      }
 {$I _var_StdLibrary}            { Include all Standard variables,      }
 {------------------------------}{ And add User defined variables:      }
 {$I _var_NetRS485}              { Include RS485Proxy   variables.      }
 cmd_DimCmdMy      : Integer;    { System Command @DIMCMDMY                 }
 cmd_DimTagUpdate  : Integer;    { System Command @DimTagUpdate             }
 cmd_RsReply       : Integer;    { Command @RS485.Reply                     }
 cmd_RsPoll        : Integer;    { Command @RS485.Poll                      }
 cmd_RsRefuse      : Integer;    { Command @RS485.Refuse                    }
 cmd_RsTimeout     : Integer;    { Command @RS485.Timeout                   }
 cmd_RsClrSumm     : Integer;    { Command @ClearRS485Sum                   }
 cmd_Poll          : Integer;    { Console Command @Poll                    }
 cmd_SetCfg        : Integer;    { Console Command @SetCfg NNTTCCFF         }
 cmd_ReadCfg       : Integer;    { Console Command @ReadCfg                 }
 cmd_ReadVer       : Integer;    { Console Command @ReadVer                 }
 cmd_ReadName      : Integer;    { Console Command @ReadName                }
 cmd_SetName       : Integer;    { Console Command @SetName NAME            }
 cmd_ReadAI        : Integer;    { Console Command @ReadAI                  }
 cmd_SetMesAI      : Integer;    { Console Command @SetMesureAI VV          }
 cmd_ReadMesAI     : Integer;    { Console Command @ReadMesureAI            }
 cmd_CalAllow      : Integer;    { Console Command @CalibrateAllow 0/1      }
 cmd_CalRange      : Integer;    { Console Command @CalibrateRange          }
 cmd_CalZero       : Integer;    { Console Command @CalibrateZero           }
 cmd_ReadTCJ       : Integer;    { Console Command @ReadTCJ                 }
 cmd_SetShiftTCJ   : Integer;    { Console Command @SetShiftTCJ data        }
 cmd_ReadWDStatus  : Integer;    { Console Command @ReadWDStatus            }
 cmd_ReadWDTimeout : Integer;    { Console Command @ReadWDTimeout           }
 cmd_SetWDStatus   : Integer;    { Console Command @SetWDStatus T Enabled   }
 cmd_RstWDStatus   : Integer;    { Console Command @ResetWDStatus           }
 cmd_MenuTools     : Integer;    { Console Command @MenuTools               }
 ColorNorm         : Integer;    { Color in normal state: lime,aqua    }
 ColorWarn         : Integer;    { Color in warning state: yellow      }
 I7018             : record      { I7018 Data                               }
  Simulator         : Boolean;   { Simulator Mode                           }
  WDReconnect       : Boolean;   { WatchDog AutoReconnect                   }
  isCheckSum        : Boolean;   { CheckSum Enable                          }
  isOpenDet         : Boolean;   { Open Det event                           }
  isCalibr          : Boolean;   { Calibrate?                               }
  CalibrMode        : Integer;   { Calibrate Mode                           }
  Shortcut          : Integer;   { Channel Number of Shortcut               }
  ChNumber          : Integer;   { Channel Number                           }
  ChState           : Boolean;   { Channel State                            }
  StartPoll         : Integer;   { Start Poll                               }
  SelfId            : String;
  SERVID            : TTagRef;   { Server Identifier pid@hostname           }
  POLL              : TTagRef;   { Tag: Poll                                }
  CLOCK             : TTagRef;   { Server Date-Time                         }
  RX                : TTagRef;   { Tag: Recieve rate data                   }
  TX                : TTagRef;   { Tag: Transmite rate data                 }
  STATE             : TTagRef;   { Tag: State of device                     }
  AI                : array [0..MaxChannelNum] of TTagRef;{  Tags: Analog Input 0..7 }
  CRV_AI            : array [0..MaxChannelNum] of Integer;{ Curve: Analog Input 0..7}
  CAL               : record
   AI                : array [0..MaxChannelNum] of TTagRef;{  Tags: Calibrate Analog Input 0..7 }
   CRV_AI            : array [0..MaxChannelNum] of Integer;{ Curve: Calibrate Analog Input 0..7 }
  end;
  TCJ               : TTagRef;   { Tag: TCJ DATA                            }
  SHIFT             : TTagRef;   { Tag: Shift TCJ                           }
  AIS               : TTagRef;   { Tag: AI Enabled                          }
  DEVCAL            : record
   AUTH              : TTagRef;  { Tag: Checkbox Authorized Calibration     }
   ZERO              : TTagRef;  { Tag: Button Zero Calibration             }
   RANGE             : TTagRef;  { Tag: Button Range Calibration            }
   TimeSendZero      : Real;     { Time Send Cmd to Device Perform Zero  Calibr }
   TimeSendRange     : Real;     { Time Send Cmd to Device Perform Range Calibr }
  end;
  NAME              : TTagRef;   { Tag: Name of device                  }
  VERSION           : TTagRef;   { Tag: Firmware Version of device      }
  ADDRESS           : TTagRef;   { Tag: Address of device               }
  BAUDRATE          : TTagRef;   { Tag: Baudrate of device              }
  CHECKSUM          : TTagRef;   { Tag: Checksum Enabled?               }
  PARITY            : TTagRef;   { Tag: Parity parameter                }
  INPTYPE           : TTagRef;   { Tag: Inpute Type                     }
  REJFILTER         : TTagRef;   { Tag: Rejector                        }
  DATATYPE          : TTagRef;   { Tag: Data Type (Data Formate)        }
  WATCHDOG          : record
   TIMEOUT           : TTagRef;   { Tag: WatchDog TimeOut                }
   STATUS            : TTagRef;   { Tag: WatchDog Status                 }
   ENABLE            : TTagRef;   { Tag: Enable/Disable WatchDog button  }
  end;
  CmdTab         : Record
   Num            : Integer;      { Current running command number          }
   Enabled        : array [1..MaxCmdNum] of Boolean; { Enable polling       }
   //CmdChar        : array [1..MaxCmdNum] of Char;    { Command Char         }
   Cmd            : array [1..MaxCmdNum] of String;  { Command Mode         }
   Acronym        : array [1..MaxCmdNum] of String;  { Accronym of command  }
   OpData         : array [1..MaxCmdNum] of String;  { Operation Data       }
   Comment        : array [1..MaxCmdNum] of String;  { Comment to command   }
   PollCnt        : Integer;                         { Poll Counter         }
  end;
  RS485             : Record
   Port              : Integer;       { Logical Port on &RS485Proxy         }
   UnitId            : Integer;       { Modbus unit id                      }
   Timeout           : Integer;       { Modbus timeout, ms                  }
   Polling           : Integer;       { Modbus polling period, ms           }
   Deadline          : Integer;       { Modbus deadline time, ms            }
   DelayOnStart      : Integer;       { Command cycle delay on start        }
   Poll              : record
    ref               : Integer;          { Last sent device reference      }
    cid               : Integer;          { Last sent command id            }
    tim               : Real;             { Last polling time, ms           }
    port              : Integer;          { Last polling port               }
    uid               : Integer;          { Last polling unit id            }
    dat               : String;           { Last sent data                  }
    Summ              : record
     Rx                : Integer;         { Summary Recieve Data            }
     Tx                : Integer;         { Summary Transmite Data          }
    end;
    Rate              : record
     Rx                : Integer;         { Rate Recieve Data               }
     Tx                : Integer;         { Rate Tramnsmite Data            }
    end;
   end;
  end;
  WatchdogTime   : Real;         { Watchdog deadline time, sec          }
  HostIsOkPeriod : Integer;      { Watchdog HostIsOk period, ms         }
  TypeCode       : Integer;      { Type code of module                  }
  NewBaudRate    : Integer;      { BaudRate  of module                  }
  DataFormat     : Integer;      { Data format of module                }
  EnableWD       : Integer;      { Enable WatchDog for send             }
  TimeoutWD      : Real;         { TimeOut WatchDog for send            }
  ShiftTCJ       : Real;         { Shift TCJ                            }
 end;

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_NetRS485}              { Include RS485Proxy   functions.  }


 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  StdSensorHelpTooltip(s,15000);
 end;
 {
 Procedure to show warning tooltip on problem/error
 }
 procedure CallWarningMsg(msg:String; delay,level:integer);
 var s:String;
 begin
  s:='';
  s:=str(delay);
  if level>0 then begin
   Trouble(msg); ShowTooltip('text "'+msg+'" delay '+s+' preset stdError');
  end else begin
   Problem(msg); ShowTooltip('text "'+msg+'" delay '+s+' preset stdWarning');
  end;
  s:='';
 end;
 {
 Custom Set Tag procedure
 }
 procedure CustomSetTag(var R:TTagRef; numdata:Real; sdata:String);
 const type_tag_nil=0; type_tag_int=1; type_tag_real=2; type_tag_string=3;
 var tag:Integer;
 begin
  tag:=R.tag;
  case TypeTag(tag) of
   type_tag_nil   :Problem('Incorrect tag reference');
   type_tag_int   :bNul(iSetTag(tag,Round(numdata)));
   type_tag_real  :bNul(rSetTag(tag,numdata));
   type_tag_string:bNul(sSetTag(tag,sdata));
  end;
 end;
 {
 Prefix for DIM @remote commands. (for future use)
 }
 function DimRemote:String;
 var CanRemote:Boolean;
 begin
  CanRemote:=DIM_IsServerMode or DIM_IsClientMode;
  if (DIM_GuiClickTag=0) then CanRemote:=false;
  if (devDimSrv=0) then CanRemote:=false;
  if CanRemote
  then DimRemote:='@remote '
  else DimRemote:='';
 end;
 {
 Clear Strings
 }
 Procedure I7018_ClearStrings;
 var i:Integer;
 begin
  I7018.RS485.Poll.dat:='';
  I7018.SelfId:='';
  for i:=1 to MaxCmdNum do begin
   I7018.CmdTab.Acronym[i]:='';
   I7018.CmdTab.OpData[i]:='';
   I7018.CmdTab.Cmd[i]:='';
   I7018.CmdTab.Comment[i]:='';
  end;
 end;
 {
 Clear Data
 }
 Procedure I7018_Clear;
 var i:Integer;
 begin
  I7018.RS485.Port        :=0;
  I7018.RS485.UnitId      :=0;
  I7018.RS485.Timeout     :=0;
  I7018.RS485.Polling     :=0;
  I7018.RS485.Deadline    :=0;
  I7018.RS485.DelayOnStart:=0;
  I7018.RS485.Poll.ref    :=0;
  I7018.RS485.Poll.cid    :=0;
  I7018.RS485.Poll.tim    :=0;
  I7018.RS485.Poll.port   :=0;
  I7018.RS485.Poll.uid    :=0;
  I7018.RS485.Poll.Rate.Rx:=0;
  I7018.RS485.Poll.Rate.Tx:=0;
  I7018.RS485.Poll.Summ.Rx:=0;
  I7018.RS485.Poll.Summ.Tx:=0;
  I7018.CmdTab.Num        :=0;
  for i:=1 to MaxCmdNum do begin
   I7018.CmdTab.Enabled[i]:=false;
  end;
  I7018.HostIsOkPeriod:=0;
 end;
 {
 I7018 tag Initialization
 }
 procedure I7018_FillTags(InitVal:Real);
 var i:Integer;
 begin
  I7018.SERVID.val := InitVal;
  I7018.POLL.val   := InitVal;
  I7018.CLOCK.val  := InitVal;
  I7018.RX.val     := InitVal;
  I7018.TX.val     := InitVal;
  I7018.STATE.val  := InitVal;
  for i:=0 to MaxChannelNum do begin
   I7018.AI[i].val     := InitVal;
   I7018.CAL.AI[i].val := InitVal;
  end;
  I7018.TCJ.val              := InitVal;
  I7018.SHIFT.val            := InitVal;
  I7018.AIS.val              := InitVal;
  I7018.DEVCAL.AUTH.val      := InitVal;
  I7018.DEVCAL.ZERO.val      := InitVal;
  I7018.DEVCAL.RANGE.val     := InitVal;
  I7018.NAME.val             := InitVal;
  I7018.VERSION.val          := InitVal;
  I7018.ADDRESS.val          := InitVal;
  I7018.BAUDRATE.val         := InitVal;
  I7018.CHECKSUM.val         := InitVal;
  I7018.PARITY.val           := InitVal;
  I7018.INPTYPE.val          := InitVal;
  I7018.REJFILTER.val        := InitVal;
  I7018.DATATYPE.val         := InitVal;
  I7018.WATCHDOG.TIMEOUT.val := InitVal;
  I7018.WATCHDOG.STATUS.val  := InitVal;
  I7018.WATCHDOG.ENABLE.val  := InitVal;
 end;
 procedure I7018_InitTags(Prefix:String; InitVal:Real);
 var i:Integer; s:String;
  procedure CustomInitTag(Prefix,Name:String; TypeTag:Integer; var R:TTagRef);
  begin
   InitTag(R.tag,Prefix+'.'+Name,TypeTag); //todo init R.nai
  end;
 begin
  s:='';
  if not IsEmptyStr(Prefix) then begin
   DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
   CustomInitTag(Prefix,'DIM.SERVID',3,I7018.SERVID);
   CustomInitTag(Prefix,'POLL'      ,1,I7018.POLL  );
   CustomInitTag(Prefix,'DIM.CLOCK' ,3,I7018.CLOCK );
   CustomInitTag(Prefix,'RX'        ,1,I7018.RX    );
   CustomInitTag(Prefix,'TX'        ,1,I7018.TX    );
   CustomInitTag(Prefix,'STATE'     ,1,I7018.STATE );
   for i:=0 to MaxChannelNum do begin
    s:=str(i);
    CustomInitTag(Prefix,'AI'+s        ,2,I7018.AI[i]    );
    CustomInitTag(Prefix,'CALDATA.AI'+s,2,I7018.CAL.AI[i]);
   end;
   CustomInitTag(Prefix,'TCJ'             ,2,I7018.TCJ             );
   CustomInitTag(Prefix,'SHIFT'           ,2,I7018.SHIFT           );
   CustomInitTag(Prefix,'AIS'             ,1,I7018.AIS             );
   CustomInitTag(Prefix,'DEVCAL.AUTH'     ,1,I7018.DEVCAL.AUTH     );
   CustomInitTag(Prefix,'DEVCAL.ZERO'     ,1,I7018.DEVCAL.ZERO     );
   CustomInitTag(Prefix,'DEVCAL.RANGE'    ,1,I7018.DEVCAL.RANGE    );
   CustomInitTag(Prefix,'NAME'            ,3,I7018.NAME            );
   CustomInitTag(Prefix,'VERSION'         ,3,I7018.VERSION         );
   CustomInitTag(Prefix,'ADDRESS'         ,1,I7018.ADDRESS         );
   CustomInitTag(Prefix,'BAUDRATE'        ,1,I7018.BAUDRATE        );
   CustomInitTag(Prefix,'CHECKSUM'        ,1,I7018.CHECKSUM        );
   CustomInitTag(Prefix,'PARITY'          ,1,I7018.PARITY          );
   CustomInitTag(Prefix,'INPTYPE'         ,1,I7018.INPTYPE         );
   CustomInitTag(Prefix,'REJFILTER'       ,1,I7018.REJFILTER       );
   CustomInitTag(Prefix,'DATATYPE'        ,1,I7018.DATATYPE        );
   CustomInitTag(Prefix,'WATCHDOG.TIMEOUT',2,I7018.WATCHDOG.TIMEOUT);
   CustomInitTag(Prefix,'WATCHDOG.STATUS' ,1,I7018.WATCHDOG.STATUS );
   CustomInitTag(Prefix,'WATCHDOG.ENABLE' ,1,I7018.WATCHDOG.ENABLE );
   I7018_FillTags(InitVal);
  end;
  s:='';
 end;
 {
 Allowed Success/Trouble/ViewImp/ViewExp/Details.
 }
 function AllowedSuccess:Boolean; begin AllowedSuccess:=iAnd(dfSuccess,DebugFlags)>0; end;
 function AllowedTrouble:Boolean; begin AllowedTrouble:=iAnd(dfTrouble,DebugFlags)>0; end;
 function AllowedViewExp:Boolean; begin AllowedViewExp:=iAnd(dfViewExp,DebugFlags)>0; end;
 function AllowedViewImp:Boolean; begin AllowedViewImp:=iAnd(dfViewImp,DebugFlags)>0; end;
 function AllowedDetails:Boolean; begin AllowedDetails:=iAnd(dfDetails,DebugFlags)>0; end;
 {
 Enable command with given command identifier.
 }
 procedure I7018_EnableCmdId(CmdId:Integer; Enabled:Boolean);
 begin
  if CmdId<=MaxCmdNum then I7018.CmdTab.Enabled[CmdId]:=Enabled;
 end;
 Procedure I7018_InitCmdItem(CmdId,Enabled:Integer; Cmd,OpData,Comment:String);
 var s:String;
 begin
  s:='';
  I7018.CmdTab.Enabled[CmdId]:=Enabled>0;
  I7018.CmdTab.Cmd[CmdId]:=Cmd;
  s:=HexB(I7018.RS485.UnitId);
  if CmdId<>cm_HostIsOk
  then I7018.CmdTab.Acronym[CmdId]:=StringReplace(Cmd,'AA',s,RfIgnoreCase)
  else I7018.CmdTab.Acronym[CmdId]:=Cmd;
  I7018.CmdTab.OpData[CmdId]:=OpData;
  I7018.CmdTab.Comment[CmdId]:=Comment;
  s:=str(CmdId);
  Details('Command Id: '+s+' - '+I7018.CmdTab.Acronym[CmdId]+' - '+I7018.CmdTab.Comment[CmdId]);
  s:='';
 end;
 
 {
 Reset command table.
 }
 procedure I7018_InitCmdTable;
 begin
  //                CMD ID           Enabled  Cmd      OpData  Comment
  I7018_InitCmdItem(cm_SetModuleConf ,  0 ,   '%AA'   ,''                 ,'Set Module Configuration');       //ans !AA
  I7018_InitCmdItem(cm_SetModuleName ,  0 ,   '~AAO'  ,''                 ,'Set Module Name');                //ans !AA
  I7018_InitCmdItem(cm_SetMesureAI   ,  0 ,   '$AA5'  ,''                 ,'Set Enabled AI');                 //ans !AA
  I7018_InitCmdItem(cm_SetShiftTCJ   ,  0 ,   '$AA9'  ,''                 ,'Set Shift Temperature Cold Junction');//ans !AA
  I7018_InitCmdItem(cm_CalAllow      ,  1 ,   '~AAE'  ,stringofchar('0',1),'Set Allow Calibration');          //ans !AA
  I7018_InitCmdItem(cm_CalRange      ,  0 ,   '$AA0'  ,''                 ,'Perform Range Calibration');      //ans !AA
  I7018_InitCmdItem(cm_CalZero       ,  0 ,   '$AA1'  ,''                 ,'Perform Zero Calibration');       //ans !AA
  I7018_InitCmdItem(cm_HostIsOk      ,  0 ,   '~**'   ,''                 ,'HostIsOk Notification');          //ans no ans
  I7018_InitCmdItem(cm_SetWDTStatus  ,  0 ,   '~AA3'  ,''                 ,'Set WatchDog Timeout Status');    //ans !AA
  I7018_InitCmdItem(cm_RstWDTStatus  ,  0 ,   '~AA1'  ,''                 ,'Reset WatchDog TimeOut Status');  //ans !AA
  I7018_InitCmdItem(cm_ReadModuleConf,  1 ,   '$AA2'  ,''                 ,'Read Module Configuration');      //ans !AATTCCFF
  I7018_InitCmdItem(cm_ReadFirmwarVer,  1 ,   '$AAF'  ,''                 ,'Read Firmware Version');          //ans !AAdata
  I7018_InitCmdItem(cm_ReadModuleName,  1 ,   '$AAM'  ,''                 ,'Read Module Name');               //ans !AAdata
  I7018_InitCmdItem(cm_ReadAnalogInp ,  1 ,   '#AA'   ,''                 ,'Read Analog Inputs');             //ans >Data
  I7018_InitCmdItem(cm_ReadMesureAI  ,  1 ,   '$AA6'  ,''                 ,'Read Enabled AI');                //ans !AAVV
  I7018_InitCmdItem(cm_ReadTCJ       ,  1 ,   '$AA3'  ,''                 ,'Read Temperature Cold Junction'); //ans >Data
  I7018_InitCmdItem(cm_ReadWDTStatus ,  1 ,   '~AA0'  ,''                 ,'Read WatchDog Status');           //ans !AASS
  I7018_InitCmdItem(cm_ReadWDTimeOut ,  1 ,   '~AA2'  ,''                 ,'Read WatchDog TimeOut');          //ans !AAVV
  I7018.CmdTab.PollCnt:=0;
 end;
 {
 Initialize I7018 parameters
 }
 Procedure I7018_Init;
 var r:Real; s:String;
 begin
  s:='';
  I7018_Clear;
  I7018_ClearStrings;
  I7018_InitTags(ReadIni('tagPrefix'),-MaxReal);
  I7018.Simulator:=iValDef(ReadIni('Simulator'),0)>0;
  I7018.WDReconnect:=iValDef(ReadIni('WDReconnect'),0)>0;
  I7018.isChecksum:=iValDef(ReadIni('Checksum'),0)>0;
  if not I7018.Simulator then begin
   I7018.RS485.Port        :=iValDef(ReadIni('RS485Port'),1);
   I7018.RS485.UnitId      :=iValDef(ReadIni('RS485UnitId'),1);
   I7018.RS485.Timeout     :=iValDef(ReadIni('RS485Timeout'),250);
   I7018.RS485.Polling     :=iValDef(ReadIni('RS485Polling'),10);
   I7018.RS485.Deadline    :=iValDef(ReadIni('RS485Deadline'),60000);
   I7018.RS485.DelayOnStart:=iValDef(ReadIni('DelayOnStart'),1000);
   r:=Max(0,Min(25.5,rValDef(ReadIni('WatchdogTime'),0)));
   if r<>0 then CustomSetTag(I7018.WATCHDOG.TIMEOUT,r,'');
   s:=Str(rGetTag(I7018.WATCHDOG.TIMEOUT.tag));
   Success('Watchdog Timeout = '+s);
  end else begin
   CustomSetTag(I7018.NAME,0,'Simula');
   CustomSetTag(I7018.VERSION,0,'SimVer');
   CustomSetTag(I7018.BAUDRATE,defBaudRate,'');
  end;
  I7018_InitCmdTable;
  {
  Initialize Array
  }
  I7018.CRV_AI[0]:=ao_ai0; I7018.CAL.CRV_AI[0]:=ao_cal_ai0; //todo: Use TTagRef.nai
  I7018.CRV_AI[1]:=ao_ai1; I7018.CAL.CRV_AI[1]:=ao_cal_ai1; //todo: Use TTagRef.nai
  I7018.CRV_AI[2]:=ao_ai2; I7018.CAL.CRV_AI[2]:=ao_cal_ai2; //todo: Use TTagRef.nai
  I7018.CRV_AI[3]:=ao_ai3; I7018.CAL.CRV_AI[3]:=ao_cal_ai3; //todo: Use TTagRef.nai
  I7018.CRV_AI[4]:=ao_ai4; I7018.CAL.CRV_AI[4]:=ao_cal_ai4; //todo: Use TTagRef.nai
  I7018.CRV_AI[5]:=ao_ai5; I7018.CAL.CRV_AI[5]:=ao_cal_ai5; //todo: Use TTagRef.nai
  I7018.CRV_AI[6]:=ao_ai6; I7018.CAL.CRV_AI[6]:=ao_cal_ai6; //todo: Use TTagRef.nai
  I7018.CRV_AI[7]:=ao_ai7; I7018.CAL.CRV_AI[7]:=ao_cal_ai7; //todo: Use TTagRef.nai
  {
  Initialize variables...
  }
  I7018.TypeCode:=defTypeCode;
  I7018.NewBaudRate:=defBaudRate;
  I7018.DataFormat:=defDataFormat;
  {
  Register console commands...
  }
  cmd_DimTagUpdate:= RegisterStdInCmd('@DimTagUpdate','');   //cmd_DimCmdMy    
  cmd_DimCmdMy    := RegisterStdInCmd('@DIMCMDMY','');       //cmd_DimTagUpdate
  cmd_RsReply     := RegisterStdInCmd('@RS485.Reply'   ,''); //cmd_RsReply     
  cmd_RsPoll      := RegisterStdInCmd('@RS485.Poll'    ,''); //cmd_RsPoll      
  cmd_RsRefuse    := RegisterStdInCmd('@RS485.Refuse'  ,''); //cmd_RsRefuse    
  cmd_RsTimeout   := RegisterStdInCmd('@RS485.Timeout' ,''); //cmd_RsTimeout   
  cmd_RsClrSumm   := RegisterStdInCmd('@ClearRS485Sum' ,''); //cmd_RsClrSumm   
  cmd_Poll         :=RegisterStdInCmd('@Poll'          ,''); //cmd_Poll        
  cmd_SetCfg       :=RegisterStdInCmd('@SetCfg'        ,''); //cmd_SetCfg      
  cmd_ReadCfg      :=RegisterStdInCmd('@ReadCfg'       ,''); //cmd_ReadCfg     
  cmd_ReadVer      :=RegisterStdInCmd('@ReadVer'       ,''); //cmd_ReadVer     
  cmd_ReadName     :=RegisterStdInCmd('@ReadName'      ,''); //cmd_ReadName    
  cmd_SetName      :=RegisterStdInCmd('@SetName'       ,''); //cmd_SetName     
  cmd_ReadAI       :=RegisterStdInCmd('@ReadAI'        ,''); //cmd_ReadAI      
  cmd_SetMesAI     :=RegisterStdInCmd('@SetMesureAI'   ,''); //cmd_SetMesAI    
  cmd_ReadMesAI    :=RegisterStdInCmd('@ReadMesureAI'  ,''); //cmd_ReadMesAI   
  cmd_CalAllow     :=RegisterStdInCmd('@CalibrateAllow',''); //cmd_CalAllow    
  cmd_CalRange     :=RegisterStdInCmd('@CalibrateRange',''); //cmd_CalRange    
  cmd_CalZero      :=RegisterStdInCmd('@CalibrateZero' ,''); //cmd_CalZero     
  cmd_ReadTCJ      :=RegisterStdInCmd('@ReadTCJ'       ,''); //cmd_ReadTCJ     
  cmd_SetShiftTCJ  :=RegisterStdInCmd('@SetShiftTCJ'   ,''); //cmd_SetShiftTCJ 
  cmd_ReadWDStatus :=RegisterStdInCmd('@ReadWDStatus'  ,''); //cmd_ReadWDStatus 
  cmd_ReadWDTimeout:=RegisterStdInCmd('@ReadWDTimeout' ,''); //cmd_ReadWDTimeout
  cmd_SetWDStatus  :=RegisterStdInCmd('@SetWDStatus'   ,''); //cmd_SetWDStatus  
  cmd_RstWDStatus  :=RegisterStdInCmd('@ResetWDStatus' ,''); //cmd_RstWDStatus  
  cmd_MenuTools    :=RegisterStdInCmd('@MenuTools'     ,''); //cmd_MenuTools    
  CustomSetTag(I7018.ADDRESS,I7018.RS485.UnitId,'');
  I7018.Shortcut:=iValDef(ReadIni('Shortcut'),-1);
  if I7018.Shortcut>MaxChannelNum then I7018.Shortcut:=-1;
  I7018.CalibrMode:=iValDef(ReadIni('Calibrate'),0);
  I7018.isCalibr:=(I7018.CalibrMode>0) and (numcals>0);
  {
  Server Identifier.
  }
  s:=Str(getpid);
  if DIM_IsServerMode
  then I7018.SelfId:=s+'@'+ParamStr('HostName')
  else I7018.SelfId:=s+'@'+ParamStr('ComputerName');
  {
  Colors
  }
  ColorNorm:=clAqua;
  if DIM_IsServerMode then ColorNorm:=clLime;
  if DIM_IsClientMode then ColorNorm:=clLime;
  ColorWarn:=clYellow;
  bNul(SetTagColor(I7018.SERVID.tag,ColorNorm));
  bNul(SetTagColor(I7018.CLOCK.tag,ColorNorm));
  s:='';
 end;
 {
 Command cycle routines.
 }
 function IsValidCmdNum(Num:Integer):Boolean;
 begin
  IsValidCmdNum:=(1<=Num) and (Num<=MaxCmdNum);
 end;
 function ValidateCmdNum(Num:Integer):Integer;
 begin
  if IsValidCmdNum(Num)
  then ValidateCmdNum:=Num
  else ValidateCmdNum:=1;
 end;
 function IsEnabledCmdNum(Num:Integer):Boolean;
 begin
  if IsValidCmdNum(Num)
  then IsEnabledCmdNum:=I7018.CmdTab.Enabled[Num]
  else IsEnabledCmdNum:=False;
 end;
 function NextEnabledCmdNum(Num:Integer):Integer;
 var i:Integer; oldNum:Integer;
 begin
  i:=1;
  oldNum:=Num;
  while (i<=MaxCmdNum) do begin
   Num:=ValidateCmdNum(Num+1);
   { writeln(str(Num)); }
   if IsEnabledCmdNum(Num)
   then begin 
   i:=MaxCmdNum+1; 
   { writeln('Enabled'+str(Num)); }
   end
   else i:=i+1;
  end;
  if Num<=oldNum then I7018.CmdTab.PollCnt:=I7018.CmdTab.PollCnt+1;
  NextEnabledCmdNum:=Num;
 end;
 {
 Find the current Proxy device to send @RS485.Poll message.
 }
 function devTheProxy:Integer;
 begin
  //if I7018.Simulator                // In simulation mode        (for future sim)
  //then devTheProxy:=devMySelf       // send messages to himself  (for future sim)
  //else devTheProxy:=devRS485Proxy; // otherwise use &ModbusProxy (for future sim)
  devTheProxy:=devRS485Proxy;
 end;
 {
 Check if communication port is opened or not.
 As long as we use Proxy device, check Port number & device reference.
 }
 function IsPortOpened:Boolean;
 begin
  IsPortOpened:=(I7018.RS485.Port>0) and (devTheProxy<>0);
 end;
 {
 Assign RS485 last sent polling request record.
 }
 procedure AssignRS485Poll(ref,cid:Integer; tim:Real; port,uid:Integer; dat:String);
 begin
  I7018.RS485.Poll.ref:=ref;     I7018.RS485.Poll.cid:=cid;     I7018.RS485.Poll.tim:=tim;
  I7018.RS485.Poll.port:=port;   I7018.RS485.Poll.uid:=uid;     I7018.RS485.Poll.dat:=dat;
 end;
 {
 Clear RS485 polling request to be ready for next polling.
 }
 procedure ClearRS485Poll;
 begin
  I7018.RS485.Poll.cid:=0;
  I7018.RS485.Poll.dat:='';
 end;
 {
 Clear RS485 pollrate
 }
 procedure ClearRS485Rate;
 begin
  I7018.RS485.Poll.Rate.Rx:=0;
  I7018.RS485.Poll.Rate.Tx:=0;
 end;
 //
 // Increment summ and rate counters.
 //
 procedure IncSummRate(var summ,rate:Integer);
 begin
  summ:=summ+1; rate:=rate+1;
 end;
 {
 Validating Type Code
 I7018 Type Code: $00..$06 or $0E..$18 (hex)
 }
 function isValidTypeCode(TT:Integer):boolean;
 begin
  isValidTypeCode:=(((TT>=minLowValidTypeCode) and (TT<=maxLowValidTypeCode)) or ((TT>=minHighValidTypeCode) and (TT<=maxHighValidTypeCode)));
 end;
 function ValidateTypeCode(oldTT,newTT:Integer):Integer;
 begin
  ValidateTypeCode:=oldTT;
  if isValidTypeCode(newTT) then ValidateTypeCode:=newTT else Problem('Invalid Type Code: '+HexB(newTT));
 end;
 {
 Validating BaudRate
 I7018 BaudRate: $03-$0A (hex)
 }
 function isValidBaudrate(CC:Integer):boolean;
 begin
  isValidBaudrate:=(ge(CC,3) and le(CC,10));
 end;
 function ValidateBaudRate(oldCC,newCC:Integer):Integer;
 begin
  ValidateBaudRate:=oldCC;
  if isValidBaudrate(newCC) then ValidateBaudRate:=newCC else Problem('Invalid BaudRate: '+HexB(newCC));
 end;
 {
 Validating DataFormat
 I7018 DataFormat: $00,$40 (hex)
 }
 function isValidDataFormat(FF:integer):boolean;
 const maxbit=3; minbit=0;
 var i,bit03,bit74:integer; 
 begin
  for i:=minbit to maxbit do bit03:=iSetBit(bit03,i,iGetBit(FF,i));
  for i:=maxbit downto minbit do bit74:=iSetBit(bit74,abs(i-3),iGetBit(FF,i+4));
  isValidDataFormat:=(((bit03>=0) and (bit03<=2)) and ((bit74>=0) and (bit03<=3)));
 end;
 function ValidateDataFormat(oldFF,newFF:Integer):Integer;
 begin
  ValidateDataFormat:=oldFF;
  if isValidDataFormat(newFF) then ValidateDataFormat:=newFF else Problem('Invalid DataFormat: '+HexB(newFF));
 end;
 {
 EnablePoll
 }
 Procedure CalcHostIsOkPeriod;
 begin
  I7018.HostIsOkPeriod:=Ord(iGetTag(I7018.WATCHDOG.STATUS.tag)=1)*Round(rGetTag(I7018.WATCHDOG.TIMEOUT.tag)*1000*0.5);
 end;
 {
 Calc Shift TCJ
 }
 Function CalcShiftTCJ(t:Real):String;
 var rtemp:Real; itemp:Integer; s:String;
 begin
  s:='';
  rtemp:=t;
  if rtemp>40.96 then rtemp:=40.96 else
  if rtemp<-40.96 then rtemp:=-40.96;
  itemp:=round(rtemp*100);
  if itemp>=0 then s:='+'+hexw(abs(itemp));
  if itemp<0 then s:='-'+hexw(abs(itemp));
  CalcShiftTCJ:=s;
  s:='';
 end;
 {
 Update data curves & tags
 }
 Procedure UpdateModuleConfig(cfg:String);
 var units,i:Integer;s:String;
 begin
  s:='';
  s:=cfg;
  if AllowedSuccess then Success('Module Config: '+s);
  I7018.TypeCode:=ValidateTypeCode(I7018.TypeCode,Val('$'+Copy(s,3,2)));
  CustomSetTag(I7018.INPTYPE,I7018.TypeCode,'');
  if AllowedSuccess then Success('TypeCode: '  +HexB(I7018.TypeCode));
  I7018.NewBaudRate:=ValidateBaudRate(iGetTag(I7018.BAUDRATE.tag),Val('$'+Copy(s,5,2)));
  CustomSetTag(I7018.BAUDRATE,I7018.NewBaudRate,'');
  if AllowedSuccess then Success('BaudRate: '  +HexB(I7018.NewBaudRate));
  I7018.DataFormat:=ValidateDataFormat(I7018.DataFormat,Val('$'+Copy(s,7,2)));
  bNul(iSetTagBitState(I7018.CHECKSUM.tag,0,isbit(I7018.DataFormat,6)));
  bNul(iSetTagBitState(I7018.REJFILTER.tag,0,isbit(I7018.DataFormat,7)));
  for i:=0 to 3 do units:=iSetBitState(units,i,iGetBitState(I7018.DataFormat,i));
  CustomSetTag(I7018.DATATYPE,units,'');
  if AllowedSuccess then Success('DataFormat: '+HexB(I7018.DataFormat));
  s:='';
 end;
 procedure UpdateModuleVersion(ver:String);
 begin
  if AllowedSuccess then Success('FirmwareVersion: '+ver);
  CustomSetTag(I7018.VERSION,0,ver);
 end;
 procedure UpdateModuleName(name:String);
 begin
  if AllowedSuccess then Success('ModuleName: '+name);
  CustomSetTag(I7018.NAME,0,name);
 end;
 {
 Update Data of Analog Inputs
 }
 function ConvertData(FormatCode,InpType:Integer; data:String):Real;
 var res:Real;s:String;
  function GetSmallInt(worddata:integer):Integer;
  const maxword=65535; minusbit=15; bitmask=-65536;
  var i,res:Integer;
  begin
   res:=0;
   if ((0<=worddata) and (worddata<=maxword)) then begin
    res:=worddata;
    if iGetBitState(worddata,minusbit) then res:=ior(worddata,bitmask);
   end else Problem('Bad data in GetSmallInt');
   GetSmallInt:=res;
  end;
  function Str2Real(data:String; var rdata:Real):Boolean;
  begin
   rdata:=rval(data);
   Str2Real:=(not isNan(rdata));
  end;
  function StrHex2Int(data:String; var idata:Integer):Boolean;
  begin
   StrHex2Int:=false;
   if length(data)=4 then
    if IsSameText(data,'0000') then begin
     idata:=0; StrHex2Int:=true;
    end else begin
     idata:=val('$'+data);
     if idata<>0 then StrHex2Int:=true;
    end;
  end;
  function HexCode2Value(Code:Integer; Range:Real):Real;
  begin
   HexCode2Value:=Range*GetSmallInt(Code)/val('$7FFF');
  end;
  function AdcCode2Value(Code:Integer; Range:Real):Real;
  begin
   AdcCode2Value:=Max(-Range,Range*GetSmallInt(Code)/val('$7FFF'));
  end;
  function Adc2mV(FSR:Real):Real;
  var rd,res:Real; id,i:Integer;
  begin
   res:=0;
   if ((FormatCode<dfEngineering) or (FormatCode>dfHexadecimal)) 
   then Problem('Invalid FormatCode in Adc2Mv') else
   case FormatCode of
    dfEngineering: 
     if (Length(data)=7) and Str2Real(data,rd) then 
     if (FSR<1000) then res:=rd else res:=rd*1000;
    dfPercentFSR:  
     if (Length(data)=7) and Str2Real(data,rd) then res:=rd/100*FSR;
    dfHexadecimal: 
     if (Length(data)=4) and StrHex2Int(data,id) then res:=AdcCode2Value(id,FSR);
   end;
   Adc2mV:=res;
  end;
  function Adc2Tc(NFSR,PFSR:Real):Real;
  var d,res:real; i:Integer;
  begin
   res:=0;
   if ((FormatCode<dfEngineering) or (FormatCode>dfHexadecimal)) 
   then Problem('Invalid FormateCode in Adc2Tc') else
   case FormatCode of
    dfEngineering:
     if IsSameText(data,'+9999') then res:=PFSR else
     if IsSameText(data,'-0000') then res:=NFSR else
     if IsSameText(data,'+9999.9') then res:=PFSR else
     if IsSameText(data,'-9999.9') then res:=NFSR else
     if (Length(data)=7) and Str2Real(data,d) then res:=d;
    dfPercentFSR:
     if IsSameText(Data,'+9999') then res:=PFSR else
     if IsSameText(Data,'-0000') then res:=NFSR else
     if IsSameText(Data,'+999.99') then res:=PFSR else
     if IsSameText(Data,'-999.99') then res:=NFSR else
     if (Length(data)=7) and Str2Real(data,d)
     then res:=d/100*max(abs(NFSR),abs(PFSR));
    dfHexadecimal:
     if IsSameText(Data,'7FFF') then res:=PFSR else
     if IsSameText(Data,'8000') then res:=NFSR else
     if (Length(Data)=4) and StrHex2Int(Data,i)
     then res:=HexCode2Value(i,max(abs(NFSR),abs(PFSR)));
   end;
   Adc2Tc:=res;
  end;
 begin
  s:='';
  s:=str(InpType);
  if ((InpType<0) or (InpType>24)) 
  then Problem('Invalid InpType in ConverData') else
  case InpType of
   0 : res:=Adc2mV(15);                              //  -15..15  mV ADC input range
   1 : res:=Adc2mV(50);                              //  -50..50  mV ADC input range
   2 : res:=Adc2mV(100);                             // -100..100 mV ADC input range
   3 : res:=Adc2mV(500);                             // -500..500 mV ADC input range
   4 : res:=Adc2mV(1000);                            //   -1..1    V ADC input range
   5 : res:=Adc2mV(2500);                            // -2.5..2.5  V ADC input range
   6 : res:=Adc2mV(20);                              //  -20..20  mA ADC input range
   7,8,9,10,11,12,13 : Problem('Invalid Input Type: '+s);
   14: res:=Adc2Tc(-210,760);                        // Type J, -210..760  C
   15: res:=Adc2Tc(-270,1372);                       // Type K, -270..1372 C
   16: res:=Adc2Tc(-270,400);                        // Type T, -270..400  C
   17: res:=Adc2Tc(-270,1000);                       // Type E, -270..1000 C
   18: res:=Adc2Tc(   0,1768);                       // Type R,    0..1768 C
   19: res:=Adc2Tc(   0,1768);                       // Type S,    0..1768 C
   20: res:=Adc2Tc(   0,1820);                       // Type B,    0..1820 C
   21: res:=Adc2Tc(-270,1300);                       // Type N, -270..1300 C
   22: res:=Adc2Tc(   0,2320);                       // Type C,    0..2320 C
   23: res:=Adc2Tc(-200,800);                        // Type L, -200..800  C
   24: res:=Adc2Tc(-200,100);                        // Type M, -200..100  C
  end;
  ConvertData:=res;
  s:='';
 end;
 procedure UpdateAnalogInputsData(aidata:String);
 const Space4 = '    '; Space7 = '       '; Zero4  = '0000'; Zero7  = '+0.0000';
 var FormatCode,TypeCode,i:Integer; dat:String;
  procedure UpdateAnalogInputData(numch:Integer;numdata:Real);
  var datsubz:Real;
  begin
   CustomSetTag(I7018.AI[numch],numdata,'');
   UpdateAo(I7018.CRV_AI[numch],time,numdata);
   if I7018.isCalibr then begin
    datsubz:=numdata;
    if I7018.Shortcut>=0 then datsubz:=numdata-rGetTag(I7018.AI[I7018.Shortcut].tag);
    case I7018.CalibrMode of
     NoCalibr: ;
     PuCalibr: if rSetTag(I7018.CAL.AI[numch].tag,calibr(numch,datsubz,0)) then
               UpdateAo(I7018.CAL.CRV_AI[numch],time,rGetTag(I7018.CAL.AI[numch].tag));
     TcCalibr: if rSetTag(I7018.CAL.AI[numch].tag,calibr(numch,datsubz,rGetTag(I7018.TCJ.tag))) then
               UpdateAo(I7018.CAL.CRV_AI[numch],time,rGetTag(I7018.CAL.AI[numch].tag));
    end;
   end;
  end;
 begin
  dat:='';
  if AllowedSuccess then Success('AnalogInputs: '+aidata);
  FormatCode:=iGetTag(I7018.DATATYPE.tag);
  TypeCode:=iGetTag(I7018.INPTYPE.tag);
  if not IsEmptyStr(aidata) then dat:=aidata;
  if ((FormatCode<dfEngineering) or (FormatCode>dfHexadecimal)) 
  then Problem('Invalid FormatCode in UpdateAnalogInputsData') else
  if (Pos(' ',aidata)>0) then
   if (FormatCode=dfHexadecimal) then dat:=StringReplace(dat,Space4,Zero4,rfReplaceAll)
   else dat:=StringReplace(dat,Space7,Zero7,rfReplaceAll);
  for i:=0 to MaxChannelNum do begin
   case FormatCode of
    dfEngineering:
     UpdateAnalogInputData(i,ConvertData(FormatCode,TypeCode,copy(dat,(i*7)+1,7)));
    dfPercentFSR:
     UpdateAnalogInputData(i,ConvertData(FormatCode,TypeCode,copy(dat,(i*7)+1,7)));
    dfHexadecimal:
     UpdateAnalogInputData(i,ConvertData(FormatCode,TypeCode,copy(dat,(i*4)+1,4)));
   end;
  end;
  dat:='';
 end;
 {
 Update WatchDog
 }
 procedure UpdateWDStatus(wds:String);
 begin
  if AllowedSuccess then Success('WatchDogStatus: '+wds);
  if iGetBitState(val('$'+wds),2) then begin
   CustomSetTag(I7018.WATCHDOG.STATUS,2,'');
   if iGetBitState(val('$'+wds),7) then begin
    CustomSetTag(I7018.WATCHDOG.ENABLE,1,'');
   end;
  end else begin
   if iGetBitState(val('$'+wds),7) then begin
    CustomSetTag(I7018.WATCHDOG.STATUS,1,'');
    CustomSetTag(I7018.WATCHDOG.ENABLE,1,'');
   end else begin
    CustomSetTag(I7018.WATCHDOG.ENABLE,0,'');
    CustomSetTag(I7018.WATCHDOG.STATUS,0,'');
   end;
  end;
  CalcHostIsOkPeriod;
 end;
 procedure UpdateWDTimeout(wdt:String);
 var rwdt:Real; s:String; i:Integer;
 begin
  s:='';
  s:=wdt;
  if AllowedSuccess then Success('WatchDogTimeout: '+s);
  if not (iGetTag(I7018.WATCHDOG.STATUS.tag)=2) 
  then begin 
   i:=val(copy(s,1,1));
   CustomSetTag(I7018.WATCHDOG.STATUS,i,'');
  end;
  i:=val('$'+copy(s,2,2));
  rwdt:=i/10;
  CustomSetTag(I7018.WATCHDOG.TIMEOUT,rwdt,'');
  CalcHostIsOkPeriod;
  s:='';
 end;
 procedure UpdateActiveChannel(data:String);
 var r:Real;
 begin
  if AllowedSuccess then Success('Active Channel = '+data);
  r:=rval('$'+data);
  if not isNan(r) 
  then CustomSetTag(I7018.AIS,Round(r),'') 
  else Trouble('Incorrect data in answer Active Channels: '+data);
 end;
 procedure UpdateTCJ(data:String);
 var rdata:Real;
 begin
  rdata:=rVal(data);
  if not isNan(rdata) then begin
   CustomSetTag(I7018.TCJ,rdata,'');
   UpdateAo(ao_tcj,time,rdata);
  end else Trouble('Incorrect data at read TCJ: '+data);
 end;
 procedure UpdateShiftTCJ(data:String);
 var rdata:Real;
 begin
  rdata:=rVal(data);
  if not isNan(rdata) then begin
   CustomSetTag(I7018.SHIFT,rdata,'');
   UpdateAo(ao_shift,time,rdata);
  end else Trouble('Incorrect data at Set Shift TCJ: '+data);
 end;
 
 {
 Handle "Answer received"...
 }
 function CalcCheckSum(dat:String):String;
 var i,sum:integer;
 begin
   sum:=0;
   for i:=1 to length(dat) do sum:=sum+ord(strfetch(dat,i));
   CalcCheckSum:=HexB(sum);
 end;
 function isCorrectChecksum(dat:String; var checkdat:String):boolean;
 var tempchecksum,tempdat:String;
 begin
  tempchecksum:='';tempdat:='';checkdat:='';
  isCorrectChecksum:=false;
  tempdat:=trim(dat);
  if I7018.isCheckSum then begin
   tempchecksum:=copy(dat,length(tempdat)-1);
   tempdat:=copy(dat,1,length(tempdat)-2);
   if CalcCheckSum(tempdat)=tempchecksum then begin
    isCorrectChecksum:=true;
   end else begin
    Trouble('Bad checksum: "'+tempchecksum+'" in answer '+tempdat);
    tempdat:='';
   end;
  end else begin
   isCorrectChecksum:=true;
  end;
  checkdat:=tempdat;
  tempchecksum:='';tempdat:='';
 end;
 function CheckData(cid:Integer; dat:String):String;
 var res:Boolean;tempdat,ans:String;
 begin
  res:=false;
  ans:='';tempdat:='';
  if IsEmptyStr(dat) then begin
   if cid<>cm_HostIsOk then Trouble('Answer is empty') else res:=true;
  end else begin
   if StrFetch(dat,1)='?' then Trouble('Bad answer : "'+dat+'" on comand '+I7018.CmdTab.Acronym[cid]) else begin
    if isCorrectChecksum(dat,tempdat) then begin
     if ((cid=cm_ReadAnalogInp) or (cid=cm_ReadTCJ)) then begin
       if StrFetch(tempdat,1)='>' then begin
         ans:=tempdat;
         res:=true;
       end else Trouble('Bad answer : "'+tempdat+'" on comand '+I7018.CmdTab.Acronym[cid]);
     end else begin
       if StrFetch(tempdat,1)='!' then begin
         ans:=tempdat;
         res:=true;
       end else Trouble('Bad answer : "'+tempdat+'" on comand '+I7018.CmdTab.Acronym[cid]);
     end;
    end;
   end;
  end;
  CheckData:=ans;
  tempdat:='';ans:='';
 end;
 procedure I7018_OnCommand(cid:Integer; dat:String);
 var v:Real; ans,hexadr:String;
 begin
  ans:='';hexadr:='';
  ans:=CheckData(cid,dat);
  if length(ans)>0 then begin
   // Set Module Config
   if cid=cm_SetModuleConf then begin
    hexadr:='$'+copy(ans,2,2);
    I7018.RS485.UnitId:=val(hexadr);
    if iGetTag(I7018.ADDRESS.tag)<>I7018.RS485.UnitId then begin
     CustomSetTag(I7018.ADDRESS,I7018.RS485.UnitId,'');
     I7018_InitCmdTable;
    end;
    I7018_EnableCmdId(cid,False);
    I7018_EnableCmdId(cm_ReadModuleConf,True);
   end else
   // Read Module Configuration
   if cid=cm_ReadModuleConf then begin
    UpdateModuleConfig(Copy(ans,2));
    I7018_EnableCmdId(cid,False);
   end else
   // Read Firmware Version
   if cid=cm_ReadFirmwarVer then begin
    UpdateModuleVersion(Copy(ans,4));
    I7018_EnableCmdId(cid,False);
   end else
   // Read Module Name
   if cid=cm_ReadModuleName then begin
    UpdateModuleName(Copy(ans,4));
    I7018_EnableCmdId(cid,False);
   end else
   // Set Module Name
   if cid=cm_SetModuleName then begin
    I7018_EnableCmdId(cid,False);
    I7018_EnableCmdId(cm_ReadModuleName,True);
   end else
   // Read Analog Inputs
   if cid=cm_ReadAnalogInp then begin
    UpdateAnalogInputsData(Copy(ans,2));
    I7018_EnableCmdId(cid,iGetTag(I7018.POLL.tag)>0);
   end else
   // Host is Ok notification
   if (cid=cm_HostIsOk) then begin
   end else
   // Read Temperature Cold Junction
   if (cid=cm_ReadTCJ) then begin
    UpdateTCJ(copy(ans,2));
    I7018_EnableCmdId(cid,iGetTag(I7018.POLL.tag)>0);
   end else
   // Read Enabled AI
   if (cid=cm_ReadMesureAI) then begin
    UpdateActiveChannel(copy(ans,4,2));
    I7018_EnableCmdId(cid,False);
   end else
   // Set Enable AI
   if (cid=cm_SetMesureAI) then begin
    I7018_EnableCmdId(cid,False);
    I7018_EnableCmdId(cm_ReadMesureAI,true);
   end else
   // Set Shift TCJ
   if (cid=cm_SetShiftTCJ) then begin
    CustomSetTag(I7018.SHIFT,I7018.ShiftTCJ,'');
    I7018_EnableCmdId(cid,False);
   end else
   // Calibration Allow
   if (cid=cm_CalAllow) then begin
    I7018_EnableCmdId(cid,False);
   end else
   // Perform Range Calibration 
   if (cid=cm_CalRange) then begin
    I7018.DEVCAL.TimeSendRange:=mSecNow;
    CustomSetTag(I7018.DEVCAL.RANGE,1,'');
    I7018_EnableCmdId(cid,False);
   end else
   // Perform Zero Calibration
   if (cid=cm_CalZero) then begin
    I7018.DEVCAL.TimeSendZero:=mSecNow;
    CustomSetTag(I7018.DEVCAL.ZERO,1,'');
    I7018_EnableCmdId(cid,False);
   end else
   // Read WatchDog Status
   if cid=cm_ReadWDTStatus then begin
    UpdateWDStatus(copy(ans,4,2));
    I7018_EnableCmdId(cid,iGetTag(I7018.WATCHDOG.STATUS.tag)>0);
   end else
   // Read WatchDog Timeout
   if cid=cm_ReadWDTimeOut then begin
    UpdateWDTimeout(copy(ans,4));
    I7018_EnableCmdId(cid,iGetTag(I7018.WATCHDOG.STATUS.tag)>0);
   end else
   // Set WatchDog Status
   if cid=cm_SetWDTStatus then begin
    I7018_EnableCmdId(cid,False);
    I7018_EnableCmdId(cm_ReadWDTStatus,true);
    I7018_EnableCmdId(cm_ReadWDTimeOut,true);
   end else
   // Reset WatchDog Status
   if cid=cm_RstWDTStatus then begin
    I7018_EnableCmdId(cid,False);
    I7018_EnableCmdId(cm_ReadWDTStatus,true);
    I7018_EnableCmdId(cm_ReadWDTimeOut,true);
   end;
  end;
  ans:='';hexadr:='';
 end;
 {
 WatchDog AutoReconnect
 }
 procedure I7018_WDAutoReconnect;
 var s:String;
 begin
  s:='';
  s:=str(rGetTag(I7018.WATCHDOG.TIMEOUT.tag));
  DevPostCmdLocal('@ResetWDStatus');
  DevPostCmdLocal('@SetWDStatus '+s+' 1');
  s:='';
 end;
 {
 Update State of Device
 }
 procedure UpdateState;
 begin
  if I7018.Simulator then CustomSetTag(I7018.STATE,stt_Simulator,'')
  else begin
   if iGetTag(I7018.WATCHDOG.STATUS.tag)=2 then begin
    CustomSetTag(I7018.STATE,stt_WatchDog,'');
    { Problem('Watchdog in device '+DevName); }
    if I7018.WDReconnect then I7018_WDAutoReconnect;
   end else if I7018.isOpenDet then begin
    CustomSetTag(I7018.STATE,stt_OpenDet,'');
   end else if iGetTag(I7018.RX.tag)>0 then begin
    CustomSetTag(I7018.STATE,stt_Work,'');
   end;
  end;
  UpdateDO(do_state,time,iGetTag(I7018.STATE.tag));
  I7018_EnableCmdId(cm_ReadTCJ,iGetTag(I7018.POLL.tag)>0);
  I7018_EnableCmdId(cm_ReadAnalogInp,iGetTag(I7018.POLL.tag)>0);
 end;
 {
 DIM Update tag
 }
 procedure DimRefreshTag(var R:TTagRef);
 var tag:Integer; v:Real;
 begin
  tag:=R.tag; v:=GetStampOfTag(tag,_NaN);
  if not IsNaN(v) then if ShouldRefresh(R.val,v)>0 then DIM_UpdateTag(R.tag,'');;
 end;
 Procedure I7018_DIM_UpdateTag;
 var i:Integer;
 begin
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then I7018_FillTags(-MaxReal);
   DimRefreshTag(I7018.SERVID);
   DimRefreshTag(I7018.POLL);
   DimRefreshTag(I7018.CLOCK);
   DimRefreshTag(I7018.RX);
   DimRefreshTag(I7018.TX);
   DimRefreshTag(I7018.STATE);
   for i:=0 to MaxChannelNum do begin
    DimRefreshTag(I7018.AI[i]);
    DimRefreshTag(I7018.CAL.AI[i]);
   end;
   DimRefreshTag(I7018.TCJ);
   DimRefreshTag(I7018.SHIFT);
   DimRefreshTag(I7018.AIS);
   DimRefreshTag(I7018.DEVCAL.AUTH);
   DimRefreshTag(I7018.DEVCAL.ZERO);
   DimRefreshTag(I7018.DEVCAL.RANGE);
   DimRefreshTag(I7018.NAME);
   DimRefreshTag(I7018.VERSION);
   DimRefreshTag(I7018.ADDRESS);
   DimRefreshTag(I7018.BAUDRATE);
   DimRefreshTag(I7018.CHECKSUM);
   DimRefreshTag(I7018.PARITY);
   DimRefreshTag(I7018.INPTYPE);
   DimRefreshTag(I7018.REJFILTER);
   DimRefreshTag(I7018.DATATYPE);
   DimRefreshTag(I7018.WATCHDOG.TIMEOUT);
   DimRefreshTag(I7018.WATCHDOG.STATUS);
   DimRefreshTag(I7018.WATCHDOG.ENABLE);
  end;
 end;
 {
 Finalize I7018 device.
 }
 procedure I7018_Free;
 begin
 end;
 {
 Data handler on @RS485.Reply event. Process reply comes from RS485 device.
 }
 procedure I7018_OnReply(ref,cid,tim,port,uid:Integer; dat:String);
 var p:Integer;s:String;
  procedure GotTrouble(msg:String; numdat:Integer);
  var temps:String;
  begin
   temps:='';
   temps:=str(numdat);
   Trouble(msg+temps);
   temps:='';
  end;
 begin
  s:='';
  if not IsValidCmdNum(cid) then GotTrouble('Bad reply command ',cid) else
  if (port<>I7018.RS485.Poll.port) then GotTrouble('Bad reply port ',port) else
  if (uid<>I7018.RS485.Poll.uid) then GotTrouble('Bad reply unit id ',uid) else
  if (cid<>I7018.RS485.Poll.cid) then GotTrouble('Bad reply command id ',cid) else
  if (ref<>I7018.RS485.Poll.ref) then Trouble('Bad reply device '+RefInfo(ref,'Name')) else
  if Length(dat)=0 then Trouble('Bad reply answer') else begin
   IncSummRate(I7018.RS485.Poll.Summ.Rx,I7018.RS485.Poll.Rate.Rx);
   p:=pos('$$',dat);
   s:=copy(dat,p+2);
   I7018_OnCommand(cid,s);
  end;
  s:='';
 end;
 {
 Prepare data for send
 }
 function I7018_PrepareData(cid:Integer):String;
 var dat:String;
 begin
  dat:='';
  dat:=I7018.CmdTab.Acronym[cid];
  if length(I7018.CmdTab.OpData[cid])>0 then begin
   dat:=dat+I7018.CmdTab.OpData[cid];
   I7018.CmdTab.OpData[cid]:='';
  end;
  if cid=cm_HostIsOk then begin
   I7018_EnableCmdId(cm_HostIsOk,false);
  end;
  if I7018.isChecksum then dat:=dat+CalcCheckSum(dat);
  dat:=RS485_insert_le(dat,le_req);
  I7018_PrepareData:=dat;
  dat:='';
 end;
 {
 I7018 Polling
 }
 Procedure I7018_Poll;
 var i:Integer;
  Procedure PollRS485Proxy(cid:Integer; var num:Integer);
  var dev,ref,tot,port,uid:Integer; dat,s:String;
  begin
   dat:='';s:='';
   dev:=devTheProxy; ref:=devMySelf; tot:=I7018.RS485.Timeout; port:=I7018.RS485.Port; uid:=I7018.RS485.UnitId;
   if cid=cm_HostIsOk then tot:=0;
   dat:=I7018_PrepareData(cid);
   if DevPost(dev,RS485_proxy_poll('@RS485.Poll',ref,cid,tot,port,uid,dat))>0 then begin
    if DebugFlagEnabled(dfViewExp) then ViewExp('COM: '+RS485_proxy_nice('@RS485.Poll',ref,cid,tot,port,uid,dat,32));
    AssignRS485Poll(dev,cid,mSecNow,port,uid,dat);
    IncSummRate(I7018.RS485.Poll.Summ.Tx,I7018.RS485.Poll.Rate.Tx);
   end else begin
    s:=Str(cid);
    Trouble('Fail to send command '+s);
    ClearRS485Poll;
   end;
   if cid=cm_HostIsOk then ClearRS485Poll;
   if ((cid<>cm_ReadAnalogInp) or (cid<>cm_ReadTCJ)) then I7018_EnableCmdId(cid,false);
   num:=cid;
   dat:='';s:='';
  end;
 begin
  if IsPortOpened and not DIM_IsClientMode then begin
   if IsValidCmdNum(I7018.RS485.Poll.cid) then begin
    //
    // Request in progress, waiting @RS485Proxy.Reply/@RS485Proxy.Refuse/@RS485Proxy.Timeout.
    // Handle Deadline if was no responce for too long time: repeat polling again.
    //
    if iGetTag(I7018.POLL.tag)<>0 then begin
     if (mSecNow>I7018.RS485.Poll.tim+I7018.RS485.Deadline) then begin
      Trouble('Deadline detected, repeat polling again...');
      PollRS485Proxy(I7018.RS485.Poll.cid,I7018.CmdTab.Num);
     end;
    end else if iGetTag(I7018.POLL.tag)=0 then begin
     UpdateDo(do_state,time,stt_Disable);
     CustomSetTag(I7018.STATE,stt_Disable,'');
    end;
   end else begin
    //
    // If request is cleared, send new @RS485Proxy.Poll request by timer.
    //
    if iGetTag(I7018.POLL.tag)<>0 then begin
     if (mSecNow>=I7018.RS485.Poll.tim+I7018.RS485.Polling) then begin
      PollRS485Proxy(NextEnabledCmdNum(I7018.CmdTab.Num),I7018.CmdTab.Num);
     end;
    end else begin
     UpdateDo(do_state,time,stt_Disable);
     CustomSetTag(I7018.STATE,stt_Disable,'');
    end;
   end;
   //
   // Update Poll Rate.
   //
   if SysTimer_Pulse(1000)>0 then begin
    CustomSetTag(I7018.RX,I7018.RS485.Poll.Rate.Rx,'');
    CustomSetTag(I7018.TX,I7018.RS485.Poll.Rate.Tx,'');
    UpdateDo(do_rx,time,I7018.RS485.Poll.Rate.Rx);
    UpdateDo(do_tx,time,I7018.RS485.Poll.Rate.Tx);
    Details('PollRate: Rx='+Str(I7018.RS485.Poll.Rate.Rx)
                   +'  Tx='+Str(I7018.RS485.Poll.Rate.Tx));
    ClearRS485Rate;
   end;
   if (I7018.HostIsOkPeriod>0) and (iGetTag(I7018.WATCHDOG.ENABLE.tag)>0) then begin
    if SysTimer_Pulse(I7018.HostIsOkPeriod)>0 then begin
     I7018_EnableCmdId(cm_HostIsOk,true);
    end;
   end;
  end else if not IsPortOpened then Trouble('Port closed');
  if ((I7018.DEVCAL.TimeSendRange>0) and (mSecNow>I7018.DEVCAL.TimeSendRange+2000)) then CustomSetTag(I7018.DEVCAL.RANGE,0,'');
  if ((I7018.DEVCAL.TimeSendZero>0)  and (mSecNow>I7018.DEVCAL.TimeSendZero+2000))  then CustomSetTag(I7018.DEVCAL.ZERO ,0,'');
 end;
 
 {
 I7018 Simulator
 }
 procedure I7018_Simulator;
 var r:Real;i:Integer;
  function SimTcValue(min,max:Real):Real;
  var dat:Real;
  begin
   dat:=max*sin(2*pi*time*60);
   if dat<min then dat:=min;
   SimTcValue:=dat
  end;
 begin
  case iGetTag(I7018.INPTYPE.tag) of
   0 : r:=15*sin(2*pi*time*60);                              //  -15..15  mV ADC input range
   1 : r:=50*sin(2*pi*time*60);                              //  -50..50  mV ADC input range
   2 : r:=100*sin(2*pi*time*60);                             // -100..100 mV ADC input range
   3 : r:=500*sin(2*pi*time*60);                             // -500..500 mV ADC input range
   4 : r:=1000*sin(2*pi*time*60);                            //   -1..1    V ADC input range
   5 : r:=2500*sin(2*pi*time*60);                            // -2.5..2.5  V ADC input range
   6 : r:=20*sin(2*pi*time*60);                              //  -20..20  mA ADC input range
   7,8,9,10,11,12,13 : Problem('Invalid Input Type');
   14: r:=SimTcValue(-210,760);                        // Type J, -210..760  C
   15: r:=SimTcValue(-270,1372);                       // Type K, -270..1372 C
   16: r:=SimTcValue(-270,400);                        // Type T, -270..400  C
   17: r:=SimTcValue(-270,1000);                       // Type E, -270..1000 C
   18: r:=SimTcValue(   0,1768);                       // Type R,    0..1768 C
   19: r:=SimTcValue(   0,1768);                       // Type S,    0..1768 C
   20: r:=SimTcValue(   0,1820);                       // Type B,    0..1820 C
   21: r:=SimTcValue(-270,1300);                       // Type N, -270..1300 C
   22: r:=SimTcValue(   0,2320);                       // Type C,    0..2320 C
   23: r:=SimTcValue(-200,800);                        // Type L, -200..800  C
   24: r:=SimTcValue(-200,100);                        // Type M, -200..100  C
  end;
  CustomSetTag(I7018.TCJ,abs(random(-1,+1)*25*sin(2*pi*time*60))+rGetTag(I7018.SHIFT.tag),'');
  UpdateAo(ao_tcj,time,rGettag(I7018.TCJ.tag));
  for i:=0 to MaxChannelNum do begin
   if iGetTagBitState(I7018.AIS.tag,i) then begin
    CustomSetTag(I7018.AI[i],r,'');
    UpdateAo(I7018.CRV_AI[i],time,r);
    if I7018.isCalibr then begin
     case I7018.CalibrMode of
      NoCalibr: ;
      PuCalibr: if rSetTag(I7018.CAL.AI[i].tag,calibr(i,r,0)) then
                UpdateAo(I7018.CAL.CRV_AI[i],time,rGetTag(I7018.CAL.AI[i].tag));
      TcCalibr: if rSetTag(I7018.CAL.AI[i].tag,calibr(i,r,rGetTag(I7018.TCJ.tag))) then
                UpdateAo(I7018.CAL.CRV_AI[i],time,rGetTag(I7018.CAL.AI[i].tag));
     end;
    end;
   end else
  end;
  
 end;
 //
 // Menu Initialize
 //
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Команда "Инструменты"... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Загрузить параметры из INI файла');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@LoadIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сохранить параметры в  INI файле');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SaveIni');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Baudrate
 }
 function GetBaudrateStr(baud:Integer):String;
 var i:Integer;
 begin
  GetBaudrateStr:=HexB(I7018.RS485.UnitId)+HexB(iGetTag(I7018.INPTYPE.tag))+
  HexB(ValidateBaudRate(iGetTag(I7018.BAUDRATE.tag),baud))+HexB(I7018.DataFormat);
 end;
 procedure MenuBaudrate;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Выберите Baudrate... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('1200');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(3));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('2400');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(4));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('4800');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(5));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('9600');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(6));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('19200');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(7));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('38400');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(8));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('57600');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(9));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('115200');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetBaudrateStr(10));
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_BAUDRATE'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Data Format in Answer
 }
 function GetDataTypeStr(mode:integer):String;
 const maxbit=1;
 var datform,i:Integer;
 begin
  datform:=I7018.DataFormat;
  for i:=0 to maxbit do datform:=iSetBitState(datform,i,iGetBitState(mode,i));
  GetDataTypeStr:=HexB(I7018.RS485.UnitId)+HexB(iGetTag(I7018.INPTYPE.tag))+
  HexB(iGetTag(I7018.BAUDRATE.tag))+HexB(datform);
 end;
 procedure MenuDataType;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Выберите Формат данных... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Технические единицы');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetDataTypeStr(dfEngineering));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('% от полного диапазаона');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetDataTypeStr(dfPercentFSR));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('HEX-формат');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetDataTypeStr(dfHexadecimal));
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_DATATYPE'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Type Input Data
 }
 function GetInputTypeStr(mode:integer):String;
 begin
  GetInputTypeStr:=HexB(I7018.RS485.UnitId)+HexB(ValidateTypeCode(iGetTag(I7018.INPTYPE.tag),mode))+
                   HexB(iGetTag(I7018.BAUDRATE.tag))+HexB(I7018.DataFormat);
 end;
 procedure MenuInputType;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Выберите Тип входных данных... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('-15..+15 мВ');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(0));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('-50..+50 мВ');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(1));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('-100..+100 мВ');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(2));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('-500..+500 мВ');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(3));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('-1..+1 В');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(4));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('-2,5..+2,5 В');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(5));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('-20..+20 мА');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(6));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('J:-210..760 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(14));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('K:-270..1372  °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(15));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('T:-270..400 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(16));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('E:-270..1000 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(17));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('R:0..1768 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(18));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('S:0..1768 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(19));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('B:0..1820 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(20));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('N:-270..1300 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(21));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('C:0..2320 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(22));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('L:-200..800 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(23));
   //////////////////////////////////////////
   n:=n+EditAddInputLn('M:-200..100 °С');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SetCfg '+GetInputTypeStr(24));
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_INPTYPE'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 
 procedure EditMenuCustomHandler(id,msg:String;tag:integer;value:string); //
 var s,s1:String;
 begin
  s:='';
  s1:='';
  if EditStateDone then
  if not IsEmptyStr(id) then begin
   // Menu handler (before confirmation).
   if EditTestResultName(id) then begin
    if EditTestResultCode(mr_OK) then begin
     if LooksLikeCommand(EditGetMenuListSelectedCommand) then
     if IsEmptyStr(EditGetMenuListSelectedConfirm) then begin
      DevPostCmdLocal(EditGetMenuListSelectedCommand);
     end else begin
      s:='Вы действительно хотите выполнить:'+LineEnding+LineEnding+
         UpCaseStr(EditGetMenuListSelectedConfirm)+' ?'+
         LineEnding+LineEnding+'Эта операция требует вашего подтверждения!'+
         LineEnding+LineEnding+msg;
      s1:=id+'::Confirm';
      EditStartConfirmCommand(s,s1,EditGetMenuListSelectedCommand,EditGetSettingText);
     end;
    if tag<>0 then begin
     if typetag(tag)=1 then bNul(iSetTag(tag,val(value)));
     if typetag(tag)=2 then bNul(rSetTag(tag,rval(value)));
     if typetag(tag)=3 then bNul(sSetTag(tag,value));
    end;
    end else
    if EditTestResultCode(mr_Cancel) or EditTestResultCode(mr_Close) then begin
     if tag<>0 then begin
      if typetag(tag)=1 then bNul(iSetTag(tag,val(value)));
      if typetag(tag)=2 then bNul(rSetTag(tag,rval(value)));
      if typetag(tag)=3 then bNul(sSetTag(tag,value));
     end;
    end;
    if not EditStateBusy then EditReset;
   end;
   // Menu handler (after confirmation).
   s1:=id+'::Confirm';
   EditMessageBoxDialogExDefaultHandler(s1);
  end;
  s:='';
  s1:='';
 end;
 {
 Start Edit Data
 }
 procedure StartEditDataStrEx(Name,Caption,Data,Params:String);
 var n:Integer;
 begin
  if EditStateReady then begin
   n:=0+EditAddOpening('Введите '+Name);
   n:=n+EditAddInputLn(Caption+'|'+Data);
   if (StdEditTagFont<>'') then n:=n+EditAddSetting('@set StringGrid.Font '+StdEditTagFont);
   if (Length(Params)>0) then n:=n+EditAddText(Params,'','@>!');
   n:=n+EditAddClosing('StringGridEdit',EditGetUID(Name),'');
   if (n>0) then Problem('Error starting edit '+Name);
  end else Problem('Could not start edit tag '+Name);
  //Data:='';
 end;
 {
 Check if data editing done.
 }
 function CheckEditData(Name:String):String;
 var s,newData:String;
 begin
  s:='';newData:='';
  if EditTestResultCode(mr_OK) then begin
   s:=ExtractWordDelims(2,Edit('?ans 1'),Dump('|'));
   newData:=s;
   EditReset;
  end;
  CheckEditData:=newData;
  s:='';newData:='';
 end;
 {
 GUI Handler to process user input...
 }
 procedure I7018_GUI_Poll;
 var s,s1:String; ClickCurve,i,itemp:Integer;
  function isClickTag(var R:TTagRef):boolean;
  begin
   isClickTag:=(ClickTag=R.tag);
  end;
 begin
  s:='';s1:='';
  DIM_GuiClickBuff:='';
  {
  Handle user mouse/keyboard clicks...
  ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,cw_KeyUp,cw_MouseWheel,...)
  ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  }
  if ClickWhat<>0 then
  repeat
   //
   // Copy GUI click to DIM buffer for remote execution.
   //
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   {
   Handle MouseDown/KeyDown
   }
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    {
    Handle Left mouse button click
    }
    if (ClickButton=VK_LBUTTON) then begin
     {
     Click is local
     }
     if ClickIsLocal then begin
      {
      Click Tags
      }
      if ClickTag<>0 then begin
       if isClickTag(I7018.ADDRESS) then begin
        s:=str(iGetTag(I7018.ADDRESS.tag));
        StartEditDataStrEx('DevAdr','Адрес',s,SetFormUnderSensorLeftBottom(ClickParams('')));
       end else
       if isClickTag(I7018.BAUDRATE) then begin
        MenuBaudrate;
       end else
       if isClickTag(I7018.NAME) then begin
        StartEditDataStrEx('DevName','Имя',sGetTag(I7018.NAME.tag),SetFormUnderSensorLeftBottom(ClickParams('')));
       end else
       if isClickTag(I7018.CHECKSUM) then begin
        if iGetTag(I7018.CHECKSUM.tag)=0 then begin
         s:='Включить Checksum?'+LineEnding+LineEnding+
            'Для изменения необходимо соединить контакты *INIT и GND, в противном случае - '+
            'дальнейшее подтверждение приведет к ошибке';
         s1:='@SetCfg '+HexB(iGetTag(I7018.ADDRESS.tag))+
                        HexB(iGetTag(I7018.INPTYPE.tag))+
                        HexB(iGetTag(I7018.BAUDRATE.tag))+
                        HexB(iSetBit(I7018.DataFormat,bitCheckSum,1));
         EditStartConfirmCommand(s,EditGetUID('Enable_CHK'),s1,SetFormUnderSensorLeftBottom(ClickParams('')));
        end else 
        if iGetTag(I7018.CHECKSUM.tag)=1 then begin 
         s:='Выключить Checksum?'+LineEnding+LineEnding+
            'Для изменения необходимо соединить контакты *INIT и GND, в противном случае - '+
            'дальнейшее подтверждение приведет к ошибке';
         s1:='@SetCfg '+HexB(iGetTag(I7018.ADDRESS.tag))+
                        HexB(iGetTag(I7018.INPTYPE.tag))+
                        HexB(iGetTag(I7018.BAUDRATE.tag))+
                        HexB(iSetBit(I7018.DataFormat,bitCheckSum,0));
         EditStartConfirmCommand(s,EditGetUID('Disable_CHK'),s1,SetFormUnderSensorLeftBottom(ClickParams('')));
        end;
       end else
       if isClickTag(I7018.REJFILTER) then begin
        if iGetTag(I7018.REJFILTER.tag)=0 then begin
         s:='Установить фильтр на 50 Гц?';
         s1:='@SetCfg '+HexB(iGetTag(I7018.ADDRESS.tag))+
                        HexB(iGetTag(I7018.INPTYPE.tag))+
                        HexB(iGetTag(I7018.BAUDRATE.tag))+
                        HexB(iSetBit(I7018.DataFormat,bitReject,1));
         EditStartConfirmCommand(s,EditGetUID('Enable_Rej50'),s1,SetFormUnderSensorLeftBottom(ClickParams('')));
        end else
        if iGetTag(I7018.REJFILTER.tag)=1 then begin
         s:='Установить фильтр на 60 Гц?';
         s1:='@SetCfg '+HexB(iGetTag(I7018.ADDRESS.tag))+
                        HexB(iGetTag(I7018.INPTYPE.tag))+
                        HexB(iGetTag(I7018.BAUDRATE.tag))+
                        HexB(iSetBit(I7018.DataFormat,bitReject,0));
         EditStartConfirmCommand(s,EditGetUID('Enable_Rej50'),s1,SetFormUnderSensorLeftBottom(ClickParams('')));
        end;
       end else
       if isClickTag(I7018.DATATYPE) then begin
        MenuDataType;
       end else
       if isClickTag(I7018.INPTYPE) then begin
        MenuInputType;
       end else
       if isClickTag(I7018.DEVCAL.AUTH) then begin
        s:=str(ord(iXor(iGetTag(I7018.DEVCAL.AUTH.tag),1)));
        DevSendCmdLocal('@CalibrateAllow '+s);
       end else
       if isClickTag(I7018.DEVCAL.RANGE) then begin
        if iGetTag(I7018.DEVCAL.AUTH.tag)<>0 then DevPostCmdLocal('@CalibrationRange') else
        CallWarningMsg('Калибровка недоступна',5000,0);
       end else
       if isClickTag(I7018.DEVCAL.ZERO) then begin
        if iGetTag(I7018.DEVCAL.AUTH.tag)<>0 then DevPostCmdLocal('@CalibrationZero') else
        CallWarningMsg('Калибровка недоступна',5000,0);
       end else
       if isClickTag(I7018.SHIFT) then begin
        s:=str(rGetTag(I7018.SHIFT.tag));
        StartEditDataStrEx('ShifTcj','Смещение',s,SetFormUnderSensorLeftBottom(ClickParams('')));
       end else
       if isClickTag(I7018.POLL) then begin
        DevSendCmdLocal('@POLL');
       end else
       if isClickTag(I7018.WATCHDOG.TIMEOUT) then begin
        StartEditTagEx(I7018.WATCHDOG.TIMEOUT.tag,'Введите Watchdog Timeout:',SetFormUnderSensorLeftBottom(ClickParams('')));
       end else
       if isClickTag(I7018.WATCHDOG.ENABLE) then begin
        s:=str(rGetTag(I7018.WATCHDOG.TIMEOUT.tag))+' '+str(iXor(iGetTag(I7018.WATCHDOG.ENABLE.tag),1));
        DevSendCmdLocal('@SetWDStatus '+s);
       end else
       if isClickTag(I7018.WATCHDOG.STATUS) then begin
        DevSendCmdLocal('@ResetWDStatus');
       end else
       if isClickTag(I7018.AIS) then begin
        s:=ClickSensor; s:=StringReplace(s,'.',' ',rfReplaceAll);
        if ((pos('ENABLE',s)<>0) and (WordCount(s)=3)) then begin
         s:=ExtractWord(2,s); s:=StringOfChar(StrFetch(s,3),1); i:=val(s);
         if not isNan(rval(s)) then begin
          s1:=str(ord(not iGetTagBitState(I7018.AIS.tag,i)));
          DevPostCmdLocal('@SetMesureAI '+s+' '+s1); 
         end else Problem('Incorrect channel - '+s);
        end else Problem('Incorrect sensor - '+s);
       end;
      end;
     end;
     {
     Handle sensor clicks...
     }
     if IsSameText(ClickSensor,'HELP') then begin
      Cron('@Browse '+DaqFileRef(ReadIni('[DAQ] HelpFile'),'.htm'));
      bNul(Voice(snd_Click));
     end;
     if (ClickSensor='I7018.LABEL') then begin
      DevPostCmdLocal('@MenuTools');
     end;
     {
     Select Plot & Tab windows by curve...
     }
     ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
     if IsRefCurve(ClickCurve) then begin
      iNul(WinSelectByCurve(ClickCurve,ClickCurve));
      bNul(Voice(snd_Wheel));
     end;
     {
     Console commands: @url_encoded_sensor ...
     }
     if LooksLikeCommand(ClickSensor) then begin
      DevSendCmdLocal(url_decode(ClickSensor));
      bNul(Voice(snd_Click));
     end;
     //
     // Handle remote clicks comes from DIM via @DimGuiClick message.
     // @DimGuiClick default handler decode and write events to FIFO,
     // so we can find it as clicks and can handle it in usual way.
     //
     if ClickIsRemote then begin
      //
      // Show time difference.
      //
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      //
      // Handle remote console commands...
      //
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevSendCmdLocal(s);
     end;
    end;
    {
    Handle Right mouse button click
    }
    if (ClickButton=VK_RBUTTON) then begin
     SensorHelp(Url_Decode(ClickParams('Hint')));
    end;
   end;
  until (ClickRead=0);
  {
  Edit handling...
  }
  if EditStateDone then begin
   {
   Warning,Information.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
   EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
   s:='Для изменения необходимо '+
      'соединить контакты *INIT и GND, в противном случае - '+
      'дальнейшее подтверждение приведет к ошибке';
   EditMenuCustomHandler(EditGetUID('MENU_BAUDRATE'),s,0,'');
   EditMenuCustomHandler(EditGetUID('MENU_DATATYPE'),'',0,'');
   EditMenuCustomHandler(EditGetUID('MENU_INPTYPE'),'',0,'');
   EditMessageBoxDialogExDefaultHandler(EditGetUID('Enable_CHK'));
   EditMessageBoxDialogExDefaultHandler(EditGetUID('Disable_CHK'));
   EditMessageBoxDialogExDefaultHandler(EditGetUID('Enable_Rej50'));
   EditMessageBoxDialogExDefaultHandler(EditGetUID('Enable_Rej60'));
   //s:=CheckEditData('адрес_устройства');
   if EditTestResultName(EditGetUID('DevAdr')) then begin
    s:=CheckEditData('DevAdr');
    itemp:=val(s);
    if not IsNan(rVal(s))
    then begin
     DevSendCmdLocal('@SetCfg '+HexB(itemp)+HexB(iGetTag(I7018.INPTYPE.tag))+HexB(iGetTag(I7018.BAUDRATE.tag))+HexB(I7018.DataFormat));
    end else Warning('Invalid input!');
   end;
   if EditTestResultName(EditGetUID('DevName')) then begin
    s:=CheckEditData('DevName');
    DevSendCmdLocal('@SetName '+s);
   end;
   if EditTestResultName(EditGetUID('ShifTcj')) then begin
    s:=CheckEditData('ShifTcj');
    if not IsNan(rVal(s))
    then DevPostCmdLocal('@SetShiftTCJ '+s)
    else Warning('Invalid input!');
   end;
   if CheckEditTag(I7018.WATCHDOG.TIMEOUT.tag,s) then begin
    UpdateTag(I7018.WATCHDOG.TIMEOUT.tag,s,0,25.5);
    s:=str(rGetTag(I7018.WATCHDOG.TIMEOUT.tag))+' '+str(iGetTag(I7018.WATCHDOG.ENABLE.tag));
    DevPostCmdLocal('@SetWDStatus '+s);
   end;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end else
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  s:='';
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  I7018_ClearStrings;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 var s:String;
 begin
  s:='';
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  InitNetRS485;
  I7018_Init;
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  I7018.StartPoll:=iGetTag(I7018.Poll.tag);
  s:=str(rGetTag(I7018.SHIFT.tag));
  DevPostCmdLocal('@SetShiftTCJ '+s);
  s:='';
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))<>0 then iNul(CustomIniRW('W','',2));
  I7018_Clear;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  I7018_GUI_Poll;
  if I7018.CmdTab.PollCnt=1 then begin
   CustomSetTag(I7018.POLL,I7018.StartPoll,'');
   I7018.CmdTab.PollCnt:=2;
  end else
  if I7018.CmdTab.PollCnt=maxint then I7018.CmdTab.PollCnt:=2;
  if mSecNow-FixmSecNow>I7018.RS485.DelayOnStart then
  if not DIM_IsClientMode then begin
   if not I7018.Simulator then begin
    I7018_Poll;
    //UpdateState;
   end else I7018_Simulator;
  end;
  {
  Update Host Date-Time.
  }
  if DIM_IsServerMode then begin
   if (SysTimer_Pulse(1000)>0) then begin
    bNul(sSetTag(I7018.SERVID.tag,I7018.SelfId));
    bNul(sSetTag(I7018.CLOCK.tag,GetDateTime(mSecNow)));
    bNul(SetTagColor(I7018.SERVID.tag,ColorNorm));
    bNul(SetTagColor(I7018.CLOCK.tag,ColorNorm));
   end;
  end;
  if DIM_IsClientMode then begin
   if (ShouldRefresh(I7018.CLOCK.dat,GetStampOfTag(I7018.CLOCK.tag,0))>0) then begin
    bNul(SetTagColor(I7018.SERVID.tag,ColorNorm));
    bNul(SetTagColor(I7018.CLOCK.tag,ColorNorm));
    I7018.CLOCK.tim:=mSecNow;
   end;
   if (SysTimer_Pulse(1000)>0) then
   if (mSecNow-I7018.CLOCK.tim>DimDeadline) then begin
    bNul(sSetTag(I7018.SERVID.tag,'Server Disconnected'));
    bNul(SetTagColor(I7018.SERVID.tag,ColorWarn));
    bNul(SetTagColor(I7018.CLOCK.tag,ColorWarn));
   end;
  end;
  UpdateState;
  I7018_DIM_UpdateTag;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var 
  cmd,arg,cdm_arg_nocode,s,dat:String; 
  itemp,cmdid,i,j,tag,ref,cid,tim,port,uid:Integer;
  r,rtemp:Real;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  s:='';cmd:='';arg:='';cdm_arg_nocode:='';dat:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @RS485.Reply
   }
   if (cmdid = cmd_RsReply) then begin
    if RS485_proxy_reply(cmd,arg,ref,cid,tim,port,uid,dat)
    then I7018_OnReply(ref,cid,tim,port,uid,dat)
    else Trouble('Bad '+cmd+' format');
    ClearRS485Poll;
    Data:='';
   end else
   {
   @RS485.Refuse
   }
   if (cmdid = cmd_RsRefuse) then begin
    if RS485_proxy_reply(cmd,arg,ref,cid,tim,port,uid,dat)
    then Trouble(RS485_proxy_nice(cmd,ref,cid,tim,port,uid,dat,0))
    else Trouble(cmd+' '+arg);
    UpdateDo(do_state,time,stt_Refused); CustomSetTag(I7018.STATE,stt_Refused,'');
    ClearRS485Poll;
    Data:='';
   end else
   {
   @RS485.Timeout
   }
   if (cmdid = cmd_RsTimeout) then begin
    if RS485_proxy_reply(cmd,arg,ref,cid,tim,port,uid,dat)
    then Problem(RS485_proxy_nice(cmd,ref,cid,tim,port,uid,dat,64))
    else Trouble(cmd+' '+arg);
    UpdateDo(do_state,time,stt_TimeOut); CustomSetTag(I7018.STATE,stt_TimeOut,'');
    ClearRS485Poll;
    Data:='';
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    if DIM_IsClientMode and not DIM_IsServerMode then begin
     tag:=FindTag(Trim(arg));
     if tag=I7018.STATE.tag      then UpdateDo(do_state,time,iGetTag(tag));
     if tag=I7018.RX.tag         then UpdateDo(do_rx   ,time,iGetTag(tag));
     if tag=I7018.TX.tag         then UpdateDo(do_tx   ,time,iGetTag(tag));
     for i:=0 to MaxChannelNum do begin
      if tag=I7018.AI[i].tag     then UpdateAo(I7018.CRV_AI[i],  time,rGetTag(tag));
      if tag=I7018.CAL.AI[i].tag then UpdateAo(I7018.CAL.CRV_AI[i],  time,rGetTag(tag));
     end;
     if tag=I7018.TCJ.tag        then UpdateAo(ao_tcj,time,rGetTag(tag));
     if tag=I7018.SHIFT.tag      then UpdateAo(ao_shift,time,rGetTag(tag));
     if tag=I7018.AIS.tag        then UpdateDo(do_ais,time,iGetTag(tag));
    end;
   end else
   {
   @DIMCMDMY
   }
   if (cmdid=cmd_DimCmdMy) then begin
    //декодируем из base64
    cdm_arg_nocode:=base64_decode(arg);
    // Избавляемся от символ NUL (\x00).
    cdm_arg_nocode:=Trim(cdm_arg_nocode);
    if LooksLikeCommand(cdm_arg_nocode) then DevPostCmdLocal(cdm_arg_nocode);
    Success('@DIMCMDMY='+cmd+' arg<'+cdm_arg_nocode+'>');
   end else
   {
   @Poll
   }
   if (cmdid = cmd_Poll) then begin
    if not DIM_IsClientMode then
     bNul(iSetTagXor(I7018.POLL.tag,1))
    else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @Poll');
    end;
   end else
   {
   @SetCfg NNTTCCFF
   }
   if (cmdid = cmd_SetCfg) then begin
    if length(arg)=8 then begin
     s:='$'+copy(arg,1,2);
     rtemp:=rval(s);
     if not isNan(rtemp) then begin
      s:='$'+copy(arg,3,2);
      itemp:=val(s);
      if isValidTypeCode(itemp) then begin
       s:='$'+copy(arg,5,2);
       itemp:=val(s);
       if isValidBaudrate(itemp) then begin
        s:='$'+copy(arg,7,2);
        itemp:=val(s);
        if isValidDataFormat(itemp) then begin
         if not DIM_IsClientMode then begin
           if I7018.Simulator then begin
            s:=Copy(arg,1,2);
            s:='$'+s;
            I7018.RS485.UnitId:=val(s);
            CustomSetTag(I7018.ADDRESS,I7018.RS485.UnitId,'');
            s:='$'+Copy(arg,3,2);
            CustomSetTag(I7018.INPTYPE,val(s),'');
            s:='$'+Copy(arg,5,2);
            CustomSetTag(I7018.BAUDRATE,val(s),'');
            s:='$'+Copy(arg,7,2);
            bNul(iSetTagBitState(I7018.CHECKSUM.tag,0,isbit(val(s),6)));
            bNul(iSetTagBitState(I7018.REJFILTER.tag,0,isbit(val(s),7)));
            bNul(iSetTagBitState(I7018.DATATYPE.tag,0,isbit(val(s),0)));bNul(iSetTagBitState(I7018.DATATYPE.tag,1,isbit(val(s),1)));
           end else begin
            I7018.CmdTab.OpData[cm_SetModuleConf]:=arg;
            I7018_EnableCmdId(cm_SetModuleConf,true);
           end;
         end else if DIM_IsClientMode then begin
          DevPostCmdLocal('@Remote @SetCfg '+arg);
         end;
        end else Trouble('Unrecognize "Data Format" in command @SetCfg');
       end else Trouble('Unrecognize "Baudrate" in command @SetCfg');
      end else Trouble('Unrecognize "Type" in command @SetCfg');
     end else Trouble('Unrecognize "Address" in command @SetCfg');
    end else Trouble('Unrecognize parametr in command @SetCfg');
   end else
   {
   @ReadCfg
   }
   if (cmdid = cmd_ReadCfg) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator
     then Success('Command @ReadCfg complete in simulator mode')
     else I7018_EnableCmdId(cm_ReadModuleConf,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadCfg');
    end;
   end else
   {
   @ReadVer
   }
   if (cmdid = cmd_ReadVer) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator
     then CustomSetTag(I7018.VERSION,0,'SimVer')
     else I7018_EnableCmdId(cm_ReadFirmwarVer,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadVer');
    end;
   end else
   {
   @ReadName
   }
   if (cmdid = cmd_ReadName) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator
     then Success('Command @ReadName complete in simulator mode')
     else I7018_EnableCmdId(cm_ReadModuleName,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadName');
    end;
   end else
   {
   @SetName
   }
   if (cmdid = cmd_SetName) then begin
    if length(arg)>6 then arg:=copy(arg,1,6);
    if not DIM_IsClientMode then begin
     if I7018.Simulator
     then CustomSetTag(I7018.NAME,0,arg)
     else begin
      I7018.CmdTab.OpData[cm_SetModuleName]:=arg;
      I7018_EnableCmdId(cm_SetModuleName,true);
     end;
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @SetName '+arg);
    end;
   end else
   {
   @ReadAI
   }
   if (cmdid = cmd_ReadAI) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator
     then Success('Command @ReadAI complete in simulator mode')
     else I7018_EnableCmdId(cm_ReadAnalogInp,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadAI');
    end;
   end else
   {
   @SetMesureAI n state
   n - channel number 0..7
   state - 0/1 disable/enable
   }
   if (cmdid = cmd_SetMesAI) then begin
    i:=val(ExtractWord(1,arg)); j:=val(ExtractWord(2,arg));
    if not (isNan(rVal(ExtractWord(1,arg))) or isNan(rVal(ExtractWord(2,arg)))) then begin
     if (((0<=i) and (i<=MaxChannelNum)) and ((j=0) or (j=1))) then begin
      if not DIM_IsClientMode then begin
       if I7018.Simulator then begin
        Success('Command @SetMesureAI complete in simulator mode');
        bNul(iSetTagBit(I7018.AIS.tag,i,j));
       end else begin
        itemp:=iSetBit(iGetTag(I7018.AIS.tag),i,j);
        I7018.CmdTab.OpData[cm_SetMesureAI]:=HexB(itemp);
        I7018_EnableCmdId(cm_SetMesureAI,true);
       end;
      end else if DIM_IsClientMode then begin
       DevPostCmdLocal('@Remote @SetMesureAI '+arg);
      end;
     end else Problem('Incorrect channel number or state');
    end else Problem('Incorrect arg in @SetMesureAI');
   end else
   {
   @ReadMesureAI
   }
   if (cmdid = cmd_ReadMesAI) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator then
      Success('Command @ReadMesureAI complete in simulator mode')
     else I7018_EnableCmdId(cm_ReadMesureAI,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadMesureAI');
    end;
   end else
   {
   @CalibrateAllow 0/1
   }
   if (cmdid = cmd_CalAllow) then begin
    s:=trim(arg);
    i:=val(s);
    if ((i=0) or (i=1)) then begin
     if not DIM_IsClientMode then begin
      CustomSetTag(I7018.DEVCAL.AUTH,i,'');
      if not I7018.Simulator then begin
       I7018.CmdTab.OpData[cm_CalAllow]:=s;
       I7018_EnableCmdId(cm_CalAllow,true);
      end;
     end else if DIM_IsClientMode then begin
      DevPostCmdLocal('@Remote @CalibrateAllow '+s);
     end;
    end else Problem('Incorrect arg in @CalibrateAllow');
   end else
   {
   @CalibrateRange
   }
   if (cmdid = cmd_CalRange) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator then
      Success('Command @CalibrateRange complete in simulator mode')
     else I7018_EnableCmdId(cm_CalRange,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @CalibrateRange');
    end;
   end else
   {
   @CalibrateZero
   }
   if (cmdid = cmd_CalZero) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator then
      Success('Command @CalibrateZero complete in simulator mode')
     else I7018_EnableCmdId(cm_CalZero,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @CalibrateZero');
    end;
   end else
   {
   @ReadTCJ
   }
   if (cmdid = cmd_ReadTCJ) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator then
      Success('Command @ReadTCJ complete in simulator mode')
     else I7018_EnableCmdId(cm_ReadTCJ,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadTCJ');
    end;
   end else
   {
   @SetShiftTCJ data
   }
   if (cmdid = cmd_SetShiftTCJ) then begin
    rtemp:=rval(arg);
    if rtemp>40.96 then rtemp:=40.96;
    if rtemp<-40.96 then rtemp:=-40.96;
    if not isNan(rtemp) then begin
     if not DIM_IsClientMode then begin
      I7018.ShiftTCJ:=rtemp;
      if not I7018.Simulator then begin
       I7018.CmdTab.OpData[cm_SetShiftTCJ]:=CalcShiftTCJ(rtemp);
       I7018_EnableCmdId(cm_SetShiftTCJ,true);
      end;
     end else if DIM_IsClientMode then begin
      s:=str(rtemp);
      DevPostCmdLocal('@Remote @SetShiftTCJ '+s);
     end;
    end else Problem('Incorrect arg in @SetShiftTCJ');
   end else
   {
   @ReadWDStatus
   }
   if (cmdid = cmd_ReadWDStatus) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator
     then Success('Command @ReadWDStatus complete in simulator mode')
     else I7018_EnableCmdId(cm_ReadWDTStatus,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadWDStatus');
    end;
   end else
   {
   @ReadWDTimeout
   }
   if (cmdid = cmd_ReadWDTimeout) then begin
    if not DIM_IsClientMode then begin
     if I7018.Simulator
     then Success('Command @ReadWDTimeout complete in simulator mode')
     else I7018_EnableCmdId(cm_ReadWDTimeOut,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ReadWDTimeout');
    end;
   end else
   {
   @SetWDStatus TimeOut Enabled (Timeout - time in 0 - 25.5 s, Enabled - 0 or 1)
   }
   if (cmdid = cmd_SetWDStatus) then begin
    if WordCount(arg)=2 then begin
     s:=extractword(1,arg);s:=copy(s,1,pos('.',s)+1);
     rtemp:=rVal(s);
     if not isNan(rtemp) then begin
      if rtemp>25.5 then rtemp:=25.5;
      if rtemp<0    then rtemp:=0;
      if not isNan(rval(extractword(2,arg))) then begin
       s:=extractword(2,arg);
       itemp:=val(s);
       if itemp>=1 then itemp:=1;
       if itemp<=0 then itemp:=0;
        if not DIM_IsClientMode then begin
         I7018.EnableWD:=itemp;
         I7018.TimeoutWD:=rtemp;
         if I7018.Simulator then begin 
          CustomSetTag(I7018.WATCHDOG.STATUS,I7018.EnableWD,'');
          CustomSetTag(I7018.WATCHDOG.TIMEOUT,I7018.TimeoutWD,'');
         end else begin
          I7018.CmdTab.OpData[cm_SetWDTStatus]:=str(I7018.EnableWD)+HexB(trunc(I7018.TimeoutWD*10));
          I7018_EnableCmdId(cm_SetWDTStatus,true);
         end;
        end else if DIM_IsClientMode then begin
         DevPostCmdLocal('@Remote @SetWDStatus '+arg);
        end;
      end else Trouble('Unrecognize "Enabled" in command @SetWDStatus');
     end else Trouble('Unrecognize "Timeout" in command @SetWDStatus');
    end else Trouble('Unrecognize parameter in command @SetWDStatus');
   end else
   {
   @ResetWDStatus
   }
   if (cmdid = cmd_RstWDStatus) then begin
    if not DIM_IsClientMode then begin
     CustomSetTag(I7018.WATCHDOG.ENABLE,0,'');
     if I7018.Simulator
     then CustomSetTag(I7018.WATCHDOG.STATUS,0,'')
     else I7018_EnableCmdId(cm_RstWDTStatus,true);
    end else if DIM_IsClientMode then begin
     DevPostCmdLocal('@Remote @ResetWDStatus');
    end;
   end else
   {
   @MenuTools
   }
   if (cmdid=cmd_MenuTools) then begin
    MenuToolsStarter;
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  s:='';cmd:='';arg:='';cdm_arg_nocode:='';dat:='';
  Data:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{<== Please never change this code ***}
{***************************************************}
