 {
 ***********************************************************************
 MERA Modbus Driver.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @Modbus.XXX    - response come from &ModbusProxy server:
| @Modbus.Reply    ref cid tim port uid fid $$ans
| @Modbus.Refuse   ref cid tim port uid fid   msg
| @Modbus.Timeout  ref cid tim port uid fid $$req
|  ref  - sender device reference, expected &ModbusProxy.
|  cid  - command id, any user defined custom data.
|  tim  - request-response time, ms.
|  port - logical port on &ModbusProxy, expected 1..16.
|  uid  - unit id, i.e.MODBUS address, expected 1..247.
|  fid  - function id, expected 1,2,3,4,5,6,15,16.
|  ans  - answer  (response), in HEX_ENCODE format.
|  req  - request (original), in HEX_ENCODE format.
|  msg  - plain text message to explain reason of Refuse.
| @PARAM.XX      - write parameter, i.e. send to MODBUS
| @Edit PARAM.XX - edit parameter and write it
| @LoadIni - load params from INI file.
| @SaveIni - save params to   INI file.
|********************************************************
[]
 Configuration:
 ***********************************************************************
  [DeviceList]
  &MERA.DRIVER = device software program
  [&MERA.DRIVER]
  Comment        = MERA MODBUS DRIVER.
  InquiryPeriod  = 0
  DevicePolling  = 10, tpNormal
  ProgramSource  = ..\DaqPas\mera_drv.pas
  DigitalFifo    = 1024
  AnalogFifo     = 1024
  DebugFlags     = 3
  OpenConsole    = 2
  Simulator      = 0
  ModbusPort     = 1
  ModbusUnitId   = 1
  ModbusTimeout  = 250
  ModbusDeadline = 60000
  ModbusPolling  = 100
  DelayOnStart   = 1000
  tagPrefix      = MERA
  ...etc...
 ***********************************************************************
 }
program MERA_Modbus_Driver;      { MERA Modbus Driver               }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 {$I _con_NetLibrary}            { NetLibrary constants             }
 dfStatist       = 32;           { DebugFlags - polling statistics  }
 MaxCmdNum       = 6;            { Maximal number of commands:      }
 cm_ReadPress    = 1;            { Read Regs 0-1 Press measurement  }
 cm_ReadDTime    = 2;            { Read HR 140-145 Date/Time        }
 cm_ReadVer      = 3;            { Read HR 131 Device version       }
 cm_ReadAddr     = 4;            { Read HR 210 Device address       }
 cm_WritAddr     = 5;            { Write MR 210 Device address      }
 cm_WritDTime    = 6;            { Write MR 140 Date/Time           }
 ao_MEASPRESS    = 0;            { Pressure measure                 }
 ao_VERSION      = 1;            { Device version                   }
 ao_ADDRESS      = 2;            { Device address                   }
 do_STATE_INDIC  = 0;            { Status indicator                 }
 do_POLLRATERX   = 1;            { DigitalOutput - Poll rate Rx     }
 do_POLLRATETX   = 2;            { DigitalOutput - Poll rate Tx     }
 do_POLLRATEEX   = 3;            { DigitalOutput - Poll rate Ex     }
 do_ERRORCOUNT   = 4;            { DigitalOutput - Error counter    }
 do_POLLSUMMRX   = 5;            { DigitalOutput - Poll summ Rx     }
 do_POLLSUMMTX   = 6;            { DigitalOutput - Poll summ Tx     }
 do_POLLSUMMEX   = 7;            { DigitalOutput - Poll summ Ex     }
 SwapModeFloat   = 1;            { Byte swap mode for float values  }
 SwapModeInt32   = 3;            { Byte swap mode for integers      }
 mb_NORMAL       = 0;            { Modbus state normal              }
 mb_TIMEOUT      = 1;            { Modbus state timeout             }
 mb_REFUSE       = 2;            { Modbus state refuse              }
 mb_DEADLINE     = 3;            { Modbus state deadline            }
 st_NORMAL       = 0;            { Green                            }
 st_TIMEOUT      = 1;            { Red timeout                      }
 st_REFUSE       = 2;            { Red refuse                       }
 st_SIMULAT      = 3;            { Simulator mode                   }
 st_DISABLE      = 4;            { Polling disabled                 }

type
 TTagRef = record tag,nai,ndi,nao,ndo:Integer; val,tim:Real; end;

var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 {$I _var_NetLibrary}            { NetLibrary variables             }
 MERA       : record             { MERA Driver data record          }
  Simulator : Boolean;           { Simulator or Driver mode         }
  step2mm   : Integer;           { Conversion factor steps to mm    }
  Modbus    : record             { Modbus data record               }
   Port     : Integer;           { Logical Port on &ModbusProxy     }
   UnitId   : Integer;           { Modbus unit id                   }
   Timeout  : Integer;           { Modbus timeout, ms               }
   Polling  : Integer;           { Modbus polling period, ms        }
   Deadline : Integer;           { Modbus deadline time, ms         }
   DelayOnStart : Integer;       { Command cycle delay on start     }
   Cmd      : record             { Command cycle data record        }
    Num     : Integer;           { Current running command number   }
    Enabled : array [1..MaxCmdNum] of Boolean; { Enable polling     }
    FuncId  : array [1..MaxCmdNum] of Integer; { Modbus Function Id }
    SAddr   : array [1..MaxCmdNum] of Integer; { Start Address      }
    Quant   : array [1..MaxCmdNum] of Integer; { Quantity of data   }
    OpData  : array [1..MaxCmdNum] of Real;    { Operation data     }
    OpBuff  : array [1..MaxCmdNum] of Real;    { Operation buffer   }
   end;                          {                                  }
   Poll     : record             { Polling transaction data record  }
    ref     : Integer;           { Last sent device reference       }
    cid     : Integer;           { Last sent command id             }
    tim     : Real;              { Last polling time, ms            }
    port    : Integer;           { Last polling port                }
    uid     : Integer;           { Last polling unit id             }
    fid     : Integer;           { Last sent function id            }
    saddr   : Integer;           { Last sent start address          }
    quant   : Integer;           { Last sent quantity of registers  }
    dat     : String;            { Last sent PDU=(fid+dat) data     }
    Summ,                        { Poll summ counters               }
    Rate    : record             { Poll rate on last second         }
     Rx     : Real;              { Rx poll summ/rate - receiver     }
     Tx     : Real;              { Tx poll summ/rate - transmitter  }
     Ex     : Real;              { Ex poll summ/rate - errors       }
    end;                         {                                  }
   end;                          {                                  }
  end;                           {                                  }
  POLL      : record             { Poll parameters                  }
   ENABLE   : TTagRef;           { Poll enable flag                 }
  end;                           {                                  }
  GUI       : record             { Graphical User Interface data    }
   CMD      : record             { Commands                         }
    HELP    : TTagRef;           {                                  }
    SAVEINI : TTagRef;           {                                  }
    LOADINI : TTagRef;           {                                  }
    CONSOLE : TTagRef;           {                                  }
    TOOLS   : TTagRef;           {                                  }
   end;                          {                                  }
  end;                           {                                  }
  MEAS      : record             { Measure data                     }
   PRESS    : TTagRef;           { Pressure measurement             }
  end;                           {                                  }
  PARAM     : record             { Device analog parameters         }
   DTIME    : TTagRef;           { Date and time parameter          }
   VERSION  : TTagRef;           { Device version parameter         }
   ADDRESS  : TTagRef;           { Device address parameter         }
   YEAR     : TTagRef;           { Device date - year               }
   MONT     : TTagRef;           { Device date - month              }
   DATE     : TTagRef;           { Device date - date               }
   HOUR     : TTagRef;           { Device date - hour               }
   MIN      : TTagRef;           { Device date - minutes            }
   SEC      : TTagRef;           { Device date - seconds            }
  end;                           {                                  }
  STATE     : record             {                                  }
   INDIC    : TTagRef;           { Status indicator tag             }
  end;                           {                                  }
  SIM       : record             { Simulator data                   }
   MEASPRESS: Real;              {                                  }
   ADDRESS  : Integer;           {                                  }
   YEAR     : Integer;           {                                  }
   MONT     : Integer;           {                                  }
   DATE     : Integer;           {                                  }
   HOUR     : Integer;           {                                  }
   MIN      : Integer;           {                                  }
   SEC      : Integer;           {                                  }
  end;                           {                                  }
 end;                            {                                  }
 cmd_ClearModbusSumm       : Integer; { @ClearModbusSumm            }
 cmd_Edit                  : Integer; { @Edit                       }
 cmd_MenuToolsConfirmation : Integer; { @MenuToolsConfirmation      }
 cmd_LoadIni               : Integer; { @LoadIni                    }
 cmd_SaveIni               : Integer; { @SaveIni                    }
 cmd_SetAddr               : Integer; { @SetAddr                    }
 cmd_ParamRTime            : Integer; { @Param.RTime                }
 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_NetLibrary}            { NetLibrary functions             }
 //
 // Command cycle routines.
 //
 function IsValidCmdNum(Num:Integer):Boolean;
 begin
  IsValidCmdNum:=(1<=Num) and (Num<=MaxCmdNum);
 end;
 function ValidateCmdNum(Num:Integer):Integer;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then ValidateCmdNum:=Num
  else ValidateCmdNum:=1;
 end;
 function IsUsableCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsUsableCmdNum:=(MERA.Modbus.Cmd.FuncId[Num]<>0)
  else IsUsableCmdNum:=False;
 end;
 function IsEnabledCmdNum(Num:Integer):Boolean;
 begin
  if (1<=Num) and (Num<=MaxCmdNum)
  then IsEnabledCmdNum:=(MERA.Modbus.Cmd.FuncId[Num]<>0) and MERA.Modbus.Cmd.Enabled[Num]
  else IsEnabledCmdNum:=False;
 end;
 function NextEnabledCmdNum(Num:Integer):Integer;
 var i:Integer;
 begin
  i:=0;
  while (i<MaxCmdNum) do begin
   Num:=ValidateCmdNum(Num+1);
   if IsEnabledCmdNum(Num)
   then i:=MaxCmdNum
   else i:=i+1;
  end;
  NextEnabledCmdNum:=Num;
 end;
 procedure EnableCmdNum(Num:Integer; Enabled:Boolean);
 begin
  if IsUsableCmdNum(Num) then MERA.Modbus.Cmd.Enabled[Num]:=Enabled;
 end;
 procedure HoldCmdOpData(Num:Integer; OpData:Real);
 begin
  if IsUsableCmdNum(Num) then begin
   MERA.Modbus.Cmd.Enabled[Num]:=not IsNaN(OpData);
   MERA.Modbus.Cmd.OpBuff[Num]:=OpData;
  end;
 end;
 procedure ApplyCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then
  if not IsNaN(MERA.Modbus.Cmd.OpBuff[Num]) then begin
   MERA.Modbus.Cmd.OpData[Num]:=MERA.Modbus.Cmd.OpBuff[Num];
   MERA.Modbus.Cmd.OpBuff[Num]:=_NaN;
  end;
 end;
 procedure ReleaseCmdOpData(Num:Integer);
 begin
  if IsUsableCmdNum(Num) then HoldCmdOpData(Num,MERA.Modbus.Cmd.OpBuff[Num]);
 end;
 function GetCmdOpData(Num:Integer):Real;
 begin
  if IsUsableCmdNum(Num)
  then GetCmdOpData:=MERA.Modbus.Cmd.OpData[Num]
  else GetCmdOpData:=_NaN;
 end;
 procedure InitCmdItem(Num:Integer; Enabled:Boolean; FuncId,SAddr,Quant:Integer; OpData,OpBuff:Real);
 begin
  if IsValidCmdNum(Num) then begin
   MERA.Modbus.Cmd.Enabled[Num]:=Enabled;
   MERA.Modbus.Cmd.FuncId[Num]:=FuncId;
   MERA.Modbus.Cmd.SAddr[Num]:=SAddr;
   MERA.Modbus.Cmd.Quant[Num]:=Quant;
   MERA.Modbus.Cmd.OpData[Num]:=OpData;
   MERA.Modbus.Cmd.OpBuff[Num]:=OpBuff;
  end;
 end;
 //
 // Command table cleanup and initialization.
 //
 procedure ClearCmdTable;
 var Num:Integer;
 begin
  MERA.Modbus.Cmd.Num:=0;
  for Num:=1 to MaxCmdNum do InitCmdItem(Num,False,0,0,0,0,_NaN);
 end;
 procedure InitCmdTable;
 begin
  ClearCmdTable;
  // Table of CommandId     Enabled FunctionId        SAddr  Quant OpData OpBuff
  InitCmdItem(cm_ReadPress, True,   modbus_fn_ReadHR, 0,     2,    0,     _NaN);   // Read HR 0-1
  InitCmdItem(cm_ReadDTime, True,   modbus_fn_ReadHR, 320,   6,    0,     _NaN);   // Read HR 140-145
  InitCmdItem(cm_ReadVer,   True,   modbus_fn_ReadHR, 305,   1,    0,     _NaN);   // Read HR 131
  InitCmdItem(cm_ReadAddr,  True,   modbus_fn_ReadHR, 528,   1,    0,     _NaN);   // Read HR 210
  InitCmdItem(cm_WritAddr,  False,  modbus_fn_WritMR, 528,   1,    0,     _NaN);   // Write MR 210
  InitCmdItem(cm_WritDTime, False,  modbus_fn_WritMR, 320,   6,    0,     _NaN);   // Write MR 140
 end;
 //
 // Assign modbus last sent polling request record.
 //
 procedure AssignModbusPoll(ref,cid:Integer; tim:Real; port,uid,fid,saddr,quant:Integer; dat:String);
 begin
  MERA.Modbus.Poll.ref:=ref;     MERA.Modbus.Poll.cid:=cid;      MERA.Modbus.Poll.tim:=tim;
  MERA.Modbus.Poll.port:=port;   MERA.Modbus.Poll.uid:=uid;      MERA.Modbus.Poll.fid:=fid;
  MERA.Modbus.Poll.saddr:=saddr; MERA.Modbus.Poll.quant:=quant;  MERA.Modbus.Poll.dat:=dat;
 end;
 //
 // Clear modbus polling request to be ready for next polling.
 //
 procedure ClearModbusPoll;
 begin
  MERA.Modbus.Poll.cid:=0;
  MERA.Modbus.Poll.dat:='';
 end;
 //
 // Clear modbus poll summ counters.
 //
 procedure ClearModbusSumm;
 begin
  MERA.Modbus.Poll.Summ.Rx:=0;
  MERA.Modbus.Poll.Summ.Tx:=0;
  MERA.Modbus.Poll.Summ.Ex:=0;
 end;
 //
 // Clear modbus poll rate counters.
 //
 procedure ClearModbusRate;
 begin
  MERA.Modbus.Poll.Rate.Rx:=0;
  MERA.Modbus.Poll.Rate.Tx:=0;
  MERA.Modbus.Poll.Rate.Ex:=0;
 end;
 //
 // Increment summ and rate counters.
 //
 procedure IncSummRate(var summ,rate:Real);
 begin
  summ:=summ+1; rate:=rate+1;
 end;
 //
 // Format error message with with error counter increment.
 // Usage is like: Trouble(GotBug(...)); Problem(GotBug(...));
 //
 function GotBug(msg:String):String;
 begin
  IncSummRate(MERA.Modbus.Poll.Summ.Ex,MERA.Modbus.Poll.Rate.Ex);
  GotBug:=msg;
 end;
 {
 Nice tooltip notifier.
 }
 procedure NiceNotify(aText:String; aDelay:Integer);
 begin
  if Length(aText)>0 then begin
   ShowTooltip('text "'+aText+'" preset stdNotify delay '+Str(aDelay));
  end;
 end;
 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  StdSensorHelpTooltip(s,15000);
 end;
 {
 Get text of Tools menu items
 }
 function MenuToolsItem(MenuTools:Integer):String;
 begin
  if MenuTools=0 then MenuToolsItem:='Очистка счетчиков ошибок '+DevName     else
  if MenuTools=1 then MenuToolsItem:='Очистка счетчиков ошибок '+ModbusProxy else
  if MenuTools=2 then MenuToolsItem:='Перезапустить драйвер '+DevName        else
  if MenuTools=3 then MenuToolsItem:='Вызвать окно справочной информации'    else
  if MenuTools=4 then MenuToolsItem:='Сохранить параметры в  INI файле'      else
  if MenuTools=5 then MenuToolsItem:='Загрузить параметры из INI файла'      else
  if MenuTools=6 then MenuToolsItem:='Пока не реализована'                   else
  if MenuTools=7 then MenuToolsItem:='Пока не реализована'                   else
  MenuToolsItem:='';
 end;
 {
 Execute Tools menu commands
 }
 procedure MenuToolsCmnd(MenuTools:Integer);
 begin
  if MenuTools=0 then DevSendCmd(devMySelf,'@ClearModbusSumm');
  if MenuTools=1 then DevSendCmd(devModbusProxy,'@ZeroPortCounters 0');
  if MenuTools=2 then Cron('@Eval @System @Async @Daq Compile '+DevName);
  if MenuTools=3 then Cron('@Eval @System @Async @Run '+DaqFileRef(ReadIni('['+devname+'] HelpFile'),'.htm'));
  if MenuTools=4 then DevSendCmd(devMySelf,'@SaveIni');
  if MenuTools=5 then DevSendCmd(devMySelf,'@LoadIni');
  if MenuTools=6 then NiceNotify('Желаю успешной работы',10000);
  if MenuTools=7 then NiceNotify('Желаю успешной работы',10000);
 end;
 //
 // Find the current Proxy device to send @Modbus.Poll message.
 //
 function devTheProxy:Integer;
 begin
  if MERA.Simulator                 // In simulation mode
  then devTheProxy:=devMySelf       // send messages to himself
  else devTheProxy:=devModbusProxy; // otherwise use &ModbusProxy
 end;
 //
 // Check if communication port is opened or not.
 // As long as we use Proxy device, check Port number & device reference.
 //
 function IsPortOpened:Boolean;
 begin
  IsPortOpened:=(MERA.Modbus.Port>0) and (devTheProxy<>0);
 end;
 //
 // Encode/decode date/time in 6 registers.
 //
 function encode_datetime(ms:Real):String;
 var year,month,day,hour,min,sec:Integer;
 begin
  year:=ms2year(ms); month:=ms2month(ms); day:=ms2day(ms);
  hour:=ms2hour(ms); min:=ms2min(ms);     sec:=ms2sec(ms);
  encode_datetime:=modbus_dump_int16(year,SwapModeInt32)
                  +modbus_dump_int16(month,SwapModeInt32)
                  +modbus_dump_int16(day,SwapModeInt32)
                  +modbus_dump_int16(hour,SwapModeInt32)
                  +modbus_dump_int16(min,SwapModeInt32)
                  +modbus_dump_int16(sec,SwapModeInt32);
 end;
 procedure decode_datetime(year,mont,date,hour,imin,isec:Integer);
 var yea,mon,dat,hou,min,sec,s:String;
 begin
  yea:='';mon:='';dat:='';hou:='';min:='';sec:='';s:='';
  if (year>1900) and (year<2100) then yea:=Str(year) else yea:='0000';
  if (mont>=1)   and (mont<=12)  then mon:=Str(mont) else mon:='00';
  if (date>=1)   and (date<=31)  then dat:=Str(date) else dat:='00';
  if (hour>=0)   and (hour<=23)  then hou:=Str(hour) else hou:='00';
  if (imin>=0)   and (imin<=59)  then min:=Str(imin) else min:='00';
  if (isec>=0)   and (isec<=59)  then sec:=Str(isec) else sec:='00';
  if length(mon)<2 then mon:=LeftPad(mon,2,'0');
  if length(dat)<2 then dat:=LeftPad(dat,2,'0');
  if length(hou)<2 then hou:=LeftPad(hou,2,'0');
  if length(min)<2 then min:=LeftPad(min,2,'0');
  if length(sec)<2 then sec:=LeftPad(sec,2,'0');
  s:=yea+'.'+mon+'.'+dat+'-'+hou+':'+min+':'+sec;
  bNul(sSetTag(MERA.PARAM.DTIME.tag,s));
  yea:='';mon:='';dat:='';hou:='';min:='';sec:='';s:='';
 end;
 //
 // Validation of date and time
 //
 function IsValidDateTime(ms:Real):Boolean;
 begin
  if IsNaN(ms) then IsValidDateTime:=False else
  if IsInf(ms) then IsValidDateTime:=False else
  if ms<datetime2ms(1900,1,1,0,0,0,0) then IsValidDateTime:=False else
  if ms>datetime2ms(2100,1,1,0,0,0,0) then IsValidDateTime:=False else
  IsValidDateTime:=True;
 end;
 //
 // Date/time extract function
 //
 function extract_datetime(arg:String):Real;
 var r,v:Real; year,month,day,hour,min,sec,msec,p:Integer;
 begin
  r:=rVal(ExtractWord(1,arg));
  if not IsValidDateTime(r) then r:=_NaN;
  if IsNan(r) then begin
   arg:=Trim(arg);
   p:=Pos('.',arg);
   if p>1 then begin
    year:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos('.',arg);
    if (2000<=year) and (year<=3000) and (p>1) then begin
     month:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos('-',arg);
     if (1<=month) and (month<=12) and (p>1) then begin
      day:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos(':',arg);
       if (1<=day) and (day<=31) and (p>1) then begin
        hour:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1); p:=Pos(':',arg);
        if (0<=hour) and (hour<=60) and (p>1) then begin
         min:=Val(Copy(arg,1,p-1)); arg:=Copy(arg,p+1);
         if (0<=min) and (min<=60) and not IsEmptyStr(arg) then begin
          v:=rVal(arg);
          if not IsNan(v) then begin
           sec:=Trunc(v);
           msec:=Round(Frac(v)*1000);
           r:=datetime2ms(year,month,day,hour,min,sec,msec);
          end;
         end;
        end;
       end;
     end;
    end;
   end;
  end;
  extract_datetime:=r;
 end;
 //
 // Calculate Modbus request PDU by command id.
 // cid   - (in)  command id number.
 // fid   - (out) Modbus function id.
 // saddr - (out) zero-based start address of registers to read/write.
 // quant - (out) data quantity, i.e. number of r/w registers or single register r/w value.
 // dat   - (out) Modbus PDU data unit, PDU=(fid+dat).
 //
 function MERA_CalcPDU(cid:Integer; var fid,saddr,quant:Integer; var dat:String):Boolean;
 begin
  fid:=0; saddr:=0; quant:=0; dat:='';
  if IsEnabledCmdNum(cid) then begin
   fid:=MERA.Modbus.Cmd.FuncId[cid];
   saddr:=MERA.Modbus.Cmd.SAddr[cid];
   quant:=MERA.Modbus.Cmd.Quant[cid];
   if (cid=cm_ReadPress) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,'');
   end else
   if (cid=cm_ReadDTime) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,'');
   end else
   if (cid=cm_ReadVer) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,'');
   end else
   if (cid=cm_ReadAddr) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,'');
   end else
   if (cid=cm_WritAddr) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,modbus_dump_int16(round(GetCmdOpData(cid)),SwapModeInt32));
   end else
   if (cid=cm_WritDTime) then begin
    dat:=modbus_encode_pdu('R',fid,saddr,quant,encode_datetime(GetCmdOpData(cid)));
   end else
   Trouble(GotBug('Invalid command id '+Str(cid)));
  end;
  MERA_CalcPDU:=modbus_func_ok(fid) and (Length(dat)>0);
 end;
 //
 // Main Modbus command handler to process Modbus device reply.
 // cid   - (in)  command id number.
 // fid   - (in)  Modbus function id.
 // saddr - (in)  zero-based start address of registers to read/write.
 // quant - (in)  data quantity, i.e. number of r/w registers or single register r/w value.
 // raw   - (in)  raw coils/registers data array.
 //
 procedure MERA_OnCommand(cid,fid,saddr,quant:Integer; raw:String);
 var addr,offs,indic:Integer; r:Real;
 begin
  if IsValidCmdNum(cid) then begin
   if cid=cm_ReadPress then begin        // Read HR ReadPress Reg 00-01
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs;
     if (addr=0) and (quant-offs+1>=2) then begin
      r:=modbus_ext_float(raw,offs,SwapModeFloat);
      bNul(rSetTag(MERA.MEAS.PRESS.tag,r));
      UpdateAo(ao_MEASPRESS,time,r);
      Details('Pressure:'+Str(r));
     end;
    end;
   end else
   if cid=cm_ReadDTime then begin       // Read HR ReadDTime Reg 140-145
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs;
     if (addr=320) and (quant-offs+1>=6) then begin
      r:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.YEAR.tag,Round(r)));
      Details('Year:'+Str(r));
      r:=modbus_ext_int16(raw,offs+1,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.MONT.tag,Round(r)));
      Details('Month:'+Str(r));
      r:=modbus_ext_int16(raw,offs+2,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.DATE.tag,Round(r)));
      Details('Date:'+Str(r));
      r:=modbus_ext_int16(raw,offs+3,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.HOUR.tag,Round(r)));
      Details('Hours:'+Str(r));
      r:=modbus_ext_int16(raw,offs+4,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.MIN.tag,Round(r)));
      Details('Minutes:'+Str(r));
      r:=modbus_ext_int16(raw,offs+5,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.SEC.tag,Round(r)));
      Details('Seconds:'+Str(r));
      decode_datetime(iGetTag(MERA.PARAM.YEAR.tag)+2000,iGetTag(MERA.PARAM.MONT.tag),
                      iGetTag(MERA.PARAM.DATE.tag),iGetTag(MERA.PARAM.HOUR.tag),
                      iGetTag(MERA.PARAM.MIN.tag),iGetTag(MERA.PARAM.SEC.tag));
     end;
    end;
   end else
   if cid=cm_ReadVer then begin       // Read HR ReadVer Reg 131
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs;
     if (addr=305) and (quant-offs+1>=1) then begin
      r:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.VERSION.tag,Round(r)));
      UpdateAo(ao_VERSION,time,r);
      Details('Version:'+Str(r));
     end;
    end;
   end else
   if cid=cm_ReadAddr then begin       // Read HR ReadAddr Reg 210
    for offs:=0 to quant-1 do begin
     addr:=saddr+offs;
     if (addr=528) and (quant-offs+1>=1) then begin
      r:=modbus_ext_int16(raw,offs,SwapModeInt32);
      bNul(iSetTag(MERA.PARAM.ADDRESS.tag,round(r)));
      UpdateAo(ao_ADDRESS,time,r);
      Details('Address:'+Str(r));
     end;
    end;
   end else
   if (cid=cm_WritAddr)  then ReleaseCmdOpData(cid) else
   if (cid=cm_WritDTime) then ReleaseCmdOpData(cid) else
   Trouble(GotBug('Unexpected command '+Str(cid)));
   if MERA.Simulator then indic:=st_SIMULAT else
   indic:=st_NORMAL;
   bNul(iSetTag(MERA.STATE.INDIC.tag,indic));
  end;
 end;
 //
 // Data handler on @Modbus.Reply event. Process reply comes from Modbus device.
 //
 procedure MERA_OnReply(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant:Integer;
 begin
  raw:=''; // Check is coming reply corresponded to sent request
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>MERA.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>MERA.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>MERA.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>MERA.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>MERA.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=MERA.Modbus.Poll.saddr; quant:=MERA.Modbus.Poll.quant;
   if modbus_decode_pdu('A',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>MERA.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>MERA.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else
    if modbus_is_except(fid) then begin
     Problem(GotBug('MODBUS EXCEPTION '+Str(modbus_ext_byte(raw,0))+' ON COMMAND '+Str(cid)));
    end else begin
     IncSummRate(MERA.Modbus.Poll.Summ.Rx,MERA.Modbus.Poll.Rate.Rx);
     MERA_OnCommand(cid,fid,saddr,quant,raw);
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  raw:='';
 end;
 //
 // Data handler on @Modbus.Poll event.
 // This procedure calls in simulation mode only.
 //
 procedure MERA_OnSimPoll(ref,cid,tim,port,uid,fid:Integer; dat:String);
 var raw:String; saddr,quant,offs:Integer; r:Real;
 begin
  raw:=''; // Check is coming reply corresponded to sent request
  if not IsValidCmdNum(cid) then Trouble(GotBug('Bad reply command '+Str(cid))) else
  if (port<>MERA.Modbus.Poll.port) then Trouble(GotBug('Bad reply port '+Str(port))) else
  if (uid<>MERA.Modbus.Poll.uid) then Trouble(GotBug('Bad reply unit id '+Str(uid))) else
  if (cid<>MERA.Modbus.Poll.cid) then Trouble(GotBug('Bad reply command id '+Str(cid))) else
  if (ref<>MERA.Modbus.Poll.ref) then Trouble(GotBug('Bad reply device '+RefInfo(ref,'Name'))) else
  if (modbus_un_except(fid)<>MERA.Modbus.Poll.fid) then Trouble(GotBug('Bad reply fid '+Str(fid))) else begin
   saddr:=MERA.Modbus.Poll.saddr; quant:=MERA.Modbus.Poll.quant;
   if modbus_decode_pdu('R',fid,dat,saddr,quant,raw)>0 then begin
    if (saddr<>MERA.Modbus.Poll.saddr) then Trouble(GotBug('Bad reply saddr '+Str(saddr))) else
    if (quant<>MERA.Modbus.Poll.quant) then Trouble(GotBug('Bad reply quant '+Str(quant))) else begin
     if DebugFlagEnabled(dfDetails) then Details('Simulate polling '+Str(cid));
     if (cid=cm_ReadPress) then begin
      raw:=modbus_dump_float(MERA.SIM.MEASPRESS,SwapModeFloat);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevSendCmd(devMySelf,modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_ReadDTime) then begin
      // Date and time update
      r:=mSecNow;
      MERA.SIM.YEAR:=ms2year(r)-2000;MERA.SIM.MONT:=ms2month(r);MERA.SIM.DATE:=ms2day(r);
      MERA.SIM.HOUR:=ms2hour(r);MERA.SIM.MIN:=ms2min(r);MERA.SIM.SEC:=ms2sec(r);
      raw:=modbus_dump_int16(MERA.SIM.YEAR,SwapModeInt32)
          +modbus_dump_int16(MERA.SIM.MONT,SwapModeInt32)
          +modbus_dump_int16(MERA.SIM.DATE,SwapModeInt32)
          +modbus_dump_int16(MERA.SIM.HOUR,SwapModeInt32)
          +modbus_dump_int16(MERA.SIM.MIN,SwapModeInt32)
          +modbus_dump_int16(MERA.SIM.SEC,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevSendCmd(devMySelf,modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_ReadVer) then begin
      raw:=modbus_dump_int16(1,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevSendCmd(devMySelf,modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_ReadAddr) then begin
      raw:=modbus_dump_int16(MERA.SIM.ADDRESS,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,raw);
      DevSendCmd(devMySelf,modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritAddr) then begin
      MERA.SIM.ADDRESS:=modbus_ext_int16(raw,0,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmd(devMySelf,modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     if (cid=cm_WritDTime) then begin
      MERA.SIM.YEAR:=modbus_ext_int16(raw,offs+0,SwapModeInt32);
      MERA.SIM.MONT:=modbus_ext_int16(raw,offs+1,SwapModeInt32);
      MERA.SIM.DATE:=modbus_ext_int16(raw,offs+2,SwapModeInt32);
      MERA.SIM.HOUR:=modbus_ext_int16(raw,offs+3,SwapModeInt32);
      MERA.SIM.MIN:=modbus_ext_int16(raw,offs+4,SwapModeInt32);
      MERA.SIM.SEC:=modbus_ext_int16(raw,offs+5,SwapModeInt32);
      dat:=modbus_encode_pdu('A',fid,saddr,quant,'');
      DevSendCmd(devMySelf,modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,fid,dat));
     end else
     DevSendCmd(devMySelf,modbus_proxy_poll('@Modbus.Reply',devMySelf,cid,0,port,uid,
                modbus_as_except(fid,true),Dump(Chr(modbus_er_ILLFUN))));
    end;
   end else Trouble(GotBug('Bad PDU format '+modbus_errmsg(modbus_errno)));
  end;
  raw:='';
 end;
 //
 // MERA Driver Simulator mode polling.
 // This procedure calls in simulation mode only.
 //
 procedure MERA_SIM_POLL;
 begin
  if MERA.Simulator then begin
   if iGetTag(MERA.POLL.ENABLE.tag)<>0 then begin
    MERA.SIM.MEASPRESS:=10+3*sin(2*pi*time*60)+0.1*random(-1,1);
   end;
  end;
 end;
 //
 // MERA Driver GUI polling.
 //
 procedure MERA_GUI_POLL;
 var s:String; ClickCurve:Integer;
  //
  // Send to device (dev) command (cmd data) to set new tag value if one in range (min,max).
  //
  procedure DevSendCmdTag(dev,tag:Integer; cmd,data:String; min,max:Real);
  var v:Real;
  begin
   if IsRefTag(tag) then
   if IsRefDevice(dev) then
   if not IsEmptyStr(cmd) then
   if not IsEmptyStr(data) then
   if TypeTag(tag)<=2 then begin
    v:=rVal(data);
    if IsNaN(v) then Trouble(GotBug('Invalid tag edit')) else
    if (v<min) or (v>max) then Trouble(GotBug('Tag edit out of range')) else
    if TypeTag(tag)=1 then DevSendCmd(devMySelf,cmd+' '+Str(Round(v))) else
    if TypeTag(tag)=2 then DevSendCmd(devMySelf,cmd+' '+Trim(data));
   end else
   if TypeTag(tag)=3 then begin
    DevSendCmd(devMySelf,cmd+' '+Trim(data));
   end;
  end;
  //
  // Send command (cmd) to device (dev) on tag click.
  //
  procedure ClickTagDevSendCmd(tag,dev:Integer; cmd:String);
  begin
   if Length(cmd)>0 then
   if IsRefTag(tag) then
   if IsRefDevice(dev) then
   if ClickTag=tag then begin
    bNul(Voice(snd_Click));
    DevSendCmd(dev,cmd);
   end;
  end;
  //
  // Xor bit on click (local version).
  //
  procedure ClickTagXorLocal(tag,XorMask:Integer);
  var nv:Integer;
  begin
   if (ClickTag=tag) then begin
    bNul(iSetTagXor(tag,XorMask));
    bNul(Voice(snd_Click));
   end;
  end;
 begin
  s:='';
  //
  // Update state indicator
  //
  UpdateDo(do_STATE_INDIC,time,iGetTag(MERA.STATE.INDIC.tag));
  //
  // Handle commands...
  //
  if iGetTag(MERA.GUI.CMD.HELP.tag)<>0 then begin
   rNul(Eval('@Global @Async @Silent @Run /Hide cmd /c '+DaqFileRef(ReadIni('['+devname+'] HelpFile'),'.htm')));
   bNul(iSetTag(MERA.GUI.CMD.HELP.tag,0));
  end;
  if iGetTag(MERA.GUI.CMD.SAVEINI.tag)<>0 then begin
   bNul(iSetTag(MERA.GUI.CMD.SAVEINI.tag,0));
   DevSendCmd(DevMySelf,'@SaveIni');
  end;
  if iGetTag(MERA.GUI.CMD.LOADINI.tag)<>0 then begin
   bNul(iSetTag(MERA.GUI.CMD.LOADINI.tag,0));
   DevSendCmd(DevMySelf,'@LoadIni');
  end;
  if iGetTag(MERA.GUI.CMD.CONSOLE.tag)<>0 then begin
   bNul(iSetTag(MERA.GUI.CMD.CONSOLE.tag,0));
   bNul(WinShow(ParamStr('Console '+DevName)));
   bNul(WinSelect(ParamStr('Console '+DevName)));
  end;
  //
  // Handle sensor clicks...
  //
  if ClickButton=1 then begin
   //
   // Buttons...
   //
   ClickTagXorLocal(MERA.GUI.CMD.HELP.tag,1);
   ClickTagXorLocal(MERA.GUI.CMD.SAVEINI.tag,1);
   ClickTagXorLocal(MERA.GUI.CMD.LOADINI.tag,1);
   ClickTagXorLocal(MERA.GUI.CMD.CONSOLE.tag,1);
   ClickTagXorLocal(MERA.POLL.ENABLE.tag,1);
   //
   // Sensor clicks...
   //
   if IsSameText('@',Copy(ClickSensor,1,1)) then begin
    DevSendCmd(devMySelf,URL_Decode(ClickSensor));
    bNul(Voice(snd_Click));
   end;
   if IsSameText(ClickSensor,'MERA.PARAM') then begin     //Show parameters dialog
    if not IsEmptyStr(Copy(devname,2,11)+'PARAM') then begin
     Cron('@WinSelect '+Copy(devname,2,11)+'PARAM');
     bNul(Voice(snd_Wheel));
    end;
   end;
   if ClickTag=MERA.PARAM.DTIME.tag then begin
    bNul(sSetTag(MERA.PARAM.DTIME.tag,''));
   end;
   //
   // Send command to edit on Click.
   //
   ClickTagDevSendCmd(MERA.PARAM.ADDRESS.tag, devMySelf,'@Edit PARAM.ADDRESS');
   //
   // Tools menu
   //
   if ClickTag=MERA.GUI.CMD.TOOLS.tag then begin
    if EditState=0 then begin
     if Pos('?',Edit('(Команда "Инструменты" '+DevName+'... ')
               +Edit(' Что выбираете:')
               +Edit(' '+MenuToolsItem(0))
               +Edit(' '+MenuToolsItem(1))
               +Edit(' '+MenuToolsItem(2))
               +Edit(' '+MenuToolsItem(3))
               +Edit(' '+MenuToolsItem(4))
               +Edit(' '+MenuToolsItem(5))
               +Edit(' '+MenuToolsItem(6))
               +Edit(' '+MenuToolsItem(7))
               +Edit(')MenuList MENU_'+NameTag(MERA.GUI.CMD.TOOLS.tag)))>0
     then Warning('Error initializing MenuList!');
    end else Warning('Cannot edit right now!');
    MERA.GUI.CMD.TOOLS.val:=-1;
   end;
   //
   // Plot & Tab windows
   //
   ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
   if IsRefCurve(ClickCurve) then begin
    iNul(WinSelectByCurve(ClickCurve,ClickCurve));
    bNul(Voice(snd_Wheel));
   end;
  end;
  //
  // Handle right button clicks...
  //
  if ClickButton=2 then begin
   SensorHelp(Url_Decode(ClickParams('Hint')));
  end;
  //
  // Edit tags...
  //
  if EditState=ef_Done then begin
   //
   // If tag edit complete, send command to apply changes
   //
   if CheckEditTag(MERA.PARAM.ADDRESS.tag,s)
   then DevSendCmdTag(devMySelf,MERA.PARAM.ADDRESS.tag,'@SETADDR',s,0,255);
   if CheckEditTag(MERA.PARAM.DTIME.tag,s)
   then DevSendCmdTag(devMySelf,MERA.PARAM.DTIME.tag,'@PARAM.RTIME',s,0,_PlusInf);
   {
   Warning.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'Warning') then sNul(Edit(''));
   {
   Information.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'Information') then sNul(Edit(''));
   {
   TOOLS menu.
   }
   if IsSameText(ExtractWord(1,Edit('?ans 0')),'MENU_'+NameTag(MERA.GUI.CMD.TOOLS.tag)) then begin
    if Val(ExtractWord(2,Edit('?ans 0')))=1 then begin
     MERA.GUI.CMD.TOOLS.val:=Val(Edit('?ans 1'));
     if MERA.GUI.CMD.TOOLS.val>=0 then
     if (MERA.GUI.CMD.TOOLS.val>10) then begin
      DevSendCmd(devMySelf,'@MenuToolsConfirmation '+UpCaseStr(StrReplace(StrReplace(
                 MenuToolsItem(Round(MERA.GUI.CMD.TOOLS.val)),'  ',Dump(' '),3),'  ',Dump(' '),3)));
     end else begin
      MenuToolsCmnd(Round(MERA.GUI.CMD.TOOLS.val));
      MERA.GUI.CMD.TOOLS.val:=-1;
     end;
    end;
    sNul(Edit(''));
   end;
   {
   TOOLS menu (after confirmation).
   }
   if IsSameText(ExtractWord(1,Edit('?ans 0')),'YesNo_'+NameTag(MERA.GUI.CMD.TOOLS.tag)) then begin
    if MERA.GUI.CMD.TOOLS.val>=0 then
    if Val(ExtractWord(2,Edit('?ans 0')))=6 then MenuToolsCmnd(Round(MERA.GUI.CMD.TOOLS.val));
    MERA.GUI.CMD.TOOLS.val:=-1;
    sNul(Edit(''));
   end;
  end;
  if EditState=ef_Done then begin
   Problem('Unknown tag edition!');
   sNul(Edit(''));
  end;
  if iAnd(EditState,ef_ErrorFound)<>0 then begin
   Problem('Dialog error detected!');
   sNul(Edit(''));
  end;
  s:='';
 end;
 //
 // MERA Driver Command Cycle polling.
 //
 procedure MERA_CMD_POLL;
  //
  // Send @Modbus.Poll ... command to &ModbusProxy and save sent request data in Modbus.Poll record.
  //
  procedure PollModbusProxy(cid:Integer);
  var dev,ref,tot,port,uid,fid,saddr,quant:Integer; dat:String;
  begin
   dat:='';
   ApplyCmdOpData(cid);
   if MERA_CalcPDU(cid,fid,saddr,quant,dat) then begin
    dev:=devTheProxy; ref:=devMySelf; tot:=MERA.Modbus.Timeout; port:=MERA.Modbus.Port; uid:=MERA.Modbus.UnitId;
    if DevSend(dev,modbus_proxy_poll('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat)+CRLF)>0 then begin
     if DebugFlagEnabled(dfViewExp) then
     ViewExp('COM: '+modbus_proxy_nice('@Modbus.Poll',ref,cid,tot,port,uid,fid,dat,32));
     AssignModbusPoll(dev,cid,mSecNow,port,uid,fid,saddr,quant,dat);
     IncSummRate(MERA.Modbus.Poll.Summ.Tx,MERA.Modbus.Poll.Rate.Tx);
    end else begin
     Trouble(GotBug('Fail to send command '+Str(cid)));
     ClearModbusPoll;
    end;
   end else begin
    Trouble(GotBug('Fail to calc command '+Str(cid)));
    ClearModbusPoll;
   end;
   dat:='';
  end;
 begin
  if IsPortOpened then begin
   if IsValidCmdNum(MERA.Modbus.Poll.cid) then begin
    //
    // Request in progress, waiting @Modbus.Reply/@Modbus.Refuse/@Modbus.Timeout.
    // Handle Deadline if was no responce for too long time: repeat polling again.
    //
    if iGetTag(MERA.POLL.ENABLE.tag)<>0 then begin
     if (mSecNow>MERA.Modbus.Poll.tim+MERA.Modbus.Deadline) then begin
      bNul(sSetTag(MERA.PARAM.DTIME.tag,'DEADLINE DETECTED'));
      Trouble(GotBug('Deadline detected, repeat polling again...'));
      PollModbusProxy(MERA.Modbus.Poll.cid);
     end;
    end else if iGetTag(MERA.POLL.ENABLE.tag)=0 then bNul(iSetTag(MERA.STATE.INDIC.tag,st_DISABLE));
   end else begin
    //
    // If request is cleared, send new @Modbus.Poll request by timer.
    //
    if iGetTag(MERA.POLL.ENABLE.tag)<>0 then begin
     if (mSecNow>=MERA.Modbus.Poll.tim+MERA.Modbus.Polling) then begin
      MERA.Modbus.Cmd.Num:=NextEnabledCmdNum(MERA.Modbus.Cmd.Num);
      if IsEnabledCmdNum(MERA.Modbus.Cmd.Num)
      then PollModbusProxy(MERA.Modbus.Cmd.Num);
     end;
    end else if iGetTag(MERA.POLL.ENABLE.tag)=0 then bNul(iSetTag(MERA.STATE.INDIC.tag,st_DISABLE));
   end;
   //
   // Update Poll Rate.
   //
   if SysTimer_Pulse(1000)>0 then begin
    UpdateDo(do_POLLRATERX,time,MERA.Modbus.Poll.Rate.Rx);
    UpdateDo(do_POLLRATETX,time,MERA.Modbus.Poll.Rate.Tx);
    UpdateDo(do_POLLRATEEX,time,MERA.Modbus.Poll.Rate.Ex);
    UpdateDo(do_POLLSUMMRX,time,MERA.Modbus.Poll.Summ.Rx);
    UpdateDo(do_POLLSUMMTX,time,MERA.Modbus.Poll.Summ.Tx);
    UpdateDo(do_POLLSUMMEX,time,MERA.Modbus.Poll.Summ.Ex);
    UpdateDo(do_ERRORCOUNT,time,GetErrCount(-1));
    if DebugFlagEnabled(dfStatist) then
    Success('PollRate: Rx='+Str(MERA.Modbus.Poll.Rate.Rx)
                   +'  Tx='+Str(MERA.Modbus.Poll.Rate.Tx)
                   +'  Ex='+Str(MERA.Modbus.Poll.Rate.Ex));
    ClearModbusRate;
   end;
  end;
 end;
 //
 // MERA tags initialization.
 //
 procedure MERA_InitTags(Prefix:String);
 begin
  if not IsEmptyStr(Prefix) then begin
   InitTag(MERA.GUI.CMD.HELP.tag,    Prefix+'.CMD.HELP',     1);
   InitTag(MERA.GUI.CMD.SAVEINI.tag, Prefix+'.CMD.SAVEINI',  1);
   InitTag(MERA.GUI.CMD.LOADINI.tag, Prefix+'.CMD.LOADINI',  1);
   InitTag(MERA.GUI.CMD.CONSOLE.tag, Prefix+'.CMD.CONSOLE',  1);
   InitTag(MERA.GUI.CMD.TOOLS.tag,   Prefix+'.CMD.TOOLS',    1);
   InitTag(MERA.STATE.INDIC.tag,     Prefix+'.STATE.INDIC',  1);
   InitTag(MERA.POLL.ENABLE.tag,     Prefix+'.POLL.ENABLE',  1);
   InitTag(MERA.PARAM.ADDRESS.tag,   Prefix+'.PARAM.ADDRESS',1);
   InitTag(MERA.PARAM.VERSION.tag,   Prefix+'.PARAM.VERSION',1);
   InitTag(MERA.PARAM.DTIME.tag,     Prefix+'.PARAM.DTIME',  3);
   InitTag(MERA.PARAM.YEAR.tag,      Prefix+'.PARAM.YEAR',   1);
   InitTag(MERA.PARAM.MONT.tag,      Prefix+'.PARAM.MONT',   1);
   InitTag(MERA.PARAM.DATE.tag,      Prefix+'.PARAM.DATE',   1);
   InitTag(MERA.PARAM.HOUR.tag,      Prefix+'.PARAM.HOUR',   1);
   InitTag(MERA.PARAM.MIN.tag,       Prefix+'.PARAM.MIN',    1);
   InitTag(MERA.PARAM.SEC.tag,       Prefix+'.PARAM.SEC',    1);
   InitTag(MERA.MEAS.PRESS.tag,      Prefix+'.MEAS.PRESS',   2);
  end;
 end;
 //
 // MERA Driver cleanup.
 //
 procedure MERA_Clear;
 begin
  MERA.Simulator:=False;
  MERA.Modbus.Port:=0;
  MERA.Modbus.UnitId:=0;
  MERA.Modbus.Timeout:=0;
  MERA.Modbus.Polling:=0;
  MERA.Modbus.Deadline:=0;
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 //
 // MERA Driver initialization...
 //
 procedure MERA_Init;
 begin
  //
  // Read ini file variables
  //
  MERA_InitTags(ReadIni('tagPrefix'));
  MERA.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  Success('Simulator='+Str(Ord(MERA.Simulator)));
  MERA.Modbus.Port:=iValDef(ReadIni('ModbusPort'),1);
  Success('ModbusPort='+Str(MERA.Modbus.Port));
  MERA.Modbus.UnitId:=iValDef(ReadIni('ModbusUnitId'),1);
  Success('ModbusUnitId='+Str(MERA.Modbus.UnitId));
  MERA.Modbus.Polling:=iValDef(ReadIni('ModbusPolling'),1000);
  Success('ModbusPolling='+Str(MERA.Modbus.Polling));
  MERA.Modbus.Timeout:=iValDef(ReadIni('ModbusTimeout'),250);
  Success('ModbusTimeout='+Str(MERA.Modbus.Timeout));
  MERA.Modbus.Deadline:=iValDef(ReadIni('ModbusDeadline'),60000);
  Success('ModbusDeadline='+Str(MERA.Modbus.Deadline));
  MERA.Modbus.DelayOnStart:=iValDef(ReadIni('DelayOnStart'),1000);
  Success('DelayOnStart='+Str(MERA.Modbus.DelayOnStart));
  bNul(sSetTag(MERA.PARAM.DTIME.tag,'2018.03.29-00:00:00'));
  MERA.SIM.ADDRESS:=iValDef(ReadIni('ModbusUnitId'),1);
  MERA.SIM.MEASPRESS:=0;
  MERA.SIM.YEAR:=2018;
  MERA.SIM.MONT:=3;
  MERA.SIM.DATE:=29;
  MERA.SIM.HOUR:=0;
  MERA.SIM.MIN:=0;
  MERA.SIM.SEC:=0;
  //
  // Initialize Cmd command table & clear Poll record.
  //
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  InitCmdTable;
 end;
 //
 // MERA Driver Finalization.
 //
 procedure MERA_Free;
 begin
  ClearModbusPoll;
  ClearModbusSumm;
  ClearModbusRate;
  ClearCmdTable;
 end;
 //
 // MERA Driver Polling.
 //
 procedure MERA_Poll;
 begin
  MERA_GUI_POLL;
  if mSecNow-FixmSecNow>MERA.Modbus.DelayOnStart then
  if IsPortOpened then begin
   if MERA.Simulator
   then MERA_SIM_POLL;
   MERA_CMD_POLL;
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearNetLibrary;
  MERA_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  StdIn_SetScripts('@StartupScript','');
  StdIn_SetTimeouts(0,0,MaxInt,0);
  InitNetLibrary;
  MERA_Init;
  cmd_ClearModbusSumm       := RegisterStdInCmd('@ClearModbusSumm',       '');
  cmd_Edit                  := RegisterStdInCmd('@Edit',                  '');
  cmd_MenuToolsConfirmation := RegisterStdInCmd('@MenuToolsConfirmation', '');
  cmd_LoadIni               := RegisterStdInCmd('@LoadIni',               '');
  cmd_SaveIni               := RegisterStdInCmd('@SaveIni',               '');
  cmd_SetAddr               := RegisterStdInCmd('@SetAddr',               '');
  cmd_ParamRTime            := RegisterStdInCmd('@Param.RTime',           '');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  FreeNetLibrary;
  MERA_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  PollNetLibrary;
  MERA_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg,dat:String; cmdid:Integer; ref,cid,tim,port,uid,fid:Integer; ms,v:Real;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  dat:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @Modbus.Reply ref cid tim port uid fid $$dat
   }
   if (cmdid=cmd_NetModbusReply) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then MERA_OnReply(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Poll ref cid tim port uid fid $$dat
   This message uses in simulation mode only
   }
   if (cmdid=cmd_NetModbusPoll) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then MERA_OnSimPoll(ref,cid,tim,port,uid,fid,dat)
    else Trouble(GotBug('Bad '+cmd+' format'));
    Data:='';
   end else
   {
   @Modbus.Refuse ...
   }
   if (cmdid=cmd_NetModbusRefuse) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Trouble(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,0)))
    else Trouble(GotBug(cmd+' '+arg));
    bNul(sSetTag(MERA.PARAM.DTIME.tag,'CONNECTION REFUSED'));
    bNul(iSetTag(MERA.STATE.INDIC.tag,st_REFUSE));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @Modbus.Timeout ...
   }
   if (cmdid=cmd_NetModbusTimeout) then begin
    if modbus_proxy_reply(cmd,arg,ref,cid,tim,port,uid,fid,dat)
    then Problem(GotBug(modbus_proxy_nice(cmd,ref,cid,tim,port,uid,fid,dat,64)))
    else Trouble(GotBug(cmd+' '+arg));
    bNul(sSetTag(MERA.PARAM.DTIME.tag,'TIMEOUT DETECTED'));
    bNul(iSetTag(MERA.STATE.INDIC.tag,st_TIMEOUT));
    ClearModbusPoll;
    Data:='';
   end else
   {
   @ClearModbusSumm
   }
   if (cmdid=cmd_ClearModbusSumm) then begin
    ClearModbusSumm;
    Data:='';
   end else
   {
   @SETADDR 1
   }
   if (cmdid=cmd_SetAddr) then begin
    v:=rVal(ExtractWord(1,arg));
    if not IsNan(v) then HoldCmdOpData(cm_WritAddr,v);
    Data:='';
   end else
   {
   @PARAM.DTIME Now
   @PARAM.RTIME 2016.12.22-08:16:22.123
   }
   if IsSameText(cmd,'@PARAM.RTIME') or IsSameText(cmd,'@PARAM.DTIME') then begin
    if IsSameText(ExtractWord(1,arg),'Now') then ms:=mSecNow else
    ms:=extract_datetime(ExtractWord(1,arg));
    HoldCmdOpData(cm_WritDTIME,ms);
    Data:='';
   end else
   {
   @Edit PARAM.ADDRESS 1
   @Edit PARAM.DTIME Now
   }
   if (cmdid=cmd_Edit) then begin
    if IsSameText(ExtractWord(1,arg),'PARAM.ADDRESS')
    then StartEditTag(MERA.PARAM.ADDRESS.tag,'Задание сетевого адреса, 1-247');
    if IsSameText(ExtractWord(1,arg),'PARAM.DTIME') then begin
     if IsSameText(ExtractWord(2,arg),'Now')
     or not IsValidDateTime(extract_datetime(sGetTag(MERA.PARAM.DTIME.tag)))
     then begin
      bNul(sSetTag(MERA.PARAM.DTIME.tag,GetDateTime(mSecNow)));
      DevSendCmd(DevMySelf,'@PARAM.RTIME Now');
     end;
    end;
    Data:='';
   end else
   {
   @MenuToolsConfirmation
   }
   if (cmdid=cmd_MenuToolsConfirmation) then begin
    if MERA.GUI.CMD.TOOLS.val>=0 then
    if EditState=0 then begin
    if Pos('?',Edit('(Вы действительно хотите:')
              +Edit(' ')
              +Edit(' '+arg)
              +Edit(' ')
              +Edit(' Эта операция может вызвать проблемы!')
              +Edit(')YesNo YesNo_'+NameTag(MERA.GUI.CMD.TOOLS.tag)))>0
     then Problem('Could not initialize dialog!');
    end else Problem('Could not initialize dialog!');
    Data:='';
   end else
   {
   @LoadIni
   }
   if (cmdid=cmd_LoadIni) then begin
    iNul(CustomIniRW('R',arg,2*Ord(not IsEmptyStr(arg))));
    Data:='';
   end else
   {
   @SaveIni
   }
   if (cmdid=cmd_SaveIni) then begin
    iNul(CustomIniRW('W',arg,2*Ord(not IsEmptyStr(arg))));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
  dat:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
