#!/bin/bash

###########################################################
## Copyright (c) 2002-2025 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Configuration Generator for MERA devices.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

#################################
### MERA devices config generator
#################################

function call(){
 echo "$*" 1>&2;
 eval "$@";
};

function rem(){
 return 0;
};

######################
# initialize variables
######################
readonly hdwcfg="${scriptbase}_hdw.cfg"; # hardware   configuration
readonly simcfg="${scriptbase}_sim.cfg"; # simulator  configuration
declare -i numhdwcfg=0;                  # Flag to include hardware config
declare -i CTRL_TOP=0;                   # Top  of main control window
declare -i CTRL_LEFT=450;                # Left of main control window
declare -i CTRL_TOP_STEP=80;
declare FN="";                           # Facility name
declare IDEN="";                         # Device identifier name


#########################
# Generate MERA devices.
#########################
function MERADevices(){
 call DeleteFiles $hdwcfg $simcfg;
 rem  Device Facility Identitier Port UnitId Polling;
 call MERA   MERA     DRV01      1    1      10     ;
 call MERA   MERA     DRV02      1    2      10     ;
 call MERA   MERA     DRV03      1    3      10     ;
 cat $hdwcfg $simcfg;
};

function MERA(){
 call MERA_HDW "$@" >> $hdwcfg;
 call MERA_SIM "$@" >> $simcfg;
 let CTRL_TOP+=CTRL_TOP_STEP;
};

function MERA_HDW(){
 FN="$1";            shift;
 IDEN="$1";          shift;
 ModbusPort="$1";    shift;
 ModbusUnitId="$1";  shift;
 ModbusPolling="$1"; shift;
 echo ";********************************************"; 
 echo "; $FN Driver for vacuometer MERADAT VIT19IT1"; 
 echo ";********************************************"; 
 echo "[DeviceList]"; 
 echo "&$FN.$IDEN.CTRL = device software program"; 
 echo "[&$FN.$IDEN.CTRL]"; 
 echo "Comment        = MODBUS RTU DRIVER FOR VACUOMETER MERADAT VIT19IT1."; 
 echo "InquiryPeriod  = 0"; 
 echo "DevicePolling  = 10, tpHigher"; 
 echo "ProgramSource  = ../daqpas/mera_drv.pas"; 
 echo "DigitalFifo    = 1024"; 
 echo "AnalogFifo     = 1024"; 
 echo "DebugFlags     = 3"; 
 echo "OpenConsole    = 2"; 
 echo "Simulator      = 0"; 
 echo "HelpFile       = ../help/index.htm"; 
 echo "ModbusPort     = $ModbusPort"; 
 echo "ModbusUnitId   = $ModbusUnitId"; 
 echo "ModbusTimeout  = 1000"; 
 echo "ModbusDeadline = 60000"; 
 echo "ModbusPolling  = $ModbusPolling"; 
 echo "DelayOnStart   = 1000"; 
 echo "tagPrefix      = $FN.$IDEN"; 
 echo "AnalogOutputs  = 3"; 
 echo "Link AnalogOutput  0  with curve $FN.$IDEN.MEAS.PRESS    ; Pressure measurement"; 
 echo "Link AnalogOutput  1  with curve $FN.$IDEN.PARAM.VERSION ; Device version"; 
 echo "Link AnalogOutput  2  with curve $FN.$IDEN.PARAM.ADDRESS ; Device address"; 
 echo "DigitalOutputs = 8"; 
 echo "Link DigitalOutput 0 with curve $FN.$IDEN.STATE.INDIC history 1000"; 
 echo "Link DigitalOutput 1 with curve $FN.$IDEN.POLLRATE.RX history 1000"; 
 echo "Link DigitalOutput 2 with curve $FN.$IDEN.POLLRATE.TX history 1000"; 
 echo "Link DigitalOutput 3 with curve $FN.$IDEN.POLLRATE.EX history 1000"; 
 echo "Link DigitalOutput 4 with curve $FN.$IDEN.ERROR.COUNT history 1000"; 
 echo "Link DigitalOutput 5 with curve $FN.$IDEN.POLLSUMM.RX history 1000"; 
 echo "Link DigitalOutput 6 with curve $FN.$IDEN.POLLSUMM.TX history 1000"; 
 echo "Link DigitalOutput 7 with curve $FN.$IDEN.POLLSUMM.EX history 1000"; 
 echo ";------ Save custom parameters to INI file"; 
 echo "CustomIniTagList  = [$IDEN.CustomParameters.TagList]"; 
 echo "CustomIniFileRef  = ../../${FN,,}_data/${IDEN,,}_custom.ini"; 
 echo "CustomIniSection  = [$IDEN.CustomParameters]"; 
 echo "CustomIniBackups  = ../../${FN,,}_data/custom"; 
 echo "CustomIniAutoLoad = 1"; 
 echo "CustomIniAutoSave = 0"; 
 echo "[]"; 
 echo "[ConfigFileList]"; 
 echo "ConfigFile = ../../${FN,,}_data/${IDEN,,}_custom.ini"; 
 echo "[]"; 
 echo "";
 echo "[TagList]"; 
 echo "$FN.$IDEN.MEAS.PRESS    = Real    0  ; Measure pressure"; 
 echo "$FN.$IDEN.PARAM.ADDRESS = Integer 0  ; Current address"; 
 echo "$FN.$IDEN.PARAM.VERSION = Integer 0  ; Device version"; 
 echo "$FN.$IDEN.PARAM.DTIME   = String  ?  ; Date/time by device clock"; 
 echo "$FN.$IDEN.PARAM.YEAR    = Integer 0  ; Date - year"; 
 echo "$FN.$IDEN.PARAM.MONT    = Integer 0  ; Date - month"; 
 echo "$FN.$IDEN.PARAM.DATE    = Integer 0  ; Date - date"; 
 echo "$FN.$IDEN.PARAM.HOUR    = Integer 0  ; Time - hours"; 
 echo "$FN.$IDEN.PARAM.MIN     = Integer 0  ; Time - minutes"; 
 echo "$FN.$IDEN.PARAM.SEC     = Integer 0  ; Time - seconds"; 
 echo "$FN.$IDEN.POLL.ENABLE   = Integer 1  ; Enable poll cycle"; 
 echo "$FN.$IDEN.STATE.INDIC   = Integer 0  ; Status indicator tag"; 
 echo "$FN.$IDEN.CMD.HELP      = integer 0  ; Command to call Help"; 
 echo "$FN.$IDEN.CMD.SAVEINI   = integer 0  ; Command to save INI file"; 
 echo "$FN.$IDEN.CMD.LOADINI   = integer 0  ; Command to load INI file"; 
 echo "$FN.$IDEN.CMD.CONSOLE   = integer 0  ; Command to open CONSOLE window"; 
 echo "$FN.$IDEN.CMD.TOOLS     = integer 0  ; Command to call TOOLS menu"; 
 echo "[]"; 
 echo "";
 echo "[DataStorage]"; 
 echo "$FN.$IDEN.MEAS.PRESS    = Curve 0 10 Black  15 1  ; Measure pressure"; 
 echo "$FN.$IDEN.PARAM.VERSION = Curve 0 10 Black  15 1  ; Device version"; 
 echo "$FN.$IDEN.PARAM.ADDRESS = Curve 0 10 Black  15 1  ; Current address"; 
 echo "$FN.$IDEN.STATE.INDIC   = Curve 0 10 Black  15 1  ; Status indicator"; 
 echo "$FN.$IDEN.POLLRATE.RX   = Curve 0 10 Black  15 1  ; Poll rate, receiver    (per sec)"; 
 echo "$FN.$IDEN.POLLRATE.TX   = Curve 0 10 Black  15 1  ; Poll rate, transmitter (per sec)"; 
 echo "$FN.$IDEN.POLLRATE.EX   = Curve 0 10 Black  15 1  ; Poll rate, errors      (per sec)"; 
 echo "$FN.$IDEN.ERROR.COUNT   = Curve 0 10 Black  15 1  ; Error counter          (counter)"; 
 echo "$FN.$IDEN.POLLSUMM.RX   = Curve 0 10 Black  15 1  ; Poll summ, receiver    (counter)"; 
 echo "$FN.$IDEN.POLLSUMM.TX   = Curve 0 10 Black  15 1  ; Poll summ, transmitter (counter)"; 
 echo "$FN.$IDEN.POLLSUMM.EX   = Curve 0 10 Black  15 1  ; Poll summ, errors      (counter)"; 
 echo "[]"; 
 echo "";
 echo "[Windows]"; 
 echo "$FN.$IDEN.PARAM = Circuit_Window"; 
 echo "[$FN.$IDEN.PARAM]"; 
 echo "Circuit = ../circuits/mera_param_ctrl.crc"; 
 echo "Link sensor @EDIT+PARAM.DTIME+NOW                             with device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.POLL.ENABLE   with tag   $FN.$IDEN.POLL.ENABLE   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.POLLRATE.RX   with curve $FN.$IDEN.POLLRATE.RX   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.POLLRATE.TX   with curve $FN.$IDEN.POLLRATE.TX   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.POLLRATE.EX   with curve $FN.$IDEN.POLLRATE.EX   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.POLLSUMM.EX   with curve $FN.$IDEN.POLLSUMM.EX   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.PAR.SET.ADDR  with tag   $FN.$IDEN.PARAM.ADDRESS device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.PAR.ADDRESS   with curve $FN.$IDEN.PARAM.ADDRESS device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.PAR.VERSION   with curve $FN.$IDEN.PARAM.VERSION device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.PAR.DTIME     with tag   $FN.$IDEN.PARAM.DTIME   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.MEAS.PRESS    with curve $FN.$IDEN.MEAS.PRESS    device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.CMD.HELP      with tag   $FN.$IDEN.CMD.HELP      device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.CMD.SAVEINI   with tag   $FN.$IDEN.CMD.SAVEINI   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.CMD.LOADINI   with tag   $FN.$IDEN.CMD.LOADINI   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.CMD.CONSOLE   with tag   $FN.$IDEN.CMD.CONSOLE   device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.CMD.TOOLS     with tag   $FN.$IDEN.CMD.TOOLS     device &$FN.$IDEN.CTRL"; 
 echo "[]"; 
 echo "[&CronSrv.StartupScript]"; 
 echo "@cron.tab $FN.$IDEN.PARAM.HOME 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.HOME @WinShow $FN.$IDEN.PARAM"; 
 echo "@cron.job $FN.$IDEN.PARAM.HOME @WinDraw $FN.$IDEN.PARAM|Left=$CTRL_LEFT|Top=$CTRL_TOP|Width=710|Height=320|Options=+Top,+Left,-Width,-Height"; 
 echo "@cron.job $FN.$IDEN.PARAM.HOME @WinDraw $FN.$IDEN.PARAM|Options=-ViewScrollLeft,-ViewScrollRight,-ViewScrollUp,-ViewScrollDown"; 
 echo "@cron.job $FN.$IDEN.PARAM.HOME @WinDraw $FN.$IDEN.PARAM|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar"; 
 echo "@cron.tab $FN.$IDEN.PARAM.SHOW 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.SHOW @WinShow $FN.$IDEN.PARAM"; 
 echo "@cron.tab $FN.$IDEN.PARAM.HIDE 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.HIDE @WinHide $FN.$IDEN.PARAM"; 
 echo "@cron.tab $FN.$IDEN.PARAM.SELECT 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.SELECT @WinSelect $FN.$IDEN.PARAM"; 
 echo "@cron.job $FN.MAIN.CTRL.HOME @cron.run $FN.$IDEN.PARAM.HOME"; 
 echo "@cron.job $FN.MAIN.CTRL.HOME @cron.run $FN.$IDEN.PARAM.HIDE"; 
 echo "[]"; 
 echo "";
 echo "[Windows]"; 
 echo "$FN.$IDEN.CTRL = Circuit_Window"; 
 echo "[$FN.$IDEN.CTRL]"; 
 echo "Circuit = ../circuits/mera_ctrl.crc"; 
 echo "Link sensor $FN.PARAM       with tag   $FN.$IDEN.MEAS.PRESS  device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.POLL.RATE   with curve $FN.$IDEN.POLLRATE.TX device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.POLL.ENABLE with tag   $FN.$IDEN.POLL.ENABLE device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.PARAM.DTIME with tag   $FN.$IDEN.PARAM.DTIME device &$FN.$IDEN.CTRL"; 
 echo "Link sensor $FN.MEAS.PRESS  with curve $FN.$IDEN.MEAS.PRESS  device &$FN.$IDEN.CTRL"; 
 echo "[]"; 
 echo "[&CronSrv.StartupScript]"; 
 echo "@cron.tab $FN.$IDEN.CTRL.HOME 0 0 0"; 
 echo "@cron.job $FN.$IDEN.CTRL.HOME @WinShow $FN.$IDEN.CTRL"; 
 echo "@cron.job $FN.$IDEN.CTRL.HOME @WinDraw $FN.$IDEN.CTRL|Left=$CTRL_LEFT|Top=$CTRL_TOP|Width=475|Height=340|Options=+Top,+Left,-Width,-Height"; 
 echo "@cron.job $FN.$IDEN.CTRL.HOME @WinDraw $FN.$IDEN.CTRL|Options=-ViewScrollLeft,-ViewScrollRight,-ViewScrollUp,-ViewScrollDown"; 
 echo "@cron.job $FN.$IDEN.CTRL.HOME @WinDraw $FN.$IDEN.CTRL|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar"; 
 echo "@cron.tab $FN.$IDEN.CTRL.SHOW 0 0 0"; 
 echo "@cron.job $FN.$IDEN.CTRL.SHOW @WinShow $FN.$IDEN.CTRL"; 
 echo "@cron.tab $FN.$IDEN.CTRL.HIDE 0 0 0"; 
 echo "@cron.job $FN.$IDEN.CTRL.HIDE @WinHide $FN.$IDEN.CTRL"; 
 echo "@cron.tab $FN.$IDEN.CTRL.SELECT 0 0 0"; 
 echo "@cron.job $FN.$IDEN.CTRL.SELECT @WinSelect $FN.$IDEN.CTRL"; 
 echo "@cron.job $FN.MAIN.CTRL.HOME @cron.run $FN.$IDEN.CTRL.HOME"; 
 echo "@cron.job $FN.MAIN.CTRL.HOME @cron.run $FN.$IDEN.CTRL.HIDE"; 
 echo "[]"; 
 echo "";
 echo "[Windows]"; 
 echo "$FN.$IDEN.PARAM.PLOT = Curve_Window"; 
 echo "[$FN.$IDEN.PARAM.PLOT]"; 
 echo "AxisX = ^R{Часов}___^N^CВремя, 0, 0.25"; 
 echo "AxisY = ^CСостояние_каналов^N^L___{Статус}, 0, 1000"; 
 echo "CurveList = $FN.$IDEN.MEAS.PRESS"; 
 echo "CurveList = $FN.$IDEN.PARAM.VERSION"; 
 echo "CurveList = $FN.$IDEN.PARAM.ADDRESS"; 
 echo "CurveList = $FN.$IDEN.POLLRATE.RX"; 
 echo "CurveList = $FN.$IDEN.POLLRATE.TX"; 
 echo "CurveList = $FN.$IDEN.POLLRATE.EX"; 
 echo "CurveList = $FN.$IDEN.STATE.INDIC"; 
 echo "CurveList = $FN.$IDEN.ERROR.COUNT"; 
 echo "CurveList = $FN.$IDEN.POLLSUMM.RX"; 
 echo "CurveList = $FN.$IDEN.POLLSUMM.TX"; 
 echo "CurveList = $FN.$IDEN.POLLSUMM.EX"; 
 echo "[]"; 
 echo "[&CronSrv.StartupScript]"; 
 echo "@cron.tab $FN.$IDEN.PARAM.PLOT.HOME 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.PLOT.HOME @WinShow $FN.$IDEN.PARAM.PLOT"; 
 echo "@cron.job $FN.$IDEN.PARAM.PLOT.HOME @WinDraw $FN.$IDEN.PARAM.PLOT|Left=167|Top=120|Height=600|Width=1000|Options=+Top,+Left,-Width,-Height"; 
 echo "@cron.job $FN.$IDEN.PARAM.PLOT.HOME @WinDraw $FN.$IDEN.PARAM.PLOT|Options=-Min,-Max,+Close,+VScroll,-HScroll,+StatusBar"; 
 echo "@cron.tab $FN.$IDEN.PARAM.PLOT.SHOW 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.PLOT.SHOW @WinShow $FN.$IDEN.PARAM.PLOT"; 
 echo "@cron.tab $FN.$IDEN.PARAM.PLOT.HIDE 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.PLOT.HIDE @WinHide $FN.$IDEN.PARAM.PLOT"; 
 echo "@cron.tab $FN.$IDEN.PARAM.PLOT.SELECT 0 0 0"; 
 echo "@cron.job $FN.$IDEN.PARAM.PLOT.SELECT @WinSelect $FN.$IDEN.PARAM.PLOT"; 
 echo "@cron.job $FN.MAIN.CTRL.HOME @cron.run $FN.$IDEN.PARAM.PLOT.HOME"; 
 echo "@cron.job $FN.MAIN.CTRL.HOME @cron.run $FN.$IDEN.PARAM.PLOT.HIDE"; 
 echo "[]"; 
 echo "";
 echo "[$FN.MAIN.CTRL]"; 
 echo "Link sensor $FN.$IDEN.CTRL  with tag   $FN.$IDEN.STATE.INDIC device &$FN.MAIN.CTRL"; 
 echo "Link sensor $FN.$IDEN.VALUE with curve $FN.$IDEN.MEAS.PRESS  device &$FN.MAIN.CTRL"; 
 echo "[]"; 
 echo "";
 echo "[&DATSRV]"; 
 echo "CurveList = $FN.$IDEN.MEAS.PRESS"; 
 echo "CurveList = $FN.$IDEN.PARAM.VERSION"; 
 echo "CurveList = $FN.$IDEN.PARAM.ADDRESS"; 
 echo "CurveList = $FN.$IDEN.POLLRATE.RX"; 
 echo "CurveList = $FN.$IDEN.POLLRATE.TX"; 
 echo "CurveList = $FN.$IDEN.POLLRATE.EX"; 
 echo "CurveList = $FN.$IDEN.STATE.INDIC"; 
 echo "CurveList = $FN.$IDEN.ERROR.COUNT"; 
 echo "CurveList = $FN.$IDEN.POLLSUMM.RX"; 
 echo "CurveList = $FN.$IDEN.POLLSUMM.TX"; 
 echo "CurveList = $FN.$IDEN.POLLSUMM.EX"; 
 echo "[]"; 
 echo "";
 echo "[$IDEN.CustomParameters.TagList]"; 
 echo "TagList = $FN.$IDEN.POLL.ENABLE"; 
 echo "[]"; 
};

function MERA_SIM(){
 echo "[&$FN.$IDEN.CTRL]"; 
 echo "Simulator = 1"; 
 echo "[]"; 
 echo "";
 rem "Include hardware config if one not done yet";
 if [[ $numhdwcfg -eq 0 ]]; then let numhdwcfg++; else return; fi;
 echo "[ConfigFileList]"; 
 echo "ConfigFile = $hdwcfg"; 
 echo "[]"; 
 echo "";
};

##############
# delete files
##############
function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

################
# main function.
################
function main(){
 if pushd $scripthome >/dev/null 2>&1; then
  MERADevices;
 fi;
 popd >/dev/null 2>&1;
 local code=$?;
 if [ $code -eq 0 ]; then
  local msg="$(langstr ru "конфигурация успешно создана." en "configuration created.")";
  unix tooltip-notifier preset stdOk delay 7000 text "$scriptname: $msg";
 else
  local msg="$(langstr ru "ошибка создания конфигурации." en "failed on create configuration.")";
  fatal $code "$scriptname: $msg";
 fi;
 return $code;
};

main "$@";

##############
## END OF FILE
##############
