 {
 ***********************************************************************
 Daq Pascal application program PULSM.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @OpUst_I1 v    - Setpoint current 1 to new value v, ampere
| @OpUst_I2 v    - Setpoint current 2 to new value v, ampere
| @OpUst_U1 v    - Setpoint voltage 1 to new value v, volt
| @OpUst_U2 v    - Setpoint voltage 2 to new value v, volt
| @OpTime t      - Setting action time, second
| @OpCode n      - Execute Operation Code n, which may be of:
|                  0-nop, 1-Regulation I1/U1, 2-Ramping
| @Edit p        - Edit PULSM parameter p, which may be of:
|                  OpUst_I1,OpUst_I2,OpUst_U1,OpUst_U2,OpTime
| @LoadIni       - load params from INI file.
| @SaveIni       - save params to   INI file.
|********************************************************
[]
 }
program pulsm_drv;               { Pulsar Smart driver              }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 MaxPduLen  = 253;               { Maximum size of PDU block        }
 MaxCoilGet = 2000;              { Maximim Coil's per one read      }
 MaxInStGet = 2000;              { Maximim InSt's per one read      }
 MaxIRegGet = 125;               { Maximum IReg's per one read      }
 MaxHRegGet = 125;               { Maximum HReg's per one read      }
 MaxCoilPut = 1968;              { Maximim Coil's per one write     }
 MaxHRegPut = 123;               { Maximum HReg's per one write     }
 MaxRtuLen  = 256;               { Maximal length of RTU frame      }
 MaxAscLen  = 513;               { Maximum length of ASCII frame    }
 hFF00      = 65280;             { $FF00 magic value for coil write }
 mbp_LIST   = 'IP,RTU,ASCII';    { MODBUS protocol List             }
 mbp_ALIAS  = 'TCP,RTU,ASC';     { MODBUS port alias names          }
 mbp_IP     = 1;                 { MODBUS protocol is IP            }
 mbp_RTU    = 2;                 { MODBUS protocol is RTU           }
 mbp_ASCII  = 3;                 { MODBUS protocol is ASCII         }
 mbf_ReadCS = 1;                 { MB fun: Read Coil Statuses       }
 mbf_ReadIS = 2;                 { MB fun: Read Input Statuses      }
 mbf_ReadHR = 3;                 { MB fun: Read Holding Registers   }
 mbf_ReadIR = 4;                 { MB fun: Read Input Registers     }
 mbf_WritSC = 5;                 { MB fun: Write Single Coil        }
 mbf_WritSR = 6;                 { MB fun: Write Single Register    }
 mbf_ReadES = 7;                 { MB fun: Read Exception Status    }
 mbf_Diagn  = 8;                 { MB fun: Diagnostics              }
 mbf_EvCnt  = 11;                { MB fun: Get Comm Event Counter   }
 mbf_EvLog  = 12;                { MB fun: Get Comm Event Log       }
 mbf_WritMC = 15;                { MB fun: Write Multitle Coils     }
 mbf_WritMR = 16;                { MB fun: Write Multiple Registers }
 mbf_RepSID = 17;                { MB fun: Report Slave ID          }
 mbf_ReadFR = 20;                { MB fun: Read File Record         }
 mbf_WritFR = 21;                { MB fun: Write File Record        }
 mbf_MaskWR = 22;                { MB fun: Mask Write Register      }
 mbf_ReWrMR = 23;                { MB fun: Read-Write Multiple Regs }
 mbf_ReadFQ = 24;                { MB fun: Read FIFO Queue          }
 mbf_Escape = 43;                { MB fun: Encapsul.Interf.Transp.  }
 mbe_OK     = 0;                 { MB err: Ok, no errors            }
 mbe_ILLFUN = 1;                 { MB err: ILLEGAL FUNCTION         }
 mbe_ILLADR = 2;                 { MB err: ILLEGAL DATA ADDRESS     }
 mbe_ILLVAL = 3;                 { MB err: ILLEGAL DATA VALUE       }
 mbe_EXCEPT = 128;               { MB exception code                }
 mbe_BADLEN = 16;                { MB err: BAD LENGTH               }
 mbe_BADCRC = 17;                { MB err: BAD CRC                  }
 mbs_SwapFL = 7;                 { MB swap float   1+2=bytes+words  }
 mbs_SwapIN = 7;                 { MB swap integer 1+2=bytes+words  }
 mbt_LIST   = 'INT16,INT32,FLOAT,DOUBLE';    { Data types           }
 mbt_LIST2  = ',BIT0,BIT1,BIT2,BIT3,BIT4,BIT5,BIT6,BIT7,BIT8,BIT9';
 mbt_LIST3  = ',BIT10,BIT11,BIT12,BIT13,BIT14,BIT15,BIT16,BIT17';
 mbt_LIST4  = ',BIT18,BIT19,BIT20,BIT21,BIT22,BIT23,BIT24,BIT25';
 mbt_LIST5  = ',BIT26,BIT27,BIT28,BIT29,BIT30,BIT31';
 mbt_Int16  = 1;                 { 16-bit integer                   }
 mbt_Int32  = 2;                 { 32-bit long integer              }
 mbt_Float  = 3;                 { 32-bit float                     }
 mbt_Double = 4;                 { 64-bit double float              }
 mbt_Bit0   = 5;                 { Status bit 0                     }
 SwapMode           = 1;         { Swap bytes/words/dwords...       }
 st_NoReq           = 0;         { No request sent                  }
 st_WaitAns         = 1;         { Waiting answer                   }
 st_WaitGap         = 2;         { Waiting time gap                 }
 st_TimeOut         = 3;         { Timeout found                    }
 cm_ReadAll         = 1;         { Command to read all params       }
 cm_OpUst_I1        = 2;         { Command to write OpUst_I1        }
 cm_OpUst_I2        = 3;         { Command to write OpUst_I2        }
 cm_OpUst_U1        = 4;         { Command to write OpUst_U1        }
 cm_OpUst_U2        = 5;         { Command to write OpUst_U2        }
 cm_OpTime          = 6;         { Command to write OpTime          }
 cm_OpCode          = 7;         { Command to write OpCode          }
 MaxCmdNum          = 7;         { Max command id number            }
 ar_OpCode          = 0;         { 0:Stop,1:Set-I,U,2:Ramp,3:I,4:U  }
 ar_OpUst_I1        = 1;         { Current-1 preset, A*10           }
 ar_OpUst_I2        = 3;         { Current-2 preset, A*10           }
 ar_OpUst_U1        = 5;         { Voltage-1 preset, V*100          }
 ar_OpUst_U2        = 6;         { Voltage-2 preset, V*100          }
 ar_OpPolar         = 7;         { Polarity 0:Plus,1:Minus          }
 ar_OpTime          = 8;         { Ramping time, sec                }
 ar_Q               = 12;        { A*H counter                      }
 ar_CurUst_I        = 17;        { A*10  Current I preset           }
 ar_CurUst_U        = 19;        { V*100 Current U preset           }
 ar_Measure_I       = 20;        { A*10  Measured current           }
 ar_Measure_U       = 22;        { V*100 Measured voltage           }
 ar_Flags           = 28;        { Status flags                     }
 ar_GTime           = 30;        { Working time, sec                }
 ar_OpUst_I1_float  = 32;        { Preset current-1, A              }
 ar_OpUst_I2_float  = 34;        { Preset current-2, A              }
 ar_OpUst_U1_float  = 36;        { Preset voltage-1, V              }
 ar_OpUst_U2_float  = 38;        { Preset voltage-2, V              }
 ar_Measure_I_float = 40;        { Measured current, A              }
 ar_Measure_U_float = 42;        { Measured voltage, V              }
 ar_NUM_VIRT_VIPR   = 49;        { Number of virtual power supply   }
 ar_QPlus           = 91;        { A*H global counter + polarity    }
 ar_QMinus          = 95;        { A*H clobal counter - polarity    }
 ao_POLLRATE        = 0;         { Analog outputs...                }
 ao_ERRORCNT        = 1;         {                                  }
 ao_OpUst_I1        = 2;         {                                  }
 ao_OpUst_I2        = 3;         {                                  }
 ao_OpUst_U1        = 4;         {                                  }
 ao_OpUst_U2        = 5;         {                                  }
 ao_Measure_I       = 6;         {                                  }
 ao_Measure_U       = 7;         {                                  }
 ao_OpCode          = 8;         {                                  }
 ao_OpPolar         = 9;         {                                  }
 ao_Q               = 10;        {                                  }
 ao_Flags           = 11;        {                                  }
 ao_GTime           = 12;        {                                  }
 ao_NUM_VIRT_VIPR   = 13;        {                                  }
 ao_OpTime          = 14;        {                                  }
 coef_I             = 10;        {                                  }
 coef_U             = 100;       {                                  }

type
 TArray2Char = array[1..2] of Char; { Dump of int16,word            }
 TArray4Char = array[1..4] of Char; { Dump of int32,dword,float     }
 TArray8Char = array[1..8] of Char; { Dump of int64,qword,double    }
 TTagRef     = record tag,nai,ndi,nao,ndo:Integer; val:Real; end;
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 PULSM             : record      { Pulsar Smart driver data         }
  Simulator        : Boolean;    { Simulator mode                   }
  Address          : Integer;    { Modbus device address            }
  ecBadCrc         : Integer;    { Error code: bad CRC checksumm    }
  ecIllFun         : Integer;    { Error code: illegal function     }
  ecExcept         : Integer;    { Error code: MODBUS exception     }
  ecTimeOut        : Integer;    { Error code: Timeout              }
  Com              : record      { COM port data                    }
   Port            : Integer;    {  Port number                     }
   TimeOut         : Integer;    {  Timeout, ms                     }
   TimeGap         : Integer;    {  Time delay after poll, ms       }
   Status          : Integer;    {  Status flag                     }
   Req,Ans         : String;     {  Request, Answer                 }
   ReqTime,AnsTime : Real;       {  Request, Answer time, ms        }
   PollPeriod      : Integer;    {  Modbus poll period, ms          }
   LastPoll        : Real;       {  Time of last modbus poll, ms    }
   PollRate        : Integer;    {  Poll rate, poll/sec             }
  end;                           {                                  }
  Cmd              : record      { Command cycle data               }
   Num             : Integer;    {  Current command number          }
   Enabled         : array [1..MaxCmdNum] of Boolean; {             }
   OpData          : array [1..MaxCmdNum] of Real;    {             }
   OpBuff          : array [1..MaxCmdNum] of Real;    {             }
  end;                           {                                  }
  Par              : record      { Parameters                       }
   OpUst_I1        : TTagRef;    {                                  }
   OpUst_I2        : TTagRef;    {                                  }
   OpUst_U1        : TTagRef;    {                                  }
   OpUst_U2        : TTagRef;    {                                  }
   OpTime          : TTagRef;    {                                  }
  end;                           {                                  }
  Gui              : record      { Gui tags                         }
   Cmd             : record      {  Gui Commands                    }
    HELP           : TTagRef;    {                                  }
    SAVEINI        : TTagRef;    {                                  }
    LOADINI        : TTagRef;    {                                  }
    Power          : TTagRef;    {                                  }
    RAMP           : TTagRef;    {   Enable Ramping                 }
   end;                          {                                  }
   Rad             : record      {  Radio Buttons                   }
    Ust_I          : TTagRef;    {                                  }
    Ust_U          : TTagRef;    {                                  }
   end;                          {                                  }
   StepSize        : TTagRef;    {                                  }
   RampTime        : Real;       {                                  }
   RampTm          : Real;       {                                  }
   RampStat        : Integer;    {                                  }
   OpCodeFl        : Integer;    {  OpCodeFlag                      }
   val_I1          : Real;
   val_I2          : Real;
   val_U1          : Real;
   val_U2          : Real;
  end;                           {                                  }
  Sim              : record      { Simulation                       }
   OpCode          : Integer;    {                                  }
   OpTime          : Integer;    {                                  }
   OpUst_I1        : Real;       {                                  }
   OpUst_I2        : Real;       {                                  }
   OpUst_U1        : Real;       {                                  }
   OpUst_U2        : Real;       {                                  }
  end;
 end;                            {                                  }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 
 {
 Get integer or real tag value or default.
 }
 function modbus_get_tag(tag:Integer;def:Real):Real;
 begin
  if TypeTag(tag)=1 then modbus_get_tag:=iGetTag(tag) else
  if TypeTag(tag)=2 then modbus_get_tag:=rGetTag(tag) else
  modbus_get_tag:=def;
 end;
 {
 Set integer or real tag value.
 }
 function modbus_set_tag(tag:Integer; val:Real):Boolean;
 begin
  if TypeTag(tag)=1 then modbus_set_tag:=iSetTag(tag,Round(val)) else
  if TypeTag(tag)=2 then modbus_set_tag:=rSetTag(tag,val) else
  modbus_set_tag:=False;
 end;
 {
 Increment integer or real tag value. Uses for error counting.
 }
 function modbus_inc_tag(tag:Integer; inc:Real):Boolean;
 begin
  if TypeTag(tag)=1 then modbus_inc_tag:=iSetTag(tag,iGetTag(tag)+Round(inc)) else
  if TypeTag(tag)=2 then modbus_inc_tag:=rSetTag(tag,rGetTag(tag)+inc) else
  modbus_inc_tag:=False;
 end;
 {
 Handle modbus error: increment tag & code & print message.
 }
 procedure modbus_error(tag,code:Integer; msg:String);
 begin
  if not modbus_inc_tag(tag,1) then
  if code<>0 then bNul(FixError(code));
  if Length(msg)>0 then Problem(msg);
 end;
 {
 Calculate MODBUS/ASCII LRC checksum.
 }
 function modbus_calc_lrc(Buffer:String):Integer;
 var data,i:Integer;
 begin
  data:=0;
  for i:=1 to Length(Buffer) do data:=data+Ord(Buffer[i]);
  modbus_calc_lrc:=Ord(Chr(256-Ord(Chr(data))));
 end;
 {
 Calculate MODBUS/RTU CRC checksum.
 }
 function modbus_calc_crc(Buffer:String):Integer;
 const hFFFF=65535; hA001=40961;
 var data,i,j:Integer;
 begin
  data:=hFFFF;
  for i:=1 to Length(Buffer) do begin
   data:=iXor(data,Ord(Buffer[i]));
   for j:=1 to 8 do if Odd(data)
   then data:=iXor(data div 2,hA001)
   else data:=data div 2;
  end;
  modbus_calc_crc:=data;
 end;
 {
 Swap two bytes in 16-bit word
 }
 function modbus_swap_int16(w:Integer):Integer;
 begin
  modbus_swap_int16:=Ord(Chr(w))*256+Ord(Chr(w div 256));
 end;
 {
 Swap bytes, words, dwords (correspond to SwapMode bit 0,1,2).
 }
 procedure modbus_swap2(var s,t:TArray2Char; SwapMode:Integer);
 begin
  case SwapMode mod 2 of
   0: begin t[1]:=s[1]; t[2]:=s[2]; end;
   1: begin t[1]:=s[2]; t[2]:=s[1]; end;
  end;
 end;
 procedure modbus_swap4(var s,t:TArray4Char; SwapMode:Integer);
 begin
  case SwapMode mod 4 of
   0: begin t[1]:=s[1]; t[2]:=s[2]; t[3]:=s[3]; t[4]:=s[4]; end;
   1: begin t[1]:=s[2]; t[2]:=s[1]; t[3]:=s[4]; t[4]:=s[3]; end;
   2: begin t[1]:=s[3]; t[2]:=s[4]; t[3]:=s[1]; t[4]:=s[2]; end;
   3: begin t[1]:=s[4]; t[2]:=s[3]; t[3]:=s[2]; t[4]:=s[1]; end;
  end;
 end;
 procedure modbus_swap8(var s,t:TArray8Char; SwapMode:Integer);
 begin
  case SwapMode mod 8 of
   0: begin t[1]:=s[1]; t[2]:=s[2]; t[3]:=s[3]; t[4]:=s[4]; t[5]:=s[5]; t[6]:=s[6]; t[7]:=s[7]; t[8]:=s[8]; end;
   1: begin t[1]:=s[2]; t[2]:=s[1]; t[3]:=s[4]; t[4]:=s[3]; t[5]:=s[6]; t[6]:=s[5]; t[7]:=s[8]; t[8]:=s[7]; end;
   2: begin t[1]:=s[3]; t[2]:=s[4]; t[3]:=s[1]; t[4]:=s[2]; t[5]:=s[7]; t[6]:=s[8]; t[7]:=s[5]; t[8]:=s[6]; end;
   3: begin t[1]:=s[4]; t[2]:=s[3]; t[3]:=s[2]; t[4]:=s[1]; t[5]:=s[8]; t[6]:=s[7]; t[7]:=s[6]; t[8]:=s[5]; end;
   4: begin t[1]:=s[5]; t[2]:=s[6]; t[3]:=s[7]; t[4]:=s[8]; t[5]:=s[1]; t[6]:=s[2]; t[7]:=s[3]; t[8]:=s[4]; end;
   5: begin t[1]:=s[6]; t[2]:=s[5]; t[3]:=s[8]; t[4]:=s[7]; t[5]:=s[2]; t[6]:=s[1]; t[7]:=s[4]; t[8]:=s[3]; end;
   6: begin t[1]:=s[7]; t[2]:=s[8]; t[3]:=s[5]; t[4]:=s[6]; t[5]:=s[3]; t[6]:=s[4]; t[7]:=s[1]; t[8]:=s[2]; end;
   7: begin t[1]:=s[8]; t[2]:=s[7]; t[3]:=s[6]; t[4]:=s[5]; t[5]:=s[4]; t[6]:=s[3]; t[7]:=s[2]; t[8]:=s[1]; end;
  end;
 end;
 {
 Get data dump as int16, with byte reordering.
 }
 function modbus_dump_int16(data:Integer; SwapMode:Integer):String;
 var s,t:TArray2Char;
 begin
  s:=Dump(data); modbus_swap2(s,t,SwapMode); modbus_dump_int16:=t;
 end;
 {
 Get data dump as int32, with byte reordering.
 }
 function modbus_dump_int32(data:Integer; SwapMode:Integer):String;
 var s,t:TArray4Char;
 begin
  s:=Dump(data); modbus_swap4(s,t,SwapMode); modbus_dump_int32:=t;
 end;
 {
 Get data dump as float, with byte reordering.
 }
 function modbus_dump_float(data:Real; SwapMode:Integer):String;
 var s,t:TArray4Char;
 begin
  s:=DumpF(data); modbus_swap4(s,t,SwapMode); modbus_dump_float:=t;
 end;
 {
 Get data dump as double, with byte reordering.
 }
 function modbus_dump_double(data:Real; SwapMode:Integer):String;
 var s,t:TArray8Char;
 begin
  s:=Dump(data); modbus_swap8(s,t,SwapMode); modbus_dump_double:=t;
 end;
 {
 Get data dump as int16/int32/float/double, with byte reordering.
 }
 function modbus_dump_typ(typ:Integer; data:Real):String;
 begin
  if typ=mbt_int16 then modbus_dump_typ:=modbus_dump_int16(Round(data),mbs_SwapIN) else
  if typ=mbt_int32 then modbus_dump_typ:=modbus_dump_int32(Round(data),mbs_SwapIN) else
  if typ=mbt_float then modbus_dump_typ:=modbus_dump_float(data,mbs_SwapFL) else
  if typ=mbt_double then modbus_dump_typ:=modbus_dump_double(data,mbs_SwapFL) else
  modbus_dump_typ:='';
 end;
 {
 Get data size of type int16/int32/float/double, in bytes.
 }
 function modbus_typ_size(typ:Integer):Integer;
 begin
  if typ=mbt_int16 then modbus_typ_size:=2 else
  if typ=mbt_int32 then modbus_typ_size:=4 else
  if typ=mbt_float then modbus_typ_size:=4 else
  if typ=mbt_double then modbus_typ_size:=8 else
  modbus_typ_size:=0;
 end;
 {
 Take data from dump as int16, with byte reordering.
 }
 function modbus_take_int16(data:String; SwapMode:Integer):Integer;
 var s,t:TArray2Char;
 begin
  s:=data; modbus_swap2(s,t,SwapMode); data:=t; modbus_take_int16:=Dump2i(data);
 end;
 {
 Take data from dump as int32, with byte reordering.
 }
 function modbus_take_int32(data:String; SwapMode:Integer):Integer;
 var s,t:TArray4Char;
 begin
  s:=data; modbus_swap4(s,t,SwapMode); data:=t; modbus_take_int32:=Dump2i(data);
 end;
 {
 Take data from dump as float, with byte reordering.
 }
 function modbus_take_float(data:String; SwapMode:Integer):Real;
 var s,t:TArray4Char;
 begin
  s:=data; modbus_swap4(s,t,SwapMode); data:=t; modbus_take_float:=Dump2f(data);
 end;
 {
 Take data dump as double, with byte reordering.
 }
 function modbus_take_double(data:String; SwapMode:Integer):Real;
 var s,t:TArray8Char;
 begin
  s:=data; modbus_swap8(s,t,SwapMode); data:=t; modbus_take_double:=Dump2r(data);
 end;
 {
 Take data from dump as int16/int32/float/double, with byte reordering.
 }
 function modbus_take_typ(typ:Integer; data:String):Real;
 begin
  if typ=mbt_int16 then modbus_take_typ:=modbus_take_int16(data,mbs_SwapIN) else
  if typ=mbt_int32 then modbus_take_typ:=modbus_take_int32(data,mbs_SwapIN) else
  if typ=mbt_float then modbus_take_typ:=modbus_take_float(data,mbs_SwapFL) else
  if typ=mbt_double then modbus_take_typ:=modbus_take_double(data,mbs_SwapFL) else
  modbus_take_typ:=0;
 end;
 {
 Encode MODBUS RTU message: UnitId+FuncId+Data+CRC.
 }
 function modbus_encode_rtu(UnitId,FuncId:Integer; Data:String):String;
 var ans:String;
 begin
  ans:=Chr(UnitId)+(Chr(FuncId)+Data);
  ans:=ans+modbus_dump_int16(modbus_calc_crc(ans),0);
  modbus_encode_rtu:=ans;
  ans:='';
 end;
 {
 Xor bit on click (local version)
 }
 procedure ClickBitXorLocal(tag,XorMask:Integer);
 var nv:Integer;
 begin
  if ClickTag=tag then begin
   bNul(iSetTagXor(tag,XorMask));
   bNul(Voice(snd_Click));
  end;
 end;
 {
 Apply bit XOR mask to tag
 }
 procedure iBitXorTag(tag,XorMask:Integer);
 begin
  bNul(iSetTagXor(tag,XorMask));
 end;
 {
 Return Data with bit number n assigned according to value SetOn.
 }
 function iSetBitNum(Data,BitNum:Integer; SetOn:Boolean):Integer;
 begin
  if SetOn
  then iSetBitNum:=iOr(Data,iShift(1,BitNum))
  else iSetBitNum:=iAnd(Data,iNot(iShift(1,BitNum)));
 end;
 {
 Set bit in integer tag
 }
 procedure iSetTagBitNum(tag,BitNum:Integer; SetOn:Boolean);
 begin
  if TypeTag(tag)=1 then
  bNul(iSetTag(tag,iSetBitNum(iGetTag(tag),BitNum,SetOn)));
 end;
 {
 Check if COM port opened.
 }
 function ComOpened:Boolean;
 begin
  ComOpened:=(ComSpace>=0);
 end;
 {
 Main command cycle.
 }
 procedure PULSM_CMD_CYCLE;
 const nDataRegs=100;
  function ValidateCmdNum(Num:Integer):Integer;
  begin
   if (Num>=1) and (Num<=MaxCmdNum)
   then ValidateCmdNum:=Num
   else ValidateCmdNum:=1;
  end;
  function NextCmdNum(Num:Integer):Integer;
  var i:Integer;
  begin
   i:=0;
   while (i<MaxCmdNum) do begin
    Num:=ValidateCmdNum(Num+1);
    if PULSM.Cmd.Enabled[Num]
    then i:=MaxCmdNum
    else i:=i+1;
   end;
   NextCmdNum:=Num;
  end;
  procedure PrepareRequest(n:Integer);
   function Encode_ReadHR(addr,nreg:Integer):String;
   begin
    Encode_ReadHR:=modbus_encode_rtu(PULSM.Address,mbf_ReadHR,
     modbus_dump_int16(addr,1)+modbus_dump_int16(nreg,1));
   end;
   function Encode_WritMR(addr:Integer;data:String):String;
   var nreg:Integer;
   begin
    nreg:=Length(data) div 2;
    Encode_WritMR:=modbus_encode_rtu(PULSM.Address,mbf_WritMR,
     modbus_dump_int16(addr,1)+modbus_dump_int16(nreg,1)+Chr(2*nreg)+data);
   end;
  begin
   if n=cm_ReadAll then begin
    PULSM.Com.Req:=Encode_ReadHR(ar_OpCode,nDataRegs);
   end else
   if n=cm_OpUst_I1 then begin
    PULSM.Com.Req:=Encode_WritMR(ar_OpUst_I1,modbus_dump_int32(Round(coef_I*PULSM.Cmd.OpData[n]),SwapMode));
   end else
   if n=cm_OpUst_I2 then begin
    PULSM.Com.Req:=Encode_WritMR(ar_OpUst_I2,modbus_dump_int32(Round(coef_I*PULSM.Cmd.OpData[n]),SwapMode));
   end else
   if n=cm_OpUst_U1 then begin
    PULSM.Com.Req:=Encode_WritMR(ar_OpUst_U1,modbus_dump_int16(Round(coef_U*PULSM.Cmd.OpData[n]),SwapMode));
   end else
   if n=cm_OpUst_U2 then begin
    PULSM.Com.Req:=Encode_WritMR(ar_OpUst_U2,modbus_dump_int16(Round(coef_U*PULSM.Cmd.OpData[n]),SwapMode));
   end else
   if n=cm_OpTime then begin
    PULSM.Com.Req:=Encode_WritMR(ar_OpTime,modbus_dump_int16(Round(PULSM.Cmd.OpData[n]),SwapMode));
   end else
   if n=cm_OpCode then begin
    PULSM.Com.Req:=Encode_WritMR(ar_OpCode,modbus_dump_int16(Round(PULSM.Cmd.OpData[n]),SwapMode));
   end;
  end;
  procedure FailureBadCRC;
  begin
   Failure(PULSM.ecBadCrc,'MODBUS CRC error found');
  end;
  procedure FailureILLFUN;
  begin
   Failure(PULSM.ecIllFun,'FUN error found');
  end;
  procedure DecodeParams(data:String);
  var OpUst_I1,OpUst_I2,OpUst_U1,OpUst_U2,Measure_I,Measure_U,GTime,Q:Real; OpTime:Integer;
   OpCode,OpPolar,Flags:Integer;
   function GetDataFloat(data:String; addr:Integer):Real;
   begin
    GetDataFloat:=modbus_take_float(Copy(data,1+addr*2,4),SwapMode);
   end;
   function GetDataInt16(data:String; addr:Integer):Integer;
   begin
    GetDataInt16:=modbus_take_int16(Copy(data,1+addr*2,2),SwapMode);
   end;
   function GetDataInt32(data:String; addr:Integer):Integer;
   begin
    GetDataInt32:=modbus_take_int32(Copy(data,1+addr*2,4),SwapMode);
   end;
   function GetDataUInt32(data:String; addr:Integer):Real;
   var v:Real;
   begin
    v:=GetDataInt16(data,addr+1)+v; v:=v*65536.0;
    v:=GetDataInt16(data,addr+0)+v;
    GetDataUInt32:=v;
   end;
   function GetDataUInt64(data:String; addr:Integer):Real;
   var v:Real;
   begin
    v:=GetDataInt16(data,addr+3);   v:=v*65536.0;
    v:=GetDataInt16(data,addr+2)+v; v:=v*65536.0;
    v:=GetDataInt16(data,addr+1)+v; v:=v*65536.0;
    v:=GetDataInt16(data,addr+0)+v;
    GetDataUInt64:=v;
   end;
  begin
   if (Length(data)=2*nDataRegs) then begin
    OpCode:=GetDataInt16(data,ar_OpCode);
    OpPolar:=GetDataInt16(data,ar_OpPolar);
    OpTime:=GetDataInt16(data,ar_OpTime);
    OpUst_I1:=GetDataFloat(data,ar_OpUst_I1_float);
    OpUst_I2:=GetDataFloat(data,ar_OpUst_I2_float);
    OpUst_U1:=GetDataFloat(data,ar_OpUst_U1_float);
    OpUst_U2:=GetDataFloat(data,ar_OpUst_U2_float);
    Measure_I:=GetDataFloat(data,ar_Measure_I_float);
    Measure_U:=GetDataFloat(data,ar_Measure_U_float);
    Flags:=GetDataInt16(data,ar_Flags);
    Q:=GetDataUInt64(data,ar_Q);
    GTime:=GetDataUInt32(data,ar_GTime);
    bNul(rSetTag(PULSM.Par.OpUst_I1.tag,OpUst_I1));
    bNul(rSetTag(PULSM.Par.OpUst_I2.tag,OpUst_I2));
    bNul(rSetTag(PULSM.Par.OpUst_U1.tag,OpUst_U1));
    bNul(rSetTag(PULSM.Par.OpUst_U2.tag,OpUst_U2));
    bNul(iSetTag(PULSM.Par.OpTime.tag,OpTime));
    UpdateAo(ao_OpUst_I1,      time, OpUst_I1);
    UpdateAo(ao_OpUst_I2,      time, OpUst_I2);
    UpdateAo(ao_OpUst_U1,      time, OpUst_U1);
    UpdateAo(ao_OpUst_U2,      time, OpUst_U2);
    UpdateAo(ao_Measure_I,     time, Measure_I);
    UpdateAo(ao_Measure_U,     time, Measure_U);
    UpdateAo(ao_OpCode,        time, OpCode);
    UpdateAo(ao_OpPolar,       time, OpPolar);
    UpdateAo(ao_OpTime,        time, OpTime);
    UpdateAo(ao_Q,             time, Q);
    UpdateAo(ao_Flags,         time, Flags);
    UpdateAo(ao_GTime,         time, GTime);
    UpdateAo(ao_NUM_VIRT_VIPR, time, GetDataInt16(data,ar_NUM_VIRT_VIPR));
    iSetTagBitNum(PULSM.Gui.Cmd.Power.tag,1,(OpCode<>0));
   end;
   if PULSM.Gui.OpCodeFl=0 then
   if OpCode=1 then bNul(iSetTag(PULSM.Gui.Cmd.RAMP.tag,0));
  end;
  function HandleRequest(n:Integer):Boolean;
  const MinAnsLen=5;
  var unitid,funcid,excode,crc16r,crc16c:Integer; Handled:Boolean;
  begin
   Handled:=false;
   if Length(PULSM.Com.Ans)>=MinAnsLen then begin
    unitid:=Dump2i(Copy(PULSM.Com.Ans,1,1));
    funcid:=Dump2i(Copy(PULSM.Com.Ans,2,1));
    excode:=Dump2i(Copy(PULSM.Com.Ans,3,1));
    crc16r:=modbus_take_int16(Copy(PULSM.Com.Ans,Length(PULSM.Com.Ans)-1),0);
    crc16c:=modbus_calc_crc(Copy(PULSM.Com.Ans,1,Length(PULSM.Com.Ans)-2));
    if (unitid=PULSM.Address) then begin
     if iAnd(funcid,mbe_EXCEPT)<>0 then begin
      if crc16r<>crc16c
      then FailureBadCrc
      else Failure(PULSM.ecExcept,'MODBUS Exception: $'+HexB(funcid)+' $'+HexB(excode));
      Handled:=true;
     end else begin
      if n=cm_ReadAll then begin
       if funcid<>mbf_ReadHR then begin
        FailureILLFUN;
        Handled:=true;
       end else
       if Length(PULSM.Com.Ans)>=MinAnsLen+excode then begin
        if crc16r<>crc16c
        then FailureBadCrc
        else DecodeParams(Copy(PULSM.Com.Ans,4,Length(PULSM.Com.Ans)-5));
        Handled:=true;
       end;
      end else
      if (n=cm_OpUst_I1)
      or (n=cm_OpUst_I2)
      or (n=cm_OpUst_U1)
      or (n=cm_OpUst_U2)
      or (n=cm_OpTime)
      or (n=cm_OpCode) then begin
       if funcid<>mbf_WritMR then begin
        FailureILLFUN;
        Handled:=true;
       end else
       if Length(PULSM.Com.Ans)>=8 then begin
        if crc16r<>crc16c
        then FailureBadCrc
        else PULSM.Cmd.Enabled[n]:=False;
        Handled:=true;
       end;
      end;
     end;
    end;
   end;
   HandleRequest:=Handled;
  end;
  procedure UpdateOpData;
  var n:Integer;
  begin
   for n:=1 to MaxCmdNum do
   if not IsNan(PULSM.Cmd.OpBuff[n]) then begin
    PULSM.Cmd.OpData[n]:=PULSM.Cmd.OpBuff[n];
    PULSM.Cmd.OpBuff[n]:=_NAN;
   end;
  end;
 begin
  PULSM.Cmd.Num:=ValidateCmdNum(PULSM.Cmd.Num);
  if PULSM.Cmd.Enabled[PULSM.Cmd.Num] then begin
   if PULSM.Com.Status=st_NoReq then begin
    UpdateOpData;
    PULSM.Com.Req:='';
    PULSM.Com.Ans:='';
    PULSM.Com.ReqTime:=0;
    PULSM.Com.AnsTime:=0;
    if mSecNow>=PULSM.Com.LastPoll+PULSM.Com.PollPeriod
    then PrepareRequest(PULSM.Cmd.Num);
    if Length(PULSM.Com.Req)=0 then PULSM.Cmd.Num:=NextCmdNum(PULSM.Cmd.Num) else begin
     PurgeComPort;
     if ComWrite(PULSM.Com.Req) then begin
      ViewExp('COM > '+Hex_Encode(PULSM.Com.Req));
      PULSM.Com.Status:=st_WaitAns;
      PULSM.Com.ReqTime:=mSecNow;
     end else begin
      Trouble('Could not send request '+Hex_Encode(PULSM.Com.Req));
      PULSM.Com.Req:='';
      PULSM.Cmd.Num:=NextCmdNum(PULSM.Cmd.Num);
     end;
     PULSM.Com.LastPoll:=mSecNow;
    end;
   end;
   if PULSM.Com.Status=st_WaitAns then begin
    PULSM.Com.Ans:=PULSM.Com.Ans+ComRead(255);
    if HandleRequest(PULSM.Cmd.Num) then begin
     ViewImp('COM < '+Hex_Encode(PULSM.Com.Ans));
     PULSM.Com.PollRate:=PULSM.Com.PollRate+1;
     PULSM.Com.Status:=st_WaitGap;
     PULSM.Com.AnsTime:=mSecNow;
    end else
    if mSecNow>PULSM.Com.ReqTime+PULSM.Com.Timeout then begin
     PULSM.Com.Status:=st_TimeOut;
    end;
   end;
   if PULSM.Com.Status=st_WaitGap then begin
    if mSecNow>=PULSM.Com.AnsTime+PULSM.Com.TimeGap then begin
     PULSM.Com.Req:='';
     PULSM.Com.Ans:='';
     PULSM.Com.ReqTime:=0;
     PULSM.Com.AnsTime:=0;
     PULSM.Com.Status:=st_NoReq;
     PULSM.Cmd.Num:=NextCmdNum(PULSM.Cmd.Num);
    end;
   end;
   if PULSM.Com.Status=st_TimeOut then begin
    Failure(PULSM.ecTimeout,'Timeout on request '+Hex_Encode(PULSM.Com.Req));
    PULSM.Com.Req:='';
    PULSM.Com.Ans:='';
    PULSM.Com.ReqTime:=0;
    PULSM.Com.AnsTime:=0;
    PULSM.Com.Status:=st_NoReq;
    PULSM.Cmd.Num:=NextCmdNum(PULSM.Cmd.Num);
   end;
  end else PULSM.Cmd.Num:=NextCmdNum(PULSM.Cmd.Num);
 end;
 {
 Handle data requests in simulation mode.
 }
 procedure PULSM_Sim(msg:String);
 var unitid,funcid:Integer; ans:String;
  {
  Send data to COM port.
  }
  procedure ComSend(data:String);
  begin
   if Length(data)>0 then begin
    if ComWrite(data) then begin
     if iAnd(DebugFlags,dfViewExp)<>0 then
     ViewExp('COM > '+Hex_Encode(Data));
    end else begin
     Trouble('COM port write failed.');
    end;
   end;
  end;
  {
  Data string composition.
  }
  function StrCompos(unitid,funcid:Integer):String;
  var n:Integer; mes_U,mes_I:Real; ans:String;
   //
   // Simulation measure values of current/voltage.
   //
   function Sim_Measure(var data:Real):Real;
   begin
    Sim_Measure:=data+random(0,0.5);
   end;
   //
   // Flags simulation.
   //
   function Sim_Flags(fid:Integer):Integer;
   begin
    if PULSM.Sim.OpCode<>0 then begin
     if (mes_I=0) and (mes_U=0) then Sim_Flags:=2 else Sim_Flags:=3;
    end else Sim_Flags:=1;
   end;
  begin
   ans:='';
   if PULSM.Sim.OpCode<>0 then begin
    mes_I:=Sim_Measure(PULSM.Sim.OpUst_I1);
    mes_U:=Sim_Measure(PULSM.Sim.OpUst_U1);
   end else begin
    mes_I:=0;
    mes_U:=0;
   end;
   ans:=Chr(200)+modbus_dump_int16(PULSM.Sim.OpCode,1);                              // OpCode
   ans:=ans+modbus_dump_int32(Round(coef_I*PULSM.Sim.OpUst_I1),1);                   // OpUst_I1(1-2 addr)
   ans:=ans+modbus_dump_int32(Round(coef_I*PULSM.Sim.OpUst_I2),1);                   // OpUst_I2(3-4 addr)
   ans:=ans+modbus_dump_int16(Round(coef_U*PULSM.Sim.OpUst_U1),1);                   // OpUst_U1(5 addr)
   ans:=ans+modbus_dump_int16(Round(coef_U*PULSM.Sim.OpUst_U2),1)+CharStr(2,Chr(0)); // OpUst_U2(6 addr)+4 Chars
   ans:=ans+modbus_dump_int16(PULSM.Sim.OpTime,1)+CharStr(16,Chr(0));                // OpTime(8 addr)+32 Chars
   ans:=ans+modbus_dump_int32(Round(coef_I*PULSM.Sim.OpUst_I1),1);                   // CurUst_I(17-18 addr)
   ans:=ans+modbus_dump_int16(Round(coef_U*PULSM.Sim.OpUst_U1),1);                   // CurUst_U(19 addr)
   ans:=ans+modbus_dump_int32(Round(coef_I*mes_I),1);                                // Measure_I(20 addr)
   ans:=ans+modbus_dump_int16(Round(coef_U*mes_U),1)+CharStr(10,Chr(0));             // Measure_U(22 addr)
   ans:=ans+modbus_dump_int16(Sim_Flags(funcid),1)+CharStr(2,Chr(0));                // Flags(28addr)
   ans:=ans+modbus_dump_int32(Round(runcount),1);                                    // GTime
   ans:=ans+modbus_dump_float(PULSM.Sim.OpUst_I1,1);                                 // OpUst_I1_float(32-33 addr)
   ans:=ans+modbus_dump_float(PULSM.Sim.OpUst_I2,1);                                 // OpUst_I2_float(34-35 addr)
   ans:=ans+modbus_dump_float(PULSM.Sim.OpUst_U1,1);                                 // OpUst_U1_float(36-37 addr)
   ans:=ans+modbus_dump_float(PULSM.Sim.OpUst_U2,1);                                 // OpUst_U2_float(38-39 addr)
   ans:=ans+modbus_dump_float(mes_I,1);                                              // Measure_I_float(40-41 addr)
   ans:=ans+modbus_dump_float(mes_U,1);                                              // Measure_U_float(42-43 addr)
   n:=200-(length(ans)-1);
   ans:=ans+CharStr(n,Chr(0));
   ans:=modbus_encode_rtu(unitid,funcid,ans);
   StrCompos:=ans;
   ans:='';
  end;
  {
  Decode RTU frame for Pulsar Smart.
  }
  function PULSM_Ans_Prepare(msg:String; var unitid,funcid:Integer):String;
  var n:Integer;
  begin
   unitid:=0; funcid:=0;
   if Length(msg)>=4 then begin
    unitid:=Dump2i(Copy(msg,1,1));
    funcid:=Dump2i(Copy(msg,2,1));
    if funcid=mbf_ReadHR then begin
     PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
    end else
    if funcid=mbf_WritMR then begin
     n:=modbus_take_int16(Copy(msg,3,4),1);
     if n=ar_OpCode then begin
      if modbus_take_int16(Copy(msg,8,4),1)<>0 then begin
       PULSM.Sim.OpCode:=modbus_take_int16(Copy(msg,8,4),1);
      end else begin
       PULSM.Sim.OpCode:=modbus_take_int16(Copy(msg,8,4),1);
      end;
      PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
     end else
     if n=2 then begin
      PULSM.Sim.OpCode:=modbus_take_int16(Copy(msg,8,4),1);
      PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
     end else
     if n=ar_OpUst_I1 then begin
      PULSM.Sim.OpUst_I1:=modbus_take_int32(Copy(msg,8,4),1)/10;
      PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
     end else
     if n=ar_OpUst_U1 then begin
      PULSM.Sim.OpUst_U1:=modbus_take_int16(Copy(msg,8,4),1)/100;
      PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
     end;
     if n=ar_OpUst_I2 then begin
      PULSM.Sim.OpUst_I2:=modbus_take_int32(Copy(msg,8,4),1)/10;
      PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
     end else
     if n=ar_OpUst_U2 then begin
      PULSM.Sim.OpUst_U2:=modbus_take_int16(Copy(msg,8,4),1)/100;
      PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
     end;
     if n=ar_OpTime then begin
      PULSM.Sim.OpTime:=Round(modbus_take_int16(Copy(msg,8,4),1));
      PULSM_Ans_Prepare:=StrCompos(unitid,funcid);
     end;
    end;
   end;
  end;
 begin
  ans:='';
  if Length(msg)>0 then begin
   ViewImp('COM < '+Hex_Encode(msg));
   ans:=PULSM_Ans_Prepare(msg,unitid,funcid);
   ComSend(ans);
  end;
  ans:='';
 end;
 {
 Handle GUI commands, edit tags etc.
 }
 procedure PULSM_GUI_POLL;
 var s:String; ClickCurve,Pow,n,i:Integer; val_I1,val_U1,val_I,val_U:Real;
  procedure DevSendCmdTag(tag:Integer; cmd,data:String; min,max:Real);
  var v:Real;
  begin
   if not IsEmptyStr(cmd) then
   if not IsEmptyStr(data) then
   if TypeTag(tag)=1 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevSendCmd(devMySelf,cmd+' '+Str(Round(v)));
   end else
   if TypeTag(tag)=2 then begin
    v:=rVal(data);
    if isNan(v) then Problem('Invalid tag edit') else
    if (v<min) or (v>max) then Problem('Tag edit out of range') else
    DevSendCmd(devMySelf,cmd+' '+Str(v));
   end else
   if TypeTag(tag)=3 then begin
    DevSendCmd(devMySelf,cmd+' '+Trim(data));
   end;
  end;
  {
  Start ramping
  }
  procedure Ramping_Start;
  begin
   PULSM.Gui.RampTime:=msecnow;
   PULSM.Gui.RampTm:=msecnow;
   PULSM.Gui.RampStat:=1;
   PULSM.Gui.val_I1:=rgettag(PULSM.Par.OpUst_I1.tag);
   PULSM.Gui.val_I2:=rgettag(PULSM.Par.OpUst_I2.tag);
   PULSM.Gui.val_U1:=rgettag(PULSM.Par.OpUst_U1.tag);
   PULSM.Gui.val_U2:=rgettag(PULSM.Par.OpUst_U2.tag);
  end;
  {
  Setting current and voltage on ramping mode
  }
  procedure Ramp_Set(var val1,val2:Real);
  var t1,t2,dv,dt,k,y,Time:Real;
  begin
   Time:=igettag(PULSM.Par.OpTime.tag);
   t1:=PULSM.Gui.RampTime;
   t2:=t1+(Time*1000);
   dv:=val2-val1;
   dt:=t2-mSecNow;
   k:=dv/(t1-t2);
   y:=k*dt+val2;
   if mSecNow<=t2 then begin
    if PULSM.Gui.RampStat=1 then
    if msecnow>=PULSM.Gui.RampTm+200 then begin
     PULSM.Gui.RampTm:=msecnow;
     DevSendCmd(devMySelf,'@OpUst_I1 '+Str(y));
     PULSM.Gui.RampStat:=2;
    end;
    if PULSM.Gui.RampStat=2 then
    if msecnow>=PULSM.Gui.RampTm+200 then begin
     PULSM.Gui.RampTm:=msecnow;
     DevSendCmd(devMySelf,'@OpUst_U1 '+Str(y));
     PULSM.Gui.RampStat:=1;
    end;
   end else begin
    DevSendCmd(devMySelf,'@OpUst_I1 '+Str(rgettag(PULSM.Par.OpUst_I2.tag)));
    DevSendCmd(devMySelf,'@OpUst_U1 '+Str(rgettag(PULSM.Par.OpUst_U2.tag)));
    bNul(rsettag(PULSM.Par.OpUst_I1.tag,rgettag(PULSM.Par.OpUst_I2.tag)));
    bNul(rsettag(PULSM.Par.OpUst_U1.tag,rgettag(PULSM.Par.OpUst_U2.tag)));
    if PULSM.Gui.RampStat<>0 then DevSendCmd(devMySelf,'@OpCode 1');
    PULSM.Gui.OpCodeFl:=0;
    PULSM.Gui.RampStat:=0;
   end;
  end;
 begin
  s:='';
  val_I1:=rgettag(PULSM.Par.OpUst_I1.tag);
  val_U1:=rgettag(PULSM.Par.OpUst_U1.tag);
  if (igettag(PULSM.Gui.Cmd.Power.tag)<>0) and (iGetTag(PULSM.GUI.CMD.RAMP.tag)<>0) then begin
   if PULSM.Gui.RampStat=1 then Ramp_Set(PULSM.Gui.val_I1,PULSM.Gui.val_I2);
   if PULSM.Gui.RampStat=2 then Ramp_Set(PULSM.Gui.val_U1,PULSM.Gui.val_U2);
  end;
  //
  // Handle commands...
  //
  if iGetTag(PULSM.GUI.CMD.HELP.tag)<>0 then begin
   DevPostCmdLocal('@BrowseHelp '+DaqFileRef(AdaptFileName(ReadIni('HelpFile')),'.htm'));
   bNul(iSetTag(PULSM.GUI.CMD.HELP.tag,0));
  end;
  if iGetTag(PULSM.GUI.CMD.SAVEINI.tag)<>0 then begin
   bNul(iSetTag(PULSM.GUI.CMD.SAVEINI.tag,0));
   DevSendCmd(DevMySelf,'@SaveIni');
  end;
  if iGetTag(PULSM.GUI.CMD.LOADINI.tag)<>0 then begin
   bNul(iSetTag(PULSM.GUI.CMD.LOADINI.tag,0));
   DevSendCmd(DevMySelf,'@LoadIni');
  end;
  Pow:=iGetTag(PULSM.Gui.Cmd.Power.tag);
  if PULSM.Gui.Cmd.Power.val<>Ord(IsBit(Pow,0)) then begin
   if iGetTag(PULSM.GUI.CMD.RAMP.tag)<>0 then begin
    if Ord(IsBit(Pow,0))<>0 then n:=2 else n:=0;
    DevSendCmd(devMySelf,'@OpCode '+str(n));
    PULSM.Gui.Cmd.Power.val:=Ord(IsBit(Pow,0));
    if n<>0 then PULSM.Gui.OpCodeFl:=1 else PULSM.Gui.OpCodeFl:=0;
   end else begin
    DevSendCmd(devMySelf,'@OpCode '+Str(Ord(IsBit(Pow,0))));
    PULSM.Gui.Cmd.Power.val:=Ord(IsBit(Pow,0));
    PULSM.Gui.OpCodeFl:=0;
   end;
  end;
  if iGetTag(PULSM.Par.OpTime.tag)=0 then bNul(iSetTag(PULSM.GUI.CMD.RAMP.tag,0));
  //
  // Handle sensor clicks...
  //
  if ClickButton=1 then begin
   if ClickTag=PULSM.Par.OpUst_I1.tag then begin
    DevSendCmd(devMySelf,'@Edit OpUst_I1');
    bNul(Voice(snd_Click));
   end;
   if ClickTag=PULSM.Par.OpUst_U1.tag then begin
    DevSendCmd(devMySelf,'@Edit OpUst_U1');
    bNul(Voice(snd_Click));
   end;
   if ClickTag=PULSM.Par.OpUst_I2.tag then begin
    DevSendCmd(devMySelf,'@Edit OpUst_I2');
    bNul(Voice(snd_Click));
   end;
   if ClickTag=PULSM.Par.OpUst_U2.tag then begin
    DevSendCmd(devMySelf,'@Edit OpUst_U2');
    bNul(Voice(snd_Click));
   end;
   if ClickTag=PULSM.Gui.Cmd.Power.tag then begin
    iBitXorTag(PULSM.Gui.Cmd.Power.tag,1);
    Ramping_Start;
    bNul(Voice(snd_Click));
   end;
   if ClickTag=PULSM.Par.OpTime.tag then begin
    DevSendCmd(devMySelf,'@Edit OpTime');
    bNul(Voice(snd_Click));
   end;
   if ClickTag=PULSM.GUI.CMD.RAMP.tag then begin
    if iGetTag(PULSM.Par.OpTime.tag)<>0 then begin
     Ramping_Start;
     if (PULSM.Gui.val_I1<>PULSM.Gui.val_I2) or (PULSM.Gui.val_U1<>PULSM.Gui.val_U2) then begin
      if iGetTag(PULSM.Gui.Cmd.Power.tag)<>0 then DevSendCmd(devMySelf,'@OpCode 2');
      PULSM.Gui.OpCodeFl:=1;
      iBitXorTag(PULSM.GUI.CMD.RAMP.tag,1);
     end;
    end;
    bNul(Voice(snd_Click));
   end;
   ClickBitXorLocal(PULSM.GUI.CMD.HELP.tag,1);
   ClickBitXorLocal(PULSM.GUI.CMD.SAVEINI.tag,1);
   ClickBitXorLocal(PULSM.GUI.CMD.LOADINI.tag,1);
   if ClickTag=PULSM.GUI.Rad.UST_I.tag then
   if iGetTag(PULSM.GUI.Rad.UST_I.tag)=0 then begin
    bNul(iSetTag(PULSM.GUI.Rad.UST_I.tag,1));
    bNul(iSetTag(PULSM.GUI.Rad.UST_U.tag,0));
    bNul(iSetTag(PULSM.GUI.StepSize.tag,0));
   end;
   if ClickTag=PULSM.GUI.Rad.UST_U.tag then
   if iGetTag(PULSM.GUI.Rad.UST_U.tag)=0 then begin
    bNul(iSetTag(PULSM.GUI.Rad.UST_U.tag,1));
    bNul(iSetTag(PULSM.GUI.Rad.UST_I.tag,0));
    bNul(iSetTag(PULSM.GUI.StepSize.tag,1));
   end;
   if not(iGetTag(PULSM.Gui.Cmd.Power.tag)<>0) or not(iGetTag(PULSM.GUI.CMD.RAMP.tag)<>0) then begin
    if iGetTag(PULSM.GUI.Rad.UST_I.tag)=1 then begin
     for i:=1 to 3 do begin
      if IsSameText(ClickSensor, 'PULSM.UST.UP'+Str(i)) then begin
       if i=1 then val_I:=0.1;
       if i=2 then val_I:=1;
       if i=3 then val_I:=5;
       val_I:=val_I1+val_I;
       if val_I>0 then bNul(rSetTag(PULSM.Par.OpUst_I1.tag, val_I));
       DevSendCmdTag(PULSM.Par.OpUst_I1.tag,'@OpUst_I1',Str(val_I),0,200);
      end;
      if IsSameText(ClickSensor, 'PULSM.UST.DN'+Str(i)) then begin
       if i=1 then val_I:=0.1;
       if i=2 then val_I:=1;
       if i=3 then val_I:=5;
       val_I:=val_I1-val_I;
       if val_I>0 then bNul(rSetTag(PULSM.Par.OpUst_I1.tag, val_I));
       DevSendCmdTag(PULSM.Par.OpUst_I1.tag,'@OpUst_I1',Str(val_I),0,200);
      end;
     end;
    end;
    if iGetTag(PULSM.GUI.Rad.UST_U.tag)=1 then begin
     for i:=1 to 3 do begin
      if IsSameText(ClickSensor, 'PULSM.UST.UP'+Str(i)) then begin
       if i=1 then val_U:=0.05;
       if i=2 then val_U:=1;
       if i=3 then val_U:=5;
       val_U:=val_U1+val_U;
       if val_U>0 then bNul(rSetTag(PULSM.Par.OpUst_U1.tag, val_U));
       DevSendCmdTag(PULSM.Par.OpUst_U1.tag,'@OpUst_U1',Str(val_U),0,200);
      end;
      if IsSameText(ClickSensor, 'PULSM.UST.DN'+Str(i)) then begin
       if i=1 then val_U:=0.05;
       if i=2 then val_U:=1;
       if i=3 then val_U:=5;
       val_U:=val_U1-val_U;
       if val_U>0 then bNul(rSetTag(PULSM.Par.OpUst_U1.tag, val_U));
       DevSendCmdTag(PULSM.Par.OpUst_U1.tag,'@OpUst_U1',Str(val_U),0,200);
      end;
     end;
    end;
   end;
   {
   Plot & Tab windows
   }
   ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
   if ClickCurve<>0 then begin
    iNul(WinSelectByCurve(ClickCurve,ClickCurve));
    bNul(Voice(snd_Wheel));
   end;
  end;
  //
  // Edit tags...
  //
  if EditState=ef_Done then begin
   if CheckEditTag(PULSM.Par.OpUst_I1.tag,s) then begin
    DevSendCmdTag(PULSM.Par.OpUst_I1.tag,'@OpUst_I1',s,0,200);
   end;
   if CheckEditTag(PULSM.Par.OpUst_I2.tag,s) then begin
    DevSendCmdTag(PULSM.Par.OpUst_I2.tag,'@OpUst_I2',s,0,200);
   end;
   if CheckEditTag(PULSM.Par.OpUst_U1.tag,s) then begin
    DevSendCmdTag(PULSM.Par.OpUst_U1.tag,'@OpUst_U1',s,0,240);
   end;
   if CheckEditTag(PULSM.Par.OpUst_U2.tag,s) then begin
    DevSendCmdTag(PULSM.Par.OpUst_U2.tag,'@OpUst_U2',s,0,240);
   end;
   if CheckEditTag(PULSM.Par.OpTime.tag,s) then begin
    DevSendCmdTag(PULSM.Par.OpTime.tag,'@OpTime',s,0,_PlusInf);
   end;
   {
   Warning.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'Warning') then sNul(Edit(''));
   {
   Information.
   }
   if IsSameText(ExtractWord(1,edit('?ans 0')),'Information') then sNul(Edit(''));
  end;
  if EditState=ef_Done then begin
   Problem('Unknown tag edition!');
   sNul(Edit(''));
  end;
  if iAnd(EditState,ef_ErrorFound)<>0 then begin
   Problem('Dialog error detected!');
   sNul(Edit(''));
  end;
  s:='';
 end;
 {
 Initialize tags.
 }
 procedure PULSM_InitTags(Prefix:String);
 begin
  if not IsEmptyStr(Prefix) then begin
   InitTag(PULSM.Par.OpUst_I1.tag,    Prefix+'.OpUst_I1',    2);
   InitTag(PULSM.Par.OpUst_I2.tag,    Prefix+'.OpUst_I2',    2);
   InitTag(PULSM.Par.OpUst_U1.tag,    Prefix+'.OpUst_U1',    2);
   InitTag(PULSM.Par.OpUst_U2.tag,    Prefix+'.OpUst_U2',    2);
   InitTag(PULSM.Par.OpTime.tag,      Prefix+'.OpTime',      1);
   InitTag(PULSM.Gui.Cmd.Power.tag,   Prefix+'.Cmd.Power',   1);
   InitTag(PULSM.GUI.CMD.HELP.tag,    Prefix+'.CMD.HELP',    1);
   InitTag(PULSM.GUI.CMD.SAVEINI.tag, Prefix+'.CMD.SAVEINI', 1);
   InitTag(PULSM.GUI.CMD.LOADINI.tag, Prefix+'.CMD.LOADINI', 1);
   InitTag(PULSM.GUI.CMD.RAMP.tag,    Prefix+'.RAMP.ENABLE', 1);
   InitTag(PULSM.GUI.Rad.UST_I.tag,   Prefix+'.RAD.UST_I',   1);
   InitTag(PULSM.GUI.Rad.UST_U.tag,   Prefix+'.RAD.UST_U',   1);
   InitTag(PULSM.GUI.StepSize.tag,    Prefix+'.STEP.SIZE',   1);
  end;
 end;
 {
 PULSM cleanup.
 }
 procedure PULSM_Clear;
 var i:Integer;
 begin
  PULSM.Com.Req:='';
  PULSM.Com.Ans:='';
  for i:=1 to MaxCmdNum do PULSM.Cmd.OpBuff[i]:=_NAN;
 end;
 {
 PULSM initialization.
 }
 procedure PULSM_Init;
 var i:Integer;
 begin
  //
  // Register error codes
  //
  PULSM.ecBadCrc:=RegisterErr(progname+': bad CRC');
  PULSM.ecIllFun:=RegisterErr(progname+': bad FUN');
  PULSM.ecExcept:=RegisterErr(progname+': Modbus Exception');
  PULSM.ecTimeout:=RegisterErr(progname+': Timeout');
  //
  // Read ini file variables
  //
  PULSM.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  if PULSM.Simulator
  then Success('Run simulator mode.')
  else Success('Run as driver mode.');
  PULSM.Address:=iValDef(ReadIni('Address'),0);
  Success('Address = '+Str(PULSM.Address));
  PULSM.Com.Port:=iValDef(ReadIni('ComPort'),0);
  Success('ComPort = '+Str(PULSM.Com.Port));
  PULSM.Com.TimeOut:=iValDef(ReadIni('ComTimeOut'),0);
  Success('ComTimeOut = '+Str(PULSM.Com.TimeOut));
  PULSM.Com.TimeGap:=iValDef(ReadIni('ComTimeGap'),0);
  Success('ComTimeGap = '+Str(PULSM.Com.TimeGap));
  PULSM.Com.PollPeriod:=iValDef(ReadIni('PollPeriod'),100);
  Success('PollPeriod = '+Str(PULSM.Com.PollPeriod));
  //
  // Initialize tags...
  //
  PULSM_InitTags(ReadIni('tagPULSM'));
  //
  // Initialize COM port
  //
  if ComOpen('[SerialPort-COM'+Str(PULSM.Com.Port)+']')
  then Success('COM port initialized.')
  else Trouble('COM port failed.');
  //
  // Initialize variables
  //
  PULSM.Com.Status:=st_NoReq;
  PULSM.Com.ReqTime:=0;
  PULSM.Com.AnsTime:=0;
  PULSM.Com.LastPoll:=0;
  PULSM.Cmd.Num:=cm_ReadAll;
  for i:=1 to MaxCmdNum do PULSM.Cmd.Enabled[i]:=false;
  PULSM.Cmd.Enabled[cm_ReadAll]:=true;
  PULSM.Gui.Cmd.Power.val:=0;
  PULSM.Sim.OpCode:=0;
  PULSM.Sim.OpTime:=0;
  PULSM.Sim.opUst_I1:=0;
  PULSM.Sim.opUst_I2:=0;
  PULSM.Sim.opUst_U1:=0;
  PULSM.Sim.opUst_U2:=0;
  PULSM.Gui.RampStat:=0;
  PULSM.Gui.OpCodeFl:=0;
  bNul(iSetTag(PULSM.GUI.Rad.UST_I.tag,1));
  bNul(iSetTag(PULSM.GUI.Rad.UST_U.tag,0));
 end;
 {
 PULSM finalization.
 }
 procedure PULSM_Free;
 begin
  bNul(ComClose);
 end;
 {
 PULSM polling.
 }
 procedure PULSM_Poll;
 begin
  PULSM_GUI_POLL;
  if PULSM.Simulator then begin
   PULSM_Sim(ComRead(255));
  end else begin
   if SysTimer_Pulse(1000)>0 then begin
    UpdateAo(ao_ERRORCNT,time,GetErrCount(-1));
    UpdateAo(ao_POLLRATE,time,PULSM.Com.PollRate);
    PULSM.Com.PollRate:=0;
   end;
   if ComOpened then PULSM_CMD_CYCLE;
  end;
 end;
 
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  PULSM_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  PULSM_Init;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  PULSM_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  PULSM_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String;
  procedure SetCommand(n:Integer; v:Real);
  begin
   if (n>=1) and (n<=MaxCmdNum) then begin
    PULSM.Cmd.Enabled[n]:=not IsNan(v);
    Success(cmd+' '+Str(v));
    PULSM.Cmd.OpBuff[n]:=v;
   end;
  end;
  procedure UpdatePulsmPar(n:Integer);
  var i:Integer;
  begin
   if (n>=1) and (n<=MaxCmdNum) then
    if n=cm_OpUst_I1 then
    DevSendCmd(DevMySelf,'@OpUst_I1 '+str(rGetTag(PULSM.Par.OpUst_I1.tag)));
    if n=cm_OpUst_I2 then
    DevSendCmd(DevMySelf,'@OpUst_I2 '+str(rGetTag(PULSM.Par.OpUst_I2.tag)));
    if n=cm_OpUst_U1 then
    DevSendCmd(DevMySelf,'@OpUst_U1 '+str(rGetTag(PULSM.Par.OpUst_U1.tag)));
    if n=cm_OpUst_U2 then
    DevSendCmd(DevMySelf,'@OpUst_U2 '+str(rGetTag(PULSM.Par.OpUst_U2.tag)));
    if n=cm_OpTime then
    DevSendCmd(DevMySelf,'@OpTime '+str(iGetTag(PULSM.Par.OpTime.tag)));
  end;
 begin
  ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  cmd:='';
  arg:='';
  if GotCommand(Data,cmd,arg) then begin
   {
   @OpUst_I1 123.456
   }
   if IsSameText(cmd,'@OpUst_I1') then begin
    SetCommand(cm_OpUst_I1,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @OpUst_I2 123.456
   }
   if IsSameText(cmd,'@OpUst_I2') then begin
    SetCommand(cm_OpUst_I2,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @OpUst_U1 123.456
   }
   if IsSameText(cmd,'@OpUst_U1') then begin
    SetCommand(cm_OpUst_U1,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @OpUst_U2 123.456
   }
   if IsSameText(cmd,'@OpUst_U2') then begin
    SetCommand(cm_OpUst_U2,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @OpTime 123.456
   }
   if IsSameText(cmd,'@OpTime') then begin
    SetCommand(cm_OpTime,rVal(ExtractWord(1,arg)));
    Data:='';
   end else
   {
   @OpCode 123.456
   }
   if IsSameText(cmd,'@OpCode') then begin
    SetCommand(cm_OpCode,rVal(ExtractWord(1,arg)));
    iSetTagBitNum(PULSM.Gui.Cmd.Power.tag,0,PULSM.Cmd.OpBuff[cm_OpCode]<>0);
    Data:='';
   end else
   {
   @Edit OpUst_I1
   }
   if IsSameText(cmd,'@Edit') then begin
    if IsSameText(ExtractWord(1,arg),'OpUst_I1')
    then StartEditTag(PULSM.Par.OpUst_I1.tag,'Уставка тока 1, А');
    if IsSameText(ExtractWord(1,arg),'OpUst_I2')
    then StartEditTag(PULSM.Par.OpUst_I2.tag,'Уставка тока 2, А');
    if IsSameText(ExtractWord(1,arg),'OpUst_U1')
    then StartEditTag(PULSM.Par.OpUst_U1.tag,'Уставка напр 1, В');
    if IsSameText(ExtractWord(1,arg),'OpUst_U2')
    then StartEditTag(PULSM.Par.OpUst_U2.tag,'Уставка напр 2, В');
    if IsSameText(ExtractWord(1,arg),'OpTime')
    then StartEditTag(PULSM.Par.OpTime.tag,'Уставка времени, сек');
    Data:='';
   end else
   {
   @LoadIni
   }
   if IsSameText(cmd,'@LoadIni') then begin
    if not PULSM.Simulator then begin
     iNul(CustomIniRW('R',arg,2*Ord(not IsEmptyStr(arg))));
     UpdatePulsmPar(cm_OpUst_I1);
     UpdatePulsmPar(cm_OpUst_I2);
     UpdatePulsmPar(cm_OpUst_U1);
     UpdatePulsmPar(cm_OpUst_U2);
     UpdatePulsmPar(cm_OpTime);
    end;
    Data:='';
   end else
   {
   @SaveIni
   }
   if IsSameText(cmd,'@SaveIni') then begin
    if not PULSM.Simulator then
    iNul(CustomIniRW('W',arg,2*Ord(not IsEmptyStr(arg))));
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
