 {
 --------------------------------------------------------------------
 Emerson EC3-X32 controller driver & simulator.
 Driver have HostAddr, simulator is not.
 Emerson protocol description:
 -----------------------------
 1)Open TCP port 1030
 2)Send message "page 0\0" to controller.
 3)Reply will be like that:
   nvomo01=4.29&nvomo02=6.25&nvomo03=10.25&nvomo04=29.37&nvomo05=-3.92&
   nvomo06=1&nvomo07=2&nvomo08=26&nvomo09=29&nvomo10=11&nvomo11=2000&nvomo12=1&
   nvomo13=1&nvomo14=-1&nvomo15=10.0&nvomo16=0&nvomo17=4&nvomo18=129&
 4)Ampersand (&) separated fields are:
   Name    | Description              | Units
   -----------------------------------------------
   nvomo01 | Suction pressure         | bar
   nvomo02 | Suction gas temperature  | °C
   nvomo03 | Superheat                | K
   nvomo04 | Valve opening            | %
   nvomo05 | Evaporating temperature  | °C
   nvomo06 | Digital inputs           | 1: Cooling demand
   nvomo07 | Digital outputs          | 1: Alarm, 2:Compressor
   nvomo08 | ?                        | ?
   nvomo09 | ?                        | ?
   nvomo10 | ?                        | ?
   nvomo11 | ?                        | ?
   nvomo12 | ?                        | ?
   nvomo13 | ?                        | ?
   nvomo14 | ?                        | ?
   nvomo15 | Superheat setpoint       | K
   nvomo16 | ?                        | ?
   nvomo17 | ?                        | ?
   nvomo18 | ?                        | ?
 Config example:
 ---------------
  [DeviceList]
  &EC3X32_SIM = device software program
  [&EC3X32_SIM]
  Comment       = EC3X32 SIMULATOR
  InquiryPeriod = 1
  DevicePolling = 500, tpNormal
  ProgramSource = .\EC3X32_DRV.PAS
  DebugFlags    = 15
  OpenConsole   = 1
  HostPort      = 1030
  []
  [DeviceList]
  &EC3X32_DRV = device software program
  [&EC3X32_DRV]
  Comment       = EC3X32 DRIVER
  InquiryPeriod = 1
  DevicePolling = 100, tpNormal
  ProgramSource = .\EC3X32_DRV.PAS
  DebugFlags    = 15
  OpenConsole   = 1
  HostPort      = 1030
  HostAddr      = localhost
  TargetDevice  = &EC3X32_DRV
  TargetMessage = @EC3X32
  PollingPeriod = 1000
  []
 --------------------------------------------------------------------
 [@Help]
 |Command list: StdIn "@cmd=arg" or "@cmd arg"
 |********************************************************
 | @Help            - This help.
 | @DebugFlags=n    - Set DebugFlags,1/2/4/8=!/:/>/< view
 | @EC3X32 s        - uses to check driver, for tests only
 | @PollingPeriod=p - set polling period to p milliseconds
 |********************************************************
 []
 --------------------------------------------------------------------
 }
program EC3X32_DRV;           { EC3X32 driver & simulator        }
const
 dfTrouble         = 1;       { DebugFlags - Trouble             }
 dfSuccess         = 2;       { DebugFlags - Success             }
 dfViewExp         = 4;       { DebugFlags - ViewExp             }
 dfViewImp         = 8;       { DebugFlags - ViewImp             }
 dfEchoConn        = 16;      { DebugFlags - Echo on Connection  }
 MaxLen            = 1024;    { Max length of messages           }
 DefaultPeriod     = 1000;    { Default polling period           }
 URLHostCmnd       = 'page 0%00'; { URL-encoded HostCmnd         }
var
 r                 : Real;    { Temporary                        }
 s                 : String;  { Temporary                        }
 i                 : Integer; { Temporary                        }
 b                 : Boolean; { Temporary                        }
 Ok                : Boolean; { Program initialization is Ok?    }
 errors            : Integer; { Program error counter            }
 errorcode         : Integer; { Error code for this device       }
 fixmaxavail       : Integer; { String manager leak control      }
 DebugFlags        : Integer; { Debug bit flags                  }
 StdIn_Line        : String;  { Temporary variable               }
 HostPipe          : Integer; { Host TCP pipe reference          }
 HostPort          : Integer; { Host TCP port                    }
 HostAddr          : String;  { Host IP address                  }
 HostCmnd          : String;  { Host command to send             }
 HostName          : String;  { Host name                        }
 PeerName          : String;  { Peer name                        }
 PipeInit          : String;  { Command line for pipe_init       }
 IsDriver          : Boolean; { Is there driver or simulator     }
 HostTimer         : Real;    { Timer to send commands to host   }
 TargetDevice      : Integer; { Target device to send messages   }
 TargetMessage     : String;  { Target message prefix to send    }
 PollingPeriod     : Integer; { Polling period, ms               }
 LastConnected     : Integer; { Number of last connected streams }
 {
 Report on trouble.
 }
 procedure Trouble(msg:String);
 var b:Boolean;
 begin
  if iand(DebugFlags,dfTrouble)<>0 then
  if length(msg)>0 then writeln(devname+' ! '+msg);
  if runcount=1 then errors:=errors+1 else b:=fixerror(errorcode);
 end;
 {
 Report on problem.
 }
 procedure Problem(msg:String);
 begin
  if iAnd(DebugFlags,dfTrouble)<>0 then
  if Length(msg)>0 then Writeln(DevName+' ! '+msg);
 end;
 {
 Report on success.
 }
 procedure Success(msg:String);
 begin
  if iand(DebugFlags,dfSuccess)<>0 then
  if length(msg)>0 then writeln(devname+' : '+msg);
 end;
 {
 Report on data import to program.
 }
 procedure ViewImp(msg:String);
 begin
  if iand(DebugFlags,dfViewImp)<>0 then
  if length(msg)>0 then writeln(devname+' < '+msg);
 end;
 {
 Report on data export from program.
 }
 procedure ViewExp(msg:String);
 begin
  if iand(DebugFlags,dfViewExp)<>0 then
  if length(msg)>0 then writeln(devname+' > '+msg);
 end;
 {
 Check I/O status.
 }
 function IoError:Boolean;
 begin
  IoError:=false;
  if ioresult<>0 then begin
   Trouble('I/O error.');
   IoError:=true;
  end;
 end;
 {
 Read string line from standard input.
 }
 function StdIn_Readln(var Data:string):boolean;
 begin
  Data:='';
  if not IoError then
  if not Eof then Readln(Data);
  if IoError then Data:='';
  StdIn_Readln:=Length(Data)>0;
 end;
 {
 Initialize and check tag
 }      
 procedure InitTag(var tag:integer; name:string; typ:integer);
 begin
  tag:=findtag(name);
  if (typ>0) and (typetag(tag)<>typ)
  then Trouble('Could not init tag: '+name);
 end;
 {
 Show/hide device console.
 }
 procedure OpenConsole(Mode:Integer);
 var b:Boolean;
  procedure ShowWin(WinName:String);
  begin
   b:=WinShow(WinName);
   b:=WinDraw(WinName+'|top=317|left=0|width=600|height=317');
   if Mode=1 then b:=WinSelect(WinName) else b:=WinHide(WinName);
  end;
 begin
  if Mode>0 then ShowWin(ParamStr('Console '+DevName))
 end;
 {
 Show help in device console and echo to Main console if AllowEcho.
 Help text should be placed in program comment is [@Help] section.
 First symbol of help block should be | and will be ignored.
 }
 procedure ShowHelp(AllowEcho:Boolean);
 var i,p,sect:Integer; b:Boolean;
 begin
  sect:=ReadIniSection(text_New,12,DaqFileRef(ReadIni('ProgramSource'),'.pas'),'[@Help]');
  for i:=0 to text_NumLn(sect)-1 do begin
   if Copy(text_GetLn(sect,i),1,1)='|' then p:=2 else p:=1;
   if AllowEcho then b:=echo(devname+' : '+Copy(text_GetLn(sect,i),p));
   Success(Copy(text_GetLn(sect,i),p));
  end;
  b:=text_Free(sect);
  if AllowEcho then b:=WinSelect(ParamStr('MainConsole'));
 end;
 {
 Get string like 2006.09.21-00:12:30
 }
 function GetDateTime(ms:Real):String;
 var s:String;
 begin
  s:='';
  s:=Str(ms2sec(ms))+s;   while Length(s)<2  do s:='0'+s; s:=':'+s;
  s:=Str(ms2min(ms))+s;   while Length(s)<5  do s:='0'+s; s:=':'+s;
  s:=Str(ms2hour(ms))+s;  while Length(s)<8  do s:='0'+s; s:='-'+s;
  s:=Str(ms2day(ms))+s;   while Length(s)<11 do s:='0'+s; s:='.'+s;
  s:=Str(ms2month(ms))+s; while Length(s)<14 do s:='0'+s; s:='.'+s;
  s:=Str(ms2year(ms))+s;  while Length(s)<19 do s:='0'+s;
  GetDateTime:=s;
  s:='';
 end;
 {
 Clear all strings
 }
 procedure ClearStrings;
 begin
  s:='';
  HostAddr:='';
  HostCmnd:='';
  HostName:='';
  PeerName:='';
  PipeInit:='';
  StdIn_Line:='';
  TargetMessage:='';
  if runcount=1 then fixmaxavail:=maxavail;
  if isinf(runcount) then
  if maxavail<>fixmaxavail then Trouble('String Manager Leak = '+str(fixmaxavail-maxavail));
 end;
 {
 Driver check connection state.
 Initialization on connection/disconnection.
 }
 procedure DriverCheck;
 var b:Boolean;
 begin
  if HostPipe<>0 then
  if pipe_connected(HostPipe)<>LastConnected then begin
   LastConnected:=pipe_connected(HostPipe);
   if LastConnected>0 then begin
    HostName:=pipe_ctrl(pipe_stream(HostPipe,0),'HostIp');
    PeerName:=pipe_ctrl(pipe_stream(HostPipe,0),'PeerIp');
    if iAnd(DebugFlags,dfEchoConn)<>0 then
    b:=Echo(DevName+' : '+HostName+' connected to '+PeerName+' at '+GetDateTime(mSecNow));
    Success(HostName+' connected to '+PeerName+' at '+GetDateTime(mSecNow));
   end else begin
    if iAnd(DebugFlags,dfEchoConn)<>0 then
    b:=Echo(DevName+' : '+HostName+' disconnected from '+PeerName+' at '+GetDateTime(mSecNow));
    Success(HostName+' disconnected from '+PeerName+' at '+GetDateTime(mSecNow));
    HostName:='';
    PeerName:='';
   end;
   b:=pipe_txclear(HostPipe);
   b:=pipe_rxclear(HostPipe);
   HostTimer:=0;
  end;
 end;
 {
 Driver data readout from controller.
 }
 procedure DriverReadout;
 var Data:String; i,j:Integer;
 begin
  Data:='';
  if IsDriver then
  if HostPipe<>0 then
  if pipe_connected(HostPipe)>0 then begin
   {
   Periodically send "page 0\0" request.
   }
   if mSecNow-HostTimer>PollingPeriod then begin
    if pipe_send(HostPipe,HostCmnd)>0
    then ViewExp(HostName+': '+HostCmnd)
    else Trouble(HostName+': '+HostCmnd);
    HostTimer:=mSecNow;
   end;
   {
   Readout controller answer.
   Extract ampersand-separated data and send to target device.
   }
   j:=1;
   Data:=pipe_recv(HostPipe,MaxLen);
   for i:=1 to Length(Data) do
   if Data[i]='&' then begin
    if i>j then begin
     ViewImp(PeerName+': '+Copy(Data,j,i-j));
     if TargetDevice<>0 then
     if DevSend(TargetDevice,TargetMessage+Copy(Data,j,i-j)+CRLF)=0
     then Trouble('DevSend fails!');
    end;
    j:=i+1;
   end;
  end;
  Data:='';
 end;
 {
 Driver simulation.
 }
 procedure DriverSimulation;
 var Data:String;
 begin
  Data:='';
  if HostPipe<>0 then
  if not IsDriver then
  if pipe_connected(HostPipe)>0 then begin
   {
   Receive input.
   }
   Data:=pipe_recv(pipe_stream(HostPipe,0),MaxLen);
   {
   If "page 0" found, send data string.
   }
   if Length(Data)>0 then begin
    ViewImp(PeerName+': '+Data);
    if Pos(Trim(HostCmnd),Data)>0 then begin
     Data:='nvomo01=4.29&'
          +'nvomo02=6.25&'
          +'nvomo03=10.25&'
          +'nvomo04=29.37&'
          +'nvomo05=-3.92&'
          +'nvomo06=1&'
          +'nvomo07=2&'
          +'nvomo08=26&'
          +'nvomo09=29&'
          +'nvomo10=11&'
          +'nvomo11=2000&'
          +'nvomo12=1&'
          +'nvomo13=1&'
          +'nvomo14=-1&'
          +'nvomo15=10.0&'
          +'nvomo16=0&'
          +'nvomo17=4&'
          +'nvomo18=129&';
     if pipe_send(pipe_stream(HostPipe,0),Data)>0
     then ViewExp(HostName+': '+Data)
     else Trouble(HostName+': '+Data);
    end;
   end;
  end;
  Data:='';
 end;
 {
 Driver open TCP/IP port.
 }
 procedure DriverOpen;
 begin
  if HostPipe=0 then 
  if HostPort<>0 then begin
   if IsDriver then begin
    Success('Open driver...');
    PipeInit:='tcp port '+Str(HostPort)+' client '+HostAddr;
   end else begin
    Success('Open simulator...');
    PipeInit:='tcp port '+Str(HostPort)+' server 1';
   end;
   HostPipe:=pipe_init(PipeInit);
   if HostPipe=0
   then Trouble(PipeInit)
   else Success(PipeInit);
  end;
 end;
 {
 Driver initialization.
 }
 procedure DriverInit;
 begin
  HostPipe:=0;
  HostName:='';
  PeerName:='';
  PipeInit:='';
  HostTimer:=0;
  LastConnected:=0;
  HostCmnd:=URL_Decode(URLHostCmnd);
  HostPort:=Val(ReadIni('HostPort'));
  HostAddr:=LoCaseStr(ReadIni('HostAddr'));
  TargetDevice:=RefFind('Device '+ReadIni('TargetDevice'));
  TargetMessage:=ReadIni('TargetMessage');
  if Length(TargetMessage)>0 then TargetMessage:=TargetMessage+' ';
  PollingPeriod:=Val(ReadIni('PollingPeriod'));
  if PollingPeriod<1 then PollingPeriod:=DefaultPeriod;
  IsDriver:=Length(HostAddr)>0;
  DriverOpen;
 end;
 {
 Driver finalization.
 }
 procedure DriverFree;
 begin
  if HostPipe<>0 then b:=pipe_free(HostPipe);
  DriverCheck;
  HostPort:=0;
  HostPipe:=0;
  HostName:='';
  PeerName:='';
  HostTimer:=0;
  HostCmnd:='';
  HostAddr:='';
  TargetDevice:=0;
  LastConnected:=0;
  TargetMessage:='';
  PollingPeriod:=0;
 end;
 {
 Driver polling: check connection, then readout data.
 }
 procedure DriverPoll;
 begin
  DriverCheck;
  if IsDriver
  then DriverReadout
  else DriverSimulation;
 end;
 {
 Analyse data coming from standard input.
 }
 procedure StdIn_Process(Data:string);
 var cmd,arg:String; b:Boolean; tag:Integer;
 begin
  ViewImp('CON: '+Data);
  {
  "@cmd=arg" or "@cmd args" commands:
  }
  cmd:='';
  arg:='';
  if Length(Data)>0 then
  if Data[1]='@' then begin
   cmd:=ExtractWord(1,Data);
   arg:=Copy(Data,Pos(cmd,Data)+Length(cmd)+1);
   {}
   if IsSameText(cmd,'@EC3X32') then begin
    ViewExp(cmd+' '+arg);
    Data:='';
   end else
   {}
   if IsSameText(cmd,'@PollingPeriod') then begin
    if Val(arg)>0 then PollingPeriod:=Val(arg);
    Success(cmd+'='+Str(PollingPeriod));
    Data:='';
   end else
   {}
   if IsSameText(cmd,'@DebugFlags') then begin
    if not IsNan(rVal(arg)) then DebugFlags:=Round(rVal(arg));
    Success(cmd+'='+Str(DebugFlags));
    Data:='';
   end else
   {}
   if IsSameText(cmd,'@Help') then begin
    ShowHelp(true);
    Data:='';
   end;
   {}
   if Length(Data)>0 then begin
    Trouble(' Unrecognized command "'+Data+'".');
    Data:='';
   end;
  end;
  cmd:='';
  arg:='';
 end;
begin
 {
 Initialization actions on Start
 }
 if RunCount=1 then begin
  {
  Initialize errors...
  }
  errors:=0;
  errorcode:=registererr(devname);
  {
  Clear and initialize variables...
  }
  ClearStrings;
  DebugFlags:=Val(ReadIni('DebugFlags'));
  OpenConsole(Val(ReadIni('OpenConsole')));
  Success('MaxAvail='+Str(MaxAvail));
  DriverInit;
  {
  Is it Ok?
  }
  if errors=0 then Success('Start Ok.') else Trouble('Start Fails.');
  if errors<>0 then b:=fixerror(errorcode);
  Ok:=(errors=0);
 end else
 {
 Finalization actions on Stop...
 }
 if isinf(runcount) then begin
  DriverFree;
  ClearStrings;
  Success('Stop.');
  Success('MaxAvail='+Str(MaxAvail));
 end else
 {
 Actions on Poll
 }
 if Ok then begin
  {
  Process standard input...
  }
  while StdIn_Readln(StdIn_Line) do StdIn_Process(StdIn_Line);
  {
  Driver polling...
  }
  DriverPoll;
 end;
end.
