 {
 ***********************************************************************
 Daq Pascal application program DG645_DRV.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @PrintCmdTable    - Print Command Table state.
| @Clear            - Reset device status
| @Delay Ch, v      - edit channel delay
| @Amplitude BNC, v - edit output amplitude
| @Offset BNC, v    - edit output offset
| @Polarity BNC, v  - edit output polarity
| @Trigger          - trigger a delay
| @TriggerMode v    - edit trigger mode
| @TriggerLevel v   - edit trigger level
| @LoadIni          - load params from INI file.
| @SaveIni          - save params to   INI file.
| @AssignTag t v    - Assign tag t to value v
| @DimTagUpdate     - Update tag from DIM
| @DimGuiClick      - handle remote click from DIM
| @DimCmdMy         - my DIM cmd to device APC
|********************************************************
[]
 }
program DG645_DRV;                  { DG645 driver                     }
const
 {---------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}               { Include all Standard constants,  }
 {---------------------------------}{ And add User defined constants:  }
 st_NoReq           = 0;            { Status : No request sent         }
 st_WaitAns         = 1;            { Status : Waiting answer          }
 st_WaitGap         = 2;            { Status : Waiting time gap        }
 st_TimeOut         = 3;            { Status : TimeOut found           }

 stt_Offline        = 0;            { State device : Offline                            }
 stt_Work           = 1;            { State device : Work                               }
 stt_TimeOut        = 2;            { State device : TimOut (when PollRate = 0)         }
 stt_Warning        = 3;            { State device : Warning (when ESR > 0)             }
 stt_Simulator      = 4;            { State device : Simulator (when Simulator = true ) }

 cm_IDN             = 1;            {                                  }
 cm_ESR             = 2;            {                                  }
 cm_INSR            = 3;            {                                  }
 cm_CLR             = 4;            {                                  }
 cm_DelaySet        = 5;            {                                  }
 cm_DelayCh2        = 6;            {                                  }
 cm_DelayCh3        = 7;            {                                  }
 cm_DelayCh4        = 8;            {                                  }
 cm_DelayCh5        = 9;            {                                  }
 cm_DelayCh6        = 10;           {                                  }
 cm_DelayCh7        = 11;           {                                  }
 cm_DelayCh8        = 12;           {                                  }
 cm_DelayCh9        = 13;           {                                  }
 cm_LvlAmpSet       = 14;           {                                  }
 cm_LvlAmpOut0      = 15;           {                                  }
 cm_LvlAmpOut1      = 16;           {                                  }
 cm_LvlAmpOut2      = 17;           {                                  }
 cm_LvlAmpOut3      = 18;           {                                  }
 cm_LvlAmpOut4      = 19;           {                                  }
 cm_LvlOffSet       = 20;           {                                  }
 cm_LvlOffOut0      = 21;           {                                  }
 cm_LvlOffOut1      = 22;           {                                  }
 cm_LvlOffOut2      = 23;           {                                  }
 cm_LvlOffOut3      = 24;           {                                  }
 cm_LvlOffOut4      = 25;           {                                  }
 cm_LvlPolSet       = 26;           {                                  }
 cm_LvlPolOut0      = 27;           {                                  }
 cm_LvlPolOut1      = 28;           {                                  }
 cm_LvlPolOut2      = 29;           {                                  }
 cm_LvlPolOut3      = 30;           {                                  }
 cm_LvlPolOut4      = 31;           {                                  }
 cm_Trg             = 32;           {                                  }
 cm_TrigModeSet     = 33;           {                                  }
 cm_TrigMode        = 34;           {                                  }
 cm_TrigLvlSet      = 35;           {                                  }
 cm_TrigLvl         = 36;           {                                  }
 cm_InterfaceSet    = 37;           {                                  }
 cm_Interface       = 38;           {                                  }

 MaxCmdNum          = 37;           { Max command id number            }
 MaxItemNum         = 50;           { FIFO Max buffer                  }

 ao_POLL_RATE       = 0;            { Analog outputs...                }
 ao_ERROR_CNT       = 1;            {                                  }
 ao_ESR             = 2;            {                                  }
 ao_INSR            = 3;            {                                  }
 ao_DELAYCH2        = 4;            {                                  }
 ao_DELAYCH3        = 5;            {                                  }
 ao_DELAYCH4        = 6;            {                                  }
 ao_DELAYCH5        = 7;            {                                  }
 ao_DELAYCH6        = 8;            {                                  }
 ao_DELAYCH7        = 9;            {                                  }
 ao_DELAYCH8        = 10;           {                                  }
 ao_DELAYCH9        = 11;           {                                  }
 ao_LVLAMPOUT0      = 12;           {                                  }
 ao_LVLAMPOUT1      = 13;           {                                  }
 ao_LVLAMPOUT2      = 14;           {                                  }
 ao_LVLAMPOUT3      = 15;           {                                  }
 ao_LVLAMPOUT4      = 16;           {                                  }
 ao_LVLOFFOUT0      = 17;           {                                  }
 ao_LVLOFFOUT1      = 18;           {                                  }
 ao_LVLOFFOUT2      = 19;           {                                  }
 ao_LVLOFFOUT3      = 20;           {                                  }
 ao_LVLOFFOUT4      = 21;           {                                  }
 ao_LVLPOLOUT0      = 22;           {                                  }
 ao_LVLPOLOUT1      = 23;           {                                  }
 ao_LVLPOLOUT2      = 24;           {                                  }
 ao_LVLPOLOUT3      = 25;           {                                  }
 ao_LVLPOLOUT4      = 26;           {                                  }
 ao_TRIGGERMODE     = 27;           {                                  }
 ao_TRIGGERLVL      = 28;           {                                  }
 ao_INTERFACE       = 29;           {                                  }
 si_OFFLINE         = 'OFFLINE';    { To identify offline state        }
 DelayOnStart       = 500;          { Delay before start polling       }
 UseCmdHash         = false;        { Use CmdHash for fast search      }
 DimDeadline        = 5000;         { Detect DIM server is dead        }
 NormalMultiply     = 1000000000;   { Коэфициент преобразования в нсек }

type
 {---------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}               { Include all Standard types,      }
 {---------------------------------}{ And add User defined types:      }

var
 {---------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}               { Include all Standard variables,  }
 {---------------------------------}{ And add User defined variables:  }
 DG645             : record         { DG645 data                       }
  Simulator        : Boolean;       { Simulator mode                   }
  ecTimeOut        : Integer;       { Error code: TimeOut              }
  Com              : record         { COM port data                    }
   Port            : Integer;       { Port number                      }
   Status          : Integer;       { Status flag                      }
   TimeOut         : Integer;       { TimeOut, ms                      }
   TimeGap         : Integer;       { Time delay after poll, ms        }
   PollPeriod      : Integer;       { Poll period, ms                  }
   Req,Ans,Buf     : String;        { Request, Answer                  }
   ReqTime,AnsTime : Real;          { Request, Answer time, ms         }
   LastPoll        : Real;          { Time of last poll, ms            }
   PollRate        : Integer;       { Poll rate, poll/sec              }
  end;                              {                                  }
  Cmd              : record         { Command cycle data               }
   Num             : Integer;       { Current command number           }
   Enabled         : array [1..MaxCmdNum] of Boolean; {                }
   Acronym         : array [1..MaxCmdNum] of String;  {                }
   Channel         : array [1..MaxCmdNum] of Integer; {                }
   Name            : array [1..MaxCmdNum] of String;  {                }
   EqualChannel    : array [1..MaxCmdNum] of Integer; {                }
   Leng            : array [1..MaxCmdNum] of Integer; {                }
   Comment         : array [1..MaxCmdNum] of String;  {                }
   OpData          : array [1..MaxCmdNum] of Real;    {                }
   OpBuff          : array [1..MaxCmdNum] of Real;    {                }
   Arg             : array [1..MaxCmdNum] of String;  {                }
   SimDat          : array [1..MaxCmdNum] of String;  {                }
   AoNum           : array [1..MaxCmdNum] of Integer; {                }
   Tag             : array [1..MaxCmdNum] of Integer; {                }
  end;                              {                                  }
  FIFO     : record                 { FIFO buffer                      }
   Count   : Integer;               { Count of items                   }
   Head    : Integer;               { Head index                       }
   Item    : record                 { Data items                       }
    Data   : array[0..MaxItemNum] of String;  { Data items             }
    Ref    : array[0..MaxItemNum] of Integer; { Data items             }
   end;
  end;
  CmdHash                : Integer; { Hash list ref. for fast search   }
  POLL_ENABLE            : TTagRef; {                                  }
  POLL_RATE              : TTagRef; {                                  }
  ERROR_CNT              : TTagRef; {                                  }
  ERR                    : TTagRef; {                                  }
  SERVID                 : TTagRef; { Server Identifier pid@hostname   }
  CLOCK                  : TTagRef; { Server Date-Time                 }
  IDN_MODEL              : TTagRef; { Device Identifier                }
  IDN_SERIAL             : TTagRef; { Device serial number             }
  IDN_VERSION            : TTagRef; { Device software version          }
  ESR                    : TTagRef; { Event Status Register            }
  INSR                   : TTagRef; { Instrument Status Register       }
  TRIGGERMODE            : TTagRef; { Device Trigger Mode              }
  TRIGGERLEVEL           : TTagRef; { Device Trigger level             }
  AB_DELAY               : TTagRef; { Delay for channel A              }
  AB_WIDTH               : TTagRef; { Delay for channel B              }
  AB_LVLAMP              : TTagRef; { Level Amplitude AB               }
  AB_LVLOFF              : TTagRef; { Level Offset AB                  }
  AB_LVLPOL              : TTagRef; { Level Polarity AB                }
  CD_DELAY               : TTagRef; { Delay for channel C              }
  CD_WIDTH               : TTagRef; { Delay for channel D              }
  CD_LVLAMP              : TTagRef; { Level Amplitude CD               }
  CD_LVLOFF              : TTagRef; { Level Offset CD                  }
  CD_LVLPOL              : TTagRef; { Level Polarity CD                }
  EF_DELAY               : TTagRef; { Delay for channel E              }
  EF_WIDTH               : TTagRef; { Delay for channel F              }
  EF_LVLAMP              : TTagRef; { Level Amplitude EF               }
  EF_LVLOFF              : TTagRef; { Level Offset EF                  }
  EF_LVLPOL              : TTagRef; { Level Polarity EF                }
  GH_DELAY               : TTagRef; { Delay for channel G              }
  GH_WIDTH               : TTagRef; { Delay for channel H              }
  GH_LVLAMP              : TTagRef; { Level Amplitude GH               }
  GH_LVLOFF              : TTagRef; { Level Offset GH                  }
  GH_LVLPOL              : TTagRef; { Level Polarity GH                }
  STATE                  : TTagRef; { STATE tag                        }
  SYSMODE                : TTagRef; { System Mode tag                  }
  ModelCode              : Integer; { Encoded model ranges (Tab index) }
  DrvName                : String;  { Model name like DCU01            }
  SelfId                 : String;  { Self pid@hostname                }
 end;                               {                                  }
 ColorNorm               : Integer; { Color in normal state: lime,aqua }
 ColorWarn               : Integer; { Color in warning state: yellow   }
 cmd_DimCmdMy            : Integer; { @DimTagUpdate                    }
 cmd_DimTagUpdate        : Integer; { @DimTagUpdate                    }
 cmd_AssignTag           : Integer; { @AssignTag                       }
 cmd_PrintCmdTable       : Integer; { @PrintCmdTable                   }
 cmd_Test                : Integer; { @Test                            }
 cmd_Clear               : Integer; { @Clear                           }
 cmd_Delay               : Integer; { @Delay                           }
 cmd_Amplitude           : Integer; { @Amplitude                       }
 cmd_Offset              : Integer; { @Offset                          }
 cmd_Polarity            : Integer; { @Polarity                        }
 cmd_Trigger             : Integer; { @Trigger                         }
 cmd_TriggerMode         : Integer; { @TriggerMode                     }
 cmd_TriggerLevel        : Integer; { @TriggerLevel                    }

 {------------------------------}{ Declare procedures & functions:     }
 {$I _fun_StdLibrary}            { Include all Standard functions,     }
 {------------------------------}{ And add User defined functions:     }
 {
 Prefix for DIM @remote commands.
 }
 function DimRemote:String;
 var CanRemote:Boolean;
 begin
  CanRemote:=DIM_IsServerMode or DIM_IsClientMode;
  if (DIM_GuiClickTag=0) then CanRemote:=false;
  if (devDimSrv=0) then CanRemote:=false;
  if CanRemote
  then DimRemote:='@remote '
  else DimRemote:='';
 end;
 {
 Xor bit on click (remote version).
 }
 procedure ClickTagXorRemote(tag,XorMask:Integer);
 begin
  if IsRefTag(tag) then
  if (ClickTag=tag) then begin
   DevSendCmdLocal(DimRemote+'@AssignTag '+NameTag(tag)+' '+Str(iXor(iGetTag(tag),XorMask)));
  end;
 end;
 {
 Nice tooltip notifier.
 }
 procedure NiceNotify(aText:String; aDelay:Integer);
 begin
  if Length(aText)>0 then begin
   ShowTooltip('text "'+aText+'" preset stdNotify delay '+Str(aDelay));
  end;
 end;
 {
 Check ModelCode is valid or return zero.
 }
 function CheckModelCode(ModelCode:Integer):Integer;
 begin
  if (ModelCode<1) then ModelCode:=0 else
  CheckModelCode:=ModelCode;
 end;
 {
 Check & return DG645 ModelCode, nonzero if IDN detection was successful.
 }
 function DG645_ModelCode:Integer;
 begin
  DG645_ModelCode:=CheckModelCode(DG645.ModelCode);
 end;
 {
 Parse IDN string like 'DCU01', find ModelCode.
 }
 function DG645_ParseIDN(IDN:String; silent:Boolean):Integer;
 var ModelCode:Integer;
 begin
  ModelCode:=0; IDN:=Trim(IDN);
  if not IsSameText(IDN,si_OFFLINE) then ModelCode:=1;
  if not silent then
  if CheckModelCode(ModelCode)>0
  then Success('Detected IDN: '+IDN)
  else Problem('Fail detect IDN: '+IDN);
  DG645_ParseIDN:=ModelCode;
 end;
 {
 Find command by id (Acronym) & channel number.
 }
 function DG645_CmdFind(id,ch:String):Integer;
 var i,n:Integer;
 begin
  n:=0;
  if (Length(id)>0) then
  if UseCmdHash and (DG645.CmdHash<>0) then begin
   n:=HashList_GetLink(DG645.CmdHash,id);
  end else begin
   for i:=1 to MaxCmdNum do if n=0 then
   if IsSameText(id,DG645.Cmd.Acronym[i]) then begin
    if ((DG645.Cmd.Channel[i]=-1)
    or (DG645.Cmd.Channel[i]=val(ch))) then n:=i;
   end;
  end;
  DG645_CmdFind:=n;
 end;
 {
 Find channel by Name
 }
 function DG645_FindNumByName(name:String):Integer;
 var i,n:Integer;
 begin
  n:=0;
  if (Length(name)>0) then
  for i:=1 to MaxCmdNum do if n=0 then
   if IsSameText(name,DG645.Cmd.Name[i]) 
   then n:=i;
  if n=0 then Problem('Канал '+name+' не найден');
  DG645_FindNumByName:=n;
 end;
 {
 Handle message @AssignTag arg
 }
 procedure EM2RS_OnAssignTag(arg:String);
 var tag:Integer; w1,w2:String;
  procedure Cleanup;
  begin
   w1:=''; w2:='';
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   w1:=ExtractWord(1,arg);
   tag:=FindTag(w1);
   if (tag<>0) then begin
    w2:=ExtractWord(2,arg);
    if tag=DG645.POLL_ENABLE.tag then UpdateTag(tag,w2,0,1);
   end;
  end;
  Cleanup;
 end;
 {
 Initialize tags.
 }
 procedure DG645_FillTags(InitVal:Real);
 begin
  DG645.POLL_ENABLE.val  :=InitVal;
  DG645.POLL_RATE.val    :=InitVal;
  DG645.ERROR_CNT.val    :=InitVal;
  DG645.SERVID.val       :=InitVal;
  DG645.CLOCK.val        :=InitVal;
  DG645.IDN_MODEL.val    :=InitVal;
  DG645.IDN_SERIAL.val   :=InitVal;
  DG645.IDN_VERSION.val  :=InitVal;
  DG645.ESR.val          :=InitVal;
  DG645.INSR.val         :=InitVal;
  DG645.TRIGGERMODE.val  :=InitVal;
  DG645.TRIGGERLEVEL.val :=InitVal;
  DG645.AB_DELAY.val     :=InitVal;
  DG645.AB_WIDTH.val     :=InitVal;
  DG645.AB_LVLAMP.val    :=InitVal;
  DG645.AB_LVLOFF.val    :=InitVal;
  DG645.AB_LVLPOL.val    :=InitVal;
  DG645.CD_DELAY.val     :=InitVal;
  DG645.CD_WIDTH.val     :=InitVal;
  DG645.CD_LVLAMP.val    :=InitVal;
  DG645.CD_LVLOFF.val    :=InitVal;
  DG645.CD_LVLPOL.val    :=InitVal;
  DG645.EF_DELAY.val     :=InitVal;
  DG645.EF_WIDTH.val     :=InitVal;
  DG645.EF_LVLAMP.val    :=InitVal;
  DG645.EF_LVLOFF.val    :=InitVal;
  DG645.EF_LVLPOL.val    :=InitVal;
  DG645.GH_DELAY.val     :=InitVal;
  DG645.GH_WIDTH.val     :=InitVal;
  DG645.GH_LVLAMP.val    :=InitVal;
  DG645.GH_LVLOFF.val    :=InitVal;
  DG645.GH_LVLPOL.val    :=InitVal;
  DG645.STATE.val        :=InitVal;
 end;
 procedure DG645_InitTags(Prefix:String);
 begin
  DIM_GuiClickInit(Prefix+'.DIMGUICLICK');
  InitTag(DG645.POLL_ENABLE.tag,  Prefix+'.POLL_ENABLE', 1);
  InitTag(DG645.POLL_RATE.tag,    Prefix+'.POLL_RATE',   2);
  InitTag(DG645.ERROR_CNT.tag,    Prefix+'.ERROR_CNT',   2);
  InitTag(DG645.SERVID.tag,       Prefix+'.SERVID',      3);
  InitTag(DG645.CLOCK.tag,        Prefix+'.CLOCK',       3);
  InitTag(DG645.IDN_MODEL.tag,    Prefix+'.IDN_MODEL',   3);
  InitTag(DG645.IDN_SERIAL.tag,   Prefix+'.IDN_SERIAL',  3);
  InitTag(DG645.IDN_VERSION.tag,  Prefix+'.IDN_VERSION', 3);
  InitTag(DG645.ESR.tag,          Prefix+'.ESR',         1);
  InitTag(DG645.INSR.tag,         Prefix+'.INSR',        1);
  InitTag(DG645.TRIGGERMODE.tag,  Prefix+'.TRIGGERMODE', 1);
  InitTag(DG645.TRIGGERLEVEL.tag, Prefix+'.TRIGGERLEVEL',2);
  InitTag(DG645.AB_DELAY.tag,     Prefix+'.AB.DELAY',    2);
  InitTag(DG645.AB_WIDTH.tag,     Prefix+'.AB.WIDTH',    2);
  InitTag(DG645.AB_LVLAMP.tag,    Prefix+'.AB.LVLAMP',   2);
  InitTag(DG645.AB_LVLOFF.tag,    Prefix+'.AB.LVLOFF',   2);
  InitTag(DG645.AB_LVLPOL.tag,    Prefix+'.AB.LVLPOL',   1);
  InitTag(DG645.CD_DELAY.tag,     Prefix+'.CD.DELAY',    2);
  InitTag(DG645.CD_WIDTH.tag,     Prefix+'.CD.WIDTH',    2);
  InitTag(DG645.CD_LVLAMP.tag,    Prefix+'.CD.LVLAMP',   2);
  InitTag(DG645.CD_LVLOFF.tag,    Prefix+'.CD.LVLOFF',   2);
  InitTag(DG645.CD_LVLPOL.tag,    Prefix+'.CD.LVLPOL',   1);
  InitTag(DG645.EF_DELAY.tag,     Prefix+'.EF.DELAY',    2);
  InitTag(DG645.EF_WIDTH.tag,     Prefix+'.EF.WIDTH',    2);
  InitTag(DG645.EF_LVLAMP.tag,    Prefix+'.EF.LVLAMP',   2);
  InitTag(DG645.EF_LVLOFF.tag,    Prefix+'.EF.LVLOFF',   2);
  InitTag(DG645.EF_LVLPOL.tag,    Prefix+'.EF.LVLPOL',   1);
  InitTag(DG645.GH_DELAY.tag,     Prefix+'.GH.DELAY',    2);
  InitTag(DG645.GH_WIDTH.tag,     Prefix+'.GH.WIDTH',    2);
  InitTag(DG645.GH_LVLAMP.tag,    Prefix+'.GH.LVLAMP',   2);
  InitTag(DG645.GH_LVLOFF.tag,    Prefix+'.GH.LVLOFF',   2);
  InitTag(DG645.GH_LVLPOL.tag,    Prefix+'.GH.LVLPOL',   1);
  InitTag(DG645.STATE.tag,        Prefix+'.STATE',       1);
  InitTag(DG645.SYSMODE.tag,      Prefix+'.SYSTEMMODE',  1);
  DG645_FillTags(-MaxReal);
  {
  Server Identifier.
  }
  if DIM_IsServerMode
  then DG645.SelfId:=Str(getpid)+'@'+ParamStr('HostName')
  else DG645.SelfId:=Str(getpid)+'@'+ParamStr('ComputerName');
  {
  Colors
  }
  ColorNorm:=clAqua;
  if DIM_IsServerMode then ColorNorm:=clLime;
  if DIM_IsClientMode then ColorNorm:=clLime;
  ColorWarn:=clYellow;
  bNul(SetTagColor(DG645.SERVID.tag,ColorNorm));
  bNul(SetTagColor(DG645.CLOCK.tag,ColorNorm));
 end;
 {
 Check if COM port opened.
 }
 function IsComPortOpened:Boolean;
 begin
  IsComPortOpened:=(ComSpace>=0);
 end;
 {
 Check command number is valid.
 }
 function IsValidCmdNum(n:Integer):Boolean;
 begin
  IsValidCmdNum:=(n>=1) and (n<=MaxCmdNum);
 end;
 {
 Check if command (n) is query.
 }
 function IsQueryCmdNum(n:Integer):Boolean;
 begin
  if IsValidCmdNum(n)
  then
   if DG645.Cmd.AoNum[n]>0 then IsQueryCmdNum:=true else IsQueryCmdNum:=false;
 end;
 {
 Enable/disable command n.
 }
 procedure EnableCmdNum(n:Integer; Enabled:Boolean);
 begin
  if IsValidCmdNum(n) then DG645.Cmd.Enabled[n]:=Enabled;
 end;
 {
 Check if DG645 polling enabled, with warning or not.
 }
 function DG645_CheckPolling(Warn:Integer):Boolean;
 var flag:Boolean; msg:String;
 begin
  msg:='';
  flag:=iGetTag(DG645.POLL_ENABLE.tag)>0;
  if (Warn<>0) and not flag then begin
   msg:=StrFmt('Для изменения параметров %s надо включить Опрос.',sGetTag(DG645.IDN_MODEL.tag));
   if iAnd(Warn,1)>0 then NiceNotify(msg,30);
   if iAnd(Warn,2)>0 then UnixWBox('Предупреждение',msg,600,20,15);
   if iAnd(Warn,4)>0 then Cron('@run /sw7 unix messagebox "Предупреждение" "'+msg+'" -t 15');
   if iAnd(Warn,8)>0 then Warning(msg);
  end;
  DG645_CheckPolling:=flag;
  msg:='';
 end;
 {
 Clear commands.
 }
 procedure DG645_Clear_Cmd;
 var i:Integer;
 begin
  for i:=1 to MaxCmdNum do begin
   DG645.Cmd.Acronym[i]:='';
   DG645.Cmd.Comment[i]:='';
   DG645.Cmd.Enabled[i]:=False;
   DG645.Cmd.OpData[i] :=_NAN;
   DG645.Cmd.OpBuff[i] :=_NAN;
   DG645.Cmd.Arg[i]    :='';
   DG645.Cmd.SimDat[i] :='';
   DG645.Cmd.AoNum[i]  :=-1;
   DG645.Cmd.Tag[i]    :=0;
  end;
 end;
 {
 Initialize commands.
 }
 procedure DG645_Init_Cmd;
  procedure AddHash(n:Integer; key:String);
  begin
   key:=Trim(key);
   if Length(key)>0 then
   if IsValidCmdNum(n) then
   if DG645.CmdHash<>0 then
   bNul(HashList_SetLink(DG645.CmdHash,key,n));
  end;
  procedure Add(n,AoNum:Integer; Info:String);
  begin
   if IsValidCmdNum(n) then begin
    DG645.Cmd.Acronym[n]:=ExtractWord(1,Info);
    if ExtractWord(2,Info) <> '0' then DG645.Cmd.Acronym[n]:=ExtractWord(1,Info);
    DG645.Cmd.Channel[n]:=Val(ExtractWord(2,Info));
    DG645.Cmd.EqualChannel[n]:=Val(ExtractWord(3,Info));
    DG645.Cmd.Name[n]:=ExtractWord(4,Info);
    DG645.Cmd.Comment[n]:=ExtractWord(5,Info);
    DG645.Cmd.Enabled[n]:=Val(ExtractWord(7,Info))>0;
    DG645.Cmd.OpData[n]:=rVal(ExtractWord(8,Info));
    DG645.Cmd.SimDat[n]:=ExtractWord(9,Info);
    DG645.Cmd.Tag[n]:=FindTag(CrvName(RefAo(AoNum)));
    if ExtractWord(6,Info) <> '-1' then DG645.Cmd.Tag[n]:=FindTag(ReadIni('tagPrefix')+ExtractWord(6,Info));
    DG645.Cmd.AoNum[n]:=AoNum;
    AddHash(n,DG645.Cmd.Acronym[n]);
   end;
  end;
 begin
  DG645.Cmd.Num:=1;
  // Command          AoNum           Acronym CH EQ Name Comment                Tag             En OpData SimDat
  Add(cm_IDN,         1,              '*IDN?  -1 -1 -1   IdentificationString   -1              1  0      SIM');
  Add(cm_ESR,         ao_ESR,         '*ESR?  -1 -1 -1   EventStatusRegister    -1              1  0      0');
  Add(cm_INSR,        ao_INSR,        'INSR?  -1 -1 -1   InstrumentStatusReg    -1              1  0      0');
  Add(cm_CLR,         -1,             '*CLS   -1 -1 -1   ClearStatus            -1              0  0      0');
  // Delay and Output Commands
  Add(cm_DelaySet,    -1,             'DLAY   -1 -1 -1   DelaySet               -1              0  0      0');
  Add(cm_DelayCh2,    ao_DelayCh2,    'DLAY?   2  0  A   DelayForChannelQuery2  .AB.DELAY       1  0      0.00001');
  Add(cm_DelayCh3,    ao_DelayCh3,    'DLAY?   3  2  B   DelayForChannelQuery3  .AB.WIDTH       1  0      0.000000100');
  Add(cm_DelayCh4,    ao_DelayCh4,    'DLAY?   4  0  C   DelayForChannelQuery4  .CD.DELAY       1  0      0.000020');
  Add(cm_DelayCh5,    ao_DelayCh5,    'DLAY?   5  4  D   DelayForChannelQuery5  .CD.WIDTH       1  0      0.000000100');
  Add(cm_DelayCh6,    ao_DelayCh6,    'DLAY?   6  0  E   DelayForChannelQuery6  .EF.DELAY       1  0      0.000030');
  Add(cm_DelayCh7,    ao_DelayCh7,    'DLAY?   7  6  F   DelayForChannelQuery7  .EF.WIDTH       1  0      0.000000100');
  Add(cm_DelayCh8,    ao_DelayCh8,    'DLAY?   8  0  G   DelayForChannelQuery8  .GH.DELAY       1  0      0.000090');
  Add(cm_DelayCh9,    ao_DelayCh9,    'DLAY?   9  8  H   DelayForChannelQuery9  .GH.WIDTH       1  0      0.000000100');
  Add(cm_LvlAmpSet,   -1,             'LAMP   -1 -1 -1   LevelAmplitudeSet      -1              0  0      0');
  Add(cm_LvlAmpOut0,  ao_LvlAmpOut0,  'LAMP?   0 -1  T0  LevelAmplitudeOutput0  .T0.LVLAMP      0  0      2.5');
  Add(cm_LvlAmpOut1,  ao_LvlAmpOut1,  'LAMP?   1 -1  AB  LevelAmplitudeOutput1  .AB.LVLAMP      1  0      2.5');
  Add(cm_LvlAmpOut2,  ao_LvlAmpOut2,  'LAMP?   2 -1  CD  LevelAmplitudeOutput2  .CD.LVLAMP      1  0      2.5');
  Add(cm_LvlAmpOut3,  ao_LvlAmpOut3,  'LAMP?   3 -1  EF  LevelAmplitudeOutput3  .EF.LVLAMP      1  0      2.5');
  Add(cm_LvlAmpOut4,  ao_LvlAmpOut4,  'LAMP?   4 -1  GH  LevelAmplitudeOutput4  .GH.LVLAMP      1  0      2.5');
  Add(cm_LvlOffSet,   -1,             'LOFF   -1 -1 -1   LevelOffsetSet         -1              0  0      0');
  Add(cm_LvlOffOut0,  ao_LvlOffOut0,  'LOFF?   0 -1  T0  LevelOffsetOutput0     .T0.LVLOFF      0  0      0');
  Add(cm_LvlOffOut1,  ao_LvlOffOut1,  'LOFF?   1 -1  AB  LevelOffsetOutput1     .AB.LVLOFF      1  0      0');
  Add(cm_LvlOffOut2,  ao_LvlOffOut2,  'LOFF?   2 -1  CD  LevelOffsetOutput1     .CD.LVLOFF      1  0      0');
  Add(cm_LvlOffOut3,  ao_LvlOffOut3,  'LOFF?   3 -1  EF  LevelOffsetOutput1     .EF.LVLOFF      1  0      0');
  Add(cm_LvlOffOut4,  ao_LvlOffOut4,  'LOFF?   4 -1  GH  LevelOffsetOutput1     .GH.LVLOFF      1  0      0');
  Add(cm_LvlPolSet,   -1,             'LPOL   -1 -1 -1   LevelPolaritySet       -1              0  0      0');
  Add(cm_LvlPolOut0,  ao_LvlPolOut0,  'LPOL?   0 -1  T0  LevelPolarityOutput0   .T0.LVLPOL      0  0      1');
  Add(cm_LvlPolOut1,  ao_LvlPolOut1,  'LPOL?   1 -1  AB  LevelPolarityOutput1   .AB.LVLPOL      1  0      1');
  Add(cm_LvlPolOut2,  ao_LvlPolOut2,  'LPOL?   2 -1  CD  LevelPolarityOutput2   .CD.LVLPOL      1  0      1');
  Add(cm_LvlPolOut3,  ao_LvlPolOut3,  'LPOL?   3 -1  EF  LevelPolarityOutput3   .EF.LVLPOL      1  0      1');
  Add(cm_LvlPolOut4,  ao_LvlPolOut4,  'LPOL?   4 -1  GH  LevelPolarityOutput4   .GH.LVLPOL      1  0      1');
  // Trigger Commands
  Add(cm_Trg,         -1,             '*TRG   -1 -1 -1   SoftwareTriggerSignal  -1              0  0      0');
  Add(cm_TrigModeSet, -1,             'TSRC   -1 -1 -1   TriggerModeSet         -1              0  0      0');
  Add(cm_TrigMode,    ao_TriggerMode, 'TSRC?  -1 -1 -1   TriggerModeQuery       -1              1  0      4');
  Add(cm_TrigLvlSet,  -1,             'TLVL   -1 -1 -1   TriggerLevelSet        -1              0  0      0');
  Add(cm_TrigLvl,     ao_TriggerLvl,  'TLVL?  -1 -1 -1   TriggerLevelQuery      -1              1  0      4');
  // Interface Commands
  Add(cm_InterfaceSet, -1,            'IFCF   -1 -1 -1   InterfaceConfigSet     -1              0  0      0');
  { Add(cm_Interface,    ao_Interface,  'IFCF?  -1 -1 -1   InterfaceConfigQuery   -1              1  0      4'); }
 end;

 {
 Print command table
 }
 procedure PrintCmdTable;
 var n:Integer;
 begin
  Success('Command table:');
  Success(StrFmt('%-2s ','Cm')
         +StrFmt('%-2s ','Ao')
         +StrFmt('%-6s ','Acro')
         +StrFmt('%-30s ','Comment')
         +StrFmt('%-2s ','En')
         +StrFmt('%-2s ','Tag')
         );
  for n:=1 to MaxCmdNum do
  Success(StrFmt('%-2d ',n)
         +StrFmt('%-2d ',DG645.Cmd.AoNum[n])
         +StrFmt('%-6s ',DG645.Cmd.Acronym[n])
         +StrFmt('%-30s ',DG645.Cmd.Comment[n])
         +StrFmt('%-2d ',Ord(DG645.Cmd.Enabled[n]))
         +StrFmt('%-2d ',Ord(DG645.Cmd.Tag[n]))
         );
 end;
 {
 Calculate FIFO item index.
 }
 function param_item_index(i,L,H:Integer):Integer;
 begin
  if i<L then i:=L;
  param_item_index:=L+(i-L) mod (H-L+1);
 end;
 {
 Push message to FIFO. Return 1=OK, 0=FAIL.
 }
 function param_fifo_push(v:String):Integer;
 var item,push:Integer;
 begin
  if push=0 then
  if DG645.FIFO.Count<=MaxItemNum then begin
   item:=param_item_index(DG645.FIFO.Head+DG645.FIFO.Count,0,MaxItemNum);
   DG645.FIFO.Item.Data[item]:=v;
   DG645.FIFO.Count:=DG645.FIFO.Count+1;
   push:=1;
  end;
  param_fifo_push:=push;
 end;
 {
 Pop message from FIFO.
 }
 function param_fifo_pop(var dat:String):Integer;
 var item,pop:Integer;
 begin
  pop:=0;
  if DG645.FIFO.Count>0 then begin
   item:=param_item_index(DG645.FIFO.Head,0,MaxItemNum);
   DG645.FIFO.Head:=param_item_index(item+1,0,MaxItemNum);
   DG645.FIFO.Count:=DG645.FIFO.Count-1;
   dat:=DG645.FIFO.Item.Data[item];
   pop:=1;
  end;
  if pop=0 then begin dat:=''; end;
  param_fifo_pop:=pop;
 end;
 {
 Main command cycle.
 }
 procedure DG645_CMD_CYCLE;
  function ValidateCmdNum(Num:Integer):Integer;
  begin
   if IsValidCmdNum(Num)
   then ValidateCmdNum:=Num
   else ValidateCmdNum:=1;
  end;
  function NextCmdNum(Num:Integer):Integer;
  var i:Integer;
  begin
   i:=0;
   while (i<MaxCmdNum) do begin
    Num:=ValidateCmdNum(Num+1);
    if DG645.Cmd.Enabled[Num]
    then i:=MaxCmdNum
    else i:=i+1;
   end;
   NextCmdNum:=Num;
  end;
  procedure ClearRequest(Num,Status:Integer);
  begin
   DG645.Com.Req:='';
   DG645.Com.Ans:='';
   DG645.Com.Buf:='';
   DG645.Com.ReqTime:=0;
   DG645.Com.AnsTime:=0;
   DG645.Com.Status:=Status;
   DG645.Cmd.Num:=ValidateCmdNum(Num);
  end;
  function ReqCmd(n:Integer):String;
  var s:String;
   procedure Cleanup;
   begin
    s:='';
   end;
  begin
   Cleanup;
   if IsValidCmdNum(n) then begin
    if not IsQueryCmdNum(DG645.Cmd.Num) then if param_fifo_pop(DG645.Cmd.Arg[n])<>0 then Details('ARG POP '+DG645.Cmd.Arg[n]);
    if (DG645.Cmd.Arg[n] <> '') then begin
     s:=DG645.Cmd.Acronym[n]+' '+DG645.Cmd.Arg[n];
    end else begin
     s:=DG645.Cmd.Acronym[n];
     if (DG645.Cmd.Channel[n] <> -1) then s:=DG645.Cmd.Acronym[n]+' '+Str(DG645.Cmd.Channel[n]);
    end;
    if (Length(s)>0) then s:=s+CRLF;
   end;
   ReqCmd:=s;
   Cleanup;
  end;
  procedure PrepareRequest(n:Integer);
  begin
   if IsValidCmdNum(n) then begin
    DG645.Com.Req:=ReqCmd(n);
   end;
  end;
  procedure UpdateIDN(data:String);
  begin
   if not IsEmptyStr(data) then begin
    DG645.ModelCode:=DG645_ParseIDN(data,iGetTag(DG645.POLL_ENABLE.tag)=0);
    if (DG645_ModelCode=0) then data:=si_OFFLINE;
    if IsEmptyStr(data) then data:=si_OFFLINE;
	sNul(worddelims(','));
    bNul(sSetTag(DG645.IDN_MODEL.tag,trim(ExtractWord(2,data))));
    bNul(sSetTag(DG645.IDN_SERIAL.tag,trim(ExtractWord(3,data))));
    bNul(sSetTag(DG645.IDN_VERSION.tag,trim(ExtractWord(4,data))));
	sNul(worddelims(dump(0)));
   end;
  end;
  procedure UpdateCmdValue(n:Integer; value:Real);
  var aonum,tag:Integer;
  begin
   if IsValidCmdNum(n) and not IsNan(value) and not IsInf(value) then begin
    tag:=DG645.Cmd.Tag[n]; aonum:=DG645.Cmd.AoNum[n];
    if TypeTag(tag)=1 then begin
     bNul(iSetTag(tag,round(value)));
    end;
    if TypeTag(tag)=2 then begin
     bNul(rSetTag(tag,value));
    end;
    if aonum>=0 then UpdateAo(aonum,time,value);
    DG645.Cmd.OpData[n]:=value;
   end;
  end;
  procedure DecodeData(n:Integer; data:String);
  begin
   if IsValidCmdNum(n) then begin
    if Length(data)>0 then begin
     case n of
      cm_IDN: begin
       EnableCmdNum(n,false); UpdateIDN(data);
      end;
      cm_ESR:        UpdateCmdValue(n,Val(ExtractWord(1,data)));
      cm_INSR:       UpdateCmdValue(n,Val(ExtractWord(1,data)));
      cm_DelayCh2:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_DelayCh3:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_DelayCh4:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_DelayCh5:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_DelayCh6:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_DelayCh7:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_DelayCh8:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_DelayCh9:   UpdateCmdValue(n,rval(ExtractWord(2,data))*NormalMultiply);
      cm_LvlAmpOut0: UpdateCmdValue(n,rval(data));
      cm_LvlAmpOut1: UpdateCmdValue(n,rval(data));
      cm_LvlAmpOut2: UpdateCmdValue(n,rval(data));
      cm_LvlAmpOut3: UpdateCmdValue(n,rval(data));
      cm_LvlAmpOut4: UpdateCmdValue(n,rval(data));
      cm_LvlOffOut0: UpdateCmdValue(n,rval(data));
      cm_LvlOffOut1: UpdateCmdValue(n,rval(data));
      cm_LvlOffOut2: UpdateCmdValue(n,rval(data));
      cm_LvlOffOut3: UpdateCmdValue(n,rval(data));
      cm_LvlOffOut4: UpdateCmdValue(n,rval(data));
      cm_LvlPolOut0: UpdateCmdValue(n,val(data));
      cm_LvlPolOut1: UpdateCmdValue(n,val(data));
      cm_LvlPolOut2: UpdateCmdValue(n,val(data));
      cm_LvlPolOut3: UpdateCmdValue(n,val(data));
      cm_LvlPolOut4: UpdateCmdValue(n,val(data));
      cm_TrigMode:   UpdateCmdValue(n,Val(ExtractWord(1,data)));
      cm_TrigLvl:    UpdateCmdValue(n,rVal(ExtractWord(1,data)));
     end;
    end;
   end;
  end;
  function HandleRequest(n:Integer; Ans:String):Boolean;
  var Handled:Boolean;
  begin
   Handled:=false;
   if IsValidCmdNum(n) and (Length(Ans)>0) then begin
    ViewImp('COM < '+Ans);
    DecodeData(n,Ans);
    Handled:=True;
   end;
   HandleRequest:=Handled;
  end;
  procedure PrintDetails;
  begin
   if iAnd(DebugFlags,dfDetails)>0 then Details('Run '+Str(RunCount)
    +', Cmd '+DG645.Cmd.Acronym[DG645.Cmd.Num]
    +', Enb '+Str(Ord(DG645.Cmd.Enabled[DG645.Cmd.Num]))
    +', Sta '+Str(DG645.Com.Status)
    +', Req '+Trim(DG645.Com.Req)
    +', Ans '+Trim(DG645.Com.Ans)
    +', Buf '+Trim(DG645.Com.Buf)
    );
  end;
  procedure UpdateOpData;
  var n:Integer;
  begin
   for n:=1 to MaxCmdNum do
   if not IsNan(DG645.Cmd.OpBuff[n]) then begin
    if not IsQueryCmdNum(n) then begin
     DG645.Cmd.OpData[n]:=DG645.Cmd.OpBuff[n];
     EnableCmdNum(n,true);
    end;
    DG645.Cmd.OpBuff[n]:=_NAN;
   end;
  end;
  procedure EnableIDN;
  begin
   EnableCmdNum(cm_IDN,true);
  end;
  procedure ExecuteIdleActions;
  begin
   UpdateOpData;
   if (DG645_ModelCode=0) then EnableIDN;
  end;
 begin
  {
  COM port communication handler
  }
  if not DIM_IsClientMode then begin
   if IsComPortOpened then begin
    if iGetTag(DG645.POLL_ENABLE.tag)>0 then begin
     DG645.Cmd.Num:=ValidateCmdNum(DG645.Cmd.Num);
     if iAnd(DebugFlags,dfDetails)>0 then PrintDetails;
     if DG645.Com.Status=st_NoReq then begin
      ExecuteIdleActions;
      ClearRequest(DG645.Cmd.Num,st_NoReq);
      if mSecNow>=DG645.Com.LastPoll+DG645.Com.PollPeriod then begin
       if DG645.Cmd.Enabled[DG645.Cmd.Num] then PrepareRequest(DG645.Cmd.Num);
       if Length(DG645.Com.Req)>0 then begin
        PurgeComPort;
        if ComWrite(DG645.Com.Req) then begin
         ViewExp('COM > '+DG645.Com.Req);
         DG645.Com.Status:=st_WaitAns;
         DG645.Com.ReqTime:=mSecNow;
        end else begin
         Trouble('Could not send request '+DG645.Com.Req);
         ClearRequest(NextCmdNum(DG645.Cmd.Num),st_NoReq);
        end;
        if DG645.Com.Status=st_WaitAns
        then DG645.Com.LastPoll:=mSecNow;
       end else ClearRequest(NextCmdNum(DG645.Cmd.Num),st_NoReq);
      end;
     end else
     if DG645.Com.Status=st_WaitAns then begin
      DG645.Com.Ans:=DG645.Com.Ans+ComRead(255);
      if (DG645.Cmd.aonum[DG645.Cmd.Num]<=0) then begin
       DG645.Com.PollRate:=DG645.Com.PollRate+1;
       DG645.Com.Status:=st_WaitGap;
       DG645.Com.AnsTime:=mSecNow;
       if DG645.FIFO.Count=0 then DG645.Cmd.Enabled[DG645.Cmd.Num]:=false;
       ClearRequest(NextCmdNum(DG645.Cmd.Num)-4,st_NoReq);
      end else
      if (Length(DG645.Com.Ans)>=3) then begin
       bNul(HandleRequest(DG645.Cmd.Num,DG645.Com.Ans));
       DG645.Com.PollRate:=DG645.Com.PollRate+1;
       DG645.Com.Status:=st_WaitGap;
       DG645.Com.AnsTime:=mSecNow;
      end else
      if mSecNow>DG645.Com.ReqTime+DG645.Com.TimeOut then begin
       DG645.Com.Status:=st_TimeOut;
       if DG645.Cmd.Enabled[DG645.Cmd.Num] then DG645.Com.Status:=st_WaitGap;
      end;
     end;
     if DG645.Com.Status=st_WaitGap then begin
      if mSecNow>=DG645.Com.AnsTime+DG645.Com.TimeGap then begin
       if not IsQueryCmdNum(DG645.Cmd.Num) then EnableCmdNum(DG645.Cmd.Num,false);
       ClearRequest(NextCmdNum(DG645.Cmd.Num),st_NoReq);
      end;
     end;
     if DG645.Com.Status=st_TimeOut then begin
      Failure(DG645.ecTimeOut,'TimeOut on command '+DG645.Cmd.Acronym[DG645.Cmd.Num]+' request '+Trim(DG645.Com.Req));
      ClearRequest(NextCmdNum(DG645.Cmd.Num),st_NoReq);
      EnableIDN;
     end;
     //UpdateStateDevice;
    end else begin
     ClearRequest(NextCmdNum(0),st_NoReq);
     UpdateIDN(si_OFFLINE);
     EnableIDN;
    end;
   end;
  end;
  {
  Update DIM services
  }
  if DIM_IsServerMode then begin
   // Enforce update each 10 sec
   if SysTimer_Pulse(10000)>0 then DG645_FillTags(-MaxReal);
   if ShouldRefresh(DG645.POLL_ENABLE.val,  GetStampOfTag(DG645.POLL_ENABLE.tag,0))>0  then DIM_UpdateTag(DG645.POLL_ENABLE.tag,'');
   if ShouldRefresh(DG645.POLL_RATE.val,    GetStampOfTag(DG645.POLL_RATE.tag,0))>0    then DIM_UpdateTag(DG645.POLL_RATE.tag,'');
   if ShouldRefresh(DG645.ERROR_CNT.val,    GetStampOfTag(DG645.ERROR_CNT.tag,0))>0    then DIM_UpdateTag(DG645.ERROR_CNT.tag,'');
   if ShouldRefresh(DG645.IDN_MODEL.val,    GetStampOfTag(DG645.IDN_MODEL.tag,0))>0    then DIM_UpdateTag(DG645.IDN_MODEL.tag,'');
   if ShouldRefresh(DG645.IDN_SERIAL.val,   GetStampOfTag(DG645.IDN_SERIAL.tag,0))>0   then DIM_UpdateTag(DG645.IDN_SERIAL.tag,'');
   if ShouldRefresh(DG645.IDN_VERSION.val,  GetStampOfTag(DG645.IDN_VERSION.tag,0))>0  then DIM_UpdateTag(DG645.IDN_VERSION.tag,'');
   if ShouldRefresh(DG645.ESR.val,          GetStampOfTag(DG645.ESR.tag,0))>0          then DIM_UpdateTag(DG645.ESR.tag,'');
   if ShouldRefresh(DG645.INSR.val,         GetStampOfTag(DG645.INSR.tag,0))>0         then DIM_UpdateTag(DG645.INSR.tag,'');
   if ShouldRefresh(DG645.TRIGGERMODE.val,  GetStampOfTag(DG645.TRIGGERMODE.tag,0))>0  then DIM_UpdateTag(DG645.TRIGGERMODE.tag,'');
   if ShouldRefresh(DG645.TRIGGERLEVEL.val, GetStampOfTag(DG645.TRIGGERLEVEL.tag,0))>0 then DIM_UpdateTag(DG645.TRIGGERLEVEL.tag,'');
   if ShouldRefresh(DG645.AB_DELAY.val,     GetStampOfTag(DG645.AB_DELAY.tag,0))>0     then DIM_UpdateTag(DG645.AB_DELAY.tag,'');
   if ShouldRefresh(DG645.AB_WIDTH.val,     GetStampOfTag(DG645.AB_WIDTH.tag,0))>0     then DIM_UpdateTag(DG645.AB_WIDTH.tag,'');
   if ShouldRefresh(DG645.AB_LVLAMP.val,    GetStampOfTag(DG645.AB_LVLAMP.tag,0))>0    then DIM_UpdateTag(DG645.AB_LVLAMP.tag,'');
   if ShouldRefresh(DG645.AB_LVLOFF.val,    GetStampOfTag(DG645.AB_LVLOFF.tag,0))>0    then DIM_UpdateTag(DG645.AB_LVLOFF.tag,'');
   if ShouldRefresh(DG645.AB_LVLPOL.val,    GetStampOfTag(DG645.AB_LVLPOL.tag,0))>0    then DIM_UpdateTag(DG645.AB_LVLPOL.tag,'');
   if ShouldRefresh(DG645.CD_DELAY.val,     GetStampOfTag(DG645.CD_DELAY.tag,0))>0     then DIM_UpdateTag(DG645.CD_DELAY.tag,'');
   if ShouldRefresh(DG645.CD_WIDTH.val,     GetStampOfTag(DG645.CD_WIDTH.tag,0))>0     then DIM_UpdateTag(DG645.CD_WIDTH.tag,'');
   if ShouldRefresh(DG645.CD_LVLAMP.val,    GetStampOfTag(DG645.CD_LVLAMP.tag,0))>0    then DIM_UpdateTag(DG645.CD_LVLAMP.tag,'');
   if ShouldRefresh(DG645.CD_LVLOFF.val,    GetStampOfTag(DG645.CD_LVLOFF.tag,0))>0    then DIM_UpdateTag(DG645.CD_LVLOFF.tag,'');
   if ShouldRefresh(DG645.CD_LVLPOL.val,    GetStampOfTag(DG645.CD_LVLPOL.tag,0))>0    then DIM_UpdateTag(DG645.CD_LVLPOL.tag,'');
   if ShouldRefresh(DG645.EF_DELAY.val,     GetStampOfTag(DG645.EF_DELAY.tag,0))>0     then DIM_UpdateTag(DG645.EF_DELAY.tag,'');
   if ShouldRefresh(DG645.EF_WIDTH.val,     GetStampOfTag(DG645.EF_WIDTH.tag,0))>0     then DIM_UpdateTag(DG645.EF_WIDTH.tag,'');
   if ShouldRefresh(DG645.EF_LVLAMP.val,    GetStampOfTag(DG645.EF_LVLAMP.tag,0))>0    then DIM_UpdateTag(DG645.EF_LVLAMP.tag,'');
   if ShouldRefresh(DG645.EF_LVLOFF.val,    GetStampOfTag(DG645.EF_LVLOFF.tag,0))>0    then DIM_UpdateTag(DG645.EF_LVLOFF.tag,'');
   if ShouldRefresh(DG645.EF_LVLPOL.val,    GetStampOfTag(DG645.EF_LVLPOL.tag,0))>0    then DIM_UpdateTag(DG645.EF_LVLPOL.tag,'');
   if ShouldRefresh(DG645.GH_DELAY.val,     GetStampOfTag(DG645.GH_DELAY.tag,0))>0     then DIM_UpdateTag(DG645.GH_DELAY.tag,'');
   if ShouldRefresh(DG645.GH_WIDTH.val,     GetStampOfTag(DG645.GH_WIDTH.tag,0))>0     then DIM_UpdateTag(DG645.GH_WIDTH.tag,'');
   if ShouldRefresh(DG645.GH_LVLAMP.val,    GetStampOfTag(DG645.GH_LVLAMP.tag,0))>0    then DIM_UpdateTag(DG645.GH_LVLAMP.tag,'');
   if ShouldRefresh(DG645.GH_LVLOFF.val,    GetStampOfTag(DG645.GH_LVLOFF.tag,0))>0    then DIM_UpdateTag(DG645.GH_LVLOFF.tag,'');
   if ShouldRefresh(DG645.GH_LVLPOL.val,    GetStampOfTag(DG645.GH_LVLPOL.tag,0))>0    then DIM_UpdateTag(DG645.GH_LVLPOL.tag,'');
   if ShouldRefresh(DG645.STATE.val,        GetStampOfTag(DG645.STATE.tag,0))>0        then DIM_UpdateTag(DG645.STATE.tag,'');
   if ShouldRefresh(DG645.CLOCK.val,        GetStampOfTag(DG645.CLOCK.tag,0))>0        then DIM_UpdateTag(DG645.CLOCK.tag,'');
   if ShouldRefresh(DG645.SERVID.val,       GetStampOfTag(DG645.SERVID.tag,0))>0       then DIM_UpdateTag(DG645.SERVID.tag,'');
  end;
 end;
 {
 Procedure to show sensor help
 }
 procedure SensorHelp(s:String);
 begin
  StdSensorHelpTooltip(s,15000);
 end;
 {
 Menu Tools Starter to start editing.
 }
 procedure MenuToolsStarter;
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Команда "Инструменты"... ');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Просмотр справочной информации (HELP)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@BrowseHelp');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сбросить состояние устройства');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@Clear');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Открыть окно: '+ParamStr('CONSOLE '+DevName));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@OpenConsole');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: нормальный  (3)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 3');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: ввод-вывод (15)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 15');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Режим отладки консоли: детальный  (31)');
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@DebugFlags 31');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Загрузить параметры из INI файла');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @LoadIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Сохранить параметры в  INI файле');
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Remote @SaveIni');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Перезапустить локальный драйвер '+DevName);
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@SysEval @Daq Compile '+DevName);
   //////////////////////////////////////////
   if DIM_IsClientMode then begin
    n:=n+EditAddInputLn('Перезапустить удаленный драйвер '+DevName);
    n:=n+EditAddConfirm(EditGetLastInputLn);
    n:=n+EditAddCommand('@Remote @SysEval @Daq Compile '+DevName);
   end;
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TOOLS'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Trigger Mode to start editing.
 }
 procedure MenuTriggerMode(tag:integer);
 var n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening('Режим триггера');
   n:=n+EditAddInputLn('Что выбираете:');
   //////////////////////////////////////////
   n:=n+EditAddInputLn('Внутренний');
   n:=n+EditAddInputLn('Внешний по переднему фронту');
   n:=n+EditAddInputLn('Внешний по заднему фронту');
   n:=n+EditAddInputLn('Одиночный внешний по переднему фронту');
   n:=n+EditAddInputLn('Одиночный внешний по заднему фронту');
   n:=n+EditAddInputLn('Одиночное срабатывание');
   n:=n+EditAddInputLn('Линейный');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:12\Style:[]');
   n:=n+EditAddSetting(SetFormUnderSensorLeftBottom(ClickParams('')));
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_TriggerMode'+str(tag)),Str(iGetTag(tag)));
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu Tools Handler to handle editing.
 }
 procedure MenuToolsHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_TOOLS'));
 end;
 {
 GUI Handler to process user input...
 }
 procedure DG645_GUI_POLL;
 var ClickCurve:Integer; s:String;
  procedure Cleanup;
  begin
   DIM_GuiClickBuff:=''; s:=''; ClickCurve:=0;
  end;
 begin
  Cleanup;
  {
  Handle user mouse/keyboard clicks...
  ClickWhat=(cw_Nothing,cw_MouseDown,cw_MouseUp,cw_MouseMove,cw_KeyDown,KeyUp)
  ClickButton=(VK_LBUTTON,VK_RBUTTON,VK_CANCEL,VK_MBUTTON,VK_BACK,VK_TAB,VK_CLEAR,VK_RETURN,...)
  }
  if ClickWhat<>0 then
  repeat
   {
   Copy GUI click to DIM buffer for remote execution.
   }
   DIM_GuiClickBuff:=DIM_GuiClickCopy;
   {
   Handle MouseDown/KeyDown
   }
   if (ClickWhat=cw_MouseDown) or (ClickWhat=cw_KeyDown) then begin
    {
    Handle Left mouse button click
    }
    if (ClickButton=VK_LBUTTON) then begin
     {
     Handle local clicks
     }
     if ClickIsLocal then begin
      {
      Handle Tag clicks...
      }
      if ClickTag<>0 then begin
       ClickTagXorRemote(DG645.POLL_ENABLE.tag,1);
       if (ClickTag=DG645.AB_LVLPOL.tag)    then DevPostCmdLocal(DimRemote+'@Polarity AB,'+Str(iXor(iGetTag(ClickTag),1)));
       if (ClickTag=DG645.CD_LVLPOL.tag)    then DevPostCmdLocal(DimRemote+'@Polarity CD,'+Str(iXor(iGetTag(ClickTag),1)));
       if (ClickTag=DG645.EF_LVLPOL.tag)    then DevPostCmdLocal(DimRemote+'@Polarity EF,'+Str(iXor(iGetTag(ClickTag),1)));
       if (ClickTag=DG645.GH_LVLPOL.tag)    then DevPostCmdLocal(DimRemote+'@Polarity GH,'+Str(iXor(iGetTag(ClickTag),1)));
       if (ClickTag=DG645.AB_DELAY.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала A (нс):');
       if (ClickTag=DG645.AB_WIDTH.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала B (нс):');
       if (ClickTag=DG645.AB_LVLAMP.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Амплитуда уровня AB (Вольт):');
       if (ClickTag=DG645.AB_LVLOFF.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Смещение уровня AB (Вольт):');
       if (ClickTag=DG645.CD_DELAY.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала C (нс):');
       if (ClickTag=DG645.CD_WIDTH.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала D (нс):');
       if (ClickTag=DG645.CD_LVLAMP.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Амплитуда уровня CD (Вольт):');
       if (ClickTag=DG645.CD_LVLOFF.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Смещение уровня CD (Вольт):');
       if (ClickTag=DG645.EF_DELAY.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала E (нс):');
       if (ClickTag=DG645.EF_WIDTH.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала F (нс):');
       if (ClickTag=DG645.EF_LVLAMP.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Амплитуда уровня EF (Вольт):');
       if (ClickTag=DG645.EF_LVLOFF.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Смещение уровня EF (Вольт):');
       if (ClickTag=DG645.GH_DELAY.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала G (нс):');
       if (ClickTag=DG645.GH_WIDTH.tag)     then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Задержка канала H (нс):');
       if (ClickTag=DG645.GH_LVLAMP.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Амплитуда уровня GH (Вольт):');
       if (ClickTag=DG645.GH_LVLOFF.tag)    then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Смещение уровня GH (Вольт):');
       if (ClickTag=DG645.TRIGGERLEVEL.tag) then if DG645_CheckPolling(1) then StartEditTag(ClickTag,'Уровень триггера:');
       if (ClickTag=DG645.TRIGGERMODE.tag)  then begin // Menu Trigger Mode
        bNul(Voice(snd_Click));
        MenuTriggerMode(DG645.TRIGGERMODE.tag);
       end;
      end;
      {
      Handle sensor clicks...
      }
      if ClickSensor<>'' then begin
       bNul(Voice(snd_Click));
       if ClickSensor='TOOLS'              then MenuToolsStarter;
       if ClickSensor='HELP'               then DevPostCmdLocal('@BrowseHelp');
       if ClickSensor='DG645.TRIGGER'      then DevPostCmdLocal(DimRemote+'@Trigger');
       if ClickSensor='LABEL_DEVICESTATUS' then DevPostCmdLocal(DimRemote+'@Clear');
      end;
      {
      Select Plot & Tab windows by curve...
      }
      ClickCurve:=RefFind('Curve '+ClickParams('Curve'));
      if IsRefCurve(ClickCurve) then begin
       iNul(WinSelectByCurve(ClickCurve,ClickCurve));
       bNul(Voice(snd_Wheel));
      end;
      {
      Console commands: @url_encoded_sensor ...
      }
      if LooksLikeCommand(ClickSensor) then begin
       DevPostCmdLocal(url_decode(ClickSensor));
       bNul(Voice(snd_Click));
      end;
     end;
     {
     Handle remote clicks comes from DIM via @DimGuiClick message.
     @DimGuiClick default handler decode and write events to FIFO,
     so we can find it as clicks and can handle it in usual way.
     }
     if ClickIsRemote then begin
      {
      Show time difference.
      }
      if DebugFlagEnabled(dfDetails) then
      Details('Remote Click Time Diff '+Str(mSecNow-rVal(ClickParams('When')))+' ms');
      {
      Handle remote console commands...
      }
      s:=Dim_GuiConsoleRecv(DevName,'');
      if LooksLikeCommand(s) then DevSendCmdLocal(s);
      {
      Handle remote sensor clicks...
      }
      if TypeTag(ClickTag)>0 then begin
       s:=ClickParams('NewValue');
       if Length(s)>0 then begin
        if ClickTag=DG645.POLL_ENABLE.tag then UpdateTag(ClickTag,s,0,1);
       end;
      end;
     end;
    end;
    {
    Handle Right mouse button click
    }
    if (ClickButton=VK_RBUTTON) then begin
     SensorHelp(Url_Decode(ClickParams('Hint')));
    end;
   end;
  until (ClickRead=0);
  {
  Edit handling...
  }
  if EditStateDone then begin
   if CheckEditTag(DG645.AB_DELAY.tag,s)     then DevSendCmdLocal('@Delay A,'+s);
   if CheckEditTag(DG645.AB_WIDTH.tag,s)     then DevSendCmdLocal('@Delay B,'+s);
   if CheckEditTag(DG645.AB_LVLAMP.tag,s)    then DevSendCmdLocal('@Amplitude AB,'+s);
   if CheckEditTag(DG645.AB_LVLOFF.tag,s)    then DevSendCmdLocal('@Offset AB,'+s);
   if CheckEditTag(DG645.CD_DELAY.tag,s)     then DevSendCmdLocal('@Delay C,'+s);
   if CheckEditTag(DG645.CD_WIDTH.tag,s)     then DevSendCmdLocal('@Delay D,'+s);
   if CheckEditTag(DG645.CD_LVLAMP.tag,s)    then DevSendCmdLocal('@Amplitude CD,'+s);
   if CheckEditTag(DG645.CD_LVLOFF.tag,s)    then DevSendCmdLocal('@Offset CD,'+s);
   if CheckEditTag(DG645.EF_DELAY.tag,s)     then DevSendCmdLocal('@Delay E,'+s);
   if CheckEditTag(DG645.EF_WIDTH.tag,s)     then DevSendCmdLocal('@Delay F,'+s);
   if CheckEditTag(DG645.EF_LVLAMP.tag,s)    then DevSendCmdLocal('@Amplitude EF,'+s);
   if CheckEditTag(DG645.EF_LVLOFF.tag,s)    then DevSendCmdLocal('@Offset EF,'+s);
   if CheckEditTag(DG645.GH_DELAY.tag,s)     then DevSendCmdLocal('@Delay G,'+s);
   if CheckEditTag(DG645.GH_WIDTH.tag,s)     then DevSendCmdLocal('@Delay H,'+s);
   if CheckEditTag(DG645.GH_LVLAMP.tag,s)    then DevSendCmdLocal('@Amplitude GH,'+s);
   if CheckEditTag(DG645.GH_LVLOFF.tag,s)    then DevSendCmdLocal('@Offset GH,'+s);
   if CheckEditTag(DG645.TRIGGERLEVEL.tag,s) then DevSendCmdLocal('@TriggerLevel '+s);
   {
   Menu TriggerMode
   }
   if EditTestResultName(EditGetUID('MENU_TriggerMode'+str(DG645.TriggerMode.tag))) then begin
    if EditTestResultCode(mr_OK) then begin
     s:=Str(EditGetMenuListSelectedIndex);
     DIM_GuiConsoleSend(DevName, '@TriggerMode '+s);
    end;
    EditReset;
   end;
   {
   Menu TOOLS
   }
   MenuToolsHandler;
   {
   Warning, Information dialog completion.
   }
   if EditTestResultName('Warning') then EditReset;
   if EditTestResultName('Information') then EditReset;
  end;
  if EditStateDone then begin
   Problem('Uncompleted edit detected!');
   EditReset;
  end else
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
  Cleanup;
 end;
 {
 Update State Device Tag and Sensor
 }
 procedure UpdateStateDevice;
  begin
   if sGetTag(DG645.IDN_MODEL.tag)='Симулятор' then bNul(iSetTag(DG645.STATE.tag, stt_Simulator)) else
   begin
    if sGetTag(DG645.IDN_MODEL.tag)='OFFLINE' then bNul(iSetTag(DG645.STATE.tag, stt_Offline)) else
    if DG645.Com.Status=st_TimeOut  then bNul(iSetTag(DG645.STATE.tag, stt_TimeOut)) else
    if iGetTag(DG645.ESR.tag)>0     then bNul(iSetTag(DG645.STATE.tag, stt_Warning)) else
    bNul(iSetTag(DG645.STATE.tag, stt_Work));
   end;
  end;
 {
 Handle data requests in simulation mode.
 }
 procedure DG645_Sim(Req:String);
 var cmd,req_cmd:String; n,ChannelID:integer;
  procedure Cleanup;
  begin
   cmd:=''; req_cmd:='';
  end;
 begin
  Cleanup;
  req_cmd:=ExtractWord(1,req);
  if Length(Req)>0 then begin
   ViewImp('COM < '+Req);
   n:=DG645_CmdFind(ExtractWord(1,Req),ExtractWord(2,Req));
   { DG645.Com.Ans:=Req+CRLF; //Одобряем все запросы, без проверки}
   if IsValidCmdNum(n)  then DG645.Com.Ans:=DG645.Cmd.SimDat[n];
   if req_cmd='*IDN?'   then DG645.Com.Ans:='Режим симулятора';
   if req_cmd='DLAY?'   then DG645.Com.Ans:='0,'+DG645.Cmd.SimDat[n];
   if req_cmd='TSRC'    then begin
    DG645.Cmd.SimDat[cm_TrigMode]:=ExtractWord(2,req);
    DG645.Com.Ans:=ExtractWord(2,req);
   end;
   // Запрос на изменение задержки канала
   if req_cmd='DLAY' then begin
    ChannelID:=DG645_CmdFind('DLAY?',ExtractWord(2,Req));
    DG645.Cmd.SimDat[ChannelID]:=ExtractWord(4,req);
   end;
   // Запрос на изменение амплитуды канала
   if req_cmd='LAMP' then begin
    ChannelID:=DG645_CmdFind('LAMP?',ExtractWord(2,Req));
    DG645.Cmd.SimDat[ChannelID]:=ExtractWord(3,req);
   end;
   // Запрос на изменение смещения канала
   if req_cmd='LOFF' then begin
    ChannelID:=DG645_CmdFind('LOFF?',ExtractWord(2,Req));
    DG645.Cmd.SimDat[ChannelID]:=ExtractWord(3,req);
   end;
   // Запрос на изменение полярности канала
   if req_cmd='LPOL' then begin
    ChannelID:=DG645_CmdFind('LPOL?',ExtractWord(2,Req));
    DG645.Cmd.SimDat[ChannelID]:=ExtractWord(3,req);
   end; 
   // Запрос на изменение источника триггера
   if req_cmd='TSRC' then begin
    ChannelID:=DG645_CmdFind('TSRC?',Req);
    DG645.Cmd.SimDat[ChannelID]:=ExtractWord(2,req);
   end;
   // Запрос на изменение уровня триггера
   if req_cmd='TLVL' then begin
    ChannelID:=DG645_CmdFind('TLVL?',Req);
    DG645.Cmd.SimDat[ChannelID]:=ExtractWord(2,req);
   end;
   {
   Send reply to requests
   }
   if Length(DG645.Com.Ans)>0 then begin
    if ComWrite(DG645.Com.Ans+CRLF) then
    begin
     ViewExp('COM > '+DG645.Com.Ans);
    end
    else Trouble('Could not send '+DG645.Com.Ans);
   end;
  end;
  DG645.Com.Req:='';
  DG645.Com.Ans:='';
  Cleanup;
 end;
 {
 DG645 cleanup.
 }
 procedure DG645_Clear;
 begin
  DG645_Clear_Cmd;
  DG645.CmdHash:=0;
  DG645.Com.Req:='';
  DG645.Com.Ans:='';
  DG645.Com.Buf:='';
  DG645.ModelCode:=0;
  DG645.DrvName:='';
  DG645.SelfId:='';
 end;
 {
 DG645 initialization.
 }
 procedure DG645_Init;
 begin
  {
  Register error codes
  }
  DG645.ecTimeOut:=RegisterErr(progname+': TimeOut');
  {
  Initialize variables
  }
  DG645.Com.ReqTime:=0;
  DG645.Com.AnsTime:=0;
  DG645.Com.LastPoll:=0;
  DG645.Com.Status:=st_NoReq;
  DG645.Fifo.Count:=0;
  DG645.Fifo.Head:=0;
  {
  Initialize command table.
  }
  DG645.CmdHash:=HashList_Init(0);
  DG645_Init_Cmd;
  {
  Read ini file variables
  }
  DG645.Simulator:=iValDef(ReadIni('Simulator'),0)<>0;
  if DG645.Simulator
  then Success('Run simulator mode.')
  else Success('Run as driver mode.');
  DG645.Com.Port:=iValDef(ReadIni('ComPort'),0);
  Success('ComPort = '+Str(DG645.Com.Port));
  DG645.Com.TimeOut:=iValDef(ReadIni('ComTimeOut'),250);
  Success('ComTimeOut = '+Str(DG645.Com.TimeOut));
  DG645.Com.TimeGap:=iValDef(ReadIni('ComTimeGap'),10);
  Success('ComTimeGap = '+Str(DG645.Com.TimeGap));
  DG645.Com.PollPeriod:=iValDef(ReadIni('PollPeriod'),100);
  Success('PollPeriod = '+Str(DG645.Com.PollPeriod));
  DG645.DrvName:=ReadIniVar('DrvName',28);
  Success('DrvName = '+DG645.DrvName);
  DG645.ModelCode:=DG645_ParseIDN(DG645.DrvName,false);
  {
  Initialize tags...
  }
  if not DG645.Simulator then begin
   DG645_InitTags(ReadIni('tagPrefix'));
   bNul(sSetTag(DG645.IDN_MODEL.tag,DG645.DrvName));
  end;
  {
  Initialize COM port
  }
  if not DIM_IsClientMode then begin
   if ComOpen('[SerialPort-COM'+Str(DG645.Com.Port)+']')
   then Success('COM port initialized.')
   else Trouble('COM port failed.');
   bNul(iSetTag(DG645.SYSMODE.tag,0));
  end else bNul(iSetTag(DG645.SYSMODE.tag,1));
 end;
 {
 DG645 finalization.
 }
 procedure DG645_Free;
 begin
  bNul(ComClose);
  if (DG645.CmdHash<>0) then begin
   bNul(HashList_Free(DG645.CmdHash));
   DG645.CmdHash:=0;
  end;
 end;
 {
 DG645 polling.
 }
 procedure DG645_Poll;
 begin
  UpdateStateDevice;
  if DG645.Simulator then begin
   if IsComPortOpened then begin
    DG645.Com.Req:=DG645.Com.Req+ComRead(255);
    if (Length(DG645.Com.Req)>=3) then
    begin
     if (StrFetch(DG645.Com.Req,Length(DG645.Com.Req))=Chr(10)) then
     begin
      DG645_Sim(DG645.Com.Req);
     end;
    end;
   end;
  end else begin
   DG645_GUI_POLL;
   if not DIM_IsClientMode then begin
    if SysTimer_Pulse(1000)>0 then begin
     bNul(rSetTag(DG645.ERROR_CNT.tag,GetErrCount(-1)));
     UpdateAo(ao_ERROR_CNT,time,GetErrCount(-1));
     bNul(rSetTag(DG645.POLL_RATE.tag,DG645.Com.PollRate));
     UpdateAo(ao_POLL_RATE,time,DG645.Com.PollRate);
     DG645.Com.PollRate:=0;
     {
     Update Host Date-Time.
     }
     bNul(sSetTag(DG645.SERVID.tag,DG645.SelfId));
     bNul(sSetTag(DG645.CLOCK.tag,GetDateTime(mSecNow)));
     bNul(SetTagColor(DG645.SERVID.tag,ColorNorm));
     bNul(SetTagColor(DG645.CLOCK.tag,ColorNorm));
    end;
    if mSecNow-FixmSecNow>DelayOnStart then
    DG645_CMD_CYCLE;
   end;
  end;
  if DIM_IsClientMode then begin
   if (ShouldRefresh(DG645.CLOCK.dat,GetStampOfTag(DG645.CLOCK.tag,0))>0) then begin
    bNul(SetTagColor(DG645.SERVID.tag,ColorNorm));
    bNul(SetTagColor(DG645.CLOCK.tag,ColorNorm));
    DG645.CLOCK.tim:=mSecNow;
   end;
   if (SysTimer_Pulse(1000)>0) then
   if (mSecNow-DG645.CLOCK.tim>DimDeadline) then begin
    bNul(sSetTag(DG645.SERVID.tag,'Server Disconnected'));
    bNul(SetTagColor(DG645.SERVID.tag,ColorWarn));
    bNul(SetTagColor(DG645.CLOCK.tag,ColorWarn));
   end;
  end;
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  DG645_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  if Val(ReadIni('CustomIniAutoLoad'))=1 then iNul(CustomIniRw('R','',2));
  DG645_Init;
  cmd_DimCmdMy      := RegisterStdInCmd('@DimCmdMy',      '');
  cmd_DimTagUpdate  := RegisterStdInCmd('@DimTagUpdate',  '');
  cmd_AssignTag     := RegisterStdInCmd('@AssignTag',     '');
  cmd_PrintCmdTable := RegisterStdInCmd('@PrintCmdTable', '');
  cmd_Test          := RegisterStdInCmd('@Test',          '');
  cmd_Clear         := RegisterStdInCmd('@Clear',         '');
  cmd_Delay         := RegisterStdInCmd('@Delay',         '');
  cmd_Amplitude     := RegisterStdInCmd('@Amplitude',     '');
  cmd_Offset        := RegisterStdInCmd('@Offset',        '');
  cmd_Polarity      := RegisterStdInCmd('@Polarity',      '');
  cmd_Trigger       := RegisterStdInCmd('@Trigger',       '');
  cmd_TriggerMode   := RegisterStdInCmd('@TriggerMode',   '');
  cmd_TriggerLevel  := RegisterStdInCmd('@TriggerLevel',  '');
  RunStartupScript;
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then iNul(CustomIniRW('W','',2));
  RunFinallyScript;
  DG645_Free;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  DG645_Poll;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg,cdm_arg_nocode,v:String; cmdid,tag,n,push:Integer;
  procedure Cleanup;
  begin
   cmd:=''; arg:=''; cdm_arg_nocode:=''; v:='';
  end;
  procedure SettingCmd(n:Integer; v:String);
  begin
   if IsValidCmdNum(n) then begin
    Success(cmd+' '+DG645.Cmd.Acronym[n]+' '+v);
    DG645.Cmd.Arg[n]:=v;
    EnableCmdNum(n,true);
   end;
  end;
  procedure UpdateAoByTag(AoNum,tag:Integer);
  begin
   if (AoNum>=0) then begin
    if TypeTag(tag)=1 then UpdateAo(AoNum,time,iGetTag(tag));
    if TypeTag(tag)=2 then UpdateAo(AoNum,time,rGetTag(tag));
   end;
  end;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  Cleanup;
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @DIMCMDMY - my DIM cmd to device APC
   }
   if (cmdid=cmd_DimCmdMy) then begin
    cdm_arg_nocode:=trim(base64_decode(arg));
    if LooksLikeCommand(cdm_arg_nocode) then DevSendCmdLocal(cdm_arg_nocode);
    Success('@DIMCMDMY='+cmd+' arg<'+cdm_arg_nocode+'>');
   end else
   {
   @DimTagUpdate tag
   }
   if (cmdid=cmd_DimTagUpdate) then begin
    if DIM_IsClientMode and not DIM_IsServerMode then begin
     tag:=FindTag(Trim(arg));
     if TypeTag(tag)>0 then begin
      if tag=DG645.ESR.tag          then UpdateAoByTag(ao_ESR,tag);
      if tag=DG645.INSR.tag         then UpdateAoByTag(ao_INSR,tag);
      if tag=DG645.AB_DELAY.tag     then UpdateAoByTag(ao_DELAYCH2,tag);
      if tag=DG645.AB_WIDTH.tag     then UpdateAoByTag(ao_DELAYCH3,tag);
      if tag=DG645.CD_DELAY.tag     then UpdateAoByTag(ao_DELAYCH4,tag);
      if tag=DG645.CD_WIDTH.tag     then UpdateAoByTag(ao_DELAYCH5,tag);
      if tag=DG645.EF_DELAY.tag     then UpdateAoByTag(ao_DELAYCH6,tag);
      if tag=DG645.EF_WIDTH.tag     then UpdateAoByTag(ao_DELAYCH7,tag);
      if tag=DG645.GH_DELAY.tag     then UpdateAoByTag(ao_DELAYCH8,tag);
      if tag=DG645.GH_WIDTH.tag     then UpdateAoByTag(ao_DELAYCH9,tag);
      if tag=DG645.AB_LVLAMP.tag    then UpdateAoByTag(ao_LVLAMPOUT1,tag);
      if tag=DG645.CD_LVLAMP.tag    then UpdateAoByTag(ao_LVLAMPOUT2,tag);
      if tag=DG645.EF_LVLAMP.tag    then UpdateAoByTag(ao_LVLAMPOUT3,tag);
      if tag=DG645.GH_LVLAMP.tag    then UpdateAoByTag(ao_LVLAMPOUT4,tag);
      if tag=DG645.AB_LVLOFF.tag    then UpdateAoByTag(ao_LVLOFFOUT1,tag);
      if tag=DG645.CD_LVLOFF.tag    then UpdateAoByTag(ao_LVLOFFOUT2,tag);
      if tag=DG645.EF_LVLOFF.tag    then UpdateAoByTag(ao_LVLOFFOUT3,tag);
      if tag=DG645.GH_LVLOFF.tag    then UpdateAoByTag(ao_LVLOFFOUT4,tag);
      if tag=DG645.AB_LVLPOL.tag    then UpdateAoByTag(ao_LVLPOLOUT1,tag);
      if tag=DG645.CD_LVLPOL.tag    then UpdateAoByTag(ao_LVLPOLOUT2,tag);
      if tag=DG645.EF_LVLPOL.tag    then UpdateAoByTag(ao_LVLPOLOUT3,tag);
      if tag=DG645.GH_LVLPOL.tag    then UpdateAoByTag(ao_LVLPOLOUT4,tag);
      if tag=DG645.TRIGGERMODE.tag  then UpdateAoByTag(ao_TRIGGERMODE,tag);
      if tag=DG645.TRIGGERLEVEL.tag then UpdateAoByTag(ao_TRIGGERLVL,tag);
     end;
    end;
   end else
   {
   @AssignTag
   }
   if (cmdid = cmd_AssignTag) then begin
    EM2RS_OnAssignTag(arg);
    Data:='';
   end else
   {
   @Clear
   }
   if (cmdid=cmd_Clear) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     SettingCmd(cm_CLR,'');
    end;
    Data:='';
   end else
   {
   @Delay A,3
   @Delay Channel, Value
   }
   if (cmdid=cmd_Delay) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     n:=DG645_FindNumByName(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=str(DG645.Cmd.Channel[n])+','+str(DG645.Cmd.EqualChannel[n])+','+str(rval(ExtractWord(2,arg))/NormalMultiply);
      push:=param_fifo_push(v);
      if push=0 then if DebugFlagEnabled(dfDetails) then Details('FIFO OVER');
      if push>0 then if DebugFlagEnabled(dfDetails) then Details(Cmd+'='+Str(push));
      EnableCmdNum(cm_DelaySet,true);
     end;
    end;
    Data:='';
   end else
   {
   @Amplitude AB,3
   @Amplitude BNC, Value
   }
   if (cmdid=cmd_Amplitude) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     n:=DG645_FindNumByName(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=str(DG645.Cmd.Channel[n])+','+ExtractWord(2,arg);
      push:=param_fifo_push(v);
      if push=0 then if DebugFlagEnabled(dfDetails) then Details('FIFO OVER');
      if push>0 then if DebugFlagEnabled(dfDetails) then Details(Cmd+'='+Str(push));
      EnableCmdNum(cm_LvlAmpSet,true);
     end;
    end;
    Data:='';
   end else
   {
   @Offset AB,3
   @Offset BNC, Value
   }
   if (cmdid=cmd_Offset) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     n:=DG645_FindNumByName(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=str(DG645.Cmd.Channel[n])+','+ExtractWord(2,arg);
      push:=param_fifo_push(v);
      if push=0 then if DebugFlagEnabled(dfDetails) then Details('FIFO OVER');
      if push>0 then if DebugFlagEnabled(dfDetails) then Details(Cmd+'='+Str(push));
      EnableCmdNum(cm_LvlOffSet,true);
     end;
    end;
    Data:='';
   end else
   {
   @Polarity AB,1
   @Polarity BNC, Value
   }
   if (cmdid=cmd_Polarity) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     n:=DG645_FindNumByName(ExtractWord(1,arg));
     if IsValidCmdNum(n) then begin
      v:=str(DG645.Cmd.Channel[n])+','+ExtractWord(2,arg);
      push:=param_fifo_push(v);
      if push=0 then if DebugFlagEnabled(dfDetails) then Details('FIFO OVER');
      if push>0 then if DebugFlagEnabled(dfDetails) then Details(Cmd+'='+Str(push));
      EnableCmdNum(cm_LvlPolSet,true);
     end;
    end;
    Data:='';
   end else
   {
   @Trigger
   }
   if (cmdid=cmd_Trigger) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     SettingCmd(cm_Trg,'');
    end;
    Data:='';
   end else
   {
   @TriggerMode 5
   0 = Internal
   1 = External rising edges
   2 = External falling edges
   3 = Single shot external rising edges
   4 = Single shot external falling edges
   5 = Single shot
   6 = Line
   }
   if (cmdid=cmd_TriggerMode) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     v:=ExtractWord(1,arg);
     push:=param_fifo_push(v);
     if push=0 then if DebugFlagEnabled(dfDetails) then Details('FIFO OVER');
     if push>0 then if DebugFlagEnabled(dfDetails) then Details(Cmd+'='+Str(push));
     SettingCmd(cm_TrigModeSet,v);
    end;
    Data:='';
   end else
   {
   @TriggerLevel AB,3
   @TriggerLevel BNC, Value
   }
   if (cmdid=cmd_TriggerLevel) then begin
    if not DG645.Simulator and not DIM_IsClientMode then begin
     v:=ExtractWord(1,arg);
     push:=param_fifo_push(v);
     if push=0 then if DebugFlagEnabled(dfDetails) then Details('FIFO OVER');
     if push>0 then if DebugFlagEnabled(dfDetails) then Details(Cmd+'='+Str(push));
     EnableCmdNum(cm_TrigLvlSet,true);
     end;
    Data:='';
   end else
   {
   @PrintCmdTable
   }
   if (cmdid=cmd_PrintCmdTable) then begin
    PrintCmdTable;
    Data:='';
   end else
   {
   @Test
   }
   if (cmdid=cmd_Test) then begin
    DevSendCmdLocal('@Delay A,1');
    DevSendCmdLocal('@Delay B,12');
    DevSendCmdLocal('@Delay C,123');
    DevSendCmdLocal('@Delay D,1234');
    DevSendCmdLocal('@Delay E,12345');
    DevSendCmdLocal('@Delay F,123456');
    DevSendCmdLocal('@Delay G,1234567');
    DevSendCmdLocal('@Delay H,12345789');
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{<== Please never change this code ***}
{***************************************************}
