package flashx.textLayout.conversion
{
   import flashx.textLayout.elements.TextFlow;
   import flashx.textLayout.tlf_internal;
   import flashx.textLayout.elements.ParagraphElement;
   import flashx.textLayout.elements.FlowLeafElement;
   import flashx.textLayout.elements.LinkElement;
   import flashx.textLayout.formats.ITextLayoutFormat;
   import flashx.textLayout.elements.TCYElement;
   import flashx.textLayout.elements.SpanElement;
   import flashx.textLayout.elements.InlineGraphicElement;
   import flashx.textLayout.formats.FormatValue;
   import flashx.textLayout.formats.Float;
   import flashx.textLayout.elements.BreakElement;
   import flashx.textLayout.elements.TabElement;
   import flashx.textLayout.formats.TabStopFormat;
   import flashx.textLayout.formats.Direction;
   import flashx.textLayout.formats.TextAlign;
   import flashx.textLayout.formats.LeadingModel;
   import flashx.textLayout.formats.TextLayoutFormat;
   import flash.text.engine.TabAlignment;
   import flashx.textLayout.formats.TextDecoration;
   import flash.text.engine.FontPosture;
   import flash.text.engine.FontWeight;
   import flashx.textLayout.elements.FlowElement;
   import flash.text.engine.Kerning;
   import flash.utils.getQualifiedClassName;
   import flashx.textLayout.elements.FlowGroupElement;
   
   use namespace tlf_internal;
   
   [ExcludeClass]
   class HtmlExporter implements ITextExporter
   {
      
      private static var _config:flashx.textLayout.conversion.ImportExportConfiguration;
      
      private static const brRegEx:RegExp = / /;
       
      function HtmlExporter()
      {
         super();
         if(!_config)
         {
            _config = new flashx.textLayout.conversion.ImportExportConfiguration();
            _config.addIEInfo("P",ParagraphElement,null,this.exportParagraph,true);
            _config.addIEInfo("A",LinkElement,null,this.exportLink,false);
            _config.addIEInfo("TCY",TCYElement,null,this.exportTCY,false);
            _config.addIEInfo("SPAN",SpanElement,null,this.exportSpan,false);
            _config.addIEInfo("IMG",InlineGraphicElement,null,this.exportImage,false);
            _config.addIEInfo("TAB",TabElement,null,this.exportTab,false);
            _config.addIEInfo("BR",BreakElement,null,this.exportBreak,false);
         }
      }
      
      private static function getSpanTextReplacementXML(ch:String) : XML
      {
         return <BR/>;
      }
      
      public function export(source:TextFlow, conversionType:String) : Object
      {
         if(conversionType == ConversionType.STRING_TYPE)
         {
            return this.exportToString(source);
         }
         if(conversionType == ConversionType.XML_TYPE)
         {
            return this.exportToXML(source);
         }
         return null;
      }
      
      private function exportToString(textFlow:TextFlow) : String
      {
         var result:String = null;
         var originalSettings:Object = null;
         originalSettings = XML.settings();
         try
         {
            XML.ignoreProcessingInstructions = false;
            XML.ignoreWhitespace = false;
            XML.prettyPrinting = false;
            result = this.exportToXML(textFlow).toXMLString();
            XML.setSettings(originalSettings);
         }
         catch(e:Error)
         {
            XML.setSettings(originalSettings);
            throw e;
         }
         return result;
      }
      
      private function exportToXML(textFlow:TextFlow) : XML
      {
         var para:ParagraphElement = null;
         var lastPara:ParagraphElement = null;
         var xml:XML = <BODY/>;
         var firstLeaf:FlowLeafElement = textFlow.getFirstLeaf();
         if(firstLeaf)
         {
            para = firstLeaf.getParagraph();
            for(lastPara = textFlow.getLastLeaf().getParagraph(); true; )
            {
               xml.appendChild(this.exportElement(para));
               if(para == lastPara)
               {
                  break;
               }
               para = textFlow.findLeaf(para.getAbsoluteStart() + para.textLength).getParagraph();
            }
         }
         var html:XML = <HTML/>;
         html.appendChild(xml);
         return html;
      }
      
      private function exportParagraph(name:String, para:ParagraphElement) : XML
      {
         var xml:XML = new XML("<" + name + "/>");
         var fontXML:XML = this.exportFont(para.computedFormat);
         this.exportChildren(fontXML,para);
         this.nest(xml,fontXML);
         return this.exportParagraphFormat(xml,para);
      }
      
      private function exportLink(name:String, link:LinkElement) : Object
      {
         var xml:XML = new XML("<" + name + "/>");
         if(link.href)
         {
            xml.@HREF = link.href;
         }
         if(link.target)
         {
            xml.@TARGET = link.target;
         }
         else
         {
            xml.@TARGET = "_blank";
         }
         this.exportChildren(xml,link);
         var format:ITextLayoutFormat = link.computedFormat;
         var ifDifferentFromFormat:ITextLayoutFormat = link.getParagraph().computedFormat;
         var font:XML = this.exportFont(format,ifDifferentFromFormat);
         return Boolean(font)?this.nest(font,xml):xml;
      }
      
      private function exportTCY(name:String, tcy:TCYElement) : XMLList
      {
         var xml:XML = new XML("<" + name + "/>");
         this.exportChildren(xml,tcy);
         return xml.children();
      }
      
      private function exportSpan(name:String, span:SpanElement) : Object
      {
         var xml:XML = new XML("<" + name + "/>");
         BaseTextLayoutExporter.exportSpanText(xml,span,brRegEx,getSpanTextReplacementXML);
         var children:Object = xml.children();
         if(Boolean(children.length() == 1) && Boolean(children[0].nodeKind() == "text"))
         {
            children = xml.text()[0];
         }
         return this.exportSpanFormat(children,span);
      }
      
      private function exportImage(name:String, image:InlineGraphicElement) : XML
      {
         var xml:XML = new XML("<" + name + "/>");
         if(image.id)
         {
            xml.@ID = image.id;
         }
         if(image.source)
         {
            xml.@SRC = image.source;
         }
         if(Boolean(image.width !== undefined) && Boolean(image.width != FormatValue.AUTO))
         {
            xml.@WIDTH = image.width;
         }
         if(Boolean(image.height !== undefined) && Boolean(image.height != FormatValue.AUTO))
         {
            xml.@HEIGHT = image.height;
         }
         if(image.float != Float.NONE)
         {
            xml.@ALIGN = image.float;
         }
         return xml;
      }
      
      private function exportBreak(name:String, breakElement:BreakElement) : XML
      {
         return new XML("<" + name + "/>");
      }
      
      private function exportTab(name:String, tabElement:TabElement) : Object
      {
         return this.exportSpan(name,tabElement);
      }
      
      private function exportTextFormatAttribute(textFormatXML:XML, attrName:String, attrVal:*) : XML
      {
         if(!textFormatXML)
         {
            textFormatXML = <TEXTFORMAT/>;
         }
         textFormatXML[attrName] = attrVal;
         return textFormatXML;
      }
      
      private function exportParagraphFormat(xml:XML, para:ParagraphElement) : XML
      {
         var textAlignment:String = null;
         var textFormat:XML = null;
         var firstLeaf:FlowLeafElement = null;
         var lineHeight:Number = NaN;
         var tabStopsString:String = null;
         var tabStop:TabStopFormat = null;
         var paraFormat:ITextLayoutFormat = para.computedFormat;
         switch(paraFormat.textAlign)
         {
            case TextAlign.START:
               textAlignment = paraFormat.direction == Direction.LTR?TextAlign.LEFT:TextAlign.RIGHT;
               break;
            case TextAlign.END:
               textAlignment = paraFormat.direction == Direction.LTR?TextAlign.RIGHT:TextAlign.LEFT;
               break;
            default:
               textAlignment = paraFormat.textAlign;
         }
         xml.@ALIGN = textAlignment;
         if(paraFormat.paragraphStartIndent != 0)
         {
            textFormat = this.exportTextFormatAttribute(textFormat,paraFormat.direction == Direction.LTR?"LEFTMARGIN":"RIGHTMARGIN",paraFormat.paragraphStartIndent);
         }
         if(paraFormat.paragraphEndIndent != 0)
         {
            textFormat = this.exportTextFormatAttribute(textFormat,paraFormat.direction == Direction.LTR?"RIGHTMARGIN":"LEFTMARGIN",paraFormat.paragraphEndIndent);
         }
         if(paraFormat.textIndent != 0)
         {
            textFormat = this.exportTextFormatAttribute(textFormat,"INDENT",paraFormat.textIndent);
         }
         if(paraFormat.leadingModel == LeadingModel.APPROXIMATE_TEXT_FIELD)
         {
            firstLeaf = para.getFirstLeaf();
            if(firstLeaf)
            {
               lineHeight = TextLayoutFormat.lineHeightProperty.computeActualPropertyValue(firstLeaf.computedFormat.lineHeight,firstLeaf.getEffectiveFontSize());
               if(lineHeight != 0)
               {
                  textFormat = this.exportTextFormatAttribute(textFormat,"LEADING",lineHeight);
               }
            }
         }
         var tabStops:Array = paraFormat.tabStops;
         if(tabStops)
         {
            tabStopsString = "";
            for each(tabStop in tabStops)
            {
               if(tabStop.alignment != TabAlignment.START)
               {
                  break;
               }
               if(tabStopsString.length)
               {
                  tabStopsString = tabStopsString + ", ";
               }
               tabStopsString = tabStopsString + tabStop.position;
            }
            if(tabStopsString.length)
            {
               textFormat = this.exportTextFormatAttribute(textFormat,"TABSTOPS",tabStopsString);
            }
         }
         return Boolean(textFormat)?this.nest(textFormat,xml):xml;
      }
      
      private function exportSpanFormat(xml:Object, span:SpanElement) : Object
      {
         var format:ITextLayoutFormat = span.computedFormat;
         var outerElement:Object = xml;
         if(format.textDecoration.toString() == TextDecoration.UNDERLINE)
         {
            outerElement = this.nest(<U/>,outerElement);
         }
         if(format.fontStyle.toString() == FontPosture.ITALIC)
         {
            outerElement = this.nest(<I/>,outerElement);
         }
         if(format.fontWeight.toString() == FontWeight.BOLD)
         {
            outerElement = this.nest(<B/>,outerElement);
         }
         var exportedParent:FlowElement = span.getParentByType(LinkElement);
         if(!exportedParent)
         {
            exportedParent = span.getParagraph();
         }
         var font:XML = this.exportFont(format,exportedParent.computedFormat);
         if(font)
         {
            outerElement = this.nest(font,outerElement);
         }
         return outerElement;
      }
      
      private function exportFontAttribute(fontXML:XML, attrName:String, attrVal:*) : XML
      {
         if(!fontXML)
         {
            fontXML = <FONT/>;
         }
         fontXML[attrName] = attrVal;
         return fontXML;
      }
      
      private function exportFont(format:ITextLayoutFormat, ifDifferentFromFormat:ITextLayoutFormat = null) : XML
      {
         var font:XML = null;
         var rgb:String = null;
         if(Boolean(!ifDifferentFromFormat) || Boolean(ifDifferentFromFormat.fontFamily != format.fontFamily))
         {
            font = this.exportFontAttribute(font,"FACE",format.fontFamily);
         }
         if(Boolean(!ifDifferentFromFormat) || Boolean(ifDifferentFromFormat.fontSize != format.fontSize))
         {
            font = this.exportFontAttribute(font,"SIZE",format.fontSize);
         }
         if(Boolean(!ifDifferentFromFormat) || Boolean(ifDifferentFromFormat.color != format.color))
         {
            rgb = format.color.toString(16);
            while(rgb.length < 6)
            {
               rgb = "0" + rgb;
            }
            rgb = "#" + rgb;
            font = this.exportFontAttribute(font,"COLOR",rgb);
         }
         if(Boolean(!ifDifferentFromFormat) || Boolean(ifDifferentFromFormat.trackingRight != format.trackingRight))
         {
            font = this.exportFontAttribute(font,"LETTERSPACING",format.trackingRight);
         }
         if(Boolean(!ifDifferentFromFormat) || Boolean(ifDifferentFromFormat.kerning != format.kerning))
         {
            font = this.exportFontAttribute(font,"KERNING",format.kerning == Kerning.OFF?"0":"1");
         }
         return font;
      }
      
      private function exportElement(flowElement:FlowElement) : Object
      {
         var className:String = getQualifiedClassName(flowElement);
         var info:FlowElementInfo = _config.lookupByClass(className);
         if(info != null)
         {
            return info.exporter(_config.lookupName(className),flowElement);
         }
         return null;
      }
      
      private function exportChildren(xml:XML, flowGroupElement:FlowGroupElement) : void
      {
         for(var i:int = 0; i < flowGroupElement.numChildren; i++)
         {
            xml.appendChild(this.exportElement(flowGroupElement.getChildAt(i)));
         }
      }
      
      private function nest(parent:XML, children:Object) : XML
      {
         parent.setChildren(children);
         return parent;
      }
   }
}
