#!/bin/sh

# Usage: _ROOT_HIST_1D.sh  file.dat
# Prepare environment for call ROOT
# Then call ROOT, wait and show GIF
# set ZSH_DONTSETCP=1 for cyrillic.

#####################################################################
## Usage: delete_file $1
## Check file $1 and delete if one exists.
#####################################################################
function delete_file() {                # delete specified file
 [ -z "$1" ] && return 0                # file name is empty?
 [ -e "$1" ] && rm -f "$1"              # delete if exists
 [ -e "$1" ] && return 1 || return 0    # return exit code
}

#####################################################################
## Usage: show_usage $1
## Print usage information on program $1.
#####################################################################
function show_usage() {                 # show usage information
 echo "Usage: $1 datafile.dat"                         # show syntax
 echo "Process datafile.dat via ROOT script ${1%.*}.C" # show comment
 return 0
}

#####################################################################
## init_and_check_variables "$1" - initialize & check uses variables
#####################################################################
function init_and_check_variables() {            # init and check variables
 iam="$(whoami)"                                 # assign user name
 startdir="$(pwd)"                               # assign startup directory
 osname="$(uname -o)";                           # OS name: GNU/Linux,Cygwin,...
 if [ "$osname" = "Cygwin" ]; then               # On Cygwin, i.e. Windows_NT
  startdir="$(cygpath -m "$startdir")"           # we should convert file names
  arg0="$(echo "$0" | tr '\\' '/')"              # we should replace all
  arg1="$(echo "$1" | tr '\\' '/')"              # backslash chars
 else                                            # to slash chars
  arg0="$0"                                      # On Unix systems
  arg1="$1"                                      # we should not
 fi                                              # do that
 if [ $# -eq 0 ]; then                           # if no arguments
  echo "No arguments specified!"                 # show report
  show_usage "$(basename $arg0)"                 # show usage info
  exit 0                                         # end exit
 fi                                              #
 progdir="${arg0%/*}"                            # called program directory
 [ "$progdir" = "$arg0" ] && progdir="$startdir" # case no directory specified
 progext="${arg0%.*}"                            # program without extension
 progext="${arg0#$progext}"                      # program extension, like .sh
 progname="$(basename $arg0)"                    # program name+extension, like name.sh
 datadir="${arg1%/*}"                            # data files directory
 [ "$datadir" = "$arg1" ] && datadir="$(pwd)"    # case no directory specified
 export logFile="${arg1%.*}.log"                 # assign and export log file name
 export datFile="${arg1%.*}.dat"                 # assign and export dat file name
 export gifFile="${arg1%.*}.gif"                 # assign and export gif file name
 delete_file "$logFile"                          # delete log file
 delete_file "$gifFile"                          # delete gif file
 date +%Y.%m.%d-%H.%M.%S:   | tee -a "$logFile"  # show report...
 echo "User:     $iam"      | tee -a "$logFile"
 echo "OSname:   $osname"   | tee -a "$logFile"
 echo "Command:  $arg0"     | tee -a "$logFile"
 echo "Argument: $arg1"     | tee -a "$logFile"
 echo "StartDir: $startdir" | tee -a "$logFile"
 echo "ProgDir:  $progdir"  | tee -a "$logFile"
 echo "ProgExt:  $progext"  | tee -a "$logFile"
 echo "ProgName: $progname" | tee -a "$logFile"
 echo "DataDir:  $datadir"  | tee -a "$logFile"
 echo "logFile:  $logFile"  | tee -a "$logFile"
 echo "datFile:  $datFile"  | tee -a "$logFile"
 echo "gifFile:  $gifFile"  | tee -a "$logFile"
 runhide="saferun -th"
 runshow="saferun -t"
 return 0
}

#####################################################################
## Usage: progress_bar $1 $2
## Show/hide ($1) progress bar with identifier $2.
#####################################################################
function progress_bar() {               # show/hide progress bar
 local mode="$(echo $1 | tr A-Z a-z)"   # translate to lower case
 local title="pid#$$: Wait for $2..."   # find title and comment
 local comment="$2 is running, please wait..."
 [ "$mode" = "hide" ] && $runshow wbusy "$title" /stop /sound
 [ "$mode" = "show" ] && $runshow wbusy "$title" "$comment" /marquee
 return 0
}

#####################################################################
## Usage: wait_for_file $1 $2
## Wait until file $1 exist or timeout $2 seconds.
#####################################################################
function wait_for_file() {
 [ -z "$1" ] && return 0                  # if empty FileName, return
 [ -e "$1" ] && return 0                  # if exist FileName, return
 [ -z "$2" ] && return 0                  # if empty Timeout, return
 [ "$2" -le 0 ] && return 0               # if Timeout <= 0,  return
 sleep 1                                  # sleep 1 second
 wait_for_file "$1" "$(echo $2-1 | bc)"   # recursive call, Timeout-1
 return 0                                 # done
}

#####################################################################
## Usage: show_report_on_success $1
## Show report on success and wait timeout $1 seconds.
#####################################################################
function show_report_on_success() {
 echo ""                 | tee -a "$logFile"
 echo "Result: SUCCESS." | tee -a "$logFile"
 echo ""                 | tee -a "$logFile"
 $runhide speak "Success."
 if [ -e "$logFile" ]; then
  $runshow unix Title View "$logFile"... \& cat "$logFile" \& waitkey ${1-60} "" "Press any key to continue..."
 fi
 printf "@echo Data analysis done (_ROOT_HIST_1D) - SUCCESS.\n@open $(cygpath -w $logFile)\n@tty close 0" | u2d | send2crwdaq 
 return 0
}

#####################################################################
## Usage: show_report_on_error Timeout
## Show report on error and wait Timeout sec
#####################################################################
function show_report_on_error() {
 echo ""                  | tee -a "$logFile"
 echo "Result: ERROR."    | tee -a "$logFile"
 echo ""                  | tee -a "$logFile"
 $runhide speak "Failure."
 if [ -e "$datFile" ]; then
  $runshow unix Title View "$datFile"... \& head -n 23 "$datFile" \& waitkey ${1-60} "" "Press any key to continue..."
 fi
 sleep 1
 if [ -e "$logFile" ]; then
  $runshow unix Title View "$logFile"... \& cat "$logFile" \& waitkey ${1-60} "" "Press any key to continue..."
 fi
 sleep 1
 $runhide wprompt "Error!" "Could not process $datFile" Ok 1:10 x
 printf "@echo Data analysis done (_ROOT_HIST_1D) - FAILS.\n@open $(cygpath -w $logFile)\n@tty close 0" | u2d | send2crwdaq 
 return 0
}

#####################################################################
## Usage: execute_root FileName
## Call ROOT to execute program filename.C
#####################################################################
execute_root() {
 [ -z "$1" ] && return 0                  # if empty FileName, return
 [ -e "$1" ] || return 0                  # if exist FileName, return
 echo ""                    | tee -a "$logFile"
 echo "Now execute ROOT..." | tee -a "$logFile"
 echo ""                    | tee -a "$logFile"
 root -l -b -x -q "$1" 2>&1 | tee -a "$logFile"
 echo ""                    | tee -a "$logFile"
}

#####################################################################
## Usage: view_file filename
## Show specified file by Imagine viewer
#####################################################################
view_file()  {                        # open file by using Imagine viewer    
 [ -z "$1" ] && return 0              # if not filename specified, return
 [ -e "$1" ] && $runshow imagine "$1" # if file exist call Imagine viewer
 return 0
}


#####################################################################
## Usage: main datfile
## Call ROOT and show result of datfile processing
#####################################################################
function main() {
 init_and_check_variables $*
 progress_bar show $progname
 pushd "$progdir" && {
  echo "WorkDir:  $(cygpath -m $(pwd))" | tee -a "$logFile"
  execute_root "${progname%.*}.C"
  wait_for_file "$gifFile" 5
  if [ -e "$gifFile" ]; then
   view_file "$gifFile"
   sleep 1
   show_report_on_success 60
   delete_file "$datFile"
  else
   show_report_on_error 60
   delete_file "$datFile"
  fi
 }
 popd
 progress_bar hide $progname
 return 0
}

################
### Main program
################

main "$1" "$2" "$3" "$4" "$5" "$6" "$7" "$8" "$9"

exit $?

###############
### Program end
###############
