////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для построения гистограмы распределения величины,
заданной формулой (по умолчанию y) по точкам данной кривой с помощью
пакета математической обработки ROOT (http://root.cern.ch).
Если задана "Область интересов" ROI, то в обработке будут участвовать
только точки, попавшие в эту область. При вызове надо задать следующие
параметры. Caption - надпись над графиком гистограммы. HistName - имя
гисторгаммы (без пробелов). Formula - формула, по которой вычисляется
величина для гистограмирования, это функция (x,y,i) координат (x,y) и
номера точки i, по умолчанию y. NumBins - число бинов (то есть ячеек)
гистограмы. RangeLeft, RangeRight задают диапазон гистограмы, они
вычисляются как функция (a,b) - вычисленного по точкам кривой
диапазона изменения величины. Но можно задавать и просто числа, если
они откуда-то известны. FitWith - метод фитирования, по умолчанию gaus,
пустая строка отменяет фитирование. По умолчанию доступны методы
фитирования: (gaus,expo,landau,pol1,pol2,..polN). При необходимости
в набор методов можно добавить и другие, но тогда надо править файл C.
FillColor,FillStyle - задает цвет (0=White, 1=Black, ... etc) и стиль
заливки (0=None,1=Solid,3244=nice, ... etc) гистограммы, которые
можно посмотреть в c:\root\root-colors.gif, c:\root\root-fills.gif.
ScreenWidth,ScreenHeight - задают размер экрана, то есть размер
изображения в пикселях.
[]
[Manual.Eng]
This plugin uses to plot histogram of value via ROOT.
[]
[Arguments.Rus]
Caption = Distribution of %Value% %FitWith%
HistName = Value
Formula = y
NumBins = 100
RangeLeft = a-(b-a)/4
RangeRight = b+(b-a)/4
FitWith = gaus
FillColor = 20
FillStyle = 3244
ScreenWidth = 1280
ScreenHeight = 900
UsesShell = .sh or .bat
[]
[Arguments.Eng]
Caption = Distribution of %Value% %FitWith%
HistName = Value
Formula = y
NumBins = 100
RangeLeft = a-(b-a)/4
RangeRight = b+(b-a)/4
FitWith = gaus
FillColor = 20
FillStyle = 3344
ScreenWidth = 1280
ScreenHeight = 900
UsesShell = .sh or .bat
[]
}

library _root_hist_1d;

{$I _crw_sysdef}

{$IFDEF FPC}{$mode Delphi}{$ENDIF}{$H+}

{$R *.res}

uses
 _crw_sharemem, // NB: THIS UNIT MUST BE FIRST !!!
 {$IFDEF UNIX} cthreads, dl, {$ENDIF}
 {$IFDEF WINDOWS} windows, {$ENDIF}
 sysutils, classes, math, graphics,
 _crw_crwapi;

//////////////////////////////////////////////////
{$I _crw_plugin_declare} // Declare CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:TCrwApi):Integer;
//////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNanInf;
var
 Roi : TRect2D; p : TPoint2D; Interpreter : TScriptInterpreter;
 Caption, HistName, Formula, RangeLeft, RangeRight, FitWith, OutName, shName, BatName : String;
 i, NumBins, FillColor, FillStyle, ScreenWidth, ScreenHeight, TheCurve : Integer;
 r, a, b, Left, Right, ms, ri : Double;
 OutFile :Text;
 {}
 function GoodPoint(const p:TPoint2d; constRoi:TRect2d):Boolean;
 begin
  Result:=False;
  with CrwApi.SysApi do begin
   if IsNan(p.x) or IsNan(p.y) or IsInf(p.x) or IsInf(p.y) then Exit;
   if not IsNan(Roi.A.X+Roi.B.X+Roi.A.Y+Roi.B.Y) then begin
    if (p.X<Roi.A.X) or (p.X>Roi.B.X) then Exit;
    if (p.Y<Roi.A.Y) or (p.Y>Roi.B.Y) then Exit;
   end;
  end;
  Result:=True;
 end;
 {}
 procedure SetValue(const expr:String; var val:Double);
 begin
  if not Interpreter.SetValue(PChar(expr),val) then Raise EDanApi.Create('Interpreter error!');
 end;
 {}
 procedure GetValue(const expr:String; var val:Double);
 begin
  if not Interpreter.GetValue(PChar(expr),val) then Raise EDanApi.Create('Interpreter error!');
 end;
 {}
 procedure RunScript;
 begin
  if Interpreter.RunScript<>0 then Raise EDanApi.Create('Interpreter error!');
 end;
 {}
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;               
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  //
  // Check input data.
  //
  ms:=mSecNow;
  RedirectStdIn(Input);            
  RedirectStdOut(Output);
  if Target <> ForDataAnalysis then Raise EDanApi.Create(RusEng('Неверное значение Target!','Invalid Target!'));
  if not WindowExists(swin) then Raise EDanApi.Create(RusEng('Не найдено окно - источник!','Source window not found!'));
  if not WindowExists(twin) then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!','Target window not found!'));
  if CurvesCount[swin]=0 then Raise EDanApi.Create(RusEng('Нет данных для обработки!','No input data curves found!'));
  Roi:=WindowRoi[swin];
  //
  // Check if input curve exists (have to be selected or single and have points).
  //
  TheCurve:=SelectedCurve[swin];
  if TheCurve=0 then if CurvesCount[swin]=1 then TheCurve:=1;
  if TheCurve=0 then Raise EDanApi.Create(RusEng('Не выбрана кривая!','No selected curve!'));
  if CurveFlags(TheCurve,0,1E-10,3) and CheckFlags <> 0
  then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для обработки!','Curve %d is not convenient for processing!'),[TheCurve]));
  //
  // Read user defined parameters.
  //
  Caption:=Trim(GetArgumentAsString('Caption'));
  HistName:=Trim(ExtractWord(1,GetArgumentAsString('HistName'),ScanSpaces));
  Caption:=Trim(StringReplace(Caption,'%Value%',HistName,[rfReplaceAll, rfIgnoreCase]));
  Formula:=Trim(GetArgumentAsString('Formula'));
  NumBins:=StrToIntDef(GetArgumentAsString('NumBins'),0);
  RangeLeft:=Trim(GetArgumentAsString('RangeLeft'));
  RangeRight:=Trim(GetArgumentAsString('RangeRight'));
  FitWith:=Trim(GetArgumentAsString('FitWith'));
  if IsEmptyStr(FitWith)
  then Caption:=Trim(StringReplace(Caption,'%FitWith%',FitWith,[rfReplaceAll, rfIgnoreCase]))
  else Caption:=Trim(StringReplace(Caption,'%FitWith%','+ '+FitWith+' fit',[rfReplaceAll, rfIgnoreCase]));
  FillColor:=StrToIntDef(GetArgumentAsString('FillColor'),20);
  FillStyle:=StrToIntDef(GetArgumentAsString('FillStyle'),1);
  ScreenWidth:=StrToIntDef(GetArgumentAsString('ScreenWidth'),1280);
  ScreenHeight:=StrToIntDef(GetArgumentAsString('ScreenHeight'),900);
  if IsEmptyStr(Caption) or IsEmptyStr(HistName) or IsEmptyStr(Formula) or (NumBins<10)
  or IsEmptyStr(RangeLeft) or IsEmptyStr(RangeRight) or (ScreenWidth<50) or (ScreenHeight<50)
  then Raise EDanApi.Create(RusEng('Ошибка задания аргументов!','Invalid input arguments!'));
  //
  // Define temporary data file and bat script file.
  // *.dat - uses as temporary buffer to transfer data to ROOT.
  // *.bat - script uses to call ROOT and to show result.
  //
  OutName:=TempDir+'\_ROOT_HIST_1D.dat';
  ShName:=HomeDir+'\Resource\DataAnalysisPlugins\_ROOT_HIST_1D.sh';
  BatName:=HomeDir+'\Resource\DataAnalysisPlugins\_ROOT_HIST_1D.bat';
  //
  // Use DAQ Script interpreter to calculate Value's for histogram.
  //
  Interpreter:=CreateScriptInterpreter;
  try
   //
   // Calculate Left,Right - Value's range.
   //
   Left:=MaxDouble; Right:=-Left;
   Interpreter.Script:='_='+Trim(Formula);
   for i:=0 to CurveLength[TheCurve]-1 do begin
    p:=CurvePoint[TheCurve,i];
    if not GoodPoint(p,Roi) then Continue;
    SetValue('x',p.x);
    SetValue('y',p.y);
    ri:=i;
    SetValue('i',ri);
    RunScript;
    GetValue('_',r);
    Left:=Min(Left,r);
    Right:=Max(Right,r);
   end;
   if Left>Right then Raise EDanApi.Create(RusEng('Нет точек в ROI!','No poins in ROI!'));
   //
   // Calculate a - left axis range for histogram.
   //
   Interpreter.Script:='_='+Trim(RangeLeft);
   SetValue('a',Left);
   SetValue('b',Right);
   RunScript;
   GetValue('_',r);
   a:=r;
   //
   // Calculate b - right axis range for histogram.
   //
   Interpreter.Script:='_='+Trim(RangeRight);
   SetValue('a',Left);
   SetValue('b',Right);
   RunScript;
   GetValue('_',r);
   b:=r;
   //
   // Open temporary dat-file.
   //
   Assign(OutFile,OutName);
   try
    Rewrite(OutFile);
    //
    // Write header - user defined parameters.
    //
    Writeln(OutFile,Caption);
    Writeln(Outfile,HistName);
    Writeln(OutFile,Formula);
    Writeln(OutFile,NumBins);
    Writeln(OutFile,a);
    Writeln(OutFile,b);
    Writeln(OutFile,FitWith);
    Writeln(OutFile,FillColor);
    Writeln(OutFile,FillStyle);
    Writeln(OutFile,ScreenWidth);
    Writeln(OutFile,ScreenHeight);
    //
    // Calculate Value's and write to dat-file.
    //
    Interpreter.Script:='_='+Trim(Formula);
    for i:=0 to CurveLength[TheCurve]-1 do begin
     p:=CurvePoint[TheCurve,i];
     if not GoodPoint(p,Roi) then Continue;
     SetValue('x',p.x);
     SetValue('y',p.y);
     ri:=i;
     SetValue('i',ri);
     RunScript;
     GetValue('_',r);
     Writeln(OutFile,r);
    end;
   finally
    Close(OutFile);
   end;
   //
   // Send command to execute bat-script to process dat-file with ROOT.
   //
   if IsSameText(ExtractWord(1,GetArgumentAsString('UsesShell'),ScanSpaces),'.bat')
   then Interpreter.Script:=Format('@system @async @run /hide "%s" "%s"',[BatName,OutName])
   else Interpreter.Script:=Format('@system @async @run /hide unix bash "%s" "%s"',[ShName,OutName]);
   // Open TTY port to listen incoming data from script (feedback message)
   Interpreter.Script:=Interpreter.Script+CRLF+'@system @async @tty listen 0';
   RunScript;
  finally
   Interpreter.Free;
  end;
  //
  // Done!
  //
  Writeln(Format('Execution time %1.3f sec',[(mSecNow-ms)*1e-3]));
 except
  on E:Exception do begin
   Result:=-1;
   if WindowExists(twin) then CurvesCount[twin]:=0; 
   Echo(E.Message);
   if UsesBlaster then Voice('EXCEPTION');
   Error(E.Message);
  end;
 end;
end;

//////////////////////////////////////////////////
{$I _crw_plugin_exports} // Exports CRWDAQ_PLUGIN.
//////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID; //
//////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
