'****************************************************************
'** Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru      **
'** Under MIT License, see https://opensource.org/licenses/MIT **
'****************************************************************
'** Command line tool to acrivate application by title         **
'** Usage: Shell-AppActivate "Application Title"               **
'****************************************************************

Option Explicit

'******* System objects
'**********************

dim StdIn  : set StdIn  = WScript.StdIn
dim StdOut : set StdOut = WScript.StdOut
dim StdErr : set StdErr = WScript.StdErr
dim Shell  : set Shell  = WScript.CreateObject("WScript.Shell")
dim FSO    : set FSO    = WScript.CreateObject("Scripting.FileSystemObject")

'******* Utility routines
'************************

sub WriteLn(line)
 StdOut.WriteLine(line)
end sub

sub PrintStdErr(line)
 StdErr.WriteLine(line)
end sub

sub Abort(ExitCode,ErrorMessage)
 PrintStdErr(ErrorMessage)
 WScript.Quit(ExitCode)
end sub

sub Assert(Condition,ExitCode,ErrorMessage)
 if not cbool(Condition) then
  call Abort(ExitCode,ErrorMessage)
 end if
end sub

function GetBaseName(path)
 GetBaseName = FSO.GetBaseName(path)
end function

function ShellAppActivate(Title)
 if (Title<>"") then
  ShellAppActivate = Shell.AppActivate(Title)
 else
  ShellAppActivate = false
 end if
end function

'******* Print Version
'*********************

sub PrintVersion(name)
 WriteLn(ucase(name)+" version 1.0")
end sub

'******* Print help screen
'*************************

sub PrintHelp(name)
 PrintVersion(name)
 WriteLn("Command to activate application by window Title.")
 WriteLn("Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru")
 WriteLn("Under MIT License, see https://opensource.org/licenses/MIT")
 WriteLn("Help on "+ucase(name)+":")
 WriteLn(" =================> Syntax:")
 WriteLn("  "+ucase(name)+" [Arguments] [Options] ")
 WriteLn(" =================> Options:")
 WriteLn("   --            => options ended, next is params")
 WriteLn("   --version     => print program version and exit")
 WriteLn("   -h,--help     => print this help screen and exit")
 WriteLn(" =================> Arguments:")
 WriteLn("   Title         => Window title to be activated")
 WriteLn(" =================> Exit Code:")
 WriteLn("   0             => specified application activated")
 WriteLn("   1             => specified application is not found ")
 WriteLn("   2             => some error found (bad arguments/options)")
 WriteLn("   else          => some error found (internal script error)")
 WriteLn(" =================> Note:")
 WriteLn("   First search application with given window Title then try")
 WriteLn("   to find window which starting or ending with given Title.")
 WriteLn("   When search a Title the comparison is not case sensitive.")
 WriteLn("   Only visible (not hidden) windows take part in searching.")
 WriteLn("   If any window(s) was found, activate first found of them.")
 WriteLn(" =================> Examples:")
 WriteLn("  call "+name+" --help")
 WriteLn("  call "+name+" --version")
 WriteLn("  call "+name+" ""cmd.exe""")
 WriteLn("  call "+name+" ""CRW32"" && echo Window ""CRW32"" activated. ")
end sub

'******* Program data and options
'********************************

dim argnum  : argnum  = 0
dim Title   : Title   = ""

'******* Handle arguments
'************************

sub HandleArgs(arg)
 argnum=argnum+1
 select case argnum
  case 1    : Title=arg
  case else : call Abort(2,"Error: excess argument found. See --help.")
 end select
end sub

'******* Command line parsing
'****************************

dim i,arg,opt,isopt : arg="" : opt="" : isopt=true
for i=0 to WScript.Arguments.Count-1
 arg=WScript.Arguments(i)
 if (left(arg,1)="-") and isopt and (opt="") then
  select case arg
   case "--"                    : isopt=false ' end of options, interpret next arguments as params
   case "--version"             : PrintVersion(GetBaseName(WScript.ScriptName)) : WScript.Quit(0)
   case "-h","--help"           : PrintHelp(GetBaseName(WScript.ScriptName))    : WScript.Quit(0)
   case else                    : call Abort(2,"Error: unknown option "+arg+". See --help.")
  end select
 else
  select case opt
   case ""                      : call HandleArgs(arg)
   case else                    : call Abort(2,"Error: unknown option "+opt+". See --help.")
  end select
  opt=""
 end if
next

'******* Task execution
'**********************

if (WScript.Arguments.Count=0) then : PrintHelp(GetBaseName(WScript.ScriptName)) : WScript.Quit(0) : end if

call Assert(Title<>"",2,"Error: Application window Title is not specified. See --help.")

call Assert(ShellAppActivate(Title),1,"Window """+Title+""" not found.")

WScript.Quit(0)

'******* Done
'************
