////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2022 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpObjReg; // ObjectRegistry and TMasterObject.

interface

uses  dpCmdArgs,dpSystem,dpSysUtils;

 {
 TMasterObject is class of object which self-registered in ObjectRegistry.
 obj:=TMasterObject.Create;
 ref:=obj.ref; // reference
 // ... usage of reference ...
 obj:=ObjectRegistry.GetItem(ref);
 }
type
 TMasterObject = class(TObject)
 private
  myRef      : Integer;
  function    GetOk:Boolean;
  function    GetRef:Integer;
 public
  constructor Create;
  destructor  Destroy;
 public
  property    Ok              : Boolean        read  GetOk;
  property    Ref             : Integer        read  GetRef;
 end;

var
 BornKillBalance : Integer = 0;

 {
 *********************************************************
 TObjectRegistry is a storage for TMasterObject references
 *********************************************************
 }
const
 ObjectRegistryOffset = $100000;
type
 TObjectRegistry = class(TObject)
 private
  myTable : array of TObject;
  myStack : array of Integer;
  myCount : Integer;
  mySpace : Integer;
  myCapac : Integer;
  function GetCount:Integer;
  function GetSpace:Integer;
  function GetCapacity:Integer;
 public
  constructor Create;
  destructor  Destroy;
  function    InitRef(Obj:TObject):Integer;
  function    FreeRef(Ref:Integer):Boolean;
 public
  property Count            : Integer read GetCount;
  property Space            : Integer read GetSpace;
  property Capacity         : Integer read GetCapacity;
 public
  function GetItems(Ref:Integer):TObject;
 end;

 {
 *****************************************************
 ObjectRegistry contains ALL TMasterObject references.
 So you may use Object.Ref as unique handle of Object.
  Ref:=TSomeObject.Create.Ref;
  (ObjectRegistry[Ref] as TSomeObject).DoSomething;
 *****************************************************
 }
function ObjectRegistry:TObjectRegistry;

implementation
 {
 ****************************
 TMasterObject implementation
 ****************************
 }
constructor TMasterObject.Create;
begin
 inherited Create;
 Inc(BornKillBalance);
 myRef:=ObjectRegistry.InitRef(Self);
end;

procedure TMasterObject.Destroy;
begin
 ObjectRegistry.FreeRef(myRef);
 Dec(BornKillBalance);
 inherited Destroy;
end;

function TMasterObject.GetOk:Boolean;
begin
 if (Self<>nil) then Result:=true else Result:=false;
end;

function TMasterObject.GetRef:Integer;
begin
 if (Self<>nil) then Result:=myRef else Result:=0;
end;

 {
 ******************************
 TObjectRegistry implementation
 ******************************
 }
var
 TheObjectRegistry : TObjectRegistry = nil;

function ObjectRegistry:TObjectRegistry;
begin
 Result:=TheObjectRegistry;
end;

constructor TObjectRegistry.Create;
begin
 inherited Create;
 SetArrayLength(myTable,0);
 SetArrayLength(myStack,0);
 myCount:=0;
 mySpace:=0;
 myCapac:=0;
end;

destructor TObjectRegistry.Destroy;
begin
 SetArrayLength(myTable,0);
 SetArrayLength(myStack,0);
 myCapac:=0;
 myCount:=0;
 mySpace:=0;
 inherited Destroy;
end;

function TObjectRegistry.GetCount:Integer;
begin
 if (Self<>nil) then Result:=myCount else Result:=0;
end;

function TObjectRegistry.GetSpace:Integer;
begin
 if (Self<>nil) then Result:=mySpace else Result:=0;
end;

function TObjectRegistry.GetCapacity:Integer;
begin
 if (Self<>nil) then Result:=myCapac else Result:=0;
end;

function TObjectRegistry.GetItems(Ref:Integer):TObject;
begin
 Result:=nil;
 if (Self<>nil) then begin
  Dec(Ref,ObjectRegistryOffset);
  if (Ref>=0) and (Ref<myCapac) then Result:=myTable[Ref];
 end;
end;

function TObjectRegistry.InitRef(Obj:TObject):Integer;
var i:Integer;
begin
 Result:=0;
 if (Obj<>nil) then
 if (Self<>nil) then
 try
  if mySpace=0 then begin
   if myCapac=0 then begin
    SetArrayLength(myTable,1);
    SetArrayLength(myStack,1);
    myTable[0]:=nil;
    myStack[0]:=0;
    Inc(myCapac);
    Inc(mySpace);
   end else begin
    SetArrayLength(myTable,2*myCapac);
    SetArrayLength(myStack,2*myCapac);
    for i:=myCapac to 2*myCapac-1 do begin
     myStack[mySpace]:=i;
     myTable[i]:=nil;
     Inc(mySpace);
    end;
    Inc(myCapac,myCapac);
   end;
  end;
  i:=myStack[mySpace-1];
  if (i<0) or (i>=myCapac) or (myTable[i]<>nil)
  then RaiseException('InitRef Error!');
  myTable[i]:=Obj;
  Dec(mySpace);
  Inc(myCount);
  if GetItems(i+ObjectRegistryOffset)<>Obj
  then RaiseException('InitRef Error!');
  Result:=i+ObjectRegistryOffset;
 except
  on E:Exception do BugReport(E,Self,'InitRef');
 end;
end;

function TObjectRegistry.FreeRef(Ref:Integer):Boolean;
begin
 Result:=false;
 if Ref<>0 then
 if (Self<>nil) then
 try
  Dec(Ref,ObjectRegistryOffset);
  if (Ref>=0) and (Ref<myCapac) then
  if (myTable[Ref]<>nil) then begin
   myTable[Ref]:=nil;
   Dec(myCount);
   if mySpace>=myCapac then RaiseException('FreeRef Error!');
   myStack[mySpace]:=Ref;
   Inc(mySpace);
   Result:=true;
  end;
 except
  on E:Exception do BugReport(E,Self,'FreeRef');
 end;
end;

initialization

 TheObjectRegistry:=TObjectRegistry.Create;

end.
