////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020-2023 Alexey Kuryakin kouriakine@mail.ru - LGPL license. //
////////////////////////////////////////////////////////////////////////////////

unit dpLinux; // Diesel Pascal Linux style constants and functions.

interface

uses dpCmdArgs,dpSystem,dpSysUtils,dpTask;

////////////////////////////////////////////////////////////////////////////////
// Default Options for execute_command_as_text function:
// Unix:    'PATH=$PATH|HomeDir=$TMPDIR|Display=SW_HIDE|TimeOut=1000|BuffKb=64'
// Windows: 'PATH=$PATH|HomeDir=$TEMP|Display=SW_HIDE|TimeOut=1000|BuffKb=64'
////////////////////////////////////////////////////////////////////////////////
function execute_command_as_text_defaults(HomeDir:String='$TMPDIR';
 Display:String='SW_HIDE'; TimeOut:Integer=1000; BuffKb:Integer=64):String;

////////////////////////////////////////////////////////////////////////////////
// Execute command (cmd) and return it`s StdOut (Result), process ID (pid), exit
// code (ExitCode). The Options is '|' - delimited string of NAME=Value options.
// PATH=$PATH set search path to find command executable;   HomeDir=/tmp specify
// directory to start in; Display=SW_HIDE contain display mode SW_HIDE/SHOW/...;
// TimeOut=1000 specify timeout milliseconds; BuffKb=64 specify buffer size, Kb;
// send is string to send to StdIn of executing command.
////////////////////////////////////////////////////////////////////////////////
function execute_command_as_text(cmd:String; var pid,ExitCode:Integer;
                              Options:String='Defaults'; send:String=''):String;

////////////////////////////////////////////////////////////////////////////////
// Execute wmctrl command with given arguments/options. Required wmctrl package.
// execute_wmctrl('-l')                - return desktop window list (defaults).
// execute_wmctrl('-l -p')             - return desktop window list (with PID).
// execute_wmctrl('-l -x -p')          - desktop window list (with Class, PID).
// execute_wmctrl('-a "Window Title"') - activate a window with specified title.
// execute_wmctrl('-a "Win Title" -F') - activate window by title, strict match.
////////////////////////////////////////////////////////////////////////////////
// execute_wmctrl('-l') - return text lines:
//  HWND        D HOST  TITLE
//  0x04400065  1 y510p Double Commander
//  0x05000048  2 y510p Object Inspector
//  0x050003b6  0 y510p CodeEditor
//  0x050003bb  3 y510p MainForm
// where
//  HWND=windowHandle; D=desktopNumber; HOST=hostName; TITLE=windowTitle
////////////////////////////////////////////////////////////////////////////////
function execute_wmctrl(arg:String; Options:String='Defaults'):String;

////////////////////////////////////////////////////////////////////////////////
// Search file (name) in PATH - just like 'which' shell command does.
////////////////////////////////////////////////////////////////////////////////
function file_which(name:String):String;

////////////////////////////////////////////////////////////////////////////////
// Read special file /proc/pid/name - properties of process pid as long string.
////////////////////////////////////////////////////////////////////////////////
function read_proc_pid_file(pid:Integer; name:String):String;

////////////////////////////////////////////////////////////////////////////////
// Get list of modules loaded by process(pid) from /proc/pid/maps.
////////////////////////////////////////////////////////////////////////////////
function read_proc_pid_modules(pid:Integer):String;

implementation

function execute_command_as_text_defaults(HomeDir:String='$TMPDIR';
 Display:String='SW_HIDE'; TimeOut:Integer=1000; BuffKb:Integer=64):String;
begin
 Result:='';
 if Linux or MacOS then begin
  HomeDir:=StringReplace(HomeDir,'$TEMP','$TMPDIR',rfReplaceAll+rfIgnoreCase);
  Result:='PATH=$PATH'
        +'|HomeDir='+HomeDir
        +'|Display='+Trim(Display)
        +'|TimeOut='+IntToStr(TimeOut)
        +'|BuffKb='+IntToStr(BuffKb);
 end else
 if Windows then begin
  HomeDir:=StringReplace(HomeDir,'$TMPDIR','$TEMP',rfReplaceAll+rfIgnoreCase);
  Result:='PATH=$PATH'
        +'|HomeDir='+HomeDir
        +'|Display='+Trim(Display)
        +'|TimeOut='+IntToStr(TimeOut)
        +'|BuffKb='+IntToStr(BuffKb);
 end;
end;

function execute_command_as_text(cmd:String; var pid,ExitCode:Integer;
 Options:String='Defaults'; send:String=''):String;
var exe,arg,PATH,HomeDir,Display:String; tid,TimeOut,BuffKb:Integer;
var OptList:TStringList;
begin
 Result:=''; pid:=0; ExitCode:=-1; if (cmd='') then Exit;
 try
  OptList:=TStringList.Create; tid:=0;
  try
   if SameText(Options,'Defaults')
   then Options:=execute_command_as_text_defaults;
   exe:=ExtractWord(1,cmd,' '); if IsEmptyStr(exe) then Exit;
   OptList.Text:=StringReplace(Options,'|',LineEnding,rfReplaceAll);
   PATH:=Trim(ExpEnv(Trim(OptList.Values['PATH'])));
   if (PATH<>'') then begin
    exe:=Trim(FileSearch(exe,PATH,false));
    if IsEmptyStr(exe) or not FileExists(exe) then Exit;
   end;
   arg:=SkipWords(1,cmd,' ');
   cmd:=Trim(exe+' '+arg);
   tid:=task_init(cmd);
   if (tid<>0) then begin
    Display:=Trim(OptList.Values['Display']);
    HomeDir:=Trim(ExpEnv(OptList.Values['HomeDir']));
    BuffKb:=StrToIntDef(Trim(OptList.Values['BuffKb']),0);
    TimeOut:=StrToIntDef(Trim(OptList.Values['TimeOut']),0);
    if (Display<>'') then task_ctrl(tid,'Display='+Display);
    if (BuffKb>0) then task_ctrl(tid,'StdInPipeSize='+IntToStr(1024*BuffKb));
    if (BuffKb>0) then task_ctrl(tid,'StdOutPipeSize='+IntToStr(1024*BuffKb));
    if (HomeDir<>'') and DirectoryExists(HomeDir) then task_ctrl(tid,'HomeDir='+HomeDir);
    if task_run(tid) then begin
     pid:=task_pid(tid);
     if (send<>'') then task_send(tid,send);
     if (TimeOut>0) then task_wait(tid,TimeOut);
     if (task_rxcount(tid)>0) then Result:=AdjustLineBreaks(task_recv(tid,MaxInt));
     ExitCode:=task_result(tid);
    end;
    if (TimeOut>0) then if task_wait(tid,0) then task_kill(tid,1,0,0);
   end;
  finally
   if (OptList<>nil) then OptList.Free;
   if (tid<>0) then task_free(tid);
  end;
 except
  on E:Exception do BugReport(E,Application,'execute_command_as_text');
 end;
end;

function execute_wmctrl(arg:String; Options:String='Defaults'):String;
var cmd:String; pid,ExitCode:Integer;
begin
 Result:='';
 if not Windows then
 try
  cmd:=Trim('wmctrl '+Trim(arg));
  Result:=execute_command_as_text(cmd,pid,ExitCode,Options);
 except
  on E:Exception do BugReport(E,Application,'execute_wmctrl');
 end;
end;

function file_which(name:String):String;
begin
 if (name='') then Result:='' else
 Result:=FileSearch(name,GetEnv('PATH'),false);
end;

function read_proc_pid_file(pid:Integer; name:String):String;
var fname:String;
begin
 Result:='';
 if IsUnix then name:=Trim(name) else Exit;
 if (name<>'') then
 try
  if (pid=0) then pid:=GetProcessId;
  fname:='/proc/'+IntToStr(pid)+'/'+name;
  if FileExists(fname) then Result:=FileReadAsText(fname,ioModeAsIs);
 except
  on E:Exception do BugReport(E,Application,'read_proc_pid_file');
 end;
end;

function read_proc_pid_modules(pid:Integer):String;
var list,temp:TStringList; i:Integer; s,delims:String;
begin
 Result:='';
 try
  if IsWindows then begin
   Result:=GetListOfModules(pid);
  end else
  if IsUnix then begin
   delims:=LineEnding+' '+Chr(9);
   list:=TStringList.Create;
   temp:=TStringList.Create;
   try
    temp.Text:=read_proc_pid_file(pid,'maps');
    for i:=0 to temp.Count-1 do begin
     s:=Trim(SkipWords(5,temp.Strings[i],delims));
     if (StrFetch(s,1)<>'/') then continue;
     if (list.IndexOf(s)>=0) then continue;
     if not FileExists(s) then continue;
     list.Add(s);
    end;
    Result:=list.Text;
   finally
    list.Free;
    temp.Free;
   end;
  end;
 except
  on E:Exception do BugReport(E,Application,'read_proc_pid_modules');
 end;
end;

end.
