////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpEasyIpc; // Diesel Pascal routines for EasyIpc.

interface

uses dpCmdArgs;

////////////////////////////////////////////////////////////////////////////////
// Base EasyIpc functions exported with DieselPascal API (supported by default).
////////////////////////////////////////////////////////////////////////////////
//function EasyIpc_Init(const PipeName,Options:String):THandle;
//function EasyIpc_Free(hIpc:THandle):Boolean;
//function EasyIpc_Poll(hIpc:THandle):Boolean;
//function EasyIpc_Send(hIpc:THandle; const TextLines:String):Boolean;
//function EasyIpc_Recv(hIpc:THandle; Count:Integer):String;
//function EasyIpc_Ctrl(hIpc:THandle; const Request:String):String;
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Addon useful EasyIpc functions which is not exported with DieselPascal API.
////////////////////////////////////////////////////////////////////////////////
// Note: Rx is Receiver, Tx is Transmitter, hIpc is EasyIpc object reference.
////////////////////////////////////////////////////////////////////////////////
function EasyIpc_Valid(hIpc:THandle):Boolean;       // Is hIpc reference valid?
function EasyIpc_Connected(hIpc:THandle):Boolean;   // hIpc is connected?
function EasyIpc_IsServer(hIpc:THandle):Boolean;    // hIpc is server?
function EasyIpc_IsClient(hIpc:THandle):Boolean;    // hIpc is client?
function EasyIpc_FileName(hIpc:THandle):String;     // Like \\.\pipe\demo
function EasyIpc_PipeName(hIpc:THandle):String;     // Like .\demo or demo
function EasyIpc_HostName(hIpc:THandle):String;     // Like . or empty
function EasyIpc_BaseName(hIpc:THandle):String;     // Like demo
function EasyIpc_Handle(hIpc:THandle):THandle;      // File handle
function EasyIpc_TimeOut(hIpc:THandle):Integer;     // Timeout, ms
function EasyIpc_RxBuffSize(hIpc:THandle):Integer;  // Rx buffer size
function EasyIpc_TxBuffSize(hIpc:THandle):Integer;  // Tx buffer size
function EasyIpc_RxLost(hIpc:THandle):Int64;        // Rx lost bytes counter
function EasyIpc_TxLost(hIpc:THandle):Int64;        // Tx lost bytes counter
function EasyIpc_RxTotal(hIpc:THandle):Int64;       // Rx total traffic
function EasyIpc_TxTotal(hIpc:THandle):Int64;       // Tx total traffic
function EasyIpc_RxLength(hIpc:THandle):Integer;    // Rx FIFO data length
function EasyIpc_TxLength(hIpc:THandle):Integer;    // Tx FIFO data length
function EasyIpc_RxFifoLimit(hIpc:THandle):Integer; // Rx FIFO size limit
function EasyIpc_TxFifoLimit(hIpc:THandle):Integer; // Tx FIFO size limit
function EasyIpc_LogsHistory(hIpc:THandle):Integer; // Max history stored in log
function EasyIpc_LogsCount(hIpc:THandle):Integer;   // Log history, lines
function EasyIpc_LogsTextMove(hIpc:THandle):String; // Copy logs and clear
function EasyIpc_LogsTextCopy(hIpc:THandle):String; // Copy logs and keep
function EasyIpc_Clear(hIpc:THandle; What:String):Boolean; // Clear *=All

// Initialize EasyIpc by command line options.
// hIpc:=EasyIpc_InitFromCmdArgs('--easyipc');
// call demo.lm9 --easyipc=.\test,TimeOut=0,LogsHistory=100
function EasyIpc_InitFromCmdArgs(Opt:String):THandle;
function EasyIpc_PipeNameFromCmdArgs(Opt:String):String;
function EasyIpc_OptionsFromCmdArgs(Opt:String):String;

implementation

function EasyIpc_Valid(hIpc:THandle):Boolean;
begin
 Result:=false; if (hIpc=0) then Exit;
 Result:=(EasyIpc_Ctrl(hIpc,'Connected')<>'');
end;

function EasyIpc_Connected(hIpc:THandle):Boolean;
begin
 Result:=false; if (hIpc=0) then Exit;
 Result:=(StrToIntDef(EasyIpc_Ctrl(hIpc,'Connected'),0)>0);
end;

function EasyIpc_IsServer(hIpc:THandle):Boolean;
begin
 Result:=false; if (hIpc=0) then Exit;
 Result:=(StrToIntDef(EasyIpc_Ctrl(hIpc,'IsServer'),0)>0);
end;

function EasyIpc_IsClient(hIpc:THandle):Boolean;
begin
 Result:=false; if (hIpc=0) then Exit;
 Result:=(StrToIntDef(EasyIpc_Ctrl(hIpc,'IsClient'),0)>0);
end;

function EasyIpc_FileName(hIpc:THandle):String;
begin
 Result:=''; if (hIpc=0) then Exit;
 Result:=EasyIpc_Ctrl(hIpc,'FileName');
end;

function EasyIpc_PipeName(hIpc:THandle):String;
begin
 Result:=''; if (hIpc=0) then Exit;
 Result:=EasyIpc_Ctrl(hIpc,'PipeName');
end;

function EasyIpc_HostName(hIpc:THandle):String;
begin
 Result:=''; if (hIpc=0) then Exit;
 Result:=EasyIpc_Ctrl(hIpc,'HostName');
end;

function EasyIpc_BaseName(hIpc:THandle):String;
begin
 Result:=''; if (hIpc=0) then Exit;
 Result:=EasyIpc_Ctrl(hIpc,'BaseName');
end;

function EasyIpc_Handle(hIpc:THandle):THandle;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'Handle'),0);
end;

function EasyIpc_TimeOut(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'TimeOut'),0);
end;

function EasyIpc_RxBuffSize(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'RxBuffSize'),0);
end;

function EasyIpc_TxBuffSize(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'TxBuffSize'),0);
end;

function EasyIpc_RxLost(hIpc:THandle):Int64;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToInt64Def(EasyIpc_Ctrl(hIpc,'RxLost'),0);
end;

function EasyIpc_TxLost(hIpc:THandle):Int64;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToInt64Def(EasyIpc_Ctrl(hIpc,'TxLost'),0);
end;

function EasyIpc_RxTotal(hIpc:THandle):Int64;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToInt64Def(EasyIpc_Ctrl(hIpc,'RxTotal'),0);
end;

function EasyIpc_TxTotal(hIpc:THandle):Int64;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToInt64Def(EasyIpc_Ctrl(hIpc,'TxTotal'),0);
end;

function EasyIpc_RxLength(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'RxLength'),0);
end;

function EasyIpc_TxLength(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'TxLength'),0);
end;

function EasyIpc_RxFifoLimit(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'RxFifoLimit'),0);
end;

function EasyIpc_TxFifoLimit(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'TxFifoLimit'),0);
end;

function EasyIpc_LogsHistory(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'LogsHistory'),0);
end;

function EasyIpc_LogsCount(hIpc:THandle):Integer;
begin
 Result:=0; if (hIpc=0) then Exit;
 Result:=StrToIntDef(EasyIpc_Ctrl(hIpc,'LogsCount'),0);
end;

function EasyIpc_LogsTextMove(hIpc:THandle):String;
begin
 Result:=''; if (hIpc=0) then Exit;
 Result:=EasyIpc_Ctrl(hIpc,'LogsTextMove');
end;

function EasyIpc_LogsTextCopy(hIpc:THandle):String;
begin
 Result:=''; if (hIpc=0) then Exit;
 Result:=EasyIpc_Ctrl(hIpc,'LogsTextCopy');
end;

function EasyIpc_Clear(hIpc:THandle; What:String):Boolean;
begin
 Result:=false; if (hIpc=0) then Exit;
 if SameText(What,'*') or SameText(What,'All')
 then What:='RxLost,TxLost,RxFifo,TxFifo,TxTotal,TxTotal,Logs';
 EasyIpc_Ctrl(hIpc,'Clear='+What);
 Result:=true;
end;

function EasyIpc_InitFromCmdArgs(Opt:String):THandle;
var PipeName,Options:String;
begin
 Result:=0;
 if (Opt='') then Exit;
 if not CmdArgs.HasOption(Opt) then Exit;
 PipeName:=EasyIpc_PipeNameFromCmdArgs(Opt);
 Options:=EasyIpc_OptionsFromCmdArgs(Opt);
 Result:=EasyIpc_Init(PipeName,Options);
end;

function EasyIpc_PipeNameFromCmdArgs(Opt:String):String;
var Params:String;
begin
 Result:='';
 if (Opt='') then Exit;
 if not CmdArgs.HasOption(Opt) then Exit;
 Params:=CmdArgs.GetOptionValue(Opt);
 Params:=StringReplace(Params,',',LineEnding,rfReplaceAll);
 Params:=StringReplace(Params,';',LineEnding,rfReplaceAll);
 Result:=ExtractWord(1,Params,LineEnding);
end;

function EasyIpc_OptionsFromCmdArgs(Opt:String):String;
var Params:String;
begin
 Result:='';
 if (Opt='') then Exit;
 if not CmdArgs.HasOption(Opt) then Exit;
 Params:=CmdArgs.GetOptionValue(Opt);
 Params:=StringReplace(Params,',',LineEnding,rfReplaceAll);
 Params:=StringReplace(Params,';',LineEnding,rfReplaceAll);
 Result:=SkipWords(1,Params,LineEnding);
end;

end.

