////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020 Alexey Kuryakin kouriakine@mail.ru under LGPL license.  //
////////////////////////////////////////////////////////////////////////////////

unit dpStdIn;

interface

uses dpSysUtils;

////////////////////////////////////////////////////////////////////////////////
// Virtual StdIn console routines. Simulation of DaqPascal - style console.   //
////////////////////////////////////////////////////////////////////////////////

 {
 Have got a command?
 Check if input line (Data) contains a command.
 Command looks like "@cmd arg" or "@cmd=arg".
 Then extract cmd and arg.
 }
function GotCommand(var Data,cmd,arg:String):Boolean;

 {
 Have got a command?
 Check if input line (Data) contains a command.
 Command looks like "@cmd arg" or "@cmd=arg".
 Then extract cmd,arg, and find command identifier cid from StdIn_CmndRegList.
 }
function GotCommandId(var Data,cmd,arg:String; var cid:Integer):Boolean;

 {
 Find positive string ID by string (str) or return -1 if one not found.
 }
function FindStdInStrId(str:String):Integer;

 {
 Register StdIn command/string.
 Command must start from @ char.
 Return positive identifier or 0 on error.
 }
function RegisterStdInCmd(cmd:String; help:String):Integer;
function RegisterStdInStr(str:String; help:String):Integer;

var
 StdIn_CmndCount   : Integer     = 0;     // StdIn command incomes count.
 StdIn_CmndRegList : TStringList = nil;   // StdIn command registry list.

implementation

const
 Delims = ' ,='+#9+#10+#13;

procedure StdIn_CmndRegList_Init;
begin
 StdIn_CmndRegList:=TStringList.Create;
 StdIn_CmndRegList.Sorted:=true;
end;

function GotCommand(var Data,cmd,arg:String):Boolean;
begin
 if (strFetch(Data,1)='@') then begin
  Result:=True;
  cmd:=ExtractWord(1,Data,Delims);
  arg:=SkipWords(1,Data,Delims);
  Inc(StdIn_CmndCount);
 end else begin
  Result:=False;
  cmd:='';
  arg:='';
 end;
end;

function GotCommandId(var Data,cmd,arg:String; var cid:Integer):Boolean;
begin
 if (strFetch(Data,1)='@') then begin
  Result:=True;
  cmd:=ExtractWord(1,Data,Delims);
  arg:=SkipWords(1,Data,Delims);
  cid:=FindStdInStrId(cmd);
  if cid=0 then cid:=-1;
  Inc(StdIn_CmndCount);
 end else begin
  Result:=False;
  cmd:='';
  arg:='';
  cid:=-1;
 end;
end;

function FindStdInStrId(str:String):Integer;
var i:Integer;
begin
 Result:=-1;
 if (Length(str)>0) then
 if (StdIn_CmndRegList<>nil) then begin
  i:=StdIn_CmndRegList.IndexOf(str);
  if (i>=0) then Result:=Integer(StdIn_CmndRegList.Objects[i]);
 end;
end;

function RegisterStdInCmd(cmd:String; help:String):Integer;
var data,arg:String;
begin
 data:=cmd;
 if GotCommand(data,cmd,arg)
 then Result:=RegisterStdInStr(cmd,help)
 else Result:=0;
end;

function RegisterStdInStr(str:String; help:String):Integer;
var sid:Integer;
begin
 sid:=0; str:=Trim(str);
 if (str<>'') then begin
  if (StdIn_CmndRegList=nil) then StdIn_CmndRegList_Init;
  sid:=FindStdInStrId(str);
  if (sid<=0) then begin
   sid:=StdIn_CmndRegList.Count+1;
   StdIn_CmndRegList.AddObject(str,TObject(sid))
  end;
  sid:=FindStdInStrId(str);
  if sid<0 then sid:=0;
 end;
 Result:=sid;
end;


initialization

 StdIn_CmndRegList_Init;

end.
