#!/bin/bash

###########################################################
## Copyright (c) 2019-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## DIM Configurator tool for CRW-DAQ.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

let abort_notify_uses=0;

###############
# print version
###############
function print_version(){
 local name="$1";
 echo "$name version 1.0";
};

###################
# print help screen
###################
function print_help(){
 local name="$1";
 print_version $name;
 echo "xterm configuraation tool for daqgroup.";
 echo "Copyright (c) 2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "Help on $name:";
 echo " ====================> Syntax:";
 echo "  $name [Options]";
 echo " ====================> Options:";
 echo "   --               => options ending, next is params";
 echo "   -h,--help        => print this help screen and exit";
 echo "   --version        => print program version and exit";
 echo "   -l,--list        => list current X config file";
 echo "   -d,--drop        => drop xterm changes made for daqgroup";
 echo "   -u,--update      => update xterm config of current user for daqgroup";
 echo "   -a,--apply       => apply current X configuration of current user";
 echo " ====================> Arguments:";
 echo "   Not uses, should be empty.";
 echo " ====================> Exit Code:";
 echo "  Return 0 on success, otherwise return error code";
 echo " ====================> Examples:";
 echo "  $name --help";
 echo "  $name --version";
 echo "  $name -u -l -a";
 echo "  $name -d";
};

##########################
# program data and options
##########################
readonly xconf="$HOME/.Xresources";            # current user X config
readonly nconf="$scriptHOME/$scriptbase.conf"; # new X config for daqgroup
readonly bsign="! XTerm configuration for DaqGroup users: BEGIN";
readonly esign="! XTerm configuration for DaqGroup users: END";

##################
# Program routines
##################

#####################
# list X config file
#####################
function list_xconf(){
 if [ -e "$xconf" ]; then
  cat $xconf;
 else
  fatal 1 "Error: not found $xconf";
 fi;
};

#####################
# apply X config file
#####################
function apply_xconf(){
 if [ -e "$xconf" ]; then
  if xrdb $xconf; then
   echo "Applied: $xconf";
  else
   colorize_bold echo_to_stderr "Error: could not apply $xconf";
   return 1;
  fi;
 fi;
};

####################################################
# print line number of $1 sign in X config file or 0
####################################################
function line_of_sign(){
 local n=0;
 local sign="$1";
 if [ -e "$xconf" ]; then
  local line="$(list_xconf | grep -n "$1" | sed 's/:.*//' | xargs)";
  if is_number "$line"; then let n=$line; fi;
 fi;
 echo "$n";
};

####################################
# check X config has sign of changes
####################################
function has_sign(){
 if [ -e "$xconf" ]; then
  local nb="$(line_of_sign "$bsign")";
  local ne="$(line_of_sign "$esign")";
  if [ $nb -gt 0 ] && [ $ne -gt $nb ]; then return 0; fi;
 fi;
 return 1;
};

###########################
# drop changes made before.
###########################
function drop_changes(){
 if has_sign; then
  local nb="$(line_of_sign "$bsign")";
  local ne="$(line_of_sign "$esign")";
  if [ $nb -gt 0 ] && [ $ne -gt $nb ]; then
   sed "${nb},${ne}d" -i $xconf;
   # drop empty lines in the end of file
   # https://unix.stackexchange.com/questions/552188/how-to-remove-empty-lines-from-beginning-and-end-of-file
   sed -e :a -e '/^\n*$/{$d;N;ba' -e '}' -i $xconf && echo "" >> $xconf;
  fi;
 fi;
};

###########################
# update changes, i.e. copy
# new X config to X config.
###########################
function update_changes(){
 drop_changes;
 echo ""    >> $xconf;
 cat $nconf >> $xconf;
};

##################
# handle arguments
##################
function handle_args(){
 fatal 1 "Error: unexpected parameter: $1.";
};

################
# main processor
################
function main(){
 ######################
 # command line parsing
 ######################
 if [ $# -lt 1 ]; then
  abort 2 "Error: No arguments. See --help.";
 fi;
 local opt="";
 local isopt=1;
 for i in $(seq $#); do
  local arg="$1"; shift;
  if is_option "$arg" && [ $isopt -eq 1 ] && [ -z "$opt" ]; then
   case $arg in
    --)          isopt=0; ;; # marker of options end
    --version)   print_version $scriptname; exit 0; ;;
    -h|--help)   print_help    $scriptname; exit 0; ;;
    -l|--list)   list_xconf; opt=""; ;;
    -d|--drop)   drop_changes; opt=""; ;;
    -a|--apply)  apply_xconf; opt=""; ;;
    -u|--update) update_changes; opt=""; ;;
    *)           handle_args "$arg"; opt=""; ;;
   esac;
  else
   case $opt in
    '')         handle_args "$arg"; ;;
    *)          abort 1 "Error: unknown option $opt. See --help."; ;;
   esac;
   opt="";
  fi;
 done;
 return 0;
};

main "$@";

##############
## END OF FILE
##############
