////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Application modal services.                                                //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20250902 - Created by A.K.                                                 //
////////////////////////////////////////////////////////////////////////////////

unit _crw_appmodal; // 

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
forms, controls, lmessages, lcltype,
 _crw_alloc, _crw_ef, _crw_fio, _crw_str, _crw_wmctrl,
 _crw_rtc, _crw_polling, _crw_dbglog,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
 TAppModal = class(TMasterObject)
 private
  myLastPoll:Int64;
  myInterval:Integer;
  myPolling:TPolling;
 private
  function  GetPolling:TPolling;
  function  GetInterval:Integer;
  procedure SetInterval(i:Integer);
  function  GetStarted:Boolean;
  procedure SetStarted(s:Boolean);
 public
  constructor Create(aIntervalSeconds:Integer=5);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  property  Polling:TPolling read GetPolling;
  property  Interval:Integer read GetInterval write SetInterval;
  function  PollPulser:Boolean;
  function  ModalLevel:Integer;
  function  FindTopmostModalForm:TCustomForm;
  function  GetModalFormZOrderIndex(Form:TCustomForm):Integer;
  procedure ActivateTopmostModalForm;
  function  ActivateModalWindows:Integer;
 public
  property  Started:Boolean read GetStarted write SetStarted;
  procedure Start;
  procedure Stop;
 end;

function AppModal:TAppModal;

procedure Timer_CheckTopmostModalForm;

implementation

procedure Timer_CheckTopmostModalForm;
begin
 if IsMainThread then
 if AppModal.PollPulser then
 if (AppModal.ModalLevel>0) then begin
  AppModal.ActivateTopmostModalForm;
 end;
end;

procedure Polling_AwakeModal(aPolling:TPolling; var Terminate:Boolean);
var Obj:TObject;
begin
 if Assigned(aPolling) then begin
  Obj:=aPolling.LinkObject;
  if not (Obj is TAppModal) then Exit;
  (Obj as TAppModal).ActivateModalWindows;
 end;
end;

///////////////////////////
// TAppModal implementation
///////////////////////////

constructor TAppModal.Create(aIntervalSeconds:Integer=5);
begin
 inherited Create;
 Interval:=aIntervalSeconds*1000;
 myPolling:=NewPolling(Polling_AwakeModal,1000,tpNormal,False,'AwakeModal');
 myPolling.Master:=@myPolling;
 myPolling.LinkObject:=Self;
end;

destructor TAppModal.Destroy;
begin
 Kill(myPolling);
 inherited Destroy;
end;

procedure TAppModal.AfterConstruction;
begin
 inherited AfterConstruction;
end;

procedure TAppModal.BeforeDestruction;
begin
 Stop;
 inherited BeforeDestruction;
end;

function TAppModal.GetPolling:TPolling;
begin
 if Assigned(Self)
 then Result:=myPolling
 else Result:=nil;
end;

function TAppModal.GetInterval:Integer;
begin
 if Assigned(Self)
 then Result:=myInterval
 else Result:=0;
end;

procedure TAppModal.SetInterval(i:Integer);
begin
 if Assigned(Self)
 then myInterval:=Max(0,i);
end;

function TAppModal.PollPulser:Boolean;
var ms,dt:Int64;
begin
 Result:=False;
 if Assigned(Self) then
 if (myInterval>=1000) then begin
  ms:=IntMsecNow; dt:=ms-myLastPoll;
  if (dt>myInterval) then begin
   myLastPoll:=ms;
   Result:=True;
  end;
 end;
end;

function TAppModal.GetStarted:Boolean;
begin
 Result:=False;
 if Assigned(Self) then begin
  //Result:=SecondActions.HasAction(Timer_CheckTopmostModalForm);
  Result:=Polling.Enabled;
 end;
end;

procedure TAppModal.SetStarted(s:Boolean);
begin
 if Assigned(Self) then begin
  if s then Start else Stop;
 end;
end;

procedure TAppModal.Start;
begin
 if Assigned(Self) then begin
  // SecondActions.Add(Timer_CheckTopmostModalForm);
  Polling.Enabled:=True;
 end;
end;

procedure TAppModal.Stop;
begin
 if Assigned(Self) then begin
  //SecondActions.Remove(Timer_CheckTopmostModalForm);
  Polling.Enabled:=False;
 end;
end;

function TAppModal.ModalLevel:Integer;
begin
 if Assigned(Self) and Assigned(Application)
 then Result:=Application.ModalLevel
 else Result:=0;
end;

function TAppModal.FindTopmostModalForm:TCustomForm;
begin
 if Assigned(Self) and Assigned(Screen)
 then Result:=Screen.GetCurrentModalForm
 else Result:=nil;
end;

function TAppModal.GetModalFormZOrderIndex(Form:TCustomForm):Integer;
begin
 if Assigned(Self) and Assigned(Screen)
 then Result:=Screen.GetCurrentModalFormZIndex
 else Result:=-1;
end;

function TAppModal.ActivateModalWindows:Integer;
var wList,wLine:LongString; NeedActivate:Boolean;
var Wnd:HWND; wsf,wtf:QWord; i:Integer;
begin
 Result:=0;
 //if IsMainThread then
 if Assigned(Self) then
 if (ModalLevel>0) then
 try
  wLine:=''; wsf:=0; wtf:=0;
  wList:=wmctrl.ListWindows(GetCurrentProcessId,'','');
  for i:=1 to WordCount(wList,EolnDelims) do begin
   wLine:=ExtractWord(i,wList,EolnDelims);
   Wnd:=wmctrl.StrToWnd(ExtractWord(1,wLine,ScanSpaces));
   if wmctrl.IsWindow(Wnd) then begin
    wsf:=wmctrl.WindowStateFlags(Wnd);
    wtf:=wmctrl.WindowTypeFlags(Wnd);
    NeedActivate:=False;
    if HasFlags(wsf,WSF_MODAL) then begin
     if HasFlags(wsf,WSF_HIDDEN) then NeedActivate:=True;
    end;
    if NeedActivate then begin
     wmctrl.SetWindowStateFlags(Wnd,WSC_REMOVE,WSF_HIDDEN);
     Inc(Result);
    end;
   end;
  end;
  FakeNop(wsf+wtf);
 except
  on E:Exception do BugReport(E,nil,'FindModalWindow');
 end;
end;

procedure TAppModal.ActivateTopmostModalForm;
var CForm:TCustomForm;
begin
 if Assigned(Self) then
 if (ModalLevel>0) then
 try
  ActivateModalWindows;
  if IsMainThread then begin
   CForm:=FindTopmostModalForm;
   if Assigned(CForm) then begin
    if (CForm.WindowState=wsMinimized)
    or (GetModalFormZOrderIndex(CForm)>0)
    or not CForm.Visible
    then CForm.ShowOnTop;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'ActivateTopmostModalForm');
 end;
end;

//////////////////////////
// AppModal implementation
//////////////////////////
const
 TheAppModal:TAppModal=nil;

function AppModal:TAppModal;
begin
 if not Assigned(TheAppModal) then begin
  TheAppModal:=TAppModal.Create;
  TheAppModal.Master:=@TheAppModal;
 end;
 Result:=TheAppModal;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_appmodal;
begin
 AppModal.Ok;
end;

procedure Free_crw_appmodal;
begin
 Kill(TObject(TheAppModal));
end;

initialization

 Init_crw_appmodal;

finalization

 Free_crw_appmodal;

end.

//////////////
// END OF FILE
//////////////

