#!/bin/bash

##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################

##########################################################
# Utility to find stock icon (png,svg,xpm from the stock).
# References:
# 1) XDG Desktop Menu Specification.
# 2) XDG Base Directory Specification.
# 3) XDG Icon Theme Specification.
# 4) XDG Icon Naming Specification.
# 5) XDG Shared MIME-info Database.
# 6) XDG Desktop Entry Specification.
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

source "$scriptHOME/ftype.in";

function do_version(){
 echo "$scriptname version 1.0";
};

function do_print_help(){
 do_version;
cat<<EOF
$scriptname Copyright (c) 2025 Alexey Kuryakin daqgroup@mail.ru
Utility to find stock icon by short name.
Usage:
 $scriptname [-options] [ico]
Options:
 --version    - print version
 -h,--help    - print help screen
 -l,--list    - print list of Icon Themes
 -s,--size s  - select icons by size (s) 
 -t,--theme t - select icons by theme (t) 
Parameters:
 icon         - short name of icon (png,xpm,svg picture)
Examples:
 $scriptname kate                # find icon 'kate'
 $scriptname -t hicolor kate     # find icon 'kate' of theme 'hicolor'
 $scriptname -s scalable kate    # find icon 'kate' of size 'scalable'
 $scriptname -s 48 kate          # find icon 'kate' of size 48 or 48x48
 $scriptname --list              # print list of Icon Themes
 $scriptname --help              # print help screen
 $scriptname --version           # print version
EOF
};

function list_icon_themes(){
 for fn in $(find -L $(enum_ico_search_dirs) -mindepth 2 -maxdepth 2 -name 'index.theme' | grep -v ' '); do
  local theme="$(basename $(dirname "$fn"))"; if [[ "$theme" = "default" ]]; then continue; fi;
  local dirs="$(unix $readini "$fn" "[Icon Theme]" "Directories")";
  if [[ -z $dirs ]]; then dirs="$(unix $readini "$fn" "[Icon Theme]" "ScaledDirectories")"; fi;
  if [[ -z $dirs ]]; then continue; fi;  
  echo "$theme";
 done;
};

function print_icon_themes(){
 echo "Icon Themes: $(list_icon_themes | sort -u | xargs)";
};

function find_icon_file(){
 local ico="$1";
 local gop_size=".*";
 if [[ -n $opt_size ]]; then
  gop_size=".*/(${opt_size}|${opt_size}x${opt_size})/.*";
 fi;
 local gop_theme=".*";
 if [[ -n $opt_theme ]]; then
  gop_theme=".*/$opt_theme/.*";
 fi;
 if [[ -n $ico ]]; then
  find -L $(enum_ico_search_dirs) -maxdepth $icodepth -type f -and -readable \
  -and '(' -name "$ico.png" -or -name "$ico.svg" -or -name "$ico.xpm" ')' \
  2>/dev/null | grep -P "$gop_size" | grep -P "$gop_theme";
 fi;
};

function find_icon(){
 local line="";
 local ico="$1";
 if [[ -n $ico ]]; then
  local line="$(find_icon_file "$ico")";
 else
  fatal 1 "Error: icon name is not specified.";
 fi;
 if [[ -n $line ]]; then
  echo "$line";
 else
  fatal 1 "Error: missed icon $ico";
 fi;
};

function find_icons(){
 if [[ -z $1 ]]; then
  fatal 1 "Error: no arguments specified. Use -h option for help.";
  return;
 fi;
 while [[ -n $1 ]]; do
  find_icon "$1";
  shift;
 done;
};

######
# MAIN
######

function main(){
 check_readini;
 while [[ -n $1 ]]; do
  case $1 in
   --version)         do_version; return 0; ;;
   -h|-help|--help)   do_print_help; return 0; ;;
   -l|-list|--list)   print_icon_themes; return 0; ;;
   -s|-size|--size)   shift; opt_size="$1"; ;;
   -t|-theme|--theme) shift; opt_theme="$1"; ;;
   -*)                fatal 1 "Error: bad option $1"; ;;
   *)                 break; ;;
  esac;
  shift;
 done;
 if [[ -n $1 ]]; then
  find_icons "$@";
 else
  find_icons "*";
 fi;
};

main "$@";

##############
## END OF FILE
##############
