#!/bin/bash

##########################################################
# Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru
##########################################################

function note(){ return; };

function fatal(){
 local date="$(date +%Y.%m.%d-%H:%M:%S)"; local script="$(basename ${BASH_SOURCE[0]})";
 if which notify-send; then notify-send -t 30000 -i dialog-error "$date: $script" "$1"; fi;
 echo -ne "\n$1\n\n";
 exit 1;
};

note "Script identification.";
note "######################";
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

declare -i nsucc=0;
declare -i nfail=0;

function succed(){ echo "SUCCED: $1"; let nsucc++; };
function failed(){ echo "FAILED: $1"; let nfail++; };

function unpack_xz(){
 if [ -n "$1" ] && [ -n "$2" ]; then
  xz -d -c $1 > $2;
 else
  return 1;
 fi;
};

function test_tar(){
 tar -tf $1 >/dev/null 2>&1;
};

function call(){
 if $1; then
  succed "$1";
  true;
 else
  failed "$1";
  false;
 fi;
};

function build_product(){
 local product="$1"; echo "Release build of $product";
 local xzfile="$(ls -1 *-$product.tar.xz | sort | tail -n 1)";
 if [ -e "$xzfile" ]; then
  xzfile="$(realpath "$xzfile")";
  dir="../../build-makeself/$product";
  if [ -d "$dir" ]; then
   dir="$(realpath "$dir")";
   if pushd $dir; then
    local tarfile="$dir/payload-$product/$product.tar";
    if call "unpack_xz $xzfile $tarfile"; then
     if call "test_tar $tarfile"; then
      call "./build-$product.sh --silent";
     fi;
    fi;
   fi;
   popd;
   echo "";
  else
   1>&2 echo "Error: not found directory for $1";
  fi;
 else
  1>&2 echo "Error: not found archive for $1";
 fi;
};

function build_products(){
 local ask="Hello, $USER.";
 ask="$ask\nDo you want to run the script:";
 ask="$ask\n\n<big><b><span fgcolor='green'>$scriptname</span></b>?</big>";
 ask="$ask\n\n<b><span fgcolor='blue'>Confirm</span>?</b>";
 if zenity --question --text "$ask" --timeout 180 >/dev/null 2>&1; then
  note "Run $scriptname …";
 else
  return 0;
 fi;
 local built_date="$(date +%Y%m%d)";
 local built_time="$(date +%Y%m%d-%H%M%S)";
 echo "";
 echo "daqgroup-suite built $built_time …";
 echo "";
 for product in $*; do
  build_product $product;
 done;
 echo "$(date +%Y.%m.%d-%H:%M:%S) - Done $nsucc operations succed, $nfail failed.";
 echo "";
 if [ -t 0 ]; then
  read -t 180 -p "Press any key to continue …" -n 1;
 else
  local icon="dialog-ok-apply";
  if [ $nfail -gt 0 ]; then icon="dialog-error"; fi;
  local date="$(date +%Y.%m.%d-%H:%M:%S)"; local script="$(basename ${BASH_SOURCE[0]})";
  notify-send -t 300000 -i $icon "$date: $script" "Succed:$nsucc, Failed:$nfail.\nWork is Done.";
 fi;
}

function main(){
 if [ -z "$XZ_OPT" ]; then export XZ_OPT="-9 -T4"; fi;
 local products="daqgroup-crwdaq daqgroup-crwkit daqgroup-crwlib";
 if [ -t 1 ]; then
  build_products $products;
 else
  build_products $products > $scriptHOME/$scriptbase.log;
  unix grun wintail          $scriptHOME/$scriptbase.log;
 fi;
};

main "$@";

##############
## END OF FILE
##############
