////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Text Editor Templates.                                                //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231202 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_texteditortemplates; // Form Text Editor Templates

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 StdCtrls, Buttons, ExtCtrls, ComCtrls, Clipbrd,
 Menus, Grids,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 _crw_alloc, _crw_fio, _crw_str, _crw_eldraw,
 _crw_rtc, _crw_ef, _crw_sect,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormTextEditorTemplates }

  TFormTextEditorTemplates = class(TMasterForm)
    PanelButtons: TPanel;
    PanelTop: TPanel;
    GroupBoxCategory: TGroupBox;
    GroupBoxParameters: TGroupBox;
    PanelBottom: TPanel;
    GroupBoxName: TGroupBox;
    GroupBoxText: TGroupBox;
    ListBoxCategory: TListBox;
    ListBoxName: TListBox;
    StringGridParameters: TStringGrid;
    MemoText: TMemo;
    BitBtnInsert: TBitBtn;
    BitBtnInsertOffs: TBitBtn;
    BitBtnCancel: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ListBoxCategoryClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure ListBoxNameClick(Sender: TObject);
    procedure StringGridParametersExit(Sender: TObject);
    procedure BitBtnInsertClick(Sender: TObject);
    procedure BitBtnInsertOffsClick(Sender: TObject);
    procedure StringGridParametersSelectCell(Sender: TObject; ACol,ARow: Integer; var CanSelect: Boolean);
    procedure StringGridParametersClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private declarations }
    RootDir : LongString;
    procedure UpdateControl(What:Integer);
    function  GetTemplate(Offset:Integer):LongString;
    procedure ClearAll;
  public
    { Public declarations }
  end;

function FormTextEditorTemplatesExecute(const aCaption:LongString; aOffs:Integer=0;
                       ToMouse:Boolean=False; const aParams:LongString=''):LongString;

implementation

{$R *.lfm}

const
 TheCategory : Integer = 0;
 TheName     : Integer = 0;

function FormTextEditorTemplatesExecute(const aCaption:LongString; aOffs:Integer=0;
                       ToMouse:Boolean=False; const aParams:LongString=''):LongString;
const
 TheForm : TFormTextEditorTemplates = nil;
var apFlags:Integer;
begin
 Result:='';
 if CanShowModal(TheForm) then
 try
  if not TheForm.Ok then begin
   Application.CreateForm(TFormTextEditorTemplates, TheForm);
   TheForm.Master:=@TheForm;
  end;
  if TheForm.Ok then begin
   TheForm.Caption:=aCaption;
   apFlags:=TheForm.ApplyParams(aParams);
   if not HasFlags(apFlags,apf_FormPos)
   then
   if ToMouse
   then LocateFormToCenterOfMouse(TheForm)
   else LocateFormToCenterOfScreen(TheForm);
   case mrVoice(TheForm.ShowModal) of
    mrOk:  Result:=TheForm.GetTemplate(0);
    mrYes: Result:=TheForm.GetTemplate(aOffs);
   end;
   TheName:=TheForm.ListBoxName.ItemIndex;
   TheCategory:=TheForm.ListBoxCategory.ItemIndex;
   TheForm.ClearAll;
  end;
 except
  on E:Exception do BugReport(E,nil,'FormTextEditorTemplatesExecute');
 end;
end;

function GetDirList(const ThePath:LongString):LongString;
var
 P          : TText;
 Found      : TSearchRec;
 FindResult : Integer;
begin
 Result:='';
 try
  P:=NewText;
  FindResult:=sysutils.FindFirst(AddPathDelim(ThePath)+AnyFileMask, faAnyFile, Found);
  try
   while FindResult = 0 do begin
    if ( Found.Attr and faDirectory <> 0 ) and
       ( Found.Name <> '.'   ) and
       ( Found.Name <> '..'  ) and
       ( Found.Name <> '...' ) and
       ( Found.Name <> '//'  ) then P.Addln(Trim(Found.Name));
    FindResult:=sysutils.FindNext(Found);
   end;
   Result:=P.Text;
  finally
   sysutils.FindClose(Found);
   Kill(P);
  end;
 except
  on E:Exception do BugReport(E,nil,'GetDirList');
 end;
end;

procedure LookNames(const FileName:LongString; const FileDetails:TSearchRec;
                       SubDirLevel:Integer; var Terminate:Boolean; CustomData:Pointer);
var
 TheName : LongString;
begin
 if SubDirLevel = 0 then
 if not HasFlags(FileDetails.Attr,faDirectory) then begin
  TheName:=ExtractFileName(FileName);
  TFormTextEditorTemplates(CustomData).ListBoxName.Items.Add(Trim(TheName));
 end;
end;

procedure TFormTextEditorTemplates.UpdateControl(What:Integer);
var
 P  : TText;
 i  : Integer;
 r  : Integer;
 c  : Integer;
 sn : LongString;
 sv : LongString;
 function GetCategory:LongString;
 begin
  with ListBoxCategory do
  if InRange(ItemIndex,0,Items.Count-1)
  then Result:=Trim(Items[ItemIndex])
  else Result:='';
 end;
 function GetName:LongString;
 begin
  with ListBoxName do
  if InRange(ItemIndex,0,Items.Count-1)
  then Result:=Trim(Items[ItemIndex])
  else Result:='';
 end;
 function GetFullName:LongString;
 begin
  if IsEmptyStr(RootDir) or IsEmptyStr(GetCategory) or IsEmptyStr(GetName)
  then Result:=''
  else Result:=AddPathDelim(AddPathDelim(RootDir)+GetCategory)+GetName+'.txt';
  if not FileExists(Result) then Result:='';
 end;
begin
 if Ok then
 try
  if HasFlags(What,1) then begin
   if ReadIniFilePath(SysIniFile,SectSystem,'EditorTemplates',HomeDir,RootDir)
   then RootDir:=FExpand(RootDir)
   else RootDir:='';
   if IsEmptyStr(RootDir) or not DirExists(RootDir) then RootDir:='';
  end;
  if HasFlags(What,2) then begin
   ListBoxCategory.Items.Clear;
   ListBoxName.Items.Clear;
   for r:=0 to StringGridParameters.RowCount-1 do
   for c:=0 to StringGridParameters.ColCount-1 do StringGridParameters.Cells[c,r]:='';
   MemoText.Lines.Clear;
   if IsNonEmptyStr(RootDir) then
   if DirExists(RootDir) then ListBoxCategory.Items.Text:=GetDirList(RootDir);
   if ListBoxCategory.Items.Count>0 then ListBoxCategory.ItemIndex:=0;
  end;
  if HasFlags(What,4) then begin
   ListBoxName.Items.Clear;
   for r:=0 to StringGridParameters.RowCount-1 do
   for c:=0 to StringGridParameters.ColCount-1 do StringGridParameters.Cells[c,r]:='';
   MemoText.Lines.Clear;
   if IsNonEmptyStr(RootDir) then
   if IsNonEmptyStr(GetCategory) then
   if DirExists(AddPathDelim(RootDir)+GetCategory) then begin
    ForEachFile(AddPathDelim(RootDir)+GetCategory,'*.txt',LookNames,0,Self);
    if ListBoxName.Items.Count>0 then ListBoxName.ItemIndex:=0;
   end;
  end;
  if HasFlags(What,8) then begin
   for r:=0 to StringGridParameters.RowCount-1 do
   for c:=0 to StringGridParameters.ColCount-1 do StringGridParameters.Cells[c,r]:='';
   MemoText.Lines.Clear;
   if GetFullName<>'' then begin
    P:=NewText;
    try
     r:=0;
     P.ReadFile(GetFullName);
     for i:=0 to P.Count-1 do
     if Copy(P[i],1,2)='%%' then begin
      c:=pos('=',P[i]);
      if c>0 then begin
       sn:=Trim(Copy(P[i],3,c-3));
       sv:=Trim(Copy(P[i],c+1,255));
      end else begin
       sn:='';
       sv:='';
      end;
      if IsNonEmptyStr(sn) and IsNonEmptyStr(sv) then begin
       StringGridParameters.Cells[0,r]:=sn;
       StringGridParameters.Cells[1,r]:=sv;
       r:=r+1;
       if r>=StringGridParameters.RowCount then Break;
      end;
     end;
    finally
     Kill(P);
    end;
   end;
  end;
  if HasFlags(What,16) then begin
   MemoText.Lines.Clear;
   if GetFullName<>'' then begin
    P:=NewText;
    try
     P.ReadFile(GetFullName);
     for i:=0 to P.Count-1 do
     if Copy(P[i],1,2)<>'%%' then begin
      for r:=0 to StringGridParameters.RowCount-1 do begin
       sn:=StringGridParameters.Cells[0,r];
       sv:=StringGridParameters.Cells[1,r];
       if IsNonEmptyStr(sn)
       then P[i]:=StringReplace(P[i],'%'+sn+'%',sv,[rfReplaceAll])
       else Break;
      end;
      MemoText.Lines.Add(P[i]);
     end;
     MemoText.SelLength:=0;
     MemoText.SelStart:=0;
    finally
     Kill(P);
    end;
   end;
  end;
  if HasFlags(What,32) then begin
   ListBoxCategory.ItemIndex:=Min(Max(0,TheCategory),ListBoxCategory.Items.Count-1);
   ListBoxName.ItemIndex:=Min(Max(0,TheName),ListBoxName.Items.Count-1);
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateControl');
 end;
end;

procedure TFormTextEditorTemplates.ClearAll;
var
 r : Integer;
 c : Integer;
begin
 if Ok then
 try
  RootDir:='';
  ListBoxCategory.Items.Clear;
  ListBoxName.Items.Clear;
  for r:=0 to StringGridParameters.RowCount-1 do
  for c:=0 to StringGridParameters.ColCount-1 do StringGridParameters.Cells[c,r]:='';
  MemoText.Lines.Clear;
 except
  on E:Exception do BugReport(E,Self,'ClearAll');
 end;
end;

function TFormTextEditorTemplates.GetTemplate(Offset:Integer):LongString;
var
 i : Integer;
 P : TText;
begin
 Result:='';
 if Ok then
 try
  P:=NewText;
  try
   P.Text:=MemoText.Text;
   if (Offset>0) and (Offset<100) then
   for i:=1 to P.Count-1 do P[i]:=Format('%*s%s',[Offset,'',P[i]]);
   while (P.Count>0) and IsEmptyStr(P[P.Count-1]) do P.Count:=P.Count-1;
   Result:=TrimRight(P.Text);
  finally
   Kill(P);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetTemplate');
 end;
end;

procedure TFormTextEditorTemplates.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 BitBtnInsert.Caption:=RusEng('Вставить','Insert');
 BitBtnInsertOffs.Caption:=RusEng('Вставить с отступом','Insert with offset');
 BitBtnCancel.Caption:=mrCaption(mrCancel);
 GroupBoxCategory.Caption:=RusEng('Категория','Category');
 GroupBoxName.Caption:=RusEng('Имя','Name');
 GroupBoxParameters.Caption:=RusEng('Параметры','Parameters');
 GroupBoxText.Caption:=RusEng('Текст','Text');
 ClearAll;
end;

procedure TFormTextEditorTemplates.FormDestroy(Sender: TObject);
begin
 RootDir:='';
 inherited;
end;

procedure TFormTextEditorTemplates.FormActivate(Sender: TObject);
begin
 UpdateControl(1+2);
 UpdateControl(32);
 UpdateControl(4+8);
 UpdateControl(32);
 UpdateControl(8+16);
end;

procedure TFormTextEditorTemplates.ListBoxCategoryClick(Sender: TObject);
begin
 UpdateControl(4+8+16);
end;

procedure TFormTextEditorTemplates.ListBoxNameClick(Sender: TObject);
begin
 UpdateControl(8+16);
end;

procedure TFormTextEditorTemplates.StringGridParametersExit(Sender: TObject);
begin
 UpdateControl(16);
end;

procedure TFormTextEditorTemplates.StringGridParametersSelectCell(Sender: TObject; ACol, ARow: Integer; var CanSelect: Boolean);
begin
 UpdateControl(16);
end;

procedure TFormTextEditorTemplates.StringGridParametersClick(Sender: TObject);
begin
 UpdateControl(16);
end;

procedure TFormTextEditorTemplates.BitBtnInsertClick(Sender: TObject);
begin
 ModalResult:=mrOk;
end;

procedure TFormTextEditorTemplates.BitBtnInsertOffsClick(Sender: TObject);
begin
 ModalResult:=mrYes;
end;

procedure TFormTextEditorTemplates.FormResize(Sender: TObject);
var w:Integer;
begin
 inherited;
 try
  if (StringGridParameters.ColCount<2) then Exit;
  if (StringGridParameters.ColWidths[0]<StringGridParameters.DefaultColWidth)
  then StringGridParameters.ColWidths[0]:=StringGridParameters.DefaultColWidth;
  w:=StringGridParameters.Width-StringGridParameters.ColWidths[0];
  w:=w-StringGridParameters.GridLineWidth;
  w:=w-GetSystemMetrics(SM_CXVSCROLL)-10;
  w:=Max(w,StringGridParameters.DefaultColWidth);
  StringGridParameters.ColWidths[1]:=w;
 except
  on E:Exception do BugReport(E,Self,'FormResize');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_texteditortemplates;
begin
end;

procedure Free_form_texteditortemplates;
begin
end;

initialization

 Init_form_texteditortemplates;

finalization

 Free_form_texteditortemplates;

end.

//////////////
// END OF FILE
//////////////

