////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Surf Tools Smoothing Dialog.                                          //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231125 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_surftoolssmoothingdialog; // Form Surf Tools Smoothing Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Spin,
 lcltype, lclintf,
 Form_SurfWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormSurfToolsSmoothingDialog = class(TMasterForm)
    GroupBoxNxNy: TGroupBox;
    LabelNx: TLabel;
    SpinEditNx: TSpinEdit;
    LabelNy: TLabel;
    SpinEditNy: TSpinEdit;
    GroupBoxK1XK2X: TGroupBox;
    LabelK1X: TLabel;
    SpinEditK1X: TSpinEdit;
    LabelK2X: TLabel;
    SpinEditK2X: TSpinEdit;
    GroupBoxK1YK2Y: TGroupBox;
    LabelK1Y: TLabel;
    SpinEditK1Y: TSpinEdit;
    LabelK2Y: TLabel;
    SpinEditK2Y: TSpinEdit;
    PanelButtons: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure LabelNxClick(Sender: TObject);
    procedure LabelNyClick(Sender: TObject);
    procedure LabelK1XClick(Sender: TObject);
    procedure LabelK2XClick(Sender: TObject);
    procedure LabelK1YClick(Sender: TObject);
    procedure LabelK2YClick(Sender: TObject);
  end;

function FormSurfToolsSmoothingDialogExecute(aForm:TForm; const aParams:LongString=''):TFormSurfWindow;

implementation

{$R *.lfm}

const
  FormSurfToolsSmoothingDialog : TFormSurfToolsSmoothingDialog = nil;

function FormSurfToolsSmoothingDialogExecute(aForm:TForm; const aParams:LongString=''):TFormSurfWindow;
var Source:TFormSurfWindow; n,hw,k1,k2:TPoint2I; i,j,ii,jj:LongInt; w,summ,norm:Double; apFlags:Integer;
 function Kern(x:Double; k1,k2:Integer):Double;
 begin
  x:=abs(x);
  if x<1 then begin
   case k1 of
    0  : Result:=1;
    1  : Result:=1-x;
    2  : Result:=1-x*x;
    else Result:=1-IntPower(x,k1);
   end;
   case k2 of
    0  : Result:=1;
    1  : ;
    2  : Result:=sqr(Result);
    else Result:=IntPower(Result,k2);
   end;
  end else Result:=0;
 end;
begin
 Result:=nil;
 if CanShowModal(FormSurfToolsSmoothingDialog) then
 try
  if FormExists(aForm) and (aForm is TFormSurfWindow)
  then Source:=TFormSurfWindow(aForm)
  else Source:=ActiveSurfWindow;
  if Source.Ok then begin
   if not Assigned(FormSurfToolsSmoothingDialog)  then begin
    Application.CreateForm(TFormSurfToolsSmoothingDialog, FormSurfToolsSmoothingDialog);
    FormSurfToolsSmoothingDialog.Master:=@FormSurfToolsSmoothingDialog;
   end;
   if Assigned(FormSurfToolsSmoothingDialog) then begin
    apFlags:=FormSurfToolsSmoothingDialog.ApplyParams(aParams);
    if not HasFlags(apFlags,apf_FormPos)
    then LocateFormToCenterOfScreen(FormSurfToolsSmoothingDialog);
    if mrVoice(FormSurfToolsSmoothingDialog.ShowModal)=mrOk then begin
     with FormSurfToolsSmoothingDialog do begin
      hw:=Point2I(SpinEditNx.Value,SpinEditNy.Value);
      k1:=Point2I(SpinEditK1X.Value,SpinEditK1Y.Value);
      k2:=Point2I(SpinEditK2X.Value,SpinEditK2Y.Value);
     end;
     Result:=Source.Clone;
     Result.Caption:=RusEng('СГЛАЖИВАНИЕ:','SMOOTHING:')+Source.Caption;
     Result.LockDraw;
     with Source do begin
      n:=NumPoints;
      for i:=0 to n.x-1 do if Result.Ok then
      for j:=0 to n.y-1 do if Result.Ok then begin
       summ:=0;
       norm:=0;
       for ii:=i-hw.x to i+hw.x do
       for jj:=j-hw.y to j+hw.y do begin
        w:=Kern((ii-i)/(hw.x+1),K1.x,K2.x)*Kern((jj-j)/(hw.y+1),K1.y,K2.y);
        summ:=summ+w*z[max(0,min(n.x-1,ii)),max(0,min(n.y-1,jj))];
        norm:=norm+w;
       end;
       if norm>0 then Result[i,j]:=summ/norm else Result[i,j]:=0;
       if Trouble(isNanOrInf(Result[i,j]),RusEng('Ошибка вычислений!','Calculation error!')) then Kill(Result);
      end;
     end;
     Result.UnlockDraw;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'FormSurfToolsSmoothingDialogExecute');
 end;
end;

procedure TFormSurfToolsSmoothingDialog.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('Диалог сглаживания поверхности','Surface smoothing dialog');
 SmartUpdate(GroupBoxNxNy,RusEng('Полуширина окна сглаживания','Smoothing window half width'));
 SmartUpdate(GroupBoxK1XK2X,RusEng('Ядро F(X)=(1-X^K1)^K2 вдоль X','Kernel F(X)=(1-X^K1)^K2 along X'));
 SmartUpdate(GroupBoxK1YK2Y,RusEng('Ядро F(Y)=(1-Y^K1)^K2 вдоль Y','Kernel F(Y)=(1-Y^K1)^K2 along Y'));
 SmartUpdate(LabelNx,RusEng('вдоль X','along X'));
 SmartUpdate(LabelNy,RusEng('вдоль Y','along Y'));
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
end;

procedure TFormSurfToolsSmoothingDialog.LabelNxClick(Sender: TObject);
begin
 SmartFocus(SpinEditNx);
end;

procedure TFormSurfToolsSmoothingDialog.LabelNyClick(Sender: TObject);
begin
 SmartFocus(SpinEditNy);
end;

procedure TFormSurfToolsSmoothingDialog.LabelK1XClick(Sender: TObject);
begin
 SmartFocus(SpinEditK1X);
end;

procedure TFormSurfToolsSmoothingDialog.LabelK2XClick(Sender: TObject);
begin
 SmartFocus(SpinEditK2X);
end;

procedure TFormSurfToolsSmoothingDialog.LabelK1YClick(Sender: TObject);
begin
 SmartFocus(SpinEditK1Y);
end;

procedure TFormSurfToolsSmoothingDialog.LabelK2YClick(Sender: TObject);
begin
 SmartFocus(SpinEditK2Y);
end;


///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_surftoolssmoothingdialog;
begin
end;

procedure Free_form_surftoolssmoothingdialog;
begin
end;

initialization

 Init_form_surftoolssmoothingdialog;

finalization

 Free_form_surftoolssmoothingdialog;

end.

//////////////
// END OF FILE
//////////////

