unit form_spectrmanfitwindow;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_SpectrWindow,
 Form_SpectrDaqWindow, Form_SpectrPikWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_sort, _crw_ef,  _crw_ee, _crw_zm,
 _crw_curves, _crw_riff, _crw_funmin,
 _crw_calib, _crw_lsqpoly,
 _crw_daqsys, _crw_daqdev,
 _crw_appforms, _crw_apptools, _crw_apputils;

const
 MaxManPoly  = 4;   { макс. число коэффициентов полинома фона       }
 MaxManPeaks = 15;  { максимальное возможное число фитируемых пиков }

type
 TManPeakParam = packed record Chan,Ampl,FWHM : Double  end; { параметры описания пиков }
 TManPeakFixed = packed record Chan,Ampl,FWHM : Boolean end; { флаги фиксации параметров }

const
 ManPeakDim = sizeof(TManPeakParam) div sizeof(Double);

type
 PManParams = ^TManParams;
 TManParams = packed record
  case byte of
  1:(P    : packed array[0..MaxManPoly+MaxManPeaks*ManPeakDim-1] of Double);
  2:(Poly : packed array[0..MaxManPoly-1 ] of Double;
     Peak : packed array[0..MaxManPeaks-1] of TManPeakParam);
 end;
 TManFixParams = packed record
  case byte of
  1:(P    : packed array[0..MaxManPoly+MaxManPeaks*ManPeakDim-1] of Boolean);
  2:(Poly : packed array[0..MaxManPoly-1 ] of Boolean;
     Peak : packed array[0..MaxManPeaks-1] of TManPeakFixed);
 end;

type
  TFormSpectrManFitWindow = class(TFormSpectrPikWindow)
    ActionSpectrManFitControl: TAction;
    MenuSpectrManFitControl: TMenuItem;
    ToolButtonSpectrManFitControl: TToolButton;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ActionSpectrManFitControlExecute(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    ManFit      : packed record
     Control    : TMasterForm;       { Пульт управления фитированием пиков    }
     FitThread  : TThread;           { Поток минимизации                      }
     UserBreak  : Boolean;           { Флаг остановки минимизации             }
     NumPeaks   : Integer;           { Текущее число пиков спектра            }
     Left       : Integer;           { Начало интервала ROI для фитирования   }
     Right      : Integer;           { Конец интервала ROI для фитирования    }
     FitResult  : Integer;           { Минимизация успешна                    }
     CurrParam  : TManParams;        { Текущие параметры описания пиков       }
     CurrFixed  : TManFixParams;     { Флажки фиксации этих параметров        }
     LoScale    : TManParams;        { Масштаб аргумента                      }
     HiScale    : TManParams;        { Масштаб аргумента                      }
     Chi2Crit   : Double;            { Хи-квадрат на 5% уровене доверия       }
     Covar      : TDoubleMatrix;     { Ковариационная матрица                 }
     Preset     : packed record      { Параметры фиттинга                     }
      Method    : Integer;           { Метод поиска                           }
      MaxCalls  : Integer;           { Максимальное число обращений к функции }
      Tolx      : Double;            { Требуемая точность по аргументу        }
      Tolf      : Double;            { Требуемая точность по функции          }
      Tolg      : Double;            { Требуемая точность по градиенту        }
      Size      : Double;            { Начальный размер симплекса             }
     end;
     FunValue   : Double;            { Значение невязки                       }
     FunCalls   : Integer;           { Сетчик вызовов функции                 }
     WorkParam  : TManParams;        { Рабочий массив                         }
     WorkGrad   : TManParams;        { Рабочий массив                         }
     TempGrad   : TManParams;        { Рабочий массив                         }
     LastReport : Int64;             { Для таймирования                       }
     FieldName  : packed array[0..ManPeakDim+2] of string[9];   { Имена полей }
    end;
    function  EvalGround(Channel:Double; var Param,Grad:TManParams):Double;
    function  EvalFitAmpl(Channel:Double; var Param,Grad:TManParams):Double;
    function  EvalFitArea(Channel:Double; var Param,Grad:TManParams):Double;
    function  EvalFitGround(Channel:Double; var Param,Grad:TManParams):Double;
    procedure ManFitSort;
    procedure DrawFul; override;
    procedure DrawExp; override;
    procedure MarkerChanged; override;
    procedure SpectrClear; override;
    procedure ManFitClear;
    procedure ManFitControlOpen;
    procedure ManFitControlUpdate(Flags:Integer);
    procedure ManFitExecute;
    procedure ManFitExecuteThread;
    function  ManFitInitInterval:Boolean;
    procedure ManFitChangeInterval;
    procedure ManFitCallPeakin;
    procedure ManFitPeakInsert(Channel:LongInt);
    procedure ManFitPeakDelete(Num:integer);
    function  ManFitFindNearest(Ch:double):integer;
    function  ManFitDetailsList(P:TText):TText;
    function  ManFitCovarList(P:TText):TText;
    procedure ManFitSavePik(FName:LongString; isAppend:Boolean);
  end;

const
 regSpecManFitWin = 'SpecManFitWin';

function  NewSpecManFitWin(const aCaption    : LongString;
                                 aSpectr     : TCurve;
                                 aOwnsSpectr : Boolean):TFormSpectrWindow;
procedure Kill(var TheObject:TFormSpectrManFitWindow); overload;

function GaussPeakArea(Ampl,FWHM:Double):Double;

implementation

{$R *.lfm}

uses Form_SpectrManFitControl,Form_SpectrManFitBreaker;

 {
 *******************************************************************************
 Частные производные Гауссовских пиков можно вычислить в пакете
 Matlab / Symbolic math toolbox как показано здесь:
        a    = амплитуда
        w    = полуширина на полувысоте
        с    = положение пика
        s    = площадь пика
        alfa = 4*ln(2)  = 2.77258872223978E+0000 - константа для экспоненты
        (pi/alfa)^(1/2) = 1.06446701943123E+0000 - для перевода площади s/a/w
        (alfa/pi)^(1/2) = 9.39437278699651E-0001 - для перевода площади a*w/s
 Гауссовский пик можно брать в двух формах:
  1.(положение,амплитуда,ширина)
  2.(положение,площадь,ширина)
 В первом случае используем код
        f=sym('a*exp(-alfa*(x-c)^2/w^2)')
        diff(f,'a')=exp(-alfa*(x-c)^2/w^2)
        diff(f,'c')=2*a*alfa*(x-c)/w^2*exp(-alfa*(x-c)^2/w^2)
        diff(f,'w')=2*a*alfa*(x-c)^2/w^3*exp(-alfa*(x-c)^2/w^2)
        s=(pi/alfa)^(1/2)*a*w
 Во втором случае используем код
        f=sym('(alfa/pi)^(1/2)*s/w*exp(-alfa*(x-c)^2/w^2)')
        diff(f,'s')=(alfa/pi)^(1/2)/w*exp(-alfa*(x-c)^2/w^2)
        diff(f,'c')=2*(alfa/pi)^(1/2)*s/w^3*alfa*(x-c)*exp(-alfa*(x-c)^2/w^2)
        diff(f,'w')=-(alfa/pi)^(1/2)*s/w^2*exp(-alfa*(x-c)^2/w^2)
                   +2*(alfa/pi)^(1/2)*s/w^4*alfa*(x-c)^2*exp(-alfa*(x-c)^2/w^2)
        diff(f,'w')=(alfa/pi)^(1/2)*s/w^2*exp(-alfa*(x-c)^2/w^2)*
             (2*alfa*(x-c)^2/w^2-1)
        a=(alfa/pi)^(1/2)*s/w
 Для вычислений в подпрограмме нужны экономичные схемы вычисления производных:
         f=a*exp(-alfa*(x-c)^2/w^2):
           t=(x-c)/w
           alfat:=alfa*t
           alfatt:=alfat*t
           if alfatt>200 then diffa=0 else diffa=exp(-alfatt) чтобы не было INF
           fun=a*diffa;
           diffc=2*fun*alfat/w
           diffw=diffc*t
         f=(alfa/pi)^(1/2)*s/w*exp(-alfa*(x-c)^2/w^2)
          t=(x-c)/w
          alfat:=alfa*t
          alfatt:=alfat*t
          if alfatt>200 then diffs=0 else diffs=sqrt(alfa/pi)*exp(-alfatt)/w
          fun=s*diffs
          diffc=2*fun*alfat/w
          diffw=(2*alfatt-1)*fun/w
 *******************************************************************************
 }
const
 alfa = 2.77258872223978123766892848583271; {=4*ln(2)}

 {
 Функция вычисляет площадь теретического пика
 }
function GaussPeakArea(Ampl,FWHM:Double):Double;
begin
 GaussPeakArea:=Ampl*FWHM*sqrt(pi/alfa);
end;

 {
 Функция вычисляет амплитуду теретического пика
 }
function GaussPeakAmpl(Area,FWHM:Double):Double;
begin
 GaussPeakAmpl:=Area/FWHM*sqrt(alfa/pi);
end;

 {
 Функция фитирования спектра
 }
function SpectrFit(t:double; n:integer; var x:array of double; var f:double;
                   var g:array of double; Custom:Pointer):integer;
var xParam,gParam:PManParams; SpecWin:TFormSpectrManFitWindow;
begin
 SpectrFit:=ecOk;
 SpecWin:=Custom; xParam:=@x; gParam:=@g;
 {данные допустимы?}
 if not Assigned(SpecWin) then begin
  Daq.Report(RusEng('NIL в МНК!','NIL in fitting!'));
  SpectrFit:=ecBadArg;
  exit;
 end;
 {прерывание пользователя?}
 if SpecWin.ManFit.UserBreak then begin
  Daq.Report(RusEng('Пользователь прервал МНК!','User break fitting!'));
  SpectrFit:=ecUserBreak;
  exit;
 end;
 f:=SpecWin.EvalFitGround(t,xParam^,gParam^);
 {все в порядке?}
 if IsNanOrInf(F) then begin
  Daq.Report(RusEng('NAN/INF в МНК!','NAN/INF in fitting!'));
  SpectrFit:=ecBadArg;
  exit;
 end;
end;

 {
 Функция фитирования спектра для вычисления ковариации площади
 }
function SpectrFitArea(t:double;n:integer; var x:array of double; var f:double;
                       var g:array of double; Custom:Pointer):integer;
var xParam,gParam:PManParams; SpecWin:TFormSpectrManFitWindow;
begin
 SpectrFitArea:=ecOk;
 SpecWin:=Custom; xParam:=@x; gParam:=@g;
 {данные допустимы?}
 if not Assigned(SpecWin) then begin
  Daq.Report(RusEng('NIL в МНК!','NIL in fitting!'));
  SpectrFitArea:=ecBadArg;
  exit;
 end;
 {прерывание пользователя?}
 if SpecWin.ManFit.UserBreak then begin
  Daq.Report(RusEng('Пользователь прервал фитирование!','User break fitting!'));
  SpectrFitArea:=ecUserBreak;
  exit;
 end;
 f:=SpecWin.EvalFitArea(t,xParam^,gParam^)+SpecWin.EvalGround(t,xParam^,gParam^);
 {все в порядке?}
 if IsNanOrInf(F) then begin
  Daq.Report(RusEng('Недопустимое значение при фитировании!','Invalid value in fitting!'));;
  SpectrFitArea:=ecBadArg;
  exit;
 end;
end;

 {
 доклад о ходе итераций
 }
procedure MinReport(m:longint; const t,y,w:array of double;
                    n:integer; const x:array of double; const F:double;
                    const g:array of double; Custom:Pointer; Count:LongInt; const Met,Com:LongString);
var SpecWin:TFormSpectrManFitWindow;
begin
 SpecWin:=Custom;
 if Assigned(SpecWin) then
 if (PosI('START', UnifyAlias(Com))>0) or
    (PosI('STOP',  UnifyAlias(Com))>0) or
    (abs(GetTickCount64-SpecWin.ManFit.LastReport)>500)
 then begin
  Daq.Report(Format('Peak fit: N=%-5d, F=%-11.6g, G=%-11.6g, M="%s", C="%s".',[Count,f,sqrt(scpr(g,g,n)),Met,Com]));
  SpecWin.ManFit.LastReport:=GetTickCount64;
 end;
end;

 {
 *******************************************************************************
 General purpose utilites
 *******************************************************************************
 }
function  NewSpecManFitWin(const aCaption    : LongString;
                                 aSpectr     : TCurve;
                                 aOwnsSpectr : Boolean ) : TFormSpectrWindow;
begin
 Application.CreateForm(TFormSpectrManFitWindow, Result);
 if Result.Ok then
 with Result do
 try
  LockDraw;
  if aCaption=''
  then Caption:=RusEng('Спектрометрическое окно','Spectrometry window')
  else Caption:=aCaption;
  AssignSpectr(aSpectr,aOwnsSpectr);
 finally
  UnlockDraw;
 end;
end;

procedure Kill(var TheObject:TFormSpectrManFitWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

 {
 *******************************************************************************
 TFormSpectrManFitWindow implementation
 *******************************************************************************
 }
procedure TFormSpectrManFitWindow.FormCreate(Sender: TObject);
begin
 inherited;
 ManFit.Control:=nil;
 ManFit.FitThread:=nil;
 ManFit.UserBreak:=true;
 ManFit.NumPeaks:=0;
 ManFit.Left:=0;
 ManFit.Right:=0;
 ManFit.FitResult:=-1;
 SafeFillChar(ManFit.CurrParam,sizeof(ManFit.CurrParam),0);
 SafeFillChar(ManFit.CurrFixed,sizeof(ManFit.CurrFixed),0);
 SafeFillChar(ManFit.LoScale,sizeof(ManFit.LoScale),0);
 SafeFillChar(ManFit.HiScale,sizeof(ManFit.HiScale),0);
 ManFit.Chi2Crit:=0;
 ManFit.Covar:=nil;
 ManFit.Preset.Method:=mDavidonFletcherPowell;
 ManFit.Preset.MaxCalls:=10000;
 ManFit.Preset.Tolx:=1E-5;
 ManFit.Preset.Tolf:=1E-5;
 ManFit.Preset.Tolg:=1E-5;
 ManFit.Preset.Size:=0.1;
 ManFit.FunValue:=0;
 ManFit.FunCalls:=0;
 SafeFillChar(ManFit.WorkParam,sizeof(ManFit.WorkParam),0);
 SafeFillChar(ManFit.WorkGrad,sizeof(ManFit.WorkGrad),0);
 SafeFillChar(ManFit.TempGrad,sizeof(ManFit.TempGrad),0);
 ManFit.LastReport:=0;
 ManFit.FieldName[0]:='Channel';
 ManFit.FieldName[1]:='Amplitude';
 ManFit.FieldName[2]:='FWHM';
 ManFit.FieldName[3]:='Area';
 ManFit.FieldName[4]:='Ground';
 ManFit.FieldName[5]:='Energy';
 UpdateMenu(MenuSpectrManFitControl,
            RusEng('Фитирование пиков','Peak fit'),
            RusEng('Открыть диалог фитирования пиков.','Open peak fit dialog.'),
            0);
end;

procedure TFormSpectrManFitWindow.FormDestroy(Sender: TObject);
begin
 try
  LockDraw;
  ManFitClear;
  Kill(ManFit.Control);
  Kill(TObject(ManFit.FitThread));
  Kill(ManFit.Covar);
 finally
  UnlockDrawHidden;
 end;
 inherited;
end;

function TFormSpectrManFitWindow.EvalGround(Channel:Double; var Param,Grad:TManParams):Double;
var i:integer; sum,t:double;
begin
 t:=(Channel-ManFit.Left)/(ManFit.Right-ManFit.Left+1);
 sum:=0;
 for i:=MaxManPoly-1 downto 0 do sum:=sum*t+Param.Poly[i];
 EvalGround:=sum;
 Grad.Poly[0]:=1;
 for i:=1 to MaxManPoly-1 do Grad.Poly[i]:=Grad.Poly[i-1]*t;
end;

function TFormSpectrManFitWindow.EvalFitAmpl(Channel:Double; var Param,Grad:TManParams):Double;
var i:integer; t,alfat,alfatt,sum,fun,diffa,diffc,diffw:Extended;
begin
 sum:=0;
 for i:=0 to ManFit.NumPeaks-1 do begin
  with Param.Peak[i] do begin
   t:=(Channel-Chan)/FWHM;
   alfat:=alfa*t;
   alfatt:=alfat*t;
   if alfatt>200 then diffa:=0 else diffa:=exp(-alfatt);
   fun:=Ampl*diffa;
   diffc:=2*fun*alfat/FWHM;
   diffw:=diffc*t;
  end;
  sum:=sum+fun;
  with Grad.Peak[i] do begin
   Ampl:=diffa;
   Chan:=diffc;
   FWHM:=diffw;
  end;
 end;
 EvalFitAmpl:=sum;
end;

function TFormSpectrManFitWindow.EvalFitArea(Channel:Double; var Param,Grad:TManParams):Double;
var i:integer; t,alfat,alfatt,fun,diffa,diffc,diffw,sum:Extended;
begin
 sum:=0;
 for i:=0 to ManFit.NumPeaks-1 do begin
  with Param.Peak[i] do begin
   t:=(Channel-Chan)/FWHM;
   alfat:=alfa*t;
   alfatt:=alfat*t;
   if alfatt>200 then diffa:=0 else diffa:=sqrt(alfa/pi)*exp(-alfatt)/FWHM;
   fun:=Ampl*diffa;
   diffc:=2*fun*alfat/FWHM;
   diffw:=(2*alfatt-1)*fun/FWHM;
  end;
  sum:=sum+fun;
  with Grad.Peak[i] do begin
   Ampl:=diffa;
   Chan:=diffc;
   FWHM:=diffw;
  end;
 end;
 EvalFitArea:=sum;
end;

function TFormSpectrManFitWindow.EvalFitGround(Channel:Double; var Param,Grad:TManParams):Double;
begin
 EvalFitGround:=EvalFitAmpl(Channel,Param,Grad)+EvalGround(Channel,Param,Grad);
end;

procedure TFormSpectrManFitWindow.ManFitSort;
 procedure Sort(l,r:Integer);
 var i,j:integer; c:double; p:TManPeakParam;  f:TManPeakFixed;
 begin
  i:=l;
  j:=r;
  c:=ManFit.CurrParam.Peak[(l+r) shr 1].Chan;
  repeat
    while ManFit.CurrParam.Peak[i].Chan<c do inc(i);
    while c<ManFit.CurrParam.Peak[j].Chan do dec(j);
    if i<=j then begin
      p:=ManFit.CurrParam.Peak[i];
      ManFit.CurrParam.Peak[i]:=ManFit.CurrParam.Peak[j];
      ManFit.CurrParam.Peak[j]:=p;
      f:=ManFit.CurrFixed.Peak[i];
      ManFit.CurrFixed.Peak[i]:=ManFit.CurrFixed.Peak[j];
      ManFit.CurrFixed.Peak[j]:=f;
      inc(i);
      dec(j);
    end;
  until i>j;
  if l<j then Sort(l,j);
  if i<r then Sort(i,r);
 end;
begin
 if Ok and (ManFit.NumPeaks>1) then
 try
  Sort(0,ManFit.NumPeaks-1);
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrManFitWindow.DrawFul;
begin
 inherited DrawFul;
end;

procedure TFormSpectrManFitWindow.DrawExp;
var G,P:TRect2I; ch,gr,pk,k:double; i,n,l,r:Integer;
begin
 inherited DrawExp;
 if (SpectrSize>0) and (ManFit.NumPeaks>0) and Assigned(ManFit.Control) then
 try
  L:=max(ManFit.Left,BegX);
  R:=min(ManFit.Right,EndX);
  if r>l then begin
   n:=ExpandView.ConvertX(R-BegX)-ExpandView.ConvertX(L-BegX);
   k:=(R-L)/n;
   for i:=0 to n do begin
    Ch:=L+k*i;
    Gr:=EvalGround(ch,ManFit.CurrParam,ManFit.TempGrad);
    Pk:=EvalFitGround(ch,ManFit.CurrParam,ManFit.TempGrad);
    G.B.X:=ExpandView.ConvertX(Ch-BegX);
    G.B.Y:=ExpandView.ConvertY(Gr,isLog);
    P.B.X:=G.B.X;
    P.B.Y:=ExpandView.ConvertY(Pk,isLog);
    if i>0 then begin
     DrawLine(PaintBoxExpand.Canvas, G.A, G.B, clSpectrPeakGnd);
     DrawLine(PaintBoxExpand.Canvas, P.A, P.B, clSpectrPeakForm);
    end;
    P.A:=P.B;
    G.A:=G.B;
   end;
   for i:=0 to ManFit.NumPeaks-1 do with ManFit.CurrParam.Peak[i] do
   ExpandView.DrawPeakMarker(Chan,EvalFitGround(Chan,ManFit.CurrParam,
                             ManFit.TempGrad),clSpectrPeakMarker);
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrManFitWindow.MarkerChanged;
begin
 inherited MarkerChanged;
 ManFitControlUpdate(2);
end;

procedure TFormSpectrManFitWindow.SpectrClear;
begin
 try
  LockDraw;
  ManFitClear;
  inherited SpectrClear;
 finally
  UnLockDraw;
 end;
end;

procedure TFormSpectrManFitWindow.ManFitClear;
begin
 try
  LockDraw;
  ROIUnMark;
  ManFit.Left:=0;
  ManFit.Right:=0;
  ManFit.NumPeaks:=0;
  Kill(ManFit.Covar);
  ManFit.FunCalls:=0;
  ManFit.FunValue:=0;
  ManFit.FitResult:=-1;
  Kill(ManFit.Covar);
 finally
  UnlockDraw;
  ManFitControlUpdate(1+2);
 end;
end;

procedure TFormSpectrManFitWindow.ManFitControlOpen;
begin
 if Ok then
 if ManFit.Control is TFormSpectrManFitControl then begin
  ManFit.Control.Show;
  ManFit.Control.BringToFront;
  ManFit.Control.WindowState:=wsNormal;
  ManFitControlUpdate(1+2+$10);
 end else begin
  ManFit.Control:=NewFormSpectrManFitControl(Self);
  ManFit.Control.Master:=@ManFit.Control;
  ManFit.Control.Show;
  ManFit.Control.BringToFront;
  ManFitControlUpdate(1+2+$10);
 end;
end;

procedure TFormSpectrManFitWindow.ManFitControlUpdate(Flags:Integer);
begin
 if Ok then
 if ManFit.Control is TFormSpectrManFitControl
 then (ManFit.Control as TFormSpectrManFitControl).UpdateControls(Flags);
end;

procedure TFormSpectrManFitWindow.ManFitExecute;
var CovResult,nv,n,nfree,i,j,ii,jj,m:LongInt; buf,v,t,y,w:PDoubleArray;
begin
 if Ok then begin
  ManFit.UserBreak:=false;
  if ManFit.NumPeaks>0 then begin
   m:=ManFit.Right-ManFit.Left+1;             { число точек спектра минимизации  }
   n:=ManPeakDim*ManFit.NumPeaks+MaxManPoly;  { число параметров минимизации     }
   nv:=(n+10)*(n+1);                          { размерность буфера v минимизации }
   buf:=Allocate((nv+3*m)*sizeof(Double));    { буфер для всех массивов          }
   if AllocSize(buf)>0 then
   try
    v:=buf;                                   { внутренний массив минимизации    }
    t:=@v[nv];                                { буфер для массива каналов        }
    y:=@t[m];                                 { буфер для массива счетов         }
    w:=@y[m];                                 { буфер для массива весов          }
    {
    Заполнить массив исходных данных которые будут фитироваться
    }
    for i:=ManFit.Left to ManFit.Right do begin
     t[i-ManFit.Left]:=i;
     y[i-ManFit.Left]:=SpectrValue[i];
     w[i-ManFit.Left]:=1/max(1,abs(SpectrValue[i]));
    end;
    {
    Сортировать пики по возрастанию канала
    }
    ManFitSort;
    {
    факторы сдвига и масштаба:
     chan меняется по порядку величины на 1 канал около начального значения,
     fwhm меняется по порядку величины на 1 канал,
     ampl меняется в широком диапазоне от 0 до средней амплитуды
    }
    for i:=0 to ManFit.NumPeaks-1 do begin
     ManFit.LoScale.Peak[i].Chan:=ManFit.CurrParam.Peak[i].Chan;
     ManFit.HiScale.Peak[i].Chan:=ManFit.LoScale.Peak[i].Chan+1;
     ManFit.LoScale.Peak[i].Ampl:=0;
     ManFit.HiScale.Peak[i].Ampl:=max(ManFit.CurrParam.Peak[i].Ampl,1)/5;
     ManFit.LoScale.Peak[i].FWHM:=0;
     ManFit.HiScale.Peak[i].FWHM:=max(ManFit.LoScale.Peak[i].FWHM,1);
    end;
    {
    Функцию масштабируем на значение Хи-квадрат на 5% уровне.
    Это характерное значение, с которым надо сравнивать значение
    остаточной суммы квадратов.
    }
    ManFit.WorkParam:=ManFit.CurrParam;
    nfree:=m-NumUnfixed(ManFit.CurrFixed.P,n);
    ManFit.Chi2Crit:=Chi2CumDistrInv(1-0.05,nfree);
    {
    вызов задачи минимизации
    }
    ManFit.WorkParam:=ManFit.CurrParam;
    with ManFit.Preset do
    ManFit.FitResult:=LSQFit(m,t^,y^,w^,SpectrFit, MinReport,n,ManFit.WorkParam.P,
                      ManFit.FunValue,ManFit.WorkGrad.P,Self,ManFit.FunCalls,
                      MaxCalls,Method,Tolx,Tolf,Tolg,
                      0,Size,1E-6,1E-14,v^,ManFit.CurrFixed.P,ManFit.Chi2Crit,
                      ManFit.LoScale.P,ManFit.HiScale.P);
    ManFit.CurrParam:=ManFit.WorkParam;
    {
    Сортировать пики по возрастанию канала
    }
    ManFitSort;
    {
    вычислить ковариацию и занести ее в поле Covar
    }
    {удаляем старый экземпляр ковариации}
    Kill(ManFit.Covar);
    {ковариацию вычисляем если минимизация прошла хорошо}
    if ManFit.FitResult=ecOk then begin
     {нас интересует ковариация в переменных Chan,Area,FWHM-конвертируем Area}
     for i:=0 to ManFit.NumPeaks-1 do
     with ManFit.WorkParam.Peak[i] do Ampl:=GaussPeakArea(Ampl,FWHM);
     ManFit.Covar:=NewDoubleMatrix(n,n);
     if Assigned(ManFit.Covar) then begin
      {вычисление ковариации в упакованном виде}
      CovResult:=LSQFitCov(m,t^,y^,w^,SpectrFitArea,n,ManFit.WorkParam.P,
                           ManFit.CurrFixed.P,v^,n,Self);
      if CovResult<>0 then begin
       Daq.Report(RusEng('Ошибка при вычислении ковариации ','Error evaluate covariation ')+d2s(CovResult));
       Kill(ManFit.Covar);
      end else begin
       {распаковка ковариации в исходном пространстве}
       i:=0;
       for ii:=0 to n-1 do begin
        if ManFit.CurrFixed.P[ii] then continue;
        j:=0;
        for jj:=0 to n-1 do begin
         if ManFit.CurrFixed.P[jj] then continue;
         ManFit.Covar[ii,jj]:=v[i*n+j];
         inc(j);
        end;
        inc(i);
       end;
      end;
     end;
    end;
    {
    прорисовки
    }
    case ManFit.FitResult of
     ecOk        : if ManFit.FunValue<ManFit.Chi2Crit
                   then Daq.Report(RusEng('Минимизация завершена, удовлетворительно.','Minimization done, good.'))
                   else Daq.Report(RusEng('Минимизация завершена, не удовлетворительно.','Minimization done, bad.'));
     ecUserBreak : Daq.Report(RusEng('Пользователь прервал фитирование!','User break fit!'));
     else          Daq.Report(RusEng('Ошибка минимизации ','Minimize error ')+d2s(ManFit.FitResult)+'!');
    end;
   except
    on E: Exception do Daq.Report(E.Message);
   end else Daq.Report(RusEng('Мало памяти!','Out of memory!'));
   Deallocate(Pointer(buf));
  end else Daq.Report(RusEng('Нет пиков в списке!','Have no peaks to fit!'));
  ManFit.UserBreak:=true;
 end;
end;

type
 TFitThread=class(TThread)
 public
  Win:TFormSpectrManFitWindow;
  procedure Execute; override;
 end;

procedure TFitThread.Execute;
begin
 Win.ManFitExecute;
end;

procedure TFormSpectrManFitWindow.ManFitExecuteThread;
begin
 if Ok then
 try
  LockDraw;
  if ManFit.Control.Ok then ManFit.Control.Hide;
  try
   ManFit.UserBreak:=false;
   ManFit.FitThread:=TFitThread.Create(true);
   (ManFit.FitThread as TFitThread).Win:=Self;
   ManFit.FitThread.Suspended:=false; // Resume
   FormSpectrManFitBreakerExecute(Self);
   Kill(TObject(ManFit.FitThread));
  except
   on E: Exception do Daq.Report(E.Message);
  end;
  if ManFit.Control.Ok then ManFit.Control.Show;
  ManFitControlUpdate(1+2);
 finally
  UnlockDraw;
 end;
end;

function TFormSpectrManFitWindow.ManFitInitInterval:Boolean;
var i:Integer; AverageGround:Double;
begin
 Result:=false;
 if Ok then
 try
  LockDraw;
  try
   Kill(ManFit.Covar);
   ManFit.FitResult:=-1;
   ManFit.Left:=MarkerL;
   ManFit.Right:=MarkerR;
   if not Trouble(ManFit.Left>=ManFit.Right,'Недопустимый интервал!') and
      not Trouble(Integral<10,'Статистика слишком мала!')
   then begin
    RoiMark;
    AverageGround:=max(0.5*(SpectrValue[ManFit.Right]+SpectrValue[ManFit.Left]),1);
    for i:=0 to MaxManPoly-1 do begin
     ManFit.CurrParam.Poly[i]:=0;
     ManFit.CurrFixed.Poly[i]:=false;
     if i>1 then ManFit.CurrFixed.Poly[i]:=true;
     ManFit.LoScale.Poly[i]:=0;
     ManFit.HiScale.Poly[i]:=AverageGround/(5*Gamma(i+1));
    end;
    ManFit.CurrParam.Poly[0]:=SpectrValue[ManFit.Left];
    ManFit.CurrParam.Poly[1]:=SpectrValue[ManFit.Right]-SpectrValue[ManFit.Left];
    Result:=true;
   end;
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  UnLockDraw;
 end;
end;

procedure TFormSpectrManFitWindow.ManFitChangeInterval;
var i:Integer; f:Boolean;
begin
 if Ok then
 try
  LockDraw;
  try
   repeat
    f:=true;
    for i:=0 to ManFit.NumPeaks-1 do begin
     with ManFit.CurrParam.Peak[i] do
     if (Chan<=MarkerL) or (Chan>=MarkerR) then begin
      ManFitPeakDelete(i);
      f:=false;
      break;
     end;
    end;
   until f;
   ManFitSort;
   if ManFitInitInterval then RoiMark else ManFitClear;
   ManFit.FitResult:=-1;
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  UnLockDraw;
  ManFitControlUpdate(1+2);
 end;
end;


procedure TFormSpectrManFitWindow.ManFitCallPeakin;
var i:integer; GroundPoly:TPolynom; L,R:LongInt;
begin
 if NoProblem(ManFit.Left<ManFit.Right,'Не инициализирована ROI!')
 then
 try
  LockDraw;
  try
   L:=MarkerL;
   R:=MarkerR;
   MarkerL:=ManFit.Left;
   MarkerR:=ManFit.Right;
   PeakClear;
   PeakSearch;
   MarkerL:=L;
   MarkerR:=R;
   GroundPoly:=nil;
   if Peak.NumPeaks>0 then begin
    ManFit.NumPeaks:=min(Peak.NumPeaks,MaxManPeaks);
    for i:=0 to ManFit.NumPeaks-1 do begin
     ManFit.CurrParam.Peak[i].Chan:=Peak.Channel[i];
     ManFit.CurrParam.Peak[i].Ampl:=Peak.Amplitude[i];
     ManFit.CurrParam.Peak[i].FWHM:=Peak.FWHM[i];
    end;
    GroundPoly:=TPolynom.Create(1,ManFit.Left,ManFit.Right-ManFit.Left+1);
    if GroundPoly.Find(Double(Peak.Channel.ItemPtr[0]^),Double(Peak.Ground.ItemPtr[0]^),Peak.NumPeaks) then begin
     for i:=0 to GroundPoly.Power do begin
       ManFit.CurrParam.Poly[i]:=GroundPoly.Coeff[i];
       ManFit.CurrFixed.Poly[i]:=false;
     end;
    end;
   end;
   Kill(GroundPoly);
   Kill(ManFit.Covar);
   ManFit.FitResult:=-1;
   PeakClear;
   ManFitSort;
   if Peak.Control.Ok then Peak.Control.Hide;
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  UnLockDraw;
  ManFitControlUpdate(1+2);
 end;
end;

procedure TFormSpectrManFitWindow.ManFitPeakInsert(Channel:LongInt);
 function PrepareFit:Boolean;
 begin
  Result:=false;
  ManFitClear;
  if ManFitInitInterval then begin
   Result:=true;
   ManFitControlUpdate(1+2);
   ROIMark;
  end;
 end;
begin
 if Ok then
 try
  LockDraw;
  try
   if (ManFit.NumPeaks>0) or PrepareFit then begin
    if (Channel>ManFit.Left) and (Channel<ManFit.Right) then begin
     if ManFit.NumPeaks<MaxManPeaks then begin
      with ManFit.CurrParam.Peak[ManFit.NumPeaks] do begin
       Chan:=Channel;
       Ampl:=SpectrValue[Channel]-EvalGround(Chan,ManFit.CurrParam,ManFit.TempGrad);
       FWHM:=HwCalibr(Channel)*1.2;
      end;
      with ManFit.CurrFixed.Peak[ManFit.NumPeaks] do begin
       Chan:=false;
       Ampl:=false;
       FWHM:=false;
      end;
      inc(ManFit.NumPeaks);
      Kill(ManFit.Covar);
      ManFit.FitResult:=-1;
      ManFitSort;
     end else Daq.Report(RusEng('Слишком много пиков!','Too many peaks!'));
    end else Daq.Report(RusEng('Маркер за пределами РОИ!','Marker out of ROI!'));
   end else Daq.Report(RusEng('Не могу инициализировать пики!','Could not init peaks!'));
  except
   on E:Exception do Daq.Report(E.Message);
  end;
 finally
  UnlockDraw;
  ManFitControlUpdate(1+2);
 end;
end;

procedure TFormSpectrManFitWindow.ManFitPeakDelete(Num:integer);
var j:integer;
begin
 if Ok and (Num>=0) and (Num<ManFit.NumPeaks) then
 try
  for j:=Num+1 to ManFit.NumPeaks-1 do begin
   with ManFit.CurrParam do Peak[j-1]:=Peak[j];
   with ManFit.CurrFixed do Peak[j-1]:=Peak[j];
  end;
  ManFit.NumPeaks:=max(0,ManFit.NumPeaks-1);
  ManFit.FitResult:=-1;
  Kill(ManFit.Covar);
  ManFitSort;
  ManFitControlUpdate(1+2);
  DrawView;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

function TFormSpectrManFitWindow.ManFitFindNearest(Ch:double):integer;
var m,c:double; i:integer;
begin
 Result:=0;
 if Ok then
 try
  m:=1E100;
  for i:=0 to ManFit.NumPeaks-1 do begin
   c:=abs(Ch-ManFit.CurrParam.Peak[i].Chan);
   if m>c then begin
    m:=c;
    Result:=i;
   end;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end; 
end;

function TFormSpectrManFitWindow.ManFitDetailsList(P:TText):TText;
var Area,Phone,Total,Chi2,Sum,Defect,Criteria:Double;
var s:LongString; i,StepSvob:Integer;
begin
 Result:=P;
 if Ok and P.Ok then
 try
  {интеграл под фоном}
  Phone:=0;
  for i:=ManFit.Left to ManFit.Right-1 do
  Phone:=Phone+Trapezium(i,   EvalGround(i,   ManFit.CurrParam, ManFit.TempGrad),
                         i+1, EvalGround(i+1, ManFit.CurrParam, ManFit.TempGrad));
  {полный интеграл}
  Total:=0;
  for i:=ManFit.Left to ManFit.Right-1 do
  Total:=Total+Trapezium(i, SpectrValue[i], i+1, SpectrValue[i]);
  {корень из Хи-квадрат}
  Chi2:=ManFit.FunValue;
  {число степеней свободы для критерия Хи-квадрат}
  StepSvob:=(ManFit.Right-ManFit.Left+1)-NumUnFixed(ManFit.CurrFixed.P,ManPeakDim*ManFit.NumPeaks+MaxManPoly);
  {критерий Хи-квадрат по уровню 0.05}
  Criteria:=Chi2CumDistrInv(1-0.05,StepSvob);
  {формирование таблицы}
  if ManFit.FitResult<>0
  then P.Addln(RusEng('Минимизация не проводилась или не была успешной.',
                      'Minimization was not call or was unsuccessful.'))
  else begin
   P.Addln(RusEng('МИНИМИЗАЦИЯ:','MINIMIZATION:'));
   P.Addln(RusEng('************','*************'));
   if Chi2<Criteria
   then P.Addln(RusEng('Фитирование удовлетворительное','Fit result is good'))
   else P.Addln(RusEng('Фитирование НЕ удовлетворительное','Fit result is NOT good'));
   P.Addln(Format(RusEng('Chi^2=%g  ( для сравнения Chi^2(0.05)=%g )',
                         'Chi^2=%g  ( compare with  Chi^2(0.05)=%g )'),[Chi2,Criteria]));
   P.Addln(Format(RusEng('Число вызовов целевой функции = %d',
                         'Number of obj. function calls = %d'),[ManFit.FunCalls]));
   P.Addln(RusEng('Минимизируемая функция: сумма квадратов с весами 1/Cnt',
                  'Objective function: summ of squares with 1/Cnt weights'));
   P.Addln(' Chi^2=sum( sqr(Fit[i]-Cnt[i])/Cnt[i], i=1..N )');
   P.Addln(RusEng('ПАРАМЕТРЫ ПИКОВ:','PEAK PARAMETERS:'));
   P.Addln(RusEng('****************','****************'));
   Sum:=0;
   for i:=0 to ManFit.NumPeaks-1 do
   with ManFit.CurrParam.Peak[i] do begin
    Area:=GaussPeakArea(Ampl,FWHM);
    Sum:=Sum+Area;
    P.Addln(Format(RusEng('ПИК НОМЕР %d :','PEAK NUMBER %d :'),[i+1]));
    P.Addln(Format('Channel = %14.3f   Amplitude = %14.3f',[Chan,Ampl]));
    P.Addln(Format('FWHM    = %14.3f   Area      = %14.3f',[FWHM,Area]));
   end;
   P.Addln(RusEng('ПЛОЩАДИ И ФОН:','AREA AND PHONE'));
   P.Addln(RusEng('**************','**************'));
   P.Addln(Format(RusEng('Общая   = %14.3f  Под пиками (реал.) = %14.3f',
                         'Total   = %14.3f  Under peaks (real) = %14.3f'),[Total,Total-Phone]));
   P.Addln(Format(RusEng('Фоновая = %14.4f  Под пиками (теор.) = %14.3f',
                         'Phone   = %14.3f  Under peaks  (fit) = %14.3f'),[Phone, Sum]));
   {дефект площади}
   defect:=abs(Total-Phone-Sum);
   P.Addln(Format(RusEng('Разница реал. и теор. площади = %g или %7.3f %s общей площади',
                         'Theor. & real area difference = %g or  %7.3f %s of total area'),
                         [defect,100*defect/total,'%']));
   P.Addln('*');
   P.Addln(RusEng('Фон описывается полиномом:','Phone polynom is:'));
   P.Addln(Format('Ground( Z=(Channel-%d)/%d )=',[ManFit.Left,ManFit.Right-ManFit.Left+1]));
   for i:=0 to MaxManPoly-1 do begin
    s:=Format('%g',[ManFit.CurrParam.Poly[i]]);
    if (s[1]<>'-') then s:='+'+s;
    s:=LeftPad(s,20);
    case i of
     0:;
     1:s:=s+'*Z';
     else s:=s+'*Z^'+d2s(i);
    end;
    P.Addln(s);
   end;
   P.Addln(Format(RusEng('Фон слева  = %14.3f  в точке  %d',
                         'Phone left = %14.3f  in point %d'),
                         [EvalGround(ManFit.Left,ManFit.CurrParam,ManFit.TempGrad),ManFit.Left]));
   P.Addln(Format(RusEng('Фон справа = %14.3f  в точке  %d',
                         'Phone right= %14.3f  in point %d'),
                         [EvalGround(ManFit.Right,ManFit.CurrParam,ManFit.TempGrad),ManFit.Right]));
   P.Addln('*');
   ManFitCovarList(P);
  end;
 except
  on E: Exception do Daq.Report(E.Message);
 end;
end;

function TFormSpectrManFitWindow.ManFitCovarList(P:TText):TText;
var i,j,n:Integer; si,sj,cor:Double; ss,sv:LongString;
 function getname(i:integer):LongString;
 var s:LongString;
 begin
  if (i<MaxManPoly)
  then s:='GndC('+d2s(i)+')'
  else begin
   s:=Copy(ManFit.FieldName[(i-MaxManPoly) mod ManPeakDim],1,4)+
           '('+d2s(1+(i-MaxManPoly) div ManPeakDim)+')';
   if (pos('AMPL',UpcaseStr(s))>0)
   then s:='Area'+'('+d2s(1+(i-MaxManPoly) div ManPeakDim)+')';
  end;
  Result:=s;
 end;
begin
 Result:=P;
 if Ok and ManFit.Covar.Ok then
 try
  {конвертируем амплитуду в площадь так как ковариация вычислена для площади}
  ManFit.WorkParam:=ManFit.CurrParam;
  for i:=0 to ManFit.NumPeaks-1 do
  with ManFit.WorkParam.Peak[i] do Ampl:=GaussPeakArea(Ampl,FWHM);
  P.Addln(RusEng('Ковариационная матрица:','Covariation matrix:'));
  P.Addln(RusEng('***********************','*******************'));
  n:=ManPeakDim*ManFit.NumPeaks+MaxManPoly;
  for i:=0 to n-1 do begin
   si:=sqrt(ManFit.Covar[i,i]);
   if si=0 then continue;
   ss:='Sigma( '+getname(i)+' ) = '+Pad(f2s(si),15);
   if (i>=MaxManPoly) and (abs(ManFit.WorkParam.P[i])>0) then begin
    str(100*si/abs(ManFit.WorkParam.P[i]):8:3,sv);
    ss:=ss+' = '+sv+' %';
   end;
   P.Addln(ss);
  end;
  for i:=0 to n-1 do begin
   si:=sqrt(ManFit.Covar[i,i]);
   if si=0 then continue;
   for j:=0 to i-1 do begin
    sj:=sqrt(ManFit.Covar[j,j]);
    if sj=0 then continue;
    cor:=ManFit.Covar[i,j]/si/sj;
    str(cor:6:3,sv);
    P.Addln('Correlation( '+getname(i)+' , '+getname(j)+' ) = '+sv);
   end;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

 {
 Сохранение параметров пиков в файле в режиме Append  или OverWrite
 }
procedure TFormSpectrManFitWindow.ManFitSavePik(FName:LongString; isAppend:Boolean);
var f:System.Text; i:integer; Channel,Energy,Amplitude,FWHM,Area,Ground:Double;
var LeftBound,LeftGround,RightBound,RightGround,StatError:Double;
begin
 if Ok then
 if (ManFit.NumPeaks>0) then
 try
  FName:=UnifyFileAlias(FName);
  isAppend:=isAppend and FileExists(FName);
  System.Assign(f,FName);
  if isAppend then System.Append(f) else begin
   System.Rewrite(f);
   {заголовок}
   System.Write(f,'Channel':14,ASCII_HT);
   System.Write(f,'Energy':14,ASCII_HT);
   System.Write(f,'Amplitude':14,ASCII_HT);
   System.Write(f,'FWHM':14,ASCII_HT);
   System.Write(f,'Area':14,ASCII_HT);
   System.Write(f,'Ground':14,ASCII_HT);
   System.Write(f,'LeftBound':14,ASCII_HT);
   System.Write(f,'LeftGround':14,ASCII_HT);
   System.Write(f,'RightBound':14,ASCII_HT);
   System.Write(f,'RightGround':14,ASCII_HT);
   System.Write(f,'StatError':14,ASCII_HT);
   System.Writeln(f);
  end;
  {таблица}
  for i:=0 to ManFit.NumPeaks-1 do begin
   Channel:=ManFit.CurrParam.Peak[i].Chan;
   Energy:=EnCalibr(Channel);
   Amplitude:=ManFit.CurrParam.Peak[i].Ampl;
   FWHM:=ManFit.CurrParam.Peak[i].FWHM;
   Area:=GaussPeakArea(Amplitude,FWHM);
   Ground:=EvalGround(Channel,ManFit.CurrParam,ManFit.TempGrad);
   LeftBound:=Channel-FWHM*3;
   LeftGround:=EvalGround(LeftBound,ManFit.CurrParam,ManFit.TempGrad);
   RightBound:=Channel+FWHM*3;
   RightGround:=EvalGround(RightBound,ManFit.CurrParam,ManFit.TempGrad);
   if ManFit.Covar.Ok
   then StatError:=sqrt(ManFit.Covar[MaxManPoly+i*ManPeakDim+1,MaxManPoly+i*ManPeakDim+1])
   else StatError:=0;
   System.Write(f,Channel:14:3,ASCII_HT);
   System.Write(f,Energy:14:3,ASCII_HT);
   System.Write(f,Amplitude:14:3,ASCII_HT);
   System.Write(f,FWHM:14:3,ASCII_HT);
   System.Write(f,Area:14:3,ASCII_HT);
   System.Write(f,Ground:14:3,ASCII_HT);
   System.Write(f,LeftBound:14:3,ASCII_HT);
   System.Write(f,LeftGround:14:3,ASCII_HT);
   System.Write(f,RightBound:14:3,ASCII_HT);
   System.Write(f,RightGround:14:3,ASCII_HT);
   System.Write(f,StatError:14:3,ASCII_HT);
   System.Writeln(f);
  end;
  System.Close(f);
  if (IOResult<>0) then Daq.Report(RusEng('Ошибка сохранения ','Error save ')+FName);
 except
  on E: Exception do Daq.Report(E.Message);
 end else Daq.Report(RusEng('Нет пиков в списке!','Have no peaks to save!'));
end;


procedure TFormSpectrManFitWindow.ActionSpectrManFitControlExecute(Sender: TObject);
begin
 inherited;
 ManFitControlOpen;
 ManFitControlUpdate($20);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrmanfitwindow;
begin
 RegisterSpectrWindowConstructor(NewSpecManFitWin, regSpecManFitWin);
end;

procedure Free_form_spectrmanfitwindow;
begin
end;

initialization

 Init_form_spectrmanfitwindow;

finalization

 Free_form_spectrmanfitwindow;

end.

//////////////
// END OF FILE
//////////////

