////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWDAQ.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Create new projects from sample.                                           //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241022 - Created by A.K.                                                 //
////////////////////////////////////////////////////////////////////////////////

unit form_createnewprojectfromsample; // Create new project from sample

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface


uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_ListBoxSelection,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_proc,
 _crw_dynar, _crw_snd, _crw_guard, _crw_syscal, _crw_gloss,
 _crw_crwapi, _crw_crwapiserver, _crw_fpcup, _crw_sect, _crw_sesman,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormCreateNewProjectFromSample }

  TFormCreateNewProjectFromSample = class(TMasterForm)
    BitBtnCancel: TBitBtn;
    BitBtnCopyCmdLine: TBitBtn;
    BitBtnOk: TBitBtn;
    BitBtnOpenProject: TBitBtn;
    BitBtnTargetDirDan: TBitBtn;
    BitBtnTargetDirWorks: TBitBtn;
    BitBtnTargetDir: TBitBtn;
    CheckBoxPrefixRun: TCheckBox;
    CheckBoxRelative: TCheckBox;
    ComboBoxProjectType: TComboBox;
    EditNewProjectName: TEdit;
    EditTargetDir: TEdit;
    GroupBoxCmdLine: TGroupBox;
    GroupBoxSamples: TGroupBox;
    ImageList16: TImageList;
    LabelListSamples: TLabel;
    LabelNewProjectName: TLabel;
    LabelProjectType: TLabel;
    ListBoxSamples: TListBox;
    MemoCmdLine: TMemo;
    PanelButtons: TPanel;
    PanelListBoxSamples: TPanel;
    PanelProjectTypes: TPanel;
    PanelRelative: TPanel;
    PanelSamples: TPanel;
    TimerUpdater: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure BitBtnCopyCmdLineClick(Sender: TObject);
    procedure BitBtnTargetDirClick(Sender: TObject);
    procedure BitBtnTargetDirWorksClick(Sender: TObject);
    procedure BitBtnOpenProjectClick(Sender: TObject);
    procedure BitBtnTargetDirDanClick(Sender: TObject);
    procedure ComboBoxProjectTypeChange(Sender: TObject);
    procedure EditNewProjectNameChange(Sender: TObject);
    procedure EditTargetDirChange(Sender: TObject);
    procedure ListBoxSamplesClick(Sender: TObject);
    procedure CheckBoxPrefixRunChange(Sender: TObject);
    procedure CheckBoxRelativeChange(Sender: TObject);
    procedure TimerUpdaterTimer(Sender: TObject);
  private
    myProjectMaker:LongString;
    function  PrefixRun:LongString;
    function  GetProjectType:LongString;
    function  GetTargetDir:LongString;
    procedure SetTargetDir(Dir:LongString);
    function  GetNewProjectName:LongString;
    function  FixCase(const s:LongString):LongString;
  public
    property  TargetDir:LongString read GetTargetDir write SetTargetDir;
    property  NewProjectName:LongString read GetNewProjectName;
    property  ProjectType:LongString read GetProjectType;
    function  ProjectPathBySample(sam:LongString):LongString;
    function  ProjectNoteBySample(sam:LongString):LongString;
    function  CmdLineBySample(sam:LongString; rel:Boolean):LongString;
    function  DefaultListOfProjectType:LongString;
    function  GetCmdLine(rel:Boolean):LongString;
    function  ListOfProjectType:LongString;
    function  DefaultTargetDir:LongString;
    function  HasSelectedSample:Boolean;
    function  SelectedSample:LongString;
    function  ProjectMaker:LongString;
    function  TargetDirDan:LongString;
    procedure OpenProjectInTargetDir;
    procedure UpdateNewProjectName;
    procedure UpdateListOfSamples;
    procedure UpdateUserControls;
    procedure UpdateCmdLine;
    procedure RunCmdLine;
  end;

function RunCreateNewProjectFromSample(const Params:LongString=''):TModalResult;

implementation

{$R *.lfm}

function RunCreateNewProjectFromSample(const Params:LongString=''):TModalResult;
const TheForm:TFormCreateNewProjectFromSample=nil;
begin
 Result:=mrCancel;
 if CanShowModal(TheForm) then
 try
  if not TheForm.Ok then begin
   Application.CreateForm(TFormCreateNewProjectFromSample,TheForm);
   TheForm.Master:=@TheForm;
  end;
  if TheForm.Ok then with TheForm do begin
   UpdateUserControls;
   UpdateListOfSamples;
   UpdateNewProjectName;
   UpdateCmdLine;
   TimerUpdater.Enabled:=true;
   if (Params<>'') then ApplyParams(Params);
   if (ShowModal=mrOk) then begin
    UpdateCmdLine;
    RunCmdLine;
    Result:=mrOk;
   end;
   TimerUpdater.Enabled:=false;
  end;
 except
  on E:Exception do BugReport(E,nil,'RunCreateNewProjectFromSample');
 end;
end;


{ TFormCreateNewProjectFromSample }

procedure TFormCreateNewProjectFromSample.FormCreate(Sender: TObject);
begin
 inherited;
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
 ComboBoxProjectType.Items.Text:=ListOfProjectType;
 ComboBoxProjectType.ItemIndex:=Min(0,ComboBoxProjectType.Items.Count-1);
 UpdateUserControls; ProjectMaker; TargetDir:=DefaultTargetDir;
 UpdateListOfSamples; UpdateNewProjectName; UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.FormDestroy(Sender: TObject);
begin
 myProjectMaker:='';
 inherited;
end;

function TFormCreateNewProjectFromSample.FixCase(const s:LongString):LongString;
begin
 if (s<>'') and not FileNameCaseSensitive
 then Result:=LowerCase(s)
 else Result:=s;
end;

function  TFormCreateNewProjectFromSample.GetCmdLine(rel:Boolean):LongString;
begin
 if not Assigned(Self) then Exit('');
 Result:=CmdLineBySample(SelectedSample,rel);
end;

function  TFormCreateNewProjectFromSample.GetTargetDir:LongString;
begin
 if not Assigned(Self) then Exit('');
 Result:=FixCase(Trim(EditTargetDir.Text));
end;

procedure TFormCreateNewProjectFromSample.SetTargetDir(Dir:LongString);
begin
 if not Assigned(Self) then Exit;
 EditTargetDir.Text:=FixCase(UnifyFileAlias(Dir,ua_RealPath));
end;

function TFormCreateNewProjectFromSample.GetProjectType:LongString;
begin
 Result:=Trim(ComboBoxProjectType.Text);
end;

procedure TFormCreateNewProjectFromSample.ComboBoxProjectTypeChange(Sender: TObject);
begin
 if not Assigned(Self) then Exit;
 UpdateListOfSamples;
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.CheckBoxRelativeChange(Sender: TObject);
begin
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.CheckBoxPrefixRunChange(Sender: TObject);
begin
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.TimerUpdaterTimer(Sender: TObject);
begin
 SetEnabledControls(HasSelectedSample,[BitBtnOk,BitBtnOpenProject,BitBtnCopyCmdLine]);
end;

procedure TFormCreateNewProjectFromSample.EditNewProjectNameChange(Sender: TObject);
begin
 if not Assigned(Self) then Exit;
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.EditTargetDirChange(Sender: TObject);
begin
 if not Assigned(Self) then Exit;
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.BitBtnTargetDirWorksClick(Sender: TObject);
begin
 if not Assigned(Self) then Exit;
 TargetDir:=DefaultTargetDir;
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.BitBtnTargetDirClick(Sender: TObject);
var Cap,Dir:String;
begin
 if not Assigned(Self) then Exit;
 Cap:=RusEng('Выбрать каталог …','Select Directory …');
 if SelectDirectory(Cap,AddPathDelim(TargetDir)+'.',Dir)
 then TargetDir:=UnifyFileAlias(Dir,ua_RealPath);
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.BitBtnTargetDirDanClick(Sender: TObject);
begin
 if not Assigned(Self) then Exit;
 TargetDir:=TargetDirDan;
 UpdateCmdLine;
end;

procedure TFormCreateNewProjectFromSample.BitBtnCopyCmdLineClick(Sender: TObject);
begin
 if not Assigned(Self) then Exit;
 try
  UpdateCmdLine;
  Clipboard.AsText:=Trim(MemoCmdLine.Text);
 except
  on E:Exception do BugReport(E,Self,'BitBtnCopyCmdLineClick');
 end;
end;

procedure TFormCreateNewProjectFromSample.BitBtnOpenProjectClick(Sender: TObject);
begin
 OpenProjectInTargetDir;
end;

procedure TFormCreateNewProjectFromSample.OpenProjectInTargetDir;
var list,cap,tit,lpr,msg,Params:LongString; n:Integer;
begin
 if not Assigned(Self) then Exit;
 if IsEmptyStr(ProjectType) then Exit;
 cap:=''; tit:=''; lpr:=''; msg:=''; Params:='';
 list:=FindAllFilesAsText(TargetDir,'*.'+ProjectType,True);
 if IsNonEmptyStr(list) then begin
  cap:=RusEng('Открыть Проект …','Open Project …');
  tit:=RusEng('Выбор Проекта …','Select Project …');
  Params:=ControlPosParams(ListBoxSamples,'LT');
  n:=ListBoxMenu(cap,tit,ValidateEol(list),0,Params);
  if (n>=0) then lpr:=ExtractWord(n+1,list,EolnDelims);
  if IsNonEmptyStr(lpr) then begin
   if FileExists(lpr)
   then SendToMainConsole('@silent @open -e '+AnsiQuotedIfNeed(lpr)+EOL);
   if (Application.ModalLevel>0) then ModalResult:=mrCancel;
  end;
 end else begin
  msg:=RusEng(ProjectType+' проекты не найдены.',ProjectType+' projects not found.');
  SendToMainConsole(Format('@silent @tooltip text "%s: %s" preset stdWarning delay 30000',
                   [SessionManager.TitlePidAtHost,msg])+EOL);
  Echo(msg);
 end;
end;

procedure TFormCreateNewProjectFromSample.ListBoxSamplesClick(Sender: TObject);
begin
 if not Assigned(Self) then Exit;
 UpdateNewProjectName;
 UpdateCmdLine;
end;

function TFormCreateNewProjectFromSample.DefaultListOfProjectType:LongString;
begin
 Result:='lpr'+EOL;
end;

function TFormCreateNewProjectFromSample.ListOfProjectType:LongString;
var txt:TText;
begin
 Result:='';
 if Assigned(Self) then
 try
  txt:=ExtractEnumWordList(SysIniFile,SectProjectSamples,'ProjectType',efConfigNC);
  try
   Result:=txt.Text;
   if (Result='') then Result:=DefaultListOfProjectType;
  finally
   Kill(txt);
  end;
 except
  on E:Exception do BugReport(E,Self,'ListOfProjectType');
 end;
end;

procedure TFormCreateNewProjectFromSample.UpdateUserControls;
begin
 if not Assigned(Self) then Exit;
 Caption:=RusEng('Создать новый Проект по Образцу','Create new Project from Sample');
 GroupBoxSamples.Caption:=RusEng('Список Образцов и Параметры для Нового Проекта:',
                                 'List of Samples and Parameters of New Project:');
 LabelListSamples.Caption:=RusEng('Список Образцов для сознания новых Проектов:',
                                  'List of Samples to create new Projects:');
 LabelProjectType.Caption:=RusEng('Тип Проекта:','Project Type:');
 LabelNewProjectName.Caption:=RusEng('Имя Нового Проекта:','Name of New Project:');
 BitBtnTargetDir.Caption:=RusEng('Целевой Каталог …','Tagret Directory …');
 BitBtnTargetDir.Hint:=RusEng('Выбрать Целевой Каталог Проекта.',
                              'Select Project Target Directory.');
 BitBtnTargetDirWorks.Caption:=ExtractBaseName(DefaultTargetDir);
 BitBtnTargetDirWorks.Hint:=RusEng('Вернуть Целевой Каталог Проекта по умолчанию.',
                                   'Set Default Project Target Direcory.');
 BitBtnTargetDirDan.Caption:=ExtractBaseName(TargetDirDan);
 BitBtnTargetDirDan.Hint:=RusEng('Задать Целевой Каталог для Плагинов Обработки Данных.',
                                 'Set Project Target Direcory for Data Analysis Plugins.');
 GroupBoxCmdLine.Caption:=RusEng('Исполняемая Команда:','Command to Run:');
 BitBtnCopyCmdLine.Caption:=RusEng('Копировать Команду','Copy Command');
 BitBtnCopyCmdLine.Hint:=RusEng('Копировать Исполняемую Команду в Буфер Обмена.',
                                'Copy Command Line to Clipboard.');
 BitBtnCancel.Hint:=RusEng('Отменить и закрыть диалог.',
                           'Cancel and close dialog.');
 BitBtnOk.Hint:=RusEng('Запустить Команду Создания Нового Проекта.',
                       'Run Command Line to Create New Project.');
 BitBtnOpenProject.Hint:=RusEng('Открыть ранее созданный Проект …',
                                'Open a previously created Project …');
 BitBtnOpenProject.Caption:=RusEng('Открыть Проект …','Open Project …');
 CheckBoxRelative.Caption:=RusEng('Задать пути относительно каталога программы',
                                  'Use file path relative program home dir');
 CheckBoxPrefixRun.Caption:=RusEng('Добавлять префикс @run …',
                                   'Add prefix @run …');
end;

function TFormCreateNewProjectFromSample.PrefixRun:LongString;
begin
 Result:='@run -sw7 -cd ~~ ';
end;

procedure TFormCreateNewProjectFromSample.UpdateCmdLine;
begin
 if not Assigned(Self) then Exit;
 MemoCmdLine.Text:=IfThen(CheckBoxPrefixRun.Checked,PrefixRun,'')
                  +GetCmdLine(CheckBoxRelative.Checked);
end;

procedure TFormCreateNewProjectFromSample.RunCmdLine;
var cmd:LongString;
begin
 if not Assigned(Self) then Exit;
 cmd:=GetCmdLine(true);
 if IsNonEmptyStr(cmd) then SendToMainConsole(PrefixRun+cmd+EOL);
end;

procedure TFormCreateNewProjectFromSample.UpdateListOfSamples;
var typ,sam,ext,note,line:LongString; txt:TText; i,n:Integer;
const spaces='  =  ';
begin
 if Assigned(Self) then
 try
  typ:=ProjectType;
  if (typ<>'') then begin
   ListBoxSamples.Clear;
   ListBoxSamples.Items.BeginUpdate;
   txt:=ExtractEnumWordList(SysIniFile,SectProjectSamples,'ProjectSample',efConfigNC);
   try
    for i:=txt.Count-1 downto 0 do begin
     sam:=ExtractWord(1,txt[i],ScanSpaces); ext:=ExtractFileExt(sam);
     if IsSameText('.'+typ,ext) then Continue;
     txt.DelLn(i);
    end;
    n:=1;
    for i:=txt.Count-1 downto 0 do begin
     sam:=ExtractWord(1,txt[i],ScanSpaces);
     note:=Trim(ProjectNoteBySample(sam));
     if (note<>'')
     then txt[i]:=sam+' '+note
     else begin txt.DelLn(i); continue; end;
     n:=Max(n,Length(sam));
    end;
    for i:=0 to txt.Count-1 do begin
     line:=Trim(txt[i]);
     sam:=Pad(ExtractWord(1,line,ScanSpaces),n);
     note:=SkipWords(1,line,ScanSpaces);
     line:=FixCase(sam)+spaces+note;
     ListBoxSamples.Items.Add(line);
    end;
    UpdateNewProjectName;
    UpdateCmdLine;
   finally
    Kill(txt);
    ListBoxSamples.Items.EndUpdate;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ListOfProjectType');
 end;
end;

function TFormCreateNewProjectFromSample.ProjectMaker:LongString;
var cmd:LongString;
begin
 if (myProjectMaker='') then begin
  cmd:='';
  if SysGlossary.ReadIniPath(SysIniFile,SectProjectSamples,'ProjectMaker',HomeDir,cmd)
  then myProjectMaker:=FixCase(UnifyFileAlias(AdaptExeFileName(cmd),ua_RealPath));
 end;
 Result:=myProjectMaker;
end;

function TFormCreateNewProjectFromSample.HasSelectedSample:Boolean;
begin
 if not Assigned(Self) then Exit(false);
 Result:=(ListBoxSamples.Count>0) and (ListBoxSamples.ItemIndex>=0);
end;

function TFormCreateNewProjectFromSample.SelectedSample:LongString;
var i:Integer;
begin
 if not Assigned(Self) then Exit('');
 Result:=''; i:=ListBoxSamples.ItemIndex;
 if InRange(i,0,ListBoxSamples.Count-1) then Result:=ListBoxSamples.Items[i];
 if (Result<>'') then Result:=ExtractWord(1,Result,ScanSpaces);
end;

procedure TFormCreateNewProjectFromSample.UpdateNewProjectName;
begin
 if Assigned(Self) then
 try
  EditNewProjectName.Text:=Trim(ExtractBaseName(SelectedSample));
 except
  on E:Exception do BugReport(E,Self,'UpdateNewProjectName');
 end;
end;

function TFormCreateNewProjectFromSample.DefaultTargetDir:LongString;
var Path:LongString;
begin
 Result:=''; Path:='';
 if (Self=nil) then Exit;
 if SysGlossary.ReadIniPath(SysIniFile,SectProjectSamples,'TargetDir',HomeDir,path)
 then Result:=FixCase(UnifyFileAlias(Path,ua_RealPath));
end;

function TFormCreateNewProjectFromSample.TargetDirDan:LongString;
var Path:LongString;
begin
 Result:=''; Path:='';
 if (Self=nil) then Exit;
 if SysGlossary.ReadIniPath(SysIniFile,SectProjectSamples,'TargetDirDan',HomeDir,path)
 then Result:=FixCase(UnifyFileAlias(Path,ua_RealPath));
end;

function TFormCreateNewProjectFromSample.GetNewProjectName:LongString;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=LowerCase(Trim(EditNewProjectName.Text));
end;

function TFormCreateNewProjectFromSample.ProjectPathBySample(sam:LongString):LongString;
var Path:LongString;
begin
 Result:='';
 if (Self=nil) then Exit;
 Path:=''; if (sam='') then Exit;
 if SysGlossary.ReadIniPath(SysIniFile,SectProjectSamples(0,sam),'ProjectPath',HomeDir,path)
 then Result:=FixCase(UnifyFileAlias(Path,ua_RealPath));
end;

function TFormCreateNewProjectFromSample.ProjectNoteBySample(sam:LongString):LongString;
var Note:LongString;
begin
 Result:='';
 if (Self=nil) then Exit;
 if (sam='') then Exit; Note:='';
 if SysGlossary.ReadIniString(SysIniFile,SectProjectSamples(0,sam),'ProjectNote_'+RusEng('ru','en'),note)
 then Result:=Trim(Note);
end;

function TFormCreateNewProjectFromSample.CmdLineBySample(sam:LongString; rel:Boolean):LongString;
var exe,lpr,dir,nam:LongString;
begin
 Result:='';
 if (Self=nil) then Exit; if (sam='') then Exit;
 exe:=ProjectMaker;
 lpr:=ProjectPathBySample(sam);
 dir:=TargetDir;
 nam:=Trim(NewProjectName);
 if rel then exe:=FixCase(MakeRelativePath(exe,ProgName));
 if rel then lpr:=FixCase(MakeRelativePath(lpr,ProgName));
 if rel then dir:=FixCase(MakeRelativePath(dir,ProgName));
 Result:=QArg(exe)+' '+QArg(lpr)+' '+QArg(dir)+' '+nam;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_createnewprojectfromsample;
begin
end;

procedure Free_form_createnewprojectfromsample;
begin
end;

initialization

 Init_form_createnewprojectfromsample;

finalization

 Free_form_createnewprojectfromsample;

end.

//////////////
// END OF FILE
//////////////

