////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Secret Service.                                                       //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231202 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_secretservice; // Form Secret Service

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, strutils,
 Graphics, Interfaces, LMessages, lclintf,
 Forms, Controls, StdCtrls, Dialogs,
 Spin, ExtCtrls, ComCtrls, Buttons, {DbLogDlg,}
 ActnList, Menus, ToolWin, ImgList, Printers, Clipbrd, FileUtil,
 _crw_alloc, _crw_str, _crw_eldraw, _crw_guard,
 _crw_polling, _crw_fio, _crw_rtc, _crw_sesman,
 _crw_appforms, _crw_apptools, _crw_apputils,
 _crw_task, _crw_crypt, _crw_base64, _crw_sect,
 _crw_lm, _crw_snd, _crw_ef, _crw_colors,
 Unit_SystemConsole, Form_TextEditor,
 Form_LoginDialog, Form_ListBoxSelection,
 Form_CrwDaqLogo;

type
  TFormSecretService = class(TMasterForm)
    PanelControls: TPanel;
    PageControlTools: TPageControl;
    TabSheetAccount: TTabSheet;
    OpenDialogAccount: TOpenDialog;
    PanelAccountControls: TPanel;
    LabelAccountFileName: TLabel;
    EditAccountFileName: TEdit;
    BitBtnAccountFileOpen: TBitBtn;
    LabelAccountUser: TLabel;
    EditAccountUser: TEdit;
    LabelAccountDomain: TLabel;
    EditAccountDomain: TEdit;
    LabelAccountPassword1: TLabel;
    EditAccountPassword1: TEdit;
    LabelAccountPassword2: TLabel;
    EditAccountPassword2: TEdit;
    ButtonAccountTest: TButton;
    ButtonAccountGenerate: TButton;
    LabelAccountKey: TLabel;
    EditAccountKey: TEdit;
    TabSheetAuthorization: TTabSheet;
    PanelAuthorizationControls: TPanel;
    GroupBoxWantedAccessLevel: TGroupBox;
    BitBtnWantsGuest: TBitBtn;
    BitBtnWantsUser: TBitBtn;
    BitBtnWantsRoot: TBitBtn;
    MemoWantsGuest: TMemo;
    MemoWantsUser: TMemo;
    MemoWantsRoot: TMemo;
    BitBtnGuestPassword: TBitBtn;
    BitBtnUserPassword: TBitBtn;
    BitBtnRootPassword: TBitBtn;
    TimerCurrentLevel: TTimer;
    BitBtnWantsLock: TBitBtn;
    MemoWantsLock: TMemo;
    BitBtnLevelClose: TBitBtn;
    GroupBoxTrustList: TGroupBox;
    BitBtnTrustListAdd: TBitBtn;
    BitBtnTrustListDel: TBitBtn;
    ListBoxTrustList: TListBox;
    OpenDialogTrustList: TOpenDialog;
    BitBtnAccountDomain: TBitBtn;
    BitBtnAccountUser: TBitBtn;
    CheckBoxAccountDomainDetail: TCheckBox;
    CheckBoxAccountUserDetail: TCheckBox;
    ImageAccountWait: TImage;
    SpinEditAccountTimeOut: TSpinEdit;
    LabelAccountTimeOut: TLabel;
    TimerSecurity: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure ButtonAccountTestClick(Sender: TObject);
    procedure BitBtnAccountFileOpenClick(Sender: TObject);
    procedure ButtonAccountGenerateClick(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure TimerCurrentLevelTimer(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure BitBtnWantsGuestClick(Sender: TObject);
    procedure BitBtnGuestPasswordClick(Sender: TObject);
    procedure BitBtnWantsUserClick(Sender: TObject);
    procedure BitBtnUserPasswordClick(Sender: TObject);
    procedure BitBtnWantsRootClick(Sender: TObject);
    procedure BitBtnRootPasswordClick(Sender: TObject);
    procedure BitBtnLevelCloseClick(Sender: TObject);
    procedure BitBtnWantsLockClick(Sender: TObject);
    procedure BitBtnTrustListAddClick(Sender: TObject);
    procedure BitBtnTrustListDelClick(Sender: TObject);
    procedure BitBtnAccountUserClick(Sender: TObject);
    procedure BitBtnAccountDomainClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure TimerSecurityTimer(Sender: TObject);
  private
    { Private declarations }
    procedure Burn;
    procedure UpdateGuardLevel;
    function  AccountAsText:LongString;
    procedure LoginToLevel(aLevel:Cardinal);
    procedure PasswordToLevel(aLevel:Cardinal);
    procedure ShowWaiting(aShow:Boolean);
  public
    { Public declarations }
  end;

procedure OpenFormSecretService(Visible:Boolean);
procedure KillFormSecretService;
procedure ShowLockWarning;

implementation

{$R *.lfm}

const
  FormSecretService: TFormSecretService = nil;

procedure OpenFormSecretService(Visible:Boolean);
begin
 try
  if not FormSecretService.Ok then begin
   Application.CreateForm(TFormSecretService, FormSecretService);
   FormSecretService.Master:=@FormSecretService;
  end;
  if FormSecretService.Ok then begin
   if Visible then begin
    FormSecretService.Burn;
    FormSecretService.Show;
    FormSecretService.BringToFront;
    FormSecretService.PageControlTools.ActivePageIndex:=0;
    FormSecretService.ListBoxTrustList.Items.Text:=Guard.TrustList;
    FormSecretService.ListBoxTrustList.ItemIndex:=Min(0,FormSecretService.ListBoxTrustList.Items.Count-1);
   end else begin
    FormSecretService.Hide;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'OpenFormSecretService');
 end;
end;

procedure KillFormSecretService;
begin
 FormSecretService.Free;
end;

const
 LockTimeOut = 5000;
 LockTime : Double = 0;

procedure HideLockWarning;
begin
 if (LockTime>0) then
 if (msecnow-LockTime>LockTimeOut) then begin
  SecondActions.Remove(HideLockWarning);
  HideFormCrwDaqLogo;
  LockTime:=0;
 end;
end;

procedure ShowLockWarning;
var s:LongString;
begin
 s:='';
 if (Guard.Level=ga_Lock) then
 if ReadIniFilePath(SysIniFile,SectSystem,'LockLogoFile',HomeDir,s) then
 ShowFormCrwDaqLogo('CRWDAQ LOCK WARNING',
                   RusEng('Программа заблокирована! Login = F11.',
                          'Program are blocked now! Login = F11.'),
                   s,true);
 SecondActions.Add(HideLockWarning);
 LockTime:=msecnow;
end;

procedure TFormSecretService.Burn;
begin
 try
  EditAccountPassword1.Text:='';
  EditAccountPassword2.Text:='';
  EditAccountKey.Text:='';
 except
  on E:Exception do BugReport(E,Self,'Burn');
 end;
end;

function  TFormSecretService.AccountAsText:LongString;
begin
 Result:=EditAccountUser.Text+EOL+EditAccountDomain.Text+EOL+EditAccountPassword1.Text;
end;

procedure TFormSecretService.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('Служба Защиты CRW-DAQ','CRW-DAQ Guard Service');
 EditAccountUser.Text:=UserName;
 EditAccountDomain.Text:='localhost';
 EditAccountPassword1.Text:='';
 EditAccountPassword2.Text:='';
 EditAccountPassword1.PasswordChar:=ThePasswordChar;
 EditAccountPassword2.PasswordChar:=ThePasswordChar;
 TimerSecurity.Enabled:=False;
 EditAccountKey.Text:='';
 ShowWaiting(False);
 TabSheetAuthorization.Caption:=RusEng('Авторизация','Authorization');
 GroupBoxWantedAccessLevel.Caption:=RusEng('Желаемый уровень доступа','Wanted access level');
 MemoWantsLock.Lines.Text:=RusEng(
  '"ЗАБЛОКИРОВАН": Запрещены все команды, кроме изменения уровня доступа.',
  '"LOCK": Disable all commands, except login.'
 );
 MemoWantsGuest.Lines.Text:=RusEng(
  '"ГОСТЬ": Можно делать расчеты, графики. Нельзя управлять DAQ-системой.',
  '"GUEST": Enable calculation commands. Disable DAQ system commands.'
 );
 MemoWantsUser.Lines.Text:=RusEng(
  '"ОПЕРАТОР": Можно управлять DAQ-системой. Нельзя менять программы и конфигурации.',
  '"USER": Disable edit programs & configurations, run applications.'
 );
 MemoWantsRoot.Lines.Text:=RusEng(
  '"АДМИНИСТРАТОР": Права не ограничены, все команды разрешены.',
  '"ROOT": unlimited rights, all commands enabled.'
 );
 BitBtnGuestPassword.Caption:=RusEng('Задать'+EOL+'пароль',
                                     'Assign'+EOL+'password');
 BitBtnUserPassword.Caption:=BitBtnGuestPassword.Caption;
 BitBtnRootPassword.Caption:=BitBtnGuestPassword.Caption;
 GroupBoxTrustList.Caption:=RusEng('Список разрешенных конфигураций CRW-DAQ',
                                   'List of enabled CRW-DAQ configurations');
 OpenDialogTrustList.Title:=RusEng('Добавить файлы в список разрешенных',
                                   'Add files to list of enabled files');
end;

procedure TFormSecretService.ButtonAccountTestClick(Sender: TObject);
var tid:Integer; Path:LongString;
begin
 try
  tid:=task_init('');
  try
   path:='';
   ShowWaiting(True);
   if (Length(EditAccountPassword1.Text)=0)
   then RAISE ETask.Create('Password should not be empty string!');
   if (EditAccountPassword1.Text<>EditAccountPassword2.Text)
   then RAISE ETask.Create('Password#1 is not equal to Password#2!');
   if ReadIniFilePath(SysIniFile,'[Dcc32.Exe]','Dcc32BinDir',HomeDir,Path)
   then task_ctrl(tid,'AppName='+AddPathDelim(Path)+'dcc32.exe');
   if (GetMd5FromFile(task_ctrl(tid,'ExeName'),df_Hex)<>'9417865A7CC56D6D95D408F1396CD89E')
   then RAISE ETask.Create('Could not test account due to internal error!');
   task_ctrl(tid,'Account='+task_ctrl(tid,'Encrypt='+AccountAsText));
   task_ctrl(tid,'Display=0');
   if task_run(tid)
   then EditAccountKey.Text:='Account is good to use.'
   else EditAccountKey.Text:='Account is not valid to use.';
  finally
   if (tid<>0) then task_free(tid);
   ShowWaiting(False);
  end;
 except
  on E:Exception do EditAccountKey.Text:=E.Message;
 end;
end;

procedure TFormSecretService.BitBtnAccountFileOpenClick(Sender: TObject);
begin
 try
  if GuardOpenDialog(OpenDialogAccount).Execute
  then EditAccountFileName.Text:=OpenDialogAccount.FileName;
 except
  on E:Exception do EditAccountKey.Text:=E.Message;
 end;
end;

procedure TFormSecretService.ButtonAccountGenerateClick(Sender: TObject);
var tid:Integer;
begin
 try
  tid:=task_init('');
  try
   if not FileExists(EditAccountFileName.Text)
   then RAISE ETask.Create('Executable file not found!');
   if Length(EditAccountPassword1.Text)=0
   then RAISE ETask.Create('Password should not be empty string!');
   if EditAccountPassword1.Text<>EditAccountPassword2.Text
   then RAISE ETask.Create('Password#1 is not equal to Password#2!');
   task_ctrl(tid,'AppName='+EditAccountFileName.Text);
   EditAccountKey.Text:=task_ctrl(tid,'Encrypt='+AccountAsText);
   if IsEmptyStr(EditAccountKey.Text)
   then EditAccountKey.Text:='Could not generate key.';
  finally
   if (tid<>0) then task_free(tid);
  end;
 except
  on E:Exception do EditAccountKey.Text:=E.Message;
 end;
end;

procedure TFormSecretService.FormShow(Sender: TObject);
begin
 TimerSecurity.Enabled:=True;
 TimerCurrentLevel.Enabled:=True;
end;

procedure TFormSecretService.FormHide(Sender: TObject);
begin
 TimerCurrentLevel.Enabled:=False;
 TimerSecurity.Enabled:=False;
 Burn;
end;

procedure TFormSecretService.FormActivate(Sender: TObject);
begin
 UpdateGuardLevel;
 //TimerCurrentLevel.Enabled:=true;
end;

procedure TFormSecretService.FormDeactivate(Sender: TObject);
begin
 //TimerCurrentLevel.Enabled:=false;
 UpdateGuardLevel;
 Hide;
end;

procedure TFormSecretService.UpdateGuardLevel;
const Cnt:Integer=0;
 procedure UpdateColor(Btn:TBitBtn; IsOn:Boolean);
 begin
  if (Btn is TBitBtn) then
  if IsOn then Btn.Font.Color:=clRed else Btn.Font.Color:=clBtnText;
 end;
begin
 if Ok then
 try
  inc(Cnt);
  UpdateColor(BitBtnWantsLock,  (Guard.Level=ga_Lock)  and odd(Cnt));
  UpdateColor(BitBtnWantsGuest, (Guard.Level=ga_Guest) and odd(Cnt));
  UpdateColor(BitBtnWantsUser,  (Guard.Level=ga_User)  and odd(Cnt));
  UpdateColor(BitBtnWantsRoot,  (Guard.Level=ga_Root)  and odd(Cnt));
  SetEnabledControls(Guard.Level=ga_Root,[BitBtnGuestPassword,BitBtnUserPassword,
                                          BitBtnRootPassword,BitBtnTrustListAdd,
                                          BitBtnTrustListDel]);
 except
  on E:Exception do BugReport(E,Self,'UpdateGuardLevel');
 end;
end;

procedure TFormSecretService.TimerCurrentLevelTimer(Sender: TObject);
begin
 UpdateGuardLevel;
end;

procedure TFormSecretService.LoginToLevel(aLevel:Cardinal);
var aUser,aDomain,aPassword,aOriginal:LongString;
begin
 if (Guard.Level<>aLevel) then
 try
  if (Guard.Level<aLevel) then begin
   aPassword:='';
   aDomain:=Guard.Domain;
   aUser:=Guard.LevelName[aLevel];
   aOriginal:=Guard.ReadPassword(aLevel);
   if (Length(aOriginal)>0) then begin
    if FormLoginDialogExecute(RusEng('Введите пароль для уровня ',
                                     'Type password for level ')+aUser,
                              aUser, aDomain, aPassword,ld_ShowUser+ld_ShowDomain)<>mrOk
    then Exit;
    if (aPassword<>aOriginal) then begin
     Warning(RusEng('Пароль неверный.','Incorrect password.'));
     Guard.Log:='Deny '+aUser;
     Exit;
    end;
   end;
  end;
  Guard.Level:=aLevel;
  Close;
  if (Guard.Level=ga_Lock) then ShowLockWarning;
 except
  on E:Exception do BugReport(E,Self,'LoginToLevel');
 end;
end;

procedure TFormSecretService.PasswordToLevel(aLevel:Cardinal);
var aUser,aDomain,aPassword:LongString;
begin
 try
  if aLevel=ga_Lock
  then Warning(RusEng('Для блокировки не нужен пароль.','Do not requires password to lock.'))
  else begin
   aPassword:='';
   aDomain:=Guard.Domain;
   aUser:=Guard.LevelName[aLevel];
   if Guard.Check(ga_Root,RusEng('Только Root может изменять пароли!',
                                 'Only Root can change password!'),gf_Warning+gf_Sound)<0
   then Exit;
   if FormLoginDialogExecute(RusEng('Задайте новый пароль для уровня ',
                                    'Set new password for level ')+aUser,
                             aUser, aDomain, aPassword,
                             ld_ShowUser+ld_ShowDomain+ld_RetypePassword)<>mrOk
   then Exit;
   if IsEmptyStr(aPassword) then
   if YesNo(RusEng('Вы действительно хотите отменить пароль?',
                   'Are you really want to clear password?'))<>mrYes then Exit;
   Guard.WritePassword(aLevel,Trim(aPassword));
   Guard.Log:='Password '+aUser;
  end;
 except
  on E:Exception do BugReport(E,Self,'PasswordToLevel');
 end;
end;

procedure TFormSecretService.BitBtnWantsLockClick(Sender: TObject);
begin
 LoginToLevel(ga_Lock);
end;

procedure TFormSecretService.BitBtnWantsGuestClick(Sender: TObject);
begin
 LoginToLevel(ga_Guest);
end;

procedure TFormSecretService.BitBtnGuestPasswordClick(Sender: TObject);
begin
 PasswordToLevel(ga_Guest);
end;

procedure TFormSecretService.BitBtnWantsUserClick(Sender: TObject);
begin
 LoginToLevel(ga_User);
end;

procedure TFormSecretService.BitBtnUserPasswordClick(Sender: TObject);
begin
 PasswordToLevel(ga_User);
end;

procedure TFormSecretService.BitBtnWantsRootClick(Sender: TObject);
begin
 LoginToLevel(ga_Root);
end;

procedure TFormSecretService.BitBtnRootPasswordClick(Sender: TObject);
begin
 PasswordToLevel(ga_Root);
end;

procedure TFormSecretService.BitBtnTrustListAddClick(Sender: TObject);
var i:Integer;
begin
 if Guard.Check(ga_Root,RusEng('Только Root может изменять список!',
                               'Only Root can change this list!'),gf_Warning+gf_Sound)<0
 then Exit;
 try
  if (ListBoxTrustList.ItemIndex>=0) then
  if (ListBoxTrustList.ItemIndex<ListBoxTrustList.Items.Count) then
  OpenDialogTrustList.FileName:=ListBoxTrustList.Items[ListBoxTrustList.ItemIndex];
  if GuardOpenDialog(OpenDialogTrustList).Execute then begin
   ListBoxTrustList.Items.Text:=Guard.TrustList;
   for i:=0 to OpenDialogTrustList.Files.Count-1 do
   if ListBoxTrustList.Items.IndexOf(OpenDialogTrustList.Files[i])<0
   then ListBoxTrustList.Items.Add(OpenDialogTrustList.Files[i]);
   Guard.TrustList:=ListBoxTrustList.Items.Text;
   Guard.WriteTrustList;
   Guard.ReadTrustList;
   ListBoxTrustList.Items.Text:=Guard.TrustList;
   ListBoxTrustList.ItemIndex:=ListBoxTrustList.Items.IndexOf(OpenDialogTrustList.FileName);
   Guard.Log:='TrustList +'+IntToStr(OpenDialogTrustList.Files.Count);
  end;
 except
  on E:Exception do BugReport(E,Self,'BitBtnTrustListAddClick');
 end;
end;

procedure TFormSecretService.BitBtnTrustListDelClick(Sender: TObject);
var i:Integer; s:LongString;
begin
 if Guard.Check(ga_Root,RusEng('Только Root может изменять список!',
                               'Only Root can change this list!'),gf_Warning+gf_Sound)<0
 then Exit;
 try
  if ListBoxTrustList.ItemIndex>=0 then
  if ListBoxTrustList.ItemIndex<ListBoxTrustList.Items.Count then begin
   i:=ListBoxTrustList.ItemIndex;
   s:=ListBoxTrustList.Items[ListBoxTrustList.ItemIndex];
   ListBoxTrustList.Items.Text:=Guard.TrustList;
   if ListBoxTrustList.Items.IndexOf(s)>=0
   then ListBoxTrustList.Items.Delete(ListBoxTrustList.Items.IndexOf(s));
   Guard.TrustList:=ListBoxTrustList.Items.Text;
   Guard.WriteTrustList;
   Guard.ReadTrustList;
   ListBoxTrustList.Items.Text:=Guard.TrustList;
   ListBoxTrustList.ItemIndex:=Min(i,ListBoxTrustList.Items.Count-1);
   Guard.Log:='TrustList -1';
  end;
 except
  on E:Exception do BugReport(E,Self,'BitBtnTrustListDelClick');
 end;
end;

function TheGuardChecker(aGuard:TGuard; aLevel:Cardinal; aMessage:LongString; aFlags:Cardinal):Integer;
 function Act:LongString;
 begin
  Result:='';
  if (WordCount(aMessage,ScanSpaces)=2)
  then Result:=' '+ExtractWord(2,aMessage,ScanSpaces);
 end;
begin
 if (aGuard.Level<aLevel) then Result:=-1 else
 if (aGuard.Level>aLevel) then Result:=+1 else Result:=0;
 if (Result<0) and (Length(aMessage)>0) then begin
  if (ExtractWord(1,aMessage,ScanSpaces)='*')
  then aMessage:=RusEng(Format(' Доступ закрыт.'+
                               ' Вам поможет F11 - "Служба Защиты CRW-DAQ".'+EOL+
                               ' Для выполнения действия'+Act+' прав "%s" мало, нужен уровень доступа "%s".',
                               [aGuard.LevelName[aGuard.Level],aGuard.LevelName[aLevel]]),
                        Format(' Access denied.'+
                               ' Use F11 to open "CRW-DAQ Guard Service" control window.'+EOL+
                               ' To execute action'+Act+' level "%s" is not enough, require level "%s".',
                               [aGuard.LevelName[aGuard.Level],aGuard.LevelName[aLevel]]));
  with aGuard do
  if HasFlags(aFlags,gf_Log) then aGuard.Log:='Deny '+LevelName[Level]+', need '+LevelName[aLevel]+' for action'+Act;
  if HasFlags(aFlags,gf_Sound) then Voice(aGuard.Sound);
  if HasFlags(aFlags,gf_Raise) then RAISE EGuard.Create('Guard: '+Trim(aMessage));
  if HasFlags(aFlags,gf_Echo) then begin
   Echo(StdDateTimePrompt(mSecNow)+'Guard: '+Trim(aMessage));
   if HasFlags(aFlags,gf_Open) then SystemConsole.Activate;
  end;
  if HasFlags(aFlags,gf_Warning) then if CanShowModal(nil,3) then Warning('Guard: '+Trim(aMessage));
 end;
end;

procedure TFormSecretService.BitBtnLevelCloseClick(Sender: TObject);
begin
 Close;
end;

procedure TFormSecretService.ShowWaiting(aShow:Boolean);
begin
 if aShow then begin
  ImageAccountWait.Show;
  CheckBoxAccountUserDetail.Hide;
  CheckBoxAccountDomainDetail.Hide;
  SafeApplicationProcessMessages;
 end else begin
  ImageAccountWait.Hide;
  CheckBoxAccountUserDetail.Show;
  CheckBoxAccountDomainDetail.Show;
  SafeApplicationProcessMessages;
 end;
end;

procedure TFormSecretService.BitBtnAccountDomainClick(Sender: TObject);
var
 t      : TText;
 i,k    : Integer;
 Patt   : LongString;
 Domain : LongString;
begin
 try
  Domain:='';
  t:=NewText;
  try
   k:=0;
   ShowWaiting(True);
   if CheckBoxAccountDomainDetail.Checked then begin
    t.Text:=GetDomainList('.',1,True,SpinEditAccountTimeOut.Value*1000);
   end else begin
    t.Text:=GetDomainList('.',0,False,SpinEditAccountTimeOut.Value*1000);
    i:=Max(t.MaxLength,Max(Length(ComputerName),Length('localhost')));
    t.InsLn(0,Format('%-*s localhost',[i,'.']));
    t.InsLn(1,Format('%-*s this_host',[i,'localhost']));
    t.InsLn(2,Format('%-*s localhost',[i,ComputerName]));
   end;
   Patt:=Trim(EditAccountDomain.Text);
   if IsEmptyStr(Patt) then Patt:=ComputerName;
   for i:=0 to t.Count-1 do
   if IsSameText(ExtractWord(1,t[i],ScanSpaces),Patt) then begin
    k:=i;
    Break;
   end;
   ShowWaiting(False);
   if ListBoxSelection(RusEng('Выбор домена','Domain selection'),
                       RusEng('Выбрать домен','Select domain'),t.Text,k)=mrOk
   then Domain:=ExtractWord(1,t[k],ScanSpaces);
   if IsSameText(Domain,NetEnumTimeoutId) then Domain:='';
  finally
   Kill(t);
   ShowWaiting(False);
  end;
  if Length(Domain)>0 then begin
   EditAccountDomain.Text:=Domain;
   ActiveControl:=EditAccountDomain;
  end;
 except
  on E:Exception do EditAccountKey.Text:=E.Message;
 end;
end;

procedure TFormSecretService.BitBtnAccountUserClick(Sender: TObject);
var
 t    : TText;
 i,k  : Integer;
 Host : LongString;
 Patt : LongString;
 User : LongString;
begin
 try
  User:='';
  t:=NewText;
  try
   k:=0;
   ShowWaiting(True);
   Host:=Trim(EditAccountDomain.Text);
   if IsEmptyStr(Host) then Host:='.' else
   if IsSameText(Host,'localhost') then Host:='.' else
   if IsSameText(Host,ComputerName) then Host:=ComputerName else begin
    Host:=ExtractWord(1,GetHostList('',Host,0,SV_TYPE_DOMAIN_CTRL,SpinEditAccountTimeOut.Value*1000),ScanSpaces);
   end;
   t.Text:=GetUserList(Host,Ord(CheckBoxAccountUserDetail.Checked));
   if t.Count=0 then t.Text:=UserName;
   Patt:=Trim(EditAccountUser.Text);
   if IsEmptyStr(Patt) then Patt:=UserName;
   for i:=0 to t.Count-1 do
   if IsSameText(ExtractWord(1,t[i],ScanSpaces),Patt) then begin
    k:=i;
    Break;
   end;
   ShowWaiting(False);
   Host:=Trim(EditAccountDomain.Text);
   if IsEmptyStr(Host) or IsSameText(Host,'.') then Host:='localhost';
   if ListBoxSelection(RusEng('Выбор пользователя','User selection'),
                       RusEng('Выбрать пользователя на ','Select user on ')+Host,t.Text,k)=mrOk
   then User:=ExtractWord(1,t[k],ScanSpaces);
   if IsSameText(User,NetEnumTimeoutId) then User:='';
  finally
   Kill(t);
   ShowWaiting(False);
  end;
  if Length(User)>0 then begin
   EditAccountUser.Text:=User;
   ActiveControl:=EditAccountUser;
  end;
 except
  on E:Exception do EditAccountKey.Text:=E.Message;
 end;
end;

procedure TFormSecretService.TimerSecurityTimer(Sender: TObject);
begin
 try
  if (CheckEditPasswordChar(EditAccountPassword1,ThePasswordChar)<>ThePasswordChar)
  or (CheckEditPasswordChar(EditAccountPassword2,ThePasswordChar)<>ThePasswordChar) then begin
   EditAccountKey.Text:=RusEng('Программа атакована! Кто-то хочет украсть пароль...',
                               'Program attacked! Someone want to steal password...');
  end;
 except
  on E:Exception do BugReport(E,Self,'TimerSecurityTimer');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_secretservice;
begin
 Guard.Checker:=TheGuardChecker;
end;

procedure Free_form_secretservice;
begin
 Guard.Checker:=DefaultGuardChecker;
end;

initialization

 Init_form_secretservice;

finalization

 Free_form_secretservice;

end.

//////////////
// END OF FILE
//////////////

