////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form MistimingService to detect mistiming problems.                        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231116 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
// 20240801 - Init_MistimingService                                           //
////////////////////////////////////////////////////////////////////////////////

unit form_mistimingservice; // Form Mistiming Service

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, strutils,
 Graphics, Interfaces, LMessages, lclintf,
 Forms, Controls, StdCtrls, Dialogs,
 Spin, ExtCtrls, ComCtrls,
 ActnList, Menus, ToolWin, ImgList,
 _crw_alloc, _crw_str, _crw_eldraw, _crw_guard, _crw_sect,
 _crw_polling, _crw_fio, _crw_rtc, _crw_sesman, _crw_gloss,
 _crw_appforms, _crw_apptools, _crw_apputils,
 _crw_guiutils,  _crw_fonts, _crw_ef;

type
  TFormMistimingService = class(TMasterForm)
    TimerCheckTimeDelta: TTimer;
    GroupBoxStatus: TGroupBox;
    LabelStatus: TLabel;
    MemoStatusComment: TMemo;
    LabelTimeDelta: TLabel;
    EditTimeDelta: TEdit;
    LabelMs1: TLabel;
    LabelTimeFails: TLabel;
    EditTimeFails: TEdit;
    GroupBoxPreset: TGroupBox;
    CheckBoxMistimingAlert: TCheckBox;
    SpinEditMaxTimeDelta: TSpinEdit;
    LabelMs2: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormHide(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure TimerCheckTimeDeltaTimer(Sender: TObject);
    procedure SpinEditMaxTimeDeltaChange(Sender: TObject);
  private
    { Private declarations }
    function  GetMistimingAlert:Boolean;
    procedure SetMistimingAlert(AlertOn:Boolean);
  public
    { Public declarations }
    property MistimingAlert : Boolean read GetMistimingAlert write SetMistimingAlert;
  end;

procedure CheckMistiming;
procedure OpenFormMistimingService(Show:Boolean=true; const aParams:LongString='');
procedure HideFormMistimingService;
procedure Init_MistimingService;

implementation

{$R *.lfm}

const
  TheTimeFails         : Int64                 = 0;
  TheTimeDelta         : Double                = 0;
  MaxTimeDelta         : Integer               = 1000;
  FormMistimingService : TFormMistimingService = nil;
  MistimingDelay       : Integer               = 1000*60*60*24;
  MistimingUseTooltip  : Boolean               = true;
  MistimingCmdOpen     : LongString            = 'resource\shell\crwdaqopenmistiming.cmd';

procedure Init_MistimingService;
var b:Boolean;
begin
 b:=MistimingUseTooltip;
 if ReadIniFileBoolean(SysIniFile,SectSystem,'MistimingUseTooltip%b',b)
 then MistimingUseTooltip:=b;
 MistimingCmdOpen:=SysGlossary.ReadIniParamDef(SysIniFile,SectSystem,'MistimingCmdOpen',MistimingCmdOpen);
end;

function TooltipMistimingMessage:LongString;
var pid:DWORD; ProgName:LongString;
begin
 ProgName:=ParamStr(0); pid:=GetCurrentProcessId;
 if MistimingUseTooltip then
 // The crwdaqopenmistiming.cmd should send @menu run FormCrwDaq.ActionWindowsMistimingService
 Result:=Format('guid %s:MTS text "%s: %s" preset stdFails delay %d btn1 "%s" cmd1 "%s -p %d"'
               ,[SessionManager.SessionHead,SessionManager.SessionSign
               ,RusEng('СЛУЖБА СИСТЕМНОГО ВРЕМЕНИ','SYSTEM TIME SERVICE'),MistimingDelay,RusEng('Открыть','Open')
               ,SmartFileRef(AdaptExeFileName(MistimingCmdOpen),'',ProgName),pid]);
end;

procedure CheckMistiming;
var
 TimeFails : Int64;
 TimeDelta : Double;
 Code      : Integer;
 Warn      : Boolean;
begin
 TimeFails:=msecnowerrors;
 TimeDelta:=msecnow(rtc_LMT_SYS)-msecnow(rtc_LMT_DAQ);
 if (TimeFails<>TheTimeFails) or (abs(TimeDelta-TheTimeDelta)>MaxTimeDelta) then
 try
  Echo(RusEng(Format('Найдена рассинхронизация времени (%g ms)',[TimeDelta-TheTimeDelta]),
              Format('System clock mistiming was found (%g ms)',[TimeDelta-TheTimeDelta])));
  if SysLogNotable(SeverityOfMistimes)
  then SysLogNote(0,SeverityOfMistimes,sdr_System,Format('System clock mistiming (%g ms)',[TimeDelta-TheTimeDelta]));
  if FormMistimingService.Ok then Warn:=FormMistimingService.CheckBoxMistimingAlert.Checked else Warn:=true;
  if PreferTooltip and MistimingUseTooltip and Warn
  then Code:=ShowTooltip(TooltipMistimingMessage) else Code:=0;
  if not FormMistimingService.Ok then OpenFormMistimingService(Code=0);
  if FormMistimingService.Ok then
  if FormMistimingService.CheckBoxMistimingAlert.Checked then begin
   if (Code=0) then begin
    FormMistimingService.Show;
    FormMistimingService.BringToFront;
   end;
   FormMistimingService.MistimingAlert:=true;
  end;
 except
  on E:Exception do BugReport(E,nil,'CheckMistiming');
 end;
 TheTimeDelta:=TimeDelta;
 TheTimeFails:=TimeFails;
end;

procedure OpenFormMistimingService(Show:Boolean; const aParams:LongString='');
var apFlags:Integer;
begin
 try
  if not FormMistimingService.Ok then begin
   Application.CreateForm(TFormMistimingService, FormMistimingService);
   FormMistimingService.Master:=@FormMistimingService;
   FormMistimingService.Visible:=Show;
  end;
  if FormMistimingService.Ok then begin
   if Show then begin
    Application.Restore;
    Application.BringToFront;
    FormMistimingService.Show;
    FormMistimingService.BringToFront;
    apFlags:=FormMistimingService.ApplyParams(aParams);
    if not HasFlags(apFlags,apf_FormPos)
    then LocateFormToCenterOfScreen(FormMistimingService);
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'OpenFormMistimingService');
 end;
end;

procedure HideFormMistimingService;
begin
 try
  if FormMistimingService.Ok then FormMistimingService.Hide;
 except
  on E:Exception do BugReport(E,nil,'HideFormMistimingService');
 end;
end;

procedure TFormMistimingService.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=Format('%s - %s',[ExtractFileNameExt(ParamStr(0)),RusEng('Служба системного времени','System time service')]);
 GroupBoxStatus.Caption:=RusEng('Статус','Status');
 GroupBoxPreset.Caption:=RusEng('Уставки','Preset');
 CheckBoxMistimingAlert.Caption:=RusEng('Сообщать при разнице >','Warning on mistiming >');
 LabelTimeDelta.Caption:=RusEng('Рассинхронизация часов','Clock mistiming');
 LabelTimeFails.Caption:=RusEng('Найдено ошибок часов','Clock errors found');
 SetDefaultMonoFontName(LabelStatus.Font);
 MistimingAlert:=false;
end;

procedure TFormMistimingService.FormClose(Sender: TObject; var Action: TCloseAction);
begin
 Action:=caHide;
end;

function  TFormMistimingService.GetMistimingAlert:Boolean;
begin
 Result:=False;
 if Ok then Result:=(LabelStatus.Font.Color=clRed);
end;

procedure TFormMistimingService.SetMistimingAlert(AlertOn:Boolean);
begin
 if Ok then
 try
  if AlertOn then begin
   LabelStatus.Color:=clYellow;
   LabelStatus.Font.Color:=clRed;
   LabelStatus.Caption:=RusEng('РАССИНХРОНИЗАЦИЯ!','MISTIMING FOUND!');
   MemoStatusComment.Text:=RusEng('Возможные причины рассинхронизации часов:'+EOL+
                                  '1)Системное время было изменено'+EOL+
                                  '2)Переход на летнее/зимнее время'+EOL+
                                  '3)Программный сбой'+EOL+
                                  'РЕКОМЕНДУЕТСЯ ПЕРЕЗАГРУЗИТЬ СИСТЕМУ',
                                  'Possible reason of clock mistiming:'+EOL+
                                  '1)System time was changed'+EOL+
                                  '2)Winter/summer time'+EOL+
                                  '3)Internal program error'+EOL+
                                  'SYSTEM RESTART IS RECOMMENDED'
                                  );
  end else begin
   LabelStatus.Color:=clSilver;
   LabelStatus.Font.Color:=clGreen;
   LabelStatus.Caption:=RusEng('ВРЕМЯ СИНХРОНИЗОВАНО','TIME IS SYNCHRONIZED');
   MemoStatusComment.Text:=RusEng('Служба системного времени не нашла проблем с синхронизацией часов.',
                                  'System time service did not find any time synchronization problem.');
  end;
 except
  on E:Exception do BugReport(E,Self,'SetMistimingAlert');
 end;
end;

procedure TFormMistimingService.TimerCheckTimeDeltaTimer(Sender: TObject);
begin
 if Ok then
 try
  EditTimeFails.Text:=Format('%d',[msecnowerrors]);
  EditTimeDelta.Text:=Format('%g',[msecnow(rtc_LMT_SYS)-msecnow(rtc_LMT_DAQ)]);
 except
  on E:Exception do BugReport(E,Self,'TimerCheckTimeDeltaTimer');
 end;
end;

procedure TFormMistimingService.FormHide(Sender: TObject);
begin
 if Assigned(Self) then
 try
  MistimingAlert:=false;
  TimerCheckTimeDelta.Enabled:=false;
 except
  on E:Exception do BugReport(E,Self,'FormHide');
 end;
end;

procedure TFormMistimingService.FormShow(Sender: TObject);
begin
 if Assigned(Self) then
 try
  TimerCheckTimeDelta.Enabled:=true;
 except
  on E:Exception do BugReport(E,Self,'FormShow');
 end;
end;

procedure TFormMistimingService.FormActivate(Sender: TObject);
begin
 if Assigned(Self) then
 try
  SpinEditMaxTimeDelta.Value:=MaxTimeDelta;
 except
  on E:Exception do BugReport(E,Self,'FormActivate');
 end;
end;

procedure TFormMistimingService.SpinEditMaxTimeDeltaChange(Sender: TObject);
begin
 if Assigned(Self) then
 try
  MaxTimeDelta:=SpinEditMaxTimeDelta.Value;
 except
  on E:Exception do BugReport(E,Self,'SpinEditMaxTimeDeltaChange');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_mistimingservice;
begin
 TheTimeDelta:=msecnow(rtc_LMT_SYS)-msecnow(rtc_LMT_DAQ);
end;

procedure Free_form_mistimingservice;
begin
end;

initialization

 Init_form_mistimingservice;

finalization

 Free_form_mistimingservice;

end.

//////////////
// END OF FILE
//////////////

