////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form DAQ DateTime Calculator.                                              //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231201 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_daqdatetimecalculator; // Form DAQ DateTime Calculator

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Spin,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormDaqDateTimeCalculator = class(TMasterForm)
    GroupBoxStartStop: TGroupBox;
    SpinEditYear1: TSpinEdit;
    SpinEditMonth1: TSpinEdit;
    SpinEditDay1: TSpinEdit;
    SpinEditHour1: TSpinEdit;
    SpinEditMin1: TSpinEdit;
    SpinEditSec1: TSpinEdit;
    SpinEditMSec1: TSpinEdit;
    SpinEditYear2: TSpinEdit;
    SpinEditMonth2: TSpinEdit;
    SpinEditDay2: TSpinEdit;
    SpinEditHour2: TSpinEdit;
    SpinEditMin2: TSpinEdit;
    SpinEditSec2: TSpinEdit;
    SpinEditMSec2: TSpinEdit;
    LabelStart: TLabel;
    LabelStop: TLabel;
    LabelYear: TLabel;
    LabelMonth: TLabel;
    LabelDay: TLabel;
    LabelHour: TLabel;
    LabelMin: TLabel;
    LabelSec: TLabel;
    LabelMSec: TLabel;
    RadioGroupUnits: TRadioGroup;
    EditDuration: TEdit;
    LabelDuration: TLabel;
    GroupBoxWhatToDo: TGroupBox;
    ButtonStart: TButton;
    ButtonStop: TButton;
    ButtonDuration: TButton;
    ButtonNowStart: TButton;
    ButtonNowStop: TButton;
    ButtonXmasStart: TButton;
    ButtonXmasStop: TButton;
    ButtonDaqTimeStart: TButton;
    ButtonDaqTimeStop: TButton;
    procedure FormCreate(Sender: TObject);
    procedure ButtonStartClick(Sender: TObject);
    procedure ButtonStopClick(Sender: TObject);
    procedure ButtonDurationClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ButtonNowStartClick(Sender: TObject);
    procedure ButtonNowStopClick(Sender: TObject);
    procedure ButtonXmasStartClick(Sender: TObject);
    procedure ButtonXmasStopClick(Sender: TObject);
    procedure ButtonDaqTimeStartClick(Sender: TObject);
    procedure ButtonDaqTimeStopClick(Sender: TObject);
  private
    { Private declarations }
    function  GetTime1:Double;
    procedure SetTime1(aTime:Double);
    function  GetTime2:Double;
    procedure SetTime2(aTime:Double);
  public
    { Public declarations }
    procedure UpdateControls(Code:Integer);
    property Time1 : Double read GetTime1 write SetTime1;
    property Time2 : Double read GetTime2 write SetTime2;
  end;

procedure OpenFormDaqDateTimeCalculator(ms1,ms2:Double; const Params:LongString='');
procedure KillFormDaqDateTimeCalculator;

implementation

uses
 _crw_daqsys;

{$R *.lfm}

const
  FormDaqDateTimeCalculator: TFormDaqDateTimeCalculator = nil;

function GetUnits(DataUnits:Integer):Double;
begin
 case DataUnits of
  0 :  Result := 1.0;
  1 :  Result := 1000.0;
  2 :  Result := 1000.0*60;
  3 :  Result := 1000.0*60*60;
  4 :  Result := 1000.0*60*60*24;
  else Result := 1.0;
 end;
end;

procedure OpenFormDaqDateTimeCalculator(ms1,ms2:Double; const Params:LongString='');
var i:Integer;
begin
 try
  if not Assigned(FormDaqDateTimeCalculator) then begin
   Application.CreateForm(TFormDaqDateTimeCalculator, FormDaqDateTimeCalculator);
   FormDaqDateTimeCalculator.Master:=@FormDaqDateTimeCalculator;
  end;
  if Assigned(FormDaqDateTimeCalculator) then with FormDaqDateTimeCalculator do begin
   with MSecToNativeTime(ms1) do begin
    SpinEditYear1.Value:=Year;
    SpinEditMonth1.Value:=Month;
    SpinEditDay1.Value:=Day;
    SpinEditHour1.Value:=Hour;
    SpinEditMin1.Value:=Minute;
    SpinEditSec1.Value:=Second;
    SpinEditMSec1.Value:=Millisecond;
   end;
   with MSecToNativeTime(ms2) do begin
    SpinEditYear2.Value:=Year;
    SpinEditMonth2.Value:=Month;
    SpinEditDay2.Value:=Day;
    SpinEditHour2.Value:=Hour;
    SpinEditMin2.Value:=Minute;
    SpinEditSec2.Value:=Second;
    SpinEditMSec2.Value:=Millisecond;
   end;
   RadioGroupUnits.ItemIndex:=1;
   for i:=0 to 4 do if Daq.Timer.LocalTimeUnits=GetUnits(i)
   then RadioGroupUnits.ItemIndex:=i;
   if IsNonEmptyStr(Params) then ApplyParams(Params);
   RadioGroupUnits.Repaint;
   UpdateControls(2);
   Show;
   BringToFront;
  end;
 except
  on E:Exception do BugReport(E,nil,'OpenFormDaqDateTimeCalculator');
 end;
end;

procedure KillFormDaqDateTimeCalculator;
begin
 Kill(TObject(FormDaqDateTimeCalculator));
end;

procedure TFormDaqDateTimeCalculator.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 LocateFormToCenterOfScreen(Self);
 Caption:=RusEng('Калькулятор времени','Date Time calculator');
 LabelYear.Caption:=RusEng('Год','Year');
 LabelMonth.Caption:=RusEng('Месяц','Month');
 LabelDay.Caption:=RusEng('День','Day');
 LabelHour.Caption:=RusEng('Час','Hour');
 LabelMin.Caption:=RusEng('Мин','Min');
 LabelSec.Caption:=RusEng('Сек','Sec');
 LabelMSec.Caption:=RusEng('мСек','mSec');
 LabelStart.Caption:=RusEng('Начало','Start');
 LabelStop.Caption:=RusEng('Конец','Stop');
 LabelDuration.Caption:=RusEng('Длительность','Start/stop duration');
 GroupBoxStartStop.Caption:=RusEng('Время начала/конца интервала','Time of interval start/stop');
 RadioGroupUnits.Caption:=RusEng('Единицы измерения времени','Time units');
 RadioGroupUnits.Items.Text:=RusEng('мСек'+EOL+'Сек'+EOL+'Мин'+EOL+'Часы'+EOL+'Дни',
                                    'mSec'+EOL+'Sec'+EOL+'Min'+EOL+'Hour'+EOL+'Day');
 GroupBoxWhatToDo.Caption:=RusEng('Что надо вычислить','What to do');
 ButtonStart.Caption:=RusEng('Найти дату/время начала интервала','Find date / time of start');
 ButtonStop.Caption:=RusEng('Найти дату/время конца интервала','Find date / time of stop');
 ButtonDuration.Caption:=RusEng('Найти длительность интервала','Find start/stop duration');
end;

procedure TFormDaqDateTimeCalculator.UpdateControls(Code:Integer);
var tbeg,tend,dt:double;
begin
 if Ok then
 try
  case Code of
  0 : if Str2Real(EditDuration.Text,dt) then begin
       tend:=DateTimeToMsec(SpinEditYear2.Value,
                            SpinEditMonth2.Value,
                            SpinEditDay2.Value,
                            SpinEditHour2.Value,
                            SpinEditMin2.Value,
                            SpinEditSec2.Value,
                            SpinEditMSec2.Value);
       tbeg:=tend-dt*GetUnits(RadioGroupUnits.ItemIndex);
       if IsNanOrInf(tbeg)
       then Warning(RusEng('Неверные данные!','Invalid data!'))
       else with MSecToNativeTime(tbeg) do begin
        SpinEditYear1.Value:=Year;
        SpinEditMonth1.Value:=Month;
        SpinEditDay1.Value:=Day;
        SpinEditHour1.Value:=Hour;
        SpinEditMin1.Value:=Minute;
        SpinEditSec1.Value:=Second;
        SpinEditMSec1.Value:=Millisecond;
       end;
      end else Warning(RusEng('Неверные данные!','Invalid data!'));
  1 : if Str2Real(EditDuration.Text,dt) then begin
       tbeg:=DateTimeToMsec(SpinEditYear1.Value,
                            SpinEditMonth1.Value,
                            SpinEditDay1.Value,
                            SpinEditHour1.Value,
                            SpinEditMin1.Value,
                            SpinEditSec1.Value,
                            SpinEditMSec1.Value);
       tend:=tbeg+dt*GetUnits(RadioGroupUnits.ItemIndex);
       if IsNanOrInf(tend)
       then Warning(RusEng('Неверные данные!','Invalid data!'))
       else with MSecToNativeTime(tend) do begin
        SpinEditYear2.Value:=Year;
        SpinEditMonth2.Value:=Month;
        SpinEditDay2.Value:=Day;
        SpinEditHour2.Value:=Hour;
        SpinEditMin2.Value:=Minute;
        SpinEditSec2.Value:=Second;
        SpinEditMSec2.Value:=Millisecond;
       end;
      end else Warning(RusEng('Неверные данные!','Invalid data!'));
  2 : begin
       tbeg:=DateTimeToMsec(SpinEditYear1.Value,
                            SpinEditMonth1.Value,
                            SpinEditDay1.Value,
                            SpinEditHour1.Value,
                            SpinEditMin1.Value,
                            SpinEditSec1.Value,
                            SpinEditMSec1.Value);
       tend:=DateTimeToMsec(SpinEditYear2.Value,
                            SpinEditMonth2.Value,
                            SpinEditDay2.Value,
                            SpinEditHour2.Value,
                            SpinEditMin2.Value,
                            SpinEditSec2.Value,
                            SpinEditMSec2.Value);
       dt:=(tend-tbeg)/GetUnits(RadioGroupUnits.ItemIndex);
       if IsNanOrInf(dt)
       then Warning(RusEng('Неверные данные!','Invalid data!'))
       else EditDuration.Text:=Format('%g',[dt]);
      end;
  end;
  Update;
 except
  on E:Exception do BugReport(E,Self,'UpdateControls');
 end;
end;

function TFormDaqDateTimeCalculator.GetTime1:Double;
begin
 Result:=_NaN;
 if Ok then
 try
  Result:=DateTimeToMSec(SpinEditYear1.Value,
                         SpinEditMonth1.Value,
                         SpinEditDay1.Value,
                         SpinEditHour1.Value,
                         SpinEditMin1.Value,
                         SpinEditSec1.Value,
                         SpinEditMSec1.Value);
 except
  on E:Exception do BugReport(E,Self,'GetTime1');
 end;
end;

procedure TFormDaqDateTimeCalculator.SetTime1(aTime:Double);
begin
 if Ok then
 try
  with MSecToNativeTime(aTime) do begin
   SpinEditYear1.Value:=Year;
   SpinEditMonth1.Value:=Month;
   SpinEditDay1.Value:=Day;
   SpinEditHour1.Value:=Hour;
   SpinEditMin1.Value:=Minute;
   SpinEditSec1.Value:=Second;
   SpinEditMSec1.Value:=Millisecond;
  end;
 except
  on E:Exception do BugReport(E,Self,'SetTime1');
 end;
end;

function TFormDaqDateTimeCalculator.GetTime2:Double;
begin
 Result:=_NaN;
 if Ok then
 try
  Result:=DateTimeToMSec(SpinEditYear2.Value,
                         SpinEditMonth2.Value,
                         SpinEditDay2.Value,
                         SpinEditHour2.Value,
                         SpinEditMin2.Value,
                         SpinEditSec2.Value,
                         SpinEditMSec2.Value);
 except
  on E:Exception do BugReport(E,Self,'GetTime2');
 end;
end;

procedure TFormDaqDateTimeCalculator.SetTime2(aTime:Double);
begin
 if Ok then
 try
  with MSecToNativeTime(aTime) do begin
   SpinEditYear2.Value:=Year;
   SpinEditMonth2.Value:=Month;
   SpinEditDay2.Value:=Day;
   SpinEditHour2.Value:=Hour;
   SpinEditMin2.Value:=Minute;
   SpinEditSec2.Value:=Second;
   SpinEditMSec2.Value:=Millisecond;
  end;
 except
  on E:Exception do BugReport(E,Self,'SetTime2');
 end;
end;

procedure TFormDaqDateTimeCalculator.ButtonStartClick(Sender: TObject);
begin
 UpdateControls(0);
end;

procedure TFormDaqDateTimeCalculator.ButtonStopClick(Sender: TObject);
begin
 UpdateControls(1);
end;

procedure TFormDaqDateTimeCalculator.ButtonDurationClick(Sender: TObject);
begin
 UpdateControls(2);
end;

procedure TFormDaqDateTimeCalculator.FormClose(Sender: TObject; var Action: TCloseAction);
begin
 Action:=caFree;
end;

procedure TFormDaqDateTimeCalculator.ButtonNowStartClick(Sender: TObject);
begin
 if Ok then Time1:=msecnow;
end;

procedure TFormDaqDateTimeCalculator.ButtonNowStopClick(Sender: TObject);
begin
 if Ok then Time2:=msecnow;
end;

procedure TFormDaqDateTimeCalculator.ButtonXmasStartClick(Sender: TObject);
begin
 if Ok then Time1:=0;
end;

procedure TFormDaqDateTimeCalculator.ButtonXmasStopClick(Sender: TObject);
begin
 if Ok then Time2:=0;
end;

procedure TFormDaqDateTimeCalculator.ButtonDaqTimeStartClick(Sender: TObject);
begin
 if Ok and Daq.Timer.isStart then Time1:=Daq.Timer.StartTime;
end;

procedure TFormDaqDateTimeCalculator.ButtonDaqTimeStopClick(Sender: TObject);
begin
 if Ok and Daq.Timer.isStart then Time2:=Daq.Timer.StartTime;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_daqdatetimecalculator;
begin
end;

procedure Free_form_daqdatetimecalculator;
begin
end;

initialization

 Init_form_daqdatetimecalculator;

finalization

 Free_form_daqdatetimecalculator;

end.

//////////////
// END OF FILE
//////////////

