////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form DAQ Control Dialog.                                                   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231202 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_daqcontroldialog; // Form DAQ Control Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_daqdev, _crw_sect,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormDaqControlDialog = class(TFormCrwDaqSysChild)
    ActionListDaq: TActionList;
    ImageListDaq: TImageList;
    ActionDaqInit: TAction;
    ActionDaqDone: TAction;
    ActionDaqStart: TAction;
    ActionDaqStop: TAction;
    ActionDaqLoad: TAction;
    ActionDaqSave: TAction;
    ActionDaqRemedy: TAction;
    ActionDaqClear: TAction;
    ActionDaqWindows: TAction;
    ActionDaqNewWindow: TAction;
    ActionDaqViewCurveTable: TAction;
    ActionDaqEditCurve: TAction;
    ActionDaqHelp: TAction;
    ActionDaqEditNote: TAction;
    ActionDaqEditConfig: TAction;
    ActionDaqSound: TAction;
    ActionDaqMerge: TAction;
    ActionDaqClock: TAction;
    ActionDaqCalculator: TAction;
    ActionDaqDevice: TAction;
    ActionDaqIntegral: TAction;
    ActionDaqMedianFilter: TAction;
    ActionDaqSmoothing: TAction;
    ActionDaqCutRect: TAction;
    ActionDaqMarker: TAction;
    ActionDaqInfo: TAction;
    ActionDaqNavigator: TAction;
    ActionDaqStatus: TAction;
    ToolButtonDaqInit: TToolButton;
    ToolButtonDaqDone: TToolButton;
    ToolButtonDaqStart: TToolButton;
    ToolButtonDaqStop: TToolButton;
    ToolButtonDaqLoad: TToolButton;
    ToolButtonDaqSave: TToolButton;
    ToolButtonDaqRemedy: TToolButton;
    ToolButtonDaqClear: TToolButton;
    ToolButtonDaqWindows: TToolButton;
    ToolButtonDaqNewWindow: TToolButton;
    ToolButtonDaqViewCurveTable: TToolButton;
    ToolButtonDaqEditCurve: TToolButton;
    ToolButtonDaqHelp: TToolButton;
    ToolButtonDaqEditNote: TToolButton;
    ToolButtonDaqEditConfig: TToolButton;
    ToolButtonDaqSound: TToolButton;
    ToolButtonDaqMerge: TToolButton;
    ToolButtondaqClock: TToolButton;
    ToolButtonDaqCalculator: TToolButton;
    ToolButtonDaqDevice: TToolButton;
    ToolButtonDaqIntegral: TToolButton;
    ToolButtonDaqMedianFilter: TToolButton;
    ToolButtonDaqSmoothing: TToolButton;
    ToolButtonDaqCutRect: TToolButton;
    ToolButtonDaqMarker: TToolButton;
    ToolButtonDaqInfo: TToolButton;
    ToolButtonDaqNavigator: TToolButton;
    ToolButtonDaqStatus: TToolButton;
    MenuDaq: TMenuItem;
    MenuDaqSubControl: TMenuItem;
    MenuDaqSubSeans: TMenuItem;
    MenuDaqSubView: TMenuItem;
    MenuDaqSubHelp: TMenuItem;
    MenuDaqSubMath: TMenuItem;
    MenuDaqSubTools: TMenuItem;
    MenuDaqStatusSeparator: TMenuItem;
    MenuDaqInit: TMenuItem;
    MenuDaqDone: TMenuItem;
    MenuDaqStart: TMenuItem;
    MenuDaqStop: TMenuItem;
    MenuDaqLoad: TMenuItem;
    MenuDaqSave: TMenuItem;
    MenuDaqRemedy: TMenuItem;
    MenuDaqClear: TMenuItem;
    MenuDaqWindows: TMenuItem;
    MenuDaqNewWindow: TMenuItem;
    MenuDaqViewCurveTable: TMenuItem;
    MenuDaqEditCurve: TMenuItem;
    MenuDaqHelp: TMenuItem;
    MenuDaqEditNote: TMenuItem;
    MenuDaqEditConfig: TMenuItem;
    MenuDaqSound: TMenuItem;
    MenuDaqMerge: TMenuItem;
    MenuDaqClock: TMenuItem;
    MenuDaqCalculator: TMenuItem;
    MenuDaqDevice: TMenuItem;
    MenuDaqIntegral: TMenuItem;
    MenuDaqMedianFilter: TMenuItem;
    MenuDaqSmoothing: TMenuItem;
    MenuDaqCutRect: TMenuItem;
    MenuDaqMarker: TMenuItem;
    MenuDaqInfo: TMenuItem;
    MenuDaqNavigator: TMenuItem;
    MenuDaqStatus: TMenuItem;
    OpenDialogCfg: TOpenDialog;
    OpenDialogDaq: TOpenDialog;
    OpenDialogDat: TOpenDialog;
    TimerDaqStatus: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure ActionListDaqUpdate(Action: TBasicAction; var Handled: Boolean);
    procedure ActionDaqInitExecute(Sender: TObject);
    procedure ActionDaqDoneExecute(Sender: TObject);
    procedure ActionDaqStartExecute(Sender: TObject);
    procedure ActionDaqStopExecute(Sender: TObject);
    procedure ActionDaqLoadExecute(Sender: TObject);
    procedure ActionDaqSaveExecute(Sender: TObject);
    procedure ActionDaqRemedyExecute(Sender: TObject);
    procedure ActionDaqClearExecute(Sender: TObject);
    procedure ActionDaqWindowsExecute(Sender: TObject);
    procedure ActionDaqNewWindowExecute(Sender: TObject);
    procedure ActionDaqViewCurveTableExecute(Sender: TObject);
    procedure ActionDaqEditCurveExecute(Sender: TObject);
    procedure ActionDaqHelpExecute(Sender: TObject);
    procedure ActionDaqEditNoteExecute(Sender: TObject);
    procedure ActionDaqEditConfigExecute(Sender: TObject);
    procedure ActionDaqSoundExecute(Sender: TObject);
    procedure ActionDaqMergeExecute(Sender: TObject);
    procedure ActionDaqClockExecute(Sender: TObject);
    procedure ActionDaqCalculatorExecute(Sender: TObject);
    procedure ActionDaqDeviceExecute(Sender: TObject);
    procedure ActionDaqIntegralExecute(Sender: TObject);
    procedure ActionDaqMedianFilterExecute(Sender: TObject);
    procedure ActionDaqSmoothingExecute(Sender: TObject);
    procedure ActionDaqCutRectExecute(Sender: TObject);
    procedure ActionDaqMarkerExecute(Sender: TObject);
    procedure ActionDaqInfoExecute(Sender: TObject);
    procedure ActionDaqNavigatorExecute(Sender: TObject);
    procedure ActionDaqStatusExecute(Sender: TObject);
    procedure OpenDialogCfgTypeChange(Sender: TObject);
    procedure OpenDialogDaqTypeChange(Sender: TObject);
    procedure TimerDaqStatusTimer(Sender: TObject);
    procedure StatusBarClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
   AlertLevel : Integer;
   LastErrors : Integer;
   CurrErrors : Integer;
   procedure  PredProcessing;
   procedure  PostProcessing;
  public
    { Public declarations }
    procedure AfterConstruction; override;
    procedure UpdateCommands; override;
    procedure UpdateStatusBar;
    procedure ClickStatusBar;
    procedure DaqSaveExecute;
  end;

const
  FormDaqControlDialog : TFormDaqControlDialog = nil;

procedure OpenFormDaqControlDialog(Mode:Integer=1);
function  GoHomeFormDaqControlDialog:Boolean;
procedure KillFormDaqControlDialog;

implementation

uses
 _crw_daqsys,
 _crw_daqreaddat,
 Form_ListBoxSelection,
 Form_DaqDateTimeCalculator,
 Form_DaqDeviceControl,
 Form_TextEditor;

{$R *.lfm}

procedure OpenFormDaqControlDialog(Mode:Integer=1);
begin
 try
  if not (FormDaqControlDialog is TFormDaqControlDialog) then begin
   Application.CreateForm(TFormDaqControlDialog,FormDaqControlDialog);
   FormDaqControlDialog.Master:=@FormDaqControlDialog;
  end;
  if (FormDaqControlDialog is TFormDaqControlDialog) then begin
   FormDaqControlDialog.AdjustDesktopByForm(ApplicationMainForm);
   FormDaqControlDialog.UpdateStatusBar;
   FormDaqControlDialog.ActionDaqSound.Checked:=Daq.SoundOn;
   FormDaqControlDialog.ToolButtonDaqSound.Down:=Daq.SoundOn;
   FormDaqControlDialog.WindowState:=wsNormal;
   FormDaqControlDialog.Show;
   FormDaqControlDialog.BringToFront;
   FormDaqControlDialog.UpdateCommands;
   if HasFlags(Mode,1) then begin
    if GoHomeFormDaqControlDialog then begin
     FormDaqControlDialog.IsFixedTop:=true;
     FormDaqControlDialog.IsFixedLeft:=true;
    end else LiftFlags(Mode,2,true);
   end;
   if HasFlags(Mode,2) then begin
    FormDaqControlDialog.Top:=0;
    FormDaqControlDialog.Left:=0;
    FormDaqControlDialog.FixedTop:=0;
    FormDaqControlDialog.FixedLeft:=0;
    FormDaqControlDialog.IsFixedTop:=true;
    FormDaqControlDialog.IsFixedLeft:=true;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'OpenFormDaqControlDialog');
 end;
end;

procedure KillFormDaqControlDialog;
begin
 Kill(TObject(FormDaqControlDialog));
end;

function GoHomeFormDaqControlDialog:Boolean;
var n:Integer; act:LongString;
begin
 Result:=false;
 if Assigned(FormDaqControlDialog) then
 try
  act:='';
  if ReadIniFileString(SysIniFile,SectSystem(1),'Action:FormDaqControlDialog.GoHome%s',act,efConfigNC,svConfigNC) then begin
   if IsLexeme(act,lex_AtCmnd) then begin
    n:=SendToMainConsole('@silent '+act+EOL);
    Result:=(n>0);
    Exit;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'GoHomeFormDaqControlDialog');
 end;
end;

procedure TFormDaqControlDialog.AfterConstruction;
begin
 inherited AfterConstruction;
 AddonSdiFlags(sf_SdiControl+sf_SdiDaqCtrlDlg);
end;

procedure TFormDaqControlDialog.UpdateCommands;
var Exposed,DaqLoaded,DaqStarted,DaqLoadedAndNotStarted,DaqExposed:Boolean;
begin
 inherited UpdateCommands;
 try
  Exposed:=FormIsExposed(Self);
  DaqExposed:=true {or Exposed};
  DaqLoaded:=DaqExposed and Daq.Ok and Daq.Timer.IsStart;
  DaqStarted:=DaqExposed and DaqLoaded and  Daq.AcqTimer.IsStart;
  DaqLoadedAndNotStarted:=DaqExposed and DaqLoaded and not DaqStarted;
  ActionFileSave.Enabled:=False;
  ActionFileSaveAs.Enabled:=False;
  ActionFilePrint.Enabled:=False;
  ActionDaqInit.Enabled:=DaqExposed and not DaqLoaded;
  ActionDaqSave.Enabled:=DaqLoaded;
  ActionDaqWindows.Enabled:=DaqLoaded;
  ActionDaqNewWindow.Enabled:=DaqLoaded;
  ActionDaqViewCurveTable.Enabled:=DaqLoaded;
  ActionDaqHelp.Enabled:=DaqLoaded;
  ActionDaqEditNote.Enabled:=DaqLoaded;
  ActionDaqSound.Enabled:=DaqLoaded and UsesBlaster;
  ActionDaqDevice.Enabled:=DaqLoaded;
  ActionDaqInfo.Enabled:=DaqLoaded;
  ActionDaqNavigator.Enabled:=DaqLoaded;
  ActionDaqDone.Enabled:=DaqLoadedAndNotStarted;
  ActionDaqStart.Enabled:=DaqLoadedAndNotStarted;
  ActionDaqClear.Enabled:=DaqLoadedAndNotStarted;
  ActionDaqRemedy.Enabled:=DaqLoadedAndNotStarted;
  ActionDaqStop.Enabled:=DaqStarted;
  ActionDaqSmoothing.Enabled:=Exposed;
  ActionDaqMerge.Enabled:=Exposed;
  ActionDaqEditCurve.Enabled:=Exposed;
  ActionDaqMarker.Enabled:=Exposed;
  ActionDaqLoad.Enabled:=Exposed;
  ActionDaqEditConfig.Enabled:=Exposed;
  ActionDaqClock.Enabled:=Exposed;
  ActionDaqCalculator.Enabled:=Exposed;
  ActionDaqIntegral.Enabled:=Exposed;
  ActionDaqMedianFilter.Enabled:=Exposed;
  ActionDaqCutRect.Enabled:=Exposed;
  ActionDaqStatus.Enabled:=DaqLoaded;
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure TFormDaqControlDialog.UpdateStatusBar;
begin
 if (Self=nil) then Exit;
 if Daq.Ok and Daq.Timer.IsStart
 then StatusBar.SimpleText:=ExtractFileNameExt(Daq.ConfigFile)
 else StatusBar.SimpleText:=RusEng('DAQ не загружен.','DAQ not loaded.');
end;

procedure TFormDaqControlDialog.ClickStatusBar;
var msg:LongString; ms,ss,mm,hh,dd:int64;
const BriefStatus:Boolean=true;
begin
 if (Self=nil) then Exit;
 try
  if Daq.Ok and Daq.Timer.isStart then begin
   if Daq.Ok and Daq.AcqTimer.IsStart
   then msg:=RusEng('DAQ работает. ','DAQ is running.')
   else msg:=RusEng('DAQ загружен. ','DAQ is loaded.');
   SendToMainConsole('@silent @echo '+StdDateTimePrompt+msg+EOL);
   SendToMainConsole('@silent @echo '+RusEng('Файл ','File ')+GetRealFilePathName(Daq.ConfigFile)+EOL);
   SendToMainConsole('@silent @echo '+Daq.BriefDaqSysInfoCounters+EOL);
   if BriefStatus then begin
    msg:=Daq.BriefDaqSysBugsCounters;
    SendToMainConsole('@silent @echo '+msg+EOL);
   end else SendToMainConsole('@silent @menu run FormDaqControlDialog.ActionDaqStatus'+EOL);
   if Daq.Timer.IsStart then begin
    ms:=Trunc(mSecNow-Daq.Timer.StartTime);
    ss:=ms div 1000; mm:=ss div 60; hh:=mm div 60; dd:=hh div 24;
    msg:=Format('%d-%2.2d:%2.2d:%2.2d.%d  (%1.3f sec)',[dd,(hh mod 24),(mm mod 60),(ss mod 60),(ms mod 1000),1e-3*ms]);
    SendToMainConsole('@silent @echo '+RusEng('Время по часам DAQ: ','Time by DAQ clock: ')+msg+EOL);
   end;
   if Daq.AcqTimer.IsStart then begin
    ms:=Trunc(mSecNow-Daq.AcqTimer.StartTime);
    ss:=ms div 1000; mm:=ss div 60; hh:=mm div 60; dd:=hh div 24;
    msg:=Format('%d-%2.2d:%2.2d:%2.2d.%d  (%1.3f sec)',[dd,(hh mod 24),(mm mod 60),(ss mod 60),(ms mod 1000),1e-3*ms]);
    SendToMainConsole('@silent @echo '+RusEng('Время работы DAQ: ','DAQ work time: ')+msg+EOL);
   end;
  end else begin
   msg:=RusEng('DAQ не загружен.','DAQ not loaded.');
   SendToMainConsole('@silent @echo '+StdDateTimePrompt+msg+EOL);
  end;
  SendToMainConsole('@silent @view showfront FormConsoleWindow'+EOL);
  //SendToMainConsole('@silent @menu run FormCrwDaq.ActionWindowsOpenSystemConsole'+EOL);
  //SendToMainConsole('@silent @view pos FormConsoleWindow '+Format('%d %d',[BoundsRect.Right,Top])+EOL);
  UpdateStatusBar;
 except
  on E:Exception do BugReport(E,Self,'ClickStatusBar');
 end;
end;

procedure TFormDaqControlDialog.FormCreate(Sender: TObject);
begin
 inherited;
 CloseAction:=caMinimize;
 StatusBar.Cursor:=crHelp;
 Caption:=GetDaqControlDialogCaption;
 UpdateMenu(MenuDaq,
            RusEng('DAQ','DAQ')+MenuRightSpace,
            RusEng('Меню DAQ системы.','DAQ system menu.'),
            0);
 UpdateMenu(MenuDaqSubControl,
            RusEng('Управление','Control'),
            RusEng('Функции управления DAQ системой.','DAQ Control submenu.'),
            0);
 UpdateMenu(MenuDaqSubSeans,
            RusEng('Сеанс','Seans'),
            RusEng('Функции сеанса DAQ.','DAQ Seans submenu.'),
            0);
 UpdateMenu(MenuDaqSubView,
            RusEng('Вид','View'),
            RusEng('Функции отображения DAQ.','DAQ View submenu.'),
            0);
 UpdateMenu(MenuDaqSubHelp,
            RusEng('Помощь','Help'),
            RusEng('Вспомогательные функции DAQ.','DAQ Help/Info submenu.'),
            0);
 UpdateMenu(MenuDaqSubTools,
            RusEng('Инструменты','Tools'),
            RusEng('Инструменты DAQ.','DAQ Tools submenu.'),
            0);
 UpdateMenu(MenuDaqSubMath,
            RusEng('Обработка','Math'),
            RusEng('Обработка данных DAQ.','DAQ Math submenu.'),
            0);
 UpdateMenu(MenuDaqInit,
            RusEng('Инициализация','Init'),
            RusEng('Начать сеанс DAQ системы.','Load DAQ system.'),
            0);
 UpdateMenu(MenuDaqDone,
            RusEng('Завершение','Done'),
            RusEng('Завершить сеанс DAQ системы.','Finalize DAQ system.'),
            0);
 UpdateMenu(MenuDaqStart,
            RusEng('Старт','Start'),
            RusEng('Стартовать DAQ систему.','Start DAQ system.'),
            0);
 UpdateMenu(MenuDaqStop,
            RusEng('Стоп','Stop'),
            RusEng('Остановить DAQ систему.','Stop DAQ system.'),
            0);
 UpdateMenu(MenuDaqLoad,
            RusEng('Загрузить','Load'),
            RusEng('Загрузить файл данных.','Load data file.'),
            0);
 UpdateMenu(MenuDaqSave,
            RusEng('Сохранить','Save'),
            RusEng('Сохранить файл данных.','Save data file.'),
            0);
 UpdateMenu(MenuDaqRemedy,
            RusEng('Восстановить','Remedy'),
            RusEng('Восстановить данные из файла автосохранения.','Remedy data from backup file.'),
            0);
 UpdateMenu(MenuDaqClear,
            RusEng('Очистить','Clear'),
            RusEng('Очистить кривые.','Clear data curves.'),
            0);
 UpdateMenu(MenuDaqWindows,
            RusEng('Окна','Window operations'),
            RusEng('Операции с окнами DAQ.','DAQ window operations.'),
            0);
 UpdateMenu(MenuDaqNewWindow,
            RusEng('Новое окно','New window'),
            RusEng('Открыть новое окно DAQ.','Open new DAQ window.'),
            0);
 UpdateMenu(MenuDaqViewCurveTable,
            RusEng('Посмотреть таблицу','View DAQ curve table'),
            RusEng('Просмотр кривой в виде таблицы.','View DAQ curve table.'),
            0);
 UpdateMenu(MenuDaqEditCurve,
            RusEng('Редактировать кривую','Edit curve'),
            RusEng('Редактировать кривую.','Edit curve.'),
            0);
 UpdateMenu(MenuDaqHelp,
            RusEng('Справка','Help'),
            RusEng('Вызов сравки по данной DAQ системе.','Help on this DAQ system.'),
            0);
 UpdateMenu(MenuDaqEditNote,
            RusEng('Заметки','Edit log'),
            RusEng('Вызвать редактирование журнального файла.','Edit DAQ log file.'),
            0);
 UpdateMenu(MenuDaqEditConfig,
            RusEng('Редактировать конфигурацию','Edit config'),
            RusEng('Вызвать редактирование конфигурационного файла.','Edit config file.'),
            0);
 UpdateMenu(MenuDaqSound,
            RusEng('Звук','Sound'),
            RusEng('Включение-выключение звуковых эффектов.','Sound on/off.'),
            0);
 UpdateMenu(MenuDaqMerge,
            RusEng('Склеить файлы *.DAT','Merge *.DAT files'),
            RusEng('Прочитать и склеить файлы *.DAT.','Read and merge *.DAT data files.'),
            0); 
 UpdateMenu(MenuDaqClock,
            RusEng('Расчет времени','Calc date/time'),
            RusEng('Калькулятор расчетов с календарным временем.','Time calculations.'),
            0);
 UpdateMenu(MenuDaqCalculator,
            RusEng('Калькуляторы','Calculators'),
            RusEng('Вызов калькуляторов.','Call calculators.'),
            0);
 UpdateMenu(MenuDaqDevice,
            RusEng('Менеджер устройств','Device manager'),
            RusEng('Вызвать диалог менеджера устройств DAQ.','Call DAQ device manager.'),
            0);
 UpdateMenu(MenuDaqIntegral,
            RusEng('Интеграл','Integral'),
            RusEng('Сосчитать определенный интеграл.','Calculate curve integral.'),
            0);
 UpdateMenu(MenuDaqMedianFilter,
            RusEng('Медианный фильтр','Median filter'),
            RusEng('Обработать кривую медианным фильтром.','Apply median filter to curve.'),
            0);
 UpdateMenu(MenuDaqSmoothing,
            RusEng('Сглаживание','Smoothing'),
            RusEng('Сгладить кривую.','Curve smoothing.'),
            0);
 UpdateMenu(MenuDaqCutRect,
            RusEng('Вырезать','Cut region'),
            RusEng('Вырезать фрагмент кривой.','Cut curve region.'),
            0);
 UpdateMenu(MenuDaqMarker,
            RusEng('Маркер','Marker'),
            RusEng('Запомнить маркер.','Set marker.'),
            0);
 UpdateMenu(MenuDaqInfo,
            RusEng('Информация','Information'),
            RusEng('Информация по DAQ системе.','Information about DAQ system.'),
            0);
 UpdateMenu(MenuDaqNavigator,
            RusEng('Навигатор','Navigator'),
            RusEng('Вызвать главное окно DAQ системы.','Call main DAQ system window.'),
            0);
 UpdateMenu(MenuDaqStatus,
            RusEng('Статус DAQ','DAQ status'),
            RusEng('Показать ошибки времени исполнения DAQ.','View DAQ runtime system errors.'),
            0);
 AlertLevel:=0;
 LastErrors:=0;
 CurrErrors:=0;
end;

procedure TFormDaqControlDialog.ActionListDaqUpdate(Action: TBasicAction; var Handled: Boolean);
begin
 Handled:=true;
end;

procedure TFormDaqControlDialog.ActionDaqInitExecute(Sender: TObject);
var Ext,FName,Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqInit)<0 then Exit;
 try
  if Daq.Ok then
  if not Trouble(Daq.Timer.IsStart,RusEng('DAQ: Система уже загружена!','DAQ is already loaded!')) then begin
   OpenDialogCfg.Title:=RusEng('Открыть конфигурацию','Open configuration');
   OpenDialogCfg.Filter:=RusEng('Файлы конфигурации   (*.cfg)|*.cfg|'+
                                'Все остальные файлы    (*.*)|*.*|',
                                'Configuration files  (*.cfg)|*.cfg|'+
                                'All other files        (*.*)|*.*|');
   OpenDialogCfg.FileName:=LowerCase(Daq.ConfigFile);
   if IsEmptyStr(OpenDialogCfg.FileName)
   then OpenDialogSelectType(OpenDialogCfg,'*.cfg')
   else OpenDialogSelectType(OpenDialogCfg,OpenDialogCfg.FileName);
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqInit,'RT'),'');
   if ExecuteFileDialog(GuardOpenDialog(OpenDialogCfg),Params) then begin
    FName:=UnifyAlias(OpenDialogCfg.FileName);
    Ext:=UnifyAlias(OpenDialogCfg.DefaultExt);
    if IsEmptyStr(Ext) or IsWildCard(Ext)
    then FName:=UnifyFileAlias(FName)
    else FName:=UnifyFileAlias(ForceExtension(FName,Ext));
    if IsNonEmptyStr(FName) then begin
     Daq.InitSession(FName);
    end;
   end;
   BringToFront;
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqInitExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqDoneExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqDone)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqDone,'RT'),'');
  if Daq.Ok then
  if Daq.CheckSessionStarted and Daq.ConfirmExit(Params) then begin
   Daq.DoneSession;
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqDoneExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqStartExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_User,ActionDaqStart)<0 then Exit;
 try
  if Daq.Ok then begin
   Daq.Start;
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqStartExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqStopExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_User,ActionDaqStop)<0 then Exit;
 try
  if Daq.Ok then begin
   Daq.Stop;
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqStopExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqLoadExecute(Sender: TObject);
var Ext,FName,Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqLoad)<0 then Exit;
 try
  if Daq.Ok then begin
   OpenDialogDaq.Title:=RusEng('Загрузить данные из DAQ файла','Load data from DAQ file');
   OpenDialogDaq.Filter:=RusEng('Файлы данных DAQ    (*.daq)|*.daq|'+
                                'Все остальные файлы (*.*)|*.*|',
                                'DAQ data files  (*.daq)|*.daq|'+
                                'All other files (*.*)|*.*|');
   OpenDialogDaq.FileName:=LowerCase(Daq.DataFile);
   if IsEmptyStr(OpenDialogDaq.FileName)
   then OpenDialogSelectType(OpenDialogDaq,'*.daq')
   else OpenDialogSelectType(OpenDialogDaq,OpenDialogDaq.FileName);
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqLoad,'RT'),'');
   if ExecuteFileDialog(GuardOpenDialog(OpenDialogDaq),Params) then begin
    FName:=UnifyAlias(OpenDialogDaq.FileName);
    Ext:=UnifyAlias(OpenDialogDaq.DefaultExt);
    if IsEmptyStr(Ext) or IsWildCard(Ext)
    then FName:=UnifyFileAlias(FName)
    else FName:=UnifyFileAlias(ForceExtension(FName,Ext));
    if IsNonEmptyStr(FName) then begin
     Daq.LoadData(FName);
    end;
   end;
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqLoadExecute');
 end;
end;

procedure TFormDaqControlDialog.OpenDialogDaqTypeChange(Sender: TObject);
begin
 OpenDialogTypeChangeStdExecute(OpenDialogDaq);
end;

procedure TFormDaqControlDialog.DaqSaveExecute;
var Ext,FName,Params:LongString;
begin
 try
  if Daq.Ok then
  if Daq.CheckSessionStarted then begin
   SaveDialog.Title:=RusEng('Сохранить данные в DAQ файл','save data to DAQ file');
   SaveDialog.Filter:=RusEng('Файлы данных DAQ    (*.daq)|*.daq|'+
                             'Все остальные файлы (*.*)|*.*|',
                             'DAQ data files  (*.daq)|*.daq|'+
                             'All other files (*.*)|*.*|');
   SaveDialog.FileName:=LowerCase(Daq.DataFile);
   if IsEmptyStr(SaveDialog.FileName)
   then OpenDialogSelectType(SaveDialog,'*.daq')
   else OpenDialogSelectType(SaveDialog,SaveDialog.FileName);
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqSave,'RT'),'');
   if ExecuteFileDialog(GuardOpenDialog(SaveDialog),Params) then begin
    FName:=UnifyAlias(SaveDialog.FileName);
    Ext:=UnifyAlias(SaveDialog.DefaultExt);
    if IsEmptyStr(Ext) or IsWildCard(Ext)
    then FName:=UnifyFileAlias(FName)
    else FName:=UnifyFileAlias(ForceExtension(FName,Ext));
    if IsNonEmptyStr(FName) then begin
     Daq.SaveData(FName);
    end;
   end;
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'DaqSaveExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqSaveExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_User,ActionDaqSave)<0 then Exit;
 DaqSaveExecute;
end;

procedure TFormDaqControlDialog.ActionDaqRemedyExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqRemedy)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqRemedy,'RT'),'');
  if Daq.Ok then begin
   Daq.Remedy(Params);
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqRemedyExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqClearExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqClear)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqClear,'RT'),'');
  if Daq.Ok then begin
   Daq.CustomClear(Params);
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqClearExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqWindowsExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqWindows)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqWindows,'RT'),'');
  if Daq.Ok then begin
   Daq.WindowsOperations(Params);
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqWindowsExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqNewWindowExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqNewWindow)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqNewWindow,'RT'),'');
  if Daq.Ok then begin
   Daq.OpenDynamicWindow(Params);
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqNewWindowExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqViewCurveTableExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionDaqViewCurveTable)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqViewCurveTable,'RT'),'');
  if Daq.Ok then begin
   Daq.ViewData(Params);
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqViewCurveTableExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqEditCurveExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqEditCurve)<0 then Exit;
 try
  if Ok and Daq.Ok then begin
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqEditCurve,'RT'),'');
   PredProcessing;
   Daq.EditCurvePoint(Params);
   PostProcessing;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqEditCurveExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqHelpExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionDaqHelp)<0 then Exit;
 try
  Daq.OpenHelp;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqHelpExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqEditNoteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_User,ActionDaqEditNote)<0 then Exit;
 try
  Daq.OpenLogBook;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqEditNoteExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqEditConfigExecute(Sender: TObject);
var Ext,FName,Params:LongString; Editor:TFormTextEditor;
begin
 if Guard.CheckAction(ga_Root,ActionDaqEditConfig)<0 then Exit;
 try
  OpenDialogCfg.Title:=RusEng('Открыть конфигурацию','Open configuration');
  OpenDialogCfg.Filter:=RusEng('Файлы конфигурации   (*.cfg)|*.cfg|'+
                               'Файлы инициализации  (*.ini)|*.ini|'+
                               'Файлы мнемосхем      (*.crc)|*.crc|'+
                               'Файлы калибровок     (*.cal)|*.cal|'+
                               'Все остальные файлы    (*.*)|*.*|',
                               'Configuration files  (*.cfg)|*.cfg|'+
                               'Initialization files (*.ini)|*.ini|'+
                               'Circuit files        (*.crc)|*.crc|'+
                               'Calibration files    (*.cal)|*.cal|'+
                               'All other files       (*.*)|*.*|');
  OpenDialogCfg.FileName:=LowerCase(Daq.ConfigFile);
  if IsEmptyStr(OpenDialogCfg.FileName)
  then OpenDialogSelectType(OpenDialogCfg,'*.cfg')
  else OpenDialogSelectType(OpenDialogCfg,OpenDialogCfg.FileName);
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqEditConfig,'RT'),'');
  if ExecuteFileDialog(GuardOpenDialog(OpenDialogCfg),Params) then begin
   FName:=UnifyAlias(OpenDialogCfg.FileName);
   Ext:=UnifyAlias(OpenDialogCfg.DefaultExt);
   if IsEmptyStr(Ext) or IsWildCard(Ext)
   then FName:=UnifyFileAlias(FName)
   else FName:=UnifyFileAlias(ForceExtension(FName,Ext));
   if IsNonEmptyStr(FName) then begin
    Editor:=FindTextEditor(FName,true);
    if Editor.Ok then begin
     Editor.WindowState:=wsNormal;
     Editor.BringToFront;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqEditConfigExecute');
 end;
end;

procedure TFormDaqControlDialog.OpenDialogCfgTypeChange(Sender: TObject);
begin
 OpenDialogTypeChangeStdExecute(OpenDialogCfg);
end;

procedure TFormDaqControlDialog.ActionDaqSoundExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_User,ActionDaqSound)<0 then Exit;
 try
  Daq.SoundOn:=not Daq.SoundOn;
  ActionDaqSound.Checked:=Daq.SoundOn;
  ToolButtonDaqSound.Down:=Daq.SoundOn;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqSoundExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqMergeExecute(Sender: TObject);
var StringList:TStringList; Params:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionDaqMerge)<0 then Exit;
 try
  OpenDialogDat.Title:=RusEng('Прочитать *.dat файлы','Load data from *.dat files');
  OpenDialogDat.Filter:=RusEng('Файлы данных (*.dat)|*.dat|','Data files (*.dat)|*.dat|');
  if IsEmptyStr(OpenDialogDat.FileName)
  then OpenDialogSelectType(OpenDialogDat,AddPathDelim(Daq.DataPath)+'*.dat')
  else OpenDialogSelectType(OpenDialogDat,OpenDialogDat.FileName);
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqMerge,'RT'),'');
  if ExecuteFileDialog(GuardOpenDialog(OpenDialogDat),Params) then
  if OpenDialogDat.Files.Count>0 then begin
   StringList:=TStringList.Create;
   try
    StringList.Text:=OpenDialogDat.Files.Text;
    StringList.Sort;
    ReadDatFiles(StringList.Text);
   finally
    StringList.Free;
    OpenDialogDat.Files.Clear;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqMergeExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqClockExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionDaqClock)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqClock,'RT'),'');
  KillFormDaqDateTimeCalculator;
  if Daq.Ok and Daq.Timer.IsStart
  then OpenFormDaqDateTimeCalculator(Daq.Timer.StartTime,msecnow,Params)
  else OpenFormDaqDateTimeCalculator(0,msecnow,Params);
 except
  on E:Exception do BugReport(E,Self,'ActionDaqClockExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqCalculatorExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionDaqCalculator)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqCalculator,'RT'),'');
  Daq.OpenCalculator(Params);
 except
  on E:Exception do BugReport(E,Self,'ActionDaqCalculatorExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqDeviceExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqDevice)<0 then Exit;
 try
  if Ok then
  if Daq.Ok and Daq.CheckSessionStarted then begin
   //Params:=IfThen(Active,ControlPosParams(ToolButtonDaqDevice,'RT'),'');
   Params:=ControlPosParams(Self,'RT');
   OpenFormDaqDeviceControl(Params);
   UpdateCommands;
   UpdateStatusBar;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqDeviceExecute');
 end;
end;

procedure TFormDaqControlDialog.PredProcessing;
begin
 if Ok then
 if SdiMan.IsMdiMode then WindowState:=wsMinimized;
end;

procedure TFormDaqControlDialog.PostProcessing;
begin
 if Ok then
 if SdiMan.IsMdiMode then WindowState:=wsNormal;
end;

procedure TFormDaqControlDialog.ActionDaqIntegralExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionDaqIntegral)<0 then Exit;
 try
  if Ok then begin
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqIntegral,'RT'),'');
   PredProcessing;
   Daq.Integral(Params);
   PostProcessing;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqIntegralExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqMedianFilterExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqMedianFilter)<0 then Exit;
 try
  if Ok then begin
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqMedianFilter,'RT'),'');
   PredProcessing;
   Daq.MedianFiltr(Params);
   PostProcessing;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqMedianFilterExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqSmoothingExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqSmoothing)<0 then Exit;
 try
  if Ok and Daq.Ok then begin
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqSmoothing,'RT'),'');
   PredProcessing;
   Daq.CurveSmoothing(Params);
   PostProcessing;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqSmoothingExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqCutRectExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_User,ActionDaqCutRect)<0 then Exit;
 try
  if Ok and Daq.Ok then begin
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqCutRect,'RT'),'');
   PredProcessing;
   Daq.CutRegion(Params);
   PostProcessing;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqCutRectExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqMarkerExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionDaqMarker)<0 then Exit;
 try
  if Ok and Daq.Ok then begin
   Params:=IfThen(Active,ControlPosParams(ToolButtonDaqMarker,'RT'),'');
   PredProcessing;
   Daq.SetUserMarker(Params);
   PostProcessing;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqMarkerExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqInfoExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_User,ActionDaqInfo)<0 then Exit;
 try
  if Daq.Ok and Daq.CheckSessionStarted then Daq.DaqInfo;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqInfoExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqNavigatorExecute(Sender: TObject);
var Params:LongString;
begin
 if Guard.CheckAction(ga_Guest,ActionDaqNavigator)<0 then Exit;
 try
  Params:=IfThen(Active,ControlPosParams(ToolButtonDaqNavigator,'RT'),'');
  if Daq.Ok and Daq.CheckSessionStarted then Daq.Navigator(Params);
 except
  on E:Exception do BugReport(E,Self,'ActionDaqNavigatorExecute');
 end;
end;

procedure TFormDaqControlDialog.ActionDaqStatusExecute(Sender: TObject);
var i,aTot:Integer; aErr:packed array[0..255] of Integer;
begin
 if Guard.CheckAction(ga_User,ActionDaqStatus)<0 then Exit;
 if Ok and Daq.Ok and Daq.CheckSessionStarted then
 try
  aTot:=0;
  for i:=0 to 255 do begin
   aErr[i]:=FullDaqDeviceList.ErrorsCount[i];
   inc(aTot,aErr[i]);
  end;
  Echo(Format(RusEng('Зафиксировано ошибок DAQ: %d','DAQ errors fixed: %d'),[aTot]));
  if aTot>0 then begin
   Echo(Format('%4s %10s %s', [RusEng('Код','Code'), RusEng('Число','Count'), RusEng('Справка','Comment')]));
   for i:=0 to 255 do if (aErr[i]>0) then Echo(Format('%4d %10d %s', [i, aErr[i], Daq.ErrMsg[i]]));
  end;
  AlertLevel:=0;
  ActionDaqStatus.ImageIndex:=27;
  SystemConsole.Activate;
 except
  on E:Exception do BugReport(E,Self,'ActionDaqStatusExecute');
 end;
end;

procedure TFormDaqControlDialog.TimerDaqStatusTimer(Sender: TObject);
begin
 if Ok then
 try
  if ActionDaqSound.Enabled then begin
   ActionDaqSound.Checked:=Daq.SoundOn;
   ToolButtonDaqSound.Down:=Daq.SoundOn;
  end;
  if ActionDaqStatus.Enabled then begin
   if Daq.Timer.isStart then begin
    CurrErrors:=FullDaqDeviceList.ErrorsTotal;
    if (CurrErrors<>LastErrors) then begin
     if (CurrErrors>LastErrors) then AlertLevel:=2;
     if (CurrErrors<LastErrors) then AlertLevel:=0;
     LastErrors:=CurrErrors;
    end;
   end;
   ActionDaqStatus.ImageIndex:=27+AlertLevel;
   if AlertLevel>1 then dec(AlertLevel);
  end;
 except
  on E:Exception do BugReport(E,Self,'TimerDaqStatusTimer');
 end;
end;

procedure TFormDaqControlDialog.StatusBarClick(Sender: TObject);
begin
 inherited;
 ClickStatusBar;
end;

procedure TFormDaqControlDialog.FormActivate(Sender: TObject);
begin
 inherited;
 UpdateCommands;
 UpdateStatusBar;
end;

procedure TFormDaqControlDialog.FormShow(Sender: TObject);
begin
 inherited;
 UpdateCommands;
 UpdateStatusBar;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_daqcontroldialog;
begin
end;

procedure Free_form_daqcontroldialog;
begin
end;

initialization

 Init_form_daqcontroldialog;

finalization

 Free_form_daqcontroldialog;

end.

//////////////
// END OF FILE
//////////////

