////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// CRW-DAQ Software Device Family.                                            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231206 - Modified for FPC (A.K.)                                         //
// 20240531 - Improved UTF8 compatibility.                                    //
// 20240604 - Some command handling moved to OnIdle event                     //
// 20250127 - Modified handling of FixedTop/Left/Width/Height                 //
// 20250129 - Use TAtomicCounter                                              //
////////////////////////////////////////////////////////////////////////////////

unit _crw_softdev; // CRW-DAQ Software Device Family

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow,
 Form_SurfWindow, Form_CircuitWindow,
 Form_ConsoleWindow, Form_Calculator,
 Form_ListBoxSelection, Form_UartTerminal,
 Form_CalibDialog, Form_DaqEditTagDialog,
 Form_TabWindow, Form_SpectrWindow, 
 Unit_CrwDaqMessages,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_ee, _crw_hl, _crw_pio,
 _crw_curves, _crw_riff, _crw_polling,
 _crw_couple, _crw_calib,
 _crw_daqtags, _crw_daqevnt,
 _crw_daqsys, _crw_daqdev, _crw_dbglog,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 Конструктор создает устройство Software по модели
 *******************************************************************************
 }
function SoftwareDeviceConstructor(const Name,Family,Model,ConfigFile:LongString):TDaqDevice;

 {
 *******************************************************************************
 Базовый класс для устройств типа Software - то есть функциональных устройств,
 которые жестко не связаны с аппаратурой и служат для генерации или
 преобразования данных.
 Каждое устройство выполняется в отдельном потоке Polling. Поток периодически
 вызывает процедуру Poll, в которой по таймеру InquiryTimer или по щелчку на
 сенсор вызывается метод Handler, который и должен делать основную полезную
 работу.
 Процедуры, которые не могут быть выполнены в потоке опроса, помещаются в очередь
 CmdFifo процедурой PostDeferredCommand, а затем асинхронно выполняются в методе
 ExecuteDeferredCommand в основном потоке VCL. Метод PostDeferredCommand вызывает
 PostDeferredCallbackRequest для отложенного выполнения команд в основном потоке.
 Вызвавший поток при этом не останавливается, а продолжает работу.
 Метод DeferredCommandProblem, вызывается если команда не может быть обработана.
 Свойства  ClickXXX вызываются из обработчика Handle для обработки нажатия
 сенсоров. После выполнения Handle они автоматически сбрасываются в ноль.
 *******************************************************************************
 }
type
 TSoftwareDevice = class(TDaqDevice)
 private
  myPolling    : TPolling;
  myCmdFifo    : TFifo;
  myCmdList    : TText;
  myCmdIdle    : TText;
  myIdleList   : TStringList;
  myLatchEvent : TSensorEvent;
  function    GetPolling:TPolling;
  function    GetCmdFifo:TFifo;
  function    GetClickWhat:TEventWhat;
  function    GetClickWrote:Integer;
  function    GetClickButton:Integer;
  function    GetClickTag:Integer;
  function    GetClickSensor:LongString;
 public
  property    Polling     : TPolling    read GetPolling;
  property    CmdFifo     : TFifo       read GetCmdFifo;
  property    ClickWhat   : TEventWhat  read GetClickWhat;
  property    ClickWrote  : Integer     read GetClickWrote;
  property    ClickButton : Integer     read GetClickButton;
  property    ClickTag    : Integer     read GetClickTag;
  property    ClickSensor : LongString  read GetClickSensor;
  function    ClickRead:TEventWhat;
  function    ClickWrite(const Lines:LongString):TEventWhat;
  procedure   ClickFree;
  function    ClickSensorParams(const aParam:LongString):LongString;
  constructor Create(const aName:LongString);
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
  procedure   Config(FileName:LongString); override;
  function    GetProperty(TheText:TText):TText; override;
  procedure   Poll; override;
  procedure   Awake; override;
  procedure   Handler; virtual;
  function    GotEvents:Boolean; override;
  procedure   ExecuteDeferredCommands;
  procedure   PollIdleDeferredCommands(Loops:Integer=1);
  function    IsIdleCommand(const Cmd:LongString):Boolean;
  procedure   AddIdleCommands(const aIdleCommands:LongString);
  function    PostDeferredCommand(const Cmd:LongString):boolean;
  function    PostDeferredCallbackRequest(Callback:TCrwDaqDeferredCallbackRequestProc; Sender:TObject):boolean;
  procedure   ExecuteDeferredCommand(var Cmd, Arg:LongString); virtual;
  procedure   DeferredCommandProblem(const Cmd:LongString); virtual;
 public
  class var HandleDpcOnIdle:Boolean;
 end;

const
 DefaultDeferredCommandFifoSize = 1024 * 16;

 {
 *******************************************************************************
 Список обеспечивает хранение и сканирование устройств типа Software.
 Метод EnablePolling разрешает или запрещает опрос всех устройств.
 Метод Start вызывает наследуемый Start и затем разрешает опрос устройств.
 Метод Stop останавливает опрос и затем вызывает наследуемый Stop.
 Метод Poll ничего не делает, так как опрос каждого устройства идет в отдельном
 потоке.
 *******************************************************************************
 }
type
 TSoftwareDeviceList = class(TDaqDeviceList)
 private
  function  GetSoftDev(Index:Integer):TSoftwareDevice;
  procedure SetSoftDev(Index:Integer; aSoftDev:TSoftwareDevice);
 public
  property  SoftDev[i:Integer] : TSoftwareDevice read GetSoftDev write SetSoftDev; default;
  procedure EnablePolling(aEnable:Boolean);
  function  Start:Boolean; override;
  procedure Stop; override;
  procedure Poll; override;
 end;

function SoftwareDeviceList:TSoftwareDeviceList;

type
 ELoadSPD = class(ESoftException);

implementation

uses
 _crw_softdevscript,
 _crw_softdevdialog,
 _crw_softdevconstgenerator,
 _crw_softdevbitsetgenerator,
 _crw_daqpascaldevice;

var // DebugLog channel
 dlc_ExecDpc:Integer=0;

var // DPC Post/Exec balance
 DpcBalance:TAtomicCounter=nil;

procedure InitDpcCounters;
begin
 LockedInit(DpcBalance);
end;

procedure FreeDpcCounters;
begin
 LockedFree(DpcBalance);
end;

 //////////////////////////////////////////
 // Private Hasher for fast string parsing.
 //////////////////////////////////////////
type
 TStringIdentifier = (
  sid_Unknown,
  sid_SOFTWARE,
  sid_DIALOG,
  sid_CONSTGENERATOR,
  sid_BITSETGENERATOR,
  sid_SCRIPT,
  sid_PROGRAM,
  sid_cmdACTION,
  sid_cmdCLEAR,
  sid_cmdCLEARDEVICE,
  sid_cmdSTART,
  sid_cmdSTOP,
  sid_cmdCLEARCURVE,
  sid_cmdWINDRAW,
  sid_cmdWINSHOW,
  sid_cmdWINHIDE,
  sid_cmdWINSELECT,
  sid_cmdSAVECRW,
  sid_1Bit,
  sid_4Bit,
  sid_8Bit,
  sid_16Bit,
  sid_24Bit,
  sid_32Bit,
  sid_TOP,
  sid_LEFT,
  sid_WIDTH,
  sid_HEIGHT,
  sid_OPTIONS,
  sid_LEGENDX,
  sid_LEGENDY,
  sid_RANGE,
  sid_ROI,
  sid_SELECTCURVE,
  sid_RELOAD,
  sid_SCROLL,
  sid_TOOLBARHEIGHT,
  sid_EVAL,
  sid_DRAWSENSOR,
  sid_UPDATESENSOR,
  sid_LEGEND,
  sid_TITLE,
  sid_PHI,
  sid_PSI,
  sid_SCALEX,
  sid_SCALEY,
  sid_SCALEZ,
  sid_MODE,
  sid_SLICEX,
  sid_SLICEY,
  sid_SAVECRW,
  sid_SAVEBMP,
  sid_LOADSPD,
  sid_SAVESPD,
  sid_Unused
 );

const
 Hasher:THashList=nil;

procedure FreeHasher;
begin
 Kill(Hasher);
end;

procedure InitHasher;
 procedure AddSid(const key:LongString; sid:TStringIdentifier);
 begin
  Hasher.KeyedLinks[key]:=Ord(sid);
 end;
begin
 if (Hasher<>nil) then Exit;
 Hasher:=NewHashList(false,HashList_DefaultHasher);
 Hasher.Master:=@Hasher;
 ////////////////////////////////////////////
 // Hash List for fast strings identification
 ////////////////////////////////////////////
 AddSid( 'SOFTWARE'          , sid_SOFTWARE);
 AddSid( 'DIALOG'            , sid_DIALOG);
 AddSid( 'CONSTGENERATOR'    , sid_CONSTGENERATOR);
 AddSid( 'BITSETGENERATOR'   , sid_BITSETGENERATOR);
 AddSid( 'SCRIPT'            , sid_SCRIPT);
 AddSid( 'PROGRAM'           , sid_PROGRAM);
 AddSid( '@ACTION'           , sid_cmdACTION);
 AddSid( '@CLEAR'            , sid_cmdCLEAR);
 AddSid( '@CLEARDEVICE'      , sid_cmdCLEARDEVICE);
 AddSid( '@START'            , sid_cmdSTART);
 AddSid( '@STOP'             , sid_cmdSTOP);
 AddSid( '@CLEARCURVE'       , sid_cmdCLEARCURVE);
 AddSid( '@WINDRAW'          , sid_cmdWINDRAW);
 AddSid( '@WINSHOW'          , sid_cmdWINSHOW);
 AddSid( '@WINHIDE'          , sid_cmdWINHIDE);
 AddSid( '@WINSELECT'        , sid_cmdWINSELECT);
 AddSid( '@SAVECRW'          , sid_cmdSAVECRW);
 AddSid( '1Bit'              , sid_1Bit);
 AddSid( '4Bit'              , sid_4Bit);
 AddSid( '8Bit'              , sid_8Bit);
 AddSid( '16Bit'             , sid_16Bit);
 AddSid( '24Bit'             , sid_24Bit);
 AddSid( '32Bit'             , sid_32Bit);
 AddSid( 'TOP'               , sid_TOP);
 AddSid( 'LEFT'              , sid_LEFT);
 AddSid( 'WIDTH'             , sid_WIDTH);
 AddSid( 'HEIGHT'            , sid_HEIGHT);
 AddSid( 'OPTIONS'           , sid_OPTIONS);
 AddSid( 'LEGENDX'           , sid_LEGENDX);
 AddSid( 'LEGENDY'           , sid_LEGENDY);
 AddSid( 'RANGE'             , sid_RANGE);
 AddSid( 'ROI'               , sid_ROI);
 AddSid( 'SELECTCURVE'       , sid_SELECTCURVE);
 AddSid( 'RELOAD'            , sid_RELOAD);
 AddSid( 'SCROLL'            , sid_SCROLL);
 AddSid( 'TOOLBARHEIGHT'     , sid_TOOLBARHEIGHT);
 AddSid( 'EVAL'              , sid_EVAL);
 AddSid( 'DRAWSENSOR'        , sid_DRAWSENSOR);
 AddSid( 'UPDATESENSOR'      , sid_UPDATESENSOR);
 AddSid( 'LEGEND'            , sid_LEGEND);
 AddSid( 'TITLE'             , sid_TITLE);
 AddSid( 'PHI'               , sid_PHI);
 AddSid( 'PSI'               , sid_PSI);
 AddSid( 'SCALEX'            , sid_SCALEX);
 AddSid( 'SCALEY'            , sid_SCALEY);
 AddSid( 'SCALEZ'            , sid_SCALEZ);
 AddSid( 'MODE'              , sid_MODE);
 AddSid( 'SLICEX'            , sid_SLICEX);
 AddSid( 'SLICEY'            , sid_SLICEY);
 AddSid( 'SAVECRW'           , sid_SAVECRW);
 AddSid( 'SAVEBMP'           , sid_SAVEBMP);
 AddSid( 'LOADSPD'           , sid_LOADSPD);
 AddSid( 'SAVESPD'           , sid_SAVESPD);
end;

function Identify(const key:LongString):TStringIdentifier;
var sid:Integer;
begin
 if (Hasher=nil) then InitHasher;
 sid:=Hasher.KeyedLinks[key];
 if (sid>=Ord(Low(TStringIdentifier))) and (sid<=Ord(High(TStringIdentifier)))
 then Result:=TStringIdentifier(sid)
 else Result:=sid_Unknown;
end;

 {
 ****************************************
 Конструктор создает устройство по модели
 ****************************************
 }
function SoftwareDeviceConstructor(const Name,Family,Model,ConfigFile:LongString):TDaqDevice;
begin
 Result:=nil;
 if (Identify(Family)=sid_SOFTWARE) then begin
  case Identify(Model) of
   sid_DIALOG:          Result:=TDialogDevice.Create(Name);
   sid_CONSTGENERATOR:  Result:=TConstGeneratorDevice.Create(Name);
   sid_BITSETGENERATOR: Result:=TBitSetGeneratorDevice.Create(Name);
   sid_SCRIPT:          Result:=TScriptDevice.Create(Name);
   sid_PROGRAM:         Result:=TProgramDevice.Create(Name);
  end;
  if Daq.Ok and not Result.Ok
  then Daq.AddWarning('Invalid device: '+Name+' = device '+Family+' '+Model);
 end;
end;

 {
 ******************************
 TSoftwareDevice implementation
 ******************************
 }
procedure DevicePollAction(aPolling:TPolling; var Terminate:Boolean);
var Device:TSoftwareDevice;
begin
 Device:=TSoftwareDevice(aPolling.LinkObject);
 if (Device is TSoftwareDevice) then Device.Poll else Terminate:=true;
end;

function TSoftwareDevice.GetPolling:TPolling;
begin
 if Assigned(Self) then Result:=myPolling else Result:=nil;
end;

function TSoftwareDevice.GetCmdFifo:TFifo;
begin
 if Assigned(Self) then Result:=myCmdFifo else Result:=nil;
end;

function TSoftwareDevice.ClickRead:TEventWhat;
begin
 Result:=evNothing;
 if Assigned(Self) then begin
  GetSensorClicked(myLatchEvent);
  Result:=myLatchEvent.What;
 end;
end;

function TSoftwareDevice.ClickWrite(const Lines:LongString):TEventWhat;
var aEvent:TSensorEvent;
begin
 Result:=evNothing;
 if Assigned(Self) then begin
  aEvent:=SensorEvent(Lines);
  if (aEvent.What<>evNothing)
  then SetSensorClicked(aEvent);
  Result:=aEvent.What;
 end;
end;

procedure TSoftwareDevice.ClickFree;
begin
 if Assigned(Self) then SensorEvent(myLatchEvent);
end;

function TSoftwareDevice.GetClickWhat:TEventWhat;
begin
 if Assigned(Self) then Result:=myLatchEvent.What else Result:=evNothing;
end;

function TSoftwareDevice.GetClickWrote:Integer;
begin
 if Assigned(Self) then Result:=myLatchEvent.Wrote else Result:=0;
end;

function TSoftwareDevice.GetClickButton:Integer;
begin
 if Assigned(Self) then Result:=myLatchEvent.Button else Result:=0;
end;

function TSoftwareDevice.GetClickTag:Integer;
begin
 if Assigned(Self) then Result:=myLatchEvent.Tag else Result:=0;
end;

function TSoftwareDevice.GetClickSensor:LongString;
begin
 if Assigned(Self) then Result:=myLatchEvent.Sensor else Result:='';
end;

function TSoftwareDevice.ClickSensorParams(const aParam:LongString):LongString;
begin
 if Assigned(Self) then Result:=SensorEventParams(myLatchEvent,aParam) else Result:='';
end;

constructor TSoftwareDevice.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceFamily('SOFTWARE');
 myPolling:=NewPolling(DevicePollAction,
                       DefaultDaqPollDelay,  DefaultDaqPollPriority,
                       false, 'Daq.'+Name);
 myPolling.Master:=@myPolling;
 myPolling.LinkObject:=Self;
 myCmdFifo:=NewFifo(DefaultDeferredCommandFifoSize);
 myCmdFifo.Master:=@myCmdFifo;
 myCmdFifo.GrowFactor:=2;
 myCmdList:=NewText;
 myCmdList.Master:=@myCmdList;
 myCmdIdle:=NewText;
 myCmdIdle.Master:=@myCmdIdle;
 myIdleList:=TStringList.Create;
 myIdleList.Sorted:=true;
 myIdleList.CaseSensitive:=false;
 myIdleList.Duplicates:=dupIgnore;
 myLatchEvent:=SensorEvent;
end;

destructor TSoftwareDevice.Destroy;
begin
 Kill(myPolling);
 Kill(myCmdFifo);
 Kill(myCmdList);
 Kill(myCmdIdle);
 Kill(myIdleList);
 inherited Destroy;
end;

procedure TSoftwareDevice.AfterConstruction;
begin
 inherited AfterConstruction;
 AddIdleCommands('@ACTION,@CLEAR,@CLEARDEVICE,@START,@STOP,@CLEARCURVE');
 AddIdleCommands('@WINSHOW,@WINHIDE,@WINDRAW,@WINSELECT'); // NB!
 AddIdleCommands('@SAVECRW');
end;

procedure TSoftwareDevice.BeforeDestruction;
begin
 inherited BeforeDestruction;
end;

procedure TSoftwareDevice.Config(FileName:LongString);
var d:Integer; p:TThreadPriority;
begin
 d:=100; p:=tpNormal;
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 if ReadIniFilePolling(FileName,   '['+Name+']', 'DevicePolling', d, p)
 or ReadIniFilePolling(FileName,   '[DAQ]',      'DevicePolling', d, p)
 or ReadIniFilePolling(SysIniFile, '[DaqSys]',   'DevicePolling', d, p)
 then begin
  Polling.Delay:=d;
  Polling.Priority:=p;
 end;
end;

function TSoftwareDevice.GetProperty(TheText:TText):TText;
begin
 Result:=inherited GetProperty(TheText);
 TheText.Addln(Format('DevicePolling = %d, %s',[Polling.Delay,GetPriorityName(Polling.Priority)]));
end;

procedure TSoftwareDevice.Poll;
begin
 if InquiryTimer.IsStart then begin
  GetSensorClicked(myLatchEvent);
  if GotEvents then Handler;
  SensorEvent(myLatchEvent);
 end;
end;

function TSoftwareDevice.GotEvents:Boolean;
begin
 Result:=(inherited GotEvents) or (ClickWhat<>evNothing) or Polling.AwakeFlag;
end;

procedure TSoftwareDevice.Awake;
begin
 Polling.Awake;
end;

procedure TSoftwareDevice.Handler;
begin
end;

const
 arg_delims=[' ',',',#9];

 {
 Поиск окна с данным именем
 }
type
 TFindWinRec = packed record
  Caption : LongString;
  Found   : TForm;
 end;

procedure CheckWinName(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if Form is TForm then
 with TFindWinRec(Custom^) do
 if IsSameText(UnifyAlias(Form.Caption),UnifyAlias(Caption)) then begin
  Found:=Form;
  Terminate:=true;
 end;
end;

procedure CheckDaqWinName(Index:LongInt; const aObject:TObject; var Terminate:Boolean; Custom:Pointer);
begin
 if (aObject is TForm) then
 with TFindWinRec(Custom^) do
 if IsSameText(UnifyAlias(TForm(aObject).Caption),UnifyAlias(Caption)) then begin
  Found:=TForm(aObject);
  Terminate:=true;
 end;
end;

function FindWin(const aName:LongString):TForm;
var FindWinRec:TFindWinRec;
begin
 FindWinRec:=Default(TFindWinRec);
 try
  FindWinRec.Caption:=StringBuffer(aName);
  FindWinRec.Found:=nil;
  if not Assigned(FindWinRec.Found) then Daq.CurWinList.ForEach(CheckDaqWinName, @FindWinRec);
  if not Assigned(FindWinRec.Found) then Daq.SpeWinList.ForEach(CheckDaqWinName, @FindWinRec);
  if not Assigned(FindWinRec.Found) then Daq.CirWinList.ForEach(CheckDaqWinName, @FindWinRec);
  if not Assigned(FindWinRec.Found) then Daq.TabWinList.ForEach(CheckDaqWinName, @FindWinRec);
  if not Assigned(FindWinRec.Found) then ForEachForm(CheckWinName, @FindWinRec);
  if (FindWinRec.Found=nil) and DebugLogEnabled(dlc_ExecDpc) then DebugLog(dlc_ExecDpc,'FAILED FINDWIN: '+aName);
  Result:=FindWinRec.Found;
 finally
  FindWinRec.Caption:='';
 end;
end;

 {
 Функция сохраняет данные в crw-файле, используя средства DAQ.
 args = fname winname wintitle winlable curvelist
 fname      - имя файла (можно без пути и без расширения)
 winname    - имя окна под которым оно попадет в crw-архив
 wintitle   - заголовок в верхней части окна
 winlable   - метка в нижней части окна
 curvelist  - список сохраняемых кривых
 }
function savecrw(const args:LongString):boolean;
var i:Integer; Curve:TCurve; crvname,crvlist:LongString;
var fname,wincap,wintit,winleg:LongString;
 procedure CheckSection(var Item:LongString;const Format,Default:LongString);
 begin
  if IsSameText(Item,UnifySection(Item)) then
  if ReadIniFileAlpha(Daq.ConfigFile,UnifySection(Item),Format+'%a',Item)
  then Item:=TrimChars(Item,arg_delims,arg_delims)
  else Item:=Default;
 end;
begin
 Result:=false;
 if Daq.DaqDataWindow.Ok and (WordCount(args,arg_delims)>4) then begin
  Result:=true;
  fname:=UpcaseStr(ExtractWord(1,args,arg_delims));
  CheckSection(fname,'FileName','?');
  fname:=Daq.FileRef(fname,'.crw');
  wincap:=UpcaseStr(ExtractWord(2,args,arg_delims));
  CheckSection(wincap,'Name',RemoveBrackets(wincap));
  DaqAlignStr(wincap);
  wintit:=UpcaseStr(ExtractWord(3,args,arg_delims));
  CheckSection(wintit,'AxisY','?');
  DaqAlignStr(wintit);
  winleg:=UpcaseStr(ExtractWord(4,args,arg_delims));
  CheckSection(winleg,'AxisX','?');
  DaqAlignStr(winleg);
  crvlist:='';
  for i:=5 to WordCount(args,arg_delims) do begin
   crvname:=UpcaseStr(ExtractWord(i,args,arg_delims));
   if IsSameText(crvname,UnifySection(crvname)) then begin
    crvlist:=crvlist+crvname+' ';
   end else begin
    Curve:=Daq.Curves.Find(crvname);
    if Curve.Ok then crvlist:=crvlist+Curve.Name+' ' else Result:=false;
   end;
  end;
  Daq.DaqDataWindow.DefCurveNum:=-1;
  Daq.DaqDataWindow.AutoRange;
  Daq.SaveDataCustom(fname,wincap,wintit,winleg,crvlist);
 end;
end;

 // Use Index=-1 to markup OnIdle callback
procedure ExecCmd(Index:LongInt; const TextLine:LongString; var Terminate:boolean; CustomData:Pointer);
var Cmd,Arg:LongString; Dev:TSoftwareDevice;
begin
 Cmd:=UnifyAlias(ExtractWord(1,TextLine,arg_delims));
 Arg:=Trim(SkipWords(1,TextLine,arg_delims));
 Dev:=TSoftwareDevice(CustomData);
 if not Assigned(Dev) then Exit;
 if (Index>=0) and Dev.HandleDpcOnIdle and Dev.IsIdleCommand(Cmd) then begin
  Dev.myCmdIdle.AddLn(TextLine); Exit; // Handle it in OnIdle callback
 end;
 if (Cmd<>'') then Dev.ExecuteDeferredCommand(Cmd,Arg);
 if (Cmd<>'') then Dev.DeferredCommandProblem(TextLine);
end;

procedure TSoftwareDevice.ExecuteDeferredCommands;
begin
 LockedDec(DpcBalance);
 if (CmdFifo.Count>0) then begin
  myCmdList.Text:=CmdFifo.GetText;
  myCmdList.ForEach(ExecCmd,Self);
  myCmdList.Count:=0;
 end;
end;

function TSoftwareDevice.IsIdleCommand(const Cmd:LongString):Boolean;
begin
 if Assigned(Self)
 then Result:=(myIdleList.IndexOf(Cmd)>=0)
 else Result:=false;
end;

procedure TSoftwareDevice.AddIdleCommands(const aIdleCommands:LongString);
var i:Integer;
begin
 if Assigned(Self) then
 for i:=1 to WordCount(aIdleCommands,ScanSpaces) do
 myIdleList.Add(ExtractWord(i,aIdleCommands,ScanSpaces));
end;

procedure TSoftwareDevice.PollIdleDeferredCommands(Loops:Integer=1);
var Terminate:Boolean;
begin
 Terminate:=false;
 if Assigned(Self) then begin
  while (myCmdIdle.Count>0) and (Loops>0) do begin
   ExecCmd(-1,myCmdIdle[0],Terminate,Self);
   myCmdIdle.DelLn(0);
   Dec(Loops);
  end;
 end;
end;

procedure Callback_PollDpcOnIdle;
var i:Integer;
begin
 if TSoftwareDevice.HandleDpcOnIdle then
 try
  with SoftwareDeviceList do
  for i:=0 to Count-1 do SoftDev[i].PollIdleDeferredCommands(1);
 except
  on E:Exception do BugReport(E,nil,'Callback_PollDpcOnIdle');
 end;
end;

procedure DoExecuteDeferredCommands(Sender:TObject);
begin
 if (FullDaqDeviceList.IndexOf(Sender)>=0) then
 if (Sender is TSoftwareDevice) then (Sender as TSoftwareDevice).ExecuteDeferredCommands;
end;

function  TSoftwareDevice.PostDeferredCommand(const Cmd:LongString):Boolean;
begin
 Result:=Ok and CmdFifo.PutText(Cmd) and PostDeferredCallbackRequest(DoExecuteDeferredCommands,Self);
end;

function TSoftwareDevice.PostDeferredCallbackRequest(Callback:TCrwDaqDeferredCallbackRequestProc; Sender:TObject):Boolean;
begin
 if Assigned(Application) and Assigned(Application.MainForm)
 then Result:=PostCrwDaqDeferredCallbackRequest(Application.MainForm.Handle,Callback,Sender)
 else Result:=False;
 if Result then LockedInc(DpcBalance);
end;

procedure TSoftwareDevice.ExecuteDeferredCommand(var Cmd,Arg:LongString);
var i,d,HistLen:Integer; Win:TForm; Curve:TCurve; Device:TDaqDevice;
const Separator = '|';
 procedure DoWinDraw(Form:TForm; const Arg:LongString);
 var i,j,p:Integer; r:Double; Rx:TRect2D; FullDraw:Boolean; sn,sv:LongString;
 var si:TStringIdentifier; bi:TBorderIcons; FT,FL,FW,FH,IgnFixed:Integer;
 var crv:TCurve; InitRect,FormRect:TRect;
  function EvalRect(Win:TFormCurveWindow; const Rx:TRect2D):TRect2D;
  var i:Integer; Rw,Ra,Rr:TRect2D; ee:TExpressionEvaluator;
  var buf:TParsingBuffer;
  begin
   ee:=NewExpressionEvaluator;
   try
    Rr:=Win.Roi;
    Rw:=Win.World;
    Ra:=Win.GetAutoRange;
    Result:=Rx;
    for i:=1 to 4 do
    if ee.SetValue('x1',   Rw.A.X) and ee.SetValue('y1',   Rw.A.Y) and
       ee.SetValue('x2',   Rw.B.X) and ee.SetValue('y2',   Rw.B.Y) and
       ee.SetValue('xmin', Ra.A.X) and ee.SetValue('ymin', Ra.A.Y) and
       ee.SetValue('xmax', Ra.B.X) and ee.SetValue('ymax', Ra.B.Y) and
       ee.SetValue('roix1',Rr.A.X) and ee.SetValue('roiy1',Rr.A.Y) and
       ee.SetValue('roix2',Rr.B.X) and ee.SetValue('roiy2',Rr.B.Y) and
       (ee.EvaluateExpression(StrCopyBuff(buf,Trim(ExtractWord(i,sv,[';']))))=ee_Ok)
    then
    case i of
     1 : Result.A.X:=ee.Answer;
     2 : Result.A.Y:=ee.Answer;
     3 : Result.B.X:=ee.Answer;
     4 : Result.B.Y:=ee.Answer;
    end;
   finally
    Kill(ee);
   end;
  end;
  function EvalRect3d(Win:TFormSurfWindow; const Rx:TRect2D):TRect2D;
  var i:Integer; Ra,Rc:TRect2D; ee:TExpressionEvaluator;
  var buf:TParsingBuffer;
  begin
   ee:=NewExpressionEvaluator;
   try
    Rc:=Win.Clip;
    Ra:=Win.Limits;
    Result:=Rx;
    for i:=1 to 4 do
    if ee.SetValue('x1',   Rc.A.X) and ee.SetValue('y1',   Rc.A.Y) and
       ee.SetValue('x2',   Rc.B.X) and ee.SetValue('y2',   Rc.B.Y) and
       ee.SetValue('xmin', Ra.A.X) and ee.SetValue('ymin', Ra.A.Y) and
       ee.SetValue('xmax', Ra.B.X) and ee.SetValue('ymax', Ra.B.Y) and
       (ee.EvaluateExpression(StrCopyBuff(buf,Trim(ExtractWord(i,sv,[';']))))=ee_Ok)
    then
    case i of
     1 : Result.A.X:=ee.Answer;
     2 : Result.A.Y:=ee.Answer;
     3 : Result.B.X:=ee.Answer;
     4 : Result.B.Y:=ee.Answer;
    end;
   finally
    Kill(ee);
   end;
  end;
  function DoSaveCrw(Form:TForm; const sv:LongString):Bool;
  var sf:LongString;
  begin
   Result:=false;
   if IsNonEmptyStr(sv) then begin
    sf:=Daq.FileRef(sv,'.crw');
    if MkDir(ExtractFilePath(sf)) then begin
     if Form is TFormCurveWindow
     then Result:=SaveCurveWindowToCrw(sf,Form as TFormCurveWindow);
     if Form is TFormSurfWindow
     then Result:=SaveSurfWindowToCrw(sf,Form as TFormSurfWindow);
     if Result then begin
      Daq.ConsoleEcho(StdDateTimePrompt+
          RusEng(Format('Записано "%s".',[Form.Caption]),
                 Format('Written  "%s".',[Form.Caption])));
      Daq.ConsoleEcho(RusEng('Файл ','File ')+sf);
     end else begin
      Daq.ConsoleEcho(StdDateTimePrompt+
          RusEng(Format('Ошибка записи "%s".',[Form.Caption]),
                 Format('Error writing "%s".',[Form.Caption])));
      Daq.ConsoleEcho(RusEng('Файл ','File ')+sf);
     end;
    end else begin
     Daq.ConsoleEcho(StdDateTimePrompt+
         RusEng(Format('Не могу создать каталог "%s".',[ExtractFilePath(sf)]),
                Format('Could not create folder "%s".',[ExtractFilePath(sf)])));
    end;
   end;
  end;
  function DoSaveBmp(Form:TForm; const sv:LongString):Bool;
  var sm,sf:LongString; bn:Integer; bm:TBitmap;
  begin
   Result:=false;
   if IsNonEmptyStr(sv) then
   try
    bm:=nil;
    try
     sm:=ExtractWord(1,sv,ScanSpaces);
     sf:=ExtractWord(2,sv,ScanSpaces);
     case Identify(sm) of
      sid_1Bit:  bn:=1;
      sid_4Bit:  bn:=4;
      sid_8Bit:  bn:=8;
      sid_16Bit: bn:=16;
      sid_24Bit: bn:=24;
      sid_32Bit: bn:=32;
      else       bn:=0;
     end;
     if (Length(sf)*bn=0)
     then RAISE EConvertError.Create(Format('Invalid SaveBmp format "%s".',[sv]));
     sf:=Daq.FileRef(sf,'.bmp');
     if not MkDir(ExtractFilePath(sf))
     then RAISE EConvertError.Create(Format('Could not create folder "%s".',[ExtractFilePath(sf)]));
     if Form is TFormCurveWindow then bm:=(Form as TFormCurveWindow).GetImage(bn=1) else
     if Form is TFormSurfWindow  then bm:=(Form as TFormSurfWindow).GetImage(bn=1)  else bm:=Form.GetFormImage;
     if not Assigned(bm)
     then RAISE EConvertError.Create(Format('Could not create image "%s".',[Form.Caption]));
     case bn of
      1:  bm.PixelFormat:=pf8Bit;
      4:  bm.PixelFormat:=pf4Bit;
      8:  bm.PixelFormat:=pf8Bit;
      16: bm.PixelFormat:=pf16Bit;
      24: bm.PixelFormat:=pf24Bit;
      32: bm.PixelFormat:=pf32Bit;
     end;
     bm.SaveToFile(sf);
     Result:=Assigned(bm);
     Daq.ConsoleEcho(StdDateTimePrompt+
         RusEng(Format('Записано "%s".',[Form.Caption]),
                Format('Written  "%s".',[Form.Caption])));
     Daq.ConsoleEcho(RusEng('Файл ','File ')+sf);
    finally
     FreeAndNil(bm);
    end;
   except
    on E:Exception do BugReport(E,Self,'DoSaveBmp');
   end;
  end;
  procedure ScrollCirWin(CirWin:TFormCircuitWindow; const sv:LongString);
  var i:Integer; Ppos,Pmin,Pmax:TPoint2D; ee:TExpressionEvaluator;
  var buf:TParsingBuffer;
  begin
   if (CirWin is TFormCircuitWindow) then
   try
    Pmin:=Point2D(CirWin.ScrollBarX.Min,      CirWin.ScrollBarY.Min);
    Pmax:=Point2D(CirWin.ScrollBarX.Max,      CirWin.ScrollBarY.Max);
    Ppos:=Point2D(CirWin.ScrollBarX.Position, CirWin.ScrollBarY.Position);
    ee:=NewExpressionEvaluator;
    try
     for i:=1 to 2 do
     if ee.SetValue('xpos', Ppos.X) and ee.SetValue('ypos', Ppos.Y) and
        ee.SetValue('xmin', Pmin.X) and ee.SetValue('ymin', Pmin.Y) and
        ee.SetValue('xmax', Pmax.X) and ee.SetValue('ymax', Pmax.Y) and
        (ee.EvaluateExpression(StrCopyBuff(buf,Trim(ExtractWord(i,sv,[';']))))=ee_Ok)
     then
     case i of
      1 : Ppos.X:=ee.Answer;
      2 : Ppos.Y:=ee.Answer;
     end;
    finally
     Kill(ee);
    end;
    if IsNanOrInf(Ppos.X) then Ppos.X:=CirWin.ScrollBarX.Position;
    if IsNanOrInf(Ppos.Y) then Ppos.Y:=CirWin.ScrollBarY.Position;
    Ppos.X:=Max(Pmin.X,Min(Pmax.X,Ppos.X));
    Ppos.Y:=Max(Pmin.Y,Min(Pmax.Y,Ppos.Y));
    CirWin.ScrollBarX.Position:=Round(Ppos.X);
    CirWin.ScrollBarY.Position:=Round(Ppos.Y);
   except
    on E:Exception do BugReport(E,Self,'ScrollCirWin');
   end;
  end;
  procedure EvalCirWin(CirWin:TFormCircuitWindow; const sv:LongString);
  var buf:TParsingBuffer;
  begin
   if (CirWin is TFormCircuitWindow) then
   try
    CirWin.Evaluator.EvaluateLine(StrCopyBuff(buf,sv));
   except
    on E:Exception do BugReport(E,Self,'EvalCirWin');
   end;
  end;
  procedure DrawSensorCirWin(CirWin:TFormCircuitWindow; const sv:LongString; Pending:Boolean);
  var i:Integer; Sensor:TCircuitSensor;
  begin
   if (CirWin is TFormCircuitWindow) then
   try
    for i:=1 to WordCount(sv,ScanSpaces) do begin
     Sensor:=CirWin.SensorByName(ExtractWord(i,sv,ScanSpaces));
     if Assigned(Sensor) then begin
      if Pending
      then Sensor.PendingDraw:=true
      else CirWin.DrawSensor(Sensor);
     end;
    end;
   except
    on E:Exception do BugReport(E,Self,'DrawSensorCirWin');
   end;
  end;
  procedure LoadSPD(SpeWin:TFormSpectrWindow; const sv:LongString);
  var spd:LongString;
  begin
   if IsNonEmptyStr(sv) then
   if (TObject(SpeWin) is TFormSpectrWindow) then
   try
    spd:=Daq.FileRef(Trim(sv),'.spd');
    if not FileExists(spd) then Raise ELoadSPD.Create(Format('File not exist: %s',[spd]));
    SpeWin.LoadSPD(spd);
   except
    on E:Exception do BugReport(E,Self,'LoadSPD');
   end;
  end;
  procedure SaveSPD(SpeWin:TFormSpectrWindow; const sv:LongString);
  var spd:LongString;
  begin
   if IsNonEmptyStr(sv) then
   if TObject(SpeWin) is TFormSpectrWindow then
   try
    spd:=Daq.FileRef(Trim(sv),'.spd');
    SpeWin.SaveSPD(spd);
   except
    on E:Exception do BugReport(E,Self,'SaveSPD');
   end;
  end;
  procedure SetFormTop(Form:TForm; y:Integer);
  begin
   if Assigned(Form) then begin
    Form.Top:=y;
    if (Form is TFormCrwDaqSysChild) then
    with (Form as TFormCrwDaqSysChild) do begin
     if IsFixedTop or HasFlags(IgnFixed,cfbm_T) then FixedTop:=y;
    end;
   end;
  end;
  procedure SetFormLeft(Form:TForm; x:Integer);
  begin
   if Assigned(Form) then begin
    Form.Left:=x;
    if (Form is TFormCrwDaqSysChild) then
    with (Form as TFormCrwDaqSysChild) do begin
     if IsFixedLeft or HasFlags(IgnFixed,cfbm_L) then FixedLeft:=x;
    end;
   end;
  end;
  procedure SetFormWidth(Form:TForm; w:Integer);
  begin
   if Assigned(Form) then begin
    Form.Width:=w;
    if (Form is TFormCrwDaqSysChild) then
    with (Form as TFormCrwDaqSysChild) do begin
     if IsFixedWidth or HasFlags(IgnFixed,cfbm_W) then FixedWidth:=w;
    end;
   end;
  end;
  procedure SetFormHeight(Form:TForm; h:Integer);
  begin
   if Assigned(Form) then begin
    Form.Height:=h;
    if (Form is TFormCrwDaqSysChild) then
    with (Form as TFormCrwDaqSysChild) do begin
     if IsFixedHeight or HasFlags(IgnFixed,cfbm_H) then FixedHeight:=h;
    end;
   end;
  end;
  function HasFixedMode(Form:TForm; What:Integer):Boolean;
  begin
   Result:=false;
    if (Form is TFormCrwDaqSysChild) then
    with (Form as TFormCrwDaqSysChild) do begin
     Result:=HasFlags(What,FixedWhat);
    end;
  end;
 begin
  try
   if (Form is TFormCrwDaqSysChild) then begin
    if not IsWindowVisible(Form.Handle) then begin
     Form.Show;
     Form.WindowState:=wsNormal;
    end;
    FullDraw:=not SameText('FAST',Trim(ExtractWord(2,Arg,[Separator])));  // @windraw window|fast|eval=...
    if FullDraw then (Form as TFormCrwDaqSysChild).LockDraw;
    try
     IgnFixed:=cfbm_LTWH;
     // Save initial Pos/Size for future.
     FormRect:=Form.BoundsRect; InitRect:=FormRect;
     FT:=MaxInt; FL:=MaxInt; FW:=MaxInt; FH:=MaxInt;
     for i:=2 to WordCount(Arg,[Separator]) do begin
      p:=ExtractNameValuePair(ExtractWord(i,Arg,[Separator]),sn,sv);
      if (p>0) then begin
       sn:=UpcaseStr(sn);
       si:=Identify(sn);
       // Common TForm
       ///////////////
       case si of
        sid_TOP: if Str2Int(sv,p) and InRange(p,0,Screen.Height-50) then FT:=p;
        sid_LEFT: if Str2Int(sv,p) and InRange(p,0,Screen.Width-100) then FL:=p;
        sid_WIDTH: if Str2Int(sv,p) and InRange(p,200,MaxInt) then FW:=Max(200,Min(p,Screen.DesktopWidth-90));
        sid_HEIGHT: if Str2Int(sv,p) and InRange(p,150,MaxInt) then FH:=Max(150,Min(p,Screen.DesktopHeight-70));
        sid_OPTIONS: begin
         bi:=Form.BorderIcons;
         for j:=1 to WordCount(sv,ScanSpaces) do
         case WordIndex(UpcaseStr(ExtractWord(j,sv,ScanSpaces)),
                       '-CLOSE;+CLOSE;-MIN;+MIN;-MAX;+MAX;-WIDTH;+WIDTH;-HEIGHT;+HEIGHT;'+
                       '-HSCROLL;+HSCROLL;-VSCROLL;+VSCROLL;-STATUSBAR;+STATUSBAR;-TOOLBAR;+TOOLBAR;'+
                       '-LEFT;+LEFT;-TOP;+TOP;',
                       ScanSpaces)
         of
          1:  Exclude(bi,biSystemMenu);
          2:  Include(bi,biSystemMenu);
          3:  Exclude(bi,biMinimize);
          4:  Include(bi,biMinimize);
          5:  Exclude(bi,biMaximize);
          6:  Include(bi,biMaximize);
          7:  (Form as TFormCrwDaqSysChild).IsFixedWidth:=True;
          8:  (Form as TFormCrwDaqSysChild).IsFixedWidth:=False;
          9:  (Form as TFormCrwDaqSysChild).IsFixedHeight:=True;
          10: (Form as TFormCrwDaqSysChild).IsFixedHeight:=False;
          11: begin
               if (Form is TFormCircuitWindow)
               then (Form as TFormCircuitWindow).ScrollBarShowX:=False;
               if (Form is TFormSurfWindow)
               then (Form as TFormSurfWindow).ScrollBarPhi.Visible:=False;
              end;
          12: begin
               if (Form is TFormCircuitWindow)
               then (Form as TFormCircuitWindow).ScrollBarShowX:=True;
               if (Form is TFormSurfWindow)
               then (Form as TFormSurfWindow).ScrollBarPhi.Visible:=True;
              end;
          13: begin
               if (Form is TFormCircuitWindow)
               then (Form as TFormCircuitWindow).ScrollBarShowY:=False;
               if (Form is TFormSurfWindow)
               then (Form as TFormSurfWindow).ScrollBarPsi.Visible:=False;
               if (Form is TFormCurveWindow)
               then (Form as TFormCurveWindow).ScrollBar.Visible:=False;
              end;
          14: begin
               if (Form is TFormCircuitWindow)
               then (Form as TFormCircuitWindow).ScrollBarShowY:=True;
               if (Form is TFormSurfWindow)
               then (Form as TFormSurfWindow).ScrollBarPsi.Visible:=True;
               if (Form is TFormCurveWindow)
               then (Form as TFormCurveWindow).ScrollBar.Visible:=True;
              end;
          15: (Form as TFormCrwDaqSysChild).StatusBar.Visible:=False;
          16: (Form as TFormCrwDaqSysChild).StatusBar.Visible:=True;
          17: (Form as TFormCrwDaqSysChild).ToolBar.Visible:=False;
          18: (Form as TFormCrwDaqSysChild).ToolBar.Visible:=True;
          19: (Form as TFormCrwDaqSysChild).IsFixedLeft:=True;
          20: (Form as TFormCrwDaqSysChild).IsFixedLeft:=False;
          21: (Form as TFormCrwDaqSysChild).IsFixedTop:=True;
          22: (Form as TFormCrwDaqSysChild).IsFixedTop:=False;
         end;
         {$IFDEF WINDOWS}
         if SysUtils.Win32Platform<VER_PLATFORM_WIN32_NT then Exclude(bi,biMaximize);
         {$ENDIF ~WINDOWS}
         if (bi<>Form.BorderIcons) then Form.BorderIcons:=bi;
        end;
        sid_SAVECRW: DoSaveCrw(Form,sv);
        sid_SAVEBMP: begin
         try
          if FullDraw then (Form as TFormCrwDaqSysChild).UnlockDraw;
          DoSaveBmp(Form,sv);
         finally
          if FullDraw then (Form as TFormCrwDaqSysChild).LockDraw;
         end;
        end;
       end;
       // TFormCurveWindow
       ///////////////////
       if (Form is TFormCurveWindow) then
       with (Form as TFormCurveWindow) do
       case si of
        sid_LEGENDX: Legend:=ReplaceAlignStr(sv,true);
        sid_LEGENDY: Title:=ReplaceAlignStr(sv,true);
        sid_RANGE: begin
         Rx:=EvalRect(Form as TFormCurveWindow,Rect2D(_NaN,_NaN,_NaN,_NaN));
         if not IsNanOrInf(Rx) and not RectIsEmpty(Rx) and not RectIsEqual(Rx,World) then World:=Rx;
        end;
        sid_ROI: begin
         Rx:=EvalRect(Form as TFormCurveWindow,Roi);
         Roi:=Rx;
        end;
        sid_SELECTCURVE: DefCurve:=Curves.Find(sv);
       end;
       // TFormCircuitWindow
       /////////////////////
       if (Form is TFormCircuitWindow) then
       with (Form as TFormCircuitWindow) do
       case si of
        sid_RELOAD: Reload(Trim(sv));
        sid_SCROLL: ScrollCirWin(Form as TFormCircuitWindow,Trim(sv));
        sid_TOOLBARHEIGHT: if Str2Int(sv,p) and (p>=0) and (p<=Form.Height div 4) then (Form as TFormCircuitWindow).ToolBarHeight:=p;
        sid_EVAL: EvalCirWin(Form as TFormCircuitWindow,Trim(sv));
        sid_DRAWSENSOR: DrawSensorCirWin(Form as TFormCircuitWindow,Trim(sv),false);
        sid_UPDATESENSOR: DrawSensorCirWin(Form as TFormCircuitWindow,Trim(sv),true);
       end;
       // TFormSurfWindow
       //////////////////
       if (Form is TFormSurfWindow) then
       with (Form as TFormSurfWindow) do
       case si of
        sid_LEGEND: Legend:=ReplaceAlignStr(sv,true);
        sid_TITLE: Title:=ReplaceAlignStr(sv,true);
        sid_PHI: if str2real(sv,r) then Phi:=DegToRad(r);
        sid_PSI: if str2real(sv,r) then Psi:=DegToRad(r);
        sid_SCALEX: if str2real(sv,r) then Scale:=Point3d(r,Scale.y,Scale.z);
        sid_SCALEY: if str2real(sv,r) then Scale:=Point3d(Scale.x,r,Scale.z);
        sid_SCALEZ: if str2real(sv,r) then Scale:=Point3d(Scale.x,Scale.y,r);
        sid_MODE: if str2int(sv,p) then SliceMode:=p;
        sid_ROI: begin
         Rx:=EvalRect3d(Form as TFormSurfWindow,Clip);
         Clip:=Rx;
        end;
        sid_SLICEX: if str2int(ExtractWord(1,sv,ScanSpaces),p) then begin
         crv:=Daq.Curves.Find(ExtractWord(2,sv,ScanSpaces));
         if (crv.Count>1) and (p>=0) and (p<NumPoints.x) then
         for j:=0 to NumPoints.y-1 do Z[p,j]:=crv.Interpolate(Y[j]);
        end;
        sid_SLICEY: if str2int(ExtractWord(1,sv,ScanSpaces),p) then begin
         crv:=Daq.Curves.Find(ExtractWord(2,sv,ScanSpaces));
         if (crv.Count>1) and (p>=0) and (p<NumPoints.y) then
         for j:=0 to NumPoints.x-1 do Z[j,p]:=crv.Interpolate(X[j]);
        end;
       end;
       // TFormSpectrWindow
       ////////////////////
       if (Form is TFormSpectrWindow) then
       case si of
        sid_LOADSPD: LoadSpd(Form as TFormSpectrWindow,sv);
        sid_SAVESPD: SaveSpd(Form as TFormSpectrWindow,sv);
       end;
      end;
     end;
     // Form Pos/Size changed by command?
     if (FT<>MaxInt) then SetFormTop(Form,FT);
     if (FL<>MaxInt) then SetFormLeft(Form,FL);
     if (FW<>MaxInt) then SetFormWidth(Form,FW);
     if (FH<>MaxInt) then SetFormHeight(Form,FH);
     if SameValue(0,1) then begin // Skip next code
      if (FT=MaxInt) and HasFixedMode(Form,cfbm_T) then SetFormTop(Form,InitRect.Top);
      if (FL=MaxInt) and HasFixedMode(Form,cfbm_L) then SetFormLeft(Form,InitRect.Left);
      if (FW=MaxInt) and HasFixedMode(Form,cfbm_W) then SetFormWidth(Form,InitRect.Width);
      if (FH=MaxInt) and HasFixedMode(Form,cfbm_H) then SetFormHeight(Form,InitRect.Height);
     end;
    finally
     if FullDraw then (Form as TFormCrwDaqSysChild).UnlockDraw;
    end;
   end else Form.Repaint;
  except
   on E:Exception do BugReport(E,Self,'DoWinDraw');
  end;
 end;
 procedure CheckOpenConsole(const Arg:LongString);
 var Dev:TDaqDevice;
 begin
  try
   if WordIndex(UpcaseStr(ExtractWord(1,ExtractWord(1,Arg,[Separator]),ScanSpaces)),'CONSOLE;КОНСОЛЬ',[';'])>0
   then begin
    Dev:=FullDaqDeviceList.Find(ExtractWord(2,ExtractWord(1,Arg,[Separator]),ScanSpaces));
    if (Dev is TProgramDevice) then TProgramDevice(Dev).OpenConsole;
   end;
  except
   on E:Exception do BugReport(E,Self,'CheckOpenConsole');
  end;
 end;
begin
 if DebugLogEnabled(dlc_ExecDpc) then DebugLog(dlc_ExecDpc,Cmd+' '+Arg);
 case Identify(Cmd) of
  sid_cmdACTION: begin
   for i:=1 to WordCount(Arg,arg_delims) do begin
    Device:=FullDaqDeviceList.Find(ExtractWord(i,Arg,arg_delims));
    if Device.Ok
    then Device.Action
    else DeferredCommandProblem(Cmd+' '+ExtractWord(i,Arg,arg_delims));
   end;
   Cmd:='';
  end;
  sid_cmdCLEAR: begin
   for i:=1 to WordCount(Arg,arg_delims) do begin
    Device:=FullDaqDeviceList.Find(ExtractWord(i,Arg,arg_delims));
    if Device.Ok
    then Device.Clear
    else DeferredCommandProblem(Cmd+' '+ExtractWord(i,Arg,arg_delims));
   end;
   Cmd:='';
  end;
  sid_cmdCLEARDEVICE: begin
   for i:=1 to WordCount(Arg,arg_delims) do begin
    Device:=FullDaqDeviceList.Find(ExtractWord(i,Arg,arg_delims));
    if Device.Ok
    then Device.ClearDevice
    else DeferredCommandProblem(Cmd+' '+ExtractWord(i,Arg,arg_delims));
   end;
   Cmd:='';
  end;
  sid_cmdSTART: begin
   for i:=1 to WordCount(Arg,arg_delims) do begin
    Device:=FullDaqDeviceList.Find(ExtractWord(i,Arg,arg_delims));
    if Device.Ok and not Device.InquiryTimer.IsStart
    then Device.Start
    else DeferredCommandProblem(Cmd+' '+ExtractWord(i,Arg,arg_delims));
   end;
   Cmd:='';
  end;
  sid_cmdSTOP: begin
   for i:=1 to WordCount(Arg,arg_delims) do begin
    Device:=FullDaqDeviceList.Find(ExtractWord(i,Arg,arg_delims));
    if Device.Ok and Device.InquiryTimer.IsStart
    then Device.Stop
    else DeferredCommandProblem(Cmd+' '+ExtractWord(i,Arg,arg_delims));
   end;
   Cmd:='';
  end;
  sid_cmdCLEARCURVE: begin
   HistLen:=0;
   for i:=1 to WordCount(Arg,arg_delims) do
   if Str2Int(ExtractWord(i,Arg,arg_delims),d) then HistLen:=d else begin
    Curve:=Daq.Curves.Find(ExtractWord(i,Arg,arg_delims));
    if Curve.Ok
    then DaqClearCurve(Curve,HistLen)
    else DeferredCommandProblem(Cmd+' '+ExtractWord(i,Arg,arg_delims));
   end;
   Cmd:='';
  end;
  sid_cmdWINDRAW: begin
   Win:=FindWin(ExtractWord(1,Arg,[Separator]));
   if (Win is TForm)
   then DoWinDraw(Win,Arg)
   else DeferredCommandProblem(Cmd+' '+Arg);
   Cmd:='';
  end;
  sid_cmdWINSHOW: begin
   CheckOpenConsole(Arg);
   Win:=FindWin(ExtractWord(1,Arg,[Separator]));
   if (Win is TForm) then begin
    Win.Show;
    Win.WindowState:=wsNormal;
   end else DeferredCommandProblem(Cmd+' '+Arg);
   Cmd:='';
  end;
  sid_cmdWINHIDE: begin
   Win:=FindWin(ExtractWord(1,Arg,[Separator]));
   if (Win is TForm) then begin
    if (Win=ApplicationMainForm) or SdiMan.IsChild(Win)
    then Win.WindowState:=wsMinimized
    else Win.Hide;
   end else DeferredCommandProblem(Cmd+' '+Arg);
   Cmd:='';
  end;
  sid_cmdWINSELECT: begin
   Win:=FindWin(ExtractWord(1,Arg,[Separator]));
   if SdiMan.IsChild(Win) then begin
    SdiMan.ActivateChild(Win,ascm_GtkRefresh);
   end else
   if (Win is TForm) then begin
    Win.Show;
    Win.WindowState:=wsNormal;
    Win.BringToFront;
   end else DeferredCommandProblem(Cmd+' '+Arg);
   Cmd:='';
  end;
  sid_cmdSAVECRW: begin
   if not savecrw(arg) then DeferredCommandProblem(Cmd+' '+Arg);
   Cmd:='';
  end;
 end;
end;

procedure TSoftwareDevice.DeferredCommandProblem(const Cmd:LongString);
begin
 if Ok then
 if IsNonEmptyStr(Cmd) then
 Daq.ConsoleEcho(Name+': '+RusEng('Не могу выполнить "','Cannot execute "')+Cmd+'".');
end;

 {
 **********************************
 TSoftwareDeviceList implementation
 **********************************
 }
function TSoftwareDeviceList.GetSoftDev(Index:Integer):TSoftwareDevice;
begin
 Result:=TSoftwareDevice(Items[Index]);
end;

procedure TSoftwareDeviceList.SetSoftDev(Index:Integer; aSoftDev:TSoftwareDevice);
begin
 Items[Index]:=aSoftDev;
end;

procedure TSoftwareDeviceList.EnablePolling(aEnable:Boolean);
var i:Integer;
 procedure Handle(Item:TSoftwareDevice);
 begin
  if Item.Ok then Item.Polling.Enable(aEnable,DefaultDaqTimeOut);
 end;
begin
 for i:=0 to Count-1 do Handle(Self[i]);
end;

function TSoftwareDeviceList.Start:Boolean;
begin
 EnablePolling(false);
 Result:=inherited Start;
 EnablePolling(true);
end;

procedure TSoftwareDeviceList.Stop;
begin
 EnablePolling(false);
 inherited Stop;
end;

procedure TSoftwareDeviceList.Poll;
begin
 // do nothing, because each device has own thread
end;

function SoftwareDeviceList:TSoftwareDeviceList;
const
 mySoftwareDeviceList : TSoftwareDeviceList = nil;
begin
 if not Assigned(mySoftwareDeviceList) then begin
  mySoftwareDeviceList:=TSoftwareDeviceList.Create(true);
  mySoftwareDeviceList.Master:=@mySoftwareDeviceList;
 end;
 Result:=mySoftwareDeviceList;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_softdev;
begin
 InitHasher;
 InitDpcCounters;
 SoftwareDeviceList.Ok;
 RegisterCrwDaqDeferredCallback(DoExecuteDeferredCommands);
 dlc_ExecDpc:=RegisterDebugLogChannel('_ExecuteDPC');
 OnIdleActions.Add(Callback_PollDpcOnIdle);
 TSoftwareDevice.HandleDpcOnIdle:=true;
end;

procedure Free_crw_softdev;
begin
 ResourceLeakageLog(Format('%-60s = %d',['SoftDev.DPC.Balance',LockedGet(DpcBalance)]));
 SoftwareDeviceList.Free;
 FreeDpcCounters;
 FreeHasher;
end;

initialization

 Init_crw_softdev;

finalization

 Free_crw_softdev;

end.

//////////////
// END OF FILE
//////////////

