////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Simple DAQ PASCAL compiler based on Facilis 0.20                           //
// DaqPascal compiler based on the Pascal S compiler of Niklaus Wirth,        //
// as modified by R.E. Berry, adapted for the IBMPC by John R. Naleszkiewicz  //
// extensions by Anthony M. Marcy                                             //
// finally modified by Alexey Kuryakin for DAQ applications                   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20020308 - Modified for Delphi (A.K.)                                      //
// 20231209 - Modified for FPC (A.K.)                                         //
// 20240618 - nextch: handle CRLF; FatalError(81)                             //
////////////////////////////////////////////////////////////////////////////////

unit _crw_daqpascalcompiler; // Unit DAQ Pascal Compiler

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}
{$WARN 6058 off : Call to subroutine "$1" marked as inline is not inlined}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 {$IFDEF UNIX} baseunix, unix, {$ENDIF}
 {$IFDEF WINDOWS} ActiveX, WinSock, {$ENDIF}
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf, lclproc,
 Sockets,
 Form_CrwDaqSysChild, Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow,  Form_TabWindow, Form_SpectrWindow,
 Form_Calculator, Form_ListBoxSelection, Form_UartTerminal, Form_CalibDialog,
 Form_DaqEditTagDialog, Form_TextEditDialog, Form_StringGridEditDialog,
 Form_CheckListBoxSelection,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo, _crw_ef, _crw_calib,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_curves, _crw_riff,
 _crw_pio, _crw_couple, _crw_polling, _crw_ee, _crw_daqtags, _crw_utf8,
 _crw_daqsys, _crw_daqevnt, _crw_daqdev, _crw_uart, _crw_adamdev,
 _crw_crwapiserver, _crw_pipe, _crw_delauna, _crw_task, _crw_proc,
 _crw_base64, _crw_base32, _crw_hash, _crw_crypt, _crw_syslog,
 _crw_syscal, _crw_pipeio, _crw_tcp, _crw_hl, _crw_colors,
 _crw_spcfld, _crw_sesman, _crw_regexp, _crw_bsencode,
 _crw_fsm, _crw_dbapi, _crw_dbcon, _crw_sharm,
 _crw_environ, _crw_assoc, _crw_prmstrlst,
 _crw_dynar, _crw_snd, _crw_guard, _crw_uri,
 _crw_appforms, _crw_apptools, _crw_apputils;

 // Integer types table
 // Type	Range from	        Range to        Format
 // Shortint	–128            ..      127             signed 8-bit
 // Smallint	–32768          ..      32767           signed 16-bit
 // Longint	–2147483648     ..      2147483647	signed 32-bit
 // Int64	–2^63           ..      2^63–1	        signed 64-bit
 // Byte	0               ..      255	        unsigned 8-bit
 // Word	0               ..      65535	        unsigned 16-bit
 // Longword	0               ..      4294967295	unsigned 32-bit

const
 nokeywrds   = 35;                     { no. of key words }
 alng        = 40;                     { no. of significant chars in identifiers }
 llng        = 241;                    { input line legnth }
 emax        = 308;                    { max exponent of real numbers }
 emin        = -308;                   { min exponent }
 kmax        = 11;                     { max no. of significant digits }
 csmax       = 256;                    { max no. of cases }
 levmax      = 7;                      { maximum level }
 xarmax      = 2147483647 {32767};     { maximum array bound }
 nmax        = 2147483647 {32767};     { maximum integer }
 InpFifoSize = 1024*16;                { standard input fifo size }
 OutFifoSize = 1024*32;                { standard output fifo size }
 SafetyGap   = 1;                      { size gap for safety }
 itabmax_def = 1024+256;               { default size of identifier table }
 btabmax_def = 60;                     { default size of block-table }
 atabmax_def = 60;                     { default size of array-table }
 rtabmax_def = 100;                    { default size of real constant table }
 ctabmax_def = 1024*8;                 { default size of P-code table }
 dtabmax_def = 1024*8;                 { default size of P-data table }
 stabmax_def = 1024*1;                 { default size of string table }
 dtabmin_def = 256;                    { default min startup stack space }
 stabmin_def = 256;                    { default min startup string table space }
 slenmax_min = 1024*32;                { minimal value of slenmax }
 slenmax_def = 1024*1024*2;            { default string length maximum }
 slenmax_max = 1024*1024*256;          { maximal value of slenmax }
 strngalign  = 16;                     { string allocation alignment }
 DefMortWait = 30000;                  { default postmortal wait time }
 incmax      = 9;                      { maximum include level }
 DefIncExt   = '.inc';                 { default extension for include files }
 tabFactor   = 2;                      { grow factor to reallocate compiler tables }

type
 textptr    = ^text;
 real       = double;
 symbol     = (intcon,realcon,charcon,stringcon,notsy,plus,minus,times,idiv,
               rdiv,imod,andsy,orsy,insy,eql,neq,gtr,geq,lss,leq,lparent,
               rparent,lbrack,rbrack,comma,semicolon,period,twodots,colon,
               becomes,constsy,typesy,varsy,funcsy,nilsy,procsy,filesy,arraysy,
               recordsy,packedsy,setsy,programsy,labelsy,ident,withsy,beginsy,
               ifsy,casesy,repeatsy,whilesy,forsy,gotosy,endsy,elsesy,untilsy,
               ofsy,dosy,tosy,downtosy,thensy);
 index      = -xarmax..+xarmax;
 alfa       = packed array [1..alng] of char;
 objekt     = (konstant,vvariable,type1,prozedure,funktion);
 types      = (notyp,ints,reals,bools,chars,strngs,arrays,records);
 symset     = set of symbol;
 typset     = set of types;
 itab_t     = packed array[0 .. 0 {itabmax}] of packed record { identifier table }
               name   : alfa;
               link   : index;
               obj    : objekt;
               typ    : types;
               ref    : index;
               normal : boolean;
               lev    : 0 .. levmax;
               adr    : integer
              end;
 btab_t     = packed array[1 .. 2 {btabmax}] of packed record { block-table }
               last    : index;
               lastpar : index;
               psize   : index;
               vsize   : index;
              end ;
 atab_t     = packed array[1 .. 1 {atabmax}] of packed record { array-table }
               inxtyp : types;
               eltyp  : types;
               elref  : index;
               low    : index;
               high   : index;
               elsize : index;
               size   : index;
              end ;
 rtab_t     = packed array [1 .. 1 {rtabmax}] of real;         { real const table }
 ctab_t     = packed array [0 .. 0 {ctabmax}] of packed record { P-code table }
               f : byte;
               x : 0..levmax;
               y : -nmax..+nmax;
              end;
 dtab_t     = packed record                                  { P-data table }
  s         : packed array[0 .. 5 {stacksize}] of packed record { blockmark }
               case cn:types of                  {    s[b+0] = fct result   }
               ints:  (  i: integer);            {    s[b+1] = return adr   }
               reals: (  r: real);               {    s[b+2] = static link  }
               bools: (  b: boolean);            {    s[b+3] = dynamic link }
               chars: (  c: char);               {    s[b+4] = table index  }
               strngs:(s,p: integer);            {    s[b+5] = string ptr   }
              end;
 end;
 stab_it    = packed array [1 .. 1 {stabmax}] of integer;
 stab_pt    = packed array [1 .. 1 {stabmax}] of pointer;
 stab_t     = packed record                                  { string table }
  space     : integer;
  count     : integer;
  stack     : ^stab_it;
  table     : ^stab_pt;
 end;
 strnglnt   = integer; {word;}  { string length type    }
 strnghdr   = packed record     { string header record  }
  leng      : strnglnt;         { string length actual  }
  dyna      : strnglnt;         { string dynamic space  }
 end;                           { or 0 for string const }
 strngptr   = ^strngbuf;        { string buffer pointer }
 strngbuf   = packed record     { string buffer record  }
  head      : strnghdr;         { string header record  }
  data      : packed record     { string data content   }
   case byte of                 { union of bytes/chars  }
   1 : ( buff:  packed array [0..slenmax_max-1] of char; );
   2 : ( bytes: packed array [0..slenmax_max-1] of byte; );
  end;
 end;
 inplineptr = ^inplinebuf;                   {pointer to input line buffer}
 inplinebuf = packed array[1..llng] of char; {buffer for input lines}
 TDaqPascal = class(TMasterObject)
 public
  CompileOk   : boolean;     { true only if compilation successfull complete }
  DebugMode   : boolean;     { set to true to write debug information }
  TheDevice   : TDaqDevice;  { associated DAQ device }
  ExceptResult: integer;     { exceptions manager }
  ErrorMessage: LongString;  { compilation error message if not CompileOk }
  ErrorDetails: LongString;  { compilation error details if not CompileOk }
  ErrorCode   : integer;     { compilation error code }
  MortalTime  : Int64;       { mortal time when runtime exception found }
  MortalDelay : Integer;     { postmortal delay to avoid system overload }
  MortalCount : Integer;     { mortal counter of sequenced runtime errors }
  ErrorLine   : integer;     { input file error line if not CompileOk }
  ErrorColumn : integer;     { input file error column if not CompileOk }
  ErrorColUtf8: integer;     { input file error column in UTF8 chars }
  SrcFile     : LongString;  { source file name to be compiled }
  LstFile     : LongString;  { compiler listing file name  }
  PasTop      : Integer;     { stack top index for pascal source files }
  PasStack    : array[0..incmax] of packed record { stack of include sources }
   Src        : LongString;
   Inp        : text;
   iln        : Integer;
   ch         : Char;
   cc         : Integer;
   ll         : Integer;
   line       : inplinebuf;
  end;
 private
  myBreakFlag : Integer;     { flag to break execution and generate error }
  procedure Set_BreakFlag(aFlag:Integer);
 public
  property  BreakFlag : Integer read myBreakFlag write Set_BreakFlag;
 private
  function  Get_PasSrc:LongString;
  procedure Set_PasSrc(const aPasSrc:LongString);
  property  PasSrc : LongString read Get_PasSrc write Set_PasSrc; { current pascal source }
 private
  function  Get_PasInp:textptr;
  property  PasInp : textptr read Get_PasInp; { points to pascal source text file }
 public
  LstOut      : text;         { list file output }
  StdInp      : text;         { runtime standard input }
  StdOut      : text;         { runtime standard output }
  IORes       : integer;      { IOResult for StdInp,StdOut }
  InpFifo     : TFifo;        { fifo as virtual standard input device }
  OutFifo     : TFifo;        { fifo as standard output device }
  ComIO       : packed record { COM port In/Out record }
   Pipe       : TPipe;        { uses COM named pipe }
   PortN      : integer;      { uses COM-port number }
   PipeLine   : integer;      { uses COM pipeline reference }
  end;
  Adam_Slot   : TAdamSlot;    { uses Adam_Slot_Device }
  lbuf1       : LongString;   { temporary string buf }
  lbuf2       : LongString;   { temporary string buf }
  lbuf3       : LongString;   { temporary string buf }
  sbuff       : PureString;   { temporary string buf }
  opcount     : real;         { operation count }
  runcount    : real;         { counter of runs }
  EntryCount  : Cardinal;     { recursive call protection }
 public { Compiler tables }
  display     : packed array[0 .. levmax] of integer;
  itab        : ^itab_t;      { identifier table }
  itabmax     : Integer;      { size of itab, see error 63 }
  btab        : ^btab_t;      { block-table }
  btabmax     : Integer;      { size of btab, see error 64 }
  atab        : ^atab_t;      { array-table }
  atabmax     : Integer;      { size of atab, see error 66 }
  rtab        : ^rtab_t;      { real const table}
  rtabmax     : Integer;      { size of rtab, see error 65 }
  ctab        : ^ctab_t;      { compiled P-code instructions table }
  ctabmax     : Integer;      { size of ctab, see error 68 }
  dtab        : ^dtab_t;      { data segment table, i.e. interpreter stack }
  dtabmax     : Integer;      { size of dtab, see error 20 }
  dtabmin     : Integer;      { min dtab space}
  stab        : stab_t;       { dynamic string table }
  stabmax     : Integer;      { size of stab }
  stabmin     : Integer;      { min stab space}
  slenmax     : Integer;      { max string len }
  property    stacksize:Integer read dtabmax;
 public
  stackdump   : boolean;
  nul         : integer; { ref of null string }
  stantyps    : typset;
  worddelims  : TCharSet;      { word delimiters for wordcount & extractword }
  b0_count    : Integer;       { count of b0_stack items }
  b0_stack    : PIntegerArray; { stack of string tables uses by string manager }
  b0_capacity : Integer;       { number of allocated items in b0_stack }
 private
  function  Get_iln:Integer;
  procedure Set_iln(ailn:Integer);
  property  iln  : integer read Get_iln write set_iln; { input line number }
 private
  function  Get_ch:Char;
  procedure Set_ch(ach:Char);
  property  ch : char read Get_ch write Set_ch; { last character read from source program}
 private
  rnum        : real;    { real number from insymbol }
  inum        : integer; { integer from insymbol }
  sleng       : integer; { string length }
 private
  function  Get_cc:Integer;
  procedure Set_cc(acc:Integer);
  property  cc : integer read Get_cc write Set_cc; { character counter }
 private
  lc          : integer; { program location counter }
 private
  function  Get_ll:Integer;
  procedure Set_ll(all:Integer);
  property  ll : integer read Get_ll write Set_ll; { length of current line }
 private
  it          : integer; { indices to tables}
  ia          : integer;
  ib          : integer;
  ic1         : integer;
  ic2         : integer;
  prtables    : boolean;
  sy          : symbol;  { last symbol read by insymbol }
  id          : alfa;    { identifier from insymbol }
  progname    : alfa;
  constbegsys : symset;
  typebegsys  : symset;
  blockbegsys : symset;
  facbegsys   : symset;
  statbegsys  : symset;
 private
  function Get_line: inplineptr;
  property line : inplineptr read Get_line;
 private
  key         : packed array[1..nokeywrds] of alfa;
  ksy         : packed array[1..nokeywrds] of symbol;
  sps         : packed array['!'..'~'] of symbol;
  spnt        : strngptr;       { temporary string pointer }
  timers      : TObjectStorage; { collection of interval timers }
  texts       : TObjectStorage; { collection of text lists }
  {
  general
  }
 public
  constructor Create(Dev:TDaqDevice);
  destructor  Destroy; override;
  procedure   Compile(const Src,Lst:LongString);
  function    Interpret:boolean;
  procedure   FreeStrings;
  function    InitStrings:boolean;
  function    StrAlignLeng(leng:Integer):Integer;
  function    StrBuf(s:integer):strngptr;
  function    StrRef(P:Pointer):integer;
  function    StrPtr(s,o:integer):Pointer;
  function    StrAlloc(size:integer):integer;
  procedure   StrFree(s:integer);
  procedure   Start;
  procedure   Stop;
  procedure   CleanOnStop;
  function    _Flush:integer;
  procedure   ClearFifo;
  {
  compiler internals
  }
 private
  procedure   FatalError(n: integer);
  procedure   OpenIncludeFile(const aFileName:LongString);
  procedure   CloseIncludeFile;
  procedure   nextch;
  procedure   insymbol;
  procedure   enterid(const x0:LongString; x1:objekt; x2:types; x3:integer);
  procedure   enterarray(tp: types; l,h: integer);
  procedure   enterblock;
  procedure   enterreal(x: real);
  procedure   emit(fct: integer);
  procedure   emit1(fct,b: integer);
  procedure   emit2(fct,a,b: integer);
  procedure   printtables;
  procedure   CompileBlock(fsys:symset; isfun:boolean; level:integer);
  {
  DAQ functions
  }
 public
  function    _maxavail:integer;
  function    _msecnow:real;
  function    _secnow:real;
  function    _getticks:real;
  function    _time:real;
  function    _timeunits:real;
  function    _numais:integer;
  function    _numdis:integer;
  function    _numaos:integer;
  function    _numdos:integer;
  function    _numcals:integer;
  function    _refai(n:integer):integer;
  function    _refdi(n:integer):integer;
  function    _refao(n:integer):integer;
  function    _refdo(n:integer):integer;
  function    _refcalibr(n:integer):integer;
  function    _getai_n(n:integer):real;
  function    _getdi_n(n:integer):real;
  function    _crvlen(ref:real):real;
  function    _crvx(ref,i:real):real;
  function    _crvy(ref,i:real):real;
  function    _getai_xn(n:integer):real;
  function    _getai_yn(n:integer):real;
  function    _getdi_xn(n:integer):real;
  function    _getdi_yn(n:integer):real;
  function    _getai(n:integer;t:real):real;
  function    _getdi(n:integer;t:real):real;
  function    _getai_xi(n:integer;i:real):real;
  function    _getai_yi(n:integer;i:real):real;
  function    _getdi_xi(n:integer;i:real):real;
  function    _getdi_yi(n:integer;i:real):real;
  function    _crvput(ref,i,x,y:real):boolean;
  function    _crvinteg(ref,a,b:real):real;
  function    _putev(w,c:integer; t,d0,d1:real):boolean;
  function    _putao(n:integer; t,d:real):boolean;
  function    _putdo(n:integer; t,d:real):boolean;
  function    _calibr(n:integer; t,d:real):real;
  function    _fixerror(n:integer):boolean;
  function    _reset(const arg:LongString):integer;
  function    _rewrite(const arg:LongString):integer;
  function    _append(const arg:LongString):integer;
  function    _ioresult:integer;
  function    _voice(const arg:LongString):boolean;
  function    _action(const arg:LongString):boolean;
  function    _clear(const arg:LongString):boolean;
  function    _cleardevice(const arg:LongString):boolean;
  function    _start(const arg:LongString):boolean;
  function    _stop(const arg:LongString):boolean;
  function    _devsendmsg(const arg:LongString):real;
  function    _debugout(const arg:LongString):boolean;
  function    _clearcurve(const arg:LongString):boolean;
  function    _savecrw(const arg:LongString):boolean;
  function    _specmarker(const arg:LongString):real;
  function    _specmarkerl(const arg:LongString):real;
  function    _specmarkerr(const arg:LongString):real;
  function    _specroil(const arg:LongString):real;
  function    _specroir(const arg:LongString):real;
  function    _windraw(const arg:LongString):boolean;
  function    _winshow(const arg:LongString):boolean;
  function    _winhide(const arg:LongString):boolean;
  function    _winselect(const arg:LongString):boolean;
  function    _global(const arg:LongString):real;
  function    _tm_new:integer;
  function    _tm_free(reftm:real):boolean;
  function    _tm_gettime(reftm:real):real;
  function    _tm_start(reftm:real):boolean;
  function    _tm_stop(reftm:real):boolean;
  function    _tm_isstart(reftm:real):boolean;
  function    _tm_event(reftm:real):boolean;
  function    _tm_curint(reftm:real):integer;
  function    _tm_numint(reftm:real):integer;
  function    _tm_addint(reftm,ms:real):boolean;
  function    _tm_getint(reftm:real;nint:integer):real;
  function    _tm_setint(reftm:real;nint:integer;ms:real):boolean;
  function    _ComOpen(const arg:LongString):boolean;
  function    _ComClose:boolean;
  function    _ComClear:boolean;
  function    _ComCount:integer;
  function    _ComSpace:integer;
  function    _ComWrite(const arg:LongString):boolean;
  function    _ComRead(Count:integer):LongString;
  function    _aimap(i,n:integer):real;
  function    _dimap(i,n:integer):real;
  function    _aomap(i,n:integer):real;
  function    _domap(i,n:integer):real;
  function    _diword(i,n:integer):real;
  function    _irqinit(n:integer):boolean;
  function    _irqfree:boolean;
  function    _isisrnow:boolean;
  function    _readini(s:LongString):LongString;
 public
  function    _caminitpkk(p,b,i,c:integer):boolean;
  function    _camdonepkk:boolean;
  function    _camtypepkk:integer;
  function    _camnaf(c,n,a,f:integer):boolean;
  function    _camwrite(c,n,a,f,d:integer):boolean;
  function    _camwritelong(c,n,a,f:integer;d:real):boolean;
  function    _camread(c,n,a,f:integer):integer;
  function    _camreadlong(c,n,a,f:integer):real;
  function    _camdeclare(c:integer):boolean;
  function    _camzero(c:integer):boolean;
  function    _camclear(c:integer):boolean;
  function    _camsetinhib(c:integer):boolean;
  function    _camclrinhib(c:integer):boolean;
  function    _camsetlmask(c,m:integer):boolean;
  function    _camsetrmask(c,m:integer):boolean;
  function    _camgetlmask(c:integer):integer;
  function    _camgetrword(c:integer):integer;
  function    _camgetbcard(c:integer):integer;
  function    _camisx(c:integer):boolean;
  function    _camisxq(c:integer):boolean;
  function    _camgetcsr(c:integer):integer;
  function    _camenabint(c,m:integer):boolean;
  function    _camdisabint(c,m:integer):boolean;
  function    _camismyreq(c:integer):boolean;
  function    _camgetrcode(c:integer):integer;
  function    _camclrreq(c,m:integer):boolean;
  function    _camsaveregs(c:integer):boolean;
  function    _camrestregs(c:integer):boolean;
  function    _cammaskofst(c:integer):integer;
  function    _camstoflam(c:integer):integer;
  function    _camgetrmask(c:integer):integer;
  function    _pkkclrreq(c:integer):boolean;
  function    _pkkenabint(c:integer):boolean;
  function    _pkkdisabint(c:integer):boolean;
  function    _clickbutton:integer;
  function    _clicksensor:LongString;
  function    _clicktag:integer;
  function    _crvfind(const path:LongString):real;
  function    _crvget(ref,x:real):real;
  function    _crvwhere(ref,x:real):real;
  function    _ms2year(ms:real):integer;
  function    _ms2month(ms:real):integer;
  function    _ms2day(ms:real):integer;
  function    _ms2hour(ms:real):integer;
  function    _ms2min(ms:real):integer;
  function    _ms2sec(ms:real):integer;
  function    _filerename(const arg:LongString):Boolean;
  function    _filecopy(const arg:LongString):Boolean;
 public
  function    _worddelims(const s:LongString):LongString;
  function    _getfattr(const name:LongString):integer;
  function    _setfattr(const name:LongString; Attr:integer):integer;
  function    _doserror:integer;
  function    _paramstr(const arg:LongString):LongString;
 public
  function    _daqdllinit(const s:LongString):integer;
  function    _daqdllfree(id:Integer):boolean;
  function    _daqdllcall(id:integer; cmd:integer):boolean;
 private
  myDllWrap : TDaqDllWrapper;
  function    DllWrap:TDaqDllWrapper;
 public
  function _f_reset(fname:LongString; mode:integer):integer;
  function _f_rewrite(fname:LongString; mode:integer):integer;
  function _f_read(count:integer):LongString;
  function _f_write(var s:LongString):integer;
  function _f_size:longint;
  function _f_seek(pos:longint):longint;
  function _f_close:boolean;
 private
  f_file : file;
 public
  function    _sleep(t:integer):boolean;
  function    _mksecnow:real;
  function    _ShellExecute(const s:longstring):integer;
 public
  crvlocks : record
   Count   : integer;
   Items   : array[0..7] of TCurve;
  end;
 public
  function    _crlf:LongString;
  function    _progname:LongString;
  function    _devname:LongString;
  function    _crvname(ref:real):LongString;
  function    _crvlock(ref:real):boolean;
  function    _crvunlock(ref:real):boolean;
  function    _crvins(ref,i,x,y:real):real;
  function    _crvdel(ref,i,n:real):real;
  function    _inportw(a:integer):integer;
  function    _outportw(a,d:integer):integer;
  function    _inportl(a:integer):integer;
  function    _outportl(a,d:integer):integer;
 private
  _EditInp   : TText;
  _EditAsk   : TText;
  _EditAns   : TText;
  _EditSet   : TText;      // @ settings
  _EditCmd   : TText;      // > send command
  _EditCon   : TText;      // ! confirmation
  _EditReq   : LongString; // Request string
  _EditErr   : Boolean;
  procedure   _EditInit;
  procedure   _EditFree;
  procedure   _EditClear;
 public
  function    _EditStart(const arg:longstring):boolean;
  function    _edit(s:longstring):longstring;
  function    _editstate:integer;
 private
  ee         : TExpressionEvaluator;
 public
  function    _echo(const s:longstring):boolean;
  function    _eval(const s:longstring):real;
  function    _evar(const s:longstring; v:real):boolean;
 private
  plotter3d  : TPlotter3d;
  plotterpar : packed record nx,ny:integer; x1,y1,x2,y2:real; Lock:boolean; end;
 public
  function    _add3d(x,y,z:real):boolean;
  function    _plot3d(nx,ny:integer; x1,y1,x2,y2:real; const opt:longstring):boolean;
  function    _startplot3d(const arg:LongString):boolean;
 public
  function    _task_init(const cmd_line:LongString):Integer;
  function    _task_free(tid:Integer):Boolean;
  function    _task_ref(tid:Integer):Integer;
  function    _task_pid(tid:Integer):Integer;
  function    _task_run(tid:Integer):Boolean;
  function    _task_wait(tid,timeout:Integer):Boolean;
  function    _task_send(tid:Integer; const data:LongString):Integer;
  function    _task_recv(tid,maxlen:Integer):LongString;
  function    _task_txcount(tid:Integer):Integer;
  function    _task_rxcount(tid:Integer):Integer;
  function    _task_txspace(tid:Integer):Integer;
  function    _task_rxspace(tid:Integer):Integer;
  function    _task_result(tid:Integer):Integer;
  function    _task_kill(tid,how,exit_code,timeout:Integer):Boolean;
  function    _task_ctrl(tid:Integer; const param:LongString):LongString;
  function    _getcomspec:LongString;
 public
  function    _mime_encode(const s:LongString):LongString;
  function    _mime_decode(const s:LongString):LongString;
 public
  function    _clickparams(const s:LongString):LongString;
  function    _str2shortcut(const s:LongString):Integer;
  function    _shortcut2str(shortcut:Integer):LongString;
 public
  function    _crvgetln(ref:real; i:Integer):LongString;
  function    _crvputln(ref:real; i:Integer; const s:LongString):Boolean;
  function    _crvinsln(ref:real; i:Integer; const s:LongString):Boolean;
  function    _crvaddln(ref:real; const s:LongString):Boolean;
  function    _crvdelln(ref:real; i:Integer):Boolean;
  function    _crvnumln(ref:real):Integer;
 public
  function    _awakeflag:Boolean;
 public
  function    _reffind(const s:LongString):Integer;
  function    _refinfo(refref:real; const s:LongString):LongString;
  function    _devsend(ref:Integer; const s:LongString):real;
  function    _devpost(ref:Integer; const s:LongString):real;
  function    _devpostmsg(const arg:LongString):real;
 public
  function    _issametext(const s1,s2:LongString):Boolean;
  function    _defaultextension(const s1,s2:LongString):LongString;
  function    _forceextension(const s1,s2:LongString):LongString;
  function    _defaultpath(const s1,s2:LongString):LongString;
  function    _forcepath(const s1,s2:LongString):LongString;
  function    _makerelativepath(const s1,s2:LongString):LongString;
  function    _trim(const s:LongString):LongString;
  function    _trimleft(const s:LongString):LongString;
  function    _trimright(const s:LongString):LongString;
  function    _fexpand(const s:LongString):LongString;
  function    _addbackslash(const s:LongString):LongString;
  function    _dropbackslash(const s:LongString):LongString;
  function    _extractfilepath(const s:LongString):LongString;
  function    _extractfilename(const s:LongString):LongString;
  function    _extractfileext(const s:LongString):LongString;
  function    _iswildcard(const s:LongString):Boolean;
  function    _isrelativepath(const s:LongString):Boolean;
  function    _hasextension(const s:LongString):Boolean;
  function    _direxists(const s:LongString):Boolean;
 public
  function    _text_new:Integer;
  function    _text_free(ref:Integer):Boolean;
  function    _text_getln(ref:Integer; i:Integer):LongString;
  function    _text_putln(ref:Integer; i:Integer; const s:LongString):Boolean;
  function    _text_insln(ref:Integer; i:Integer; const s:LongString):Boolean;
  function    _text_addln(ref:Integer; const s:LongString):Boolean;
  function    _text_delln(ref:Integer; i:Integer):Boolean;
  function    _text_numln(ref:Integer):Integer;
  function    _readinisection(txt,flags:integer; const fname,sec:LongString):integer;
  function    _getenv(const n:LongString):LongString;
  function    _setenv(const n,v:LongString):Boolean;
 private
  _cryptkind : TEncryptionKind;
  _cryptmode : TEncryptionMode;
  _cryptei   : TDataViewFormat;
  _crypteo   : TDataViewFormat;
  _cryptdi   : TDataViewFormat;
  _cryptdo   : TDataViewFormat;
  _cryptiv   : LongString;
 public
  function    _hex_encode(const s:LongString):LongString;
  function    _hex_decode(const s:LongString):LongString;
  function    _crypt_ctrl(const s:LongString):LongString;
  function    _crypt_encode(const s,key:LongString):LongString;
  function    _crypt_decode(const s,key:LongString):LongString;
  function    _getmd5fromstr(const s:LongString):LongString;
  function    _getmd5fromfile(const s:LongString):LongString;
  function    _getmd5fromtext(ref:Integer):LongString;
 public
  function    _wdt_reset(flag:Boolean):real;
  function    _guard_check(const s:LongString):Integer;
  function    _dirlist(txt,maxlev:integer; const dir,pat:LongString):integer;
  function    _pidlist(txt:integer):integer;
  function    _pidkill(pid,cod,lev:Integer):Integer;
  function    _enumerate(txt:integer; const what:LongString):integer;
  function    _timebase:real;
  function    _daqfileref(f,e:LongString):LongString;
 public
  function    _GetClockRes(what:Integer):real;
  function    _SetClockRes(res:real):real;
 public
  function    _url_encode(const s:LongString):LongString;
  function    _url_decode(const s:LongString):LongString;
 public
  function    _datetime2ms(year,month,day,hour,min,sec,msec:Integer):real;
 private
  CpuBase : Int64;
 public
  function    _cpu_count:Integer;
  function    _cpu_start:Boolean;
  function    _cpu_clock:Real;
  function    _cpu_MHz:Real;
  function    _pidaffinity(pid,mask:Integer):Integer;
  function    _devaffinity(ref,mask:Integer):Integer;
 private
  _pipelist:TObjectStorage;
 public
  function _pipe_init(const cmd_line:LongString):Integer;
  function _pipe_free(pid:Integer):Boolean;
  function _pipe_ref(pid:Integer):Integer;
  function _pipe_pid(pid:Integer):Integer;
  function _pipe_run(pid:Integer):Boolean;
  function _pipe_wait(pid,timeout:Integer):Boolean;
  function _pipe_send(pid:Integer; const data:LongString):Integer;
  function _pipe_recv(pid,maxlen:Integer):LongString;
  function _pipe_txcount(pid:Integer):Integer;
  function _pipe_rxcount(pid:Integer):Integer;
  function _pipe_txspace(pid:Integer):Integer;
  function _pipe_rxspace(pid:Integer):Integer;
  function _pipe_result(pid:Integer):Integer;
  function _pipe_kill(pid,how,exit_code,timeout:Integer):Boolean;
  function _pipe_ctrl(pid:Integer; const param:LongString):LongString;
  function _pipe_txclear(pid:Integer):Boolean;
  function _pipe_rxclear(pid:Integer):Boolean;
  function _pipe_count(pid:Integer):Integer;
  function _pipe_stream(pid,index:Integer):Integer;
  function _pipe_connected(pid:Integer):Integer;
  function _strconv(const how,s:LongString):LongString;
  function _dump2f(const s:LongString):Real;
  function _dumpf(x:Real):LongString;
  function _url_packed(const s:LongString):LongString;
 private
  myBase32 : TBase32Coder;
  procedure myBase32Init;
 public
  function _base32_encode(const s:LongString):LongString;
  function _base32_decode(const s:LongString):LongString;
  function _base32_alphabet(const s:LongString):LongString;
  function _hashindexof(const s:LongString; aTableSize,aMethod:Integer):Integer;
  function _getpid:DWORD;
 private
  _hashlist:TObjectStorage;
 public
  function _hashlist_init(aMode:Integer):Integer;
  function _hashlist_free(aRef:Integer):Boolean;
  function _hashlist_count(aRef:Integer):Integer;
  function _hashlist_getkey(aRef:Integer; aIndex:Integer):LongString;
  function _hashlist_delete(aRef:Integer; aKey:LongString):Boolean;
  function _hashlist_indexof(aRef:Integer; aKey:LongString):Integer;
  function _hashlist_getdata(aRef:Integer; aKey:LongString):Double;
  function _hashlist_setdata(aRef:Integer; aKey:LongString; aData:Double):Boolean;
  function _hashlist_getlink(aRef:Integer; aKey:LongString):Integer;
  function _hashlist_setlink(aRef:Integer; aKey:LongString; aLink:Integer):Boolean;
  function _hashlist_getpara(aRef:Integer; aKey:LongString):LongString;
  function _hashlist_setpara(aRef:Integer; aKey:LongString; aPara:LongString):Boolean;
 public
  function _clickwrite(const lines:LongString):Integer;
  function _clickread:Integer;
  function _clickwhat:Integer;
  function _clickwrote:Integer;
  function _clickfilter(aFilter:Integer):Integer;
  function _clickawaker(aAwaker:Integer):Integer;
 public
  taglocks : record
   Count   : Integer;
  end;
 public
  function _taglock(lock:Boolean):Integer;
 public
  function _getai_par(n:integer; id:integer):real;
  function _getao_par(n:integer; id:integer):real;
  function _getdi_par(n:integer; id:integer):real;
  function _getdo_par(n:integer; id:integer):real;
 public
  function _strfmt(const Fmt:LongString; const Args: array of const):LongString;
 private
  _easyipclist : TObjectStorage;
 public
  function _EasyIpc_Init(const PipeName,Options:LongString):PtrUint;
  function _EasyIpc_Free(hIpc:PtrUint):Boolean;
  function _EasyIpc_Poll(hIpc:PtrUint):Boolean;
  function _EasyIpc_Send(hIpc:PtrUint; const TextLines:LongString):Boolean;
  function _EasyIpc_Recv(hIpc:PtrUint; Count:Integer):LongString;
  function _EasyIpc_Ctrl(hIpc:PtrUint; const Request:LongString):LongString;
 public
  function _expenv(const n:LongString):LongString;
  function _createtempfile(const s:LongString):LongString;
  function _stringreplace(const s,p,n:LongString; f:Integer):LongString;
  function _ansiquotedstr(const s:LongString; q:Char):LongString;
  function _ansidequotedstr(const s:LongString; q:Char):LongString;
  function _ansiskipquotedstr(const s:LongString; q:Char):LongString;
 public
  function _text_tostring(ref:Integer):LongString;
 public
  function _extractfirstparam(const arg:LongString; quote:Char):LongString;
  function _skipfirstparam(const arg:LongString; quote:Char):LongString;
  function _isoption(const arg,opt:LongString):Boolean;
  function _getoptionvalue(const arg:LongString):LongString;
  function _ansiquotedifneed(const arg:LongString; quote:Char):LongString;
 private
  _regexplist : TObjectStorage;
 public
  function _regexp_init(engine:Integer; pattern:LongString; options:LongString=''):Integer;
  function _regexp_free(rex:Integer):Boolean;
  function _regexp_ref(rex:Integer):Integer;
  function _regexp_ctrl(rex:Integer; arg:LongString):LongString;
  function _regexp_test(rex:Integer; arg:LongString):Boolean;
  function _regexp_exec(rex:Integer; arg:LongString):Integer;
  function _regexp_replace(rex:Integer; arg,rep:LongString):LongString;
  function _regexp_matchnum(rex:Integer; i:Integer):Integer;
  function _regexp_matchpos(rex:Integer; i,j:Integer):Integer;
  function _regexp_matchlen(rex:Integer; i,j:Integer):Integer;
  function _regexp_matchstr(rex:Integer; i,j:Integer):LongString;
  function _regexp_escape(arg:LongString):LongString;
 private
  _backslash_esclist : TCharSet;
  _backslash_hexlist : TCharSet;
 public
  function _backslash_encode(arg:LongString):LongString;
  function _backslash_decode(arg:LongString):LongString;
  function _backslash_encoder_ctrl(arg:LongString):LongString;
 private
  _fsmlist : TObjectStorage;
 public
  function _fsm_new:Integer;
  function _fsm_free(fsm:Integer):Boolean;
  function _fsm_ref(fsm:Integer):Integer;
  function _fsm_root(fsm:Integer):Integer;
  function _fsm_type(fsm:Integer):Integer;
  function _fsm_parent(fsm:Integer):Integer;
  function _fsm_name(fsm:Integer):LongString;
  function _fsm_path(fsm:Integer):LongString;
  function _fsm_ctrl(fsm:Integer; arg:LongString):LongString;
  function _fsm_count(fsm,typ:Integer):Integer;
  function _fsm_items(fsm,typ,i:Integer):Integer;
  function _fsm_get_iparam(fsm:Integer):Integer;
  function _fsm_set_iparam(fsm:Integer; data:Integer):Boolean;
  function _fsm_get_fparam(fsm:Integer):Double;
  function _fsm_set_fparam(fsm:Integer; data:Double):Boolean;
  function _fsm_get_sparam(fsm:Integer):LongString;
  function _fsm_set_sparam(fsm:Integer; data:LongString):Boolean;
  function _fsm_add(fsm:Integer; typ:integer; key:LongString):Integer;
  function _fsm_find(fsm:Integer; typ:integer; key:LongString):Integer;
  function _fsm_get_state(fsm:Integer):Integer;
  function _fsm_set_state(fsm:Integer; state:LongString):Integer; overload;
  function _fsm_set_state(fsm:Integer; state:Integer):Integer; overload;
  function _fsm_link(fsm:Integer; arg:LongString):Integer;
  function _fsm_modified(fsm:Integer; delta:Integer):Integer;
  function _fsm_name_rule(typ:Integer):Integer;
 public
  function _upcase(c:Char):Char;
  function _locase(c:Char):Char;
 public
  function _htonl(l:Integer):Integer;
  function _ntohl(l:Integer):Integer;
  function _htons(s:Integer):Integer;
  function _ntohs(s:Integer):Integer;
 private
  _dblist : TObjectStorage;
 public
  function _db_create(arg:LongString):Boolean;
  function _db_connection(eid:Integer; arg:LongString):Integer;
  function _db_recordset(dbo:Integer; arg:LongString):Integer;
  function _db_command(dbo:Integer; arg:LongString):Integer;
  function _db_free(dbo:Integer):Boolean;
  function _db_ref(dbo:Integer):Integer;
  function _db_root(dbo:Integer):Integer;
  function _db_type(dbo:Integer):Integer;
  function _db_parent(dbo:Integer):Integer;
  function _db_state(dbo:Integer):Integer;
  function _db_close(dbo:Integer):Boolean;
  function _db_open(dbo:Integer; opt:Integer):Boolean;
  function _db_ctrl(dbo:Integer; arg:LongString):LongString;
  function _db_bugscount(dbo:Integer):Integer;
  function _db_bugsclear(dbo:Integer):Integer;
  function _db_errors(dbo:Integer):LongString;
  function _db_errorscount(dbo:Integer):Integer;
  function _db_errorsclear(dbo:Integer):Integer;
  function _db_execute(dbo:Integer; arg:LongString; opt:Integer):Integer;
  function _db_cancel(dbo:Integer):Boolean;
  function _db_update(dbr:Integer):Boolean;
  function _db_cancelupdate(dbr:Integer):Boolean;
  function _db_begintrans(dbc:Integer):Integer;
  function _db_committrans(dbc:Integer):Boolean;
  function _db_rollbacktrans(dbc:Integer):Boolean;
  function _db_bof(dbr:Integer):Boolean;
  function _db_eof(dbr:Integer):Boolean;
  function _db_movefirst(dbr:Integer):Boolean;
  function _db_movelast(dbr:Integer):Boolean;
  function _db_movenext(dbr:Integer):Boolean;
  function _db_moveprevious(dbr:Integer):Boolean;
  function _db_fieldscount(dbr:Integer):Integer;
  function _db_fieldsnames(dbr:Integer; i:Integer):LongString;
  function _db_fieldstypes(dbr:Integer; id:LongString):Integer;
  function _db_fieldsasint(dbr:Integer; id:LongString; op:Char; arg:Integer):Integer;
  function _db_fieldsasfloat(dbr:Integer; id:LongString; op:Char; arg:Double):Double;
  function _db_fieldsasstring(dbr:Integer; id:LongString; op:Char; arg:LongString):LongString;
  function _db_addnew(dbr:Integer; arg:LongString):Boolean;
  function _db_delete(dbr:Integer; aff:Integer):Boolean;
  function _db_requery(dbr:Integer; opt:Integer):Boolean;
  function _db_resync(dbr:Integer; aff,res:Integer):Boolean;
  function _db_supports(dbr:Integer; opt:Integer):Boolean;
  function _db_save(dbr:Integer; dst:LongString; fmt:Integer):Boolean;
  function _db_engineid(dbo:Integer):Integer;
  function _db_active(dbo:Integer):Boolean;
 public
  function _strtimefmt(const fmt:LongString; ms:Double):LongString;
 private
  _shmlist : TObjectStorage;
 public
  function _shm_init(name:LongString; size,mode:Integer):Integer;
  function _shm_ref(shm:Integer):Integer;
  function _shm_free(shm:Integer):Boolean;
  function _shm_delink(name:LongString):Boolean;
  function _shm_iop(shm:Integer; offset:Integer; op:Char; data:LongInt):LongInt;
  function _shm_rop(shm:Integer; offset:Integer; op:Char; data:Double):Double;
  function _shm_fop(shm:Integer; offset:Integer; op:Char; data:Single):Single;
  function _shm_sop(shm:Integer; offset:Integer; op:Char; data:LongString):LongString;
  function _shm_ctrl(shm:Integer; arg:LongString):LongString;
  function _shm_ioresult(code:Integer):Integer;
 public
  function _strtointbase(s:LongString; base,def:Integer):Integer;
  function _inttostrbase(value,base,width:Integer):LongString;
 private
  afnm_cur : Integer;
 public
  function _AdaptFileName(const FN:LongString):LongString;
  function _AdaptExeFileName(const FN:LongString):LongString;
  function _AdaptFileNameMode(m:Integer):Integer;
 public
  phrasequotes:LongString;
  function _skipwords(n:Integer; const s:LongString):LongString;
  function _phrasecount(const s:LongString):Integer;
  function _extractphrase(n:Integer; const s:LongString):LongString;
  function _skipphrases(n:Integer; const s:LongString):LongString;
  function _phraselisttotextlines(const s:LongString):LongString;
  function _phrasequotes(const s:LongString):LongString;
 public
  function _AdaptDllFileName(const FN:LongString):LongString;
  function _AdaptLnkFileName(const FN:LongString):LongString;
 public
  function _pct_encode(const arg:LongString):LongString;
  function _pct_decode(const arg:LongString):LongString;
  function _pct_encoder_ctrl(const arg:LongString):LongString;
 private
  _pct_reserved:TCharSet;
  _pct_errors:Integer;
 public
  function _ReadFileToBuffer(const FileName:LongString; Count,Offset:Integer):LongString;
  function _WriteBufferToFile(const FileName,Buffer:LongString; Offset:Integer):Integer;
  function _GetFileProperties(const FileName,Properties:LongString):LongString;
 public
  function _SysLogNotable(aSeverity:Integer):Boolean;
  function _SysLogNote(aSeverity:Integer; aData:LongString):Integer;
 public
  function _wdt_timeout(aTimeout:Integer):Integer;
 end;

type
 EDaqPascalCompileError = class(EEchoException); { exception raises at compilation time }
 EDaqPascalRuntimeError = class(EFailException); { exception raises at execution time   }

procedure Kill(var TheObject:TDaqPascal); overload;

 // Calculate UTF8 char column in line buffer at byte column cc
function GetChColUtf8(const line:inplinebuf; cc:Integer):Integer;

implementation

uses
 _crw_softdev,
 _crw_daqpascalruntime,
 _crw_daqpascaldevice,
 form_crwdaq;

 //////////////////////////////////////////
 // Private Hasher for fast string parsing.
 //////////////////////////////////////////
type
 TStringIdentifier = (
  sid_Unknown,
  // _reffind
  sid_Tag,
  sid_Curve,
  sid_Device,
  sid_Polling,
  sid_Window,
  // _refinfo
  sid_Name,
  sid_Type,
  sid_ClassName,
  sid_Bounds,
  // _strconv
  sid_Ansi_Oem,
  sid_Oem_Ansi,
  sid_Win_Dos,
  sid_Dos_Win,
  sid_Win_Koi,
  sid_Koi_Win,
  sid_Utf8_Ansi,
  sid_Ansi_Utf8,
  sid_Ascii_UpperCase,
  sid_Ascii_LowerCase,
  sid_Ansi_UpperCase,
  sid_Ansi_LowerCase,
  sid_Utf8_UpperCase,
  sid_Utf8_LowerCase,
  // _paramstr
  sid_0,
  sid_1,
  sid_2,
  sid_3,
  sid_4,
  sid_5,
  sid_6,
  sid_7,
  sid_8,
  sid_9,
  sid_HOMEDIR,
  sid_STARTUPPATH,
  sid_TEMPPATH,
  sid_DAQCONFIGPATH,
  sid_DAQCONFIGFILE,
  sid_DAQDATAPATH,
  sid_DAQBACKUPFILE,
  sid_GETCURDIR,
  sid_GETCURRDIR,
  sid_DEVICENAME,
  sid_DEVICEMODEL,
  sid_DEVICEFAMILY,
  sid_PROGNAME,
  sid_SYSINIFILE,
  sid_DAQPROGRAM,
  sid_USERNAME,
  sid_COMPUTERNAME,
  sid_HOSTNAME,
  sid_HOSTLIST,
  sid_DOMAINLIST,
  sid_USERDOMAIN,
  sid_USERLIST,
  sid_IPADDRESS,
  sid_MACADDRESS,
  sid_CONSOLE,
  sid_CONSOLE_RU,
  sid_MAINCONSOLE,
  sid_GUARD,
  sid_ComPortList,
  sid_AdamTraffic,
  sid_CreateGUID,
  sid_CreateClassID,
  sid_GETENV,
  sid_FEXPAND,
  sid_GETEXEBYFILE,
  sid_EXTRACTFILEPATH,
  sid_EXTRACTFILENAME,
  sid_EXTRACTFILEEXT,
  sid_ADDBACKSLASH,
  sid_AdaptFileName,
  sid_AdaptExeFileName,
  sid_AdaptDllFileName,
  sid_AdaptLnkFileName,
  sid_ReadShellLink,
  sid_CURVENAME,
  sid_CURWINNAME,
  sid_TABWINNAME,
  sid_CIRWINNAME,
  sid_SPEWINNAME,
  sid_FILESEARCH,
  sid_SPECFOLDER,
  sid_AnsiCodePage,
  sid_OemCodePage,
  sid_AppFormBounds,
  sid_AppClientBounds,
  sid_DaqSysInfo,
  sid_ColorCode,
  sid_ColorName,
  sid_ColorInfo,
  sid_CharsetCode,
  sid_CharsetName,
  sid_PitchCode,
  sid_PitchName,
  sid_HasherCode,
  sid_HasherName,
  sid_GetSystemAssoc,
  sid_GetSystemFType,
  sid_GetSystemAssocExe,
  sid_GetSystemFTypeExe,
  sid_Compiler_Options,
  sid_Compiler_itabmax,
  sid_Compiler_btabmax,
  sid_Compiler_atabmax,
  sid_Compiler_rtabmax,
  sid_Compiler_dtabmax,
  sid_Compiler_dtabmin,
  sid_Compiler_stabmax,
  sid_Compiler_stabmin,
  sid_Compiler_slenmax,
  sid_SYSTEM,
  sid_MAKERELATIVEPATH,
  sid_FORCEPATH,
  sid_DEFAULTPATH,
  sid_FORCEEXTENSION,
  sid_DEFAULTEXTENSION,
  sid_ADDSEARCHPATH,
  sid_REMSEARCHPATH,
  sid_REGISTRY,
  sid_OleDbProviderNames,
  sid_OdbcDriverNames,
  sid_MainInstance,
  sid_MainThreadId,
  sid_CurrentThreadId,
  sid_CurrentProcessId,
  sid_ApplicationHandle,
  sid_MainFormHandle,
  sid_PollingUseMsgPump,
  sid_PollingThreadId,
  sid_PollingDelay,
  sid_PollingPriority,
  sid_lc_ctype_digit,
  sid_lc_ctype_xdigit,
  sid_lc_ctype_upper,
  sid_lc_ctype_lower,
  sid_lc_ctype_blank,
  sid_lc_ctype_space,
  sid_lc_ctype_cntrl,
  sid_lc_ctype_punct,
  sid_lc_ctype_alpha,
  sid_lc_ctype_alnum,
  sid_lc_ctype_print,
  sid_lc_ctype_graph,
  sid_lc_ctype_word,
  sid_SessionNumber,
  sid_SessionTitle,
  sid_SessionName,
  sid_SessionNb,
  sid_str,
  sid_bin,
  sid_hex,
  sid_nice,
  sid_FetchUriScheme,
  sid_FetchUriAuthority,
  sid_FetchUriPath,
  sid_FetchUriQuery,
  sid_FetchUriFragment,
  sid_FindServicePort,
  sid_ServicePortInfo,
  sid_ServicePortName,
  sid_ServicePortList,
  sid_EtcServicesFile,
  sid_GetListOf,
  sid_ListOf,
  sid_DetectBlobImageType,
  sid_SysLogSeverityList,
  sid_SysLogSeverityCode,
  sid_SysLogSeverityName,
  sid_Unused
 );

const
 Hasher:THashList=nil;

procedure FreeHasher;
begin
 Kill(Hasher);
end;

procedure InitHasher;
 procedure AddSid(const key:LongString; sid:TStringIdentifier);
 begin
  Hasher.KeyedLinks[key]:=Ord(sid);
 end;
begin
 if (Hasher<>nil) then Exit;
 Hasher:=NewHashList(false,HashList_DefaultHasher);
 Hasher.Master:=@Hasher;
 ////////////////////////////////////////////
 // Hash List for fast strings identification
 ////////////////////////////////////////////
 AddSid( 'Tag'                 , sid_Tag);
 AddSid( 'Curve'               , sid_Curve);
 AddSid( 'Device'              , sid_Device);
 AddSid( 'Polling'             , sid_Polling);
 AddSid( 'Window'              , sid_Window);
 AddSid( 'Name'                , sid_Name);
 AddSid( 'Type'                , sid_Type);
 AddSid( 'ClassName'           , sid_ClassName);
 AddSid( 'Bounds'              , sid_Bounds);
 AddSid( 'ansi2oem'            , sid_Ansi_Oem);
 AddSid( 'ansi oem'            , sid_Ansi_Oem);
 AddSid( 'ansi_oem'            , sid_Ansi_Oem);
 AddSid( 'ansi-oem'            , sid_Ansi_Oem);
 AddSid( 'oem2ansi'            , sid_Oem_Ansi);
 AddSid( 'oem ansi'            , sid_Oem_Ansi);
 AddSid( 'oem_ansi'            , sid_Oem_Ansi);
 AddSid( 'oem-ansi'            , sid_Oem_Ansi);
 AddSid( 'win2dos'             , sid_Win_Dos);
 AddSid( 'win dos'             , sid_Win_Dos);
 AddSid( 'win_dos'             , sid_Win_Dos);
 AddSid( 'win-dos'             , sid_Win_Dos);
 AddSid( 'dos2win'             , sid_Dos_Win);
 AddSid( 'dos win'             , sid_Dos_Win);
 AddSid( 'dos_win'             , sid_Dos_Win);
 AddSid( 'dos-win'             , sid_Dos_Win);
 AddSid( 'win2koi'             , sid_Win_Koi);
 AddSid( 'win koi'             , sid_Win_Koi);
 AddSid( 'win_koi'             , sid_Win_Koi);
 AddSid( 'win-koi'             , sid_Win_Koi);
 AddSid( 'koi2win'             , sid_Koi_Win);
 AddSid( 'koi win'             , sid_Koi_Win);
 AddSid( 'koi_win'             , sid_Koi_Win);
 AddSid( 'koi-win'             , sid_Koi_Win);
 AddSid( 'utf8 ansi'           , sid_Utf8_Ansi);
 AddSid( 'utf8_ansi'           , sid_Utf8_Ansi);
 AddSid( 'utf8-ansi'           , sid_Utf8_Ansi);
 AddSid( 'ansi utf8'           , sid_Ansi_Utf8);
 AddSid( 'ansi_utf8'           , sid_Ansi_Utf8);
 AddSid( 'ansi-utf8'           , sid_Ansi_Utf8);
 AddSid( 'ascii uppercase'     , sid_Ascii_UpperCase);
 AddSid( 'ascii_uppercase'     , sid_Ascii_UpperCase);
 AddSid( 'ascii-uppercase'     , sid_Ascii_UpperCase);
 AddSid( 'ascii lowercase'     , sid_Ascii_LowerCase);
 AddSid( 'ascii_lowercase'     , sid_Ascii_LowerCase);
 AddSid( 'ascii-lowercase'     , sid_Ascii_LowerCase);
 AddSid( 'ansi uppercase'      , sid_Ansi_UpperCase);
 AddSid( 'ansi_uppercase'      , sid_Ansi_UpperCase);
 AddSid( 'ansi-uppercase'      , sid_Ansi_UpperCase);
 AddSid( 'ansi lowercase'      , sid_Ansi_LowerCase);
 AddSid( 'ansi_lowercase'      , sid_Ansi_LowerCase);
 AddSid( 'ansi-lowercase'      , sid_Ansi_LowerCase);
 AddSid( 'utf8 uppercase'      , sid_Utf8_UpperCase);
 AddSid( 'utf8_uppercase'      , sid_Utf8_UpperCase);
 AddSid( 'utf8-uppercase'      , sid_Utf8_UpperCase);
 AddSid( 'utf8 lowercase'      , sid_Utf8_LowerCase);
 AddSid( 'utf8_lowercase'      , sid_Utf8_LowerCase);
 AddSid( 'utf8-lowercase'      , sid_Utf8_LowerCase);
 AddSid( '0'                   , sid_0);
 AddSid( '1'                   , sid_1);
 AddSid( '2'                   , sid_2);
 AddSid( '3'                   , sid_3);
 AddSid( '4'                   , sid_4);
 AddSid( '5'                   , sid_5);
 AddSid( '6'                   , sid_6);
 AddSid( '7'                   , sid_7);
 AddSid( '8'                   , sid_8);
 AddSid( '9'                   , sid_9);
 AddSid( 'HOMEDIR'             , sid_HOMEDIR);
 AddSid( 'STARTUPPATH'         , sid_STARTUPPATH);
 AddSid( 'TEMPPATH'            , sid_TEMPPATH);
 AddSid( 'DAQCONFIGPATH'       , sid_DAQCONFIGPATH);
 AddSid( 'DAQCONFIGFILE'       , sid_DAQCONFIGFILE);
 AddSid( 'DAQDATAPATH'         , sid_DAQDATAPATH);
 AddSid( 'DAQBACKUPFILE'       , sid_DAQBACKUPFILE);
 AddSid( 'GETCURDIR'           , sid_GETCURDIR);
 AddSid( 'GETCURRDIR'          , sid_GETCURRDIR);
 AddSid( 'DEVICENAME'          , sid_DEVICENAME);
 AddSid( 'DEVICEMODEL'         , sid_DEVICEMODEL);
 AddSid( 'DEVICEFAMILY'        , sid_DEVICEFAMILY);
 AddSid( 'PROGNAME'            , sid_PROGNAME);
 AddSid( 'SYSINIFILE'          , sid_SYSINIFILE);
 AddSid( 'DAQPROGRAM'          , sid_DAQPROGRAM);
 AddSid( 'USERNAME'            , sid_USERNAME);
 AddSid( 'COMPUTERNAME'        , sid_COMPUTERNAME);
 AddSid( 'HOSTNAME'            , sid_HOSTNAME);
 AddSid( 'HOSTLIST'            , sid_HOSTLIST);
 AddSid( 'DOMAINLIST'          , sid_DOMAINLIST);
 AddSid( 'USERDOMAIN'          , sid_USERDOMAIN);
 AddSid( 'USERLIST'            , sid_USERLIST);
 AddSid( 'IPADDRESS'           , sid_IPADDRESS);
 AddSid( 'MACADDRESS'          , sid_MACADDRESS);
 AddSid( 'CONSOLE'             , sid_CONSOLE);
 AddSid( 'КОНСОЛЬ'             , sid_CONSOLE_RU);
 AddSid( 'MAINCONSOLE'         , sid_MAINCONSOLE);
 AddSid( 'GUARD'               , sid_GUARD);
 AddSid( 'ComPortList'         , sid_ComPortList);
 AddSid( 'AdamTraffic'         , sid_AdamTraffic);
 AddSid( 'CreateGUID'          , sid_CreateGUID);
 AddSid( 'CreateClassID'       , sid_CreateClassID);
 AddSid( 'GETENV'              , sid_GETENV);
 AddSid( 'FEXPAND'             , sid_FEXPAND);
 AddSid( 'GETEXEBYFILE'        , sid_GETEXEBYFILE);
 AddSid( 'EXTRACTFILEPATH'     , sid_EXTRACTFILEPATH);
 AddSid( 'EXTRACTFILENAME'     , sid_EXTRACTFILENAME);
 AddSid( 'EXTRACTFILEEXT'      , sid_EXTRACTFILEEXT);
 AddSid( 'ADDBACKSLASH'        , sid_ADDBACKSLASH);
 AddSid( 'ADAPTFILENAME'       , sid_AdaptFileName);
 AddSid( 'ADAPTEXEFILENAME'    , sid_AdaptExeFileName);
 AddSid( 'ADAPTDLLFILENAME'    , sid_AdaptDllFileName);
 AddSid( 'ADAPTLNKFILENAME'    , sid_AdaptLnkFileName);
 AddSid( 'READSHELLLINK'       , sid_ReadShellLink);
 AddSid( 'CURVENAME'           , sid_CURVENAME);
 AddSid( 'CURWINNAME'          , sid_CURWINNAME);
 AddSid( 'TABWINNAME'          , sid_TABWINNAME);
 AddSid( 'CIRWINNAME'          , sid_CIRWINNAME);
 AddSid( 'SPEWINNAME'          , sid_SPEWINNAME);
 AddSid( 'FILESEARCH'          , sid_FILESEARCH);
 AddSid( 'SPECFOLDER'          , sid_SPECFOLDER);
 AddSid( 'AnsiCodePage'        , sid_AnsiCodePage);
 AddSid( 'OemCodePage'         , sid_OemCodePage);
 AddSid( 'AppFormBounds'       , sid_AppFormBounds);   
 AddSid( 'AppClientBounds'     , sid_AppClientBounds);
 AddSid( 'DaqSysInfo'          , sid_DaqSysInfo);
 AddSid( 'ColorCode'           , sid_ColorCode);
 AddSid( 'ColorName'           , sid_ColorName);
 AddSid( 'ColorInfo'           , sid_ColorInfo);
 AddSid( 'CharsetCode'         , sid_CharsetCode);
 AddSid( 'CharsetName'         , sid_CharsetName);
 AddSid( 'PitchCode'           , sid_PitchCode);
 AddSid( 'PitchName'           , sid_PitchName);
 AddSid( 'HasherCode'          , sid_HasherCode);
 AddSid( 'HasherName'          , sid_HasherName);
 AddSid( 'GetSystemAssoc'      , sid_GetSystemAssoc);
 AddSid( 'GetSystemFType'      , sid_GetSystemFType);
 AddSid( 'GetSystemAssocExe'   , sid_GetSystemAssocExe);
 AddSid( 'GetSystemFTypeExe'   , sid_GetSystemFTypeExe);
 AddSid( '[Compiler.Options]'  , sid_Compiler_Options);
 AddSid( 'Compiler.itabmax'    , sid_Compiler_itabmax);
 AddSid( 'Compiler.btabmax'    , sid_Compiler_btabmax);
 AddSid( 'Compiler.atabmax'    , sid_Compiler_atabmax);
 AddSid( 'Compiler.rtabmax'    , sid_Compiler_rtabmax);
 AddSid( 'Compiler.dtabmax'    , sid_Compiler_dtabmax);
 AddSid( 'Compiler.dtabmin'    , sid_Compiler_dtabmin);
 AddSid( 'Compiler.stabmax'    , sid_Compiler_stabmax);
 AddSid( 'Compiler.stabmin'    , sid_Compiler_stabmin);
 AddSid( 'Compiler.slenmax'    , sid_Compiler_slenmax);
 AddSid( 'SYSTEM'              , sid_SYSTEM);
 AddSid( 'MAKERELATIVEPATH'    , sid_MAKERELATIVEPATH);
 AddSid( 'FORCEPATH'           , sid_FORCEPATH);
 AddSid( 'DEFAULTPATH'         , sid_DEFAULTPATH);
 AddSid( 'FORCEEXTENSION'      , sid_FORCEEXTENSION);
 AddSid( 'DEFAULTEXTENSION'    , sid_DEFAULTEXTENSION);
 AddSid( 'ADDSEARCHPATH'       , sid_ADDSEARCHPATH);
 AddSid( 'REMSEARCHPATH'       , sid_REMSEARCHPATH);
 AddSid( 'REGISTRY'            , sid_REGISTRY);
 AddSid( 'OleDbProviderNames'  , sid_OleDbProviderNames);
 AddSid( 'OdbcDriverNames'     , sid_OdbcDriverNames);
 AddSid( 'MainInstance'        , sid_MainInstance);
 AddSid( 'MainThreadId'        , sid_MainThreadId);
 AddSid( 'CurrentThreadId'     , sid_CurrentThreadId);
 AddSid( 'CurrentProcessId'    , sid_CurrentProcessId);
 AddSid( 'ApplicationHandle'   , sid_ApplicationHandle);
 AddSid( 'MainFormHandle'      , sid_MainFormHandle);
 AddSid( 'Polling.UseMsgPump'  , sid_PollingUseMsgPump);
 AddSid( 'Polling.ThreadID'    , sid_PollingThreadId);
 AddSid( 'Polling.Delay'       , sid_PollingDelay);
 AddSid( 'Polling.Priority'    , sid_PollingPriority);
 AddSid( '[:digit:]'           , sid_lc_ctype_digit);
 AddSid( '[:xdigit:]'          , sid_lc_ctype_xdigit);
 AddSid( '[:upper:]'           , sid_lc_ctype_upper);
 AddSid( '[:lower:]'           , sid_lc_ctype_lower);
 AddSid( '[:blank:]'           , sid_lc_ctype_blank);
 AddSid( '[:space:]'           , sid_lc_ctype_space);
 AddSid( '[:cntrl:]'           , sid_lc_ctype_cntrl);
 AddSid( '[:punct:]'           , sid_lc_ctype_punct);
 AddSid( '[:alpha:]'           , sid_lc_ctype_alpha);
 AddSid( '[:alnum:]'           , sid_lc_ctype_alnum);
 AddSid( '[:print:]'           , sid_lc_ctype_print);
 AddSid( '[:graph:]'           , sid_lc_ctype_graph);
 AddSid( '[:word:]'            , sid_lc_ctype_word);
 AddSid( 'SessionNumber'       , sid_SessionNumber);
 AddSid( 'SessionTitle'        , sid_SessionTitle);
 AddSid( 'SessionName'         , sid_SessionName);
 AddSid( 'SessionNb'           , sid_SessionNb);
 AddSid( 'str'                 , sid_str);
 AddSid( 'bin'                 , sid_bin);
 AddSid( 'hex'                 , sid_hex);
 AddSid( 'nice'                , sid_nice);
 AddSid( 'FetchUriScheme'      , sid_FetchUriScheme);
 AddSid( 'FetchUriAuthority'   , sid_FetchUriAuthority);
 AddSid( 'FetchUriPath'        , sid_FetchUriPath);
 AddSid( 'FetchUriQuery'       , sid_FetchUriQuery);
 AddSid( 'FetchUriFragment'    , sid_FetchUriFragment);
 AddSid( 'FindServicePort'     , sid_FindServicePort);
 AddSid( 'ServicePortInfo'     , sid_ServicePortInfo);
 AddSid( 'ServicePortName'     , sid_ServicePortName);
 AddSid( 'ServicePortList'     , sid_ServicePortList);
 AddSid( 'EtcServicesFile'     , sid_EtcServicesFile);
 AddSid( 'GetListOf'           , sid_ListOf);
 AddSid( 'ListOf'              , sid_ListOf);
 AddSid( 'DetectBlobImageType' , sid_DetectBlobImageType);
 AddSid( 'SysLogSeverityList'  , sid_SysLogSeverityList);
 AddSid( 'SysLogSeverityCode'  , sid_SysLogSeverityCode);
 AddSid( 'SysLogSeverityName'  , sid_SysLogSeverityName);
end;

function Identify(const key:LongString):TStringIdentifier;
var sid:Integer;
begin
 if (Hasher=nil) then InitHasher;
 sid:=Hasher.KeyedLinks[key];
 if (sid>=Ord(Low(TStringIdentifier))) and (sid<=Ord(High(TStringIdentifier)))
 then Result:=TStringIdentifier(sid)
 else Result:=sid_Unknown;
end;

procedure Kill(var TheObject:TDaqPascal); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

function GetChColUtf8(const line:inplinebuf; cc:Integer):Integer;
var s:LongString;
begin
 Result:=EnsureRange(cc,Low(line),High(Line));
 if (cc<=1) or not IsSysUtf8 then Exit;
 SetString(s,PChar(@line),cc-1);
 if not utf8_valid(s) then Exit;
 Result:=utf8_length(s)+1;s:='';
end;

procedure ClearTable(tab:Pointer);
begin
 if AllocSize(tab)>0 then SafeFillChar(tab^,AllocSize(tab),0);
end;

procedure AllocateTable(var tab:Pointer; var tabmax:Integer; size,itemsize,n:Integer; tabClear:Boolean);
begin
 Reallocate(tab, (size+n+SafetyGap) * itemsize);
 tabmax:=(AllocSize(tab) div itemsize)-n-SafetyGap;
 if tabClear then ClearTable(tab);
end;

 {
 Set a alfa value by given string s
 }
procedure setalfa(var a:alfa; const s:LongString);
var len:integer;
begin
 fillchar(a,sizeof(a),' ');
 if length(s)<sizeof(a) then len:=length(s) else len:=sizeof(a);
 if len>0 then move(s[1],a,len);
end;

 {
 Init compiler
 }
constructor TDaqPascal.Create(Dev:TDaqDevice);
var i:Integer;
begin
 inherited Create;
 TheDevice:=Dev;
 CompileOk:=false;
 DebugMode:=false;
 ErrorMessage:='';
 ErrorDetails:='';
 SrcFile:='';
 LstFile:='';
 ErrorCode:=0;
 MortalTime:=0;
 MortalDelay:=DefMortWait;
 MortalCount:=0;
 ErrorLine:=0;
 ErrorColumn:=0;
 ErrorColUtf8:=0;
 BreakFlag:=0;
 AllocateTable(Pointer(itab), itabmax, itabmax_def, sizeof(itab[0]), 1, true);
 AllocateTable(Pointer(btab), btabmax, btabmax_def, sizeof(btab[1]), 0, true);
 AllocateTable(Pointer(atab), atabmax, atabmax_def, sizeof(atab[1]), 0, true);
 AllocateTable(Pointer(rtab), rtabmax, rtabmax_def, sizeof(rtab[1]), 0, true);
 AllocateTable(Pointer(ctab), ctabmax, ctabmax_def, sizeof(ctab[0]), 1, true);
 AllocateTable(Pointer(dtab), dtabmax, dtabmax_def, sizeof(dtab.s[0]), 1, true);
 AllocateTable(Pointer(stab.stack), stabmax, stabmax_def, sizeof(stab.stack[1]), 0, true);
 AllocateTable(Pointer(stab.table), stabmax, stabmax_def, sizeof(stab.table[1]), 0, true);
 dtabmin:=dtabmin_def;
 stabmin:=stabmin_def;
 slenmax:=slenmax_def;
 stab.space:=0;
 stab.count:=0;
 InpFifo:=NewFifo(InpFifoSize);
 OutFifo:=NewFifo(OutFifoSize);
 ComIO.Pipe:=nil; ComIO.PortN:=0; ComIO.PipeLine:=0;
 PasTop:=Low(PasStack);
 for i:=Low(PasStack) to High(PasStack) do PasStack[i].Src:='';
 for i:=Low(PasStack) to High(PasStack) do AssignNull(PasStack[i].Inp);
 AssignNull(LstOut);
 AssignNull(StdInp);
 AssignNull(StdOut);
 IORes:=0;
 Adam_Slot:=nil;
 timers:=NewObjectStorage(true);
 timers.Master:=@timers;
 texts:=NewObjectStorage(true);
 texts.Master:=@texts;
 worddelims:=ScanSpaces;
 phrasequotes:=QuoteMark+Apostrophe;
 _pct_reserved:=[];
 _pct_errors:=0;
 System.Assign(f_file,'');
 myDllWrap:=nil;
 _EditInit;
 ee:=NewExpressionEvaluator;
 ee.Master:=@ee;
 ee.Exceptions:=false;
 ee.Custom:=Self;
 plotter3d:=TPlotter3d.Create;
 plotterpar.Lock:=false;
 _cryptkind:=ek_Blowfish;
 _cryptmode:=em_CBC;
 _cryptei:=df_Bin;
 _crypteo:=df_Mime;
 _cryptdi:=df_Mime;
 _cryptdo:=df_Bin;
 _cryptiv:='';
 crvlocks.Count:=0;
 b0_count:=0;
 b0_stack:=nil;
 b0_capacity:=0;
 _pipelist:=NewObjectStorage(False,0,16);
 _pipelist.Master:=@_pipelist;
 _hashlist:=NewObjectStorage(False,0,16);
 _hashlist.Master:=@_hashlist;
 _regexplist:=NewObjectStorage(False,0,16);
 _regexplist.Master:=@_regexplist;
 _fsmlist:=NewObjectStorage(False,0,16);
 _fsmlist.Master:=@_fsmlist;
 _dblist:=NewObjectStorage(False,0,16);
 _dblist.Master:=@_dblist;
 _shmlist:=NewObjectStorage(False,0,16);
 _shmlist.Master:=@_shmlist;
 taglocks.Count:=0;
 _easyipclist:=NewObjectStorage(False,0,16);
 _easyipclist.Master:=@_easyipclist;
 _backslash_esclist:=[];
 _backslash_hexlist:=[];
 afnm_cur:=afnm_Def;
end;

 {
 Destroy compiler
 }
destructor TDaqPascal.Destroy;
var i:Integer;
begin
 ErrorMessage:='';
 ErrorDetails:='';
 SrcFile:='';
 LstFile:='';
 phrasequotes:='';
 Finalize(_cryptiv);
 Kill(TObject(plotter3d));
 Kill(ee);
 _EditFree;
 SetInOutRes(0);
 PasTop:=Low(PasStack);
 for i:=Low(PasStack) to High(PasStack) do PasStack[i].Src:='';
 for i:=Low(PasStack) to High(PasStack) do SmartFileClose(PasStack[i].Inp);
 SmartFileClose(LstOut);
 SmartFileClose(StdInp);
 SmartFileClose(StdOut);
 SmartFileClose(f_file);
 FreeStrings;
 Kill(texts);
 Kill(timers);
 Kill(InpFifo);
 Kill(OutFifo);
 Kill(myDllWrap);
 Deallocate(Pointer(itab));
 Deallocate(Pointer(btab));
 Deallocate(Pointer(atab));
 Deallocate(Pointer(rtab));
 Deallocate(Pointer(ctab));
 Deallocate(Pointer(dtab));
 Deallocate(Pointer(stab.stack));
 Deallocate(Pointer(stab.table));
 Deallocate(Pointer(b0_stack));
 Kill(_pipelist);
 Kill(_hashlist);
 Kill(_regexplist);
 Kill(_fsmlist);
 Kill(_dblist);
 Kill(_shmlist);
 Kill(myBase32);
 Kill(_easyipclist);
 inherited Destroy;
end;

procedure TDaqPascal.Set_BreakFlag(aFlag:Integer);
begin
 LockedExchange(myBreakFlag,aFlag);
end;

function TDaqPascal.Get_PasSrc:LongString;
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 Result:=PasStack[PasTop].Src;
end;
procedure TDaqPascal.Set_PasSrc(const aPasSrc:LongString);
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 PasStack[PasTop].Src:=aPasSrc;
end;

function TDaqPascal.Get_PasInp:textptr;
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 Result:=@(PasStack[PasTop].Inp);
end;

function TDaqPascal.Get_iln:Integer;
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 Result:=PasStack[PasTop].iln;
end;
procedure TDaqPascal.Set_iln(ailn:Integer);
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 PasStack[PasTop].iln:=ailn;
end;

function TDaqPascal.Get_ch:Char;
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 Result:=PasStack[PasTop].ch;
end;
procedure TDaqPascal.Set_ch(ach:Char);
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 PasStack[PasTop].ch:=ach;
end;

function TDaqPascal.Get_cc:Integer;
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 Result:=PasStack[PasTop].cc;
end;
procedure TDaqPascal.Set_cc(acc:Integer);
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 PasStack[PasTop].cc:=acc;
end;

function TDaqPascal.Get_ll:Integer;
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 Result:=PasStack[PasTop].ll;
end;
procedure TDaqPascal.Set_ll(all:Integer);
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 PasStack[PasTop].ll:=all;
end;

function TDaqPascal.Get_line:inplineptr;
begin
 if PasTop<Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 Result:=@(PasStack[PasTop].line);
end;

 {
 Compile file Src, comments write to Lst
 }
procedure TDaqPascal.Compile(const Src,Lst:LongString);
var i:Integer; CompTime:Double; eline:LongString;
 function CompilerOption(const OptName:LongString; var OptValue:Integer):Boolean;
 var s:LongString; buff:TParsingBuffer; ee:TExpressionEvaluator;
 begin
  s:='';
  Result:=false;
  if ReadIniFileString(SrcFile,'[Compiler.Options]',OptName+'%s',s)
  or ReadIniFileString(Daq.ConfigFile,'[Compiler.Options]',OptName+'%s',s)
  or ReadIniFileString(SysIniFile,'[Compiler.Options]',OptName+'%s',s) then
  if IsNonEmptyStr(s) then begin
   ee:=NewExpressionEvaluator;
   try
    if ee.EvaluateExpression(StrCopyBuff(buff,Trim(s)))=ee_Ok then
    if not IsNan(ee.Answer) and not IsInf(ee.Answer) then begin
     OptValue:=Round(ee.Answer);
     Result:=(OptValue>0);
    end;
   finally
    Kill(ee);
   end;
  end;
 end;
begin
 CompileOk:=false;
 ErrorMessage:='';
 ErrorDetails:='';
 ErrorCode:=0;
 ErrorLine:=0;
 ErrorColumn:=0;
 ErrorColUtf8:=0;
 BreakFlag:=0;
 CompTime:=mSecNow;
 SetInOutRes(0);
 PasTop:=Low(PasStack);
 for i:=Low(PasStack) to High(PasStack) do PasStack[i].Src:='';
 for i:=Low(PasStack) to High(PasStack) do SmartFileClose(PasStack[i].Inp);
 SmartFileClose(LstOut);
 SmartFileClose(StdInp);
 SmartFileClose(StdOut);
 SmartFileClose(f_file);
 FreeStrings;
 ExceptResult:=-1;
 try
  {
  Assign source and list file
  }
  SrcFile:=FExpand(DefaultExtension(Src,'.pas'));
  PasTop:=Low(PasStack);
  PasSrc:=SrcFile;
  assign(PasInp^,PasSrc);
  LstFile:=FExpand(DefaultExtension(Lst,'.lst'));
  assign(LstOut,LstFile);
  {
  Allocate and clear all compiler tables
  }
  {identifier table}
  if CompilerOption('Compiler.itabmax',i) then
  AllocateTable(Pointer(itab), itabmax, i, sizeof(itab[0]), 1, true);
  ClearTable(itab);
  {block table}
  if CompilerOption('Compiler.btabmax',i) then
  AllocateTable(Pointer(btab), btabmax, i, sizeof(btab[1]), 0, true);
  ClearTable(btab);
  {array table}
  if CompilerOption('Compiler.atabmax',i) then
  AllocateTable(Pointer(atab), atabmax, i, sizeof(atab[1]), 0, true);
  ClearTable(atab);
  {real const table}
  if CompilerOption('Compiler.rtabmax',i) then
  AllocateTable(Pointer(rtab), rtabmax, i, sizeof(rtab[1]), 0, true);
  ClearTable(rtab);
  {code segment}
  if CompilerOption('Compiler.ctabmax',i) then
  AllocateTable(Pointer(ctab), ctabmax, i, sizeof(ctab[0]), 1, true);
  ClearTable(ctab);
  {data segment}
  if CompilerOption('Compiler.dtabmax',i) then
  AllocateTable(Pointer(dtab), dtabmax, i, sizeof(dtab.s[0]), 1, true);
  ClearTable(dtab);
  if CompilerOption('Compiler.dtabmin',i) then dtabmin:=i;
  {string table segment}
  if CompilerOption('Compiler.stabmax',i) then begin
   AllocateTable(Pointer(stab.stack), stabmax, i, sizeof(stab.stack[1]), 0, true);
   AllocateTable(Pointer(stab.table), stabmax, i, sizeof(stab.table[1]), 0, true);
  end;
  {string length limit}
  if CompilerOption('Compiler.slenmax',i) then begin
   slenmax:=Max(slenmax_min,Min(slenmax_max,i));
  end;
  ClearTable(stab.stack);
  ClearTable(stab.table);
  if CompilerOption('Compiler.stabmin',i) then stabmin:=i;
  {
  Open source and list file
  }
  rewrite(LstOut);
  if IOResult <> 0 then FatalError(62);
  if not FileExists(PasSrc) then FatalError(61);
  reset(PasInp^);
  if IOResult <> 0 then FatalError(61);
  write(LstOut,'<<<<<  Opened[',PasTop,']: ',PasSrc,EOL,0:5,'  ');
  {
  initialize compiler keywords and symbols
  }
  setalfa( key[ 1] , 'and                 ');
  setalfa( key[ 2] , 'array               ');
  setalfa( key[ 3] , 'begin               ');
  setalfa( key[ 4] , 'case                ');
  setalfa( key[ 5] , 'const               ');
  setalfa( key[ 6] , 'div                 ');
  setalfa( key[ 7] , 'do                  ');
  setalfa( key[ 8] , 'downto              ');
  setalfa( key[ 9] , 'else                ');
  setalfa( key[10] , 'end                 ');
  setalfa( key[11] , 'file                ');
  setalfa( key[12] , 'for                 ');
  setalfa( key[13] , 'function            ');
  setalfa( key[14] , 'goto                ');
  setalfa( key[15] , 'if                  ');
  setalfa( key[16] , 'in                  ');
  setalfa( key[17] , 'label               ');
  setalfa( key[18] , 'mod                 ');
  setalfa( key[19] , 'nil                 ');
  setalfa( key[20] , 'not                 ');
  setalfa( key[21] , 'of                  ');
  setalfa( key[22] , 'or                  ');
  setalfa( key[23] , 'packed              ');
  setalfa( key[24] , 'procedure           ');
  setalfa( key[25] , 'program             ');
  setalfa( key[26] , 'record              ');
  setalfa( key[27] , 'repeat              ');
  setalfa( key[28] , 'set                 ');
  setalfa( key[29] , 'then                ');
  setalfa( key[30] , 'to                  ');
  setalfa( key[31] , 'type                ');
  setalfa( key[32] , 'until               ');
  setalfa( key[33] , 'var                 ');
  setalfa( key[34] , 'while               ');
  setalfa( key[35] , 'with                ');
  ksy[ 1] := andsy;
  ksy[ 2] := arraysy;
  ksy[ 3] := beginsy;
  ksy[ 4] := casesy;
  ksy[ 5] := constsy;
  ksy[ 6] := idiv;
  ksy[ 7] := dosy;
  ksy[ 8] := downtosy;
  ksy[ 9] := elsesy;
  ksy[10] := endsy;
  ksy[11] := filesy;
  ksy[12] := forsy;
  ksy[13] := funcsy;
  ksy[14] := gotosy;
  ksy[15] := ifsy;
  ksy[16] := insy;
  ksy[17] := labelsy;
  ksy[18] := imod;
  ksy[19] := nilsy;
  ksy[20] := notsy;
  ksy[21] := ofsy;
  ksy[22] := orsy;
  ksy[23] := packedsy;
  ksy[24] := procsy;
  ksy[25] := programsy;
  ksy[26] := recordsy;
  ksy[27] := repeatsy;
  ksy[28] := setsy;
  ksy[29] := thensy;
  ksy[30] := tosy;
  ksy[31] := typesy;
  ksy[32] := untilsy;
  ksy[33] := varsy;
  ksy[34] := whilesy;
  ksy[35] := withsy;
  sps['+'] := plus;
  sps['-'] := minus;
  sps['*'] := times;
  sps['/'] := rdiv;
  sps[')'] := rparent;
  sps['='] := eql;
  sps[','] := comma;
  sps['['] := lbrack;
  sps[']'] := rbrack;
  sps['~'] := notsy;
  sps['&'] := andsy;
  sps[';'] := semicolon;
  sps['|'] := orsy;
  constbegsys := [plus,minus,intcon,realcon,charcon,stringcon,ident];
  typebegsys  := [ident,arraysy,recordsy];
  blockbegsys := [constsy,typesy,varsy,procsy,funcsy,beginsy];
  facbegsys   := [intcon,realcon,charcon,stringcon,ident,lparent,notsy];
  statbegsys  := [beginsy,ifsy,whilesy,repeatsy,forsy,casesy];
  stantyps    := [notyp,ints,reals,bools,chars,strngs];
  {
  clear variables
  }
  iln := 0;
  lc := 0;
  ll := 0;
  cc := 0;
  ch := ' ';
  it := -1;
  ia := 0;
  ib :=  1;
  ic2 :=  0;
  display[0] := 1;
  prtables:= false;
  stackdump:= false;
  {
  Check device
  }
  if not Assigned(TheDevice) then FatalError(73);
  if not (TheDevice is TProgramDevice) then FatalError(73);
  {
  init counters and string manager
  }
  opcount:=0;
  runcount:=0;
  EntryCount:=0;
  if not InitStrings then FatalError(72);
  {
  compile "program(input,output);" statement
  }
  insymbol;
  if sy <> programsy then FatalError(3);
  insymbol;
  if sy <> ident then FatalError(2);
  progname := id;
  insymbol;
  if sy = lparent then begin
   repeat
    insymbol;
    if sy<> ident then FatalError(2) else insymbol
   until sy <> comma;
   if sy = rparent then insymbol else FatalError(4);
  end;
  {
  init standard functions
  }
  enterid('                        ', vvariable, notyp,  0);  { sentinel }
  enterid('false                   ', konstant,  bools,  0);
  enterid('true                    ', konstant,  bools,  1);
  enterid('sizeofreal              ', konstant,  ints,   sizeof(real));
  enterid('sizeofchar              ', konstant,  ints,   sizeof(char));
  enterid('sizeofboolean           ', konstant,  ints,   sizeof(boolean));
  enterid('sizeofinteger           ', konstant,  ints,   sizeof(integer));
  enterid('daq_cmd_init            ', konstant,  ints,   1);
  enterid('daq_cmd_free            ', konstant,  ints,   2);
  enterid('daq_cmd_poll            ', konstant,  ints,   3);
  enterid('tag_type_nil            ', konstant,  ints,   tag_type_nil);
  enterid('tag_type_int            ', konstant,  ints,   tag_type_int);
  enterid('tag_type_real           ', konstant,  ints,   tag_type_real);
  enterid('tag_type_string         ', konstant,  ints,   tag_type_string);
  enterid('tag_ref_min             ', konstant,  ints,   TAG_REF_MIN);
  enterid('tag_ref_max             ', konstant,  ints,   TAG_REF_MAX);
  enterid('task_ref_min            ', konstant,  ints,   TASK_REF_MIN);
  enterid('task_ref_max            ', konstant,  ints,   TASK_REF_MAX);
  enterid('rfreplaceall            ', konstant,  ints,   GetBitMask(Ord(rfReplaceAll)));
  enterid('rfignorecase            ', konstant,  ints,   GetBitMask(Ord(rfIgnoreCase)));
  enterid('quotemark               ', konstant,  chars,  Ord(QuoteMark));
  enterid('apostrophe              ', konstant,  chars,  Ord(Apostrophe));
  enterid('lex_ansi                ', konstant,  ints,   lex_Ansi);
  enterid('lex_utf8                ', konstant,  ints,   lex_Utf8);
  enterid('lex_name                ', konstant,  ints,   lex_Name);
  enterid('lex_word                ', konstant,  ints,   lex_Word);
  enterid('lex_blank               ', konstant,  ints,   lex_Blank);
  enterid('lex_space               ', konstant,  ints,   lex_Space);
  enterid('lex_cntrl               ', konstant,  ints,   lex_Cntrl);
  enterid('lex_alpha               ', konstant,  ints,   lex_Alpha);
  enterid('lex_lower               ', konstant,  ints,   lex_Lower);
  enterid('lex_upper               ', konstant,  ints,   lex_Upper);
  enterid('lex_digit               ', konstant,  ints,   lex_Digit);
  enterid('lex_alnum               ', konstant,  ints,   lex_Alnum);
  enterid('lex_xdigit              ', konstant,  ints,   lex_xDigit);
  enterid('lex_punct               ', konstant,  ints,   lex_Punct);
  enterid('lex_print               ', konstant,  ints,   lex_Print);
  enterid('lex_graph               ', konstant,  ints,   lex_Graph);
  enterid('lex_ascii               ', konstant,  ints,   lex_Ascii);
  enterid('lex_iparam              ', konstant,  ints,   lex_iParam);
  enterid('lex_fparam              ', konstant,  ints,   lex_fParam);
  enterid('lex_sparam              ', konstant,  ints,   lex_sParam);
  enterid('lex_base64              ', konstant,  ints,   lex_Base64);
  enterid('lex_fsmname             ', konstant,  ints,   lex_FsmName);
  enterid('lex_sminame             ', konstant,  ints,   lex_SmiName);
  enterid('lex_dimname             ', konstant,  ints,   lex_DimName);
  enterid('lex_sqlname             ', konstant,  ints,   lex_SqlName);
  enterid('lex_fbdname             ', konstant,  ints,   lex_FbdName);
  enterid('lex_section             ', konstant,  ints,   lex_Section);
  enterid('lex_atcall              ', konstant,  ints,   lex_AtCall);
  enterid('lex_atcmnd              ', konstant,  ints,   lex_AtCmnd);
  enterid('lex_lquote              ', konstant,  ints,   lex_lquote);
  enterid('lex_rquote              ', konstant,  ints,   lex_rquote);
  enterid('lex_quotes              ', konstant,  ints,   lex_quotes);
  enterid('lex_quoted              ', konstant,  ints,   lex_quoted);
  enterid('lex_domname             ', konstant,  ints,   lex_domname);
  enterid('lex_dnshost             ', konstant,  ints,   lex_dnshost);
  enterid('lex_adduser             ', konstant,  ints,   lex_adduser);
  enterid('lex_domuser             ', konstant,  ints,   lex_domuser);
  enterid('lex_usrhost             ', konstant,  ints,   lex_usrhost);
  enterid('lex_ip4addr             ', konstant,  ints,   lex_ip4addr);
  enterid('lex_ip6addr             ', konstant,  ints,   lex_ip6addr);
  enterid('lex_macaddr             ', konstant,  ints,   lex_macaddr);
  enterid('lex_dotname             ', konstant,  ints,   lex_dotname);
  enterid('lex_uriaddr             ', konstant,  ints,   lex_uriaddr);
  enterid('lex_pctchar             ', konstant,  ints,   lex_pctchar);
  enterid('lex_pctcode             ', konstant,  ints,   lex_pctcode);
  enterid('lex_pctdata             ', konstant,  ints,   lex_pctdata);
  enterid('lex_pctneed             ', konstant,  ints,   lex_pctneed);
  enterid('regexp_def              ', konstant,  ints,   regexp_def);
  enterid('regexp_pas              ', konstant,  ints,   regexp_pas);
  enterid('regexp_vbs              ', konstant,  ints,   regexp_vbs);
  enterid('regexp_pcrelib          ', konstant,  ints,   regexp_pcrelib);
  enterid('clnone                  ', konstant,  ints,   clNone);
  enterid('clblack                 ', konstant,  ints,   clBlack);
  enterid('clmaroon                ', konstant,  ints,   clMaroon);
  enterid('clgreen                 ', konstant,  ints,   clGreen);
  enterid('clolive                 ', konstant,  ints,   clOlive);
  enterid('clnavy                  ', konstant,  ints,   clNavy);
  enterid('clpurple                ', konstant,  ints,   clPurple);
  enterid('clteal                  ', konstant,  ints,   clTeal);
  enterid('clgray                  ', konstant,  ints,   clGray);
  enterid('clsilver                ', konstant,  ints,   clSilver);
  enterid('clred                   ', konstant,  ints,   clRed);
  enterid('cllime                  ', konstant,  ints,   clLime);
  enterid('clyellow                ', konstant,  ints,   clYellow);
  enterid('clblue                  ', konstant,  ints,   clBlue);
  enterid('clfuchsia               ', konstant,  ints,   clFuchsia);
  enterid('claqua                  ', konstant,  ints,   clAqua);
  enterid('clwhite                 ', konstant,  ints,   clWhite);
  enterid('fsm_type_nil            ', konstant,  ints,   fsm_type_nil);
  enterid('fsm_type_int            ', konstant,  ints,   fsm_type_int);
  enterid('fsm_type_float          ', konstant,  ints,   fsm_type_float);
  enterid('fsm_type_string         ', konstant,  ints,   fsm_type_string);
  enterid('fsm_type_parameter      ', konstant,  ints,   fsm_type_parameter);
  enterid('fsm_type_manager        ', konstant,  ints,   fsm_type_manager);
  enterid('fsm_type_domain         ', konstant,  ints,   fsm_type_domain);
  enterid('fsm_type_class          ', konstant,  ints,   fsm_type_class);
  enterid('fsm_type_object         ', konstant,  ints,   fsm_type_object);
  enterid('fsm_type_state          ', konstant,  ints,   fsm_type_state);
  enterid('fsm_type_action         ', konstant,  ints,   fsm_type_action);
  enterid('fsm_type_objectset      ', konstant,  ints,   fsm_type_objectset);
  enterid('fsm_type_function       ', konstant,  ints,   fsm_type_function);
  enterid('fsm_type_parent         ', konstant,  ints,   fsm_type_parent);
  enterid('fsm_type_entity         ', konstant,  ints,   fsm_type_entity);
  enterid('fsm_type_any            ', konstant,  ints,   fsm_type_any);
  enterid('db_type_nil             ', konstant,  ints,   db_type_nil);
  enterid('db_type_connection      ', konstant,  ints,   db_type_connection);
  enterid('db_type_recordset       ', konstant,  ints,   db_type_recordset);
  enterid('db_type_command         ', konstant,  ints,   db_type_command);
  enterid('adstateclosed           ', konstant,  ints,   adStateClosed);
  enterid('adstateopen             ', konstant,  ints,   adStateOpen);
  enterid('adstateconnecting       ', konstant,  ints,   adStateConnecting);
  enterid('adstateexecuting        ', konstant,  ints,   adStateExecuting);
  enterid('adstatefetching         ', konstant,  ints,   adStateFetching);
  enterid('adcmdtext               ', konstant,  ints,   adCmdText);
  enterid('adcmdtable              ', konstant,  ints,   adCmdTable);
  enterid('adcmdstoredproc         ', konstant,  ints,   adCmdStoredProc);
  enterid('adcmdunknown            ', konstant,  ints,   adCmdUnknown);
  enterid('adasyncexecute          ', konstant,  ints,   adAsyncExecute);
  enterid('adasyncfetch            ', konstant,  ints,   adAsyncFetch);
  enterid('adexecutenorecords      ', konstant,  ints,   adExecuteNoRecords);
  enterid('adconnectunspecified    ', konstant,  ints,   adConnectUnspecified);
  enterid('adasyncconnect          ', konstant,  ints,   adAsyncConnect);
  enterid('shm_err_ok              ', konstant,  ints,   shm_err_ok);
  enterid('shm_err_op              ', konstant,  ints,   shm_err_op);
  enterid('shm_err_ref             ', konstant,  ints,   shm_err_ref);
  enterid('shm_err_arg             ', konstant,  ints,   shm_err_arg);
  enterid('shm_err_bug             ', konstant,  ints,   shm_err_bug);
  enterid('isunix                  ', konstant,  bools,  Ord(IsUnix));
  enterid('islinux                 ', konstant,  bools,  Ord(IsLinux));
  enterid('iswindows               ', konstant,  bools,  Ord(IsWindows));
  enterid('cpubitness              ', konstant,  ints,   CpuBitness);
  enterid('sizeofpointer           ', konstant,  ints,   SizeOf(Pointer));
  enterid('defaultsystemcodepage   ', konstant,  ints,   DefaultSystemCodePage);
  enterid('cp_utf8                 ', konstant,  ints,   CP_UTF8);
  enterid('cp_none                 ', konstant,  ints,   CP_NONE);
  enterid('cp_1251                 ', konstant,  ints,   CP_1251);
  enterid('cp_866                  ', konstant,  ints,   CP_866);
  enterid('afnm_trim               ', konstant,  ints,   afnm_trim);
  enterid('afnm_delim              ', konstant,  ints,   afnm_delim);
  enterid('afnm_drive              ', konstant,  ints,   afnm_drive);
  enterid('afnm_lower              ', konstant,  ints,   afnm_lower);
  enterid('afnm_utf8               ', konstant,  ints,   afnm_utf8);
  enterid('afnm_nodup              ', konstant,  ints,   afnm_nodup);
  enterid('db_engine_ado           ', konstant,  ints,   db_engine_ado);
  enterid('db_engine_sqldb         ', konstant,  ints,   db_engine_sqldb);
  enterid('db_engine_zeos          ', konstant,  ints,   db_engine_zeos);
  enterid('sev_debug               ', konstant,  ints,   sev_debug);
  enterid('sev_info                ', konstant,  ints,   sev_info);
  enterid('sev_warn                ', konstant,  ints,   sev_warn);
  enterid('sev_error               ', konstant,  ints,   sev_error);
  enterid('sev_fatal               ', konstant,  ints,   sev_fatal);
  enterid('sev_off                 ', konstant,  ints,   sev_off);
  enterid('sev_remark              ', konstant,  ints,   sev_remark);
  enterid('sev_comment             ', konstant,  ints,   sev_comment);
  enterid('sev_notice              ', konstant,  ints,   sev_notice);
  enterid('sev_mark                ', konstant,  ints,   sev_mark);
  enterid('sev_details             ', konstant,  ints,   sev_details);
  enterid('sev_viewexp             ', konstant,  ints,   sev_viewexp);
  enterid('sev_viewimp             ', konstant,  ints,   sev_viewimp);
  enterid('sev_report              ', konstant,  ints,   sev_report);
  enterid('sev_attention           ', konstant,  ints,   sev_attention);
  enterid('sev_voice               ', konstant,  ints,   sev_voice);
  enterid('sev_tooltip             ', konstant,  ints,   sev_tooltip);
  enterid('sev_input               ', konstant,  ints,   sev_input);
  enterid('sev_print               ', konstant,  ints,   sev_print);
  enterid('sev_notify              ', konstant,  ints,   sev_notify);
  enterid('sev_success             ', konstant,  ints,   sev_success);
  enterid('sev_succeed             ', konstant,  ints,   sev_succeed);
  enterid('sev_great               ', konstant,  ints,   sev_great);
  enterid('sev_perfect             ', konstant,  ints,   sev_perfect);
  enterid('sev_worry               ', konstant,  ints,   sev_worry);
  enterid('sev_hazard              ', konstant,  ints,   sev_hazard);
  enterid('sev_disturb             ', konstant,  ints,   sev_disturb);
  enterid('sev_hardship            ', konstant,  ints,   sev_hardship);
  enterid('sev_problem             ', konstant,  ints,   sev_problem);
  enterid('sev_mistimes            ', konstant,  ints,   sev_mistimes);
  enterid('sev_exception           ', konstant,  ints,   sev_exception);
  enterid('sev_watchdog            ', konstant,  ints,   sev_watchdog);
  enterid('sev_alert               ', konstant,  ints,   sev_alert);
  enterid('sev_bug                 ', konstant,  ints,   sev_bug);
  enterid('sev_glitch              ', konstant,  ints,   sev_glitch);
  enterid('sev_fail                ', konstant,  ints,   sev_fail);
  enterid('sev_fault               ', konstant,  ints,   sev_fault);
  enterid('sev_trouble             ', konstant,  ints,   sev_trouble);
  enterid('sev_alarm               ', konstant,  ints,   sev_alarm);
  enterid('sev_danger              ', konstant,  ints,   sev_danger);
  enterid('sev_siren               ', konstant,  ints,   sev_siren);
  enterid('sev_critical            ', konstant,  ints,   sev_critical);
  enterid('sev_failure             ', konstant,  ints,   sev_failure);
  enterid('sev_abort               ', konstant,  ints,   sev_abort);
  enterid('sev_crash               ', konstant,  ints,   sev_crash);
  enterid('sev_misfortune          ', konstant,  ints,   sev_misfortune);
  enterid('sev_emergency           ', konstant,  ints,   sev_emergency);
  enterid('sev_accident            ', konstant,  ints,   sev_accident);
  enterid('sev_catastrophe         ', konstant,  ints,   sev_catastrophe);
  enterid('sev_disaster            ', konstant,  ints,   sev_disaster);
  enterid('sev_doomsday            ', konstant,  ints,   sev_doomsday);
  // types
  enterid('real                    ', type1,     reals,  1);
  enterid('char                    ', type1,     chars,  1);
  enterid('boolean                 ', type1,     bools,  1);
  enterid('integer                 ', type1,     ints ,  1);
  enterid('string                  ', type1,     strngs, 1);
  // functions
  enterid('abs                     ', funktion,  reals,  0);
  enterid('sqr                     ', funktion,  reals,  2);
  enterid('odd                     ', funktion,  bools,  4);
  enterid('chr                     ', funktion,  chars,  5);
  enterid('ord                     ', funktion,  ints,   6);
  enterid('succ                    ', funktion,  chars,  7);
  enterid('pred                    ', funktion,  chars,  8);
  enterid('round                   ', funktion,  ints,   9);
  enterid('trunc                   ', funktion,  ints,   10);
  enterid('sin                     ', funktion,  reals,  11);
  enterid('cos                     ', funktion,  reals,  12);
  enterid('exp                     ', funktion,  reals,  13);
  enterid('ln                      ', funktion,  reals,  14);
  enterid('sqrt                    ', funktion,  reals,  15);
  enterid('arctan                  ', funktion,  reals,  16);
  enterid('eof                     ', funktion,  bools,  17);
  enterid('eoln                    ', funktion,  bools,  18);
  enterid('maxavail                ', funktion,  ints,   19);
  enterid('length                  ', funktion,  ints,   20);
  enterid('copy                    ', funktion,  strngs, 23);
  enterid('pos                     ', funktion,  ints,   26);
  enterid('str                     ', funktion,  strngs, 33);
  enterid('val                     ', funktion,  ints,   35);
  enterid('rval                    ', funktion,  reals,  37);
  {
  init DAQ standard functions
  }
  enterid('rand                    ', funktion,  reals,  39);
  enterid('msecnow                 ', funktion,  reals,  40);
  enterid('secnow                  ', funktion,  reals,  41);
  enterid('getticks                ', funktion,  reals,  42);
  enterid('time                    ', funktion,  reals,  43);
  enterid('timeunits               ', funktion,  reals,  44);
  enterid('numais                  ', funktion,  ints,   45);
  enterid('numdis                  ', funktion,  ints,   46);
  enterid('numaos                  ', funktion,  ints,   47);
  enterid('numdos                  ', funktion,  ints,   48);
  enterid('numcals                 ', funktion,  ints,   49);
  enterid('pi                      ', funktion,  reals,  50);
  enterid('deg                     ', funktion,  reals,  51);
  enterid('rad                     ', funktion,  reals,  52);
  enterid('tan                     ', funktion,  reals,  53);
  enterid('asin                    ', funktion,  reals,  54);
  enterid('acos                    ', funktion,  reals,  55);
  enterid('atan                    ', funktion,  reals,  56);
  enterid('sinh                    ', funktion,  reals,  57);
  enterid('cosh                    ', funktion,  reals,  58);
  enterid('tanh                    ', funktion,  reals,  59);
  enterid('gamma                   ', funktion,  reals,  60);
  enterid('isnan                   ', funktion,  bools,  61);
  enterid('isinf                   ', funktion,  bools,  62);
  enterid('sign                    ', funktion,  ints,   63);
  enterid('int                     ', funktion,  reals,  64);
  enterid('frac                    ', funktion,  reals,  65);
  enterid('floor                   ', funktion,  reals,  66);
  enterid('ceil                    ', funktion,  reals,  67);
  enterid('log                     ', funktion,  reals,  68);
  enterid('hypot                   ', funktion,  reals,  69);
  enterid('random                  ', funktion,  reals,  70);
  enterid('max                     ', funktion,  reals,  71);
  enterid('rmax                    ', funktion,  reals,  71); // recommended synonym
  enterid('min                     ', funktion,  reals,  72);
  enterid('rmin                    ', funktion,  reals,  72); // recommended synonym
  enterid('eq                      ', funktion,  bools,  73);
  enterid('ne                      ', funktion,  bools,  74);
  enterid('lt                      ', funktion,  bools,  75);
  enterid('gt                      ', funktion,  bools,  76);
  enterid('le                      ', funktion,  bools,  77);
  enterid('ge                      ', funktion,  bools,  78);
  enterid('bitor                   ', funktion,  reals,  79);
  enterid('bitxor                  ', funktion,  reals,  80);
  enterid('bitand                  ', funktion,  reals,  81);
  enterid('refai                   ', funktion,  ints,   82);
  enterid('refdi                   ', funktion,  ints,   83);
  enterid('refao                   ', funktion,  ints,   84);
  enterid('refdo                   ', funktion,  ints,   85);
  enterid('refcalibr               ', funktion,  ints,   86);
  enterid('getai_n                 ', funktion,  reals,  87);
  enterid('getdi_n                 ', funktion,  reals,  88);
  enterid('crvlen                  ', funktion,  reals,  89);
  enterid('crvx                    ', funktion,  reals,  90);
  enterid('crvy                    ', funktion,  reals,  91);
  enterid('getai_xn                ', funktion,  reals,  92);
  enterid('getai_yn                ', funktion,  reals,  93);
  enterid('getdi_xn                ', funktion,  reals,  94);
  enterid('getdi_yn                ', funktion,  reals,  95);
  enterid('getai                   ', funktion,  reals,  96);
  enterid('getdi                   ', funktion,  reals,  97);
  enterid('getai_xi                ', funktion,  reals,  98);
  enterid('getai_yi                ', funktion,  reals,  99);
  enterid('getdi_xi                ', funktion,  reals, 100);
  enterid('getdi_yi                ', funktion,  reals, 101);
  enterid('crvput                  ', funktion,  bools, 102);
  enterid('crvinteg                ', funktion,  reals, 103);
  enterid('putev                   ', funktion,  bools, 104);
  enterid('putao                   ', funktion,  bools, 105);
  enterid('putdo                   ', funktion,  bools, 106);
  enterid('calibr                  ', funktion,  reals, 107);
  enterid('power                   ', funktion,  reals, 108);
  enterid('bitnot                  ', funktion,  reals, 109);
  enterid('getbitmask              ', funktion,  reals, 110);
  enterid('fixerror                ', funktion,  bools, 111);
  enterid('inportb                 ', funktion,  ints,  112);
  enterid('outportb                ', funktion,  ints,  113);
  enterid('isbit                   ', funktion,  bools, 114);
  enterid('reset                   ', funktion,  ints,  115);
  enterid('rewrite                 ', funktion,  ints,  118);
  enterid('append                  ', funktion,  ints,  121);
  enterid('ioresult                ', funktion,  ints,  124);
  enterid('fileexists              ', funktion,  bools, 125);
  enterid('runcount                ', funktion,  reals, 128);
  enterid('voice                   ', funktion,  bools, 129);
  enterid('action                  ', funktion,  bools, 132);
  enterid('clear                   ', funktion,  bools, 135);
  enterid('cleardevice             ', funktion,  bools, 138);
  enterid('start                   ', funktion,  bools, 141);
  enterid('stop                    ', funktion,  bools, 144);
  enterid('devmsg                  ', funktion,  reals, 147); // obsolete, keep for backward capability
  enterid('devsendmsg              ', funktion,  reals, 147); // recommended synonym
  enterid('debugout                ', funktion,  bools, 150);
  enterid('clearcurve              ', funktion,  bools, 153);
  enterid('savecrw                 ', funktion,  bools, 156);
  enterid('specmarker              ', funktion,  reals, 159);
  enterid('specmarkerl             ', funktion,  reals, 162);
  enterid('specmarkerr             ', funktion,  reals, 165);
  enterid('specroil                ', funktion,  reals, 168);
  enterid('specroir                ', funktion,  reals, 171);
  enterid('windraw                 ', funktion,  bools, 174);
  enterid('winshow                 ', funktion,  bools, 177);
  enterid('winhide                 ', funktion,  bools, 180);
  enterid('winselect               ', funktion,  bools, 183);
  enterid('global                  ', funktion,  reals, 186);
  enterid('tm_new                  ', funktion,  ints,  189);
  enterid('tm_free                 ', funktion,  bools, 190);
  enterid('tm_gettime              ', funktion,  reals, 191);
  enterid('tm_start                ', funktion,  bools, 192);
  enterid('tm_stop                 ', funktion,  bools, 193);
  enterid('tm_isstart              ', funktion,  bools, 194);
  enterid('tm_event                ', funktion,  bools, 195);
  enterid('tm_curint               ', funktion,  ints,  196);
  enterid('tm_numint               ', funktion,  ints,  197);
  enterid('tm_addint               ', funktion,  bools, 198);
  enterid('tm_getint               ', funktion,  reals, 199);
  enterid('tm_setint               ', funktion,  bools, 200);
  enterid('_nan                    ', funktion,  reals, 201);
  enterid('_plusinf                ', funktion,  reals, 202); // obsolete, keep for backward capability
  enterid('_inf                    ', funktion,  reals, 202); // recommended synonym
  enterid('_minusinf               ', funktion,  reals, 203);
  enterid('_nil                    ', funktion,  ints,  204);
  enterid('macheps                 ', funktion,  reals, 205);
  enterid('maxint                  ', funktion,  ints,  206);
  enterid('comopen                 ', funktion,  bools, 207);
  enterid('comclose                ', funktion,  bools, 210);
  enterid('comcount                ', funktion,  ints,  211);
  enterid('comwrite                ', funktion,  bools, 212);
  enterid('comread                 ', funktion,  strngs,215);
  enterid('aimap                   ', funktion,  reals, 216);
  enterid('dimap                   ', funktion,  reals, 217);
  enterid('aomap                   ', funktion,  reals, 218);
  enterid('domap                   ', funktion,  reals, 219);
  enterid('diword                  ', funktion,  reals, 220);
  enterid('irqinit                 ', funktion,  bools, 221);
  enterid('irqfree                 ', funktion,  bools, 222);
  enterid('isisrnow                ', funktion,  bools, 223);
  enterid('readini                 ', funktion,  strngs,224);
  enterid('caminitpkk              ', funktion,  bools, 227);
  enterid('camdonepkk              ', funktion,  bools, 228);
  enterid('camtypepkk              ', funktion,  ints,  229);
  enterid('camnaf                  ', funktion,  bools, 230);
  enterid('camwrite                ', funktion,  bools, 231);
  enterid('camwritelong            ', funktion,  bools, 232);
  enterid('camread                 ', funktion,  ints,  233);
  enterid('camreadlong             ', funktion,  reals, 234);
  enterid('camdeclare              ', funktion,  bools, 235);
  enterid('camzero                 ', funktion,  bools, 236);
  enterid('camclear                ', funktion,  bools, 237);
  enterid('camsetinhib             ', funktion,  bools, 238);
  enterid('camclrinhib             ', funktion,  bools, 239);
  enterid('camsetlmask             ', funktion,  bools, 240);
  enterid('camsetrmask             ', funktion,  bools, 241);
  enterid('camgetlmask             ', funktion,  ints,  242);
  enterid('camgetrword             ', funktion,  ints,  243);
  enterid('camgetbcard             ', funktion,  ints,  244);
  enterid('camisx                  ', funktion,  bools, 245);
  enterid('camisxq                 ', funktion,  bools, 246);
  enterid('camgetcsr               ', funktion,  ints,  247);
  enterid('camenabint              ', funktion,  bools, 248);
  enterid('camdisabint             ', funktion,  bools, 249);
  enterid('camismyreq              ', funktion,  bools, 250);
  enterid('camgetrcode             ', funktion,  ints,  251);
  enterid('camclrreq               ', funktion,  bools, 252);
  enterid('camsaveregs             ', funktion,  bools, 253);
  enterid('camrestregs             ', funktion,  bools, 254);
  enterid('cammaskofst             ', funktion,  ints,  255);
  enterid('camstoflam              ', funktion,  ints,  256);
  enterid('camgetrmask             ', funktion,  ints,  257);
  enterid('pkkclrreq               ', funktion,  bools, 258);
  enterid('pkkenabint              ', funktion,  bools, 259);
  enterid('pkkdisabint             ', funktion,  bools, 260);
  enterid('findtag                 ', funktion,  ints,  261);
  enterid('inittag                 ', funktion,  ints,  264); // obsolete, keep for backward capability
  enterid('createtag               ', funktion,  ints,  264); // recommended synonym
  enterid('freetag                 ', funktion,  bools, 267);
  enterid('typetag                 ', funktion,  ints,  268);
  enterid('nametag                 ', funktion,  strngs,269);
  enterid('igettag                 ', funktion,  ints,  270);
  enterid('rgettag                 ', funktion,  reals, 271);
  enterid('sgettag                 ', funktion,  strngs,272);
  enterid('isettag                 ', funktion,  bools, 273);
  enterid('rsettag                 ', funktion,  bools, 274);
  enterid('ssettag                 ', funktion,  bools, 275);
  enterid('clickbutton             ', funktion,  ints,  278);
  enterid('clicksensor             ', funktion,  strngs,279);
  enterid('clicktag                ', funktion,  ints,  280);
  enterid('crvfind                 ', funktion,  reals, 281);
  enterid('crvget                  ', funktion,  reals, 284);
  enterid('crvwhere                ', funktion,  reals, 285);
  enterid('ms2year                 ', funktion,  ints,  286);
  enterid('ms2month                ', funktion,  ints,  287);
  enterid('ms2day                  ', funktion,  ints,  288);
  enterid('ms2hour                 ', funktion,  ints,  289);
  enterid('ms2min                  ', funktion,  ints,  290);
  enterid('ms2sec                  ', funktion,  ints,  291);
  enterid('fileerase               ', funktion,  bools, 292);
  enterid('filerename              ', funktion,  bools, 295);
  enterid('filecopy                ', funktion,  bools, 298);
  enterid('inot                    ', funktion,  ints,  301);
  enterid('ior                     ', funktion,  ints,  302);
  enterid('ixor                    ', funktion,  ints,  303);
  enterid('iand                    ', funktion,  ints,  304);
  enterid('ishift                  ', funktion,  ints,  305);
  enterid('rshift                  ', funktion,  reals, 306);
  enterid('upcasestr               ', funktion,  strngs,307);
  enterid('locasestr               ', funktion,  strngs,310);
  enterid('worddelims              ', funktion,  strngs,313);
  enterid('wordcount               ', funktion,  ints,  316);
  enterid('extractword             ', funktion,  strngs,319);
  enterid('getfattr                ', funktion,  ints,  322);
  enterid('setfattr                ', funktion,  ints,  325);
  enterid('doserror                ', funktion,  ints,  328);
  enterid('daqdllinit              ', funktion,  ints,  329);
  enterid('daqdllfree              ', funktion,  bools, 332);
  enterid('daqdllcall              ', funktion,  bools, 333);
  enterid('paramstr                ', funktion,  strngs,334);
  enterid('registererr             ', funktion,  ints,  337);
  enterid('geterrcount             ', funktion,  reals, 340);
  enterid('adam_get                ', funktion,  strngs,341);
  enterid('adam_request            ', funktion,  bools, 344);
  enterid('adam_status             ', funktion,  ints,  347);
  enterid('adam_reqtime            ', funktion,  reals, 348);
  enterid('hexb                    ', funktion,  strngs,349);
  enterid('hexw                    ', funktion,  strngs,350);
  enterid('hexl                    ', funktion,  strngs,351);
  enterid('strfix                  ', funktion,  strngs,352);
  enterid('f_reset                 ', funktion,  ints,  353);
  enterid('f_rewrite               ', funktion,  ints,  356);
  enterid('f_read                  ', funktion,  strngs,359);
  enterid('f_write                 ', funktion,  ints,  360);
  enterid('f_size                  ', funktion,  reals, 363);
  enterid('f_seek                  ', funktion,  reals, 364);
  enterid('f_close                 ', funktion,  bools, 365);
  enterid('dump                    ', funktion,  strngs,366);
  enterid('dump2b                  ', funktion,  bools, 370);
  enterid('dump2c                  ', funktion,  chars, 373);
  enterid('dump2i                  ', funktion,  ints,  376);
  enterid('dump2r                  ', funktion,  reals, 379);
  enterid('mkdir                   ', funktion,  bools, 382);
  enterid('sleep                   ', funktion,  bools, 385);
  enterid('mksecnow                ', funktion,  reals, 386);
  enterid('shellexecute            ', funktion,  ints,  387);
  enterid('crlf                    ', funktion,  strngs,390);
  enterid('progname                ', funktion,  strngs,391);
  enterid('devname                 ', funktion,  strngs,392);
  enterid('crvname                 ', funktion,  strngs,393);
  enterid('crvlock                 ', funktion,  bools, 394);
  enterid('crvunlock               ', funktion,  bools, 395);
  enterid('crvins                  ', funktion,  reals, 396);
  enterid('crvdel                  ', funktion,  reals, 397);
  enterid('inportw                 ', funktion,  ints,  398);
  enterid('outportw                ', funktion,  ints,  399);
  enterid('inportl                 ', funktion,  ints,  400);
  enterid('outportl                ', funktion,  ints,  401);
  enterid('edit                    ', funktion,  strngs,402);
  enterid('editstate               ', funktion,  ints,  405);
  enterid('echo                    ', funktion,  bools, 406);
  enterid('eval                    ', funktion,  reals, 409);
  enterid('evar                    ', funktion,  bools, 412);
  enterid('comspace                ', funktion,  ints,  415);
  enterid('comclear                ', funktion,  bools, 416);
  enterid('add3d                   ', funktion,  bools, 417);
  enterid('plot3d                  ', funktion,  bools, 418);
  enterid('task_init               ', funktion,  ints,  421);
  enterid('task_free               ', funktion,  bools, 424);
  enterid('task_ref                ', funktion,  ints,  425);
  enterid('task_pid                ', funktion,  ints,  426);
  enterid('task_run                ', funktion,  bools, 427);
  enterid('task_wait               ', funktion,  bools, 428);
  enterid('task_send               ', funktion,  ints,  429);
  enterid('task_recv               ', funktion,  strngs,432);
  enterid('task_txcount            ', funktion,  ints,  433);
  enterid('task_rxcount            ', funktion,  ints,  434);
  enterid('task_txspace            ', funktion,  ints,  435);
  enterid('task_rxspace            ', funktion,  ints,  436);
  enterid('task_result             ', funktion,  ints,  437);
  enterid('task_kill               ', funktion,  bools, 438);
  enterid('task_ctrl               ', funktion,  strngs,439);
  enterid('getcomspec              ', funktion,  strngs,442); //  recommended synonym on Windows
  enterid('getshell                ', funktion,  strngs,442); //  recommended synonym on Unix
  enterid('mime_encode             ', funktion,  strngs,443);
  enterid('base64_encode           ', funktion,  strngs,443);
  enterid('mime_decode             ', funktion,  strngs,446);
  enterid('base64_decode           ', funktion,  strngs,446);
  enterid('clickparams             ', funktion,  strngs,449);
  enterid('str2shortcut            ', funktion,  ints,  452);
  enterid('shortcut2str            ', funktion,  strngs,455);
  enterid('crvgetln                ', funktion,  strngs,456);
  enterid('crvputln                ', funktion,  bools, 457);
  enterid('crvinsln                ', funktion,  bools, 460);
  enterid('crvaddln                ', funktion,  bools, 463);
  enterid('crvdelln                ', funktion,  bools, 466);
  enterid('crvnumln                ', funktion,  ints,  467);
  enterid('awakeflag               ', funktion,  bools, 468);
  enterid('reffind                 ', funktion,  ints,  469);
  enterid('refinfo                 ', funktion,  strngs,472);
  enterid('devsend                 ', funktion,  reals, 475);
  enterid('issametext              ', funktion,  bools, 478);
  enterid('defaultextension        ', funktion,  strngs,487);
  enterid('forceextension          ', funktion,  strngs,496);
  enterid('defaultpath             ', funktion,  strngs,505);
  enterid('forcepath               ', funktion,  strngs,514);
  enterid('makerelativepath        ', funktion,  strngs,523);
  enterid('trim                    ', funktion,  strngs,532);
  enterid('trimleft                ', funktion,  strngs,535);
  enterid('trimright               ', funktion,  strngs,538);
  enterid('fexpand                 ', funktion,  strngs,541);
  enterid('addbackslash            ', funktion,  strngs,544); // obsolete, keep for backward capability
  enterid('addpathdelim            ', funktion,  strngs,544); // recommended synonym
  enterid('dropbackslash           ', funktion,  strngs,547); // obsolete, keep for backward capability
  enterid('droppathdelim           ', funktion,  strngs,547); // recommended synonym
  enterid('extractfilepath         ', funktion,  strngs,550);
  enterid('extractfilename         ', funktion,  strngs,553);
  enterid('extractfileext          ', funktion,  strngs,556);
  enterid('iswildcard              ', funktion,  bools, 559);
  enterid('isrelativepath          ', funktion,  bools, 562);
  enterid('hasextension            ', funktion,  bools, 565);
  enterid('direxists               ', funktion,  bools, 568);
  enterid('text_new                ', funktion,  ints,  571);
  enterid('text_free               ', funktion,  bools, 572);
  enterid('text_getln              ', funktion,  strngs,573);
  enterid('text_putln              ', funktion,  bools, 574);
  enterid('text_insln              ', funktion,  bools, 577);
  enterid('text_addln              ', funktion,  bools, 580);
  enterid('text_delln              ', funktion,  bools, 583);
  enterid('text_numln              ', funktion,  ints,  584);
  enterid('readinisection          ', funktion,  ints,  585);
  enterid('getenv                  ', funktion,  strngs,594);
  enterid('setenv                  ', funktion,  bools, 597);
  enterid('vdpm_opcount            ', funktion,  reals, 606);
  enterid('hex_encode              ', funktion,  strngs,607);
  enterid('base16_encode           ', funktion,  strngs,607);
  enterid('hex_decode              ', funktion,  strngs,610);
  enterid('base16_decode           ', funktion,  strngs,610);
  enterid('crypt_ctrl              ', funktion,  strngs,613);
  enterid('crypt_encode            ', funktion,  strngs,616);
  enterid('crypt_decode            ', funktion,  strngs,625);
  enterid('getmd5fromstr           ', funktion,  strngs,634);
  enterid('getmd5fromfile          ', funktion,  strngs,637);
  enterid('getmd5fromtext          ', funktion,  strngs,640);
  enterid('wdt_reset               ', funktion,  reals, 641);
  enterid('guard_check             ', funktion,  ints,  642);
  enterid('dirlist                 ', funktion,  ints,  645);
  enterid('pidlist                 ', funktion,  ints,  654);
  enterid('pidkill                 ', funktion,  ints,  655);
  enterid('enumerate               ', funktion,  ints,  656);
  enterid('timebase                ', funktion,  reals, 659);
  enterid('daqfileref              ', funktion,  strngs,660);
  enterid('getclockres             ', funktion,  reals, 669);
  enterid('setclockres             ', funktion,  reals, 670);
  enterid('url_encode              ', funktion,  strngs,671);
  enterid('url_decode              ', funktion,  strngs,674);
  enterid('datetime2ms             ', funktion,  reals, 677);
  enterid('stackavail              ', funktion,  ints,  678);
  enterid('cpu_count               ', funktion,  ints,  679);
  enterid('cpu_start               ', funktion,  bools, 680);
  enterid('cpu_clock               ', funktion,  reals, 681);
  enterid('cpu_mhz                 ', funktion,  reals, 682);
  enterid('pidaffinity             ', funktion,  ints,  683);
  enterid('devaffinity             ', funktion,  ints,  684);
  enterid('pipe_init               ', funktion,  ints,  685);
  enterid('pipe_free               ', funktion,  bools, 688);
  enterid('pipe_ref                ', funktion,  ints,  689);
  enterid('pipe_pid                ', funktion,  ints,  690);
  enterid('pipe_run                ', funktion,  bools, 691);
  enterid('pipe_wait               ', funktion,  bools, 692);
  enterid('pipe_send               ', funktion,  ints,  693);
  enterid('pipe_recv               ', funktion,  strngs,696);
  enterid('pipe_txcount            ', funktion,  ints,  697);
  enterid('pipe_rxcount            ', funktion,  ints,  698);
  enterid('pipe_txspace            ', funktion,  ints,  699);
  enterid('pipe_rxspace            ', funktion,  ints,  700);
  enterid('pipe_result             ', funktion,  ints,  701);
  enterid('pipe_kill               ', funktion,  bools, 702);
  enterid('pipe_ctrl               ', funktion,  strngs,703);
  enterid('pipe_txclear            ', funktion,  bools, 706);
  enterid('pipe_rxclear            ', funktion,  bools, 707);
  enterid('pipe_count              ', funktion,  ints,  708);
  enterid('pipe_stream             ', funktion,  ints,  709);
  enterid('pipe_connected          ', funktion,  ints,  710);
  enterid('strconv                 ', funktion,  strngs,711);
  enterid('dump2f                  ', funktion,  reals, 720);
  enterid('dumpf                   ', funktion,  strngs,723);
  enterid('url_packed              ', funktion,  strngs,724);
  enterid('nice_encode             ', funktion,  strngs,727);
  enterid('base32_encode           ', funktion,  strngs,727);
  enterid('nice_decode             ', funktion,  strngs,730);
  enterid('base32_decode           ', funktion,  strngs,730);
  enterid('base32_alphabet         ', funktion,  strngs,733);
  enterid('hashindexof             ', funktion,  ints,  736);
  enterid('getpid                  ', funktion,  ints,  739);
  enterid('hashlist_init           ', funktion,  ints,  740);
  enterid('hashlist_free           ', funktion,  bools, 741);
  enterid('hashlist_count          ', funktion,  ints,  742);
  enterid('hashlist_getkey         ', funktion,  strngs,743);
  enterid('hashlist_delete         ', funktion,  bools, 744);
  enterid('hashlist_indexof        ', funktion,  ints,  747);
  enterid('hashlist_getdata        ', funktion,  reals, 750);
  enterid('hashlist_setdata        ', funktion,  bools, 753);
  enterid('hashlist_getlink        ', funktion,  ints,  756);
  enterid('hashlist_setlink        ', funktion,  bools, 759);
  enterid('hashlist_getpara        ', funktion,  strngs,762);
  enterid('hashlist_setpara        ', funktion,  bools, 765);
  enterid('devpost                 ', funktion,  reals, 774);
  enterid('devpostmsg              ', funktion,  reals, 777);
  enterid('clickwrite              ', funktion,  ints,  780);
  enterid('clickread               ', funktion,  ints,  783);
  enterid('clickwhat               ', funktion,  ints,  784);
  enterid('clickwrote              ', funktion,  ints,  785);
  enterid('clickfilter             ', funktion,  ints,  786);
  enterid('clickawaker             ', funktion,  ints,  787);
  enterid('strfetch                ', funktion,  chars, 788);
  enterid('cookiescan              ', funktion,  strngs,791);
  enterid('taglock                 ', funktion,  ints,  800);
  enterid('iatomictagop            ', funktion,  bools, 801);
  enterid('ratomictagop            ', funktion,  bools, 802);
  enterid('stringofchar            ', funktion,  strngs,803);
  enterid('igetdump                ', funktion,  ints,  804);
  enterid('isetdump                ', funktion,  bools, 805);
  enterid('rgetdump                ', funktion,  reals, 806);
  enterid('rsetdump                ', funktion,  bools, 807);
  enterid('getai_par               ', funktion,  reals, 808);
  enterid('getao_par               ', funktion,  reals, 809);
  enterid('getdi_par               ', funktion,  reals, 810);
  enterid('getdo_par               ', funktion,  reals, 811);
  enterid('strfmt                  ', funktion,  strngs,812);
  enterid('utf8_encode_ansi        ', funktion,  strngs,822);
  enterid('utf8_decode_ansi        ', funktion,  strngs,825);
  enterid('utf8_uppercase          ', funktion,  strngs,828);
  enterid('utf8_lowercase          ', funktion,  strngs,831);
  enterid('utf8_length             ', funktion,  ints,  834);
  enterid('utf8_copy               ', funktion,  strngs,837);
  enterid('utf8_ord                ', funktion,  ints,  840);
  enterid('utf8_chr                ', funktion,  strngs,843);
  enterid('utf8_bom                ', funktion,  strngs,844);
  enterid('adjustlinebreaks        ', funktion,  strngs,845);
  enterid('eol                     ', funktion,  strngs,848); // recommended synonym
  enterid('lineending              ', funktion,  strngs,848); // recommended synonym
  enterid('slinebreak              ', funktion,  strngs,848); // recommended synonym
  enterid('directoryseparator      ', funktion,  chars, 849); // recommended synonym
  enterid('pathdelim               ', funktion,  chars, 849); // recommended synonym
  enterid('pathseparator           ', funktion,  chars, 850); // recommended synonym
  enterid('pathsep                 ', funktion,  chars, 850); // recommended synonym
  enterid('easyipc_init            ', funktion,  ints,  851);
  enterid('easyipc_free            ', funktion,  bools, 860);
  enterid('easyipc_poll            ', funktion,  bools, 861);
  enterid('easyipc_send            ', funktion,  bools, 862);
  enterid('easyipc_recv            ', funktion,  strngs,865);
  enterid('easyipc_ctrl            ', funktion,  strngs,866);
  enterid('expenv                  ', funktion,  strngs,869);
  enterid('createtempfile          ', funktion,  strngs,872);
  enterid('stringreplace           ', funktion,  strngs,875);
  enterid('ansiquotedstr           ', funktion,  strngs,902);
  enterid('ansidequotedstr         ', funktion,  strngs,905);
  enterid('ansiskipquotedstr       ', funktion,  strngs,908);
  enterid('posex                   ', funktion,  ints,  911);
  enterid('text_tostring           ', funktion,  strngs,920);
  enterid('imax                    ', funktion,  ints,  921);
  enterid('imin                    ', funktion,  ints,  922);
  enterid('hasflags                ', funktion,  bools, 923);
  enterid('gettagcolor             ', funktion,  ints,  924);
  enterid('settagcolor             ', funktion,  bools, 925);
  enterid('gettagparam             ', funktion,  reals, 926);
  enterid('settagparam             ', funktion,  bools, 927);
  enterid('gettagtimer             ', funktion,  reals, 928);
  enterid('settagtimer             ', funktion,  bools, 929);
  enterid('extractfirstparam       ', funktion,  strngs,930);
  enterid('skipfirstparam          ', funktion,  strngs,933);
  enterid('isoption                ', funktion,  bools, 936);
  enterid('getoptionvalue          ', funktion,  strngs,945);
  enterid('ansiquotedifneed        ', funktion,  strngs,948);
  enterid('islexeme                ', funktion,  bools, 951);
  enterid('regexp_init             ', funktion,  ints,  954);
  enterid('regexp_free             ', funktion,  bools, 957);
  enterid('regexp_ref              ', funktion,  ints,  958);
  enterid('regexp_ctrl             ', funktion,  strngs,959);
  enterid('regexp_test             ', funktion,  bools, 962);
  enterid('regexp_exec             ', funktion,  ints,  965);
  enterid('regexp_replace          ', funktion,  strngs,968);
  enterid('regexp_matchnum         ', funktion,  ints,  977);
  enterid('regexp_matchpos         ', funktion,  ints,  978);
  enterid('regexp_matchlen         ', funktion,  ints,  979);
  enterid('regexp_matchstr         ', funktion,  strngs,980);
  enterid('regexp_escape           ', funktion,  strngs,981);
  enterid('backslash_encode        ', funktion,  strngs,984);
  enterid('backslash_decode        ', funktion,  strngs,987);
  enterid('backslash_encoder_ctrl  ', funktion,  strngs,990);
  enterid('fsm_new                 ', funktion,  ints,  993);
  enterid('fsm_free                ', funktion,  bools, 994);
  enterid('fsm_ref                 ', funktion,  ints,  995);
  enterid('fsm_root                ', funktion,  ints,  996);
  enterid('fsm_type                ', funktion,  ints,  997);
  enterid('fsm_parent              ', funktion,  ints,  998);
  enterid('fsm_name                ', funktion,  strngs,999);
  enterid('fsm_path                ', funktion,  strngs,1000);
  enterid('fsm_ctrl                ', funktion,  strngs,1001);
  enterid('fsm_count               ', funktion,  ints,  1004);
  enterid('fsm_items               ', funktion,  ints,  1005);
  enterid('fsm_get_iparam          ', funktion,  ints,  1006);
  enterid('fsm_set_iparam          ', funktion,  bools, 1007);
  enterid('fsm_get_fparam          ', funktion,  reals, 1008);
  enterid('fsm_set_fparam          ', funktion,  bools, 1009);
  enterid('fsm_get_sparam          ', funktion,  strngs,1010);
  enterid('fsm_set_sparam          ', funktion,  bools, 1011);
  enterid('fsm_add                 ', funktion,  ints,  1014);
  enterid('fsm_find                ', funktion,  ints,  1017);
  enterid('fsm_get_state           ', funktion,  ints,  1020);
  enterid('fsm_set_state           ', funktion,  ints,  1021);
  enterid('fsm_link                ', funktion,  ints,  1025);
  enterid('fsm_modified            ', funktion,  ints,  1028);
  enterid('fsm_name_rule           ', funktion,  ints,  1029);
  enterid('upcase                  ', funktion,  chars, 1030);
  enterid('locase                  ', funktion,  chars, 1031);
  enterid('htonl                   ', funktion,  ints,  1032);
  enterid('ntohl                   ', funktion,  ints,  1033);
  enterid('htons                   ', funktion,  ints,  1034);
  enterid('ntohs                   ', funktion,  ints,  1035);
  enterid('db_create               ', funktion,  bools, 1036);
  enterid('db_connection           ', funktion,  ints,  1039);
  enterid('db_recordset            ', funktion,  ints,  1042);
  enterid('db_command              ', funktion,  ints,  1045);
  enterid('db_free                 ', funktion,  bools, 1048);
  enterid('db_ref                  ', funktion,  ints,  1049);
  enterid('db_root                 ', funktion,  ints,  1050);
  enterid('db_type                 ', funktion,  ints,  1051);
  enterid('db_parent               ', funktion,  ints,  1052);
  enterid('db_state                ', funktion,  ints,  1053);
  enterid('db_close                ', funktion,  bools, 1054);
  enterid('db_open                 ', funktion,  bools, 1055);
  enterid('db_ctrl                 ', funktion,  strngs,1056);
  enterid('db_bugscount            ', funktion,  ints,  1059);
  enterid('db_bugsclear            ', funktion,  ints,  1060);
  enterid('db_errors               ', funktion,  strngs,1061);
  enterid('db_errorscount          ', funktion,  ints,  1062);
  enterid('db_errorsclear          ', funktion,  ints,  1063);
  enterid('db_execute              ', funktion,  ints,  1064);
  enterid('db_cancel               ', funktion,  bools, 1067);
  enterid('db_update               ', funktion,  bools, 1068);
  enterid('db_cancelupdate         ', funktion,  bools, 1069);
  enterid('db_begintrans           ', funktion,  ints,  1070);
  enterid('db_committrans          ', funktion,  bools, 1071);
  enterid('db_rollbacktrans        ', funktion,  bools, 1072);
  enterid('db_bof                  ', funktion,  bools, 1073);
  enterid('db_eof                  ', funktion,  bools, 1074);
  enterid('db_movefirst            ', funktion,  bools, 1075);
  enterid('db_movelast             ', funktion,  bools, 1076);
  enterid('db_movenext             ', funktion,  bools, 1077);
  enterid('db_moveprevious         ', funktion,  bools, 1078);
  enterid('db_fieldscount          ', funktion,  ints,  1079);
  enterid('db_fieldsnames          ', funktion,  strngs,1080);
  enterid('db_fieldstypes          ', funktion,  ints,  1081);
  enterid('db_fieldsasint          ', funktion,  ints,  1084);
  enterid('db_fieldsasfloat        ', funktion,  reals, 1087);
  enterid('db_fieldsasstring       ', funktion,  strngs,1090);
  enterid('db_addnew               ', funktion,  bools, 1099);
  enterid('db_delete               ', funktion,  bools, 1102);
  enterid('db_requery              ', funktion,  bools, 1103);
  enterid('db_resync               ', funktion,  bools, 1104);
  enterid('db_supports             ', funktion,  bools, 1105);
  enterid('db_save                 ', funktion,  bools, 1106);
  enterid('poseol                  ', funktion,  ints,  1109);
  enterid('strtimefmt              ', funktion,  strngs,1112);
  enterid('shm_init                ', funktion,  ints,  1115);
  enterid('shm_ref                 ', funktion,  ints,  1118);
  enterid('shm_free                ', funktion,  bools, 1119);
  enterid('shm_delink              ', funktion,  bools, 1120);
  enterid('shm_iop                 ', funktion,  ints,  1123);
  enterid('shm_rop                 ', funktion,  reals, 1124);
  enterid('shm_fop                 ', funktion,  reals, 1125);
  enterid('shm_sop                 ', funktion,  strngs,1126);
  enterid('shm_ctrl                ', funktion,  strngs,1129);
  enterid('shm_ioresult            ', funktion,  ints,  1132);
  enterid('strtointbase            ', funktion,  ints,  1133);
  enterid('inttostrbase            ', funktion,  strngs,1136);
  enterid('adaptfilename           ', funktion,  strngs,1137);
  enterid('adaptexefilename        ', funktion,  strngs,1140);
  enterid('adaptfilenamemode       ', funktion,  ints,  1143);
  enterid('skipwords               ', funktion,  strngs,1144);
  enterid('phrasecount             ', funktion,  ints,  1147);
  enterid('extractphrase           ', funktion,  strngs,1150);
  enterid('skipphrases             ', funktion,  strngs,1153);
  enterid('phraselisttotextlines   ', funktion,  strngs,1156);
  enterid('phrasequotes            ', funktion,  strngs,1159);
  enterid('adaptdllfilename        ', funktion,  strngs,1162);
  enterid('adaptlnkfilename        ', funktion,  strngs,1165);
  enterid('db_engineid             ', funktion,  ints,  1168);
  enterid('db_active               ', funktion,  bools, 1169);
  enterid('pct_encode              ', funktion,  strngs,1170);
  enterid('pct_decode              ', funktion,  strngs,1173);
  enterid('pct_encoder_ctrl        ', funktion,  strngs,1176);
  enterid('percent_encode          ', funktion,  strngs,1170); // synonym
  enterid('percent_decode          ', funktion,  strngs,1173); // synonym
  enterid('percent_encoder_ctrl    ', funktion,  strngs,1176); // synonym
  enterid('readfiletobuffer        ', funktion,  strngs,1179);
  enterid('writebuffertofile       ', funktion,  ints,  1182);
  enterid('getfileproperties       ', funktion,  strngs,1191);
  enterid('syslognotable           ', funktion,  bools, 1200);
  enterid('syslognote              ', funktion,  ints,  1201);
  enterid('wdt_timeout             ', funktion,  ints,  1204);
  {next is 1205}
  {
  standard procedures
  }
  enterid('read                    ', prozedure, notyp,  1);
  enterid('readln                  ', prozedure, notyp,  2);
  enterid('write                   ', prozedure, notyp,  3);
  enterid('writeln                 ', prozedure, notyp,  4);
  enterid('                        ', prozedure, notyp,  0);
  {
  init block table
  }
  with btab[1] do begin
   last := it;
   lastpar := 1;
   psize := 0;
   vsize := 0;
  end ;
  {
  compile main begin..end. block
  }
  CompileBlock(blockbegsys+statbegsys, false, 1);
  if sy <> period then FatalError(22);
  {
  last P-code must be HALT
  }
  emit(131);
  {
  Check stack size
  }
  if stacksize-btab[2].vsize+1<dtabmin then FatalError(20);
  {
  Check string table space
  }
  if stab.space<stabmin then FatalError(69);
  {
  if need, print compiler tables
  }
  if prtables then printtables;
  {
  Ok!
  }
  CompTime:=mSecNow-CompTime;
  write(LstOut,EOL,'>>>>>  Closed[',PasTop,']: ',PasSrc,EOL);
  write(LstOut,'COMPILED: OK. Compile time: ',CompTime:1:0,' ms.',EOL);
  for i:=Low(PasStack) to High(PasStack) do SmartFileClose(PasStack[i].Inp);
  SmartFileClose(LstOut);
  CompileOk:=(IOResult=0);
 except
  {
  Compilation error detected
  }
  ErrorCode:=ExceptResult;
  case ExceptResult of
   0  : ErrorMessage:='Undefined identifier';
   1  : ErrorMessage:='Attempt at multiple definition';
   2  : ErrorMessage:='Identifier expected';
   3  : ErrorMessage:='PROGRAM expected';
   4  : ErrorMessage:=') expected';
   5  : ErrorMessage:=': expected';
   6  : ErrorMessage:='Syntax error';
   7  : ErrorMessage:='Identifier or VAR expected';
   8  : ErrorMessage:='OF expected';
   9  : ErrorMessage:='( expected';
   10 : ErrorMessage:='TYPE expected';
   11 : ErrorMessage:='[ expected';
   12 : ErrorMessage:='] expected';
   13 : ErrorMessage:='.. expected';
   14 : ErrorMessage:='; expected';
   15 : ErrorMessage:='Error in function result type';
   16 : ErrorMessage:='= expected';
   17 : ErrorMessage:='Expression must have BOOLEAN result';
   18 : ErrorMessage:='Invalid type for control variable';
   19 : ErrorMessage:='Type conflict between control variable and expression';
   20 : ErrorMessage:='Compiler table overflow - data segment';
   21 : ErrorMessage:='Number too large';
   22 : ErrorMessage:='. expected';
   23 : ErrorMessage:='Invalid expression type following CASE';
   24 : ErrorMessage:='Illegal character';
   25 : ErrorMessage:='Identifier of type CONST expected';
   26 : ErrorMessage:='Array index type conflict';
   27 : ErrorMessage:='Type conflict in declaration of array index bounds';
   28 : ErrorMessage:='No such array';
   29 : ErrorMessage:='TYPE identifier expected';
   30 : ErrorMessage:='Undefined type';
   31 : ErrorMessage:='No such record';
   32 : ErrorMessage:='BOOLEAN type expected';
   33 : ErrorMessage:='Arithmetic type expected';
   34 : ErrorMessage:='INTEGER type expected';
   35 : ErrorMessage:='Incompatible operands of relation operator';
   36 : ErrorMessage:='Actual/formal parameter type conflict';
   37 : ErrorMessage:='Variable identifier expected';
   38 : ErrorMessage:='String expected';
   39 : ErrorMessage:='Wrong number of actual parameters';
   40 : ErrorMessage:='Digit expected';
   41 : ErrorMessage:='Incorrect type';
   42 : ErrorMessage:='Type REAL expected';
   43 : ErrorMessage:='Integer expected';
   44 : ErrorMessage:='Variable or constant expected';
   45 : ErrorMessage:='Variable or procedure identifier expected';
   46 : ErrorMessage:='Incompatible operands of :=';
   47 : ErrorMessage:='Label type incompatible with selecting expression';
   48 : ErrorMessage:='Incorrect parameter type';
   49 : ErrorMessage:='Store overflow';
   50 : ErrorMessage:='Constant expected';
   51 : ErrorMessage:=':= expected';
   52 : ErrorMessage:='THEN expected';
   53 : ErrorMessage:='UNTIL expected';
   54 : ErrorMessage:='DO expected';
   55 : ErrorMessage:='TO or DOWNTO expected';
   56 : ErrorMessage:='BEGIN expected';
   57 : ErrorMessage:='END expected';
   58 : ErrorMessage:='Factor expected';
   59 : ErrorMessage:=', expected';
   60 : ErrorMessage:='Indexed string not allowed here';
   61 : ErrorMessage:='File '+PasSrc+' not found or corrupted';
   62 : ErrorMessage:='Can''t open file '+LstFile;
   63 : ErrorMessage:='Compiler table overflow - identifiers';
   64 : ErrorMessage:='Compiler table overflow - procedures';
   65 : ErrorMessage:='Compiler table overflow - reals';
   66 : ErrorMessage:='Compiler table overflow - arrays';
   67 : ErrorMessage:='Compiler table overflow - levels';
   68 : ErrorMessage:='Compiler table overflow - code segment';
   69 : ErrorMessage:='Compiler table overflow - strings';
   70 : ErrorMessage:='Compiler table overflow - input line';
   71 : ErrorMessage:='Program incomplete';
   72 : ErrorMessage:='Out of memory';
   73 : ErrorMessage:='Internal compiler error';
   74 : ErrorMessage:='Include file stack underflow';
   75 : ErrorMessage:='Include file stack overflow';
   76 : ErrorMessage:='CHAR type expected';
   77 : ErrorMessage:='STRING variable expected';
   78 : ErrorMessage:='Hexadecimal constant expected';
   79 : ErrorMessage:='Octal constant expected';
   80 : ErrorMessage:='Binary constant expected';
   81 : ErrorMessage:='File '+PasSrc+' has corrupted CRLF delimiter';
   else ErrorMessage:='Unknown error or bug in compiler';
  end;
  if not IsFileClosed(LstOut) then begin
   writeln(LstOut);
   if cc>0 then writeln(LstOut,'*****',' ': GetChColUtf8(PasStack[PasTop].line,cc), '^ - "',ErrorMessage, '"');
   writeln(LstOut,Format('Compile error %d detected : "%s"',[ExceptResult,ErrorMessage]));
   for i:=Low(PasStack) to PasTop do with PasStack[i] do
   writeln(LstOut,Format('Error found at line %5d, column %3d, in file[%d]: %s',[iln,GetChColUtf8(line,cc),i,Src]));
   with PasStack[PasTop] do
   if SysLogNotable(SeverityOfDaqCompiler)
   then SysLogNote(0,SeverityOfDaqCompiler,TheDevice.SysLogSign,
        Format('Compile error %d "%s" at line %d column %d in file "%s".',
        [ExceptResult,ErrorMessage,iln,GetChColUtf8(line,cc),Src]));
  end;
  for i:=Low(PasStack) to PasTop do with PasStack[i] do begin
   eline:=Format('%d, %d, %s',[iln,GetChColUtf8(line,cc),QArg(Src)]);
   ErrorDetails:=ErrorDetails+eline+EOL;
  end;
  SetInOutRes(0);
  for i:=Low(PasStack) to High(PasStack) do SmartFileClose(PasStack[i].Inp);
  SmartFileClose(LstOut);
  SetInOutRes(0);
 end;
end;

 {
 Call runtime P-code interpreter
 }
function TDaqPascal.Interpret:boolean;
begin
 Interpret:=RtmInterpreter(Self);
end;

 {
 Free all dinamic arrays for strings
 }
procedure TDaqPascal.FreeStrings;
var i:integer;
begin
 for i:=1 to stabmax do Deallocate(stab.table[i]);
 ClearTable(stab.stack);
 ClearTable(stab.table);
 ClearTable(dtab);
 stab.space:=0;
 stab.count:=0;
 if Assigned(timers) then timers.Count:=0;
 if Assigned(texts) then texts.Count:=0;
 _ComClose;
 lbuf1:='';
 lbuf2:='';
 lbuf3:='';
 sbuff:='';
end;

 {
 Init strings manager
 }
function TDaqPascal.InitStrings:boolean;
label Failure;
begin
 lbuf1:='';
 lbuf2:='';
 lbuf3:='';
 sbuff:='';
 FreeStrings;
 with stab do begin
  space:=0;
  count:=stabmax;
  while count>0 do begin
   inc(space);
   table[count]:=nil;
   stack[space]:=count;
   dec(count);
  end;
 end;
 nul := StrAlloc(StrAlignLeng(0));
 if nul=0 then goto Failure;
 spnt:=StrBuf(nul);
 if spnt=nil then goto Failure;
 spnt.head.leng := 0;
 spnt.head.dyna := 0;
 InitStrings:=true;
 exit;
Failure:
 InitStrings:=false;
 FreeStrings;
end;

 {
 Calculate aligned buffer size for string length leng with granularity strngalign
 }
function TDaqPascal.StrAlignLeng(leng:Integer):Integer;
begin
 Result:=(((leng+sizeof(strnghdr)-1) div strngalign)+1)*strngalign;
end;

 {
 Get string buffer pointer for given string index
 }
function TDaqPascal.StrBuf(s:integer):strngptr;
begin
 Result:=nil;
 if s<1 then exit;
 if s>stabmax then exit;
 Result:=stab.table[s];
end;

 {
 Get string index for given pointer
 }
function TDaqPascal.StrRef(P:Pointer):integer;
var i:integer;
begin
 StrRef:=0;
 if P=nil then exit;
 for i:=1 to stabmax do
 if P=stab.table[i] then begin
  StrRef:=i;
  exit;
 end;
end;

 {
 Get string pointer for given string index
 }
function TDaqPascal.StrPtr(s,o:integer):Pointer;
begin
 StrPtr:=nil;
 if s<1 then exit;
 if s>stabmax then exit;
 if stab.table[s]=nil then exit;
 StrPtr:=PChar(stab.table[s])+o;
end;

 {
 Allocates string of given size and return index of this string
 }
function TDaqPascal.StrAlloc(size:integer):integer;
var i:integer;
begin
 StrAlloc:=0;
 if size<1 then exit;
 with stab do begin
  if space=0 then exit;
  i:=stack[space];
  if i<1 then exit;
  if i>stabmax then exit;
  if table[i]<>nil then exit;
  table[i]:=Allocate(size);
  if table[i]=nil then exit;
  stack[space]:=0;
  dec(space);
  inc(count);
  StrAlloc:=i;
 end;
end;

 {
 Free string with given index s
 }
procedure   TDaqPascal.StrFree(s:integer);
begin
 if s<1 then exit;
 if s>stabmax then exit;
 with stab do begin
  if table[s]=nil then exit;
  Deallocate(Pointer(table[s]));
  if space=stabmax then exit;
  inc(space);
  dec(count);
  stack[space]:=s;
 end;
end;

 {
 Calls on device start
 }
procedure   TDaqPascal.Start;
var s:LongString;
begin
 opcount:=0;
 runcount:=0;
 BreakFlag:=0;
 SetInOutRes(0);
 SmartFileClose(StdInp);
 SmartFileClose(StdOut);
 SmartFileClose(f_file);
 ClearFifo;
 s:='';
 _reset(s);
 s:='';
 _rewrite(s);
 _EditClear;
 plotter3d.Clear;
 plotterpar.Lock:=false;
 _pipelist.Clear;
 _hashlist.Clear;
 _regexplist.Clear;
 _fsmlist.Clear;
 _dblist.Clear;
 _shmlist.Clear;
 lbuf1:='';
 lbuf2:='';
 lbuf3:='';
 _easyipclist.Clear;
 _backslash_esclist:=[];
 _backslash_hexlist:=[];
 s:=dump(Integer(0));_worddelims(s);s:='';
 worddelims:=ScanSpaces;
 phrasequotes:=QuoteMark+Apostrophe;
 _pct_reserved:=[];
 _pct_errors:=0;
 afnm_cur:=afnm_Def;
end;

 {
 Calls on device stop
 }
procedure   TDaqPascal.Stop;
begin
 try
  opcount:=0;
  runcount:=0;
  BreakFlag:=0;
  SetInOutRes(0);
  SmartFileClose(StdInp);
  SmartFileClose(StdOut);
  SmartFileClose(f_file);
  ClearFifo;
  _ComClose;
  _EditClear;
  plotter3d.Clear;
  plotterpar.Lock:=false;
  CleanOnStop;
  lbuf1:='';
  lbuf2:='';
  lbuf3:='';
  _backslash_esclist:=[];
  _backslash_hexlist:=[];
 except
  on E:Exception do BugReport(E,Self,'Stop');
 end;
end;

 {
 Calls to clean all objects on device stop.
 }
procedure   TDaqPascal.CleanOnStop;
var i:integer;
begin
 try
  for i:=_pipelist.Count-1 downto 0 do _pipe_free(Integer(_pipelist[i]));
  _pipelist.Clear;
  for i:=_hashlist.Count-1 downto 0 do _hashlist_free(Integer(_hashlist[i]));
  _hashlist.Clear;
  for i:=_regexplist.Count-1 downto 0 do _regexp_free(Integer(_regexplist[i]));
  _regexplist.Clear;
  for i:=_fsmlist.Count-1 downto 0 do _fsm_free(Integer(_fsmlist[i]));
  _fsmlist.Clear;
  for i:=_dblist.Count-1 downto 0 do _db_free(Integer(_dblist[i]));
  _dblist.Clear;
  for i:=_shmlist.Count-1 downto 0 do _shm_free(Integer(_shmlist[i]));
  _shmlist.Clear;
  for i:=_easyipclist.Count-1 downto 0 do _easyipc_free(PtrUint(_easyipclist[i]));
  _easyipclist.Clear;
 except
  on E:Exception do BugReport(E,Self,'CleanOnStop');
 end;
end;

 {
 Generate compile error
 }
procedure TDaqPascal.FatalError(n:integer);
begin
 ExceptResult:=n;
 ErrorLine:=PasStack[Low(PasStack)].iln;
 ErrorColumn:=PasStack[Low(PasStack)].cc;
 with PasStack[Low(PasStack)] do ErrorColUtf8:=GetChColUtf8(line,cc);
 RAISE EDaqPascalCompileError.CreateFmt('Compile error %d in file %s',[n,SrcFile]);
end;

procedure TDaqPascal.OpenIncludeFile(const aFileName:LongString);
 function SearchFile(aFileName:LongString):LongString;
 const SearchList='CRW_DAQ_CONFIG_PATH;CRW_DAQ_INCLUDE_PATH';
 var i:Integer;
 begin
  // NB!
  // On Unix filenames is case sensitive, so we
  // always use lower case filenames to avoid problems.
  if IsFileNameCaseSensitive and IsRelativePath(aFileName)
  then aFileName:=LowerCase(aFileName);
  // NB!
  Result:=SmartFileRef(aFileName,DefIncExt,SrcFile);
  if IsEmptyStr(Result) or not FileExists(Result) then begin
   if IsEmptyStr(ExtractFilePath(aFileName))
   then Result:=SmartFileSearch(DefaultExtension(aFileName,DefIncExt),SearchList)
   else Result:='';
   if IsEmptyStr(Result) or not FileExists(Result) then begin
    for i:=PasTop-1 downto Low(PasStack)+1 do begin
     if IsEmptyStr(Result) or not FileExists(Result)
     then Result:=SmartFileRef(aFileName,DefIncExt,PasStack[i].Src)
     else Break;
    end;
    if IsEmptyStr(Result) or not FileExists(Result)
    then Result:=SmartFileRef(aFileName,DefIncExt,SrcFile);
   end;
  end;
 end;
begin
 if PasTop<Low(PasStack)-1 then FatalError(74);
 if PasTop>=High(PasStack) then FatalError(75);
 PasTop:=PasTop+1;
 iln:=0; ll:=0; cc:=0; ch:=' ';
 PasSrc:=SearchFile(aFileName);
 if not FileExists(PasSrc) then FatalError(61);
 assign(PasInp^,PasSrc); reset(PasInp^);
 if IOResult <> 0 then FatalError(61);
 write(LstOut,EOL,'<<<<<  Opened[',PasTop,']: ',PasSrc,EOL,lc:5,'  ');
end;

procedure TDaqPascal.CloseIncludeFile;
begin
 if PasTop<=Low(PasStack) then FatalError(74);
 if PasTop>High(PasStack) then FatalError(75);
 write(LstOut,EOL,'>>>>>  Closed[',PasTop,']: ',PasSrc,EOL,lc:5,'  ');
 SmartFileClose(PasInp^); PasSrc:='';
 iln:=0; ll:=0; cc:=0; ch:=' ';
 PasTop:=PasTop-1;
end;

 {
 Read next character; process line end
 }
procedure TDaqPascal.nextch;
var  rch:Char;
begin
 if cc = ll then begin
  if eof(PasInp^) then begin
   if PasTop=Low(PasStack) then FatalError(71);
   CloseIncludeFile;
   nextch;
   Exit;
  end;
  //write(LstOut,lc:5, '  ');
  ll := 0;
  cc := 0;
  while not eoln(PasInp^) do begin
   if ll > llng-2 then FatalError(70);
   read(PasInp^,rch); ch:=rch;
   if IOResult<>0 then FatalError(61);
   if ch <> chr(10) then begin
    if ord(ch) < 32 then ch := ' ';
    //write(LstOut,ch);
    ll := ll+1;
    line[ll] := ch
   end
  end;
  ll := ll+1;
  line[ll] := ' ';
  read(PasInp^,rch); ch:=rch;
  if IOResult<>0 then FatalError(61);
  if (ch=Chr(13)) then begin // Handle CRLF delimeter
   read(PasInp^,rch); ch:=rch;
   if IOResult<>0 then FatalError(61);
   if (ch<>Chr(10)) then FatalError(81);
  end;
  //writeln(LstOut);
  iln := iln+1;
 end ;
 cc := cc+1;
 ch := line[cc];
 if cc = ll then write(LstOut,EOL,lc:5,'  ') else write(LstOut,ch);
end; { nextch }

 {
 reads next symbol
 }
procedure TDaqPascal.insymbol;
const dotdot = #31; sbufflen = 240;
label 1,2,3;
var i,j,k,e: integer; sbuff:string[sbufflen]; incfile:LongString;
 {}
 procedure readscale;
 var s,sign: integer;
 begin
  nextch;
  sign := 1;
  s := 0;
  if ch = '+'
  then nextch
  else if ch = '-' then begin
   nextch;
   sign := -1
  end;
  if not ((ch>='0') and (ch<='9')) then FatalError(40);
  repeat
   s := 10*s + ord(ch)-ord('0');
   nextch
  until not ((ch>='0') and (ch<='9'));
  e := s*sign + e
 end; { readscale }
 {}
 procedure adjustscale;
 var s:integer; d,t:real;
 begin
  if k+e > emax then FatalError(21);
  if k+e < emin then rnum := 0 else begin
   s := abs(e);
   t := 1.0;
   d := 10.0;
   repeat
    while not odd(s) do begin
     s := s div 2;
     d := sqr(d)
    end ;
    s := s-1;
    t := d*t
   until s = 0;
   if e >= 0 then rnum := rnum*t else rnum := rnum/t
  end
 end; { adjustscale }
 {}
 procedure options;
  procedure switch(var b: boolean);
  begin
   b:=ch='+';
   if not b
   then if not (ch='-') then FatalError(6) else nextch
   else nextch
  end; { switch }
  procedure ReadIncludeFileName;
  const IncFileChars = ['a'..'z','A'..'Z','0'..'9','.','_',':','\','/',' ','~'];
  begin
   incfile:='';
   if ch<>' ' then FatalError(6);
   while ch = ' ' do nextch;
   if not (ch in IncFileChars) then FatalError(6);
   while not (ch in [' ','}','*']) do begin
    if ch in IncFileChars then begin
     incfile:=incfile+ch;
     nextch;
    end else FatalError(6);
   end;
   while ch = ' ' do nextch;
   if not (ch in ['}','*']) then FatalError(6);
  end; {readincfile}
 begin      {options}
  repeat
   nextch;
   if (ch <> '*') and (ch <> '}') then begin
    if ((ch='i') or (ch='I')) then begin
     nextch;
     ReadIncludeFileName; Break;
    end else
    if ((ch='t') or (ch='T')) then begin
     nextch;
     switch(prtables)
    end else
    if ((ch='s') or (ch='S')) then begin
     nextch;
     switch(stackdump)
    end
   end
  until ch<>','
 end;  { options }
 procedure HexOctBinNumber(basech:Char; next:Boolean);
 var snum:LongString; kmax:Integer; digs:TCharSet;
 begin
  k:=0; inum:=0; sy:=intcon;
  snum:=''; kmax:=0; digs:=[];
  case basech of
   '$': begin digs:=['0'..'9','A'..'F','a'..'f']; kmax:=8; end;
   '&': begin digs:=['0'..'7']; kmax:=11; end;
   '%': begin digs:=['0'..'1']; kmax:=32; end;
   else FatalError(73);
  end;
  if next then nextch;
  while (ch in digs) do begin
   snum:=snum+ch;
   k:=k+1;
   nextch;
  end;
  {$WARNINGS OFF}
  if (k>kmax) or (inum>nmax) then FatalError(21);
  {$WARNINGS ON}
  case basech of
   '$': if not StrHex2Long(snum,inum) then FatalError(78);
   '&': if not StrOct2Long(snum,inum) then FatalError(79);
   '%': if not StrBin2Long(snum,inum) then FatalError(80);
   else FatalError(73);
  end;
 end; { HexOctBinNumber }
begin    { insymbol }
1:
 while ch = ' ' do nextch;
 if ch in ['a'..'z','A'..'Z','_'] then begin { identifier or wordsymbol }
  k := 0;
  setalfa(id,'            ');
  if ch in ['A'..'Z'] then ch := chr(ord(ch)+32);
  repeat
   if k < alng then begin
    k := k+1;
    id[k] := ch
   end ;
   nextch;
   if ch in ['A'..'Z'] then ch := chr(ord(ch)+32)
  until not ( (ch in ['a'..'z']) or (ch in ['0'..'9']) or (ch='_') );
  { binary search }
  i := 1;
  j:= nokeywrds;
  repeat
   k := (i+j) div 2;
   if id <= key[k] then j := k-1;
   if id >= key[k] then i := k+1
  until i > j;
  if i-1 > j then sy := ksy[k] else sy := ident
 end else
 if ch in ['+','-','*',')','=',',','[',']',';','&','|','~'] then begin
  sy := sps[ch];
  nextch;
  if (sy=sps['&']) and (ch in ['0'..'7']) then HexOctBinNumber('&',false);
 end else
 if ch in ['$','%'] then begin
  HexOctBinNumber(ch,true);
 end else
 if ch in ['0'..'9'] then begin { number }
  k := 0;
  inum := 0;
  sy := intcon;
  repeat
   inum := inum*10 + ord(ch) - ord('0');
   k := k+1;
   nextch
  until not ((ch>='0') and (ch<='9'));                           {$WARNINGS OFF}
  if (k > kmax) or (inum > nmax) then FatalError(21);            {$WARNINGS ON}
  if ch = '.' then begin
   nextch;
   if ch = '.' then ch := dotdot else begin
    sy := realcon;
    rnum := inum;
    e := 0;
    while (ch>='0') and (ch<='9') do begin
     e := e-1;
     rnum := 10.0*rnum + (ord(ch)-ord('0'));
     nextch
    end ;
    if e = 0 then FatalError(40);
    if ((ch = 'e') or (ch = 'E')) then readscale;
    if e <> 0 then adjustscale end
   end else
   if ((ch = 'e') or (ch = 'E')) then begin
    sy := realcon;
    rnum := inum;
    e := 0;
    readscale;
    if e <> 0 then adjustscale
   end;
  end else
  case ch of
   ':' :
    begin
     nextch;
     if ch = '=' then begin
      sy := becomes;
      nextch
     end  else sy := colon
    end;
   '<' :
    begin
     nextch;
     if ch = '=' then begin
      sy := leq;
      nextch
     end else
     if ch = '>' then begin
      sy := neq;
      nextch
     end else sy := lss
    end;
   '>' :
    begin
     nextch;
     if ch = '=' then begin
      sy := geq;
      nextch
     end else sy := gtr
    end;
   '.' :
    begin
     nextch;
     if ch = '.' then begin
      sy := twodots;
      nextch
     end else sy := period
    end;
   dotdot:
    begin
     sy := twodots;
     nextch
    end;
   '''' :
    begin
     sbuff := '';
 2:  nextch;
     if ch = '''' then begin
      nextch;
      if ch <> '''' then goto 3
     end ;
     if length(sbuff) < sbufflen then sbuff := sbuff + ch else FatalError(38);
     if cc = 1 then FatalError(38) else goto 2;
3:   if length(sbuff) = 1 then begin
      sy := charcon;
      inum := ord(sbuff[1])
     end else begin
      sy := stringcon;
      sleng := length(sbuff);
      if sleng=0 then begin
       if nul=0 then FatalError(69);
       spnt := StrBuf(nul);
       if spnt=nil then FatalError(72);
      end else begin
       k := StrAlloc(StrAlignLeng(sleng));
       if k=0 then FatalError(69);
       spnt:=StrBuf(k);
       if spnt=nil then FatalError(72);
       spnt.head.leng := sleng;
       spnt.head.dyna := 0;
       move(sbuff[1],spnt.data.buff,sleng);
      end;
     end
    end;
   '(' :
    begin
     nextch;
     if ch <> '*'
     then sy := lparent
     else begin { comment }
      nextch;
      incfile:='';
      if ch='$' then options;
      repeat
       while ch <>  '*' do nextch;
       nextch
      until ch = ')';
      if incfile<>'' then OpenIncludeFile(incfile);
      nextch;
      goto 1
     end
    end;
   '{' :
    begin { comment }
     nextch;
     incfile:='';
     if ch='$' then options;
     while ch <> '}' do nextch;
     if incfile<>'' then OpenIncludeFile(incfile);
     nextch;
     goto 1
    end;
   '/' :
    begin
     nextch;
     if ch <> '/'
     then sy := sps['/']
     else begin               { C - like comment: }
      while cc > 1 do nextch; { skip current line }
      goto 1
     end
    end;
   else nextch; FatalError(24); goto 1
 end; { case }
end; { insymbol }

 {
 Enter standard identifier to identifier table
 }
procedure TDaqPascal.enterid(const x0:LongString; x1:objekt; x2:types; x3:integer);
begin
 it := it+1;
 if it > itabmax then AllocateTable(Pointer(itab), itabmax, Round(itabmax*tabFactor), sizeof(itab[0]), 1, false);
 if it > itabmax then FatalError(63);
 with itab[it] do begin
  setalfa(name,x0);
  link   := it-1;
  obj    := x1;
  typ    := x2;
  ref    := 0;
  normal := true;
  lev    := 0;
  adr    := x3
 end
end;

 {
 Enter array to array table
 }
procedure TDaqPascal.enterarray(tp:types; l,h:integer);
begin
 if l > h then FatalError(27);                                   {$WARNINGS OFF}
 if (abs(l)>xarmax) or (abs(h)>xarmax) then FatalError(27);      {$WARNINGS ON}
 if ia = atabmax then AllocateTable(Pointer(atab), atabmax, Round(atabmax*tabFactor), sizeof(atab[1]), 0, false);
 if ia = atabmax then FatalError(66);
 ia := ia+1;
 with atab[ia] do begin
  inxtyp := tp;
  low    := l;
  high   := h;
 end;
end;

 {
 Enter block to block table
 }
procedure TDaqPascal.enterblock;
begin
 if ib = btabmax then AllocateTable(Pointer(btab), btabmax, Round(btabmax*tabFactor), sizeof(btab[1]), 0, false);
 if ib = btabmax then FatalError(64);
 ib := ib+1;
 btab[ib].last := 0;
 btab[ib].lastpar := 0
end;

 {
 Enter real variable
 }
procedure TDaqPascal.enterreal(x: real);
begin
 if ic2 = rtabmax-1 then AllocateTable(Pointer(rtab), rtabmax, Round(rtabmax*tabFactor), sizeof(rtab[1]), 0, false);
 if ic2 = rtabmax-1 then FatalError(65);
 rtab[ic2+1] := x;
 ic1 := 1;
 while rtab[ic1] <> x do ic1 := ic1+1;
 if ic1 > ic2 then ic2 := ic1;
end;

 {
 Write operation code to P-code table
 }
procedure TDaqPascal.emit(fct: integer);
begin
 if lc = ctabmax then AllocateTable(Pointer(ctab), ctabmax, Round(ctabmax*tabFactor), sizeof(ctab[0]), 1, false);
 if lc = ctabmax then FatalError(68);
 ctab[lc].f:=fct;
 lc:=lc+1
end;

 {
 Write operation code with one argument to P-code table
 }
procedure TDaqPascal.emit1(fct,b: integer);
begin
 if lc = ctabmax then AllocateTable(Pointer(ctab), ctabmax, Round(ctabmax*tabFactor), sizeof(ctab[0]), 1, false);
 if lc = ctabmax then FatalError(68);
 with ctab[lc] do begin
  f:=fct;
  y:=b
 end;
 lc:=lc+1;
end;

 {
 Write operation code with two arguments to P-code table
 }
procedure TDaqPascal.emit2(fct,a,b: integer);
begin
 if lc = ctabmax then AllocateTable(Pointer(ctab), ctabmax, Round(ctabmax*tabFactor), sizeof(ctab[0]), 1, false);
 if lc = ctabmax then FatalError(68);
 with ctab[lc] do begin
  f:=fct;
  x:=a;
  y:=b
 end;
 lc:=lc+1
end;

procedure TDaqPascal.printtables;
var i:integer;
begin
 {
 itab
 }
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut,'Table of Identifiers:');
 writeln(LstOut,'---------------------');
 writeln(LstOut,'i':5,' ','name':alng,' ','link':5,' ','obj':5,' ','typ':5,' ',
                'ref':5,' ','norm':5,' ','lev':5,' ','adr':5);
 writeln(LstOut);
 for i := btab[1].last to it do
 with itab[i] do
 writeln(LstOut,i:5,' ',Trim(name):alng,' ',link:5,' ',ord(obj):5,' ',ord(typ):5,' ',
                ref:5,' ',ord(normal):5,' ',lev:5,' ',adr:5);
 {
 btab
 }
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut,'Table of Blocks:');
 writeln(LstOut,'----------------');
 writeln(LstOut,'i':5,' ','last':5,' ','lpar':5,' ','psze':5,' ','vsze':5);
 writeln(LstOut);
 for i := 1 to ib do
 with btab[i] do
 writeln(LstOut,i:5,' ',last:5,' ',lastpar:5,' ',psize:5,' ',vsize:5);
 {
 atab
 }
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut,'Table of Arrays:');
 writeln(LstOut,'----------------');
 writeln(LstOut,'i':5,' ','xtyp':5,' ','etyp':5,' ','eref':5,' ','low':5,' ',
                'high':5,' ','elsz':5,' ','size':5);
 writeln(LstOut);
 for i := 1 to ia do
 with atab[i] do
 writeln(LstOut,i:5,' ',ord(inxtyp):5,' ',ord(eltyp):5,' ',elref:5,' ',low:5,' ',
                high:5,' ',elsize:5,' ',size:5);
 {
 ctab
 }
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut,'Table of PCodes:');
 writeln(LstOut,'----------------');
 writeln(LstOut,'i':5,' ','pcode':5,' ','xoper':5,' ','yoper':5,' ');
 for i:=0 to lc-1 do begin
  write(LstOut);
  write(LstOut,i:5,' ');
  write(LstOut,ctab[i].f:5,' ');
  if ctab[i].f < 100 then
  if ctab[i].f<4 then write(LstOut,ctab[i].x:5,' ',ctab[i].y:5,' ') else write(LstOut,ctab[i].y:11,' ')
  else write(LstOut,' ':11,' ');
  writeln(LstOut,',')
 end;
 writeln(LstOut);
 writeln(LstOut,'Starting address is ',itab[btab[1].last].adr:5);
 writeln(LstOut);
 writeln(LstOut);
 writeln(LstOut);
end;

 {
 ***************************
 BEGIN Block syntax analizer
 ***************************
 }
procedure TDaqPascal.CompileBlock(fsys:symset; isfun:boolean; level:integer);
type
 item = packed record
  typ  : types;
  ref  : index;
  temp : boolean
 end;
 conrec = packed record
  case tp: types of
  ints,chars,bools: (i:integer);
  reals:            (r: real)
 end ;
var
 dx : integer;    { data allocation index }
 prt: integer;    { t-index of this procedure }
 prb: integer;    { b-index of this procedure }
 x  : integer;
 { clear item x }
 procedure clrit(out x:item);
 begin
  x.typ:=notyp; x.ref:=0; x.temp:=false;
 end;
 {}
 procedure test(s1:symset; n:integer);
 begin
  if not (sy in s1) then FatalError(n)
 end;
 {}
 procedure testsemicolon;
 begin
  if sy = semicolon then insymbol else FatalError(14);
  test([ident]+blockbegsys, 6)
 end;
 {}
 procedure enter(id: alfa; k:objekt);
 var j,l: integer;
 begin
  if it = itabmax then AllocateTable(Pointer(itab), itabmax, Round(itabmax*tabFactor), sizeof(itab[0]), 1, false);
  if it = itabmax then FatalError(63) else begin
   itab[0].name := id;
   j := btab[display[level]].last;
   l := j;
   while itab[j].name <> id do  j := itab[j].link;
   if j <> 0 then FatalError(1) else begin
    it := it+1;
    with itab[it] do begin
     name:= id;
     link := l;
     obj := k;
     typ := notyp;
     ref := 0;
     lev := level;
     adr := 0
    end;
    btab[display[level]].last := it
   end
  end
 end;
 {}
 function loc(id: alfa): integer;
 var i,j: integer;      { locate id in tabel }
 begin
  i := level;
  itab[0].name := id;    { sentinel }
  repeat
   j := btab[display[i]].last;
   while itab[j].name <> id do  j := itab[j].link;
   i := i-1;
  until (i<0) or (j<>0);
  if j = 0 then FatalError(0);
  loc := j
 end;
 {}
 procedure entervariable;
 begin
  if sy = ident then begin
   enter(id,vvariable);
   insymbol
  end else FatalError(2)
 end;
 {}
 procedure constant(fsys: symset; out c: conrec);
 var x,sign:integer;
 begin
  c.tp := notyp;
  c.i := 0;
  test(constbegsys, 50);
  if sy in constbegsys then begin
   if sy = charcon  then begin
    c.tp := chars;
    c.i := inum;
    insymbol
   end else
   if sy = stringcon then begin
    c.tp := strngs;
    c.i := StrRef(spnt);
    insymbol
   end else begin
    sign := 0;
    if sy in [plus,minus] then begin
     if sy = minus then sign := -1 else sign := 1;
     insymbol
    end;
    if sy = ident then begin
     x := loc(id);
     if x <> 0 then
     if itab[x].obj <> konstant then FatalError(25) else begin
      c.tp := itab[x].typ;
      if c.tp in [ints,reals] then if sign=0 then sign := 1;
      if c.tp = reals then c.r := sign*rtab[itab[x].adr] else
      if c.tp = ints then c.i := sign*itab[x].adr else begin
       if sign<>0 then FatalError(33);
       c.i := itab[x].adr
      end
     end ;
     insymbol
    end else begin
     if sign=0 then sign := 1;
     if sy = intcon then begin
      c.tp := ints;
      c.i := sign*inum;
      insymbol
     end else
     if sy = realcon then begin
      c.tp := reals;
      c.r := sign*rnum;
      insymbol
     end else FatalError(50)
    end
   end;
   test(fsys, 6)
  end
 end;
 {}
 procedure typ(fsys: symset; out tp: types; out rf, sz: integer);
 var    eltp: types; elrf,elsz,offset,x,t0,t1: integer; dummy: conrec;
  procedure arraytyp(out aref,arsz: integer);
  var eltp: types; low, high: conrec; elrf, elsz: integer;
  begin
   constant([twodots,rbrack,rparent,ofsy]+fsys, low);
   if low.tp in [reals,strngs] then FatalError(27);
   if sy = twodots then insymbol else FatalError(13);
   constant([rbrack,comma,rparent,ofsy]+fsys, high);
   if high.tp <> low.tp then FatalError(27);
   enterarray(low.tp, low.i,high.i);
   aref := ia;
   if sy = comma then begin
    insymbol;
    eltp := arrays;
    arraytyp(elrf,elsz)
   end else begin
    if sy = rbrack then insymbol else FatalError(12);
    if sy = ofsy then insymbol else FatalError(8);
    typ(fsys,eltp,elrf,elsz)
   end ;
   with atab[aref] do begin
    arsz := (high-low+1)*elsz; size := arsz;
    if arsz > stacksize then FatalError(20);
    eltyp := eltp;
    elref := elrf;
    elsize := elsz
   end ;
  end;  {arraytyp }
 begin  { typ }
  tp := notyp;
  rf := 0;
  sz := 0;
  test(typebegsys, 10);
  if sy in typebegsys then begin
   if sy = ident then begin
    x := loc(id);
    if x <> 0 then with itab[x] do
    if obj <> type1 then FatalError(29) else begin
     tp := typ;
     rf := ref;
     sz := adr;
     if tp = notyp then FatalError(30)
    end ;
    insymbol;
    if (tp=strngs) and (sy=lbrack) then begin
     insymbol;
     constant([rbrack]+fsys,dummy);
     if sy=rbrack then insymbol else FatalError(12);
    end;
   end else
   if sy = arraysy then begin
    insymbol;
    if sy = lbrack then insymbol else FatalError(11);
    tp := arrays;
    arraytyp(rf,sz)
   end else begin  { records }
    insymbol;
    enterblock;
    tp := records;
    rf := ib;
    if level = levmax then FatalError(67);
    level := level+1;
    display[level] := ib;
    offset := 0;
    { field section }
    while not (sy in fsys-[semicolon,comma,ident]+[endsy]) do begin
     if sy = ident then begin
      t0 := it;
      entervariable;
      while sy = comma do begin
       insymbol;
       entervariable;
      end ;
      if sy = colon then insymbol else FatalError(5);
      t1 := it;
      typ(fsys+[semicolon,endsy,comma,ident],eltp,elrf,elsz);
      while t0 < t1 do begin
       t0 := t0+1;
       with itab[t0] do begin
        typ := eltp;
        ref := elrf;
        normal := true;
        adr := offset;
        offset := offset + elsz
       end
      end
     end ; {sy = ident}
     if sy <> endsy then begin
      if sy = semicolon then insymbol else FatalError(14);
      test([ident,endsy,semicolon], 6)
     end
    end ; {field section}
    btab[rf].vsize := offset;
    sz := offset;
    if sz > stacksize then FatalError(20);
    btab[rf].psize := 0;
    insymbol;
    level := level-1
   end ; {records}
   test(fsys, 6)
  end
 end  { typ } ;
 {}
 procedure parameterlist;      { formal parameter list }
 var tp:types; valpar:boolean; rf,sz,x,t0:integer;
 begin
  insymbol;
  tp := notyp;
  rf := 0;
  sz := 0;
  test([ident, varsy], 7);
  while sy in [ident, varsy] do begin
   if sy <> varsy then valpar := true else begin
    insymbol;
    valpar := false
   end ;
   t0 := it;
   entervariable;
   while sy = comma do begin
    insymbol;
    entervariable;
   end;
   if sy = colon then begin
    insymbol;
    if sy <> ident then FatalError(2) else begin
     x := loc(id);
     insymbol;
     if x <> 0 then with itab[x] do
     if obj <> type1 then FatalError(29) else begin
      tp := typ;
      rf := ref;
      if valpar then sz := adr else sz := 1
     end ;
    end ;
    test([semicolon,rparent], 14)
   end else FatalError(5);
   while t0 < it do begin
    t0 := t0+1;
    with itab[t0] do begin
     typ := tp;
     ref := rf;
     adr := dx;
     lev := level;
     normal := valpar;
     dx := dx + sz
    end
   end ;
   if sy <> rparent then begin
    if sy = semicolon then insymbol else FatalError(14);
    test([ident,varsy], 6)
   end
  end;  { while }
  if sy = rparent then begin
   insymbol;
   test([semicolon,colon], 6)
  end else FatalError(4)
 end;  { parameterlist }
 {
 }
 procedure constdec;
 var c:conrec;
 begin
  insymbol;
  test([ident], 2);
  while sy = ident do begin
   enter(id,konstant);
   insymbol;
   if sy = eql then insymbol else FatalError(16);
   constant([semicolon,comma,ident]+fsys,c);
   itab[it].typ := c.tp;
   itab[it].ref := 0;
   if c.tp = reals then begin
    enterreal(c.r);
    itab[it].adr := ic1
   end else itab[it].adr := c.i;
   testsemicolon
  end
 end;  { constdec }
 {}
 procedure typedeclaration;
 var tp:types; rf, sz, t1: integer;
 begin
  insymbol;
  test([ident], 2);
  while sy = ident do begin
   enter(id,type1);
   t1 := it;
   insymbol;
   if sy = eql then insymbol else FatalError(16);
   typ([semicolon,comma,ident]+fsys, tp, rf, sz);
   with itab[t1] do begin
    typ := tp;
    ref := rf;
    adr := sz
   end;
   testsemicolon
  end
 end;  { typedeclaration }
 {}
 procedure variabledeclaration;
 var tp:types; t0, t1, rf, sz: integer;
 begin
  insymbol;
  while sy = ident do begin
   t0 := it;
   entervariable;
   while sy = comma do begin
    insymbol;
    entervariable;
   end ;
   if sy = colon then insymbol else FatalError(5);
   t1 := it;
   typ([semicolon,comma,ident]+fsys, tp, rf, sz);
   while t0 < t1 do begin
    t0 := t0+1;
    with itab[t0] do begin
     typ := tp;
     ref := rf;
     lev := level;
     adr := dx;
     normal := true;
     dx := dx + sz
    end
   end ;
  testsemicolon
  end
 end;  { variabledeclaration }
 {}
 procedure procdeclaration;
 var isfun: boolean;
 begin
  isfun := sy = funcsy;
  insymbol;
  if sy <> ident then FatalError(2);
  if isfun then enter(id,funktion) else enter(id,prozedure);
  itab[it].normal := true;
  insymbol;
  CompileBlock([semicolon]+fsys, isfun, level+1);
  if sy = semicolon then insymbol else FatalError(14);
  emit(132+ord(isfun))     { exit }
 end  { procdeclaration } ;
 {}
 procedure statement(fsys: symset);
 var i:integer;
  procedure expression(fsys: symset; var x: item); forward;
  procedure selector(fsys: symset; var v: item);
  var x:item; a,j:integer;
  begin  { sy in [lparent, lbrack, period] }
   clrit(x);
   repeat
    if sy = period then begin
     insymbol;   { field selector }
     if sy <> ident then FatalError(2) else begin
      if v.typ <> records
      then FatalError(31)
      else begin  {search field identifier }
       j := btab[v.ref].last;
       itab[0].name := id;
       while itab[j].name <> id do j := itab[j].link;
       if j = 0 then FatalError(0);
       v.typ := itab[j].typ;
       v.ref := itab[j].ref;
       a := itab[j].adr;
       if a <> 0 then emit1(9,a)
      end ;
      insymbol
     end
    end else begin  { array selector }
     if sy <> lbrack then FatalError(11);
     if v.typ=strngs then begin
      insymbol;
      expression(fsys+[rbrack],x);
      if x.typ<>ints then FatalError(34) else emit(165);
      v.typ := chars
     end else
     repeat
      insymbol;
      expression(fsys+[comma,rbrack], x);
      if v.typ <> arrays then FatalError(28) else begin
       a := v.ref;
       if atab[a].inxtyp <> x.typ
       then FatalError(26)
       else if atab[a].elsize = 1 then emit1(20,a) else emit1(21,a);
       v.typ := atab[a].eltyp;
       v.ref := atab[a].elref
      end
     until sy <> comma;
     if sy = rbrack then insymbol else FatalError(12);
    end;
   until not (sy in [lbrack,lparent,period]);
   test (fsys, 6)
  end  { selector } ;
  {}
  procedure call(fsys: symset; i: integer);
  var x:item; lastp, cp, k: integer;
  begin
   clrit(x);
   emit1(18,i);   { mark stack }
   lastp := btab[itab[i].ref].lastpar;
   cp := i;
   if sy = lparent then begin  { actual parameter list }
    repeat
     insymbol;
     if cp >= lastp then FatalError(39) else begin
      cp := cp+1;
      if itab[cp].normal then begin  {value parameter }
       expression(fsys+[comma,colon,rparent], x);
       if x.typ=itab[cp].typ then begin
        if x.ref <> itab[cp].ref
        then FatalError(36)
        else if x.typ = arrays
        then emit1(22,atab[x.ref].size)
        else if x.typ = records
        then emit1(22,btab[x.ref].vsize)
        else if x.typ = strngs
        then if x.temp then emit(173)
        else emit(172)
       end
       else if (x.typ=ints) and (itab[cp].typ=reals)
       then emit1(26,0)
       else if x.typ<>notyp then FatalError(36);
      end else begin  { var parameter }
       if sy <> ident then FatalError(2) else begin
        k := loc(id);
        insymbol;
        if k <> 0 then begin
         if itab[k].obj <> vvariable then FatalError(37);
         x.typ := itab[k].typ;
         x.ref := itab[k].ref;
         if itab[k].normal
         then emit2(0,itab[k].lev,itab[k].adr)
         else emit2(1,itab[k].lev,itab[k].adr);
         if sy in [lbrack,lparent,period] then begin
          if x.typ=strngs then FatalError(60);
          selector(fsys+[comma,colon,rparent], x);
         end;
         if (x.typ<>itab[cp].typ) or (x.ref<>itab[cp].ref) then FatalError(36)
        end
       end
      end {var parameter}
     end ;
     test([comma,rparent], 6)
    until sy <> comma;
    if sy = rparent then insymbol else FatalError(4)
   end ;
   if cp < lastp then FatalError(39);  { too few actual parameters }
   emit1(19, btab[itab[i].ref].psize-1);
   if itab[i].lev < level then emit2(3, itab[i].lev, level)
  end  { call } ;
  {}
  function resulttype(a,b: types): types;
  begin
   if (a>reals) or (b>reals) then FatalError(33);
   if (a=notyp) or (b=notyp) then resulttype := notyp else
   if a=ints then if b=ints then resulttype := ints else begin
    resulttype := reals;
    emit1(26,1)
   end else begin
    resulttype := reals;
    if b=ints then emit1(26,0)
   end
  end;   { resulttype }
  {}
  procedure expression {fsys:symset; var x:item};
  var y:item;  op:symbol; t:integer;
   {}
   procedure simpleexpression(fsys:symset; var x:item);
   var y:item; op:symbol; t:integer;
    {}
    procedure term(fsys:symset; var x:item);
    var y:item; op:symbol;
     {}
     procedure factor(fsys:symset; var x:item);
     var i,f:integer;
      {standard functions call}
      procedure standfct(n: integer);
      var ts:typset;
       {add real argument}
       procedure argreals(sym:symbol);
       begin
        test([comma], 59);
        insymbol;
        expression(fsys+[sym],y);
        if not (y.typ in [ints,reals,notyp]) then FatalError(42);
        if y.typ = ints then emit1(26,0);
       end;
       {add integer argument}
       procedure argints(sym:symbol);
       begin
        test([comma], 59);
        insymbol;
        expression(fsys+[sym],y);
        if not (y.typ in [ints,notyp]) then FatalError(34);
       end;
       {add string argument; fac=1,3,9,27...}
       procedure argstrngs(sym:symbol;fac:integer);
       begin
        test([comma], 59);
        insymbol;
        expression(fsys+[sym],y);
        if not (y.typ in [strngs,chars,notyp]) then FatalError(38);
        if y.typ = chars then n := n+2*fac else if y.temp then n := n+1*fac;
       end;
       {add char argument}
       procedure argchars(sym:symbol);
       begin
        test([comma], 59);
        insymbol;
        expression(fsys+[sym],y);
        if not (y.typ in [chars,notyp]) then FatalError(76);
       end;
       {add variant argument:string/temporary,char,integer,real}
       procedure argSTCIRs(sym:symbol;fac:integer);
       begin
        test([comma], 59);
        insymbol;
        expression(fsys+[sym],y);
        if not (y.typ in [strngs,chars,ints,reals,notyp]) then FatalError(48);
        if y.typ = strngs then if y.temp then n := n+1*fac;
        if y.typ = chars then n := n+2*fac;
        if y.typ = ints then n := n+3*fac;
        if y.typ = reals then n:= n+4*fac;
       end;
       {add variant argument:string/temporary,char,integer}
       procedure argSTCIs(sym:symbol;fac:integer);
       begin
        test([comma], 59);
        insymbol;
        expression(fsys+[sym],y);
        if not (y.typ in [strngs,chars,ints,reals,notyp]) then FatalError(48);
        if y.typ = strngs then if y.temp then n := n+1*fac;
        if y.typ = chars then n := n+2*fac;
        if y.typ = ints then n := n+3*fac;
       end;
      begin
       { standard function no. n }
       case n of
        {func without args}
        17,18,19,39,40,41,42,43,44,45,46,47,48,49,50,124,128,189,
        201,202,203,204,205,206,210,211,222,223,228,229,278,279,280,328,
        347,348,363,365,386,390,391,392,405,415,416,442,468,571,606,659,
        678,679,680,681,682,739,783,784,785,844,848,849,850,993:
         emit1(8,n);
        {func with args}
        else begin
         if sy = lparent then insymbol else FatalError(9);
         expression(fsys+[comma,rparent],x);
         case n of
          0,2: { abs,sqr }
           begin
            ts := [ints,reals];
            itab[i].typ := x.typ;
            if x.typ = reals then n := n+1
           end;
          4,5: { odd,chr }
           ts := [ints];
          6: { ord }
           ts := [ints,bools,chars];
          7,8:{ succ,pred }
           begin
            ts := [ints,bools,chars];
            itab[i].typ := x.typ
           end;
          { function(boolean) }
          641,800:
           ts:=[bools];
          { function(real) as round,trunc,sin,cos,... }
          9,10,11,12,13,14,15,16,51,52,53,54,55,56,57,58,59,60,61,62,63,
          64,65,66,67,89,109,190,191,192,193,194,195,196,197,
          286,287,288,289,290,291,351,364,393,394,395,467,670,723:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0)
           end;
          { function(real,real) }
          68,69,70,71,72,73,74,75,76,77,78,79,80,81,90,91,108,198,284,285:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argreals(rparent);
           end;
          { function(integer,real) }
          96,97,98,99,100,101,274,927,929,1009:
           begin
            ts := [ints];
            argreals(rparent);
           end;
          { function(real,integer) }
          114,199,306,456,466:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argints(rparent);
           end;
          { function(real,integer,integer) }
          352:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argints(comma);
            argints(rparent);
           end;
          { function(integer,integer) }
          113,216,217,218,219,220,240,241,248,249,252,273,302,303,304,305,333,
          399,401,428,432,573,583,683,684,692,696,709,743,808,809,810,811,865,
          921,922,923,925,977,1004,1007,1028,1055,1080,1102,1103,1105:
           begin
            ts := [ints];
            argints(rparent);
           end;
          { function(char) }
          1030,1031:
           ts := [chars];
          { function(integer) }
          82,83,84,85,86,87,88,92,93,94,95,110,111,112,215,221,235,236,237,
          238,239,242,243,244,245,246,247,250,251,253,254,255,256,257,258,
          259,260,267,268,269,270,271,272,301,332,340,349,350,359,385,398,400,
          424,425,426,427,433,434,435,436,437,455,572,584,640,654,669,688,689,
          690,691,697,698,699,700,701,706,707,708,710,740,741,742,786,787,843,
          860,861,920,924,926,928,957,958,994,995,996,997,998,999,1000,1006,
          1008,1010,1020,1029,1032,1033,1034,1035,1048,1049,1050,1051,1052,
          1053,1054,1059,1060,1061,1062,1063,1067,1068,1069,1070,1071,1072,
          1073,1074,1075,1076,1077,1078,1079,1118,1119,1132,1143,1168,1169,
          1200,1204:
           ts := [ints];
          { function(real,real,real,real) }
          102,396:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argreals(comma);
            argreals(comma);
            argreals(rparent);
           end;
          { function(real,real,real) }
          103,397,417:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argreals(comma);
            argreals(rparent);
           end;
          { function(real,integer,real) }
          200:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argints(comma);
            argreals(rparent);
           end;
          { function(integer,real,real) }
          105,106,107:
           begin
            ts := [ints];
            argreals(comma);
            argreals(rparent);
           end;
          { function(integer,integer,real,real,real) }
          104:
           begin
            ts := [ints];
            argints(comma);
            argreals(comma);
            argreals(comma);
            argreals(rparent);
           end;
          { function(integer,integer,integer)}
          655,978,979,980,1005,1104,1136:
           begin
            ts := [ints];
            argints(comma);
            argints(rparent);
           end;
          { function(integer,integer,integer,integer)}
          227,230,233,234,438,702:
           begin
            ts := [ints];
            argints(comma);
            argints(comma);
            argints(rparent);
           end;
          { function(integer,integer,integer,integer,integer)}
          231:
           begin
            ts := [ints];
            argints(comma);
            argints(comma);
            argints(comma);
            argints(rparent);
           end;
          { function(integer,integer,integer,integer,real)}
          232:
           begin
            ts := [ints];
            argints(comma);
            argints(comma);
            argints(comma);
            argreals(rparent);
           end;
          { function(integer,integer,integer,integer,integer,integer,integer)}
          677:
           begin
            ts := [ints];
            argints(comma);
            argints(comma);
            argints(comma);
            argints(comma);
            argints(comma);
            argints(rparent);
           end;
          { function(string) }
          20,115,118,121,125,129,132,135,138,141,144,147,150,153,156,159,162,
          165,168,171,174,177,180,183,186,207,212,224,261,281,292,295,298,
          307,310,313,316,322,329,334,337,341,360,370,373,376,379,382,387,402,
          406,409,421,443,446,449,452,469,532,535,538,541,544,547,550,553,556,
          559,562,565,568,594,607,610,613,634,637,642,671,674,685,720,724,727,
          730,733,777,780,822,825,828,831,834,845,869,872,945,981,984,987,990,
          1036,1120,1137,1140,1147,1156,1159,1162,1165,1170,1173,1176:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
           end;
          { function(string,integer,integer) }
          736,837,1109,1115,1133,1179:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            argints(comma);
            argints(rparent);
           end;
          { function(string,integer) }
          264,325,344,353,356,788,840,951:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            argints(rparent);
           end;
          { function(integer,string) }
          275,319,429,439,475,580,656,693,703,744,747,750,756,762,774,862,866,954,959,962,965,
          1001,1011,1025,1039,1042,1045,1056,1081,1099,1129,1144,1150,1153,1201:
           begin
            ts:=[ints];
            argstrngs(rparent,1);
           end;
          { function(integer,string,real) }
          753:
           begin
            ts:=[ints];
            argstrngs(comma,1);
            argreals(rparent);
           end;
          { function(integer,string,integer) }
          759,1064,1106:
           begin
            ts:=[ints];
            argstrngs(comma,1);
            argints(rparent);
           end;
          { function(integer,string,char,integer) }
          1084:
           begin
            ts:=[ints];
            argstrngs(comma,1);
            argchars(comma);
            argints(rparent);
           end;
          { function(integer,string,char,real) }
          1087:
           begin
            ts:=[ints];
            argstrngs(comma,1);
            argchars(comma);
            argreals(rparent);
           end;
          { function(integer,string,char,string) }
          1090:
           begin
            ts:=[ints];
            argstrngs(comma,1);
            argchars(comma);
            argstrngs(rparent,3);
           end;
          { function(string,real) }
          412,1112:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            argreals(rparent);
           end;
          { function(integer,integer,real,real,real,real,string) }
          418:
           begin
            ts:=[ints];
            argints(comma);
            argreals(comma);
            argreals(comma);
            argreals(comma);
            argreals(comma);
            argstrngs(rparent,1);
           end;
          { function(real,integer,string) }
          457,460:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argints(comma);
            argstrngs(rparent,1);
           end;
          { function(integer,integer,string) }
          574,577,1014,1017:
           begin
            ts := [ints];
            argints(comma);
            argstrngs(rparent,1);
           end;
          { function(integer,integer,string,string) }
          585,645:
           begin
            ts := [ints];
            argints(comma);
            argstrngs(comma,1);
            argstrngs(rparent,3);
           end;
          { function(integer,string,string) }
          765,968:
           begin
            ts := [ints];
            argstrngs(comma,1);
            argstrngs(rparent,3);
           end;
          { function(real,string) }
          463,472:
           begin
            ts := [ints,reals];
            if x.typ = ints then emit1(26,0);
            argstrngs(rparent,1);
           end;
          { function(string,string) }
          478,487,496,505,514,523,597,616,625,660,711,851,936,1191:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            argstrngs(rparent,3);
           end;
          { function(string,string,integer) }
          791,911,1182:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            argstrngs(comma,3);
            argints(rparent);
           end;
          { function(string,string,string,integer) }
          875:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            argstrngs(comma,3);
            argstrngs(comma,9);
            argints(rparent);
           end;
          { function(string,char) }
          902,905,908,930,933,948:
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            argchars(rparent);
           end;
          { function(integer,char,integer) }
          801:
           begin
            ts := [ints];
            argchars(comma);
            argints(rparent);
           end;
          { function(integer,char,real) }
          802:
           begin
            ts := [ints];
            argchars(comma);
            argreals(rparent);
           end;
          { function(integer,integer,char,integer) }
          1123:
           begin
            ts := [ints];
            argints(comma);
            argchars(comma);
            argints(rparent);
           end;
          { function(integer,integer,char,real) }
          1124,1125:
           begin
            ts := [ints];
            argints(comma);
            argchars(comma);
            argreals(rparent);
           end;
          { function(integer,integer,char,string) }
          1126:
           begin
            ts := [ints];
            argints(comma);
            argchars(comma);
            argstrngs(rparent,3);
           end;
          { function(char,integer) }
          803:
           begin
            ts := [chars];
            argints(rparent);
           end;
          { function(var string,integer) }
          804,806:
           begin
            ts := [strngs];
            if x.temp then FatalError(77);
            argints(rparent);
           end;
          { function(var string,integer,integer) }
          805:
           begin
            ts := [strngs];
            if x.temp then FatalError(77);
            argints(comma);
            argints(rparent);
           end;
          { function(var string,integer,real) }
          807:
           begin
            ts := [strngs];
            if x.temp then FatalError(77);
            argints(comma);
            argreals(rparent);
           end;
          812: { function(string,variant) - strfmt }
           begin
            ts := [strngs];
            if x.temp then n := n+1;
            argSTCIRs(rparent,2);
           end;
          { function(integer,variant) - fsm_set_state }
          1021:
           begin
            ts := [ints];
            argSTCIs(rparent,1);
           end;
          366: { dump }
           begin
            ts := [bools,chars,ints,reals];
            if x.typ = chars then n := n+1;
            if x.typ = ints  then n := n+2;
            if x.typ = reals then n := n+3;
           end;
          23: { copy }
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            test([comma], 59);
            if sy = comma then begin
             insymbol;
             expression(fsys+[comma,rparent],y);
             if y.typ <> ints then if y.typ <> notyp then FatalError(34);
             test([comma,rparent], 6);
             if sy = comma then begin
              insymbol;
              expression(fsys+[rparent],y);
              if y.typ <> ints then if y.typ <> notyp then FatalError(34);
             end else emit1(24,nmax);
            end;
           end;
          26: { pos }
           begin
            ts := [strngs,chars];
            if x.typ = chars then n := n+2 else if x.temp then n := n+1;
            test([comma], 59);
            if sy = comma then begin
             insymbol;
             expression(fsys+[rparent],y);
             if y.typ <> strngs
             then if y.typ <> notyp
             then FatalError(38) else else if y.temp then n := n+4;
            end
           end;
          33: { str }
           begin
            ts := [ints,reals];
            if x.typ = reals then n := n+1
           end;
          35,37: { val,rval }
           begin
            ts := [strngs];
            if x.temp then n := n+1
           end;
         end ; { case }
         if x.typ in ts then emit1(8,n) else if x.typ<>notyp then FatalError(48);
         x.typ := itab[i].typ;
         x.temp := true;
         if sy = rparent then insymbol else FatalError(4)
        end; { else }
       end;{ case }
      end; { standfct }
     begin  { factor }
      x.typ := notyp;
      x.ref := 0;
      test(facbegsys, 58);
      while sy in facbegsys do begin
       case sy of
        ident:
         begin
          i := loc(id);
          insymbol;
          with itab[i] do
          case obj of
           konstant:
            begin
             x.typ := typ;
             x.ref := 0;
             x.temp := false;
             if x.typ = reals then emit1(25,adr) else emit1(24,adr)
            end ;
           vvariable:
            begin
             x.typ := typ;
             x.ref := ref;
             x.temp := false;
             if sy in [lbrack,lparent,period] then begin
              if normal then f := 0 else f := 1;
              if x.typ=strngs then begin
               emit2(f+1,lev,adr);
               selector(fsys,x);
              end
              else begin
               emit2(f,lev,adr);
               selector(fsys,x);
               if x.typ in stantyps then emit(134);
              end
             end else begin
              if x.typ in stantyps
              then if normal then f := 1 else f := 2
              else if normal then f := 0 else f :=1;
              emit2(f, lev, adr)
             end
            end ;
           type1, prozedure:
            FatalError(44);
           funktion :
            begin
             x.typ := typ;
             x.temp := true;
             if lev <> 0 then call(fsys, i) else standfct(adr)
            end
          end  { case obj, with }
         end;   { ident }
        realcon:
         begin
          x.typ := reals;
          x.ref := 0;
          enterreal(rnum);
          emit1(25, ic1);
          insymbol
         end;
        charcon:
         begin
          x.typ := chars;
          x.ref := 0;
          x.temp := false;
          emit1(24, inum);
          insymbol
         end;
        intcon:
         begin
          x.typ := ints;
          x.ref := 0;
          emit1(24, inum);
          insymbol
         end;
        stringcon:
         begin
          x.typ := strngs;
          x.ref := 0;
          x.temp := false;
          emit1(24,StrRef(spnt));
          insymbol
         end;
        lparent:
         begin
          insymbol;
          expression(fsys+[rparent], x);
          if sy = rparent then insymbol else FatalError(4)
         end;
        notsy:
         begin
          insymbol;
          factor(fsys,x);
          if x.typ=bools then emit(135) else if x.typ<>notyp then FatalError(32)
         end;
       end;  { case sy }
       test(fsys, 6);
      end { while }
     end { factor } ;
    begin { term }
     factor(fsys+[times,rdiv,idiv,imod,andsy], x);
     while sy in [times,rdiv,idiv,imod,andsy] do begin
      op := sy;
      insymbol;
      factor(fsys+[times,rdiv,idiv,imod,andsy], y);
      if op = times then begin
       x.typ := resulttype(x.typ, y.typ);
       case x.typ of
        notyp: ;
        ints : emit(157);
        reals: emit(160);
       end
      end else
      if op = rdiv then begin
       if x.typ = ints then begin
        emit1(26,1);
        x.typ := reals
       end ;
       if y.typ = ints then begin
        emit1(26,0);
        y.typ := reals
       end ;
       if (x.typ=reals) and (y.typ=reals) then emit(161) else begin
        if (x.typ<>notyp) and (y.typ<>notyp) then FatalError(33);
        x.typ := notyp
       end
      end else
      if op = andsy then begin
       if (x.typ=bools) and (y.typ=bools) then emit(156) else begin
        if (x.typ<>notyp) and (y.typ<>notyp) then FatalError(32);
         x.typ := notyp
       end
      end else begin     { op in [idiv,imod] }
       if (x.typ=ints) and (y.typ=ints) then
       if op=idiv then emit(158) else emit(159)
       else begin
        if (x.typ<>notyp) and (y.typ<>notyp) then FatalError(34);
        x.typ := notyp
       end
      end
     end {while}
    end; { term }
   begin { simpleexpression }
    clrit(y);
    if sy in [plus,minus] then begin
     op := sy;
     insymbol;
     term(fsys+[plus,minus], x);
     if x.typ > reals then FatalError(33) else
     if op = minus then if x.typ = reals then emit(164) else emit(136)
    end else term(fsys+[plus,minus,orsy], x);
    while sy in [plus,minus,orsy] do begin
     op := sy;
     insymbol;
     term(fsys+[plus,minus,orsy], y);
     if op = orsy then begin
      if (x.typ=bools) and (y.typ=bools) then emit(151) else begin
       if (x.typ <> notyp) and (y.typ<>notyp) then FatalError(32);
       x.typ := notyp
      end
     end else
     if (x.typ = chars) or (x.typ = strngs) then begin
      if not((y.typ = chars) or (y.typ = strngs)) then begin
       FatalError(38);
       x.typ := notyp;
      end
      else begin
       if x.typ = chars then t := 0 else t := 1;
       if y.typ = strngs then t := t+2;
       if x.temp then t := t+4;
       if y.temp then t := t+8;
       emit1(7,t);
       x.typ := strngs;
       x.temp := true;
      end
     end
     else begin
      x.typ := resulttype(x.typ, y.typ);
      case x.typ of
       notyp: ;
       ints : if op = plus then emit(152) else emit(153);
       reals: if op = plus then emit(154) else emit(155)
      end {case}
     end
    end {while}
   end; { simpleexpression }
  begin { expression }
   clrit(y);
   simpleexpression(fsys+[eql,neq,lss,leq,gtr,geq], x);
   if sy in [eql,neq,lss,leq,gtr,geq] then begin
    op := sy;
    insymbol;
    simpleexpression(fsys, y);
    if (x.typ in [notyp,ints,bools,chars]) and (x.typ = y.typ) then
    case op of
     eql: emit(145);
     neq: emit(146);
     lss: emit(147);
     leq: emit(148);
     gtr: emit(149);
     geq: emit(150);
    end
    else begin
     if x.typ = ints then begin
      x.typ := reals;
      emit1(26,1)
     end else
     if y.typ = ints then begin
      y.typ := reals;
      emit1(26,0)
     end ;
     if (x.typ=reals) and (y.typ=reals) then
     case op of
      eql: emit(139);
      neq: emit(140);
      lss: emit(141);
      leq: emit(142);
      gtr: emit(143);
      geq: emit(144);
     end else
     if (x.typ in [chars,strngs]) and (y.typ in [chars,strngs]) then begin
      if x.typ=strngs then t := 1 else t := 0;
      if y.typ=strngs then t := t+2;
      if x.temp then t := t+4;
      if y.temp then t := t+8;
      if op in [eql,leq,geq] then t := t+16;
      if op in [neq,gtr,geq] then t := t+32;
      if op in [neq,lss,leq] then t := t+64;
      emit1(32,t);
     end
     else FatalError(35)
    end ;
    x.typ := bools
   end
  end; { expression }
  {}
  procedure assignment(lv,ad: integer);
  var x,y:item;  f:integer;
  begin              { tab[i].obj in [vvariable,funktion] }
   clrit(x); clrit(y);
   x.typ := itab[i].typ;
   x.ref := itab[i].ref;
   if itab[i].normal then f := 0 else f := 1;
   emit2(f, lv, ad);
   if sy in [lbrack,lparent,period] then
   if x.typ<>strngs then selector([becomes,eql]+fsys, x) else FatalError(60);
   if sy = becomes then insymbol else FatalError(51);
   expression(fsys, y);
   if x.typ = y.typ
    then if x.typ in stantyps
     then if x.typ=strngs
      then if y.temp
       then emit(166)
       else emit(169)
      else emit(138)
     else if x.ref <> y.ref
      then FatalError(46)
      else if x.typ = arrays
       then emit1(23,atab[x.ref].size)
       else emit1(23,btab[x.ref].vsize)
      else if (x.typ=reals) and (y.typ=ints)
       then begin
        emit1(26,0);
        emit(138)
       end
       else if (x.typ=chars) and (y.typ=strngs) then begin
        if y.temp then it := 8 else it := 0;
        emit1(31,it);
       end
       else if (x.typ=strngs) and (y.typ=chars)
        then emit(168)
        else if (x.typ=strngs) and (y.typ=arrays)
         then if atab[y.ref].eltyp = chars
          then begin
           emit1(167,atab[y.ref].size);
           emit(166)
          end
          else
         else if (x.typ=arrays) and (y.typ=strngs)
          then if atab[x.ref].eltyp = chars
           then if y.temp
            then emit1(175,atab[x.ref].size)
            else emit1(174,atab[x.ref].size)
           else
          else if (x.typ<>notyp) and (y.typ<>notyp)
           then FatalError(46)
  end; { assignment }
  {}
  procedure compoundstatement;
  begin
   insymbol;
   statement([semicolon,endsy]+fsys);
   while sy in [semicolon]+statbegsys do begin
    if sy = semicolon then insymbol else FatalError(14);
    statement([semicolon,endsy]+fsys)
   end ;
   if sy = endsy then insymbol else FatalError(57)
  end; { compoundstatement }
  {}
  procedure ifstatement;
  var x:item; lc1,lc2: integer;
  begin
   clrit(x);
   insymbol;
   expression(fsys+[thensy,dosy], x);
   if not (x.typ in [bools,notyp]) then FatalError(17);
   lc1 := lc;
   emit(11);     { jmpc }
   if sy = thensy then insymbol else FatalError(52);
   statement(fsys+[elsesy]);
   if sy = elsesy then begin
    insymbol;
    lc2 := lc;
    emit(10);
    ctab[lc1].y := lc;
    statement(fsys);
    ctab[lc2].y := lc
   end else ctab[lc1].y := lc
  end; { ifstatement }
  {}
  procedure casestatement;
  var x:item; i,j,k,lc1: integer;
      casetab: packed array [1..csmax] of packed record val, lc: index end;
      exittab: packed array [1..csmax] of integer;
   {}
   procedure caselabel;
   var lab:conrec; k:integer;
   begin
    constant(fsys+[comma,colon], lab);
    if lab.tp <> x.typ then FatalError(47) else
    if i = csmax then FatalError(68) else begin
     i := i+1;
     k := 0;
     casetab[i].val :=lab.i;
     casetab[i].lc  := lc;
     repeat
      k := k+1
     until casetab[k].val = lab.i;
     if k < i then FatalError(1);   { multiple definition }
    end
   end { caselabel } ;
   {}
   procedure onecase;
   begin
    if sy in constbegsys then begin
     caselabel;
     while sy = comma do begin
      insymbol;
      caselabel
     end ;
     if sy = colon then insymbol else FatalError(5);
     statement([semicolon,endsy]+fsys);
     j := j+1;
     exittab[j] := lc;
     emit(10)
    end
   end { onecase } ;
  begin {casestatement}
   clrit(x);
   insymbol;
   i := 0;
   j := 0;
   expression(fsys+[ofsy,comma,colon], x);
   if not (x.typ in [ints,bools,chars,notyp]) then FatalError(23);
   lc1 := lc;
   emit(12);  { jmpx }
   if sy = ofsy then insymbol else FatalError(8);
   onecase;
   while sy = semicolon do begin
    insymbol;
    onecase
   end ;
   ctab[lc1].y := lc;
   for k := 1 to i do begin
    emit1(13,casetab[k].val);
    emit1(13,casetab[k].lc)
   end ;
   emit1(10,lc+1); // ending case table
   for k := 1 to j do ctab[exittab[k]].y := lc;
   if sy = endsy then insymbol else FatalError(57)
  end; { casestatement }
  {}
  procedure repeatstatement;
  var x:item; lc1:integer;
  begin
   clrit(x);
   lc1 := lc;
   insymbol;
   statement([semicolon,untilsy]+fsys);
   while sy in [semicolon]+statbegsys do begin
    if sy = semicolon then insymbol else FatalError(14);
    statement([semicolon,untilsy]+fsys)
   end ;
   if sy = untilsy then begin
    insymbol;
    expression(fsys, x);
    if not (x.typ in [bools,notyp]) then FatalError(17);
    emit1(11, lc1)
   end else FatalError(53)
  end; { repeatstatement }
  {}
  procedure whilestatement;
  var x:item; lc1,lc2:integer;
  begin
   clrit(x);
   insymbol;
   lc1 := lc;
   expression(fsys+[dosy], x);
   if not (x.typ in [bools,notyp]) then FatalError(17);
   lc2 := lc;
   emit(11);
   if sy = dosy then insymbol else FatalError(54);
   statement(fsys);
   emit1(10,lc1);
   ctab[lc2].y := lc
  end { whilestatement } ;
  {}
  procedure forstatement;
  var cvt:types; x:item; i,f,lc1,lc2:integer;
  begin
   clrit(x);
   cvt:=notyp;
   insymbol;
   if sy = ident then begin
    i := loc(id);
    insymbol;
    if i = 0
    then cvt := ints
    else if itab[i].obj = vvariable
    then begin
     cvt := itab[i].typ;
     if itab[i].normal then f := 0 else f := 1;
     emit2(f, itab[i].lev, itab[i].adr);
     if not (cvt in [notyp,ints,bools,chars]) then FatalError(18)
    end else FatalError(37);
   end else FatalError(2);
   if sy = becomes then begin
    insymbol;
    expression([tosy,downtosy,dosy]+fsys, x);
    if x.typ <> cvt then FatalError(19);
   end else FatalError(51);
   f := 14;
   if sy in [tosy, downtosy] then begin
    if sy = downtosy then f := 16;
    insymbol;
    expression([dosy]+fsys, x);
    if x.typ <> cvt then FatalError(19)
   end else FatalError(55);
   lc1 := lc;
   emit(f);
   if sy = dosy then insymbol else FatalError(54);
   lc2 := lc;
   statement(fsys);
   emit1(f+1,lc2);
   ctab[lc1].y := lc
  end { forstatement } ;
  {}
  procedure standproc(n: integer);
  var i,f: integer; x,y: item;
  begin
   clrit(x); clrit(y);
   case n of
    1,2: { read }
     begin
      if sy = lparent then begin
       repeat
        insymbol;
        if sy <> ident then FatalError(2) else begin
         i := loc(id);
         insymbol;
         if i <> 0 then if itab[i].obj <> vvariable then FatalError(37)
         else begin
          x.typ := itab[i].typ;
          x.ref := itab[i].ref;
          if itab[i].normal then f := 0 else f := 1;
          emit2(f, itab[i].lev, itab[i].adr);
          if sy in [lbrack,lparent,period] then begin
           if x.typ=strngs then FatalError(60);
           selector(fsys+[comma,rparent], x);
          end;
          if x.typ in [ints,reals,chars,strngs,notyp]
          then emit1(27,ord(x.typ))
          else FatalError(41)
         end
        end ;
        test([comma,rparent], 6);
       until sy <> comma;
       if sy = rparent then insymbol else FatalError(4)
      end ;
      if n = 2 then emit(162)
     end ;
    3,4: { write }
     begin
      if sy = lparent then begin
       repeat
        insymbol;
        expression(fsys+[comma,colon,rparent], x);
        if not (x.typ in stantyps) then FatalError(41);
        if sy = colon then begin
         insymbol;
         expression(fsys+[comma,colon,rparent], y);
         if y.typ <> ints then FatalError(43);
         if sy = colon then begin
          if x.typ <> reals then FatalError(42);
          insymbol;
          expression(fsys+[comma,rparent], y);
          if y.typ <> ints then FatalError(43);
          emit(137)
         end else begin
         if x.typ=strngs
          then if x.temp then emit(177) else emit(176)
          else emit1(30, ord(x.typ))
         end
        end else if x.typ=strngs
                 then if x.temp then emit(171)
                                else emit(170)
                 else emit1(29, ord(x.typ))
       until sy <> comma;
       if sy = rparent then insymbol else FatalError(4)
      end ;
      if n = 4 then emit(163)
     end ; {write}
   end { case }
  end; { standproc }
 begin { statement }
  if sy in statbegsys+[ident] then
  case sy of
   ident:
    begin
     i := loc(id);
     insymbol;
     if i <> 0 then
     case itab[i].obj of
      konstant, type1 : FatalError(45);
      vvariable       : assignment(itab[i].lev, itab[i].adr);
      prozedure       : if itab[i].lev <> 0
                        then call(fsys, i)
                        else standproc(itab[i].adr);
      funktion        : if itab[i].ref = display[level]
                        then assignment(itab[i].lev+1, 0)
                        else FatalError(45)
     end {case}
    end ;
   beginsy  : compoundstatement;
   ifsy     : ifstatement;
   casesy   : casestatement;
   whilesy  : whilestatement;
   repeatsy : repeatstatement;
   forsy    : forstatement;
  end; {case}
  test(fsys, 14)
 end { statement } ;
begin { block }
 dx := 6;
 prt := it;
 if level > levmax then FatalError(67);
 test([lparent,colon,semicolon], 14);
 enterblock;
 prb := ib;
 display[level] := ib;
 itab[prt].typ := notyp;
 itab[prt].ref := prb;
 if (sy = lparent) and (level > 1) then parameterlist;
 btab[prb].lastpar := it;
 btab[prb].psize := dx;
 if isfun then if sy = colon then begin
  insymbol;   { function type }
  if sy = ident then begin
   x := loc(id);
   insymbol;
   if x <> 0 then if itab[x].obj <> type1 then FatalError(29) else
   if itab[x].typ in stantyps then itab[prt].typ := itab[x].typ else FatalError(15)
  end else FatalError(2)
 end else FatalError(5);
 if sy = semicolon then insymbol else FatalError(14);
 repeat
  if sy = constsy then constdec;
  if sy = typesy then typedeclaration;
  if sy = varsy then variabledeclaration;
  btab[prb].vsize := dx;
  while sy in [procsy,funcsy] do procdeclaration;
  test([beginsy], 56)
 until sy in statbegsys;
 itab[prt].adr := lc;
 insymbol;
 statement([semicolon,endsy]+fsys);
 while sy in [semicolon]+statbegsys do begin
  if sy = semicolon then insymbol else FatalError(14);
  statement([semicolon,endsy]+fsys)
 end ;
 if sy = endsy then insymbol else FatalError(57);
 test(fsys+[period], 6)
end; { block }
 {
 *************************
 END Block syntax analizer
 *************************
 }

 {
 **************
 User functions
 **************
 }
function CurveByRef(ref:real):TCurve;
begin
 Result:=TCurve(ObjectRegistry[Round(ref)]);
 if TObject(Result) is TCurve then {Ok} else Result:=nil;
end;

function RefByCurve(Curve:TCurve):Integer;
begin
 Result:=Curve.Ref;
end;

function CalibrByRef(ref:real):TPolynomCalibration;
begin
 Result:=TPolynomCalibration(ObjectRegistry[Round(ref)]);
 if TObject(Result) is TPolynomCalibration then {Ok} else Result:=nil;
end;

function RefByCalibr(Calibr:TPolynomCalibration):Integer;
begin
 Result:=Calibr.Ref;
end;

function TimerByRef(ref:real):TIntervalTimer;
begin
 Result:=TIntervalTimer(ObjectRegistry[Round(ref)]);
 if TObject(Result) is TIntervalTimer then {Ok} else Result:=nil;
end;

function RefByTimer(Timer:TIntervalTimer):Integer;
begin
 Result:=Timer.Ref;
end;

function TextByRef(ref:Integer):TText;
begin
 Result:=TText(ObjectRegistry[ref]);
 if TObject(Result) is TText then {Ok} else Result:=nil;
end;

function RefByText(Txt:TText):Integer;
begin
 Result:=Txt.Ref;
end;

function TDaqPascal._maxavail:integer;
begin
 _maxavail:=stab.space;
end;

function TDaqPascal._msecnow:real;
begin
 _msecnow:=_crw_rtc.msecnow;
end;

function TDaqPascal._secnow:real;
begin
 _secnow:=_crw_rtc.msecnow*1e-3;
end;

function TDaqPascal._getticks:real;
begin
 _getticks:=_crw_rtc.msecnow * TicksPerMSec;
end;

function TDaqPascal._time:real;
begin
 _time:=Daq.Timer.LocalTime;
end;

function TDaqPascal._timeunits:real;
begin
 _timeunits:=Daq.Timer.LocalTimeUnits;
end;

function TDaqPascal._numais:integer;
begin
 _numais:=TheDevice.NumAnalogInputs;
end;

function TDaqPascal._numdis:integer;
begin
 _numdis:=TheDevice.NumDigitalInputs;
end;

function TDaqPascal._numaos:integer;
begin
 _numaos:=TheDevice.NumAnalogOutputs;
end;

function TDaqPascal._numdos:integer;
begin
 _numdos:=TheDevice.NumDigitalOutputs;
end;

function TDaqPascal._numcals:integer;
begin
 _numcals:=TheDevice.NumCalibrations;
end;

function TDaqPascal._refai(n:integer):integer;
begin
 _refai:=RefByCurve(TheDevice.AnalogInputCurve[n]);
end;

function TDaqPascal._refdi(n:integer):integer;
begin
 _refdi:=RefByCurve(TheDevice.DigitalInputCurve[n]);
end;

function TDaqPascal._refao(n:integer):integer;
begin
 _refao:=RefByCurve(TheDevice.AnalogOutputCurve[n]);
end;

function TDaqPascal._refdo(n:integer):integer;
begin
 _refdo:=RefByCurve(TheDevice.DigitalOutputCurve[n]);
end;

function TDaqPascal._refcalibr(n:integer):integer;
begin
 _refcalibr:=RefByCalibr(TheDevice.Calibration[n]);
end;

function TDaqPascal._getai_n(n:integer):real;
begin
 _getai_n:=TheDevice.AnalogInputCurve[n].Count;
end;

function TDaqPascal._getdi_n(n:integer):real;
begin
 _getdi_n:=TheDevice.DigitalInputCurve[n].Count;
end;

function TDaqPascal._crvlen(ref:real):real;
begin
 _crvlen:=CurveByRef(ref).Count;
end;

function TDaqPascal._crvx(ref,i:real):real;
begin
 _crvx:=CurveByRef(ref)[round(i)-1].x;
end;

function TDaqPascal._crvy(ref,i:real):real;
begin
 _crvy:=CurveByRef(ref)[round(i)-1].y;
end;

function TDaqPascal._getai_xn(n:integer):real;
begin
 _getai_xn:=TheDevice.AnalogInputCurve[n].LastPoint.X;
end;

function TDaqPascal._getai_yn(n:integer):real;
begin
 _getai_yn:=TheDevice.AnalogInputCurve[n].LastPoint.Y;
end;

function TDaqPascal._getdi_xn(n:integer):real;
begin
 _getdi_xn:=TheDevice.DigitalInputCurve[n].LastPoint.X;
end;

function TDaqPascal._getdi_yn(n:integer):real;
begin
 _getdi_yn:=TheDevice.DigitalInputCurve[n].LastPoint.Y;
end;

function TDaqPascal._getai(n:integer;t:real):real;
begin
 _getai:=TheDevice.SmoothAnalogInputCurve(n,t);
end;

function TDaqPascal._getdi(n:integer;t:real):real;
var c:TCurve;
begin
 _getdi:=0;
 c:=TheDevice.DigitalInputCurve[n];
 if Assigned(c) then begin
  c.Lock;
  if (c.Count>0) and (t>=0) then begin
   if t>=c.LastPoint.X
   then _getdi:=c.LastPoint.Y
   else _getdi:=c[c.GetIndexAt(t)].Y;
  end;
  c.Unlock;
 end;
end;

function TDaqPascal._getai_xi(n:integer;i:real):real;
begin
 _getai_xi:=TheDevice.AnalogInputCurve[n][round(i)-1].X;
end;

function TDaqPascal._getai_yi(n:integer;i:real):real;
begin
 _getai_yi:=TheDevice.AnalogInputCurve[n][round(i)-1].Y;
end;

function TDaqPascal._getdi_xi(n:integer;i:real):real;
begin
 _getdi_xi:=TheDevice.DigitalInputCurve[n][round(i)-1].X;
end;

function TDaqPascal._getdi_yi(n:integer;i:real):real;
begin
 _getdi_yi:=TheDevice.DigitalInputCurve[n][round(i)-1].Y;
end;

function TDaqPascal._crvput(ref,i,x,y:real):boolean;
var c:TCurve; j:longint;
begin
 _crvput:=false;
 c:=CurveByRef(ref);
 j:=round(i);
 if Assigned(c) then begin
  c.Lock;
  _crvput:=(j>=1) and (j<=c.Count+1);
  c[j-1]:=Point2D(x,y);
  c.Unlock;
 end;
end;

function TDaqPascal._crvinteg(ref,a,b:real):real;
begin
 _crvinteg:=DaqIntegral(CurveByRef(ref),a,b);
end;

function TDaqPascal._putev(w,c:integer; t,d0,d1:real):boolean;
begin
 _putev:=TheDevice.PutDaqEvent(DaqEvent(w,c,t,d0,d1));
end;

function TDaqPascal._putao(n:integer; t,d:real):boolean;
begin
 _putao:=TheDevice.PutDaqEvent(DaqEvent(evAnalog+evCompress,n,t,d));
end;

function TDaqPascal._putdo(n:integer; t,d:real):boolean;
begin
 _putdo:=TheDevice.PutDaqEvent(DaqEvent(evDigital,n,t,d));
end;

function TDaqPascal._calibr(n:integer; t,d:real):real;
begin
 _calibr:=TheDevice.Transform(n,t,d);
end;

function TDaqPascal._fixerror(n:integer):boolean;
begin
 _fixerror:=(n=Byte(n));
 TheDevice.FixError(n);
end;

procedure _assign(var f:text; fname:LongString; fifo:TFifo);
begin
 fname:=UpcaseStr(Trim(fname));
 if SameText(fname,'NULL:') then AssignNull(f) else
 if fname='' then AssignFifo(f,fifo) else
 if SameText(fname,'CON:') then begin
  AssignFifo(f,fifo);
  TTextRec(f).Name:='CON:';
 end else begin
  fname:=UnifyFileAlias(AdaptFileName(fname));
  Assign(f,fname);
 end;
end;

function TDaqPascal._reset(const arg:LongString):integer;
var SaveFileMode:byte;
begin
 SetInOutRes(0);
 SmartFileClose(StdInp);
 _assign(StdInp,arg,InpFifo);
 SaveFileMode:=System.FileMode;
 System.FileMode:=0;
 System.Reset(StdInp);
 System.FileMode:=SaveFileMode;
 _reset:=IOResult;
 IORes:=0;
end;

function TDaqPascal._rewrite(const arg:LongString):integer;
var SaveFileMode:byte;
begin
 SetInOutRes(0);
 SmartFileClose(StdOut);
 _assign(StdOut,arg,OutFifo);
 SaveFileMode:=System.FileMode;
 System.FileMode:=1;
 System.Rewrite(StdOut);
 System.FileMode:=SaveFileMode;
 _rewrite:=IOResult;
 IORes:=0;
end;

function TDaqPascal._append(const arg:LongString):integer;
var SaveFileMode:byte;
begin
 SetInOutRes(0);
 SmartFileClose(StdOut);
 _assign(StdOut,arg,OutFifo);
 SaveFileMode:=System.FileMode;
 System.FileMode:=1;
 System.Append(StdOut);
 System.FileMode:=SaveFileMode;
 _append:=IOResult;
 IORes:=0;
end;

function TDaqPascal._ioresult:integer;
begin
 _ioresult:=IORes;
 IORes:=0;
end;

function TDaqPascal._Flush:integer;
begin
 _Flush:=0;
 if not IsFileClosed(StdOut) then begin
  Flush(StdOut);
  _Flush:=IOResult;
 end;
end;

procedure TDaqPascal.ClearFifo;
begin
 if Assigned(InpFifo) then InpFifo.Clear;
 if Assigned(OutFifo) then OutFifo.Clear;
end;

const
 arg_delims=[' ',',',#9];

 {
 Ссылка на SoftwareDevice, связанного с DaqPascal
 }
function TheScript(Who:TDaqPascal):TSoftwareDevice;
begin
 Result:=TSoftwareDevice(Who.TheDevice);
end;

 {
 Поиск окна с данным именем
 }
type
 TFindWinRec = packed record
  Caption : LongString;
  Found   : TForm;
 end;

procedure CheckWinName(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if Form is TForm then
 with TFindWinRec(Custom^) do
 if SameText(UnifyAlias(Form.Caption),UnifyAlias(Caption)) then begin
  Found:=Form;
  Terminate:=true;
 end;
end;

function FindWin(const aName:LongString):TForm;
var FindWinRec:TFindWinRec;
begin
 Result:=nil;
 FindWinRec:=Default(TFindWinRec);
 try
  FindWinRec.Caption:=StringBuffer(aName);
  FindWinRec.Found:=nil;
  ForEachForm(CheckWinName, @FindWinRec);
  Result:=FindWinRec.Found;
 finally
  FindWinRec.Caption:='';
 end;
end;

procedure CheckWinCrv(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if Form is TFormCurveWindow then begin
  TForm(Custom^):=Form;
  Terminate:=true;
 end;
end;

function FindFirstCurveWin:TFormCurveWindow;
begin
 Result:=nil;
 SdiMan.ForEachChildInZOrder(CheckWinCrv, @Result);
end;

 {
 Найти спектрометрическое окно по имени в списке окон DAQ
 }
function FindSpecWin(const aName:LongString):TFormSpectrWindow;
var
 i : Integer;
begin
 Result:=nil;
 for i:=0 to Daq.SpeWinList.Count-1 do
 with Daq.SpeWinList[i] do if Ok then
 if SameText(UnifyAlias(Caption),UnifyAlias(aName)) then begin
  Result:=Daq.SpeWinList[i];
  break;
 end;
end;

function crvfind(path:LongString):real;
var name:LongString; w:TFormCurveWindow; c:TCurve;
 function FindCurveInWindow(aWin:TFormCurveWindow; aName:LongString):TCurve;
 var i:integer;
 begin
  Result:=nil;
  if aWin.Ok then
  for i:=0 to aWin.Curves.Count-1 do
  if aWin.Curves[i].Ok and SameText(UnifyAlias(aWin.Curves[i].Name),UnifyAlias(aName)) then begin
   FindCurveInWindow:=aWin.Curves[i];
   break;
  end;
 end;
begin
 c:=nil;
 path:=TrimChars(path,ScanSpaces,ScanSpaces);
 name:=ExtractWord(1,path,ScanSpaces);
 if name<>'' then begin
  System.delete(path,1,length(name));
  path:=Trim(path);
  if path<>'' then begin
   if path='*'
   then w:=ActiveCurveWindow
   else begin
    if path='**'
    then w:=FindFirstCurveWin
    else w:=TFormCurveWindow(FindWin(path));
    if not (w is TFormCurveWindow) then w:=nil;
   end;
   if w is TFormCurveWindow then begin
    if name='*'
    then c:=w.DefCurve
    else c:=FindCurveInWindow(w,name);
   end;
  end else begin
   c:=Daq.Curves.Find(name);
  end;
 end;
 crvfind:=RefByCurve(c);
end;

function TDaqPascal._voice(const arg:LongString):boolean;
begin
 _voice:=Daq.SoundOn;
 Daq.Voice(arg);
end;

function TDaqPascal._action(const arg:LongString):boolean;
var
 i      : Integer;
 Device : TDaqDevice;
begin
 Result:=true;
 for i:=1 to WordCount(arg,arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,arg,arg_delims));
  if Device.Ok
  then Result:=TheScript(Self).PostDeferredCommand('@ACTION '+Device.Name+EOL) and Result
  else Result:=false;
 end;
end;

function TDaqPascal._clear(const arg:LongString):boolean;
var
 i      : Integer;
 Device : TDaqDevice;
begin
 Result:=true;
 for i:=1 to WordCount(arg,arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,arg,arg_delims));
  if Device.Ok
  then Result:=TheScript(Self).PostDeferredCommand('@CLEAR '+Device.Name+EOL) and Result
  else Result:=false;
 end;
end;

function TDaqPascal._cleardevice(const arg:LongString):boolean;
var
 i      : Integer;
 Device : TDaqDevice;
begin
 Result:=true;
 for i:=1 to WordCount(arg,arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,arg,arg_delims));
  if Device.Ok
  then Result:=TheScript(Self).PostDeferredCommand('@CLEARDEVICE '+Device.Name+EOL) and Result
  else Result:=false;
 end;
end;

function TDaqPascal._start(const arg:LongString):boolean;
var
 i      : Integer;
 Device : TDaqDevice;
begin
 Result:=true;
 for i:=1 to WordCount(arg,arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,arg,arg_delims));
  if Device.Ok
  then Result:=TheScript(Self).PostDeferredCommand('@START '+Device.Name+EOL) and Result
  else Result:=false;
 end;
end;

function TDaqPascal._stop(const arg:LongString):boolean;
var
 i      : Integer;
 Device : TDaqDevice;
begin
 Result:=true;
 for i:=1 to WordCount(arg,arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,arg,arg_delims));
  if Device.Ok
  then Result:=TheScript(Self).PostDeferredCommand('@STOP '+Device.Name+EOL) and Result
  else Result:=false;
 end;
end;

 {147}
function TDaqPascal._devsendmsg(const arg:LongString):real;
var Device:TDaqDevice; name,msg:LongString;
begin
 Result:=0;
 name:=ExtractWord(1,arg,arg_delims);
 Device:=FullDaqDeviceList.Find(name);
 if Assigned(Device) then begin
  msg:=SkipWords(1,arg,arg_delims);
  Result:=Device.HandleMessage(msg);
 end;
end;

function TDaqPascal._debugout(const arg:LongString):boolean;
begin
 _debugout:=false;
 if not _isISRnow then begin
  _debugout:=true;
  _crw_fio.DebugOut(stdfDebug,arg);
 end;
end;

function TDaqPascal._clearcurve(const arg:LongString):boolean;
var
 i       : Integer;
 d       : Integer;
 Curve   : TCurve;
 HistLen : Integer;
begin
 Result:=true;
 HistLen:=0;
 for i:=1 to WordCount(arg,arg_delims) do
 if Str2Int(ExtractWord(i,arg,arg_delims),d) then HistLen:=d else begin
  Curve:=Daq.Curves.Find(ExtractWord(i,arg,arg_delims));
  if Curve.Ok
  then DaqClearCurve(Curve,HistLen)
  else Result:=false;
 end;
end;

 {
 выполняет сохранение списка кривых в файл CRW
 @savecrw fname winname wintitle winlable curvelist
 fname      - имя файла (можно без пути и без расширения)
 winname    - имя окна под которым оно попадет в crw-архив
 wintitle   - заголовок в верхней части окна
 winlable   - метка в нижней части окна
 curvelist  - список сохраняемых кривых
 }
function TDaqPascal._savecrw(const arg:LongString):boolean;
begin
 Result:=TheScript(Self).PostDeferredCommand('@SAVECRW '+arg+EOL);
end;

function TDaqPascal._specmarker(const arg:LongString):real;
begin
 Result:=FindSpecWin(arg).Marker;
end;

function TDaqPascal._specmarkerl(const arg:LongString):real;
begin
 Result:=FindSpecWin(arg).MarkerL;
end;

function TDaqPascal._specmarkerr(const arg:LongString):real;
begin
 Result:=FindSpecWin(arg).MarkerR;
end;

function TDaqPascal._specroil(const arg:LongString):real;
begin
 Result:=FindSpecWin(arg).RoiL;
end;

function TDaqPascal._specroir(const arg:LongString):real;
begin
 Result:=FindSpecWin(arg).RoiR;
end;

function TDaqPascal._windraw(const arg:LongString):boolean;
begin
 Result:=TheScript(Self).PostDeferredCommand('@WINDRAW '+arg+EOL);
end;

function TDaqPascal._winshow(const arg:LongString):boolean;
begin
 Result:=TheScript(Self).PostDeferredCommand('@WINSHOW '+arg+EOL);
end;

function TDaqPascal._winhide(const arg:LongString):boolean;
begin
 Result:=TheScript(Self).PostDeferredCommand('@WINHIDE '+arg+EOL);
end;

function TDaqPascal._winselect(const arg:LongString):boolean;
begin
 Result:=TheScript(Self).PostDeferredCommand('@WINSELECT '+arg+EOL);
end;

function TDaqPascal._global(const arg:LongString):real;
begin
 Result:=SystemCalculator.Eval(arg);
end;

function TDaqPascal._tm_new:integer;
var tm:TIntervalTimer;
begin
 tm:=NewIntervalTimer(tmCyclic,nil);
 timers.Add(tm);
 _tm_new:=RefByTimer(tm);
end;

function TDaqPascal._tm_free(reftm:real):boolean;
var tm:TIntervalTimer;
begin
 _tm_free:=false;
 tm:=TimerByRef(reftm);
 if tm<>nil then begin
  timers.Remove(tm);
  _tm_free:=true;
 end;
end;

function TDaqPascal._tm_addint(reftm,ms:real):boolean;
var tm:TIntervalTimer;
begin
 _tm_addint:=false;
 tm:=TimerByRef(reftm);
 if tm<>nil then begin
  tm.AddIntervalMs(ms,tm.NumIntervals+1);
  _tm_addint:=true;
 end;
end;

function TDaqPascal._tm_numint(reftm:real):integer;
var tm:TIntervalTimer;
begin
 _tm_numint:=0;
 tm:=TimerByRef(reftm);
 if tm<>nil then _tm_numint:=tm.NumIntervals;
end;

function TDaqPascal._tm_getint(reftm:real;nint:integer):real;
var tm:TIntervalTimer;
begin
 _tm_getint:=0;
 tm:=TimerByRef(reftm);
 if tm<>nil then _tm_getint:=tm.IntervalMs[nint];
end;

function TDaqPascal._tm_setint(reftm:real;nint:integer;ms:real):boolean;
var tm:TIntervalTimer;
begin
 _tm_setint:=false;
 tm:=TimerByRef(reftm);
 if tm<>nil then begin
  tm.IntervalMs[nint]:=ms;
  _tm_setint:=true;
 end;
end;

function TDaqPascal._tm_gettime(reftm:real):real;
var tm:TIntervalTimer;
begin
 _tm_gettime:=0;
 tm:=TimerByRef(reftm);
 if tm<>nil then _tm_gettime:=tm.LocalTime;
end;

function TDaqPascal._tm_start(reftm:real):boolean;
var tm:TIntervalTimer;
begin
 _tm_start:=false;
 tm:=TimerByRef(reftm);
 if tm<>nil then begin
  tm.start;
  _tm_start:=true;
 end;
end;

function TDaqPascal._tm_stop(reftm:real):boolean;
var tm:TIntervalTimer;
begin
 _tm_stop:=false;
 tm:=TimerByRef(reftm);
 if tm<>nil then begin
  tm.stop;
  _tm_stop:=true;
 end;
end;

function TDaqPascal._tm_isstart(reftm:real):boolean;
var tm:TIntervalTimer;
begin
 _tm_isstart:=false;
 tm:=TimerByRef(reftm);
 if tm<>nil then _tm_isstart:=tm.isstart;
end;

function TDaqPascal._tm_event(reftm:real):boolean;
var tm:TIntervalTimer;
begin
 _tm_event:=false;
 tm:=TimerByRef(reftm);
 if tm<>nil then _tm_event:=tm.Event;
end;

function TDaqPascal._tm_curint(reftm:real):integer;
var tm:TIntervalTimer;
begin
 _tm_curint:=0;
 tm:=TimerByRef(reftm);
 if tm<>nil then _tm_curint:=tm.CurrentInterval;
end;

function eBrokenPipe(Code:Integer):Boolean;
begin
 {$IFDEF WINDOWS}
 Result:=(Code=ERROR_BROKEN_PIPE);
 {$ENDIF ~WINDOWS}
 {$IFDEF UNIX}
 Result:=(Code=ESysEPIPE);
 {$ENDIF ~UNIX}
end;

function eOperationAborted(Code:Integer):Boolean;
begin
 {$IFDEF WINDOWS}
 Result:=(Code=ERROR_OPERATION_ABORTED);
 {$ENDIF ~WINDOWS}
 {$IFDEF UNIX}
 Result:=(Code=ESysECONNABORTED);
 {$ENDIF ~UNIX}
end;

procedure ComPipeReport(Pipe:TPipe; When:Double; const What:LongString; Code:Integer);
begin
 if Code<>ERROR_SUCCESS then
 if TObject(Pipe) is TPipe then
 if TObject(Pipe.Polling.LinkCustom) is TDaqPascal then
 with TDaqPascal(Pipe.Polling.LinkCustom) do begin
  if (Code=-1) then TheDevice.FixError(ecPipeOver) else begin
   if eBrokenPipe(Code) {and Pipe.Server} then {Ok} else
   if eOperationAborted(Code) {and Pipe.Server} then {Ok} else
   TheDevice.FixError(ecPipeIO);
   if DebugMode then OutFifo.PutText(Format('%s - %s("%s",%d="%s")%s',
   [StdDateTimeStr(When),What,Pipe.Polling.Name,Code,SysErrorMessage(Code),EOL]));
  end;
 end;
end;

 {207}
function  TDaqPascal._ComOpen(const arg:LongString):boolean;
var s:LongString; n:Integer; buf:TParsingBuffer;
begin
 Result:=false;
 _ComClose;
 s:=''; n:=0;
 if ReadIniFileString(Daq.ConfigFile,arg,'PipeLine%s',s) and IsNonEmptyStr(s) then begin
  if (Pos('%',s)>0) then s:=ExpEnv(s);
  ComIO.PipeLine:=_pipe_init(s);
  Result:=(_pipe_ref(ComIO.PipeLine)<>0);
  Exit;
 end;
 ComIO.Pipe:=NewPipeFromIniFile(Daq.ConfigFile,arg,ComPipeReport);
 if Assigned(ComIO.Pipe) then begin
  Result:=true;
  ComIO.Pipe.Master:=@ComIO.Pipe;
  ComIO.Pipe.Polling.LinkCustom:=Self;
 end else 
 if ReadIniFileAlpha(Daq.ConfigFile,arg,'Port%a',s) then
 if ScanVarInteger(svUpCase,StrCopyBuff(buf,s),'COM%i',n)<>nil then
 if uart[n].OpenConfig(Daq.ConfigFile,arg) then begin
  ComIO.PortN:=n;
  Result:=uart[ComIO.PortN].Active;
 end;
end;

 {210}
function  TDaqPascal._ComClose:boolean;
begin
 Result:=false;
 if (ComIO.PipeLine<>0) then begin
  Result:=_pipe_free(ComIO.PipeLine);
  ComIO.PipeLine:=0;
 end;
 if Assigned(ComIO.Pipe) then begin
  Result:=true;
  Kill(ComIO.Pipe);
 end;
 if (ComIO.PortN<>0) then begin
  Result:=uart[ComIO.PortN].Active;
  uart[ComIO.PortN].Close;
  ComIO.PortN:=0;
 end;
end;

 {416}
function  TDaqPascal._ComClear:boolean;
begin
 Result:=false;
 if (ComIO.PipeLine<>0) then begin
  if (_pipe_connected(ComIO.PipeLine)>0)
  then Result:=_pipe_txclear(_pipe_stream(ComIO.PipeLine,0));
 end else
 if Assigned(ComIO.Pipe) then begin
  Result:=true;
  ComIO.Pipe.TxFifo.Clear;
 end else
 if (ComIO.PortN<>0) then begin
  Result:=uart[ComIO.PortN].Active;
  uart[ComIO.PortN].TxClear;
 end;
end;

 {211}
function  TDaqPascal._ComCount:integer;
begin
 if (ComIO.PipeLine<>0) then begin
  if (_pipe_connected(ComIO.PipeLine)>0)
  then Result:=_pipe_rxcount(_pipe_stream(ComIO.PipeLine,0))
  else Result:=-1;
 end else
 if Assigned(ComIO.Pipe) then Result:=ComIO.Pipe.RxFifo.Count else
 if uart[ComIO.PortN].Active
 then Result:=uart[ComIO.PortN].RxCount
 else Result:=-1;
end;

 {415}
function  TDaqPascal._ComSpace:integer;
begin
 if (ComIO.PipeLine<>0) then begin
  if (_pipe_connected(ComIO.PipeLine)>0)
  then Result:=_pipe_txspace(_pipe_stream(ComIO.PipeLine,0))
  else Result:=-1;
 end else
 if Assigned(ComIO.Pipe) then Result:=ComIO.Pipe.TxFifo.Space else
 if uart[ComIO.PortN].Active
 then Result:=uart[ComIO.PortN].TxSpace
 else Result:=-1;
end;

 {212}
function  TDaqPascal._ComWrite(const arg:LongString):boolean;
begin
 if (ComIO.PipeLine<>0) then begin
  if (_pipe_connected(ComIO.PipeLine)>0)
  then Result:=_pipe_send(_pipe_stream(ComIO.PipeLine,0),arg)>0
  else Result:=false;
 end else
 if Assigned(ComIO.Pipe) then Result:=ComIO.Pipe.TxFifo.PutText(arg) else
 Result:=uart[ComIO.PortN].WriteStr(arg);
end;

 {215}
function  TDaqPascal._ComRead(Count:integer):LongString;
begin
 if (ComIO.PipeLine<>0) then begin
  if (_pipe_connected(ComIO.PipeLine)>0)
  then Result:=_pipe_recv(_pipe_stream(ComIO.PipeLine,0),max(0,min(Count,255)))
  else Result:='';
 end else
 if Assigned(ComIO.Pipe) then Result:=ComIO.Pipe.RxFifo.GetText(max(0,min(Count,255))) else
 Result:=uart[ComIO.PortN].ReadStr(max(0,min(Count,255)));
end;

function TDaqPascal._aimap(i,n:integer):real;
begin
 _aimap:=TheDevice.ConstructAnalogInputMap(i,n);
end;

function TDaqPascal._dimap(i,n:integer):real;
begin
 _dimap:=TheDevice.ConstructDigitalInputMap(i,n);
end;

function TDaqPascal._aomap(i,n:integer):real;
begin
 _aomap:=TheDevice.ConstructAnalogOutputMap(i,n);
end;

function TDaqPascal._domap(i,n:integer):real;
begin
 _domap:=TheDevice.ConstructDigitalOutputMap(i,n);
end;

function TDaqPascal._diword(i,n:integer):real;
begin
 _diword:=TheDevice.ConstructDigitalInputWord(i,n);
end;

function TDaqPascal._irqinit(n:integer):boolean;
begin
 _irqinit:=false; // ??
end;

function TDaqPascal._irqfree:boolean;
begin
 _irqfree:=false; // ??
end;

function TDaqPascal._isisrnow:boolean;
begin
 _isisrnow:=false; // ??
end;

function TDaqPascal._readini(s:LongString):LongString;
var cfg,sec,nam:LongString;
begin
 _readini:='';
 if _isISRnow then exit;
 case WordCount(s,ScanSpaces) of
  1:begin
     cfg:=Daq.ConfigFile;
     sec:='['+TheDevice.Name+']';
     nam:=ExtractWord(1,s,ScanSpaces);
    end;
  2:begin
     cfg:=Daq.ConfigFile;
     sec:=ExtractWord(1,s,ScanSpaces);
     nam:=ExtractWord(2,s,ScanSpaces);
    end;
  3:begin
     cfg:=Daq.FileRef(ExtractWord(1,s,ScanSpaces),'.cfg');
     sec:=ExtractWord(2,s,ScanSpaces);
     nam:=ExtractWord(3,s,ScanSpaces);
    end;
  else exit;
 end;
 if pos('%',nam)>0 then exit;
 if ReadIniFileAlpha(cfg,sec,nam+'%a',s) then _readini:=s;
end;

function TDaqPascal._caminitpkk(p,b,i,c:integer):boolean;
begin
 _caminitpkk:=false; // ??
end;

function TDaqPascal._camdonepkk:boolean;
begin
 _camdonepkk:=false; // ??
end;

function TDaqPascal._camtypepkk:integer;
begin
 _camtypepkk:=0; // ??
end;

function TDaqPascal._camnaf(c,n,a,f:integer):boolean;
begin
 _camnaf:=false; // ??
end;

function TDaqPascal._camwrite(c,n,a,f,d:integer):boolean;
begin
 _camwrite:=false; // ??
end;

function TDaqPascal._camwritelong(c,n,a,f:integer;d:real):boolean;
begin
 _camwritelong:=false; // ??
end;

function TDaqPascal._camread(c,n,a,f:integer):integer;
begin
 _camread:=0; // ??
end;

function TDaqPascal._camreadlong(c,n,a,f:integer):real;
begin
 _camreadlong:=0; // ??
end;

function TDaqPascal._camdeclare(c:integer):boolean;
begin
 _camdeclare:=false; // ??
end;

function TDaqPascal._camzero(c:integer):boolean;
begin
 _camzero:=false; // ??
end;

function TDaqPascal._camclear(c:integer):boolean;
begin
 _camclear:=false; // ??
end;

function TDaqPascal._camsetinhib(c:integer):boolean;
begin
 _camsetinhib:=false; // ??
end;

function TDaqPascal._camclrinhib(c:integer):boolean;
begin
 _camclrinhib:=false; // ??
end;

function TDaqPascal._camsetlmask(c,m:integer):boolean;
begin
 _camsetlmask:=false; // ??
end;

function TDaqPascal._camsetrmask(c,m:integer):boolean;
begin
 _camsetrmask:=false; // ??
end;

function TDaqPascal._camgetlmask(c:integer):integer;
begin
 _camgetlmask:=0; // ??
end;

function TDaqPascal._camgetrword(c:integer):integer;
begin
 _camgetrword:=0; // ??
end;

function TDaqPascal._camgetbcard(c:integer):integer;
begin
 _camgetbcard:=0; // ??
end;

function TDaqPascal._camisx(c:integer):boolean;
begin
 _camisx:=false; // ??
end;

function TDaqPascal._camisxq(c:integer):boolean;
begin
 _camisxq:=false; // ??
end;

function TDaqPascal._camgetcsr(c:integer):integer;
begin
 _camgetcsr:=0; // ??
end;

function TDaqPascal._camenabint(c,m:integer):boolean;
begin
 _camenabint:=false; // ??
end;

function TDaqPascal._camdisabint(c,m:integer):boolean;
begin
 _camdisabint:=false; // ??
end;

function TDaqPascal._camismyreq(c:integer):boolean;
begin
 _camismyreq:=false; // ??
end;

function TDaqPascal._camgetrcode(c:integer):integer;
begin
 _camgetrcode:=0; // ??
end;

function TDaqPascal._camclrreq(c,m:integer):boolean;
begin
 _camclrreq:=false; // ??
end;

function TDaqPascal._camsaveregs(c:integer):boolean;
begin
 _camsaveregs:=false; // ??
end;

function TDaqPascal._camrestregs(c:integer):boolean;
begin
 _camrestregs:=false; // ??
end;

function TDaqPascal._cammaskofst(c:integer):integer;
begin
 _cammaskofst:=0; // ??
end;

function TDaqPascal._camstoflam(c:integer):integer;
begin
 _camstoflam:=0; // ??
end;

function TDaqPascal._camgetrmask(c:integer):integer;
begin
 _camgetrmask:=0; // ??
end;

function TDaqPascal._pkkclrreq(c:integer):boolean;
begin
 _pkkclrreq:=false; // ??
end;

function TDaqPascal._pkkenabint(c:integer):boolean;
begin
 _pkkenabint:=false; // ??
end;

function TDaqPascal._pkkdisabint(c:integer):boolean;
begin
 _pkkdisabint:=false; // ??
end;

function TDaqPascal._clickbutton:integer;
begin
 _clickbutton:=TProgramDevice(TheDevice).ClickButton;
end;

function TDaqPascal._clicksensor:LongString;
begin
 _clicksensor:=TProgramDevice(TheDevice).ClickSensor;
end;

function TDaqPascal._clicktag:integer;
begin
 _clicktag:=TProgramDevice(TheDevice).ClickTag;
end;

function TDaqPascal._crvfind(const path:LongString):real;
begin
 Result:=crvfind(path);
end;

function TDaqPascal._crvget(ref,x:real):real;
begin
 _crvget:=CurveByRef(ref).Interpolate(x);
end;

function TDaqPascal._crvwhere(ref,x:real):real;
begin
 _crvwhere:=CurveByRef(ref).GetIndexAt(x)+1;
end;

function TDaqPascal._ms2year(ms:real):integer;
var Year,Month,Day,Hour,Min,Sec,MSec:word;
begin
 MSecToDateTime(ms,Year,Month,Day,Hour,Min,Sec,MSec);
 _ms2year:=Year;
end;

function TDaqPascal._ms2month(ms:real):integer;
var Year,Month,Day,Hour,Min,Sec,MSec:word;
begin
 MSecToDateTime(ms,Year,Month,Day,Hour,Min,Sec,MSec);
 _ms2month:=Month;
end;

function TDaqPascal._ms2day(ms:real):integer;
var Year,Month,Day,Hour,Min,Sec,MSec:word;
begin
 MSecToDateTime(ms,Year,Month,Day,Hour,Min,Sec,MSec);
 _ms2day:=Day;
end;

function TDaqPascal._ms2hour(ms:real):integer;
var Year,Month,Day,Hour,Min,Sec,MSec:word;
begin
 MSecToDateTime(ms,Year,Month,Day,Hour,Min,Sec,MSec);
 _ms2hour:=Hour;
end;

function TDaqPascal._ms2min(ms:real):integer;
var Year,Month,Day,Hour,Min,Sec,MSec:word;
begin
 MSecToDateTime(ms,Year,Month,Day,Hour,Min,Sec,MSec);
 _ms2min:=Min;
end;

function TDaqPascal._ms2sec(ms:real):integer;
var Year,Month,Day,Hour,Min,Sec,MSec:word;
begin
 MSecToDateTime(ms,Year,Month,Day,Hour,Min,Sec,MSec);
 _ms2sec:=Sec;
end;

procedure FetchPair(arg:LongString; out s1,s2:LongString);
begin
 arg:=Trim(arg);
 if (PosEol(arg)>0) then begin
  s1:=Trim(ExtractWord(1,arg,EolnDelims));
  s2:=Trim(ExtractWord(2,arg,EolnDelims));
  Exit;
 end;
 if HasChars(arg,[QuoteMark,Apostrophe]) then begin
  s1:=ExtractPhrase(1,arg,JustBlanks);
  s2:=ExtractPhrase(2,arg,JustBlanks);
  Exit;
 end;
 s1:=URL_Decode(ExtractWord(1,arg,JustBlanks));
 s2:=URL_Decode(ExtractWord(2,arg,JustBlanks));
end;

function TDaqPascal._filerename(const arg:LongString):Boolean;
var src,dst:LongString;
begin
 FetchPair(arg,src,dst);
 Result:=_crw_fio.filerename(src,dst);
end;

function TDaqPascal._filecopy(const arg:LongString):Boolean;
var src,dst:LongString;
begin
 FetchPair(arg,src,dst);
 Result:=_crw_fio.filecopy(src,dst);
end;

function TDaqPascal._worddelims(const s:LongString):LongString;
var s1:LongString; c:char; i:integer; const zdump:LongString='';
begin
 s1:='';
 for c:=#0 to #255 do if c in worddelims then s1:=s1+c;
 if s<>'' then begin
  worddelims:=[];
  if (zdump='') then zdump:=dump(Integer(0));
  if (s=zdump) then worddelims:=ScanSpaces else
  for i:=1 to length(s) do include(worddelims,s[i]);
 end;
 _worddelims:=s1;
end;

function TDaqPascal._getfattr(const name:LongString):integer;
begin
 _getfattr:=GetFileAttr(name);
end;

function TDaqPascal._setfattr(const name:LongString; Attr:integer):integer;
begin
 _setfattr:=1-ord(SetFileAttr(name,Attr));
end;

function GetEnv(const Name:LongString):LongString;
begin
 Result:=_crw_environ.GetEnv(Name);
end;

function TDaqPascal._paramstr(const arg:LongString):LongString;
var s1,w1,s2,w2,s3,w3,s4,w4,s5,w5,s6,w6:LongString;
var wc:word; Delims:TCharSet; i:integer;
 function FormCaption(Form:TForm):LongString;
 begin
  if Form is TForm then Result:=Form.Caption else Result:='';
 end;
 function CreateGUID(const Fmt:LongString):LongString;
 var G:TGUID;
 begin
  Result:='';
  try
   if (SysUtils.CreateGuid(G)=0) then
   case Identify(Fmt) of
    sid_bin:  Result:=Dump(G,sizeof(G));
    sid_hex:  Result:=hex_encode(Dump(G,sizeof(G)));
    sid_nice: Result:=base32_encode(Dump(G,sizeof(G)),Base32_Alphabet_Id_zbase32);
    sid_str:  Result:=GuidToString(G);
   end;
  except
   on E:Exception do BugReport(E,Self,'ParamStr_CreateGUID');
  end;
 end;
 function SpecFolder(const id:LongString):LongString;
 begin
  if (id='') or (id='*')
  then Result:=CSIDL_ListAllAsText
  else Result:=CSIDL_FolderByName(id,'');
 end;
 function ApplicationMainFormHandle:THandle;
 begin
  if (Application<>nil) and (Application.MainForm<>nil)
  then Result:=Application.MainForm.Handle
  else Result:=0;
 end;
 function DoDetectBlobImageType(const How,Data:LongString):LongString;
 var Blob:LongString;
 begin
  case WordIndex(How,'b64 hex pct url',JustSpaces) of
   1: Blob:=base64_decode(Data);
   2: Blob:=hex_decode(Data);
   3: Blob:=percent_decode(Data);
   4: Blob:=url_decode(Data);
   else Blob:='';
  end;
  Result:=DbCon.DetectBlobImageType(Blob);
 end;
begin
 Result:='';
 Delims:=[' ',#9];
 wc:=WordCount(arg,Delims);
 case wc of
  0: Exit;
  1: begin
   s1:=ExtractWord(1,arg,Delims); w1:=UpCaseStr(s1);
   case Identify(w1) of
    sid_0:                  Result:=paramstr(0);
    sid_1:                  Result:=paramstr(1);
    sid_2:                  Result:=paramstr(2);
    sid_3:                  Result:=paramstr(3);
    sid_4:                  Result:=paramstr(4);
    sid_5:                  Result:=paramstr(5);
    sid_6:                  Result:=paramstr(6);
    sid_7:                  Result:=paramstr(7);
    sid_8:                  Result:=paramstr(8);
    sid_9:                  Result:=paramstr(9);
    sid_HOMEDIR:            Result:=HomeDir;
    sid_STARTUPPATH:        Result:=StartupPath;
    sid_TEMPPATH:           Result:=TempDir;
    sid_DAQCONFIGPATH:      Result:=Daq.ConfigPath;
    sid_DAQCONFIGFILE:      Result:=Daq.ConfigFile;
    sid_DAQDATAPATH:        Result:=Daq.DataPath;
    sid_DAQBACKUPFILE:      Result:=Daq.BackUpFile;
    sid_GETCURDIR:          Result:=GetCurrentDir;
    sid_GETCURRDIR:         Result:=GetCurrentDir;
    sid_DEVICENAME:         Result:=TheDevice.Name;
    sid_DEVICEMODEL:        Result:=TheDevice.Model;
    sid_DEVICEFAMILY:       Result:=TheDevice.Family;
    sid_PROGNAME:           Result:=_crw_fio.ProgName;
    sid_SYSINIFILE:         Result:=_crw_fio.SysIniFile;
    sid_DAQPROGRAM:         Result:=Self.ProgName;
    sid_USERNAME:           Result:=_crw_fio.UserName;
    sid_COMPUTERNAME:       Result:=_crw_fio.ComputerName;
    sid_HOSTNAME:           Result:=_crw_fio.HostName;
    sid_HOSTLIST:           Result:=_crw_fio.GetHostList;
    sid_DOMAINLIST:         Result:=_crw_fio.GetDomainList;
    sid_USERDOMAIN:         Result:=_crw_fio.UserDomain;
    sid_USERLIST:           Result:=_crw_fio.GetUserList;
    sid_IPADDRESS:          Result:=_crw_fio.GetIpAddress;
    sid_MACADDRESS:         Result:=_crw_fio.GetMacAddresses;
    sid_CONSOLE:            Result:=GetLangConsoleCaption;
    sid_CONSOLE_RU:         Result:=GetLangConsoleCaption;
    sid_MAINCONSOLE:        Result:=GetMainConsoleCaption;
    sid_GUARD:              Result:=Guard.LevelName[Guard.Level];
    sid_ComPortList:        Result:=_crw_uart.EnumComPorts(',');
    sid_AdamTraffic:        Result:=GetAdamTraffic('*');
    sid_CreateGUID:         Result:=CreateGUID('str');
    sid_CreateClassID:      Result:=CreateGUID('str');
    sid_SPECFOLDER:         Result:=SpecFolder('*');
    sid_AnsiCodePage:       Result:=IntToStr(GetACP);
    sid_OemCodePage:        Result:=IntToStr(GetOEMCP);
    sid_AppFormBounds:      Result:=GetAppFormBoundsStr;
    sid_AppClientBounds:    Result:=GetAppClientBoundsStr;
    sid_OleDbProviderNames: Result:=OleDbProviderNames.Text;
    sid_OdbcDriverNames:    Result:=OdbcDriverNames.Text;
    sid_MainInstance:       Result:=IntToStr(MainInstance);
    sid_MainThreadId:       Result:=IntToStr(MainThreadId);
    sid_CurrentThreadId:    Result:=IntToStr(GetCurrentThreadId);
    sid_CurrentProcessId:   Result:=IntToStr(GetCurrentProcessId);
    sid_ApplicationHandle:  Result:=IntToStr(ApplicationHandle);
    sid_MainFormHandle:     Result:=IntToStr(ApplicationMainFormHandle);
    sid_PollingUseMsgPump:  Result:=IntToStr(Ord((TheDevice as TSoftwareDevice).Polling.UseMsgPump));
    sid_PollingThreadID:    Result:=IntToStr((TheDevice as TSoftwareDevice).Polling.ThreadID);
    sid_PollingDelay:       Result:=IntToStr((TheDevice as TSoftwareDevice).Polling.Delay);
    sid_PollingPriority:    Result:=GetPriorityName((TheDevice as TSoftwareDevice).Polling.Priority);
    sid_lc_ctype_digit:     Result:=LC_CTYPE_digit;
    sid_lc_ctype_xdigit:    Result:=LC_CTYPE_xdigit;
    sid_lc_ctype_upper:     Result:=LC_CTYPE_upper;
    sid_lc_ctype_lower:     Result:=LC_CTYPE_lower;
    sid_lc_ctype_blank:     Result:=LC_CTYPE_blank;
    sid_lc_ctype_space:     Result:=LC_CTYPE_space;
    sid_lc_ctype_cntrl:     Result:=LC_CTYPE_cntrl;
    sid_lc_ctype_punct:     Result:=LC_CTYPE_punct;
    sid_lc_ctype_alpha:     Result:=LC_CTYPE_alpha;
    sid_lc_ctype_alnum:     Result:=LC_CTYPE_alnum;
    sid_lc_ctype_print:     Result:=LC_CTYPE_print;
    sid_lc_ctype_graph:     Result:=LC_CTYPE_graph;
    sid_lc_ctype_word:      Result:=LC_CTYPE_word;
    sid_SessionNumber:      Result:=IntToStr(SessionManager.SessionNb);
    sid_SessionTitle:       Result:=GetEnv('CRW_DAQ_SYS_TITLE');
    sid_SessionName:        Result:=LowerCase(ProgBaseName)+'_'+IntToStr(SessionManager.SessionNb);
    sid_SessionNb:          Result:=IntToStr(SessionManager.SessionNb);
    sid_ServicePortList:    Result:=UriMan.ServicePortList;
    sid_EtcServicesFile:    Result:=UriMan.EtcServicesFile;
    sid_ListOf:             Result:=ParamStrListOf(SkipWords(1,arg,Delims));
    sid_SysLogSeverityList: Result:=SysLog.SeverityList;
    else                    Result:='';
   end;
  end;
  2: begin
   s1:=ExtractWord(1,arg,Delims); w1:=UpCaseStr(s1);
   s2:=ExtractWord(2,arg,Delims); w2:=UpCaseStr(s2);
   case Identify(w1) of
    sid_GETENV:             Result:=_crw_fio.getenv(s2);
    sid_FEXPAND:            Result:=fexpand(URL_Decode(s2));
    sid_GETEXEBYFILE:       Result:=GetExeByFile(URL_Decode(w2));
    sid_EXTRACTFILEPATH:    Result:=ExtractFilePath(URL_Decode(s2));
    sid_EXTRACTFILENAME:    Result:=ExtractFileName(URL_Decode(s2));
    sid_EXTRACTFILEEXT:     Result:=ExtractFileExt(URL_Decode(s2));
    sid_ADDBACKSLASH:       Result:=AddBackSlash(URL_Decode(s2));
    sid_AdaptFileName:      Result:=AdaptFileName(URL_Decode(s2));
    sid_AdaptExeFileName:   Result:=AdaptExeFileName(URL_Decode(s2));
    sid_AdaptDllFileName:   Result:=AdaptDllFileName(URL_Decode(s2));
    sid_AdaptLnkFileName:   Result:=AdaptLnkFileName(URL_Decode(s2));
    sid_ReadShellLink:      Result:=ReadShellLinkAsText(URL_Decode(s2));
    sid_DEVICENAME:         begin if str2int(w2,i) then Result:=FullDaqDeviceList[i].Name        end;
    sid_DEVICEMODEL:        begin if str2int(w2,i) then Result:=FullDaqDeviceList[i].Model       end;
    sid_DEVICEFAMILY:       begin if str2int(w2,i) then Result:=FullDaqDeviceList[i].Family      end;
    sid_CURVENAME:          begin if str2int(w2,i) then Result:=Daq.Curves[i].Name               end;
    sid_CURWINNAME:         begin if str2int(w2,i) then Result:=FormCaption(Daq.CurWinList[i])   end;
    sid_TABWINNAME:         begin if str2int(w2,i) then Result:=FormCaption(Daq.TabWinList[i])   end;
    sid_CIRWINNAME:         begin if str2int(w2,i) then Result:=FormCaption(Daq.CirWinList[i])   end;
    sid_SPEWINNAME:         begin if str2int(w2,i) then Result:=FormCaption(Daq.SpeWinList[i])   end;
    sid_CONSOLE:            Result:=RusEng('КОНСОЛЬ ','CONSOLE ')+w2;
    sid_CONSOLE_RU:         Result:=RusEng('КОНСОЛЬ ','CONSOLE ')+w2;
    sid_USERDOMAIN:         Result:=_crw_fio.UserDomain(URL_Decode(s2));
    sid_HOSTNAME:           Result:=_crw_fio.HostName(StrToIntDef(s2,0));
    sid_IPADDRESS:          Result:=_crw_fio.GetIpAddress(URL_Decode(s2));
    sid_MACADDRESS:         Result:=_crw_fio.GetMacAddresses(URL_Decode(s2));
    sid_USERLIST:           Result:=_crw_fio.GetUserList(URL_Decode(s2));
    sid_HOSTLIST:           Result:=_crw_fio.GetHostList(URL_Decode(s2));
    sid_DOMAINLIST:         Result:=_crw_fio.GetDomainList(URL_Decode(s2));
    sid_FILESEARCH:         Result:=_crw_fio.SmartFileSearch(URL_Decode(s2),DefaultPathVarStr);
    sid_SPECFOLDER:         Result:=SpecFolder(s2);
    sid_AdamTraffic:        Result:=GetAdamTraffic(s2);
    sid_DaqSysInfo:         Result:=GetDaqSysInfo(s2);
    sid_ColorCode:          Result:=Format('%.6x',[StringToColor(s2,clNone)]);
    sid_ColorName:          begin if str2int(w2,i) then Result:=ColorToString(i);                end;
    sid_ColorInfo:          Result:=ParamStrColorInfo(s2);
    sid_CharsetCode:        Result:=Format('%d',[StringToCharset(w2,DEFAULT_CHARSET)]);
    sid_CharsetName:        begin if str2int(w2,i) then Result:=CharsetToString(i,'');           end;
    sid_PitchCode:          Result:=Format('%d',[Ord(StringToPitch(w2,fpDefault))]);
    sid_PitchName:          begin if str2int(w2,i) then Result:=PitchToString(TFontPitch(i),''); end;
    sid_HasherCode:         Result:=Format('%d',[FindIndexOfHasher(w2)]);
    sid_HasherName:         begin if str2int(w2,i) then Result:=GetHasherNameByIndex(i);         end;
    sid_CreateGUID:         Result:=CreateGUID(w2);
    sid_CreateClassID:      Result:=CreateGUID(w2);
    sid_GetSystemAssoc:     Result:=GetSystemAssoc(w2);
    sid_GetSystemFType:     Result:=GetSystemFType(w2);
    sid_GetSystemAssocExe:  Result:=GetSystemAssocExe(w2);
    sid_GetSystemFTypeExe:  Result:=GetSystemFTypeExe(w2);
    sid_Compiler_Options:
    case Identify(w2) of
     sid_Compiler_itabmax:  Result:=d2s(itabmax);
     sid_Compiler_btabmax:  Result:=d2s(btabmax);
     sid_Compiler_atabmax:  Result:=d2s(atabmax);
     sid_Compiler_rtabmax:  Result:=d2s(rtabmax);
     sid_Compiler_dtabmax:  Result:=d2s(dtabmax);
     sid_Compiler_dtabmin:  Result:=d2s(dtabmin);
     sid_Compiler_stabmax:  Result:=d2s(stabmax);
     sid_Compiler_stabmin:  Result:=d2s(stabmin);
     sid_Compiler_slenmax:  Result:=d2s(slenmax);
     else                   Result:='';
    end;
    sid_PollingUseMsgPump:  begin
     if str2int(w2,i) then (TheDevice as TSoftwareDevice).Polling.UseMsgPump:=(i<>0);
     Result:=IntToStr(Ord((TheDevice as TSoftwareDevice).Polling.UseMsgPump));
    end;
    sid_PollingDelay:       begin
     if str2int(w2,i) and (i>0) and (i<5000) then (TheDevice as TSoftwareDevice).Polling.Delay:=i;
     Result:=IntToStr((TheDevice as TSoftwareDevice).Polling.Delay);
    end;
    sid_PollingPriority:    begin
     (TheDevice as TSoftwareDevice).Polling.Priority:=GetPriorityByName(w2,(TheDevice as TSoftwareDevice).Polling.Priority);
     Result:=GetPriorityName((TheDevice as TSoftwareDevice).Polling.Priority);
    end;
    sid_FetchUriScheme:     Result:=UriMan.Fetch_URI_Scheme(s2);
    sid_FetchUriAuthority:  Result:=UriMan.Fetch_URI_Authority(s2);
    sid_FetchUriPath:       Result:=UriMan.Fetch_URI_Path(s2);
    sid_FetchUriQuery:      Result:=UriMan.Fetch_URI_Query(s2);
    sid_FetchUriFragment:   Result:=UriMan.Fetch_URI_Fragment(s2);
    sid_FindServicePort:    Result:=IntToStr(UriMan.FindServicePort(s2));
    sid_ServicePortInfo:    Result:=UriMan.ServicePortInfo(StrToIntDef(s2,-1));
    sid_ServicePortName:    Result:=UriMan.ServicePortName(StrToIntDef(s2,-1));
    sid_ListOf:             Result:=ParamStrListOf(SkipWords(1,arg,Delims));
    sid_SysLogSeverityCode: Result:=IntToStr(SysLog.StringToSeverity(s2));
    sid_SysLogSeverityName: Result:=SysLog.GetSeverityIdent(SysLog.StringToSeverity(s2));
    else                    Result:='';
   end;
  end;
  3: begin
   s1:=ExtractWord(1,arg,Delims); w1:=UpCaseStr(s1);
   s2:=ExtractWord(2,arg,Delims); w2:=UpCaseStr(s2);
   s3:=ExtractWord(3,arg,Delims); w3:=UpCaseStr(s3);
   case Identify(w1) of
    sid_SYSTEM:            Result:=GetSystemParamStr(s2,s3);
    sid_MAKERELATIVEPATH:  Result:=MakeRelativePath(URL_Decode(s2),URL_Decode(s3));
    sid_FORCEPATH:         Result:=ForcePath(URL_Decode(s2),URL_Decode(s3));
    sid_DEFAULTPATH:       Result:=DefaultPath(URL_Decode(s2),URL_Decode(s3));
    sid_FORCEEXTENSION:    Result:=ForceExtension(URL_Decode(s2),URL_Decode(s3));
    sid_DEFAULTEXTENSION:  Result:=DefaultExtension(URL_Decode(s2),URL_Decode(s3));
    sid_USERLIST:          Result:=_crw_fio.GetUserList(URL_Decode(s2),StrToIntDef(s3,0));
    sid_HOSTLIST:          Result:=_crw_fio.GetHostList(URL_Decode(s2),URL_Decode(s3));
    sid_DOMAINLIST:        Result:=_crw_fio.GetDomainList(URL_Decode(s2),StrToIntDef(s3,0));
    sid_FILESEARCH:        Result:=_crw_fio.SmartFileSearch(URL_Decode(s2),URL_Decode(s3));
    sid_ADDSEARCHPATH:     Result:=_crw_fio.AddSearchPath(URL_Decode(s2),Daq.FileRef(Trim(URL_Decode(s3)),''));
    sid_REMSEARCHPATH:     Result:=_crw_fio.RemSearchPath(URL_Decode(s2),Daq.FileRef(Trim(URL_Decode(s3)),''));
    sid_FindServicePort:   Result:=IntToStr(UriMan.FindServicePort(s2,s3));
    sid_ServicePortInfo:   Result:=UriMan.ServicePortInfo(StrToIntDef(s2,-1),s3);
    sid_ServicePortName:   Result:=UriMan.ServicePortName(StrToIntDef(s2,-1),s3);
    sid_ListOf:            Result:=ParamStrListOf(SkipWords(1,arg,Delims));
    sid_DetectBlobImageType : Result:=DoDetectBlobImageType(s2,s3);
    else                   Result:='';
   end;
  end;
  4: begin
   s1:=ExtractWord(1,arg,Delims); w1:=UpCaseStr(s1);
   s2:=ExtractWord(2,arg,Delims); w2:=UpCaseStr(s2);
   s3:=ExtractWord(3,arg,Delims); w3:=UpCaseStr(s3);
   s4:=ExtractWord(4,arg,Delims); w4:=UpCaseStr(s4);
   case Identify(w1) of
    sid_REGISTRY:          Result:=_crw_fio.ReadRegistryString(GetRootKeyByName(s2),URL_Decode(s3),URL_Decode(s4));
    sid_USERLIST:          Result:=_crw_fio.GetUserList(URL_Decode(s2),StrToIntDef(w3,0),StrToIntDef(w4,2));
    sid_HOSTLIST:          Result:=_crw_fio.GetHostList(URL_Decode(s2),URL_Decode(s3),StrToIntDef(w4,0));
    sid_DOMAINLIST:        Result:=_crw_fio.GetDomainList(URL_Decode(s2),StrToIntDef(w3,0),False,StrToIntDef(w4,DefNetEnumTimeOut));
    sid_ServicePortName:   Result:=UriMan.ServicePortName(StrToIntDef(s2,-1),s3,StrToIntDef(s4,0));
    sid_ListOf:            Result:=ParamStrListOf(SkipWords(1,arg,Delims));
    else                   Result:='';
   end;
  end;
  5: begin
   s1:=ExtractWord(1,arg,Delims); w1:=UpCaseStr(s1);
   s2:=ExtractWord(2,arg,Delims); w2:=UpCaseStr(s2);
   s3:=ExtractWord(3,arg,Delims); w3:=UpCaseStr(s3);
   s4:=ExtractWord(4,arg,Delims); w4:=UpCaseStr(s4);
   s5:=ExtractWord(5,arg,Delims); w5:=UpCaseStr(s5);
   case Identify(w1) of
    sid_REGISTRY:          Result:=_crw_fio.WriteRegistryString(GetRootKeyByName(w2),URL_Decode(s3),URL_Decode(s4),URL_Decode(s5));
    sid_USERLIST:          Result:=_crw_fio.GetUserList(URL_Decode(s2),StrToIntDef(w3,0),StrToIntDef(w4,2),StrToIntDef(w5,DefNetEnumTimeOut));
    sid_HOSTLIST:          Result:=_crw_fio.GetHostList(URL_Decode(s2),URL_Decode(s3),StrToIntDef(w4,0),StrToIntDef(w5,-1));
    sid_ListOf:            Result:=ParamStrListOf(SkipWords(1,arg,Delims));
    else                   Result:='';
   end;
  end;
  6: begin
   s1:=ExtractWord(1,arg,Delims); w1:=UpCaseStr(s1);
   s2:=ExtractWord(2,arg,Delims); w2:=UpCaseStr(s2);
   s3:=ExtractWord(3,arg,Delims); w3:=UpCaseStr(s3);
   s4:=ExtractWord(4,arg,Delims); w4:=UpCaseStr(s4);
   s5:=ExtractWord(5,arg,Delims); w5:=UpCaseStr(s5);
   s6:=ExtractWord(6,arg,Delims); w6:=UpCaseStr(s6);
   case Identify(w1) of
    sid_HOSTLIST:          Result:=_crw_fio.GetHostList(URL_Decode(s2),URL_Decode(s3),StrToIntDef(w4,0),StrToIntDef(w5,-1),StrToIntDef(w6,DefNetEnumTimeOut));
    sid_ListOf:            Result:=ParamStrListOf(SkipWords(1,arg,Delims));
    else                   Result:='';
   end;
  end;
  7..255: begin
   s1:=ExtractWord(1,arg,Delims); w1:=UpCaseStr(s1);
   case Identify(w1) of
    sid_ListOf:            Result:=ParamStrListOf(SkipWords(1,arg,Delims));
    else                   Result:='';
   end;
  end;
  else Result:='';
 end;
 if Length(Result)>(slenmax-(slenmax div 8)) then Result:='?'; // To avoid string overflow
end;

function TDaqPascal._doserror:integer;
begin
 Result:=GetLastOsError; // ??
end;

function TDaqPascal.DllWrap:TDaqDllWrapper;
begin
 if myDllWrap=nil then begin
  myDllWrap:=NewDaqDllWrapper(TheDevice);
  myDllWrap.Master:=@myDllWrap;
 end;
 Result:=myDllWrap;
end;

function TDaqPascal._daqdllinit(const s:LongString):integer;
begin
 Result:=DllWrap.DaqDllInit(s);
end;

function TDaqPascal._daqdllfree(id:Integer):boolean;
begin
 Result:=DllWrap.DaqDllFree(id);
end;

function TDaqPascal._daqdllcall(id:integer; cmd:integer):boolean;
begin
 Result:=DllWrap.DaqDllCall(id,cmd);
end;

function TDaqPascal._f_reset(fname:LongString; mode:integer):integer;
var SaveFileMode:byte;
begin
 SetInOutRes(0);
 SmartFileClose(f_file);
 System.Assign(f_file,fname);
 SaveFileMode:=System.FileMode;
 System.FileMode:=mode;
 System.Reset(f_file,1);
 System.FileMode:=SaveFileMode;
 _f_reset:=IOResult;
 IORes:=0;
end;

function TDaqPascal._f_rewrite(fname:LongString; mode:integer):integer;
var SaveFileMode:byte;
begin
 SetInOutRes(0);
 SmartFileClose(f_file);
 System.Assign(f_file,fname);
 SaveFileMode:=System.FileMode;
 System.FileMode:=mode;
 System.Rewrite(f_file,1);
 System.FileMode:=SaveFileMode;
 _f_rewrite:=IOResult;
 IORes:=0;
end;

function TDaqPascal._f_read(count:integer):LongString;
var s:LongString; res:Integer;
begin
 s:='';
 if IORes=0 then begin
  SetInOutRes(0);
  count:=Min(count,System.FileSize(f_file));
  if count>0 then begin
   count:=Min(count,slenmax div 2);
   SetLength(s,count); res:=0;
   System.BlockRead(f_file,s[1],count,res);
   SetLength(s,res);
  end;
  IORes:=IOResult;
 end;
 Result:=s;
end;

function TDaqPascal._f_write(var s:LongString):integer;
var res:Integer;
begin
 res:=0;
 if IORes=0 then begin
  SetInOutRes(0);
  if length(s)>0 then System.BlockWrite(f_file,s[1],length(s),res);
  IORes:=IOResult;
 end;
 Result:=res;
end;

function TDaqPascal._f_size:longint;
begin
 Result:=0;
 if IORes=0 then begin
  SetInOutRes(0);
  Result:=System.FileSize(f_file);
  IORes:=IOResult;
 end;
end;

function TDaqPascal._f_seek(pos:longint):longint;
begin
 Result:=0;
 if IORes=0 then begin
  SetInOutRes(0);
  if pos>=0 then System.Seek(f_file,pos);
  Result:=System.FilePos(f_file);
  IORes:=IOResult;
 end;
end;

function TDaqPascal._f_close:boolean;
begin
 SetInOutRes(0);
 SmartFileClose(f_file);
 Result:=isFileClosed(f_file);
 if IORes=0 then IORes:=IOResult;
end;

 {385}
function TDaqPascal._sleep(t:integer):boolean;
begin
 SysUtils.Sleep(t);
 Result:=true;
end;

 {386}
function TDaqPascal._mksecnow:real;
begin
 Result:=_crw_rtc.mksecnow;
end;

{$IFDEF UNIX}
function ShellExecute(hwnd:THandle; lpOperation,lpFile,lpParameters,lpDirectory:PChar; nShowCmd:Integer):Integer;
var aOperation,aFile,aDirectory,CmdLine,OutStr:LongString;
begin
 Result:=0;
 aOperation:=TrimDef(StrPas(lpOperation),'open');
 aFile:=Trim(StrPas(lpFile));
 aDirectory:=Trim(StrPas(lpDirectory));
 if SameText(aOperation,'open') then begin
  CmdLine:='xdg-open '+aFile;
  if RunCommandInDir(aDirectory,CmdLine,OutStr) then Result:=33;
 end;
end;
{$ENDIF ~UNIX}

 {387}
function TDaqPascal._ShellExecute(const s:longstring):integer;
var
 hwnd           : THandle;	        // handle to parent window
 lpOperation    : TParsingBuffer;	// pointer to string that specifies operation to perform
 lpFile         : TParsingBuffer;	// pointer to filename or folder name string
 lpParameters   : TParsingBuffer;	// pointer to string that specifies executable-file parameters
 lpDirectory    : TParsingBuffer;	// pointer to string that specifies default directory
 nShowCmd       : Integer;          // whether file is shown when opened
begin
 Result:=0;
 try
  hwnd:=0;
  StrCopyBuff(lpOperation,  Trim(ExtractWord(1,s,['|'])));
  StrCopyBuff(lpFile,       Trim(ExtractWord(2,s,['|'])));
  StrCopyBuff(lpParameters, Trim(ExtractWord(3,s,['|'])));
  StrCopyBuff(lpDirectory,  Trim(ExtractWord(4,s,['|'])));
  case WordIndex(ExtractWord(5,s,['|']),'HIDE,MAXIMIZE,MINIMIZE,RESTORE,SHOW,'+
                 'SHOWDEFAULT,SHOWMAXIMIZED,SHOWMINIMIZED,SHOWMINNOACTIVE,SHOWNA,'+
                 'SHOWNOACTIVATE,SHOWNORMAL',ScanSpaces)
  of
   1  : nShowCmd:=SW_HIDE;
   2  : nShowCmd:=SW_MAXIMIZE;
   3  : nShowCmd:=SW_MINIMIZE;
   4  : nShowCmd:=SW_RESTORE;
   5  : nShowCmd:=SW_SHOW;
   6  : nShowCmd:=SW_SHOWDEFAULT;
   7  : nShowCmd:=SW_SHOWMAXIMIZED;
   8  : nShowCmd:=SW_SHOWMINIMIZED;
   9  : nShowCmd:=SW_SHOWMINNOACTIVE;
   10 : nShowCmd:=SW_SHOWNA;
   11 : nShowCmd:=SW_SHOWNOACTIVATE;
   12 : nShowCmd:=SW_SHOWNORMAL;
   else nShowCmd:=SW_SHOWNORMAL;
  end;
  Result:=ShellExecute(hwnd,lpOperation,lpFile,lpParameters,lpDirectory,nShowCmd);
 except
  on E:Exception do BugReport(E,nil,'_ShellExecute');
 end;
end;

 {390}
function TDaqPascal._crlf:LongString;
begin
 _crlf:=_crw_str.CRLF;
end;

 {391}
function TDaqPascal._progname:LongString;
begin
 _progname:=UnifyAlias(Self.progname);
end;

 {392}
function TDaqPascal._devname:LongString;
begin
 _devname:=TheDevice.Name;
end;

 {393}
function TDaqPascal._crvname(ref:real):LongString;
begin
 _crvname:=CurveByRef(ref).Name;
end;

 {394}
function TDaqPascal._crvlock(ref:real):boolean;
var c:TCurve;
begin
 _crvlock:=false;
 c:=CurveByRef(ref);
 if Assigned(c) then begin
  if crvlocks.Count<=High(crvlocks.Items) then begin
   crvlocks.Items[crvlocks.Count]:=c;
   Inc(crvlocks.Count);
   _crvlock:=true;
   c.Lock;
  end;
 end;
end;

 {395}
function TDaqPascal._crvunlock(ref:real):boolean;
var c:TCurve; i:Integer;
begin
 _crvunlock:=false;
 c:=CurveByRef(ref);
 if Assigned(c) then begin
  if crvlocks.Count>0 then
  for i:=crvlocks.Count-1 downto 0 do
  if crvlocks.Items[i]=c then begin
   SafeMove(crvlocks.Items[i+1],crvlocks.Items[i],(crvlocks.Count-i-1)*SizeOf(crvlocks.Items[0]));
   Dec(crvlocks.Count);
   _crvunlock:=true;
   c.Unlock;
   Break;
  end;
 end;
end;

 {396}
function TDaqPascal._crvins(ref,i,x,y:real):real;
var c:TCurve; j,k:longint;
begin
 _crvins:=0;
 c:=CurveByRef(ref);
 j:=round(i);
 if Assigned(c) then begin
  c.Lock;
  k:=c.Count;
  j:=min(j,k+1);
  if j>=1 then c.InsertPoint(j-1,x,y);
  _crvins:=c.Count-k;
  c.Unlock;
 end;
end;

 {397}
function TDaqPascal._crvdel(ref,i,n:real):real;
var c:TCurve; j,k,m:longint;
begin
 _crvdel:=0;
 c:=CurveByRef(ref);
 j:=round(i);
 m:=round(n);
 if Assigned(c) then begin
  c.Lock;
  k:=c.Count;
  if (j>=1) and (j<=c.Count) then c.DeletePoints(j-1,m);
  _crvdel:=k-c.Count;
  c.Unlock;
 end;
end;

 {398}
function TDaqPascal._inportw(a:integer):integer;
begin
 _inportw:=PortW[a];
end;

 {399}
function TDaqPascal._outportw(a,d:integer):integer;
begin
 PortW[a]:=d;
 _outportw:=d;
end;

 {400}
function TDaqPascal._inportl(a:integer):integer;
begin
 _inportl:=PortL[a];
end;

 {401}
function TDaqPascal._outportl(a,d:integer):integer;
begin
 PortL[a]:=d;
 _outportl:=d;
end;

const
 esAns = 1;
 esAsk = 2;
 esInp = 4;
 esErr = 8;

procedure TDaqPascal._EditInit;
begin
 _EditInp:=NewText;
 _EditAsk:=NewText;
 _EditAns:=NewText;
 _EditSet:=NewText;
 _EditCmd:=NewText;
 _EditCon:=NewText;
 _EditErr:=False;
 _EditReq:='';
end;

procedure TDaqPascal._EditFree;
begin
 Kill(_EditInp);
 Kill(_EditAsk);
 Kill(_EditAns);
 Kill(_EditSet);
 Kill(_EditCmd);
 Kill(_EditCon);
 _EditErr:=False;
 _EditReq:='';
end;

procedure TDaqPascal._EditClear;
begin
 _EditInp.Count:=0;
 _EditAsk.Count:=0;
 _EditAns.Count:=0;
 _EditSet.Count:=0;
 _EditCmd.Count:=0;
 _EditCon.Count:=0;
 _EditErr:=False;
 _EditReq:='';
end;

function TDaqPascal._EditStart(const arg:longstring):boolean;
var
 mr  : Integer;
 mi  : Integer;
 mj  : Integer;
 s1  : LongString;
 s2  : LongString;
 cmd : LongString;
 who : LongString;
 function CanEdit:Boolean;
 begin
  Result:=CanShowModal(nil,0) and (_editstate=esAsk);
 end;
begin
 _EditStart:=false;
 try
  if CanEdit then begin
   cmd:=UpcaseStr(ExtractWord(1,arg,ScanSpaces));
   who:=UpcaseStr(ExtractWord(2,arg,ScanSpaces));
   if who='' then who:=cmd;
   if SameText(cmd,'WARNING') then begin
    mr:=Warning(_EditAsk.Text,_EditSet.Text);
    _EditAns.Count:=0;
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'YESNO') then begin
    mr:=YesNo(_EditAsk.Text,_EditSet.Text);
    _EditAns.Count:=0;
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'YESNOCANCEL') then begin
    mr:=YesNoCancel(_EditAsk.Text,_EditSet.Text);
    _EditAns.Count:=0;
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'ERROR') then begin
    mr:=Error(_EditAsk.Text,_EditSet.Text);
    _EditAns.Count:=0;
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'INFORMATION') then begin
    mr:=Information(_EditAsk.Text,_EditSet.Text);
    _EditAns.Count:=0;
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'MENULIST') and (_EditAsk.Count>2) then begin
    s1:=_EditAsk[0]; _EditAsk.DelLn(0);
    s2:=_EditAsk[0]; _EditAsk.DelLn(0);
    if not Str2Int(ExtractWord(3,arg,ScanSpaces),mi) then mi:=0;
    mr:=ListBoxSelection(s1,s2,_EditAsk.Text,mi,_EditSet.Text);
    _EditAns.Count:=0;
    if mr=mrOk then _EditAns.Addln(Format('%d',[mi]));
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'SELECTIONLIST') and (_EditAsk.Count>2) then begin
    s1:=_EditAsk[0]; _EditAsk.DelLn(0);
    s2:=_EditAsk[0]; _EditAsk.DelLn(0);
    if not Str2Int(ExtractWord(3,arg,ScanSpaces),mi) then mi:=0;
    if not Str2Int(ExtractWord(4,arg,ScanSpaces),mj) then mj:=0;
    _EditAns.Text:=ListBoxMultiSelection(s1,s2,_EditAsk.Text,false,true,mi,mj,_EditSet.Text);
    //if _EditAns.Count>0 then mr:=mrOk else mr:=mrCancel;
    mr:=GetFormModalResult(FormListBoxSelection);
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'CHECKBOXLIST') and (_EditAsk.Count>2) then begin
    s1:=_EditAsk[0]; _EditAsk.DelLn(0);
    s2:=_EditAsk[0]; _EditAsk.DelLn(0);
    if not Str2Int(ExtractWord(3,arg,ScanSpaces),mi) then mi:=0;
    if not Str2Int(ExtractWord(4,arg,ScanSpaces),mj) then mj:=0;
    _EditAns.Text:=CheckListBoxMultiSelection(s1,s2,_EditAsk.Text,false,true,mi,mj,_EditSet.Text);
    //if _EditAns.Count>0 then mr:=mrOk else mr:=mrCancel;
    mr:=GetFormModalResult(FormCheckListBoxSelection);
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'TEXTEDIT') and (_EditAsk.Count>2) then begin
    s1:=_EditAsk[0]; _EditAsk.DelLn(0);
    s2:=_EditAsk[0]; _EditAsk.DelLn(0);
    _EditAns.Text:=TextEditDialog(s1,s2,_EditAsk.Text,_EditSet.Text);
    //if _EditAns.Count>0 then mr:=mrOk else mr:=mrCancel;
    mr:=GetFormModalResult(FormTextEditDialog);
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'STRINGGRIDEDIT') and (_EditAsk.Count>1) then begin
    s1:=_EditAsk[0]; _EditAsk.DelLn(0);
    mr:=ExecuteFormStringGridEditDialog(s1,_EditAsk,'|',ord(WordCount(_EditAsk[0],['|'])>2),1,true,_EditSet.Text);
    _EditAns.Count:=0;
    if mr=mrOk then _EditAns.Concat(_EditAsk);
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'FILEOPENDIALOG') and (_EditAsk.Count>1) then begin
    s1:=_EditAsk[0]; _EditAsk.DelLn(0);
    s2:=_EditAsk[0]; _EditAsk.DelLn(0);
    _EditAns.Text:=FormCrwDaq.CommonOpenDialogExecute(s1,s2,_EditAsk.Text,_EditSet.Text);
    if _EditAns.Count>0 then mr:=mrOk else mr:=mrCancel;
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if SameText(cmd,'SELECTDIRECTORYDIALOG') and (_EditAsk.Count>1) then begin
    s1:=_EditAsk[0]; _EditAsk.DelLn(0);
    s2:=_EditAsk[0]; _EditAsk.DelLn(0);
    _EditAns.Text:=FormCrwDaq.CommonSelectDirectoryDialog(s1,s2,_EditAsk.Text,_EditSet.Text);
    if _EditAns.Count>0 then mr:=mrOk else mr:=mrCancel;
    _EditAns.InsLn(0,Format('%s=%d',[who,mr]));
    _EditStart:=true;
    cmd:='';
   end;
   if cmd<>'' then _EditErr:=True;
  end else _EditErr:=true;
 except
  on E:Exception do begin BugReport(E,Self,'_EditStart'); _EditErr:=True; end;
 end;
 _EditAsk.Count:=0;
 if _EditErr then _EditAns.Count:=0;
end;

 {402}
function TDaqPascal._edit(s:longstring):longstring;
var c:char; s1,s2:LongString; i:integer; Temp:TText;
begin
 _edit:='?';
 try
  if s='' then begin
   _EditClear;
   _edit:=Format('%d',[_editstate]);
  end else begin
   c:=StrFetch(s,1);
   System.Delete(s,1,1);
   case c of
    '(': begin
          _EditClear;
          _EditInp.AddLn(s);
          _edit:=Format('%d',[_editstate]);
         end;
    ' ': begin
          _EditInp.AddLn(s);
          _edit:=Format('%d',[_editstate]);
         end;
    '@': begin
          _EditSet.AddLn('@'+s);
          _edit:=Format('%d',[_editstate]);
         end;
    '>': begin
          _EditCmd.AddLn(s);
          _edit:=Format('%d',[_editstate]);
         end;
    '!': begin
          _EditCon.AddLn(s);
          _edit:=Format('%d',[_editstate]);
         end;
    ')': begin
          Temp:=NewTextCopy(_EditInp);
          _EditInp.Count:=0;
          _EditAns.Count:=0;
          _EditAsk.Lock;
          try
           if _EditAsk.Count=0 then begin
            _EditAsk.Copy(Temp); _EditReq:=s;
            if TheScript(Self).PostDeferredCommand('@EDIT '+s+EOL) then _edit:=Format('%d',[_editstate]) else _EditReq:='';
           end else begin
            TheScript(Self).DeferredCommandProblem('@EDIT '+s);
           end;
          finally
           _EditAsk.Unlock;
           Kill(Temp);
          end;
         end;
    '?': begin
           s1:=UpcaseStr(ExtractWord(1,s,ScanSpaces));
           s2:=UpcaseStr(ExtractWord(2,s,ScanSpaces));
           if s1='' then _edit:=Format('%d',[_editstate]) else
           if s1='INP' then begin
            if s2='COUNT' then _edit:=Format('%d',[_EditInp.Count]) else
            if s2='TEXT' then _edit:=_EditInp.Text else
            if str2int(s2,i) then _edit:=_EditInp[i];
           end else
           if s1='ASK' then begin
            if s2='COUNT' then _edit:=Format('%d',[_EditAsk.Count]) else
            if s2='TEXT' then _edit:=_EditAsk.Text else
            if str2int(s2,i) then _edit:=_EditAsk[i];
           end else
           if s1='ANS' then begin
            if s2='COUNT' then _edit:=Format('%d',[_EditAns.Count]) else
            if s2='TEXT' then _edit:=_EditAns.Text else
            if str2int(s2,i) then _edit:=_EditAns[i];
           end else
           if s1='SET' then begin
            if s2='COUNT' then _edit:=Format('%d',[_EditSet.Count]) else
            if s2='TEXT' then _edit:=_EditSet.Text else
            if str2int(s2,i) then _edit:=_EditSet[i];
           end else
           if s1='CMD' then begin
            if s2='COUNT' then _edit:=Format('%d',[_EditCmd.Count]) else
            if s2='TEXT' then _edit:=_EditCmd.Text else
            if str2int(s2,i) then _edit:=_EditCmd[i];
           end else
           if s1='CON' then begin
            if s2='COUNT' then _edit:=Format('%d',[_EditCon.Count]) else
            if s2='TEXT' then _edit:=_EditCon.Text else
            if str2int(s2,i) then _edit:=_EditCon[i];
           end else
           if s1='REQ' then begin
            _edit:=_EditReq;
           end else
           if s1='TYPES' then begin
            _edit:='Warning,YesNo,YesNoCancel,Information,Error,'
                  +'MenuList,SelectionList,CheckBoxList,TextEdit,'
                  +'StringGridEdit,FileOpenDialog,SelectDirectoryDialog';
           end;
         end;
    else begin
     Echo(Format('%s: EDIT could not interpret:%s%s',[_devname,EOL,c+s]));
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'_edit');
 end;
end;

 {405}
function TDaqPascal._editstate:integer;
begin
 _editstate:=esInp*ord(_EditInp.Count>0)+
             esAsk*ord(_EditAsk.Count>0)+
             esAns*ord(_EditAns.Count>0)+
             esErr*ord(_EditErr);
end;

 {406}
function TDaqPascal._echo(const s:longstring):boolean;
begin
 _echo:=true;
 Echo(s);
end;

 {409}
function TDaqPascal._eval(const s:longstring):real;
var buf:TeeTokensBuffer;
begin
 if ee.EvaluateLine(StrCopyBuff(buf,s))=ee_Ok then _eval:=ee.Answer else _eval:=_NaN;
end;

 {412}
function TDaqPascal._evar(const s:longstring; v:real):boolean;
var buf:TParsingBuffer;
begin
 _evar:=ee.SetValue(StrCopyBuff(buf,Trim(s)),v);
end;

 {417}
function TDaqPascal._add3d(x,y,z:real):boolean;
begin
 Result:=false;
 if plotter3d.Ok and (plotter3d.N<1024*1024) and not plotterpar.Lock then
 try
  if not (isnanorinf(x) or isnanorinf(y) or isnanorinf(z)) then plotter3d.Add(x,y,z);
  Result:=true;
 except
  on E:Exception do BugReport(E,Self,'_add3d');
 end;
end;

 {418}
function TDaqPascal._plot3d(nx,ny:integer; x1,y1,x2,y2:real; const opt:longstring):boolean;
begin
 Result:=false;
 if plotter3d.Ok and not plotterpar.Lock then
 try
  plotterpar.Lock:=true;
  plotterpar.nx:=nx;
  plotterpar.ny:=ny;
  plotterpar.x1:=x1;
  plotterpar.y1:=y1;
  plotterpar.x2:=x2;
  plotterpar.y2:=y2;
  if TheScript(Self).PostDeferredCommand('@PLOT3D '+opt+EOL) then Result:=true;
 except
  on E:Exception do BugReport(E,Self,'_plot3d');
 end;
end;

function TDaqPascal._startplot3d(const arg:LongString):boolean;
var Win:TFormSurfWindow; cap,tit,leg:LongString; buf:TParsingBuffer;
    i,t,l,w,h,j,r:integer; d,phi,psi,sx,sy,sz:double;
begin
 Result:=false;
 if plotter3d.Ok and plotterpar.Lock then with plotterpar do
 try
  try
   if (plotter3d.N>3) and (nx>1) and (ny>1) and (x2>x1) and (y2>y1) then begin
    plotter3d.Accept(x1,y1,x2,y2);
    cap:='Plot-3D'; tit:='Title'; leg:='Legend';
    t:=maxint; l:=maxint; w:=maxint; h:=maxint; r:=0; phi:=_nan; psi:=_nan; sx:=1; sy:=1; sz:=1;
    for i:=1 to WordCount(arg,['|']) do begin
     StrCopyBuff(buf,trim(extractword(i,arg,['|'])));
     if StrLIComp('caption=',buf,8)=0 then cap:=StrPas(@buf[8]);
     if StrLIComp('title=',buf,6)=0 then tit:=StrPas(@buf[6]);
     if StrLIComp('legend=',buf,7)=0 then leg:=StrPas(@buf[7]);
     if StrLIComp('top=',buf,4)=0 then if atoi(@buf[4],j) then t:=j;
     if StrLIComp('left=',buf,5)=0 then if atoi(@buf[5],j) then l:=j;
     if StrLIComp('width=',buf,6)=0 then if atoi(@buf[6],j) then w:=j;
     if StrLIComp('height=',buf,7)=0 then if atoi(@buf[7],j) then h:=j;
     if StrLIComp('replace=',buf,8)=0 then if atoi(@buf[8],j) then r:=j;
     if StrLIComp('phi=',buf,4)=0 then if atof(@buf[4],d) then phi:=d;
     if StrLIComp('psi=',buf,4)=0 then if atof(@buf[4],d) then psi:=d;
     if StrLIComp('scalex=',buf,7)=0 then if atof(@buf[7],d) then sx:=d;
     if StrLIComp('scaley=',buf,7)=0 then if atof(@buf[7],d) then sy:=d;
     if StrLIComp('scalez=',buf,7)=0 then if atof(@buf[7],d) then sz:=d;
    end;
    if r=1 then begin
     Win:=FindSurfWindow(cap);
     if Win.Ok and (Win.CloseAction=caFree) then Kill(Win);
    end;
    Win:=NewSurfWindowByPlotter(Point2i(nx,ny),Rect2d(x1,y1,x2,y2),plotter3d,cap,tit,leg);
    if Win.Ok then begin
     if t<>maxint then Win.Top:=t;
     if l<>maxint then Win.Left:=l;
     if w<>maxint then Win.Width:=w;
     if h<>maxint then Win.Height:=h;
     if not isnanorinf(phi) then Win.Phi:=DegToRad(phi);
     if not isnanorinf(psi) then Win.Psi:=DegToRad(psi);
     if (sx<>1) or (sy<>1) or (sz<>1) then Win.Scale:=Point3d(sx,sy,sz);
     Result:=true;
    end;
   end;
  finally
   plotter3d.Clear;
   plotterpar.Lock:=false;
  end;
 except
  on E:Exception do BugReport(E,Self,'_startplot3d');
 end;
end;

 {421}
function TDaqPascal._task_init(const cmd_line:LongString):Integer;
begin
 Result:=_crw_task.task_init(cmd_line);
 if (Result<>0) then _pipelist.Add(PtrIntToPointer(Result));
end;

 {424}
function TDaqPascal._task_free(tid:Integer):Boolean;
begin
 Result:=_pipe_free(tid);
end;

 {425}
function TDaqPascal._task_ref(tid:Integer):Integer;
begin
 Result:=_crw_task.task_ref(tid).Ref;
end;

 {426}
function TDaqPascal._task_pid(tid:Integer):Integer;
begin
 Result:=_crw_task.task_pid(tid);
end;

 {427}
function TDaqPascal._task_run(tid:Integer):Boolean;
begin
 Result:=_crw_task.task_run(tid);
end;

 {428}
function TDaqPascal._task_wait(tid,timeout:Integer):Boolean;
begin
 Result:=_crw_task.task_wait(tid,timeout);
end;

 {429}
function TDaqPascal._task_send(tid:Integer; const data:LongString):Integer;
begin
 Result:=_crw_task.task_send(tid,data);
end;

 {432}
function TDaqPascal._task_recv(tid,maxlen:Integer):LongString;
begin
 Result:=_crw_task.task_recv(tid,max(0,maxlen));
end;

 {433}
function TDaqPascal._task_txcount(tid:Integer):Integer;
begin
 Result:=_crw_task.task_txcount(tid);
end;

 {434}
function TDaqPascal._task_rxcount(tid:Integer):Integer;
begin
 Result:=_crw_task.task_rxcount(tid);
end;

 {435}
function TDaqPascal._task_txspace(tid:Integer):Integer;
begin
 Result:=_crw_task.task_txspace(tid);
end;

 {436}
function TDaqPascal._task_rxspace(tid:Integer):Integer;
begin
 Result:=_crw_task.task_rxspace(tid);
end;

 {437}
function TDaqPascal._task_result(tid:Integer):Integer;
begin
 Result:=_crw_task.task_result(tid);
end;

 {438}
function TDaqPascal._task_kill(tid,how,exit_code,timeout:Integer):Boolean;
begin
 Result:=_crw_task.task_kill(tid,how,exit_code,timeout);
end;

 {439}
function TDaqPascal._task_ctrl(tid:Integer; const param:LongString):LongString;
begin
 Result:=_crw_task.task_ctrl(tid,param);
end;

 {442}
function TDaqPascal._getcomspec:LongString;
begin
 Result:=_crw_fio.GetComSpec;
end;

 {443}
function TDaqPascal._mime_encode(const s:LongString):LongString;
begin
 Result:=_crw_base64.mime_encode(s);
end;

 {446}
function TDaqPascal._mime_decode(const s:LongString):LongString;
begin
 Result:=_crw_base64.mime_decode(s);
end;

 {449}
function TDaqPascal._clickparams(const s:LongString):LongString;
begin
 Result:=TProgramDevice(TheDevice).ClickSensorParams(s);
end;

 {452}
function TDaqPascal._str2shortcut(const s:LongString):Integer;
 function TextToShortCut(Text:LongString): TShortCut;
 begin
  Result:=_crw_apptools.TextToShortcut(Text);
  if Result<>0 then Exit;
  Result:=_crw_apptools.TextToShortcut(StringReplace(Text,'LButton','Tab',[rfIgnoreCase]));
  if Result<>0 then begin Result:=(Result and $FF00) + VK_LBUTTON; Exit; end;
  Result:=_crw_apptools.TextToShortcut(StringReplace(Text,'RButton','Tab',[rfIgnoreCase]));
  if Result<>0 then begin Result:=(Result and $FF00) + VK_RBUTTON; Exit; end;
  Result:=_crw_apptools.TextToShortcut(StringReplace(Text,'MButton','Tab',[rfIgnoreCase]));
  if Result<>0 then begin Result:=(Result and $FF00) + VK_MBUTTON; Exit; end;
 end;
begin
 Result:=TextToShortcut(StringReplace(Trim(s),'_',' ',[rfReplaceAll]));
end;

 {455}
function TDaqPascal._shortcut2str(shortcut:Integer):LongString;
 function ShortCutToText(ShortCut:TShortCut):LongString;
 begin
  case WordRec(ShortCut).Lo of
   VK_LBUTTON : Result:='LButton';
   VK_RBUTTON : Result:='RButton';
   VK_MBUTTON : Result:='MButton';
   else begin
    Result:=_crw_apptools.ShortCutToText(ShortCut);
    Exit;
   end;
  end;
  Result:=StringReplace(_crw_apptools.ShortCutToText((ShortCut and $FF00) + VK_TAB),
                        _crw_apptools.ShortCutToText(VK_TAB),Result,[rfIgnoreCase]);
 end;
begin
 Result:=ShortcutToText(shortcut);
end;

 {456}
function TDaqPascal._crvgetln(ref:real; i:Integer):LongString;
begin
 with CurveByRef(ref) do Result:=Comment.GetLn(i);
end;

 {457}
function TDaqPascal._crvputln(ref:real; i:Integer; const s:LongString):Boolean;
begin
 Result:=false;
 with CurveByRef(ref) do if Ok then begin Result:=true; Comment.PutLn(i,s); end;
end;

 {460}
function TDaqPascal._crvinsln(ref:real; i:Integer; const s:LongString):Boolean;
begin
 Result:=false;
 with CurveByRef(ref) do if Ok then begin Result:=true; Comment.InsLn(i,s); end;
end;

 {463}
function TDaqPascal._crvaddln(ref:real; const s:LongString):Boolean;
begin
 Result:=false;
 with CurveByRef(ref) do if Ok then begin Result:=true; Comment.AddLn(s); end;
end;

 {466}
function TDaqPascal._crvdelln(ref:real; i:Integer):Boolean;
begin
 Result:=false;
 with CurveByRef(ref) do if Ok then begin Result:=true; Comment.DelLn(i); end;
end;

 {467}
function TDaqPascal._crvnumln(ref:real):Integer;
begin
 with CurveByRef(ref) do Result:=Comment.Count;
end;

 {468}
function TDaqPascal._awakeflag:Boolean;
begin
 Result:=TProgramDevice(TheDevice).Polling.AwakeFlag;
end;

 {469}
type
 TFindDaqWinRec = packed record
  Caption : LongString;
  Found   : TForm;
 end;

procedure CheckDaqWinName(Index:LongInt; const aObject:TObject; var Terminate:Boolean; Custom:Pointer);
begin
 if aObject is TForm then
 with TFindDaqWinRec(Custom^) do
 if IsSameText((aObject as TForm).Caption,Caption) then begin
  Found:=(aObject as TForm);
  Terminate:=true;
 end;
end;

function FindDaqWin(const aName:LongString):TForm;
var FindDaqWinRec:TFindDaqWinRec;
begin
 Result:=nil;
 FindDaqWinRec:=Default(TFindDaqWinRec);
 try
  FindDaqWinRec.Caption:=StringBuffer(aName);
  FindDaqWinRec.Found:=nil;
  if not Assigned(FindDaqWinRec.Found) then Daq.CurWinList.ForEach(CheckDaqWinName, @FindDaqWinRec);
  if not Assigned(FindDaqWinRec.Found) then Daq.SpeWinList.ForEach(CheckDaqWinName, @FindDaqWinRec);
  if not Assigned(FindDaqWinRec.Found) then Daq.CirWinList.ForEach(CheckDaqWinName, @FindDaqWinRec);
  if not Assigned(FindDaqWinRec.Found) then Daq.TabWinList.ForEach(CheckDaqWinName, @FindDaqWinRec);
  Result:=FindDaqWinRec.Found;
 finally
  FindDaqWinRec.Caption:='';
 end;
end;

function TDaqPascal._reffind(const s:LongString):Integer;
var i:Integer; What,Name:LongString; Win:TMasterForm;
begin
 Result:=0;
 if (s<>'') then
 try
  What:=ExtractWord(1,s,ScanSpaces);
  Name:=ExtractWord(2,s,ScanSpaces);
  case Identify(What) of
   sid_Tag:
    Result:=FindTag(Name);
   sid_Curve:
    if IsNonEmptyStr(Name) then
    for i:=0 to Daq.Curves.Count-1 do
    if IsSameText(Daq.Curves[i].Name,Name) then begin
     Result:=Daq.Curves[i].Ref;
     Break;
    end;
   sid_Device:
    if IsEmptyStr(Name) then Result:=TheDevice.Ref else
    for i:=0 to FullDaqDeviceList.Count-1 do
    if IsSameText(FullDaqDeviceList[i].Name,Name) then begin
     Result:=FullDaqDeviceList[i].Ref;
     Break;
    end;
   sid_Polling:
    Result:=FindPollingRefByName(Name);
   sid_Window:
    if IsNonEmptyStr(Name) then begin
     Win:=TMasterForm(FindDaqWin(Name));
     if TObject(Win) is TMasterForm then Result:=Win.Ref;
    end;
  end;
 except
  on E:Exception do BugReport(E,Self,'_reffind');
 end;
end;

 {472}
function TDaqPascal._refinfo(refref:real; const s:LongString):LongString;
var ref:Integer; obj:TObject;
begin
 Result:='';
 if IsNonEmptyStr(s) then
 try
  ref:=Round(refref);
  obj:=ObjectRegistry[ref];
  case Identify(s) of
   sid_Name:
    if obj is TDaqDevice     then Result:=TDaqDevice(obj).Name else
    if obj is TCurve         then Result:=TCurve(obj).Name     else
    if obj is TForm          then Result:=TForm(obj).Caption   else
    if obj is TPipe          then Result:=TPipe(obj).Polling.Name else
    if obj is TSocketPipe    then Result:=TSocketPipe(obj).Polling.Name else
    if obj is TPolling       then Result:=TPolling(obj).Name   else
    if obj is TFsmEntity     then Result:=TFsmEntity(obj).Name else
    if obj is TSharedMemory  then Result:=TSharedMemory(obj).Name else
    if typetag(ref)>0        then Result:=nametag(ref);
   sid_Type:
    if obj is TDaqDevice     then Result:='Device'   else
    if obj is TCurve         then Result:='Curve'    else
    if obj is TMasterForm    then Result:='Window'   else
    if obj is TIntervalTimer then Result:='Timer'    else
    if obj is TText          then Result:='Text'     else
    if obj is TPipe          then Result:='Pipe'     else
    if obj is TSocketPipe    then Result:='Tcp'      else
    if obj is TUartPort      then Result:='Com'      else
    if obj is THashList      then Result:='HashList' else
    if obj is TRegExpMaster  then Result:='RegExp'   else
    if task_ref(ref).Ok      then Result:='Task'     else
    if obj is TPolling       then Result:='Polling'  else
    if obj is TFsmEntity     then Result:='Fsm'      else
    if obj is TDbEntity      then Result:='DB'       else
    if obj is TSharedMemory  then Result:='Shm'      else
    if typetag(ref)>0        then Result:='Tag';
   sid_ClassName:
    if obj is TObject        then Result:=obj.ClassName else
    if task_ref(ref).Ok      then Result:='Task' else
    if typetag(ref)>0        then Result:='Tag';
   sid_Bounds:
    if obj is TForm          then with TForm(obj).BoundsRect do Result:=Format('%d %d %d %d',[Left,Top,Right,Bottom]);
  end;
 except
  on E:Exception do BugReport(E,Self,'_refinfo');
 end;
end;

 {475}
function TDaqPascal._devsend(ref:Integer; const s:LongString):real;
var obj:TObject;
begin
 Result:=0;
 if ref=0 then ref:=TheDevice.Ref;
 obj:=ObjectRegistry[ref]; if (obj=nil) then Exit;
 if obj is TDaqDevice then Result:=TDaqDevice(obj).HandleMessage(s) else
 if (Length(s)=0) then begin
  if obj is TPolling then TPolling(obj).Awake else
  if obj is TUartPort then Result:=pipe_send(ref,s,true) else
  if obj is TTask then Result:=pipe_send(ref,s,true) else
  if obj is TPipe then Result:=pipe_send(ref,s,true) else
  if obj is TSocketPipe then Result:=pipe_send(ref,s,true);
 end;
end;

 {478}
function TDaqPascal._issametext(const s1,s2:LongString):Boolean;
begin
 Result:=_crw_str.IsSameText(s1,s2);
end;

 {487}
function TDaqPascal._defaultextension(const s1,s2:LongString):LongString;
begin
 Result:=_crw_str.DefaultExtension(s1,s2);
end;

 {496}
function TDaqPascal._forceextension(const s1,s2:LongString):LongString;
begin
 Result:=_crw_str.ForceExtension(s1,s2);
end;

 {505}
function TDaqPascal._defaultpath(const s1,s2:LongString):LongString;
begin
 Result:=_crw_str.DefaultPath(s1,s2);
end;

 {514}
function TDaqPascal._forcepath(const s1,s2:LongString):LongString;
begin
 Result:=_crw_str.ForcePath(s1,s2);
end;

 {523}
function TDaqPascal._makerelativepath(const s1,s2:LongString):LongString;
begin
 Result:=_crw_str.MakeRelativePath(s1,s2);
end;

 {532}
function TDaqPascal._trim(const s:LongString):LongString;
begin
 Result:=SysUtils.Trim(s);
end;

 {535}
function TDaqPascal._trimleft(const s:LongString):LongString;
begin
 Result:=SysUtils.TrimLeft(s);
end;

 {538}
function TDaqPascal._trimright(const s:LongString):LongString;
begin
 Result:=SysUtils.TrimRight(s);
end;

 {541}
function TDaqPascal._fexpand(const s:LongString):LongString;
begin
 Result:=_crw_str.FExpand(s);
end;

 {544}
function TDaqPascal._addbackslash(const s:LongString):LongString;
begin
 Result:=_crw_str.AddBackSlash(s);
end;

 {547}
function TDaqPascal._dropbackslash(const s:LongString):LongString;
begin
 Result:=_crw_str.DropBackSlash(s);
end;

 {550}
function TDaqPascal._extractfilepath(const s:LongString):LongString;
begin
 Result:=_crw_str.ExtractFilePath(s);
end;

 {553}
function TDaqPascal._extractfilename(const s:LongString):LongString;
begin
 Result:=_crw_str.ExtractFileName(s);
end;

 {556}
function TDaqPascal._extractfileext(const s:LongString):LongString;
begin
 Result:=_crw_str.ExtractFileExt(s);
end;

 {559}
function TDaqPascal._iswildcard(const s:LongString):Boolean;
begin
 Result:=_crw_str.IsWildCard(s);
end;

 {562}
function TDaqPascal._isrelativepath(const s:LongString):Boolean;
begin
 Result:=_crw_str.IsRelativePath(s);
end;

 {565}
function TDaqPascal._hasextension(const s:LongString):Boolean;
begin
 Result:=_crw_str.HasExtension(s);
end;

 {568}
function TDaqPascal._direxists(const s:LongString):Boolean;
begin
 Result:=_crw_fio.DirExists(s);
end;

 {571}
function TDaqPascal._text_new:Integer;
var txt:TText;
begin
 txt:=NewText(0,16);
 texts.Add(txt);
 _text_new:=RefByText(txt);
end;

 {572}
function TDaqPascal._text_free(ref:Integer):Boolean;
var txt:TText;
begin
 _text_free:=false;
 txt:=TextByRef(ref);
 if txt<>nil then begin
  texts.Remove(txt);
  _text_free:=true;
 end;
end;

 {573}
function TDaqPascal._text_getln(ref:Integer; i:Integer):LongString;
begin
 with TextByRef(ref) do Result:=GetLn(i);
end;

 {574}
function TDaqPascal._text_putln(ref:Integer; i:Integer; const s:LongString):Boolean;
begin
 Result:=false;
 with TextByRef(ref) do if Ok then begin Result:=true; PutLn(i,s); end;
end;

 {577}
function TDaqPascal._text_insln(ref:Integer; i:Integer; const s:LongString):Boolean;
begin
 Result:=false;
 with TextByRef(ref) do if Ok then begin Result:=true; InsLn(i,s); end;
end;

 {580}
function TDaqPascal._text_addln(ref:Integer; const s:LongString):Boolean;
begin
 Result:=false;
 with TextByRef(ref) do if Ok then begin Result:=true; AddLn(s); end;
end;

 {583}
function TDaqPascal._text_delln(ref:Integer; i:Integer):Boolean;
begin
 Result:=false;
 with TextByRef(ref) do if Ok then begin Result:=true; DelLn(i); end;
end;

 {584}
function TDaqPascal._text_numln(ref:Integer):Integer;
begin
 with TextByRef(ref) do Result:=Count;
end;

 {585}
function TDaqPascal._readinisection(txt,flags:integer; const fname,sec:LongString):integer;
var t:TText; f,s:LongString;
begin
 t:=TextByRef(txt);
 _readinisection:=0;
 if t<>nil then begin
  _readinisection:=txt;
  if IsEmptyStr(fname)
  then f:=Daq.ConfigFile
  else f:=Daq.FileRef(fname,'.cfg');
  if IsEmptyStr(sec) then s:='['+TheDevice.Name+']' else s:=UnifySection(sec);
  t.Text:=ExtractTextSection(f,s,flags);
 end;
end;

 {594}
function TDaqPascal._getenv(const n:LongString):LongString;
begin
 _getenv:=_crw_fio.GetEnv(n);
end;

 {597}
function TDaqPascal._setenv(const n,v:LongString):Boolean;
begin
 _setenv:=_crw_fio.SetEnv(n,v);
end;

 {607}
function TDaqPascal._hex_encode(const s:LongString):LongString;
begin
 Result:=_crw_crypt.hex_encode(s);
end;

 {610}
function TDaqPascal._hex_decode(const s:LongString):LongString;
begin
 Result:=_crw_crypt.hex_decode(s);
end;

 {613}
function TDaqPascal._crypt_ctrl(const s:LongString):LongString;
var w1,w2:LongString;
begin
 Result:='?';
 ExtractNameValuePair(s,w1,w2,'=',0);
 if (s='') or (s='*') then begin
  Result:='Kind='+_crypt_ctrl('Kind='+s)+EOL+
          'Mode='+_crypt_ctrl('Mode='+s)+EOL+
          'IV='+_crypt_ctrl('IV='+s)+EOL+
          'Ei='+_crypt_ctrl('Ei='+s)+EOL+
          'Eo='+_crypt_ctrl('Eo='+s)+EOL+
          'Di='+_crypt_ctrl('Di='+s)+EOL+
          'Do='+_crypt_ctrl('Do='+s);
 end else
 if IsSameText(w1,'Kind') then begin
  if IsNonEmptyStr(w2) then
  if w2='*' then _cryptkind:=ek_Blowfish else _cryptkind:=GetEncryptionKindByName(w2);
  Result:=GetEncryptionKindName(_cryptkind);
 end else
 if IsSameText(w1,'Mode') then begin
  if IsNonEmptyStr(w2) then
  if w2='*' then _cryptmode:=em_CBC else _cryptmode:=GetEncryptionModeByName(w2);
  Result:=GetEncryptionModeName(_cryptmode);
 end else
 if IsSameText(w1,'IV') then begin
  if IsNonEmptyStr(w2) then
  if w2='*' then _cryptiv:='' else _cryptiv:=Mime_Decode(w2);
  Result:=Mime_Encode(_cryptiv);
 end else
 if IsSameText(w1,'Ei') then begin
  if IsNonEmptyStr(w2) then
  if w2='*' then _cryptei:=df_Bin else _cryptei:=GetDataViewFormatByName(w2);
  Result:=GetDataViewFormatName(_cryptei);
 end else
 if IsSameText(w1,'Eo') then begin
  if IsNonEmptyStr(w2) then
  if w2='*' then _crypteo:=df_Mime else _crypteo:=GetDataViewFormatByName(w2);
  Result:=GetDataViewFormatName(_crypteo);
 end else
 if IsSameText(w1,'Di') then begin
  if IsNonEmptyStr(w2) then
  if w2='*' then _cryptdi:=df_Mime else _cryptdi:=GetDataViewFormatByName(w2);
  Result:=GetDataViewFormatName(_cryptdi);
 end else
 if IsSameText(w1,'Do') then begin
  if IsNonEmptyStr(w2) then
  if w2='*' then _cryptdo:=df_Bin else _cryptdo:=GetDataViewFormatByName(w2);
  Result:=GetDataViewFormatName(_cryptdo);
 end else
 if IsSameText(w1,'GetFileSign') then begin
  if IsNonEmptyStr(w2) then Result:=GetFileSign(w2);
 end;
end;

 {616}
function TDaqPascal._crypt_encode(const s,key:LongString):LongString;
begin
 Result:=_crw_crypt.EncryptText(s,key,_cryptiv,_cryptkind,_cryptmode,_cryptei,_crypteo);
end;

 {625}
function TDaqPascal._crypt_decode(const s,key:LongString):LongString;
begin
 Result:=_crw_crypt.DecryptText(s,key,_cryptiv,_cryptkind,_cryptmode,_cryptdi,_cryptdo);
end;

 {634}
function TDaqPascal._getmd5fromstr(const s:LongString):LongString;
begin
 Result:=_crw_crypt.GetMd5FromText(s,df_Bin);
end;

 {637}
function TDaqPascal._getmd5fromfile(const s:LongString):LongString;
begin
 Result:=_crw_crypt.GetMd5FromFile(s,df_Bin);
end;

 {640}
function TDaqPascal._getmd5fromtext(ref:Integer):LongString;
begin
 with TextByRef(ref) do
 if Ok then Result:=_crw_crypt.GetMd5FromText(Text,df_Bin) else Result:='';
end;

 {641}
function  TDaqPascal._wdt_reset(flag:Boolean):real;
begin
 with TProgramDevice(TheDevice).Polling do begin
  if flag then begin
   TProgramDevice(TheDevice).WatchdogStarted:=mSecNow;
   WdtReset;
  end;
  Result:=LoopTime;
 end;
end;

 {642}
function TDaqPascal._guard_check(const s:LongString):Integer;
var L:Cardinal;
begin
 Result:=-1;
 for L:=ga_Root downto ga_Lock do
 if IsSameText(s,Guard.LevelName[L]) then begin
  if Guard.Level<L then Result:=-1 else
  if Guard.Level>L then Result:=+1 else Result:=0;
  Break;
 end;
end;

procedure GetDirListAction(const FileName:LongString; const FileDetails:TSearchRec;
           SubDirLevel:Integer; var Terminate:Boolean; CustomData:Pointer);
begin
 if HasFlags(FileDetails.Attr,faDirectory)
 then TText(CustomData).Addln(AddBackSlash(FileName))
 else TText(CustomData).Addln(FileName);
end;

 {645}
function TDaqPascal._dirlist(txt,maxlev:integer; const dir,pat:LongString):integer;
var t:TText;
begin
 _dirlist:=0;
 t:=TextByRef(txt);
 if t<>nil then begin
  if DirExists(dir)
  then ForEachFile(FExpand(dir),pat,GetDirListAction,maxlev,t);
  _dirlist:=txt;
 end;
end;

 {654}
function TDaqPascal._pidlist(txt:integer):integer;
var t:TText; list:TRunningProcessList; i:integer;
begin
 _pidlist:=0;
 t:=TextByRef(txt);
 if t<>nil then begin
  _pidlist:=txt;
  list:=NewRunningProcessList(glops_FixName);
  try
   with list do
   for i:=0 to Count-1 do
   t.Addln(Format('%d,%d,%d,%d,%s',[Pid[i],ParentPid[i],Threads[i],
                                    Priority[i],TaskName[i]]));
  finally
   Kill(list);
  end;
 end;
end;

 {655}
function TDaqPascal._pidkill(pid,cod,lev:Integer):Integer;
begin
 _pidkill:=KillProcessTree(pid,cod,lev);
end;

procedure EnumAction(Index:LongInt; const aObject:TObject;
                     var Terminate:Boolean; CustomData:Pointer );
begin
 if aObject is TDaqDevice then TText(CustomData).Addln(TDaqDevice(aObject).Name) else
 if aObject is TCurve then TText(CustomData).Addln(TCurve(aObject).Name) else
 if aObject is TForm then TText(CustomData).Addln(TForm(aObject).Caption) else
 if aObject is TPolling then TText(CustomData).Addln(TPolling(aObject).Name);
end;

 {656}
function TDaqPascal._enumerate(txt:integer; const what:LongString):integer;
var t:TText; i:Integer;
 procedure Enum(const Id:LongString; const Lists:array of TObjectStorage);
 var i:Integer;
 begin
  if IsSameText(Trim(what),Id) then
  for i:=Low(Lists) to High(Lists) do Lists[i].ForEach(EnumAction,t);
 end;
begin
 _enumerate:=0;
 t:=TextByRef(txt);
 if t<>nil then begin
  _enumerate:=txt;
  Enum('Curve',[Daq.Curves]);
  Enum('Device',[FullDaqDeviceList]);
  Enum('Polling',[FullPollingList]);
  Enum('Tab_Window',[Daq.TabWinList]);
  Enum('Curve_Window',[Daq.CurWinList]);
  Enum('Spectr_Window',[Daq.SpeWinList]);
  Enum('Circuit_Window',[Daq.CirWinList]);
  Enum('Window',[Daq.CirWinList,Daq.CurWinList,Daq.TabWinList,Daq.SpeWinList]);
  if IsSameText(Trim(what),'Task') then
  for i:=1 to 255 do if task_ref(i).Ok then t.Addln(IntToStr(i));
  if IsSameText(Trim(what),'Tag') then
  for i:=TAG_REF_MIN to TAG_REF_MAX do if typetag(i)>0 then t.Addln(NameTag(i));
 end;
end;

 {659}
function TDaqPascal._timebase:real;
begin
 _timebase:=Daq.Timer.StartTime;
end;

 {660}
function TDaqPascal._daqfileref(f,e:LongString):LongString;
begin
 if IsEmptyStr(f) and IsEmptyStr(e)
 then _daqfileref:=GetCurrentDir
 else _daqfileref:=Daq.FileRef(f,e);
end;

 {669}
function TDaqPascal._GetClockRes(what:Integer):real;
begin
 Result:=_crw_rtc.GetClockResolution(what)/FileTimesPerMSec;
end;

 {670}
function TDaqPascal._SetClockRes(res:real):real;
begin
 Result:=_crw_rtc.SetClockResolution(Round(res*FileTimesPerMSec))/FileTimesPerMSec;
end;

 {671}
function TDaqPascal._url_encode(const s:LongString):LongString;
begin
 Result:=_crw_str.URL_Encode(s);
end;

 {674}
function TDaqPascal._url_decode(const s:LongString):LongString;
begin
 Result:=_crw_str.URL_Decode(s);
end;

 {677}
function TDaqPascal._datetime2ms(year,month,day,hour,min,sec,msec:Integer):real;
begin
 Result:=_crw_rtc.DateTimeToMSec(year,month,day,hour,min,sec,msec);
end;

 {678 = stackavail}

 {679}
function TDaqPascal._cpu_count:Integer;
begin
 Result:=_crw_proc.cpu_count;
end;

 {680}
function TDaqPascal._cpu_start:Boolean;
begin
 CpuBase:=_crw_rtc.ReadTimeStampCounter;
 Result:=CpuBase<>0;
end;

 {681}
function TDaqPascal._cpu_clock:Real;
begin
 Result:=_crw_rtc.ReadTimeStampCounter-CpuBase;
end;

 {682}
function TDaqPascal._cpu_MHz:Real;
begin
 Result:=_crw_apptools.CpuFrequencyMHz;
end;

 {683}
function TDaqPascal._pidaffinity(pid,mask:Integer):Integer;
begin
 Result:=0;
 try
  if IsUnix and (pid=-1) then pid:=0;
  Result:=_crw_task.PidAffinity(pid,mask);
 except
  on E:Exception do BugReport(E,Self,'_pidaffinity');
 end;
end;

 {684}
function TDaqPascal._devaffinity(ref,mask:Integer):Integer;
var dev:TSoftwareDevice; h:THandle; m1,m2:QWORD;
begin
 Result:=0;
 try
  if ref=0 then ref:=TheDevice.Ref;
  TObject(dev):=ObjectRegistry[ref];
  if TObject(dev) is TSoftwareDevice then h:=dev.Polling.ThreadHandle else h:=0;
  if (h<>0) then begin
   m1:=GetProcessAffinityMask;                   // Affinity of current process
   m2:=GetThreadAffinityMask(h);                 // Read original mask
   if SetThreadAffinityMask(h,(mask and m1))<>0  // Try set new mask
   then m2:=GetThreadAffinityMask(h);            // Ok, read this one
   Result:=m2;
  end;
 except
  on E:Exception do BugReport(E,Self,'_devaffinity');
 end;
end;

function eConnReset(Code:Integer):Boolean;
begin
 {$IFDEF UNIX}
 Result:=(Code=ESysECONNRESET);
 {$ENDIF ~UNIX}
 {$IFDEF WINDOWS}
 Result:=(Code=WSAECONNRESET);
 {$ENDIF ~WINDOWS}
end;

function eConnAborted(Code:Integer):Boolean;
begin
 {$IFDEF UNIX}
 Result:=(Code=ESysECONNABORTED);
 {$ENDIF ~UNIX}
 {$IFDEF WINDOWS}
 Result:=(Code=WSAECONNABORTED);
 {$ENDIF ~WINDOWS}
end;

procedure SocketReport(Pipe:TSocketPipe; When:Double; What:PChar; Code:Integer);
begin
 if Code<>ERROR_SUCCESS then
 if TObject(Pipe) is TSocketPipe then begin
  if Pipe.Owner<>nil then Pipe:=Pipe.Owner;
  if TObject(Pipe.Polling.LinkCustom) is TDaqPascal then
  with TDaqPascal(Pipe.Polling.LinkCustom) do begin
   if Code=-1 then TheDevice.FixError(ecPipeOver) else begin
    if eConnReset(Code) {and Pipe.IsServer} then {Ok} else
    if eConnAborted(Code) {and Pipe.IsServer} then {Ok} else
    TheDevice.FixError(ecPipeIO);
    if DebugMode then OutFifo.PutText(Format('%s - %s("%s",%d="%s")%s',
    [StdDateTimeStr(When),What,Pipe.Polling.Name,Code,SysErrorMessage(Code),EOL]));
   end;
  end;
 end;
end;

 {685}
function TDaqPascal._pipe_init(const cmd_line:LongString):Integer;
begin
 Result:=_crw_pipeio.pipe_init(cmd_line,ComPipeReport,SocketReport);
 if (Result<>0) then _pipelist.Add(PtrIntToPointer(Result));
 if (Result<>0) then begin
  if (pipe_ref(Result) is TPipe) then
  with (pipe_ref(Result) as TPipe) do begin
   Polling.Enabled:=False;
   Polling.LinkCustom:=Self;
   Polling.Enabled:=True;
  end;
  if (pipe_ref(Result) is TSocketPipe) then
  with (pipe_ref(Result) as TSocketPipe) do begin
   Polling.Enabled:=False;
   Polling.LinkCustom:=Self;
   Polling.Enabled:=True;
  end;
 end;
end;

 {688}
function TDaqPascal._pipe_free(pid:Integer):Boolean;
begin
 Result:=False;
 if (pid<>0) then
 if (_pipelist.IndexOf(PtrIntToPointer(pid))>=0) then begin
  _pipelist.Remove(PtrIntToPointer(pid));
  Result:=_crw_pipeio.pipe_free(pid);
 end;
end;

 {689}
function TDaqPascal._pipe_ref(pid:Integer):Integer;
var obj:TMasterObject;
begin
 Result:=0;
 TObject(obj):=_crw_pipeio.pipe_ref(pid);
 if (obj is TTask) or (obj is TUartPort)
 or (obj is TPipe) or (obj is TSocketPipe) then Result:=obj.ref;
end;

 {690}
function TDaqPascal._pipe_pid(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_pid(pid);
end;

 {691}
function TDaqPascal._pipe_run(pid:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_run(pid);
end;

 {692}
function TDaqPascal._pipe_wait(pid,timeout:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_wait(pid,timeout);
end;

 {693}
function TDaqPascal._pipe_send(pid:Integer; const data:LongString):Integer;
begin
 Result:=_crw_pipeio.pipe_send(pid,data);
end;

 {696}
function TDaqPascal._pipe_recv(pid,maxlen:Integer):LongString;
begin
 Result:=_crw_pipeio.pipe_recv(pid,maxlen);
end;

 {697}
function TDaqPascal._pipe_txcount(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_txcount(pid);
end;

 {698}
function TDaqPascal._pipe_rxcount(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_rxcount(pid);
end;

 {699}
function TDaqPascal._pipe_txspace(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_txspace(pid);
end;

 {700}
function TDaqPascal._pipe_rxspace(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_rxspace(pid);
end;

 {701}
function TDaqPascal._pipe_result(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_result(pid);
end;

 {702}
function TDaqPascal._pipe_kill(pid,how,exit_code,timeout:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_kill(pid,how,exit_code,timeout);
end;

 {703}
function TDaqPascal._pipe_ctrl(pid:Integer; const param:LongString):LongString;
begin
 Result:=_crw_pipeio.pipe_ctrl(pid,param);
end;

 {706}
function TDaqPascal._pipe_txclear(pid:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_txclear(pid);
end;

 {707}
function TDaqPascal._pipe_rxclear(pid:Integer):Boolean;
begin
 Result:=_crw_pipeio.pipe_rxclear(pid);
end;

 {708}
function TDaqPascal._pipe_count(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_count(pid);
end;

 {709}
function TDaqPascal._pipe_stream(pid,index:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_stream(pid,index);
end;

 {710}
function TDaqPascal._pipe_connected(pid:Integer):Integer;
begin
 Result:=_crw_pipeio.pipe_connected(pid);
end;

 {711}
function TDaqPascal._strconv(const how,s:LongString):LongString;
begin
 Result:=s;
 if Length(s)>0 then
 if Length(how)>0 then
 case Identify(how) of
  {$IFDEF WINDOWS}
  sid_Ansi_Oem        : Windows.CharToOemBuff(PChar(Result),PChar(Result),Length(Result));
  sid_Oem_Ansi        : Windows.OemToCharBuff(PChar(Result),PChar(Result),Length(Result));
  {$ENDIF ~WINDOWS}
  sid_Win_Dos         : Result:=WinToDosStr(Result);
  sid_Dos_Win         : Result:=DosToWinStr(Result);
  sid_Win_Koi         : Result:=WinToKoiStr(Result);
  sid_Koi_Win         : Result:=KoiToWinStr(Result);
  sid_Utf8_Ansi       : Result:=utf8_decode_ansi(s);
  sid_Ansi_Utf8       : Result:=utf8_encode_ansi(s);
  sid_Ascii_UpperCase : Result:=UpperCase(s);
  sid_Ascii_LowerCase : Result:=LowerCase(s);
  sid_Ansi_UpperCase  : Result:=AnsiUpperCase(s);
  sid_Ansi_LowerCase  : Result:=AnsiLowerCase(s);
  sid_Utf8_UpperCase  : Result:=utf8_uppercase(s);
  sid_Utf8_LowerCase  : Result:=utf8_lowercase(s);
 end;
end;

 {720}
function TDaqPascal._dump2f(const s:LongString):Real;
var f:Single;
begin
 f:=0;
 SafeMove(s[1],f,Min(SizeOf(f),Length(s)));
 Result:=f;
end;

 {723}
function TDaqPascal._dumpf(x:Real):LongString;
var f:Single;
begin
 f:=x;
 Result:=_crw_base64.Dump(f,SizeOf(f));
end;

 {724}
function TDaqPascal._url_packed(const s:LongString):LongString;
begin
 Result:=_crw_str.URL_Packed(s);
end;

procedure TDaqPascal.myBase32Init;
begin
 if not Assigned(myBase32) then begin
  myBase32:=NewBase32Coder('nice');
  myBase32.Master:=@myBase32;
 end;
end;

 {727}
function TDaqPascal._base32_encode(const s:LongString):LongString;
begin
 if not Assigned(myBase32) then myBase32Init;
 Result:=myBase32.Encode(s);
end;

 {730}
function TDaqPascal._base32_decode(const s:LongString):LongString;
begin
 if not Assigned(myBase32) then myBase32Init;
 Result:=myBase32.Decode(s);
end;

 {733}
function TDaqPascal._base32_alphabet(const s:LongString):LongString;
begin
 if not Assigned(myBase32) then myBase32Init;
 Result:=myBase32.Alphabet;
 if Length(s)>0 then myBase32.Alphabet:=s;
end;

 {736}
function TDaqPascal._hashindexof(const s:LongString; aTableSize,aMethod:Integer):Integer;
begin
 Result:=_crw_hash.GetHashByStringKey(s,aTableSize,aMethod);
end;

 {739}
function TDaqPascal._getpid:DWORD;
begin
 Result:=GetCurrentProcessId;
end;

 {740}
function TDaqPascal._hashlist_init(aMode:Integer):Integer;
var hid:Integer; aHasher:THash32Function;
begin
 hid:=(aMode shr 8) and 255; // Second byte of aMode
 if hid in [Low(Hash32FuncTable)..High(Hash32FuncTable)]
 then aHasher:=Hash32FuncTable[hid].Func else aHasher:=nil;
 Result:=_crw_hl.HashList_Init((aMode and 1)<>0,aHasher);
 if (Result<>0) then _hashlist.Add(PtrIntToPointer(Result));
end;

 {741}
function TDaqPascal._hashlist_free(aRef:Integer):Boolean;
begin
 Result:=false;
 if (aRef<>0) then
 if (_hashlist.IndexOf(PtrIntToPointer(aRef))>=0) then begin
  _hashlist.Remove(PtrIntToPointer(aRef));
  Result:=_crw_hl.HashList_Free(aRef);
 end;
end;

 {742}
function TDaqPascal._hashlist_count(aRef:Integer):Integer;
begin
 Result:=_crw_hl.HashList_Count(aRef);
end;

 {743}
function TDaqPascal._hashlist_getkey(aRef:Integer; aIndex:Integer):LongString;
begin
 Result:=_crw_hl.HashList_GetKey(aRef,aIndex);
end;

 {744}
function TDaqPascal._hashlist_delete(aRef:Integer; aKey:LongString):Boolean;
begin
 Result:=_crw_hl.HashList_Delete(aRef,aKey);
end;

 {747}
function TDaqPascal._hashlist_indexof(aRef:Integer; aKey:LongString):Integer;
begin
 Result:=_crw_hl.HashList_IndexOf(aRef,aKey);
end;

 {750}
function TDaqPascal._hashlist_getdata(aRef:Integer; aKey:LongString):Double;
begin
 Result:=_crw_hl.HashList_GetData(aRef,aKey);
end;

 {753}
function TDaqPascal._hashlist_setdata(aRef:Integer; aKey:LongString; aData:Double):Boolean;
begin
 Result:=_crw_hl.HashList_SetData(aRef,aKey,aData);
end;

 {756}
function TDaqPascal._hashlist_getlink(aRef:Integer; aKey:LongString):Integer;
begin
 Result:=_crw_hl.HashList_GetLink(aRef,aKey);
end;

 {759}
function TDaqPascal._hashlist_setlink(aRef:Integer; aKey:LongString; aLink:Integer):Boolean;
begin
 Result:=_crw_hl.HashList_SetLink(aRef,aKey,aLink);
end;

 {762}
function TDaqPascal._hashlist_getpara(aRef:Integer; aKey:LongString):LongString;
begin
 Result:=_crw_hl.HashList_GetPara(aRef,aKey);
end;

 {765}
function TDaqPascal._hashlist_setpara(aRef:Integer; aKey:LongString; aPara:LongString):Boolean;
begin
 Result:=_crw_hl.HashList_SetPara(aRef,aKey,aPara);
end;

 {774}
function TDaqPascal._devpost(ref:Integer; const s:LongString):real;
var obj:TObject;
begin
 Result:=0;
 if ref=0 then ref:=TheDevice.Ref;
 obj:=ObjectRegistry[ref]; if (obj=nil) then Exit;
 if obj is TDaqDevice then Result:=TDaqDevice(obj).HandleMessage(s,hf_Default or hf_SkipAwake);
end;

 {777}
function TDaqPascal._devpostmsg(const arg:LongString):real;
var Device:TDaqDevice; name,msg:LongString;
begin
 Result:=0;
 name:=ExtractWord(1,arg,arg_delims);
 Device:=FullDaqDeviceList.Find(name);
 if Assigned(Device) then begin
  msg:=SkipWords(1,arg,arg_delims);
  Result:=Device.HandleMessage(msg,hf_Default or hf_SkipAwake);
 end;
end;

 {780}
function TDaqPascal._clickwrite(const lines:LongString):Integer;
begin
 Result:=Ord(TProgramDevice(TheDevice).ClickWrite(lines));
end;

 {783}
function TDaqPascal._clickread:Integer;
begin
 Result:=Ord(TProgramDevice(TheDevice).ClickRead);
end;

 {784}
function TDaqPascal._clickwhat:Integer;
begin
 Result:=Ord(TProgramDevice(TheDevice).ClickWhat);
end;

 {785}
function TDaqPascal._clickwrote:Integer;
begin
 Result:=TProgramDevice(TheDevice).ClickWrote;
end;

 {786}
function TDaqPascal._clickfilter(aFilter:Integer):Integer;
begin
 case aFilter of
  0: Result:=TProgramDevice(TheDevice).ClickFilter;
  1: Result:=cf_Default;
  else begin
   Result:=TProgramDevice(TheDevice).ClickFilter;
   TProgramDevice(TheDevice).ClickFilter:=aFilter;
  end;
 end;
end;

 {787}
function TDaqPascal._clickawaker(aAwaker:Integer):Integer;
begin
 case aAwaker of
  0: Result:=TProgramDevice(TheDevice).ClickAwaker;
  1: Result:=cf_Default;
  else begin
   Result:=TProgramDevice(TheDevice).ClickAwaker;
   TProgramDevice(TheDevice).ClickAwaker:=aAwaker;
  end;
 end;
end;

 {800}
function TDaqPascal._taglock(lock:Boolean):Integer;
begin
 if lock then begin
  LockTags;
  Inc(taglocks.Count);
 end else begin
  if taglocks.Count>0 then begin
   Dec(taglocks.Count);
   UnLockTags;
  end;
 end;
 Result:=taglocks.Count;
end;

 {808}
function TDaqPascal._getai_par(n:integer; id:integer):real;
begin
 case id of
  0  : Result:=TheDevice.AnalogInputCurve[n].Count;
  1  : Result:=TheDevice.AnalogInputCurve[n].XY[0].X;
  2  : Result:=TheDevice.AnalogInputCurve[n].XY[0].Y;
  3  : Result:=TheDevice.AnalogInputCurve[n].LastPoint.X;
  4  : Result:=TheDevice.AnalogInputCurve[n].LastPoint.Y;
  5  : Result:=TheDevice.AnalogInputCurve[n].Color;
  6  : Result:=TheDevice.AnalogInputCurve[n].Style;
  7  : Result:=TheDevice.AnalogInputCurve[n].Limits.a.x;
  8  : Result:=TheDevice.AnalogInputCurve[n].Limits.a.y;
  9  : Result:=TheDevice.AnalogInputCurve[n].Limits.b.x;
  10 : Result:=TheDevice.AnalogInputCurve[n].Limits.b.y;
  11 : Result:=TheDevice.AnalogInputSmoother[n].Window;
  12 : Result:=TheDevice.AnalogInputSmoother[n].Power;
  13 : Result:=TheDevice.AnalogInputSmoother[n].K1;
  14 : Result:=TheDevice.AnalogInputSmoother[n].K2;
  else Result:=_NaN;
 end;
end;

 {809}
function TDaqPascal._getao_par(n:integer; id:integer):real;
begin
 case id of
  0  : Result:=TheDevice.AnalogOutputCurve[n].Count;
  1  : Result:=TheDevice.AnalogOutputCurve[n].XY[0].X;
  2  : Result:=TheDevice.AnalogOutputCurve[n].XY[0].Y;
  3  : Result:=TheDevice.AnalogOutputCurve[n].LastPoint.X;
  4  : Result:=TheDevice.AnalogOutputCurve[n].LastPoint.Y;
  5  : Result:=TheDevice.AnalogOutputCurve[n].Color;
  6  : Result:=TheDevice.AnalogOutputCurve[n].Style;
  7  : Result:=TheDevice.AnalogOutputCurve[n].Limits.a.x;
  8  : Result:=TheDevice.AnalogOutputCurve[n].Limits.a.y;
  9  : Result:=TheDevice.AnalogOutputCurve[n].Limits.b.x;
  10 : Result:=TheDevice.AnalogOutputCurve[n].Limits.b.y;
  11 : Result:=TheDevice.AnalogOutputHistoryLen[n];
  12 : Result:=TheDevice.AnalogFifoSize;
  13 : Result:=TheDevice.AnalogOutputCompressor[n].AbsTol;
  14 : Result:=TheDevice.AnalogOutputCompressor[n].RelTol;
  else Result:=_NaN;
 end;
end;

 {810}
function TDaqPascal._getdi_par(n:integer; id:integer):real;
var inv:Boolean;
begin
 case id of
  0  : Result:=TheDevice.DigitalInputCurve[n].Count;
  1  : Result:=TheDevice.DigitalInputCurve[n].XY[0].X;
  2  : Result:=TheDevice.DigitalInputCurve[n].XY[0].Y;
  3  : Result:=TheDevice.DigitalInputCurve[n].LastPoint.X;
  4  : Result:=TheDevice.DigitalInputCurve[n].LastPoint.Y;
  5  : Result:=TheDevice.DigitalInputCurve[n].Color;
  6  : Result:=TheDevice.DigitalInputCurve[n].Style;
  7  : Result:=TheDevice.DigitalInputCurve[n].Limits.a.x;
  8  : Result:=TheDevice.DigitalInputCurve[n].Limits.a.y;
  9  : Result:=TheDevice.DigitalInputCurve[n].Limits.b.x;
  10 : Result:=TheDevice.DigitalInputCurve[n].Limits.b.y;
  11 : Result:=TheDevice.GetDigitalInputBit(n,inv);
  12 : begin TheDevice.GetDigitalInputBit(n,inv); Result:=ord(inv); end;
  else Result:=_NaN;
 end;
end;

 {811}
function TDaqPascal._getdo_par(n:integer; id:integer):real;
begin
 case id of
  0  : Result:=TheDevice.DigitalOutputCurve[n].Count;
  1  : Result:=TheDevice.DigitalOutputCurve[n].XY[0].X;
  2  : Result:=TheDevice.DigitalOutputCurve[n].XY[0].Y;
  3  : Result:=TheDevice.DigitalOutputCurve[n].LastPoint.X;
  4  : Result:=TheDevice.DigitalOutputCurve[n].LastPoint.Y;
  5  : Result:=TheDevice.DigitalOutputCurve[n].Color;
  6  : Result:=TheDevice.DigitalOutputCurve[n].Style;
  7  : Result:=TheDevice.DigitalOutputCurve[n].Limits.a.x;
  8  : Result:=TheDevice.DigitalOutputCurve[n].Limits.a.y;
  9  : Result:=TheDevice.DigitalOutputCurve[n].Limits.b.x;
  10 : Result:=TheDevice.DigitalOutputCurve[n].Limits.b.y;
  11 : Result:=TheDevice.DigitalOutputHistoryLen[n];
  12 : Result:=TheDevice.DigitalFifoSize;
  else Result:=_NaN;
 end;
end;

 {812}
function TDaqPascal._strfmt(const Fmt:LongString; const Args: array of const):LongString;
begin
 Result:='';
 try
  Result:=SysUtils.Format(Fmt, Args);
 except
  on E:Exception do begin
   BugReport(E,Self,'_strfmt');
   _fixerror(ecFormatErrorInStrFmt);
  end;
 end;
end;

 {851}
function TDaqPascal._EasyIpc_Init(const PipeName,Options:LongString):PtrUint;
begin
 Result:=0;
 if IsNonEmptyStr(PipeName) then
 try
  Result:=EasyIpc_Init(PipeName,Options);
  if (Result<>0) then _easyipclist.Add(PtrIntToPointer(Result));
 except
  on E:Exception do BugReport(E,Self,'EasyIpc_Init');
 end;
end;

 {860}
function TDaqPascal._EasyIpc_Free(hIpc:PtrUint):Boolean;
begin
 Result:=false;
 if (hIpc<>0) then
 try
  if (_easyipclist.IndexOf(PtrIntToPointer(hIpc))>=0) then begin
   _easyipclist.Remove(PtrIntToPointer(hIpc));
   Result:=EasyIpc_Free(hIpc);
  end;
 except
  on E:Exception do BugReport(E,Self,'EasyIpc_Free');
 end;
end;

 {861}
function TDaqPascal._EasyIpc_Poll(hIpc:PtrUint):Boolean;
begin
 Result:=false;
 if (hIpc<>0) then
 try
  Result:=EasyIpc_Poll(hIpc);
 except
  on E:Exception do BugReport(E,Self,'EasyIpc_Poll');
 end;
end;

 {862}
function TDaqPascal._EasyIpc_Send(hIpc:PtrUint; const TextLines:LongString):Boolean;
begin
 Result:=false;
 if (hIpc<>0) then
 try
  Result:=EasyIpc_Send(hIpc,TextLines);
 except
  on E:Exception do BugReport(E,Self,'EasyIpc_Send');
 end;
end;

 {865}
function TDaqPascal._EasyIpc_Recv(hIpc:PtrUint; Count:Integer):LongString;
begin
 Result:='';
 if (hIpc<>0) then
 try
  Result:=EasyIpc_Recv(hIpc,Count);
 except
  on E:Exception do BugReport(E,Self,'EasyIpc_Recv');
 end;
end;

 {866}
function TDaqPascal._EasyIpc_Ctrl(hIpc:PtrUint; const Request:LongString):LongString;
begin
 Result:='';
 if (hIpc<>0) then
 try
  Result:=EasyIpc_Ctrl(hIpc,Request);
 except
  on E:Exception do BugReport(E,Self,'EasyIpc_Ctrl');
 end;
end;

 {869}
function TDaqPascal._expenv(const n:LongString):LongString;
begin
 Result:=_crw_fio.ExpEnv(n);
end;

 {872}
function TDaqPascal._createtempfile(const s:LongString):LongString;
begin
 Result:=_crw_fio.CreateTempFile(s);
end;

 {875}
function TDaqPascal._stringreplace(const s,p,n:LongString; f:Integer):LongString;
var flags:TReplaceFlags;
begin
 flags:=[];
 if ((f and GetBitMask(Ord(rfReplaceAll)))<>0) then include(flags,rfReplaceAll);
 if ((f and GetBitMask(Ord(rfIgnoreCase)))<>0) then include(flags,rfIgnoreCase);
 Result:=StringReplace(s,p,n,flags);
end;

 {902}
function TDaqPascal._ansiquotedstr(const s:LongString; q:Char):LongString;
begin
 Result:=SysUtils.AnsiQuotedStr(s,q);
end;

 {905}
function TDaqPascal._ansidequotedstr(const s:LongString; q:Char):LongString;
var p:PChar;
begin
 Result:=s;
 if (s='') then exit;
 if (s[1]<>q) then exit;
 p:=PChar(s);
 Result:=SysUtils.AnsiExtractQuotedStr(p,q);
end;

 {908}
function TDaqPascal._ansiskipquotedstr(const s:LongString; q:Char):LongString;
var p:PChar;
begin
 Result:=s;
 if (s='') then exit;
 if (s[1]<>q) then exit;
 p:=PChar(s);
 Result:=SysUtils.AnsiExtractQuotedStr(p,q);
 if (p=nil) then Result:='' else Result:=p;
end;

 {920}
function TDaqPascal._text_tostring(ref:Integer):LongString;
begin
 with TextByRef(ref) do Result:=Text;
end;

 {930}
function TDaqPascal._extractfirstparam(const arg:LongString; quote:Char):LongString;
begin
 Result:=ExtractFirstParam(arg,quote,WordDelims);
end;

 {933}
function TDaqPascal._skipfirstparam(const arg:LongString; quote:Char):LongString;
begin
 Result:=SkipFirstParam(arg,quote,WordDelims);
end;

 {936}
function TDaqPascal._isoption(const arg,opt:LongString):Boolean;
var i:Integer;
begin
 Result:=IsOption(arg);
 if Result and (opt<>'') then
 for i:=1 to WordCount(opt,ScanSpaces) do begin
  Result:=IsOption(arg,ExtractWord(i,opt,ScanSpaces));
  if Result then Break;
 end;
end;

 {945}
function TDaqPascal._getoptionvalue(const arg:LongString):LongString;
begin
 Result:=GetOptionValue(arg);
end;

 {948}
function TDaqPascal._ansiquotedifneed(const arg:LongString; quote:Char):LongString;
begin
 Result:=AnsiQuotedIfNeed(arg,quote,WordDelims);
end;

 {954}
function TDaqPascal._regexp_init(engine:Integer; pattern:LongString; options:LongString=''):Integer;
begin
 Result:=_crw_regexp.regexp_init(engine,pattern,options);
 if (Result<>0) then _regexplist.Add(PtrIntToPointer(Result));
end;

 {957}
function TDaqPascal._regexp_free(rex:Integer):Boolean;
begin
 Result:=false;
 if (rex<>0) then
 if (_regexplist.IndexOf(PtrIntToPointer(rex))>=0) then begin
  _regexplist.Remove(PtrIntToPointer(rex));
  Result:=_crw_regexp.regexp_free(rex);
 end;
end;

 {958}
function TDaqPascal._regexp_ref(rex:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_ref(rex).ref;
end;

 {959}
function TDaqPascal._regexp_ctrl(rex:Integer; arg:LongString):LongString;
begin
 Result:=_crw_regexp.regexp_ctrl(rex,arg);
end;

 {962}
function TDaqPascal._regexp_test(rex:Integer; arg:LongString):Boolean;
begin
 Result:=_crw_regexp.regexp_test(rex,arg);
end;

 {965}
function TDaqPascal._regexp_exec(rex:Integer; arg:LongString):Integer;
begin
 Result:=_crw_regexp.regexp_exec(rex,arg);
end;

 {968}
function TDaqPascal._regexp_replace(rex:Integer; arg,rep:LongString):LongString;
begin
 Result:=_crw_regexp.regexp_replace(rex,arg,rep);
end;

 {977}
function TDaqPascal._regexp_matchnum(rex:Integer; i:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_matchnum(rex,i);
end;

 {978}
function TDaqPascal._regexp_matchpos(rex:Integer; i,j:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_matchpos(rex,i,j);
end;

 {979}
function TDaqPascal._regexp_matchlen(rex:Integer; i,j:Integer):Integer;
begin
 Result:=_crw_regexp.regexp_matchlen(rex,i,j);
end;

 {980}
function TDaqPascal._regexp_matchstr(rex:Integer; i,j:Integer):LongString;
begin
 Result:=_crw_regexp.regexp_matchstr(rex,i,j);
end;

 {981}
function TDaqPascal._regexp_escape(arg:LongString):LongString;
begin
 Result:=_crw_regexp.regexp_escape(arg);
end;

 {984}
function TDaqPascal._backslash_encode(arg:LongString):LongString;
begin
 Result:=_crw_bsencode.backslash_encode(arg,_backslash_esclist,_backslash_hexlist);
end;

 {987}
function TDaqPascal._backslash_decode(arg:LongString):LongString;
begin
 Result:=_crw_bsencode.backslash_decode(arg);
end;

 {990}
function TDaqPascal._backslash_encoder_ctrl(arg:LongString):LongString;
var p:Integer; n,v:LongString;
begin
 Result:='';
 if (arg='') then Exit;
 p:=Pos('=',arg); n:=arg; v:='';
 if (p>0) then begin n:=Copy(arg,1,p-1); v:=Copy(arg,p+1,Length(arg)-p); end;
 if SameText(n,'EscList') then begin
  Result:=CharSet2Str(_backslash_esclist);
  if (p>0) then _backslash_esclist:=Str2CharSet(v);
 end else
 if SameText(n,'HexList') then begin
  Result:=CharSet2Str(_backslash_hexlist);
  if (p>0) then _backslash_hexlist:=Str2CharSet(v);
 end;
end;

 {993}
function TDaqPascal._fsm_new:Integer;
begin
 Result:=_crw_fsm.fsm_new;
 if (Result<>0) then _fsmlist.Add(PtrIntToPointer(Result));
end;

 {994}
function TDaqPascal._fsm_free(fsm:Integer):Boolean;
begin
 Result:=false;
 if (fsm<>0) then
 if (_fsmlist.IndexOf(PtrIntToPointer(fsm))>=0) then begin
  _fsmlist.Remove(PtrIntToPointer(fsm));
  Result:=_crw_fsm.fsm_free(fsm);
 end;
end;

 {995}
function TDaqPascal._fsm_ref(fsm:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_ref(fsm).Ref;
end;

 {996}
function TDaqPascal._fsm_root(fsm:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_root(fsm);
end;

 {997}
function TDaqPascal._fsm_type(fsm:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_type(fsm);
end;

 {998}
function TDaqPascal._fsm_parent(fsm:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_parent(fsm);
end;

 {999}
function TDaqPascal._fsm_name(fsm:Integer):LongString;
begin
 Result:=_crw_fsm.fsm_name(fsm);
end;

 {1000}
function TDaqPascal._fsm_path(fsm:Integer):LongString;
begin
 Result:=_crw_fsm.fsm_path(fsm);
end;

 {1001}
function TDaqPascal._fsm_ctrl(fsm:Integer; arg:LongString):LongString;
begin
 Result:=_crw_fsm.fsm_ctrl(fsm,arg);
end;

 {1004}
function TDaqPascal._fsm_count(fsm,typ:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_count(fsm,typ);
end;

 {1005}
function TDaqPascal._fsm_items(fsm,typ,i:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_items(fsm,typ,i);
end;

 {1006}
function TDaqPascal._fsm_get_iparam(fsm:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_get_iparam(fsm);
end;

 {1007}
function TDaqPascal._fsm_set_iparam(fsm:Integer; data:Integer):Boolean;
begin
 Result:=_crw_fsm.fsm_set_iparam(fsm,data);
end;

 {1008}
function TDaqPascal._fsm_get_fparam(fsm:Integer):Double;
begin
 Result:=_crw_fsm.fsm_get_fparam(fsm);
end;

 {1009}
function TDaqPascal._fsm_set_fparam(fsm:Integer; data:Double):Boolean;
begin
 Result:=_crw_fsm.fsm_set_fparam(fsm,data);
end;

 {1010}
function TDaqPascal._fsm_get_sparam(fsm:Integer):LongString;
begin
 Result:=_crw_fsm.fsm_get_sparam(fsm);
end;

 {1011}
function TDaqPascal._fsm_set_sparam(fsm:Integer; data:LongString):Boolean;
begin
 Result:=_crw_fsm.fsm_set_sparam(fsm,data);
end;

 {1014}
function TDaqPascal._fsm_add(fsm:Integer; typ:integer; key:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_add(fsm,typ,key);
end;

 {1017}
function TDaqPascal._fsm_find(fsm:Integer; typ:integer; key:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_find(fsm,typ,key);
end;

 {1020}
function TDaqPascal._fsm_get_state(fsm:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_get_state(fsm);
end;

 {1021}
function TDaqPascal._fsm_set_state(fsm:Integer; state:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_set_state(fsm,state);
end;

 {1024}
function TDaqPascal._fsm_set_state(fsm:Integer; state:Integer):Integer; 
begin
 Result:=_crw_fsm.fsm_set_state(fsm,state);
end;

 {1025}
function TDaqPascal._fsm_link(fsm:Integer; arg:LongString):Integer;
begin
 Result:=_crw_fsm.fsm_link(fsm,arg);
end;

 {1028}
function TDaqPascal._fsm_modified(fsm:Integer; delta:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_modified(fsm,delta);
end;

 {1029}
function TDaqPascal._fsm_name_rule(typ:Integer):Integer;
begin
 Result:=_crw_fsm.fsm_name_rule(typ);
end;

 {1030}
function TDaqPascal._upcase(c:Char):Char;
begin
 case c of
  'a'..'z': dec(c,$20);
 end;
 Result:=c;
end;

 {1031}
function TDaqPascal._locase(c:Char):Char;
begin
 case c of
  'A'..'Z': inc(c,$20);
 end;
 Result:=c;
end;

 {1032}
function TDaqPascal._htonl(l:Integer):Integer;
begin
 Result:=sockets.htonl(l);
end;

 {1033}
function TDaqPascal._ntohl(l:Integer):Integer;
begin
 Result:=sockets.ntohl(l);
end;

 {1034}
function TDaqPascal._htons(s:Integer):Integer;
begin
 Result:=sockets.htons(s);
end;

 {1035}
function TDaqPascal._ntohs(s:Integer):Integer;
begin
 Result:=sockets.ntohs(s);
end;

 {1036}
function TDaqPascal._db_create(arg:LongString):Boolean;
begin
 Result:=_crw_dbapi.db_create(arg);
end;

 {1039}
function TDaqPascal._db_connection(eid:Integer; arg:LongString):Integer;
begin
 Result:=_crw_dbapi.db_connection(eid,arg);
 if (Result<>0) then _dblist.Add(PtrIntToPointer(Result));
end;

  {1042}
function TDaqPascal._db_recordset(dbo:Integer; arg:LongString):Integer;
begin
 Result:=_crw_dbapi.db_recordset(dbo,arg);
end;

 {1045}
function TDaqPascal._db_command(dbo:Integer; arg:LongString):Integer;
begin
 Result:=_crw_dbapi.db_command(dbo,arg);
end;

 {1048}
function TDaqPascal._db_free(dbo:Integer):Boolean;
begin
 Result:=false;
 if (dbo<>0) then
 if (_dblist.IndexOf(PtrIntToPointer(dbo))>=0) then begin
  _dblist.Remove(PtrIntToPointer(dbo));
  Result:=_crw_dbapi.db_free(dbo);
 end;
end;

 {1049}
function TDaqPascal._db_ref(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_ref(dbo).ref;
end;

 {1050}
function TDaqPascal._db_root(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_root(dbo).ref;
end;

 {1051}
function TDaqPascal._db_type(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_type(dbo);
end;

 {1052}
function TDaqPascal._db_parent(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_parent(dbo);
end;

 {1053}
function TDaqPascal._db_state(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_state(dbo);
end;

 {1054}
function TDaqPascal._db_close(dbo:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_close(dbo);
end;

 {1055}
function TDaqPascal._db_open(dbo:Integer; opt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_open(dbo,opt);
end;

 {1056}
function TDaqPascal._db_ctrl(dbo:Integer; arg:LongString):LongString;
begin
 Result:=_crw_dbapi.db_ctrl(dbo,arg);
end;

 {1059}
function TDaqPascal._db_bugscount(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_bugscount(dbo);
end;

 {1060}
function TDaqPascal._db_bugsclear(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_bugsclear(dbo);
end;

 {1061}
function TDaqPascal._db_errors(dbo:Integer):LongString;
begin
 Result:=_crw_dbapi.db_errors(dbo);
end;

 {1062}
function TDaqPascal._db_errorscount(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_errorscount(dbo);
end;

 {1063}
function TDaqPascal._db_errorsclear(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_errorsclear(dbo);
end;

 {1064}
function TDaqPascal._db_execute(dbo:Integer; arg:LongString; opt:Integer):Integer;
begin
 Result:=_crw_dbapi.db_execute(dbo,arg,opt).ref;
end;

 {1067}
function TDaqPascal._db_cancel(dbo:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_cancel(dbo);
end;

 {1068}
function TDaqPascal._db_update(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_update(dbr);
end;

 {1069}
function TDaqPascal._db_cancelupdate(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_cancelupdate(dbr);
end;

 {1070}
function TDaqPascal._db_begintrans(dbc:Integer):Integer;
begin
 Result:=_crw_dbapi.db_begintrans(dbc);
end;

 {1071}
function TDaqPascal._db_committrans(dbc:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_committrans(dbc);
end;

 {1072}
function TDaqPascal._db_rollbacktrans(dbc:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_rollbacktrans(dbc);
end;

 {1073}
function TDaqPascal._db_bof(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_bof(dbr);
end;

 {1074}
function TDaqPascal._db_eof(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_eof(dbr);
end;

 {1075}
function TDaqPascal._db_movefirst(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_movefirst(dbr);
end;

 {1076}
function TDaqPascal._db_movelast(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_movelast(dbr);
end;

 {1077}
function TDaqPascal._db_movenext(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_movenext(dbr);
end;

 {1078}
function TDaqPascal._db_moveprevious(dbr:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_moveprevious(dbr);
end;

 {1079}
function TDaqPascal._db_fieldscount(dbr:Integer):Integer;
begin
 Result:=_crw_dbapi.db_fieldscount(dbr);
end;

 {1080}
function TDaqPascal._db_fieldsnames(dbr:Integer; i:Integer):LongString;
begin
 Result:=_crw_dbapi.db_fieldsnames(dbr,i);
end;

 {1081}
function TDaqPascal._db_fieldstypes(dbr:Integer; id:LongString):Integer;
begin
 Result:=_crw_dbapi.db_fieldstypes(dbr,id);
end;

 {1084}
function TDaqPascal._db_fieldsasint(dbr:Integer; id:LongString; op:Char; arg:Integer):Integer;
begin
 Result:=_crw_dbapi.db_fieldsasint(dbr,id,op,arg);
end;

 {1087}
function TDaqPascal._db_fieldsasfloat(dbr:Integer; id:LongString; op:Char; arg:Double):Double;
begin
 Result:=_crw_dbapi.db_fieldsasfloat(dbr,id,op,arg);
end;

 {1090}
function TDaqPascal._db_fieldsasstring(dbr:Integer; id:LongString; op:Char; arg:LongString):LongString;
begin
 Result:=_crw_dbapi.db_fieldsasstring(dbr,id,op,arg);
end;

 {1099}
function TDaqPascal._db_addnew(dbr:Integer; arg:LongString):Boolean;
begin
 Result:=_crw_dbapi.db_addnew(dbr,arg);
end;

 {1102}
function TDaqPascal._db_delete(dbr:Integer; aff:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_delete(dbr,aff);
end;

 {1103}
function TDaqPascal._db_requery(dbr:Integer; opt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_requery(dbr,opt);
end;

 {1104}
function TDaqPascal._db_resync(dbr:Integer; aff,res:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_resync(dbr,aff,res);
end;

 {1105}
function TDaqPascal._db_supports(dbr:Integer; opt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_supports(dbr,opt);
end;

 {1106}
function TDaqPascal._db_save(dbr:Integer; dst:LongString; fmt:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_save(dbr,dst,fmt);
end;

 {1112}
function TDaqPascal._strtimefmt(const fmt:LongString; ms:Double):LongString;
begin
 Result:=_crw_rtc.StrTimeFmt(fmt,ms,stfm_Ignore);
end;

 {1115}
function TDaqPascal._shm_init(name:LongString; size,mode:Integer):Integer;
begin
 Result:=_crw_sharm.shm_init(name,size,mode);
 if (Result<>0) then _shmlist.Add(PtrIntToPointer(Result));
end;

 {1118}
function TDaqPascal._shm_ref(shm:Integer):Integer;
begin
 Result:=_crw_sharm.shm_ref(shm).Ref;
end;

 {1119}
function TDaqPascal._shm_free(shm:Integer):Boolean;
begin
 Result:=false;
 if (shm<>0) then
 if (_shmlist.IndexOf(PtrIntToPointer(shm))>=0) then begin
  _shmlist.Remove(PtrIntToPointer(shm));
  Result:=_crw_sharm.shm_free(shm);
 end;
end;

 {1120}
function TDaqPascal._shm_delink(name:LongString):Boolean;
begin
 Result:=_crw_sharm.shm_delink(name);
end;

 {1123}
function TDaqPascal._shm_iop(shm:Integer; offset:Integer; op:Char; data:LongInt):LongInt;
begin
 Result:=_crw_sharm.shm_iop(shm,offset,op,data);
end;

 {1124}
function TDaqPascal._shm_rop(shm:Integer; offset:Integer; op:Char; data:Double):Double;
begin
 Result:=_crw_sharm.shm_rop(shm,offset,op,data);
end;

 {1125}
function TDaqPascal._shm_fop(shm:Integer; offset:Integer; op:Char; data:Single):Single;
begin
 Result:=_crw_sharm.shm_fop(shm,offset,op,data);
end;

 {1126}
function TDaqPascal._shm_sop(shm:Integer; offset:Integer; op:Char; data:LongString):LongString;
begin
 Result:=_crw_sharm.shm_sop(shm,offset,op,data);
end;

 {1129}
function TDaqPascal._shm_ctrl(shm:Integer; arg:LongString):LongString;
begin
 Result:=_crw_sharm.shm_ctrl(shm,arg);
end;

 {1132}
function TDaqPascal._shm_ioresult(code:Integer):Integer;
begin
 Result:=_crw_sharm.shm_ioresult(code);
end;

 {1133}
function TDaqPascal._strtointbase(s:LongString; base,def:Integer):Integer;
begin
 Result:=StrToIntBase(s,base,def);
end;

 {1136}
function TDaqPascal._inttostrbase(value,base,width:Integer):LongString;
begin
 Result:=IntToStrBase(value,base,width);
end;

 {1137}
function TDaqPascal._AdaptFileName(const FN:LongString):LongString;
begin
 Result:=_crw_str.AdaptFileName(FN,afnm_cur);
end;

 {1140}
function TDaqPascal._AdaptExeFileName(const FN:LongString):LongString;
begin
 Result:=_crw_str.AdaptExeFileName(FN,afnm_cur);
end;

 {1143}
function TDaqPascal._AdaptFileNameMode(m:Integer):Integer;
begin
 Result:=afnm_cur;
 if (m=-1) then afnm_cur:=afnm_Def else afnm_cur:=m;
end;

 {1144}
function TDaqPascal._skipwords(n:Integer; const s:LongString):LongString;
begin
 Result:=_crw_str.SkipWords(n,s,worddelims);
end;

 {1147}
function TDaqPascal._phrasecount(const s:LongString):Integer;
begin
 Result:=_crw_str.PhraseCount(s,worddelims,phrasequotes);
end;

 {1150}
function TDaqPascal._extractphrase(n:Integer; const s:LongString):LongString;
begin
 Result:=_crw_str.ExtractPhrase(n,s,worddelims,phrasequotes);
end;

 {1153}
function TDaqPascal._skipphrases(n:Integer; const s:LongString):LongString;
begin
 Result:=_crw_str.SkipPhrases(n,s,worddelims,phrasequotes);
end;

 {1156}
function TDaqPascal._phraselisttotextlines(const s:LongString):LongString;
begin
 Result:=_crw_str.PhraseListToTextLines(s,worddelims,phrasequotes);
end;

 {1159}
function TDaqPascal._phrasequotes(const s:LongString):LongString;
const zero:Integer=0;
begin
 Result:=phrasequotes;
 if (s<>'') then
 if (s<>Dump(zero))
 then phrasequotes:=Trim(s)
 else phrasequotes:=QuoteMark+Apostrophe;
end;

 {1162}
function TDaqPascal._AdaptDllFileName(const FN:LongString):LongString;
begin
 Result:=_crw_str.AdaptDllFileName(FN,afnm_cur);
end;

 {1165}
function TDaqPascal._AdaptLnkFileName(const FN:LongString):LongString;
begin
 Result:=_crw_str.AdaptLnkFileName(FN,afnm_cur);
end;

 {1168}
function TDaqPascal._db_engineid(dbo:Integer):Integer;
begin
 Result:=_crw_dbapi.db_engineid(dbo);
end;

 {1169}
function TDaqPascal._db_active(dbo:Integer):Boolean;
begin
 Result:=_crw_dbapi.db_active(dbo);
end;

 {1170}
function TDaqPascal._pct_encode(const arg:LongString):LongString;
begin
 Result:=_crw_uri.percent_encode(arg,_pct_reserved);
end;

 {1173}
function TDaqPascal._pct_decode(const arg:LongString):LongString;
begin
 Result:=_crw_uri.percent_decode(arg,@_pct_errors);
end;

 {1176}
function TDaqPascal._pct_encoder_ctrl(const arg:LongString):LongString;
begin
 Result:=_crw_uri.percent_encoder_ctrl(arg,_pct_reserved,_pct_errors);
end;

 {1179}
function TDaqPascal._ReadFileToBuffer(const FileName:LongString; Count,Offset:Integer):LongString;
begin
 Result:='';
 if IsEmptyStr(FileName) then Exit;
 if (Count<0) or (Offset<0) then Exit;
 if not FileIsReadable(FileName) then Exit;
 if (Offset=0) then begin
  Result:=_crw_fio.StringFromFile(FileName,Count);
 end else begin
  if (Count=0) then Count:=GetFileSize(FileName)-Offset;
  Result:=_crw_fio.ReadFileToBuffer(FileName,Count,Offset);
 end;
end;

 {1182}
function TDaqPascal._WriteBufferToFile(const FileName,Buffer:LongString; Offset:Integer):Integer;
begin
 Result:=0;
 if (Buffer='') or (Offset<0) then Exit;
 Result:=_crw_fio.WriteBufferToFile(FileName,Buffer,Offset);
end;

 {1191}
function TDaqPascal._GetFileProperties(const FileName,Properties:LongString):LongString;
begin
 Result:=_crw_fio.GetFilePropertiesAsText(FileName,Properties);
end;

 {1200}
function TDaqPascal._SysLogNotable(aSeverity:Integer):Boolean;
begin
 Result:=SysLog.Notable(aSeverity);
end;

 {1201}
function TDaqPascal._SysLogNote(aSeverity:Integer; aData:LongString):Integer;
begin
 Result:=0;
 if SysLog.Notable(aSeverity) and (aData<>'')
 then Result:=SysLog.Note(0,aSeverity,TheDevice.SysLogSign,aData);
end;

 {1204}
function TDaqPascal._wdt_timeout(aTimeout:Integer):Integer;
begin
 Result:=TProgramDevice(TheDevice).Polling.WdtTimeout;
 TProgramDevice(TheDevice).Polling.WdtTimeout:=Max(0,aTimeout);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_daqpascalcompiler;
begin
 InitHasher;
end;

procedure Free_crw_daqpascalcompiler;
begin
 FreeHasher;
end;

initialization

 Init_crw_daqpascalcompiler;

finalization

 Free_crw_daqpascalcompiler;

end.

//////////////
// END OF FILE
//////////////

