#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CRW-DAQ script to print text file as text.
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

readonly logfile="$CRW_DAQ_VAR_TMP_DIR/$scriptbase.log";
declare  lpr_opt="";

function fatal(){
 1>&2 echo "$2";
 exit $1;
};

function check_environ(){
 if [ -z "$CRW_DAQ_VAR_TMP_DIR" ]; then
  fatal 1 "Error: script should be called from CRW-DAQ system only.";
 fi;
};
function enter_logfile(){
 echo ""                                              >> $logfile;
 echo "$(date +%Y.%m.%d-%H:%M:%S): Enter $scriptfile" >> $logfile;
};

function leave_logfile(){
 echo "$(date +%Y.%m.%d-%H:%M:%S): Leave $scriptfile" >> $logfile;
 echo ""                                              >> $logfile;
};

function get_file_size(){
 if [ -e "$1" ]; then
  stat -c %s "$1";
 else
  echo "0";
 fi;
};

function delete_files(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -fv "$1"; fi;
  shift;
 done;
};

function cleanup_dirs(){
 let KB=1024; let MB=$KB*$KB; let maxsize=32*$MB;
 echo "$(date +%Y.%m.%d-%H:%M:%S): Cleanup dirs: $maxsize $*";
 for dir in $*; do
  if [ -d "$dir" ]; then
   for fn in $(ls -1 "$dir"); do
    local file="$dir/$fn";
    if [ -f "$file" ]; then
     if [ "$file" = "$logfile" ]; then continue; fi;
     local size="$(get_file_size $file)";
     if [ $size -gt $maxsize ]; then
      old="$dir/obsolete";
      mkdir -p "$old";
      mv -fv "$file" "$old/$fn";
     fi;
    fi;
   done;
  fi;
 done;
};

function cleanup_files(){
 cleanup_dirs "$CRW_DAQ_VAR_TMP_DIR" >> $logfile;
};

function add_opt(){
 if [ $# -eq 2 ]; then
  if [ -z "$lpr_opt" ]; then
   lpr_opt+="-o $1=$2";
  else
   lpr_opt+=" -o $1=$2";
  fi;
 fi;
};

##################################################################################################################
# -o cpi=N
#      Sets the number of characters per inch to use when printing a text file. The default is 10.
# -o lpi=N
#      Sets the number of lines per inch to use when printing a text file. The default is 6.
# -o page-bottom=N
# -o page-left=N
# -o page-right=N
# -o page-top=N
#      Sets the page margins when printing text files. The values are in points - there are 72 points to the inch.
##################################################################################################################

##############################
# 1 inch = 2.54 cm = 72 points
##############################
function cm2pt(){
 local cm="$1";
 local pt="$(echo "$cm * 72 / 2.54" | bc)";
 printf "%1.0f" "$pt";
};

function add_options(){
 add_opt media a4;
 local pl="$(printenv LP_PAGE_INDENTS | cut -d' ' -f1)";
 local pt="$(printenv LP_PAGE_INDENTS | cut -d' ' -f2)";
 local pr="$(printenv LP_PAGE_INDENTS | cut -d' ' -f3)";
 local pb="$(printenv LP_PAGE_INDENTS | cut -d' ' -f4)";
 add_opt page-left   $(cm2pt ${pl:-0});
 add_opt page-top    $(cm2pt ${pt:-0});
 add_opt page-right  $(cm2pt ${pr:-0});
 add_opt page-bottom $(cm2pt ${pb:-0});
};

function text_print(){
 if [ -e "$1" ]; then
  echo lpr $lpr_opt "$1";
  eval lpr $lpr_opt "$1";
  case $1 in
   *.tmp) rm -fv "$1"; ;;
   *) ;;
  esac;
 else
  fatal 2 "Error: not found $1";
 fi;
};

function main(){
 add_options;
 check_environ;
 enter_logfile;
 cleanup_files   | tee -a $logfile;
 text_print "$1" | tee -a $logfile;
 leave_logfile;
};

main "$@";

#############
# END OF FILE
#############
