 {
 ****************************************************************************
 CRW32 project
 Copyright (C) by Kuryakin Alexey, Sarov, Russia, 2001, <kouriakine@mail.ru>
 Direct CPU I/O port access for user mode applications.
 Uses freeware giveio.sys driver written by Dale Roberts.
 Note: direct I/O assess is potentially dangerous thing, you may hang system,
       if will not be careful. But if direct I/O takes about 1.8/2.5/3.9 mks
       per byte/word/dword I/O, "right" kernel mode driver, via call of
       DeviceIoControl, will takes at least 30 mks.
 Modifications:
  20011211 - Creation & test Ok.
 ****************************************************************************
 }

unit _pio; { port input /output}

{$I _sysdef}

interface

uses
 sysutils,
 windows,
 winsvc,
 _alloc,
 _str,
 _fio,
 _fifo
 ;

 {
 ******************************************************************************
 Purpose:
   Port[...]  is Borland Pascal - like array to access byte  CPU port I/O.
   PortW[...] is Borland Pascal - like array to access word  CPU port I/O.
   PortL[...] is Borland Pascal - like array to access dword CPU port I/O.
 Note:
  1) Before use port I/O, you need call IOPM_Open, see below.
  1) You should not create/destroy Port,PortW,PortL, system do it automatically.
 Example:
   program IOPM_Test;
   uses _pio;
   begin
    if IOPM_Open('giveio.sys') and IOPM_CheckIO then begin
     Port[$300]:=Port[$300]+1;
     PortW[$301]:=PortW[$301] and $F000;
     PortL[$310]:=PortL[$310] or $80000000;
     // & etc.
    end;
    IOPM_Close;
   end;
 ******************************************************************************
 }
type
 TBytePort = class(TLatch)
 private
  procedure SetPort(address : Word; data : Byte);
  function  GetPort(address : Word) : Byte;
 public
  property  ByteIO[address : Word] : Byte read GetPort write SetPort;  default;
 end;

type
 TWordPort = class(TLatch)
 private
  procedure SetPort(address : Word; data : Word);
  function  GetPort(address : Word) : Word;
 public
  property  WordIO[address : Word] : Word read GetPort write SetPort;  default;
 end;

type
 TLongPort = class(TLatch)
 private
  procedure SetPort(address : Word; data : LongWord);
  function  GetPort(address : Word) : LongWord;
 public
  property  LongIO[address : Word] : LongWord read GetPort write SetPort;  default;
 end;

const
 Port  : TBytePort = nil;
 PortW : TWordPort = nil;
 PortL : TLongPort = nil;

 {
 ******************************************************************************
 Purpose:
   This functions uses to give access to direct port I/O under Windows 9x/NT/2k.
   IOPM         - "Input / Output Permission Map"
   IOPM_CheckIO - try to read some port to check direct I/O access.
   IOPM_Open    - open the IOPM driver to enable direct I/O access.
                  FileName is name of driver, if Section='' or Variable=''.
                  FileName is name of config file, if Section<>'' and Variable<>''.
                  Section is name of config file section where driver path placed.
                  Variable is name of driver path variable in this section.
                  For example:
                   IOPM_Open('example.cfg','[IOPM]','DriverPath')
                  where example.cfg contains:
                   [IOPM]
                   DriverPath = c:\winnt\system32\driver\giveio.sys
   IOPM_Close   - close the IOPM driver to free memory and disable port I/O.
   IOPM_History - history of IOPM driver work for diagnostic reasons.
 Note:
   If IOPM_Open called once, direct I/O access enabled until program stop
   execution. IOPM_Close only free memory and set internal flag off.
 ******************************************************************************
 }
const
 IOPM_DEVICE_NAME  = 'giveio';
 IOPM_DISPLAY_NAME = 'CPU port I/O permission map driver.';

function  IOPM_CheckIO:Boolean;
function  IOPM_Opened:Boolean;
function  IOPM_Open(const FileName : ShortString = IOPM_DEVICE_NAME;
                    const Section  : ShortString = '';
                    const Variable : ShortString = ''):Boolean;
procedure IOPM_Close;
function  IOPM_History:LongString;

 {
 *******************************************************************************
 Purpose:
   TSysDriverLoader encapsulate *.SYS driver loader functions.
   To load and start *.SYS driver, we have to:
    1) Open Service Control Manager
    2) Install driver via Service Control Manager, if driver was not already installed.
    3) Start the driver via Service Control Manager, if driver was not already started.
    4) Open device via CreateFile('\\.\DriverName'...) call.
   To stop working, we have to:
    1) Close device.
    2) Stop driver, if driver was not already started.
    3) Uninstall driver, if driver was not already installed.
    4) Close Service Control Manager.
 *******************************************************************************
 }
type
 TSysDriverLoader = class(TLatch)
 protected
  myManager             : SC_HANDLE;            { Handle for Service Control Manager }
  myDevice              : SC_HANDLE;            { Handle for device }
  myDriverName          : array[byte] of char;  { The name of driver }
  myDriverPath          : array[byte] of char;  { Full driver path }
  myDisplayName         : array[byte] of char;  { Driver short description }
  myStatus              : DWORD;                { Status of last operation }
  myHistory             : TText;                { History of loading }
  myWasAlreadyInstalled : Boolean;              { Driver was installed? }
  myWasAlreadyStarted   : Boolean;              { Driver was started? }
  myStayResident        : Boolean;              { Don't uninstall service }
  myStayRunning         : Boolean;              { Don't stop service }
  procedure   Annals(const aMsg:ShortString; aCod:DWORD);
  function    OpenServiceControlManager:DWORD;
  function    CloseServiceControlManager:DWORD;
  function    InstallService:DWORD;
  function    StartService:DWORD;
  function    StopService:DWORD;
  function    KillService:DWORD;
  function    OpenDevice:DWORD;
  function    CloseDevice:DWORD;
 public
  constructor Create(const TheDriverPath  : ShortString;
                     const TheDisplayName : ShortString = '';
                           StayResident   : Boolean = false;
                           StayRunning    : Boolean = false);
  destructor  Destroy; override;
  function    OpenDriver:Boolean;
  procedure   CloseDriver;
  function    History:LongString;
 end;

const
 THE_IOPM_DRIVER : TSysDriverLoader = nil;

 {
 ******************************************************************************
 Purpose:
   Borland Pascal - like PC speaker sound routines with direct port I/O.
 ******************************************************************************
 }
procedure SoundOn(wFrequency : Word);
procedure SoundOff;
procedure Sound(wFrequency : Word; wDelay : LongWord);

implementation

const
 NULL              = integer(nil);
 IOPM_ON : boolean = false;

 {
 ************************
 TBytePort implementation
 ************************
 }
function TBytePort.GetPort(address : Word) : Byte;
asm
 cmp IOPM_ON, false
 jz  @exit
 cmp eax, NULL
 jz  @exit
 in  al, dx
@exit:
end;

procedure TBytePort.SetPort(address : Word; data : Byte);
asm
 cmp IOPM_ON, false
 jz  @exit
 cmp eax, NULL
 jz  @exit
 mov al, cl
 out dx, al
@exit:
end;

 {
 ************************
 TWordPort implementation
 ************************
 }
function TWordPort.GetPort(address : Word) : Word;
asm
 cmp IOPM_ON, false
 jz  @exit
 cmp eax, NULL
 jz  @exit
 in  ax, dx
@exit:
end;

procedure TWordPort.SetPort(address : Word; data : Word);
asm
 cmp IOPM_ON, false
 jz  @exit
 cmp eax, NULL
 jz  @exit
 mov ax, cx
 out dx, ax
@exit:
end;

 {
 ************************
 TLongPort implementation
 ************************
 }
function TLongPort.GetPort(address : Word) : LongWord;
asm
 cmp IOPM_ON, false
 jz  @exit
 cmp eax, NULL
 jz  @exit
 in  eax, dx
@exit:
end;

procedure TLongPort.SetPort(address : Word; data : LongWord);
asm
 cmp IOPM_ON, false
 jz  @exit
 cmp eax, NULL
 jz  @exit
 mov eax, ecx
 out dx, eax
@exit:
end;

 {
 ************************************
 IOPM driver functions implementation
 ************************************
 }
function IOPM_CheckIO:Boolean;
begin
 Result:=true;
 try
  asm
   in al,$61                      { try read speaker port }
  end;
 except
  on EPrivilege do Result:=false; { if exception raised, direct I/O disabled }
 end;
end;

function  IOPM_Opened:Boolean;
begin
 Result:=IOPM_ON;
end;

function  IOPM_Open(const FileName : ShortString = IOPM_DEVICE_NAME;
                    const Section  : ShortString = '';
                    const Variable : ShortString = ''):Boolean;
var DriverPath:ShortString;
begin
 IOPM_Close;
 if (Win32Platform = VER_PLATFORM_WIN32_NT) then begin
  if IsEmptyStr(Section) or IsEmptyStr(Variable)
  then DriverPath:=UnifyFileAlias(SmartFileRef(FileName,'.SYS',ProgName))
  else if not ReadIniFilePath(FileName,Section,Variable,HomeDir,DriverPath) then DriverPath:='';
  THE_IOPM_DRIVER:=TSysDriverLoader.Create(DriverPath,IOPM_DISPLAY_NAME,false,false);
  Result:=THE_IOPM_DRIVER.OpenDriver;
  { Don't need driver after enable direct I/O access for THIS process }
  THE_IOPM_DRIVER.CloseDriver;
 end else Result:=true;
 IOPM_ON:=Result;
end;

procedure IOPM_Close;
begin
 IOPM_ON:=false;
 Kill(TObject(THE_IOPM_DRIVER));
end;

function  IOPM_History:LongString;
begin
 Result:='';
 if (Win32Platform = VER_PLATFORM_WIN32_NT) then begin
  if Assigned(THE_IOPM_DRIVER)
  then Result:=THE_IOPM_DRIVER.History
  else Result:='IOPM driver not assigned.';
 end else Result:='Under Windows 95/98 IOPM driver does not need.';
end;

 {
 *******************************
 TSysDriverLoader implementation
 *******************************
 }
constructor TSysDriverLoader.Create(const TheDriverPath  : ShortString;
                                    const TheDisplayName : ShortString = '';
                                          StayResident   : Boolean = false;
                                          StayRunning    : Boolean = false);
begin
 inherited Create;
 myManager:=0;
 myDevice:=INVALID_HANDLE_VALUE;
 StrPCopy(myDriverName,UnifyAlias(ExtractFileName(TheDriverPath)));
 StrPCopy(myDriverPath,UnifyFileAlias(SmartFileRef(TheDriverPath,'.SYS',ProgName)));
 StrPCopy(myDisplayName,TheDisplayName);
 if StrLen(myDisplayName)=0 then StrCopy(myDisplayName,myDriverName);
 myStatus:=ERROR_SUCCESS;
 myHistory:=NewText;
 myWasAlreadyInstalled:=false;
 myWasAlreadyStarted:=false;
 myStayResident:=StayResident;
 myStayRunning:=StayRunning;
end;

destructor TSysDriverLoader.Destroy;
begin
 CloseDriver;
 Kill(myHistory);
 inherited Destroy;
end;

function TSysDriverLoader.OpenDriver:Boolean;
begin
 Result:=false;
 if Assigned(Self) then begin
  if myDevice <> INVALID_HANDLE_VALUE then CloseDevice;
  if myManager <> 0 then CloseServiceControlManager;
  Result := (OpenServiceControlManager = ERROR_SUCCESS) and
            (myManager <> 0) and
            (InstallService = ERROR_SUCCESS) and
            (StartService = ERROR_SUCCESS) and
            (OpenDevice = ERROR_SUCCESS) and
            (myDevice <> INVALID_HANDLE_VALUE);
 end;
end;

procedure TSysDriverLoader.CloseDriver;
begin
 if Assigned(Self) then begin
  if myDevice <> INVALID_HANDLE_VALUE then CloseDevice;
  OpenServiceControlManager;
  if myManager <> 0 then begin
   if not (myWasAlreadyStarted or myStayRunning) then StopService;
   if not (myWasAlreadyInstalled or myStayResident) then KillService;
   CloseServiceControlManager;
  end;
 end;
end;

function TSysDriverLoader.History:LongString;
begin
 Result:='';
 if Assigned(Self) then Result:=myHistory.Text;
end;

procedure TSysDriverLoader.Annals(const aMsg:ShortString; aCod:DWORD);
begin
 myStatus:=aCod;
 myHistory.AddLn(Pad(aMsg,30)+' Err='+d2s(aCod,-5)+' Msg='+SysErrorMessage(aCod));
end;

function TSysDriverLoader.OpenServiceControlManager:DWORD;
const
 SC_MANAGER_USERS_ACCESS = SC_MANAGER_CONNECT or SC_MANAGER_QUERY_LOCK_STATUS or SC_MANAGER_ENUMERATE_SERVICE;
 SC_MANAGER_ADMIN_ACCESS = SC_MANAGER_CONNECT or SC_MANAGER_QUERY_LOCK_STATUS or SC_MANAGER_ENUMERATE_SERVICE or SC_MANAGER_CREATE_SERVICE;
begin
 Result := ERROR_SUCCESS;
 if myManager = 0 then begin
  myManager := winsvc.OpenSCManager(nil, nil, SC_MANAGER_ALL_ACCESS);
  if (myManager=0) and (GetLastError=ERROR_ACCESS_DENIED)
  then myManager := winsvc.OpenSCManager(nil, nil, SC_MANAGER_ADMIN_ACCESS);
  if (myManager=0) and (GetLastError=ERROR_ACCESS_DENIED)
  then myManager := winsvc.OpenSCManager(nil, nil, SC_MANAGER_USERS_ACCESS);
  if myManager=0 then Result:=GetLastError;
 end;
 Annals('OpenServiceControlManager:',Result);
end;

function TSysDriverLoader.CloseServiceControlManager:DWORD;
begin
 Result:=ERROR_SUCCESS;
 if myManager <> 0 then
 if not winsvc.CloseServiceHandle(myManager) then Result:=GetLastError;
 myManager:=0;
 Annals('CloseServiceControlManager:',Result);
end;

function TSysDriverLoader.InstallService:DWORD;
var
 hService : SC_HANDLE;
begin
 Result:=ERROR_SUCCESS;
 { Is the driver already installed in the ServiceControlManager? }
 hService:=winsvc.OpenService(myManager, myDriverName, SERVICE_QUERY_STATUS);
 if (hService<>0) then begin   { Driver already installed }
  myWasAlreadyInstalled:=true;
  if not winsvc.CloseServiceHandle(hService) then Result:=GetLastError;
  Annals('InstallDriver(already installed):',Result);
 end else begin                { No driver installed, add to SCM database }
  hService := winsvc.CreateService(myManager, myDriverName, myDisplayName,
                     SERVICE_ALL_ACCESS,
                     SERVICE_KERNEL_DRIVER, SERVICE_DEMAND_START,
                     SERVICE_ERROR_NORMAL, myDriverPath, nil, nil, nil, nil, nil);
  if (hService=0) and (GetLastError=ERROR_ACCESS_DENIED) then
  hService := winsvc.CreateService(myManager, myDriverName, myDisplayName,
                     SERVICE_START or SERVICE_STOP or _DELETE or SERVICE_QUERY_STATUS,
                     SERVICE_KERNEL_DRIVER, SERVICE_DEMAND_START,
                     SERVICE_ERROR_NORMAL, myDriverPath, nil, nil, nil, nil, nil);
  if hService = 0
  then Result := GetLastError
  else if not winsvc.CloseServiceHandle(hService)
  then Result:=GetLastError;
  Annals('DriverInstall:',Result);
 end;
end;

function TSysDriverLoader.StartService:DWORD;
var
 hService            : SC_HANDLE;
 lpServiceArgVectors : PChar;
 sStatus             : TServiceStatus;
begin
 Result:=ERROR_SUCCESS;
 lpServiceArgVectors := nil;
 hService:=winsvc.OpenService(myManager, myDriverName, SERVICE_QUERY_STATUS);
 if hService<>0 then begin
  if winsvc.QueryServiceStatus(hService, sStatus) then begin
   if sStatus.dwCurrentState=SERVICE_RUNNING then myWasAlreadyStarted:=true else begin
    winsvc.CloseServiceHandle(hService);
    hService := winsvc.OpenService(myManager, myDriverName, SERVICE_START);
    if not winsvc.StartService(hService, 0, lpServiceArgVectors) then Result:=GetLastError;
   end;
  end else Result:=GetLastError;
  if not winsvc.CloseServiceHandle(hService) then Result:=GetLastError;
 end else Result:=GetLastError;
 Annals('DriverStart:',Result);
end;

function TSysDriverLoader.StopService:DWORD;
var
 hService      : SC_HANDLE;
 serviceStatus : TServiceStatus;
begin
 Result := ERROR_SUCCESS;
 if not myWasAlreadyStarted then begin
  hService := winsvc.OpenService(myManager, myDriverName, SERVICE_STOP or SERVICE_QUERY_STATUS);
  if hService <> 0 then begin
   if not winsvc.ControlService(hService, SERVICE_CONTROL_STOP, serviceStatus)
   then Result := GetLastError;
   if not winsvc.CloseServiceHandle(hService) then Result := GetLastError;
  end else Result := GetLastError;
 end;
 Annals('DriverStop:',Result);
end;

function TSysDriverLoader.KillService:DWORD;
var
 hService : SC_HANDLE;
begin
 //StopService;
 Result:= ERROR_SUCCESS;
 if not myWasAlreadyInstalled then begin
  hService := winsvc.OpenService(myManager, myDriverName, _DELETE);
  if hService <> 0 then begin
   if not winsvc.DeleteService(hService) then Result := GetLastError;
   if not winsvc.CloseServiceHandle(hService) then Result := GetLastError;
  end else Result := GetLastError;
 end;
 Annals('DriverRemove:',Result);
end;

function TSysDriverLoader.OpenDevice:DWORD;
var buff:array[byte] of char;
begin
 Result := ERROR_SUCCESS;
 if myDevice <> INVALID_HANDLE_VALUE then CloseDevice;
 myDevice := CreateFile(StrCat(StrCopy(buff,'\\.\'),myDriverName),{ lpFileName: PChar            }
                        GENERIC_READ or GENERIC_WRITE,            { dwDesiredAccess: integer     }
                        0,                                        { dwShareMode: Integer         }
                        PSECURITY_DESCRIPTOR(nil),                { lpSecurityAttributes         }
                        OPEN_EXISTING,                            { dwCreationDisposition: DWORD }
                        FILE_ATTRIBUTE_NORMAL,                    { dwFlagsAndAttributes: DWORD  }
                        0);                                       { hTemplateFile: THandle       }
 if myDevice = INVALID_HANDLE_VALUE then Result := GetLastError;
 Annals('DeviceOpen:',Result);
end;

function TSysDriverLoader.CloseDevice:DWORD;
begin
 Result := ERROR_SUCCESS;
 if myDevice <> INVALID_HANDLE_VALUE then
 if not CloseHandle(myDevice) then Result := GetLastError;
 myDevice := INVALID_HANDLE_VALUE;
 Annals('DeviceClose:',Result);
end;

 {
 ****************************************
 PC speaker routines uses direct port I/O
 ****************************************
 }
procedure SoundOn(wFrequency : Word);
const
 QuartzFrequency = 1193180.0;
var
 wTone : Word;
begin
 if wFrequency>0 then begin
  wTone:=Round(QuartzFrequency/wFrequency);
  Port[$61]:=Port[$61] or $03;
  Port[$43]:=$B6;
  Port[$42]:=Lo(wTone);
  Port[$42]:=Hi(wTone);
 end;
end;

procedure SoundOff;
begin
 Port[$61]:=(Port[$61] and $FC);
end;

procedure Sound(wFrequency : Word; wDelay : LongWord);
begin
 SoundOn(wFrequency);
 Sleep(wDelay);
 SoundOff;
end;

initialization

 Port  := TBytePort.Create;
 PortW := TWordPort.Create;
 PortL := TLongPort.Create;

finalization

 Kill(TObject(Port));
 Kill(TObject(PortW));
 Kill(TObject(PortL));
 IOPM_Close;

end.

