 {
 ****************************************************************************
 CRW32 project
 Copyright (C) by Kuryakin Alexey, Sarov, Russia, 2002, <kouriakine@mail.ru>
 Data acquisition system tags database.
 Modifications:
 20020204 - Creation (uses CRW16) & test
 20171115 - HashList to improve performance of FindTag
 20171202 - ReinitTags, use _HASH library and HashList_DefaultHasher
 20190320 - LockTags,UnLockTags
 20190321 - iAtomicTagOp,rAtomicTagOp
 20200615 - CountTags
 20210423 - GetTagColor,GetTagParam,GetTagTimer
 20221116 - tag_type_nil,tag_type_int,tag_type_real,tag_type_string
 20230614 - SetTagsSpinlockCount
 ****************************************************************************
 }
unit _daqtags; { data acquision system tags }

{$I _sysdef}

interface

uses
 sysutils, windows, classes, contnrs, math, _alloc, _fifo, _str, _fio, _hl;

 {
 *******************************************************************************
 Daq tags is thread - safety database of common variables, uses by data
 acquisition system. Tags high protected against crash and errors.
 Each tag has:
  1) Unique name, a string to search variable by name from any thread.
  2) Unique reference, a integer value to access variable.
  3) Type of variable: 1=Integer, 2=Real=Double, 3=String.
  4) Value of specified type (iGetTag,rGetTag,sGetTag).
  5) Associated attributes (GetTagColor,GetTagParam,GetTagTimer).
 To access tag, you need find tag reference by known name via FindTag call and
 use this reference to access tag variable via xGetTag/xSetTag calls.
 If tag is not exists, FindTag returns 0, TypeTag returns 0, NameTag returns
 empty string.
 The typical code of tags usage are:
  var tag:Integer; Value:Double;
  begin
   tag:=FindTag('name');             // Find tag reference in database by name.
   if TypeTag(tag)=2 then begin      // Check, is tag of real type?
    Value:=rGetTag(tag);             // Read  real tag value.
    rSetTag(tag,Value);              // Write real tag value.
   end else Echo('Invalid tag!');    // Tag not found or has another type.
  end;
 *******************************************************************************
 }

const
 OriginTags = 256;                       // Low range of tag index
 MaxNumTags = 1024 * 64;                 // Maximum namber of tags
 TagBugs    : Integer = 0;               // Internal errors counter

const                                    // For program logic consistency
 TAG_REF_MIN = OriginTags;               // Must be TAG_REF_MIN > TASK_REF_MAX = 255
 TAG_REF_MAX = OriginTags+MaxNumTags-1;  // Must be TAG_REF_MAX < ObjectRegistryOffset

const                                   // TypeTag values:
 tag_type_nil    = 0;                   // Tag not initialized or invalid
 tag_type_int    = 1;                   // Tag has Integer type => Integer
 tag_type_real   = 2;                   // Tag has Real    type => Double
 tag_type_string = 3;                   // Tag has String  type => AnsiString

procedure ClearTags;
function  CountTags:Integer;
function  FindTag(Name:ShortString):Integer;
function  InitTag(Name:ShortString; Typ:Integer):Integer;
function  FreeTag(tag:Integer):Boolean;
function  TypeTag(tag:Integer):Integer;
function  NameTag(tag:Integer):ShortString;
function  iGetTag(tag:Integer):Integer;
function  rGetTag(tag:Integer):Double;
function  sGetTag(tag:Integer):LongString;
function  iSetTag(tag:Integer; i:Integer):Boolean;
function  rSetTag(tag:Integer; r:Double):Boolean;
function  sSetTag(tag:Integer; const s:LongString):Boolean;
function  GetTagColor(tag:Integer):Integer;
function  SetTagColor(tag:Integer; c:Integer):Boolean;
function  GetTagParam(tag:Integer):Double;
function  SetTagParam(tag:Integer; p:Double):Boolean;
function  GetTagTimer(tag:Integer):Double;
function  SetTagTimer(tag:Integer; t:Double):Boolean;
function  GetTagList(P:TText):TText;
procedure ReadTags(const IniFile,SectionName:ShortString; Warnings:TText);
procedure ReinitTags;
procedure LockTags;
procedure UnLockTags;
function  SetTagsSpinlockCount(SpinCount:Integer):Integer;

 {
 Atomic Tag Operations.
 op is operator ( + - * / ^ ) for Real,
 op is operator ( + - * / % & | ^ < > ) for Integer,
 Example: iAtomicTagOp(tag,'+',1); - atomic tag increment
 }
function iAtomicTagOp(tag:Integer; op:Char; arg:Integer):Boolean;
function rAtomicTagOp(tag:Integer; op:Char; arg:Double):Boolean;

implementation

const
 AvailTyps = [tag_type_int,tag_type_real,tag_type_string];

var
 myLatch    : TRTLCriticalSection;
 myTags     : packed array[TAG_REF_MIN..TAG_REF_MAX] of packed record
               Typ   : Integer;
               Name  : LongString;
               iVal  : Integer;
               rVal  : Double;
               sVal  : LongString;
               Color : Integer;
               Param : Double;
               Timer : Double;
              end;
 myTagCount : Integer   = 0;
 myHashBugs : Int64     = 0;
 myHashList : THashList = nil;

procedure InitTags;
begin
 try
  LockTags;
  try
   LockedExchange(myTagCount,0);
   SafeFillChar(myTags,sizeof(myTags),0);
   myHashList:=NewHashList(true,HashList_DefaultHasher);
   myHashList.Master:=myHashList;
   myHashBugs:=0;
  finally
   UnLockTags;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure FreeTags;
begin
 try
  LockTags;
  try
   ClearTags;
   myHashBugs:=0;
   Kill(myHashList);
   LockedExchange(myTagCount,0);
   SafeFillChar(myTags,sizeof(myTags),0);
  finally
   UnLockTags;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure ReinitTags;
begin
 try
  LockTags;
  try
   FreeTags;
   InitTags;
  finally
   UnLockTags;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure LockTags;
begin
 EnterCriticalSection(myLatch);
end;

procedure UnLockTags;
begin
 LeaveCriticalSection(myLatch);
end;

function  SetTagsSpinlockCount(SpinCount:Integer):Integer;
begin
 Result:=-1;
 if (SpinCount>=0) then Result:=SetCriticalSectionSpinCount(myLatch,SpinCount);
end;

procedure ClearTags;
var tag:Integer;
begin
 try
  EnterCriticalSection(myLatch);
  try
   for tag:=Low(myTags) to High(myTags) do FreeTag(tag);
   myHashList.Clear;
  finally
   LeaveCriticalSection(myLatch);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function CountTags:Integer;
begin
 Result:=LockedAdd(myTagCount,0);
end;

function  FindTag(Name:ShortString):Integer;
var tag,i:Integer;
begin
 Result:=0;
 try
  EnterCriticalSection(myLatch);
  try
   Name:=UnifyAlias(Name);
   if not IsEmptyStr(Name) then
   if Assigned(myHashList) then begin
    i:=myHashList.IndexOf(Name);
    if i>=0 then begin
     tag:=myHashList.Links[i];
     if (Low(myTags)<=tag) and (tag<=High(myTags)) then
     if (myTags[tag].Typ in AvailTyps) and (myTags[tag].Name=Name)
     then Result:=tag else inc(myHashBugs);
    end;
   end else // obsolete version
   for tag:=Low(myTags) to High(myTags) do
   if (myTags[tag].Typ in AvailTyps) and (myTags[tag].Name=Name) then begin
    Result:=tag;
    break;
   end;
  finally
   LeaveCriticalSection(myLatch);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function  InitTag(Name:ShortString; Typ:Integer):Integer;
var tag:Integer;
begin
 Result:=0;
 try
  EnterCriticalSection(myLatch);
  try
   Name:=UnifyAlias(Name);
   if (Typ in AvailTyps) and not IsEmptyStr(Name) then begin
    tag:=FindTag(Name);
    if tag>0 then begin
     if TypeTag(tag)=Typ then Result:=tag;
    end else begin
     for tag:=Low(myTags) to High(myTags) do
     if (myTags[tag].Typ=0) and (Length(myTags[tag].Name)=0) then begin
      FreeTag(tag);
      if Assigned(myHashList) then begin
       myHashList.KeyedLinks[Name]:=tag;
      end;
      LockedInc(myTagCount);
      myTags[tag].Name:=Name;
      myTags[tag].Typ:=Typ;
      Result:=tag;
      break;
     end;
    end;
   end;
  finally
   LeaveCriticalSection(myLatch);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function  FreeTag(tag:Integer):Boolean;
begin
 Result:=false;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then
 try
  EnterCriticalSection(myLatch);
  try
   with myTags[tag] do begin
    if Assigned(myHashList) then begin
     if Name<>'' then myHashList.Delete(Name);
    end;
    if Typ>0 then LockedDec(myTagCount);
    Name:='';
    Typ:=0;
    iVal:=0;
    rVal:=0.0;
    sVal:='';
    Color:=0;
    Param:=0.0;
    Timer:=0.0;
   end;
   Result:=true;
  finally
   LeaveCriticalSection(myLatch);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function  TypeTag(tag:Integer):Integer;
begin
 Result:=0;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  Result:=myTags[tag].Typ;
  LeaveCriticalSection(myLatch);
 end;
end;

function  NameTag(tag:Integer):ShortString;
begin
 Result:='';
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ in AvailTyps then Result:=myTags[tag].Name;
  LeaveCriticalSection(myLatch);
 end;
end;

function  iGetTag(tag:Integer):Integer;
begin
 Result:=0;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ = tag_type_int then Result:=myTags[tag].iVal else LockedInc(TagBugs);
  LeaveCriticalSection(myLatch);
 end;
end;

function  rGetTag(tag:Integer):Double;
begin
 Result:=0.0;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ = tag_type_real then Result:=myTags[tag].rVal else LockedInc(TagBugs);
  LeaveCriticalSection(myLatch);
 end;
end;

function  sGetTag(tag:Integer):LongString;
begin
 Result:='';
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then
 try
  EnterCriticalSection(myLatch);
  try
   if myTags[tag].Typ = tag_type_string then Result:=myTags[tag].sVal else LockedInc(TagBugs);
  finally
   LeaveCriticalSection(myLatch);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function  iSetTag(tag:Integer; i:Integer):Boolean;
begin
 Result:=false;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ = tag_type_int then begin
   myTags[tag].iVal:=i;
   Result:=true;
  end else LockedInc(TagBugs);
  LeaveCriticalSection(myLatch);
 end;
end;

function  rSetTag(tag:Integer; r:Double):Boolean;
begin
 Result:=false;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ = tag_type_real then begin
   myTags[tag].rVal:=r;
   Result:=true;
  end else LockedInc(TagBugs);
  LeaveCriticalSection(myLatch);
 end;
end;

function  sSetTag(tag:Integer; const s:LongString):Boolean;
begin
 Result:=false;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then
 try
  EnterCriticalSection(myLatch);
  try
   if myTags[tag].Typ = tag_type_string then begin
    myTags[tag].sVal:=s;
    Result:=true;
   end else LockedInc(TagBugs);
  finally
   LeaveCriticalSection(myLatch);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function  GetTagColor(tag:Integer):Integer;
begin
 Result:=0;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ in AvailTyps then Result:=myTags[tag].Color;
  LeaveCriticalSection(myLatch);
 end;
end;

function  SetTagColor(tag:Integer; c:Integer):Boolean;
begin
 Result:=false;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ in AvailTyps then begin
   myTags[tag].Color:=c;
   Result:=true;
  end;
  LeaveCriticalSection(myLatch);
 end;
end;

function  GetTagParam(tag:Integer):Double;
begin
 Result:=0;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ in AvailTyps then Result:=myTags[tag].Param;
  LeaveCriticalSection(myLatch);
 end;
end;

function  SetTagParam(tag:Integer; p:Double):Boolean;
begin
 Result:=false;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ in AvailTyps then begin
   myTags[tag].Param:=p;
   Result:=true;
  end;
  LeaveCriticalSection(myLatch);
 end;
end;

function  GetTagTimer(tag:Integer):Double;
begin
 Result:=0;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ in AvailTyps then Result:=myTags[tag].Timer;
  LeaveCriticalSection(myLatch);
 end;
end;

function  SetTagTimer(tag:Integer; t:Double):Boolean;
begin
 Result:=false;
 if (tag>=Low(myTags)) and (tag<=High(myTags)) then begin
  EnterCriticalSection(myLatch);
  if myTags[tag].Typ in AvailTyps then begin
   myTags[tag].Timer:=t;
   Result:=true;
  end;
  LeaveCriticalSection(myLatch);
 end;
end;

function  GetTagList(P:TText):TText;
var i,m:Integer;
begin
 Result:=P;
 if P is TText then
 try
  m:=3;
  for i:=Low(myTags) to High(myTags) do m:=max(m,Length(NameTag(i)));
  P.Addln(Pad('',m)+'        ');
  for i:=Low(myTags) to High(myTags) do begin
   case TypeTag(i) of
    tag_type_int:    P.Addln(Pad(NameTag(i),m)+'  INTEGER  '+format('%d',[iGetTag(i)]));
    tag_type_real:   P.Addln(Pad(NameTag(i),m)+'  REAL     '+format('%g',[rGetTag(i)]));
    tag_type_string: P.Addln(Pad(NameTag(i),m)+'  STRING   '+sGetTag(i));
   end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure AddTag(Index:LongInt; const TextLine:ShortString; var Terminate:boolean; CustomData:Pointer);
var tag,err:Integer; SName,SType,SValu:ShortString; d:LongInt; r:double; tmp:packed array[0..255] of char;
begin
 if not IsEmptyStr(TextLine) then
 try
  err:=0;
  SName:=ExtractWord(1,TextLine,ScanSpaces);
  if SName<>'' then begin
   SType:=UpcaseStr(ExtractWord(2,TextLine,ScanSpaces));
   if SType<>'' then begin
    SValu := ExtractWord(3,TextLine,ScanSpaces);
    if SType='INTEGER' then begin
     tag:=findtag(SName);              if tag>0 then inc(err);
     tag:=inittag(SName,tag_type_int); if tag=0 then inc(err);
     if SValu<>'' then
     if not ((ScanVar(svAsIs,StrPCopy(tmp,SValu),'%d',d)<>nil) and iSetTag(tag,d)) then inc(err);
    end else
    if SType='REAL' then  begin
     tag:=findtag(SName);               if tag>0 then inc(err);
     tag:=inittag(SName,tag_type_real); if tag=0 then inc(err);
     if SValu<>'' then
     if not ((ScanVar(svAsIs,StrPCopy(tmp,SValu),'%f',r)<>nil) and rSetTag(tag,r)) then inc(err);
    end else
    if SType='STRING' then  begin
     tag:=findtag(SName);                 if tag>0 then inc(err);
     tag:=inittag(SName,tag_type_string); if tag=0 then inc(err);
     if not sSetTag(tag,SValu) then inc(err);
    end;
   end;
  end;
  if (err<>0) and TText(CustomData).Ok
  then TText(CustomData).Addln('Can~t init TAG -> "'+TextLine+'"');
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure ReadTags(const IniFile,SectionName:ShortString; Warnings:TText);
var P:TText;
begin
 P:=ExtractListSection(IniFile,SectionName,efConfig);
 if P.Ok then P.ForEach(AddTag,Warnings);
 Kill(P);
end;

function iAtomicTagOp(tag:Integer; op:Char; arg:Integer):Boolean;
begin
 Result:=false;
 if TypeTag(tag)=tag_type_int then
 try
  if (op in ['/','%']) then if (arg=0) then Exit; // Prevent EDivByZero
  case op of
   '+' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) +   arg); finally UnLockTags; end; end;
   '-' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) -   arg); finally UnLockTags; end; end;
   '*' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) *   arg); finally UnLockTags; end; end;
   '/' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) div arg); finally UnLockTags; end; end;
   '%' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) mod arg); finally UnLockTags; end; end;
   '&' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) and arg); finally UnLockTags; end; end;
   '|' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) or  arg); finally UnLockTags; end; end;
   '^' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) xor arg); finally UnLockTags; end; end;
   '<' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) shl arg); finally UnLockTags; end; end;
   '>' : begin LockTags; try Result:=iSetTag(tag,iGetTag(tag) shr arg); finally UnLockTags; end; end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function rAtomicTagOp(tag:Integer; op:Char; arg:Double):Boolean;
begin
 Result:=false;
 if TypeTag(tag)=tag_type_real then
 try
  case op of
   '+' : begin LockTags; try Result:=rSetTag(tag,rGetTag(tag) +   arg);    finally UnLockTags; end; end;
   '-' : begin LockTags; try Result:=rSetTag(tag,rGetTag(tag) -   arg);    finally UnLockTags; end; end;
   '*' : begin LockTags; try Result:=rSetTag(tag,rGetTag(tag) *   arg);    finally UnLockTags; end; end;
   '/' : begin LockTags; try Result:=rSetTag(tag,rGetTag(tag) /   arg);    finally UnLockTags; end; end;
   '^' : begin LockTags; try Result:=rSetTag(tag,Power(rGetTag(tag),arg)); finally UnLockTags; end; end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

initialization

 InitializeCriticalSection(myLatch);
 InitTags;

finalization

 FreeTags;
 DeleteCriticalSection(myLatch);

 ResourceLeakageLog(Format('%-60s = %d',['DAQ Tags HashList Bugs',myHashBugs]));

end.
