////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для преобразования кривой температуры, рассчитанной по калибровке Термопара_1 в кривую по калибровке Термопара_2.
В качестве калибровок можно брать следующие имена: HRAL, HRCP, HRCN, RHPT13, RHPT10, RHPT30, WREA1, WREA2, WREA3, NISI, CUCN, FECN.
[]
[Manual.Eng]
This plugin uses to convert temperature curve measured with calibration ThermoCouple_1 to calibration ThermoCouple_2.
Next thermocouple names can be useed: HRAL, HRCP, HRCN, RHPT13, RHPT10, RHPT30, WREA1, WREA2, WREA3, NISI, CUCN, FECN.
[]
[Arguments.Rus]
Имя окна = Результат:"THERMO_COUPLE_CONVERTER"
Заголовок = ^CЗаголовок^N^L  °C
Легенда = ^RX  ^N^CЛегенда
ТемпХолСпая = 20
Термопара_1 = HrCp
Термопара_2 = HrAl
[]
[Arguments.Eng]
Caption = Result:"THERMO_COUPLE_CONVERTER"
Title = ^CTitle^N^L  °C
Legend = ^RX  ^N^CLegend
ColdJuncTemp   = 20
ThermoCouple_1 = HrCp
ThermoCouple_2 = HrAl
[]
}

library thermo_couple_converter;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;

/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNotSortedX + cfDuplicatesX + cfNanInf;
var
 tcj : Double; p : TPoint2D;
 i, c1, c2, cFirst, cLast, tc1, tc2 : Integer;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAnalysis);
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if not GetArgumentAsDouble(RusEng('ТемпХолСпая','ColdJuncTemp'),tcj)
  then Raise EDanApi.Create(RusEng('Ошибка задания аргументов!',
                                   'Invalid input arguments!'));
  tc1:=DaqApi.FindThermocouple(GetArgumentAsString(RusEng('Термопара_1','ThermoCouple_1')));
  tc2:=DaqApi.FindThermocouple(GetArgumentAsString(RusEng('Термопара_2','ThermoCouple_2')));
  if tc1<0 then Raise EDanApi.Create(RusEng('Не найдена Термопара_1!','Not found ThermoCouple_1!'));
  if tc2<0 then Raise EDanApi.Create(RusEng('Не найдена Термопара_2!','Not found ThermoCouple_2!'));
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  for c1:=cFirst to cLast do begin
   if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для обработки!',
                                          'Curve %d is not convenient for processing!'),[c1]));
   c2:=CreateCurve(twin,CurveName[c1],CurveColor[c1],CurveStyle[c1],CurveComment[c1]);
   CurveLength[c2]:=CurveLength[c1];
   for i:=0 to CurveLength[c1]-1 do with DaqApi do begin
    p:=CurvePoint[c1,i];
    p.y:=ThermoCoupleValue(tc2,ThermoCoupleValue(tc1,p.y,false)
                              -ThermoCoupleValue(tc1,tcj,false)
                              +ThermoCoupleValue(tc2,tcj,false),true);
    CurvePoint[c2,i]:=p;
    Refreshment(100);
   end;
  end;
  if SelectedCurve[swin]>0 then SelectedCurve[twin]:=1 else SelectedCurve[twin]:=0;
  with DaqApi do
  WindowCaption[twin]:=WindowCaption[twin]+' ('+ThermoCoupleName(tc1)+'->'
                                               +ThermoCoupleName(tc2)+')';
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0;
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message); Error(E.Message);
   Result:=-1;
  end;
 end;
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
