////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2020-2023 Alexey Kuryakin kouriakine@mail.ru - LGPL license. //
////////////////////////////////////////////////////////////////////////////////

unit dpTask; // Diesel Pascal routines useful for tasks, i.e. process control.

interface

uses dpCmdArgs,dpSystem,dpSysUtils,dpWindows;

type
 TTask = class(TObject)
 private
  myTid        : Integer;
  myProcess    : TProcess;
  myPriority   : Integer;
  myEncoding   : TEncoding;
  myUsesWinApi : Boolean;
  function    GetTid:Integer;
  procedure   SetTid(aTid:Integer);
  function    GetProcess:TProcess;
  function    GetAppName:String;
  procedure   SetAppName(aAppName:String);
  function    GetCmdLine:String;
  procedure   SetCmdLine(aCmdLine:String);
  function    GetHomeDir:String;
  procedure   SetHomeDir(aHomeDir:String);
  function    GetEnvironment:String;
  procedure   SetEnvironment(aEnvironment:String);
  function    GetDisplay:Integer;
  procedure   SetDisplay(aDisplay:Integer);
  function    GetStdInpPipeSize:Integer;
  procedure   SetStdInpPipeSize(aPipeSize:Integer);
  function    GetStdInpPipeFifoCount:Integer;
  function    GetStdInpPipeFifoSpace:Integer;
  function    GetStdOutPipeSize:Integer;
  procedure   SetStdOutPipeSize(aPipeSize:Integer);
  function    GetStdOutPipeFifoCount:Integer;
  function    GetStdOutPipeFifoSpace:Integer;
  function    GetProcessPriority:Integer;
  procedure   SetProcessPriority(aPriority:Integer);
  function    GetCodePage:Cardinal;
  procedure   SetCodePage(aCodePage:Cardinal);
  function    GetExitStatus:Integer;
  function    GetExitCode:Integer;
  function    GetPid:Integer;
  function    GetWnd:hWnd;
  function    GetExeName:String;
  function    GetUsesWinApi:Boolean;
  function    SetUsesWinApi(aUsesWinApi:Boolean);
 public
  property Tid                 : Integer        read GetTid             write SetTid;
  property Process             : TProcess       read GetProcess;
  property AppName             : String         read GetAppName         write SetAppName;
  property CmdLine             : String         read GetCmdLine         write SetCmdLine;
  property HomeDir             : String         read GetHomeDir         write SetHomeDir;
  property ExeName             : String         read GetExeName;
  property Environment         : String         read GetEnvironment     write SetEnvironment;
  property Display             : Integer        read GetDisplay         write SetDisplay;
  property StdInpPipeSize      : Integer        read GetStdInpPipeSize  write SetStdInpPipeSize;
  property StdInpPipeFifoCount : Integer        read GetStdInpPipeFifoCount;
  property StdInpPipeFifoSpace : Integer        read GetStdInpPipeFifoSpace;
  property StdOutPipeSize      : Integer        read GetStdOutPipeSize  write SetStdOutPipeSize;
  property StdOutPipeFifoCount : Integer        read GetStdOutPipeFifoCount;
  property StdOutPipeFifoSpace : Integer        read GetStdOutPipeFifoSpace;
  property ProcessPriority     : Integer        read GetProcessPriority write SetProcessPriority;
  property CodePage            : Cardinal       read GetCodePage        write SetCodePage;
  property ExitStatus          : Integer        read GetExitStatus;
  property ExitCode            : Integer        read GetExitCode;
  property Pid                 : Integer        read GetPid;
  property Wnd                 : hWnd           read GetWnd;
  property UsesWinApi          : Boolean        read GetUsesWinApi      write SetUsesWinApi;
 public
  constructor Create(aTid:Integer);
  destructor  Destroy;
 public
  function    Run:Boolean;
  function    Running(aTimeOut:Integer=0):Boolean;
  function    Send(const data:String):Integer;
  function    Recv(maxlen:Integer):String;
  function    Terminate(aHow:Integer=0; aExitCode:Integer=0; aTimeOut:Integer=0):Boolean;
 end;

var TaskUsesWinApiByDefault : Boolean = true;

 ///////////////////////////////////////////////////////////////////////////////
 // Easy task routines, to be use in DAQ PASCAL
 // tid          - Task index, integer value > 0 to identify task.
 //                Zero value of tid uses to indicate error or null task.
 // task_init    - Create empty task, initialize CmdLine, return task index.
 //                After task_init, task_ref<>nil,task_pid=0,task_wait=false.
 //                Task not attached to process, you need task_run call to do it.
 //                Return 0 if free task index not found, if task table overflow.
 // task_free    - Destroy task instance and free task index for future use.
 // task_ref     - Return task instance by task index or nil if not exists.
 //                Existing reference means that task_init call was done.
 // task_pid     - Process identifier (pid) or 0 if process not attached.
 //                task_pid will return pid<>0 after task_run call.
 // task_run     - Create and run process, return true if success.
 //                After task_run call, task_pid<>0, task_wait=true.
 //                Task now will be attached to process, even if one terminated.
 //                If process already attached, return false.
 //                So you don't call task_run two times.
 // task_wait    - Wait timeout while process running.
 //                Return true if process still running.
 //                Use task_wait(tid,0) to check immediately if process running.
 // task_send    - Send data string to process standard stdin stream.
 //                Works if only stdin was redirected to pipe.
 // task_recv    - Receive data string from process standard stdout stream.
 //                Works if only stdout was redirected to pipe.
 // task_txcount - Data count in stdin  pipe.
 // task_rxcount - Data count in stdout pipe.
 // task_txspace - Data space in stdin  pipe.
 // task_rxspace - Data space in stdout pipe.
 // task_result  - Return exit code of terminated process.
 //                If process still running, return 259=STILL_ACTIVE.
 //                If closed by ^C, return -1073741510=STATUS_CONTROL_C_EXIT.
 // task_kill    - Terminate process. Argument how means how to kill.
 //                0 - TerminateProcess
 //                1 - PostMessage(WM_CLOSE)
 //                2 - PostMesssage(WM_QUIT)
 //                3 - KillProcessTree
 // task_ctrl    - Task control, set parameters.
 //                task_ctrl('name=value') - assign parameter value
 //                task_ctrl('name')       - return parameter value
 //                AppName, CmdLine, HomeDir, Account, Display,
 //                StdInPipeSize=StdInpPipeSize=TxPipeSize,StdOutPipeSize=RxPipeSize
 //                StdInFileName=StdInpFileName,StdOutFileName
 ///////////////////////////////////////////////////////////////////////////////
function task_init(const cmd_line:String):Integer;
function task_free(tid:Integer):Boolean;
function task_ref(tid:Integer):TTask;
function task_pid(tid:Integer):Integer;
function task_run(tid:Integer):Boolean;
function task_wait(tid,timeout:Integer):Boolean;
function task_send(tid:Integer; const data:String):Integer;
function task_recv(tid,maxlen:Integer):String;
function task_txcount(tid:Integer):Integer;
function task_rxcount(tid:Integer):Integer;
function task_txspace(tid:Integer):Integer;
function task_rxspace(tid:Integer):Integer;
function task_result(tid:Integer):Integer;
function task_kill(tid,how,exit_code,timeout:Integer):Boolean;
function task_ctrl(tid:Integer; const param:String):String;

procedure task_initialize;
procedure task_finalize;

const task_ref_min=1;  task_ref_max=255;

const NONE_PRIORITY_LEVEL         = 0;
const IDLE_PRIORITY_LEVEL         = 4;
const LOWER_PRIORITY_LEVEL        = 6;
const NORMAL_PRIORITY_LEVEL       = 8;
const HIGHER_PRIORITY_LEVEL       = 10;
const HIGH_PRIORITY_LEVEL         = 13;
const REALTIME_PRIORITY_LEVEL     = 24;
const MIN_NORMAL_PRIORITY_LEVEL   = 1;
const MAX_NORMAL_PRIORITY_LEVEL   = 15;
const MIN_REALTIME_PRIORITY_LEVEL = 16;
const MAX_REALTIME_PRIORITY_LEVEL = 31;

function GetPriorityName(aPriority:TThreadPriority):String;
function GetPriorityByName(const aName:String):TThreadPriority;
function AdjustPriorityClassLevel(aLevel:Integer):Integer;
function GetPriorityClassLevel(aPriorityClass:DWORD):Integer;
function GetPriorityClassName(aPriorityClass:DWORD):String;
function GetPriorityClassByLevel(aLevel:Integer):DWORD;
function GetPriorityClassByName(const aName:String):DWORD;
function GetAbsolutePriority(PriorityClass:DWORD; ThreadPriority:TThreadPriority):Integer;

function KillProcess(aPid:DWORD; aExitCode:Integer):Boolean;
function KillProcessTree(aPid:DWORD; aExitCode:Integer; MaxLevel:Integer=1000):Integer;

function ExecuteProcessSafe(cmd,arg,opt:String; Timeout:Integer):Integer;

function WTERMSIG(Status:Integer):Integer;
function WSTOPSIG(Status:Integer):Integer;
function WIFEXITED(Status:Integer):Boolean;
function WIFSTOPPED(Status:Integer):Boolean;
function WEXITSTATUS(Status:Integer):Integer;
function WIFSIGNALED(Status:Integer): Boolean;
function ProcessExitStatusToCode(Status:Integer):Integer;
function ProcessExitStatusToString(Status:Integer):String;

var ExecuteProcessSafeVerbose:Boolean=true;

implementation

function WTERMSIG(Status:Integer):Integer;
begin
 Result:=(Status and $7F);
end;

function WSTOPSIG(Status:Integer):Integer;
begin
 Result:=WEXITSTATUS(Status);
end;

function WIFEXITED(Status:Integer):Boolean;
begin
  Result:=(WTERMSIG(Status)=0);
end;

function WEXITSTATUS(Status:Integer):Integer;
begin
 Result:=(Status and $FF00) shr 8;
end;

function WIFSTOPPED(Status:Integer):Boolean;
begin
 Result:=((Status and $FF)=$7F);
end;

function WIFSIGNALED(Status: longint): Boolean;
begin
  Result:=(not WIFSTOPPED(Status)) and (not WIFEXITED(Status));
end;

function ProcessExitStatusToCode(Status:Integer):Integer;
begin
 Result:=Status;
 if IsUnix then begin
  if wIfExited(Status)   then Result:=WExitStatus(Status) else
  if wIfSignaled(Status) then Result:=WTermSig(Status)    else
  if WIfStopped(Status)  then Result:=WStopSig(Status);
 end;
end;

function ProcessExitStatusToString(Status:Integer):String;
begin
 if IsUnix then begin
  if WIfExited(Status)   then Result:='ExitCode: '+IntToStr(WExitStatus(Status)) else
  if WIfSIgnaled(Status) then Result:='TermSign: '+IntToStr(WTermSig(Status)) else
  if WIfStopped(Status)  then Result:='StopSign: '+IntToStr(WStopSig(Status)) else
                              Result:='StatCode: '+IntToStr(Status);
 end else                     Result:='ExitCode: '+IntToStr(Status);
end;

function GetPriorityName(aPriority:TThreadPriority):String;
begin
 case aPriority of
  tpIdle         : Result:='tpIdle';
  tpLowest       : Result:='tpLowest';
  tpLower        : Result:='tpLower';
  tpNormal       : Result:='tpNormal';
  tpHigher       : Result:='tpHigher';
  tpHighest      : Result:='tpHighest';
  tpTimeCritical : Result:='tpTimeCritical';
  else             Result:='tpNormal';
 end;
end;

function GetPriorityByName(const aName:String):TThreadPriority;
begin
 if IsSameText(aName,GetPriorityName(tpIdle))         then Result:=tpIdle         else
 if IsSameText(aName,GetPriorityName(tpLowest))       then Result:=tpLowest       else
 if IsSameText(aName,GetPriorityName(tpLower))        then Result:=tpLower        else
 if IsSameText(aName,GetPriorityName(tpNormal))       then Result:=tpNormal       else
 if IsSameText(aName,GetPriorityName(tpHigher))       then Result:=tpHigher       else
 if IsSameText(aName,GetPriorityName(tpHighest))      then Result:=tpHighest      else
 if IsSameText(aName,GetPriorityName(tpTimeCritical)) then Result:=tpTimeCritical else
 Result:=tpNormal;
end;

function AdjustPriorityClassLevel(aLevel:Integer):Integer;
begin
 Result:=NONE_PRIORITY_LEVEL;
 if (aLevel>=MIN_NORMAL_PRIORITY_LEVEL) then
 if (aLevel<=MAX_REALTIME_PRIORITY_LEVEL) then
 if (aLevel>=MIN_REALTIME_PRIORITY_LEVEL) then Result:=REALTIME_PRIORITY_LEVEL else
 if (aLevel>=HIGH_PRIORITY_LEVEL)         then Result:=HIGH_PRIORITY_LEVEL     else
 if (aLevel>=HIGHER_PRIORITY_LEVEL)       then Result:=HIGHER_PRIORITY_LEVEL   else
 if (aLevel>=NORMAL_PRIORITY_LEVEL)       then Result:=NORMAL_PRIORITY_LEVEL   else
 if (aLevel>=LOWER_PRIORITY_LEVEL)        then Result:=LOWER_PRIORITY_LEVEL    else
 if (aLevel>=IDLE_PRIORITY_LEVEL)         then Result:=IDLE_PRIORITY_LEVEL     else
 Result:=IDLE_PRIORITY_LEVEL;
end;

function GetPriorityClassLevel(aPriorityClass:DWORD):Integer;
begin
 Result:=0;
 case aPriorityClass of
  IDLE_PRIORITY_CLASS:     Result:=IDLE_PRIORITY_LEVEL;
  LOWER_PRIORITY_CLASS:    Result:=LOWER_PRIORITY_LEVEL;
  NORMAL_PRIORITY_CLASS:   Result:=NORMAL_PRIORITY_LEVEL;
  HIGHER_PRIORITY_CLASS:   Result:=HIGHER_PRIORITY_LEVEL;
  HIGH_PRIORITY_CLASS:     Result:=HIGH_PRIORITY_LEVEL;
  REALTIME_PRIORITY_CLASS: Result:=REALTIME_PRIORITY_LEVEL;
 end;
end;

function GetPriorityClassName(aPriorityClass:DWORD):String;
begin
 Result:='';
 case aPriorityClass of
  IDLE_PRIORITY_CLASS:     Result:='Idle';
  LOWER_PRIORITY_CLASS:    Result:='Lower';
  NORMAL_PRIORITY_CLASS:   Result:='Normal';
  HIGHER_PRIORITY_CLASS:   Result:='Higher';
  HIGH_PRIORITY_CLASS:     Result:='High';
  REALTIME_PRIORITY_CLASS: Result:='RealTime';
 end;
end;

function GetPriorityClassByLevel(aLevel:Integer):DWORD;
begin
 Result:=0;
 case aLevel of
  IDLE_PRIORITY_LEVEL:     Result:=IDLE_PRIORITY_CLASS;
  LOWER_PRIORITY_LEVEL:    Result:=LOWER_PRIORITY_CLASS;
  NORMAL_PRIORITY_LEVEL:   Result:=NORMAL_PRIORITY_CLASS;
  HIGHER_PRIORITY_LEVEL:   Result:=HIGHER_PRIORITY_CLASS;
  HIGH_PRIORITY_LEVEL:     Result:=HIGH_PRIORITY_CLASS;
  REALTIME_PRIORITY_LEVEL: Result:=REALTIME_PRIORITY_CLASS;
 end;
end;

function GetPriorityClassByName(const aName:String):DWORD;
var i:Integer;
begin
 Result:=0;
 if TryStrToInt(aName,i) then Result:=GetPriorityClassByLevel(i)  else
 if IsSameText(aName,'Idle')     then Result:=IDLE_PRIORITY_CLASS   else
 if IsSameText(aName,'Lower')    then Result:=LOWER_PRIORITY_CLASS  else
 if IsSameText(aName,'Normal')   then Result:=NORMAL_PRIORITY_CLASS else
 if IsSameText(aName,'Higher')   then Result:=HIGHER_PRIORITY_CLASS else
 if IsSameText(aName,'High')     then Result:=HIGH_PRIORITY_CLASS   else
 if IsSameText(aName,'RealTime') then Result:=REALTIME_PRIORITY_CLASS;
end;

function GetAbsolutePriority(PriorityClass:DWORD; ThreadPriority:TThreadPriority):Integer;
var pmin,pmax,pri:Integer;
begin
 Result:=0;
 pri:=GetPriorityClassLevel(PriorityClass);
 if pri>0 then begin
  if pri<16 then begin
   pmin:=1; pmax:=15;
  end else begin
   pmin:=16; pmax:=31;
  end;
  case ThreadPriority of
   tpIdle         : Result:=pmin;
   tpLowest       : Result:=pri-2;
   tpLower        : Result:=pri-1;
   tpNormal       : Result:=pri;
   tpHigher       : Result:=pri+1;
   tpHighest      : Result:=pri+2;
   tpTimeCritical : Result:=pmax;
  end;
  if Result<pmin then Result:=pmin;
  if Result>pmax then Result:=pmax;
 end;
end;

function KillProcess(aPid:DWORD; aExitCode:Integer):Boolean;
var hProcess:THandle;
begin
 Result:=false;
 if (aPid<>0) then
 if (aPid<>GetProcessId) then
 try
  hProcess:=WinApi.OpenProcess(PROCESS_TERMINATE,False,aPid);
  if hProcess<>0 then
  try
   Result:=WinApi.TerminateProcess(hProcess,aExitCode);
  finally
   WinApi.CloseHandle(hProcess);
  end;
 except
  on E:Exception do BugReport(E,nil,'KillProcess');
 end;
end;

function KillProcessTree(aPid:DWORD; aExitCode:Integer; MaxLevel:Integer=1000):Integer;
var List:TStringList;
 procedure KillPid(aPid:DWORD;aLevel:Integer); // To be called recursively
 var i:Integer; pid,ppid:DWORD;
 begin
  if (aPid<>0) then
  if (aPid<>GetProcessId) then begin
   if aLevel<MaxLevel then
   for i:=0 to List.Count-1 do begin
    pid:=StrToIntDef(ExtractWord(1,List.Strings[i],', '),0);
    ppid:=StrToIntDef(ExtractWord(2,List.Strings[i],', '),0);
    if (ppid=aPid) then KillPid(pid,aLevel+1);
   end;
   if KillProcess(aPid,aExitCode) then Inc(Result);
  end;
 end;
begin
 Result:=0;
 if (aPid<>0) then
 if (aPid<>GetProcessId) then
 try
  List:=TStringList.Create;
  try
   List.Text:=GetListOfProcesses(0,0,'');
   KillPid(aPid,0);
  finally
   List.Free;
  end;
 except
  on E:Exception do BugReport(E,nil,'KillProcessTree');
 end;
end;

function ExecuteProcessSafe(cmd,arg,opt:String; Timeout:Integer):Integer;
var tid,i:Integer; List:TStringList; ms:Double;
begin
 Result:=-1;
 try
  ms:=mSecNow;
  cmd:=Trim(cmd); arg:=Trim(arg);
  if (cmd+arg='') or (Timeout<=0) then Exit;
  tid:=task_init(Trim(cmd+' '+arg));
  if (tid<>0) then
  try
   if (opt<>'') then begin
    List:=TStringList.Create;
    try
     List.Text:=Trim(opt);
     for i:=0 to List.Count-1 do
     if (Pos('=',List.Strings[i])>0)
     then task_ctrl(tid,Trim(List.Strings[i]));
    finally
     List.Free;
    end;
   end;
   if task_run(tid) then begin
    if task_wait(tid,timeout) then
    if task_wait(tid,0) then task_kill(tid,1,1,500);
    if task_wait(tid,0) then task_kill(tid,0,1,500);
    Result:=task_result(tid);
   end;
   ms:=mSecNow-ms;
   if ExecuteProcessSafeVerbose then
   LogEvents(Now,'Executed: '+task_ctrl(tid,'CmdLine')+EOL+
                 ProcessExitStatusToString(Result)+', '+
                 'Time: '+IntToStr(Round(ms))+' ms');
  finally
   task_free(tid);
  end;
 except
  on E:Exception do BugReport(E,nil,'ExecuteProcessSafe');
 end;
end;

 ///////////////////////
 // TTask implementation
 ///////////////////////

var task_array : array[1..255] of TTask;

constructor TTask.Create(aTid:Integer=0);
var aCodePage:Cardinal;
begin
 inherited Create;
 myProcess:=TProcess.Create(Application);
 myPriority:=NORMAL_PRIORITY_LEVEL;
 aCodePage:=TEncoding(nil).Default.CodePage;
 myEncoding:=TEncoding(nil).GetEncoding(aCodePage);
 UsesWinApi:=TaskUsesWinApiByDefault;
 Tid:=aTid;
end;

destructor TTask.Destroy;
begin
 if (Tid>=task_ref_min) and (Tid<=task_ref_max) then
 if (task_array[Tid]=Self) then task_array[Tid]:=nil;
 myProcess.Free; myProcess:=nil;
 myEncoding.Free; myEncoding:=nil;
 inherited Destroy;
end;


function TTask.GetTid:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 Result:=myTid;
end;

procedure TTask.SetTid(aTid:Integer);
begin
 if (Self<>nil) and (aTid<>0) then
 if (aTid>=task_ref_min) and (aTid<=task_ref_max) then begin
  if (task_array[aTid]=nil) then task_array[aTid]:=Self;
  myTid:=aTid;
 end;
end;

function TTask.GetProcess:TProcess;
begin
 Result:=nil;
 if (Self=nil) then Exit;
 Result:=myProcess;
end;

function  TTask.GetAppName:String;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=Process.ApplicationName;
end;

procedure TTask.SetAppName(aAppName:String);
begin
 if (Self=nil) then Exit;
 Process.ApplicationName:=Trim(aAppName);
end;

function  TTask.GetCmdLine:String;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=Process.CommandLine;
end;

procedure TTask.SetCmdLine(aCmdLine:String);
begin
 if (Self=nil) then Exit;
 Process.CommandLine:=Trim(aCmdLine);
end;

function  TTask.GetHomeDir:String;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=Process.CurrentDirectory;
end;

procedure TTask.SetHomeDir(aHomeDir:String);
begin
 if (Self=nil) then Exit;
 Process.CurrentDirectory:=Trim(aHomeDir);
end;

function  TTask.GetEnvironment:String;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=Process.Environment.Text;
end;

procedure TTask.SetEnvironment(aEnvironment:String);
begin
 if (Self=nil) then Exit;
 Process.Environment.Text:=aEnvironment;
end;

function TTask.GetExitStatus:Integer;
var Code:DWORD;
begin
 Result:=-1;
 if (Self=nil) then Exit;
 if UsesWinApi and IsWindows then begin
  if (Process.Handle<>0) then
  if WinApi.GetExitCodeProcess(Process.Handle,Code) then
  Result:=(Code and -1); // Convert Result to Integer
  Exit;
 end;
 Result:=Process.ExitStatus;
end;

function TTask.GetExitCode:Integer;
var Code:Integer;
begin
 Result:=-1;
 if (Self=nil) then Exit;
 Code:=Process.ExitStatus;
 Result:=ProcessExitStatusToCode(Code);
end;

function TTask.GetPid:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 Result:=Process.ProcessId;
end;

function TTask.GetWnd:hWnd;
var List:TStringList; w:hWnd; i:Integer;
 function CheckWindow(Win: HWND): Boolean;
 var ParentWnd:hWnd; ExStyle:DWORD;
 begin
  Result:=false;
  if (Win<>0) then
  if WinApi.IsWindow(Win) then begin
   ExStyle:=WinApi.GetWindowLong(Win,GWL_EXSTYLE);
   ParentWnd:=WinApi.GetWindowLong(Win,GWL_HWNDPARENT);
   if (ParentWnd=0) or (ParentWnd=WinApi.GetDesktopWindow) then
   if ((ExStyle and WS_EX_TOOLWINDOW)=0) or ((ExStyle and WS_EX_APPWINDOW)<>0) then
   if (WinApi.GetWindowText(Win)<>'') then Result:=true;
  end;
 end;
begin
 Result:=0;
 if (PID<>0) then
 try
  List:=TStringList.Create;
  try
   List.Text:=WinApi.GetListOfWindows(PID,'','');
   for i:=0 to List.Count-1 do begin
    w:=StrToIntDef(ExtractWord(1,List.Strings[i],' ,'),0);
    if (w=0) then continue;
    if CheckWindow(w) then Result:=w;
    if (Result<>0) then Break;
   end;
  finally
   List.Free;
  end;
 except
  on E:Exception do BugReport(E,Self,'GetWnd');
 end;
end;

function TTask.GetUsesWinApi:Boolean;
begin
 Result:=false;
 if (Self=nil) then Exit;
 Result:=myUsesWinApi;
end;

function TTask.SetUsesWinApi(aUsesWinApi:Boolean);
begin
 if (Self=nil) then Exit;
 myUsesWinApi:=aUsesWinApi and IsWindows;
end;

function TTask.GetExeName:String;
begin
 Result:='';
 if (Self=nil) then Exit;
 Result:=Process.Executable;
end;

function  TTask.GetDisplay:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 case Process.ShowWindow of
  swoNone            : Result:=-1;
  swoHIDE            : Result:=SW_HIDE;
  swoMaximize        : Result:=SW_MAXIMIZE;
  swoMinimize        : Result:=SW_MINIMIZE;
  swoRestore         : Result:=SW_RESTORE;
  swoShow            : Result:=SW_SHOW;
  swoShowDefault     : Result:=SW_SHOWDEFAULT;
  swoShowMaximized   : Result:=SW_SHOWMAXIMIZED;
  swoShowMinimized   : Result:=SW_SHOWMINIMIZED;
  swoshowMinNOActive : Result:=SW_SHOWMINNOACTIVE;
  swoShowNA          : Result:=SW_SHOWNA;
  swoShowNoActivate  : Result:=SW_SHOWNOACTIVATE;
  swoShowNormal      : Result:=SW_SHOWNORMAL;
  else                 Result:=-1;
 end;
end;

procedure TTask.SetDisplay(aDisplay:Integer);
begin
 if (Self=nil) then Exit;
 case aDisplay of
  SW_HIDE            : Process.ShowWindow:=swoHIDE;
  SW_MAXIMIZE        : Process.ShowWindow:=swoMaximize;
  SW_MINIMIZE        : Process.ShowWindow:=swoMinimize;
  SW_RESTORE         : Process.ShowWindow:=swoRestore;
  SW_SHOW            : Process.ShowWindow:=swoShow;
  SW_SHOWDEFAULT     : Process.ShowWindow:=swoShowDefault;
  SW_SHOWMAXIMIZED   : Process.ShowWindow:=swoShowMaximized;
  SW_SHOWMINIMIZED   : Process.ShowWindow:=swoShowMinimized;
  SW_SHOWMINNOACTIVE : Process.ShowWindow:=swoshowMinNOActive;
  SW_SHOWNA          : Process.ShowWindow:=swoShowNA;
  SW_SHOWNOACTIVATE  : Process.ShowWindow:=swoShowNoActivate;
  SW_SHOWNORMAL      : Process.ShowWindow:=swoShowNormal;
  else                 Process.ShowWindow:=swoNone;
 end;
 if (Process.ShowWindow<>swoNone)
 then Process.StartupOptions:=Process.StartupOptions+[suoUseShowWindow];
end;

function  TTask.GetStdInpPipeSize:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 if (Process.Input=nil) then Exit;
 Result:=Process.PipeBufferSize;
end;

function  TTask.GetStdInpPipeFifoCount:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 if (Process.Input=nil) then Exit;
 Result:=0;
end;

function  TTask.GetStdInpPipeFifoSpace:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 if (Process.Input=nil) then Exit;
 Result:=Process.PipeBufferSize;
end;

procedure TTask.SetStdInpPipeSize(aPipeSize:Integer);
begin
 if (Self=nil) then Exit;
 if (aPipeSize<=0) then Exit;
 try
  Process.Options:=Process.Options+[poUsePipes];
  Process.PipeBufferSize:=AdjustBufferSize(aPipeSize,1024);
 except
  on E:Exception do BugReport(E,Self,'SetStdInpPipeSize');
 end;
end;

function  TTask.GetStdOutPipeSize:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 if (Process.Output=nil) then Exit;
 Result:=Process.PipeBufferSize;
end;

procedure TTask.SetStdOutPipeSize(aPipeSize:Integer);
begin
 if (Self=nil) then Exit;
 if (aPipeSize<=0) then Exit;
 try
  Process.Options:=Process.Options+[poUsePipes];
  Process.PipeBufferSize:=AdjustBufferSize(aPipeSize,1024);
 except
  on E:Exception do BugReport(E,Self,'SetStdOutPipeSize');
 end;
end;

function  TTask.GetStdOutPipeFifoCount:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 if (Process.Output=nil) then Exit;
 Result:=Process.Output.NumBytesAvailable;
end;

function  TTask.GetStdOutPipeFifoSpace:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 if (Process.Output=nil) then Exit;
 Result:=iMax(0,Process.PipeBufferSize-Process.Output.NumBytesAvailable);
end;

function  TTask.GetProcessPriority:Integer;
begin
 Result:=0;
 if (Self=nil) then Exit;
 Result:=myPriority;
end;

procedure TTask.SetProcessPriority(aPriority:Integer);
begin
 if (Self=nil) then Exit;
 myPriority:=AdjustPriorityClassLevel(aPriority);
 case ProcessPriority of
  IDLE_PRIORITY_LEVEL:     Process.Priority:=ppIdle;
  LOWER_PRIORITY_LEVEL:    Process.Priority:=ppNormal;
  NORMAL_PRIORITY_LEVEL:   Process.Priority:=ppNormal;
  HIGHER_PRIORITY_LEVEL:   Process.Priority:=ppNormal;
  HIGH_PRIORITY_LEVEL:     Process.Priority:=ppHigh;
  REALTIME_PRIORITY_LEVEL: Process.Priority:=ppRealTime;
 end;
 if Running then WinApi.SetPriorityClass(Process.Handle,GetPriorityClassByLevel(ProcessPriority));
end;

function TTask.GetCodePage:Cardinal;
begin
 Result:=0;
 if (Self=nil) then Exit;
 Result:=myEncoding.CodePage;
end;

procedure TTask.SetCodePage(aCodePage:Cardinal);
begin
 if (Self=nil) then Exit;
 if (myEncoding.CodePage=aCodePage) then Exit;
 myEncoding.Free;
 myEncoding:=TEncoding(nil).GetEncoding(aCodePage);
end;

function TTask.Run:Boolean;
begin
 Result:=false;
 if (Self=nil) then Exit;
 if not Running then Process.Execute;
 if IsWindows and Running then WinApi.SetPriorityClass(Process.Handle,GetPriorityClassByLevel(ProcessPriority));
 Result:=Running;
end;

function TTask.Running(aTimeOut:Integer=0):Boolean;
var tick:Cardinal;
begin
 Result:=false;
 if (Self=nil) then Exit;
 if UsesWinApi then begin
  if (Process.Handle<>0) then
  Result:=(WinApi.WaitForSingleObject(Process.Handle,aTimeOut)=WAIT_TIMEOUT);
  Exit;
 end;
 if (aTimeOut<>0) then begin
  tick:=GetTickCount;
  while (GetTickCount-tick<=aTimeOut) and Process.Running do Sleep(1);
 end;
 Result:=Process.Running;
end;

function TTask.Send(const data:String):Integer;
var Str:TStringStream;
begin
 Result:=0;
 if (Self=nil) then Exit;
 if (Length(data)=0) then Exit;
 if (Process.Input=nil) then Exit;
 try
  Str:=TStringStream.CreateEncoding(data,myEncoding,false);
  try
   Str.Seek(0,soFromBeginning);
   Result:=Process.Input.CopyFrom(Str,Length(data));
  finally
   Str.Free;
  end;
 except
  on E:Exception do BugReport(E,Self,'Send');
 end;
end;

function TTask.Recv(maxlen:Integer):String;
var n,m:Integer; Str:TStringStream;
begin
 Result:='';
 if (Self=nil) then Exit;
 if (Process.Output=nil) then Exit;
 try
  n:=Process.Output.NumBytesAvailable;
  if (n>maxlen) then n:=maxlen;
  if (n=0) then Exit;
  Str:=TStringStream.CreateEncoding(StringOfChar(#0,n),myEncoding,false);
  try
   Str.Seek(0,soFromBeginning); m:=Str.CopyFrom(Process.Output,n);
   Str.Seek(0,soFromBeginning); Result:=Str.ReadString(m);
  finally
   Str.Free;
  end;
 except
  on E:Exception do BugReport(E,Self,'Recv');
 end;
end;

function TTask.Terminate(aHow:Integer=0; aExitCode:Integer=0; aTimeOut:Integer=0):Boolean;
var hWin:hWnd;
begin
 Result:=false;
 if (Self=nil) then Exit;
 try
  if UsesWinApi then begin
   if (Process.Handle<>0) then begin
    if (WinApi.WaitForSingleObject(Process.Handle,0)=WAIT_TIMEOUT) then begin
     case aHow of
      0:   Result:=WinApi.TerminateProcess(Process.Handle,aExitCode);
      1:   begin
            hWin:=Wnd;
            if (hWin<>0) and WinApi.IsWindow(hWin) then begin
             if WinApi.IsWindowVisible(hWin) then WinApi.SetForegroundWindow(hWin);
             Result:=WinApi.PostMessage(hWin, WM_CLOSE, aExitCode, 0);
            end;
           end;
      2:   begin
            hWin:=Wnd;
            if (hWin<>0) and WinApi.IsWindow(hWin) then begin
             if WinApi.IsWindowVisible(hWin) then WinApi.SetForegroundWindow(hWin);
             Result:=WinApi.PostMessage(hWin, WM_QUIT, aExitCode, 0);
            end;
           end;
      3:   Result:=KillProcessTree(Pid,aExitCode,aTimeOut)>0;
      else Result:=WinApi.TerminateProcess(Process.Handle,aExitCode);
     end;
     if Result then WinApi.WaitForSingleObject(Process.Handle,aTimeOut);
    end;
   end;
   Exit;
  end;
  if Running then Process.Terminate(aExitCode);
  Result:=not Running;
 except
  on E:Exception do BugReport(E,Self,'Terminate');
 end;
end;

 //////////////////////////////////////////////
 // Easy task routines, to be use in DAQ PASCAL
 //////////////////////////////////////////////

procedure task_initialize;
var i:Integer;
begin
 for i:=task_ref_min to task_ref_max do task_array[i]:=nil;
end;

procedure task_finalize;
var i:Integer;
begin
 for i:=task_ref_min to task_ref_max do task_free(i);
end;

function task_init(const cmd_line:String):Integer;
var i:Integer; task:TTask;
begin
 Result:=0;
 try
  for i:=task_ref_min to task_ref_max do
  if (task_array[i]=nil) then begin
   task:=TTask.Create(i);
   task.CmdLine:=cmd_line;
   task_array[i]:=task;
   task.Tid:=i;
   Result:=i;
   Break;
  end;
  if not Windows then task_ctrl(Result,'Environment=Inherited');
 except
  on E:Exception do BugReport(E,nil,'task_init');
 end;
end;

function task_free(tid:Integer):Boolean;
var task:TTask;
begin
 task:=nil;
 if (tid >= task_ref_min) then
 if (tid <= task_ref_max) then begin
  task:=task_array[tid];
  task_array[tid]:=nil;
 end;
 Result:=(task<>nil);
 if (task<>nil) then task.Free;
end;

function task_ref(tid:Integer):TTask;
begin
 Result:=nil;
 if tid >= task_ref_min then
 if tid <= task_ref_max then Result:=task_array[tid];
end;

function task_pid(tid:Integer):Integer;
begin
 Result:=task_ref(tid).PID;
end;

function task_run(tid:Integer):Boolean;
begin
 Result:=task_ref(tid).Run;
end;

function task_wait(tid,timeout:Integer):Boolean;
begin
 Result:=task_ref(tid).Running(timeout);
end;

function task_send(tid:Integer; const data:String):Integer;
begin
 Result:=task_ref(tid).Send(data);
end;

function task_recv(tid:Integer; maxlen:Integer):String;
begin
 Result:=task_ref(tid).Recv(maxlen);
end;

function task_txcount(tid:Integer):Integer;
begin
 Result:=task_ref(tid).StdInpPipeFifoCount;
end;

function task_rxcount(tid:Integer):Integer;
begin
 Result:=task_ref(tid).StdOutPipeFifoCount;
end;

function task_txspace(tid:Integer):Integer;
begin
 Result:=task_ref(tid).StdInpPipeFifoSpace;
end;

function task_rxspace(tid:Integer):Integer;
begin
 Result:=task_ref(tid).StdOutPipeFifoSpace;
end;

function task_result(tid:Integer):Integer;
begin
 Result:=task_ref(tid).ExitStatus;
end;

function task_kill(tid,how,exit_code,timeout:Integer):Boolean;
var task:TTask;
begin
 Result:=false;
 task:=task_ref(tid);
 if (task=nil) then Exit;
 task.Terminate(how,exit_code,timeout);
 if task.Running then task.Process.Terminate(exit_code);
 Result:=not task.Running;
end;

function task_ctrl(tid:Integer; const param:String):String;
var task:TTask; p,i:Integer; sn,sv,line:String; list:TStringList;
begin
 Result:='?';
 sn:=''; sv:='';
 task:=task_ref(tid);
 if (task<>nil) then
 try
  p:=Pos('=',param);
  if p=0 then begin
   sn:=param;
   sv:='';
  end else begin
   sn:=Copy(param,1,p-1);
   sv:=Copy(param,p+1,length(param)-p);
  end;
  if IsSameText(sn,'AppName') then begin
   if p=0 then Result:=task.AppName else begin
    task.AppName:=sv;
    Result:=IntToStr(Length(task.AppName));
   end;
  end else
  if IsSameText(sn,'CmdLine') then begin
   if p=0 then Result:=task.CmdLine else begin
    task.CmdLine:=sv;
    Result:=IntToStr(Length(task.CmdLine));
   end;
  end else
  if IsSameText(sn,'HomeDir') then begin
   if p=0 then Result:=task.HomeDir else begin
    task.HomeDir:=sv;
    Result:=IntToStr(Length(task.HomeDir));
   end;
  end else
  if IsSameText(sn,'ExeName') then begin
   Result:=task.ExeName;
  end else
  if IsSameText(sn,'Display') then begin
   if p>0 then begin
    task.Display:=StrToIntDef(sv,-1);
    if IsSameText(sv,'SW_HIDE')            then task.Display:=SW_HIDE;
    if IsSameText(sv,'SW_MAXIMIZE')        then task.Display:=SW_MAXIMIZE;
    if IsSameText(sv,'SW_MINIMIZE')        then task.Display:=SW_MINIMIZE;
    if IsSameText(sv,'SW_RESTORE')         then task.Display:=SW_RESTORE;
    if IsSameText(sv,'SW_SHOW')            then task.Display:=SW_SHOW;
    if IsSameText(sv,'SW_SHOWDEFAULT')     then task.Display:=SW_SHOWDEFAULT;
    if IsSameText(sv,'SW_SHOWMAXIMIZED')   then task.Display:=SW_SHOWMAXIMIZED;
    if IsSameText(sv,'SW_SHOWMINIMIZED')   then task.Display:=SW_SHOWMINIMIZED;
    if IsSameText(sv,'SW_SHOWMINNOACTIVE') then task.Display:=SW_SHOWMINNOACTIVE;
    if IsSameText(sv,'SW_SHOWNA')          then task.Display:=SW_SHOWNA;
    if IsSameText(sv,'SW_SHOWNOACTIVATE')  then task.Display:=SW_SHOWNOACTIVATE;
    if IsSameText(sv,'SW_SHOWNORMAL')      then task.Display:=SW_SHOWNORMAL;
   end;
   Result:=IntToStr(task.Display);
  end else
  if IsSameText(sn,'TxPipeSize')
  or IsSameText(sn,'StdInPipeSize')
  or IsSameText(sn,'StdInpPipeSize') then begin
   if p>0 then task.StdInpPipeSize:=StrToIntDef(Trim(sv),0);
   Result:=IntToStr(task.StdInpPipeSize);
  end else
  if IsSameText(sn,'RxPipeSize')
  or IsSameText(sn,'StdOutPipeSize') then begin
   if p>0 then task.Process.PipeBufferSize:=StrToIntDef(Trim(sv),0);
   Result:=IntToStr(task.StdOutPipeSize);
  end else
  if IsSameText(sn,'ProcessPriority') then begin
   if p>0 then task.ProcessPriority:=GetPriorityClassLevel(GetPriorityClassByName(Trim(sv)));
   Result:=GetPriorityClassName(GetPriorityClassByLevel(task.ProcessPriority));
  end else
  if IsSameText(sn,'CodePage') then begin
   if p>0 then task.CodePage:=StrToIntDef(sv,task.CodePage);
   Result:=IntToStr(task.CodePage);
  end else
  if IsSameText(sn,'ExitStatus') then begin
   Result:=IntToStr(task.ExitStatus);
  end else
  if IsSameText(sn,'ExitCode') then begin
   Result:=IntToStr(task.ExitCode);
  end else
  if IsSameText(sn,'Env')
  or IsSameText(sn,'Environ')
  or IsSameText(sn,'Environment') then begin
   if (p>0) then begin
    if (sv='') then task.Process.Environment.Clear else
    if (WordIndex(sv,'Inherit,Inherits,Inherited',',')>0) then begin
     if (task.Process.Environment.Count>0) then task.Process.Environment.Clear;
     task.Process.Environment.Text:=EnvironmentVariableList.Text;
     //Application.GetEnvironmentList(task.Process.Environment);
    end else begin
     list:=TStringList.Create;
     try
      list.Text:=sv;
      for i:=0 to list.Count-1 do begin
       line:=TrimLeft(list.Strings[i]);
       if (Pos('=',line)<2) then continue;
       task.Process.Environment.Add(line);
      end;
     finally
      list.Free;
     end;
    end;
   end;
   Result:=task.Process.Environment.Text;
  end;
 except
  on E:Exception do BugReport(E,nil,'task_ctrl');
 end;
 sn:=''; sv:='';
end;

initialization

 task_initialize;

end.
