 {
 ****************************************************************************
 CRW32 project
 Copyright (C) by Kuryakin Alexey, Sarov, Russia, 2002, <kouriakine@mail.ru>
 Routines to organize thread polling.
 Modifications:
 20010712 - Creation (uses CRW16) & test
 20030330 - Struggle for safety (add some try/except checks)...
 20030406 - FullPollingList, TPolling.Name, TPolling.GetCpuLoad
 20030409 - Wdt = Watchdog timer
 20050221 - Awake
 20050222 - AwakeFlag
 20050420 - Histogram
 20050428 - Priority class
 20061213 - ThreadHandle
 20190319 - FindPollingRefByName
 20221103 - Add CoInitialize/CoUninitialize to TPolling.Execute to use COM
 20221106 - CoIntializeBalanceCount,CoIntializeFailureCount
 20221115 - UseMsgPump; MessagePump in TPolling.Execute required for COM STA
 ****************************************************************************
 }
unit _Polling;

{$I _sysdef}

interface

uses
 sysutils, windows, messages, classes, contnrs, math, activex, _alloc, _dynar, _fifo, _str, _rtc;

 {
 *******************************************************************************
 TPolling periodically calls Action procedure to do something you need.
 Be carefully to write Action, do not forget, that this procedure executes in
 another thread.
 To do something in main VCL thread, call Polling.Syncronize, this procedure
 call aSynchro procedure. Use LonkXXX to transfer data to and from main VCL
 thread or keep any other info.
 *******************************************************************************
 }
type
 TPolling = class;
 TPollingAction = procedure(aPolling:TPolling; var Terminate:Boolean);
 TPolling = class(TLatch)
 private
  myThread     : TThread;
  myEnabled    : Boolean;
  myAction     : TPollingAction;
  myDelay      : Integer;
  myLinkObject : TObject;
  myLinkCustom : Pointer;
  myLinkText   : LongString;
  myLinkParam  : array[0..15] of Double;
  myLoopCount  : Int64;
  myLoopTime   : Int64;
  myLoopEvent  : THandle;
  myWakeEvent  : THandle;
  myAwakeFlag  : Boolean;
  myCpuLoad    : packed record
   LastTick    : DWORD;
   LastLoop    : Int64;
   LastKTime   : Int64;
   LastUTime   : Int64;
  end;
  myName       : ShortString;
  myWdt        : packed record
   Flag        : Boolean;
   LastTick    : Cardinal;
  end;
  myHist       : record
   Time        : Int64;
   Data        : array[0..511] of Int64;
  end;
  myUseMsgPump : Boolean;
  function    GetName:ShortString;
  function    GetThreadID:THandle;
  function    GetThreadHandle:THandle;
  function    GetEnabled:Boolean;
  procedure   SetEnabled(aEnabled:Boolean);
  function    GetUseMsgPump:Boolean;
  procedure   SetUseMsgPump(aUse:Boolean);
  function    GetDelay:Integer;
  procedure   SetDelay(aDelay:Integer);
  function    GetPriority:TThreadPriority;
  procedure   SetPriority(aPriority:TThreadPriority);
  function    GetLinkObject:TObject;
  procedure   SetLinkObject(aObject:TObject);
  function    GetLinkCustom:Pointer;
  procedure   SetLinkCustom(aCustom:Pointer);
  function    GetLinkText:LongString;
  procedure   SetLinkText(const aText:LongString);
  function    GetLinkParam(Index:Integer):Double;
  procedure   SetLinkParam(Index:Integer; aParam:Double);
  function    GetLoopCount:Int64;
  function    GetLoopTime:Int64;
  function    GetHistStep:Integer;
  function    GetHistSize:Integer;
  function    GetHistData(Index:Integer):Int64;
  procedure   Histogram(const aTime:Int64);
  procedure   ExecuteAction(var Terminate:Boolean);
 public
  constructor Create(aAction   : TPollingAction;
                     aDelay    : Integer         = 10;
                     aPriority : TThreadPriority = tpNormal;
                     aEnabled  : Boolean         = false;
                     aName     : ShortString     = 'NONAME');
  destructor  Destroy; override;
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
 public
  class function    DefMsgPump:Boolean;
  class procedure   SetDefMsgPump(aUse:Boolean);
 public
  property    Name                 : ShortString     read GetName;
  property    ThreadID             : THandle         read GetThreadID;
  property    ThreadHandle         : THandle         read GetThreadHandle;
  property    Enabled              : Boolean         read GetEnabled    write SetEnabled;
  property    UseMsgPump           : Boolean         read GetUseMsgPump write SetUseMsgPump;
  property    Delay                : Integer         read GetDelay      write SetDelay;
  property    Priority             : TThreadPriority read GetPriority   write SetPriority;
  property    LinkObject           : TObject         read GetLinkObject write SetLinkObject;
  property    LinkCustom           : Pointer         read GetLinkCustom write SetLinkCustom;
  property    LinkText             : LongString      read GetLinkText   write SetLinkText;
  property    LinkParam[i:Integer] : Double          read GetLinkParam  write SetLinkParam;
  property    LoopCount            : Int64           read GetLoopCount;
  property    LoopTime             : Int64           read GetLoopTime;
  property    HistStep             : Integer         read GetHistStep;
  property    HistSize             : Integer         read GetHistSize;
  property    HistData[i:Integer]  : Int64           read GetHistData;
  function    SyncLoop(aTimeOut:Integer; aNum:Integer):Int64;
  function    WaitForLoop(aTimeOut:DWORD=INFINITE):DWORD;
  function    Enable(aEnabled:Boolean; aTimeOut:DWORD=INFINITE):Boolean;
  function    GetCpuLoad(var Summ,Kern,User,Freq:Double):Boolean;
  function    WdtAlert(aWdt:Cardinal):Boolean;
  procedure   WdtReset;
  procedure   Awake;
  function    AwakeFlag:Boolean;
  procedure   HistClear;
 end;

function NewPolling(aAction   : TPollingAction;
                    aDelay    : Integer         = 10;
                    aPriority : TThreadPriority = tpNormal;
                    aEnabled  : Boolean         = false;
                    aName     : ShortString     = 'NONAME'
                            ) : TPolling;
procedure Kill(var TheObject:TPolling); overload;

 //
 // Functions to get priority (by) name and to read Polling delay and priority
 // from ini -  file. For example, to read
 //     [System]
 //     UartPolling = 50, tpNormal
 // uses ReadIniFilePolling(SysIniFile,'[System]','UartPolling',Delay,Priority);
 //
function GetPriorityName(aPriority : TThreadPriority):ShortString;
function GetPriorityByName(const aName:ShortString; Def:TThreadPriority=tpNormal):TThreadPriority;
function ReadIniFilePolling(const IniFile,Section,Name:ShortString;
                        var Delay:Integer; var Priority:TThreadPriority):Boolean;

 //
 // Functions to get priority class (by) name and to read priority class
 // from ini -  file. For example, to read
 //     [System]
 //     PriorityClass = RealTime, 100
 // uses ReadIniFilePriorityClass(SysIniFile,'[System]','PriorityClass',PriorityClass,CheckPeriod);
 // Priopity class table are:
 // Level  Class                   Name
 // 4      IDLE_PRIORITY_CLASS     'Idle'
 // 6      LOWER_PRIORITY_CLASS    'Lower'
 // 8      NORMAL_PRIORITY_CLASS   'Normal'
 // 10     HIGHER_PRIORITY_CLASS   'Higher'
 // 13     HIGH_PRIORITY_CLASS     'High'
 // 24     REALTIME_PRIORITY_CLASS 'RealTime'
 //
const
 LOWER_PRIORITY_CLASS  = $4000;     // Not defined in Windows.pas
 HIGHER_PRIORITY_CLASS = $8000;     // Not defined in Windows.pas
 WantedPriorityClass   : DWORD = 0; // Wanted priority class for this process
 PeriodPriorityClass   : DWORD = 0; // Period to check & force wanted priority

function GetPriorityClassLevel(aPriorityClass:DWORD):Integer;
function GetPriorityClassName(aPriorityClass:DWORD):LongString;
function GetPriorityClassByLevel(aLevel:Integer):DWORD;
function GetPriorityClassByName(const aName:ShortString):DWORD;
function GetAbsolutePriority(PriorityClass:DWORD; ThreadPriority:TThreadPriority):Integer;
function ReadIniFilePriorityClass(const IniFile,Section,Name:ShortString;
                        var aPriorityClass:DWORD; var aPeriod:DWORD):Boolean;
procedure ForcePriorityClass(aWantedPriorityClass:DWORD; aPeriodPriorityClass:DWORD);
procedure CheckPriorityClass;

function FindPollingRefByName(const Name:ShortString):Integer;

function FullPollingList:TObjectStorage;

implementation

uses _fio;

const
 CoIntializeBalanceCount : Integer = 0;
 CoIntializeFailureCount : Integer = 0;

 {
 *******************************************************************************
 TPollingThread implementation
 *******************************************************************************
 }
type
 TPollingThread = class(TThread)
 private
  myPolling : TPolling;
 public
  constructor Create(aPolling:TPolling);
  procedure   Execute; override;
 end;

constructor TPollingThread.Create(aPolling:TPolling);
begin
 inherited Create(true);
 myPolling:=aPolling;
end;

procedure TPollingThread.Execute;
var
 Term : Boolean;
 iCOM : Boolean;
 iMSG : Boolean;
 procedure CreateMessageQueue;
 var aMsg:TMsg;
 begin
  PeekMessage(aMsg,0,WM_USER,WM_USER,PM_NOREMOVE);
  iMSG:=true;
 end;
 procedure MessagePump; // Required for COM STA Marshaling
 var aMsg:TMsg;
 begin
  if not iMSG then CreateMessageQueue;
  while PeekMessage(aMsg,0,0,0,PM_REMOVE) do DispatchMessage(aMsg);
 end;
begin
 try
  iMSG:=false;
  iCOM:=Succeeded(CoInitialize(nil));
  try
   if iCOM then LockedInc(CoIntializeBalanceCount) else LockedInc(CoIntializeFailureCount);
   // Main Execution Loop
   while not Terminated do
   try
    Term:=false;
    myPolling.WdtReset;
    myPolling.ExecuteAction(Term);
    if myPolling.UseMsgPump then MessagePump;
    if Term then Terminate;
   except
    on E:Exception do BugReport(E,Self,'ExecuteLoop');
   end;
  finally
   if iCOM then CoUninitialize;
   if iCOM then LockedDec(CoIntializeBalanceCount);
  end;
 except
  on E:Exception do BugReport(E,Self,'Execute');
 end;
end;

 {
 *******************************************************************************
 TPolling implementation
 *******************************************************************************
 }
constructor TPolling.Create(aAction   : TPollingAction;
                            aDelay    : Integer         = 10;
                            aPriority : TThreadPriority = tpNormal;
                            aEnabled  : Boolean         = false;
                            aName     : ShortString     = 'NONAME');
begin
 inherited Create;
 myThread:=TPollingThread.Create(Self);
 myEnabled:=aEnabled;
 myAction:=aAction;
 myThread.Priority:=aPriority;
 myDelay:=aDelay;
 myLinkObject:=nil;
 myLinkCustom:=nil;
 myLinkText:='';
 SafeFillChar(myLinkParam,sizeof(myLinkParam),0);
 myLoopCount:=0;
 myLoopTime:=0;
 myLoopEvent:=CreateEvent(nil, True, False, nil);
 myWakeEvent:=CreateEvent(nil, False, False, nil);
 myAwakeFlag:=false;
 myThread.Suspended:=false;
 SafeFillChar(myCpuLoad,sizeof(myCpuLoad),0);
 myName:=Trim(aName);
 myWdt.Flag:=false;
 myWdt.LastTick:=0;
 myUseMsgPump:=DefMsgPump;
 HistClear;
end;

destructor TPolling.Destroy;
begin
 Enabled:=false;
 myThread.Terminate;
 Awake; // Fast OFF
 myThread.WaitFor;
 Kill(TObject(myThread));
 CloseHandle(myLoopEvent);
 CloseHandle(myWakeEvent);
 LinkText:='';
 inherited Destroy;
end;

procedure TPolling.AfterConstruction;
begin
 inherited AfterConstruction;
 FullPollingList.Add(Self);
end;

procedure TPolling.BeforeDestruction;
begin
 FullPollingList.Remove(Self);
 inherited BeforeDestruction;
end;

function TPolling.GetName:ShortString;
begin
 if Assigned(Self) then Result:=myName else Result:='';
end;

function TPolling.GetThreadID:THandle;
begin
 if Assigned(Self) and Assigned(myThread)
 then Result:=myThread.ThreadID
 else Result:=0;
end;

function TPolling.GetThreadHandle:THandle;
begin
 if Assigned(Self) and Assigned(myThread)
 then Result:=myThread.Handle
 else Result:=0;
end;

function TPolling.GetEnabled:Boolean;
begin
 if Assigned(Self) then begin
  Lock;
  Result:=myEnabled;
  Unlock;
 end else Result:=false;
end;

procedure TPolling.SetEnabled(aEnabled:Boolean);
begin
 if Assigned(Self) then begin
  Lock;
  myEnabled:=aEnabled;
  myHist.Time:=0;
  Unlock;
 end;
end;

function TPolling.GetUseMsgPump:Boolean;
begin
 if (Self<>nil)
 then Result:=myUseMsgPump
 else Result:=false;
end;

procedure TPolling.SetUseMsgPump(aUse:Boolean);
begin
 if (Self<>nil) then myUseMsgPump:=aUse;
end;

const TPollingDefMsgPump : Boolean = false;

class function TPolling.DefMsgPump:Boolean;
begin
 Result:=TPollingDefMsgPump;
end;

class procedure TPolling.SetDefMsgPump(aUse:Boolean);
begin
 TPollingDefMsgPump:=aUse;
end;

function TPolling.GetDelay:Integer;
begin
 if Assigned(Self) then begin
  Lock;
  Result:=myDelay;
  Unlock;
 end else Result:=0;
end;

procedure TPolling.SetDelay(aDelay:Integer);
begin
 if Assigned(Self) then begin
  Lock;
  myDelay:=Max(1,aDelay);
  Unlock;
 end;
end;

function TPolling.GetPriority:TThreadPriority;
begin
 if Assigned(Self) and Assigned(myThread)
 then Result:=myThread.Priority
 else Result:=tpNormal;
end;

procedure TPolling.SetPriority(aPriority:TThreadPriority);
begin
 if Assigned(Self) and Assigned(myThread)
 then myThread.Priority:=aPriority;
end;

function TPolling.GetLinkObject:TObject;
begin
 if Assigned(Self) then Result:=myLinkObject else Result:=nil;
end;

procedure TPolling.SetLinkObject(aObject:TObject);
begin
 if Assigned(Self) then myLinkObject:=aObject;
end;

function TPolling.GetLinkCustom:Pointer;
begin
 if Assigned(Self) then Result:=myLinkCustom else Result:=nil;
end;

procedure TPolling.SetLinkCustom(aCustom:Pointer);
begin
 if Assigned(Self) then myLinkCustom:=aCustom;
end;

function TPolling.GetLinkText:LongString;
begin
 if Assigned(Self) then Result:=myLinkText else Result:='';
end;

procedure TPolling.SetLinkText(const aText:LongString);
begin
 if Assigned(Self) then myLinkText:=aText;
end;

function TPolling.GetLinkParam(Index:Integer):Double;
begin
 if Assigned(Self) and (Cardinal(Index)<=High(myLinkParam))
 then Result:=myLinkParam[Index]
 else Result:=0;
end;

procedure TPolling.SetLinkParam(Index:Integer; aParam:Double);
begin
 if Assigned(Self) and (Cardinal(Index)<=High(myLinkParam))
 then myLinkParam[Index]:=aParam;
end;

function TPolling.GetLoopCount:Int64;
begin
 if Assigned(Self) then begin
  Lock;
  Result:=myLoopCount;
  Unlock;
 end else Result:=0;
end;

function TPolling.GetLoopTime:Int64;
begin
 if Assigned(Self) then begin
  Lock;
  Result:=myLoopTime;
  Unlock;
 end else Result:=0;
end;

function TPolling.GetHistStep:Integer;
begin
 Result:=10;
end;

function TPolling.GetHistSize:Integer;
begin
 if Assigned(Self) then Result:=High(myHist.Data)-Low(myHist.Data)+1 else Result:=0;
end;

function TPolling.GetHistData(Index:Integer):Int64;
begin
 Result:=0;
 if Assigned(Self) then
 if Index>=Low(myHist.Data) then
 if Index<=High(myHist.Data) then begin
  Lock;
  Result:=myHist.Data[Index];
  Unlock;
 end;
end;

procedure TPolling.Histogram(const aTime:Int64);
var dt:Integer;
begin
 if Assigned(Self) then begin
  if myHist.Time<>0 then begin
   dt:=aTime-myHist.Time;
   dt:=dt div HistStep;
   if dt<Low(myHist.Data) then dt:=Low(myHist.Data);
   if dt>High(myHist.Data) then dt:=High(myHist.Data);
   inc(myHist.Data[dt]);
  end;
  myHist.Time:=aTime;
 end;
end;

procedure TPolling.HistClear;
begin
 if Assigned(Self) then begin
  Lock;
  SafeFillChar(myHist,SizeOf(myHist),0);
  Unlock;
 end;
end;

procedure TPolling.ExecuteAction(var Terminate:Boolean);
var
 aTime    : Int64;
 aDelay   : Integer;
 aEnabled : Boolean;
begin
 if Assigned(Self) then begin
  aDelay:=Delay;
  aEnabled:=Enabled;
  aTime:=IntMSecNow;
  if aEnabled and (@myAction<>nil) then
  try
   myAction(Self,Terminate);
  except
   on E:Exception do BugReport(E,Self);
  end;
  Lock;
  Histogram(aTime);
  inc(myLoopCount);
  myLoopTime:=aTime;
  Unlock;
  SetEvent(myLoopEvent);
  if aDelay>=0
  then myAwakeFlag:=(WaitForSingleObject(myWakeEvent,aDelay)=WAIT_OBJECT_0)
  else myAwakeFlag:=false;
 end;
end;

function TPolling.SyncLoop(aTimeOut:Integer; aNum:Integer):Int64;
begin
 Result:=0;
 if Assigned(Self) then begin
  if (aTimeOut>0) then
  while (aNum>0) do begin
   Awake; Dec(aNum);
   WaitForLoop(aTimeOut);
  end;
  Result:=LoopCount;
 end;
end;

function TPolling.WaitForLoop(aTimeOut:DWORD=INFINITE):DWORD;
begin
 Result:=WAIT_FAILED;
 if Assigned(Self) then
 if Assigned(myThread) then
 if (GetCurrentThreadID <> myThread.ThreadID) then
 if not myThread.Suspended then begin
  ResetEvent(myLoopEvent);
  Awake;
  Result:=WaitForSingleObject(myLoopEvent,aTimeout);
 end;
end;

function TPolling.Enable(aEnabled:Boolean; aTimeOut:DWORD=INFINITE):Boolean;
begin
 Result:=false;
 if Assigned(Self) then begin
  Enabled:=aEnabled;
  Result:=(WaitForLoop(aTimeOut)=WAIT_OBJECT_0);
 end;
end;

function TPolling.GetCpuLoad(var Summ,Kern,User,Freq:Double):Boolean;
var
 CurrTick  : DWORD;
 CurrLoop  : Int64;
 CurrCTime : Int64;
 CurrETime : Int64;
 CurrKTime : Int64;
 CurrUTime : Int64;
begin
 Result:=false;
 Summ:=0;
 Kern:=0;
 User:=0;
 Freq:=0;
 try
  if Assigned(Self) and Assigned(myThread) then with myCpuLoad do begin
   CurrLoop:=myLoopCount;
   CurrTick:=GetTickCount;
   if (CurrTick>LastTick) then begin
    if GetThreadTimes(myThread.Handle,
                      TFileTime(CurrCTime),TFileTime(CurrETime),
                      TFileTime(CurrKTime),TFileTime(CurrUTime))
    then begin
     Kern:=Max(0,Min(100,0.01*(CurrKTime-LastKTime)/(CurrTick-LastTick)));
     User:=Max(0,Min(100,0.01*(CurrUTime-LastUTime)/(CurrTick-LastTick)));
     Summ:=Max(0,Min(100,Kern+User));
    end else begin
     CurrKTime:=0;
     CurrUTime:=0;
    end;
    Freq:=Max(0,1e3*(CurrLoop-LastLoop)/(CurrTick-LastTick));
   end;
   LastKTime:=CurrKTime;
   LastUTime:=CurrUTime;
   LastTick:=CurrTick;
   LastLoop:=CurrLoop;
   Result:=true;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TPolling.WdtAlert(aWdt:Cardinal):Boolean;
var
 CurrTick : Cardinal;
begin
 Result:=false;
 if aWdt>0 then
 if Assigned(Self) then begin
  CurrTick:=Windows.GetTickCount;
  if myWdt.Flag then begin
   if CurrTick - myWdt.LastTick > aWdt then Result:=true;
  end else begin
   myWdt.Flag:=true;
   myWdt.LastTick:=CurrTick;
  end;
 end;
end;

procedure TPolling.WdtReset;
begin
 if Assigned(Self) then myWdt.Flag:=false;
end;

procedure TPolling.Awake;
begin
 if Assigned(Self) then SetEvent(myWakeEvent);
end;

function TPolling.AwakeFlag:Boolean;
begin
 if Assigned(Self) then Result:=myAwakeFlag else Result:=false;
end;

function NewPolling(aAction   : TPollingAction;
                    aDelay    : Integer         = 10;
                    aPriority : TThreadPriority = tpNormal;
                    aEnabled  : Boolean         = false;
                    aName     : ShortString     = 'NONAME'
                            ) : TPolling;
begin
 Result:=nil;
 try
  Result:=TPolling.Create(aAction, aDelay, aPriority, aEnabled, aName);
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure Kill(var TheObject:TPolling); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end;
end;

function GetPriorityName(aPriority : TThreadPriority):ShortString;
begin
 case aPriority of
  tpIdle         : Result:='tpIdle';
  tpLowest       : Result:='tpLowest';
  tpLower        : Result:='tpLower';
  tpNormal       : Result:='tpNormal';
  tpHigher       : Result:='tpHigher';
  tpHighest      : Result:='tpHighest';
  tpTimeCritical : Result:='tpTimeCritical';
  else             Result:='tpNormal';
 end;
end;

function GetPriorityByName(const aName:ShortString; Def:TThreadPriority=tpNormal):TThreadPriority;
var
 tp : TThreadPriority;
begin
 Result:=Def;
 for tp:=Low(tp) to High(tp) do
 if SameText(aName,GetPriorityName(tp)) then begin
  Result:=tp;
  Break;
 end;
end;

function ReadIniFilePolling(const IniFile,Section,Name:ShortString;
                        var Delay:Integer; var Priority:TThreadPriority):Boolean;
var
 Data : packed record
  D   : Integer;
  P   : ShortString;
 end;
begin
 Result:=false;
 if ReadIniFileVariable(IniFile,Section,UnifyAlias(Name)+'%i;%a',Data) then begin
  Delay:=Data.D;
  Priority:=GetPriorityByName(Data.P);
  Result:=true;
 end else begin
  Delay:=10;
  Priority:=tpNormal;
 end;
end;

const
 PriorityClassTable : array[0..5] of
  record prLevel:Integer; prClass:DWORD; prName:PChar end = (
  (prLevel:4;  prClass:IDLE_PRIORITY_CLASS;     prName:'Idle'),
  (prLevel:6;  prClass:LOWER_PRIORITY_CLASS;    prName:'Lower'),
  (prLevel:8;  prClass:NORMAL_PRIORITY_CLASS;   prName:'Normal'),
  (prLevel:10; prClass:HIGHER_PRIORITY_CLASS;   prName:'Higher'),
  (prLevel:13; prClass:HIGH_PRIORITY_CLASS;     prName:'High'),
  (prLevel:24; prClass:REALTIME_PRIORITY_CLASS; prName:'RealTime')
  );

function GetPriorityClassLevel(aPriorityClass:DWORD):Integer;
var
 i : Integer;
begin
 Result:=0;
 for i:=Low(PriorityClassTable) to High(PriorityClassTable) do
 if aPriorityClass=PriorityClassTable[i].prClass then begin
  Result:=PriorityClassTable[i].prLevel;
  Exit;
 end;
end;

function GetPriorityClassName(aPriorityClass:DWORD):LongString;
var
 i : Integer;
begin
 Result:='';
 for i:=Low(PriorityClassTable) to High(PriorityClassTable) do
 if aPriorityClass=PriorityClassTable[i].prClass then begin
  Result:=PriorityClassTable[i].prName;
  Exit;
 end;
end;

function GetPriorityClassByLevel(aLevel:Integer):DWORD;
var
 i : Integer;
begin
 Result:=0;
 for i:=Low(PriorityClassTable) to High(PriorityClassTable) do
 if aLevel>=PriorityClassTable[i].prLevel
 then Result:=PriorityClassTable[i].prClass
 else Exit;
end;

function GetPriorityClassByName(const aName:ShortString):DWORD;
var
 i : Integer;
begin
 Result:=0;
 if not IsEmptyStr(aName) then
 if Str2Long(aName,i) then Result:=GetPriorityClassByLevel(i) else
 for i:=Low(PriorityClassTable) to High(PriorityClassTable) do
 if IsSameText(aName,PriorityClassTable[i].prName) then begin
  Result:=PriorityClassTable[i].prClass;
  Exit;
 end;
end;

function GetAbsolutePriority(PriorityClass:DWORD; ThreadPriority:TThreadPriority):Integer;
var pmin,pmax,pri:Integer;
begin
 Result:=0;
 pri:=GetPriorityClassLevel(PriorityClass);
 if pri>0 then begin
  if pri<16 then begin
   pmin:=1; pmax:=15;
  end else begin
   pmin:=16; pmax:=31;
  end;
  case ThreadPriority of
   tpIdle         : Result:=pmin;
   tpLowest       : Result:=pri-2;
   tpLower        : Result:=pri-1;
   tpNormal       : Result:=pri;
   tpHigher       : Result:=pri+1;
   tpHighest      : Result:=pri+2;
   tpTimeCritical : Result:=pmax;
  end;
  if Result<pmin then Result:=pmin;
  if Result>pmax then Result:=pmax;
 end;
end;

function ReadIniFilePriorityClass(const IniFile,Section,Name:ShortString;
                        var aPriorityClass:DWORD; var aPeriod:DWORD):Boolean;
var
 Data : packed record
  p   : ShortString;
  t   : Integer;
 end;
begin
 Result:=false;
 Data.p:='';
 Data.t:=0;
 ReadIniFileVariable(IniFile,Section,UnifyAlias(Name)+'%a;%i',Data);
 aPriorityClass:=GetPriorityClassByName(Data.p);
 if aPriorityClass>0 then begin
  aPeriod:=Data.t;
  Result:=true;
 end else begin
  aPriorityClass:=0;
  aPeriod:=0;
 end;
end;

procedure ForcePriorityClass(aWantedPriorityClass:DWORD; aPeriodPriorityClass:DWORD);
var
 hProcess : THandle;
begin
 try
  WantedPriorityClass:=aWantedPriorityClass;
  PeriodPriorityClass:=aPeriodPriorityClass;
  if WantedPriorityClass>0 then
  if GetPriorityClassLevel(WantedPriorityClass)>0 then begin
   hProcess:=GetCurrentProcess;
   if GetPriorityClass(hProcess)<>WantedPriorityClass
   then SetPriorityClass(hProcess,WantedPriorityClass);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure CheckPriorityClass;
const
 LastCheck : Cardinal = 0;
var
 hProcess  : THandle;
 TickCount : Cardinal;
begin
 try
  if PeriodPriorityClass>0 then
  if WantedPriorityClass>0 then begin
   TickCount:=GetTickCount;
   if TickCount-LastCheck>PeriodPriorityClass then begin
    if GetPriorityClassLevel(WantedPriorityClass)>0 then begin
     hProcess:=GetCurrentProcess;
     if GetPriorityClass(hProcess)<>WantedPriorityClass
     then SetPriorityClass(hProcess,WantedPriorityClass);
    end;
    LastCheck:=TickCount;
   end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

type TPollingFindRec=packed record RefToFind:Integer; NameToFind:ShortString; end;
procedure PollingFind(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if aObject is TPolling then with aObject as TPolling do
 if Assigned(CustomData) then with TPollingFindRec(CustomData^) do
 if SameText(Name,NameToFind) then begin
  Terminate:=true;
  RefToFind:=Ref;
 end;
end;

function FindPollingRefByName(const Name:ShortString):Integer;
var PFR:TPollingFindRec;
begin
 Result:=0;
 if not IsEmptyStr(Name) then begin
  PFR.RefToFind:=0;
  PFR.NameToFind:=Name;
  FullPollingList.ForEach(PollingFind,@PFR);
  Result:=PFR.RefToFind;
 end;
end;

const
 TheFullPollingList : TObjectStorage = nil;

function FullPollingList:TObjectStorage;
begin
 Result:=TheFullPollingList;
end;

initialization

 TheFullPollingList:=NewObjectStorage(false);
 TheFullPollingList.Master:=TheFullPollingList;
 TheFullPollingList.OwnsObjects:=false;

finalization

 ResourceLeakageLog(Format('%-60s = %d',['FullPollingList.Count', TheFullPollingList.Count]));
 Kill(TheFullPollingList);

 ResourceLeakageLog(Format('%-60s = %d',['TPolling.CoIntializeBalance.Count', CoIntializeBalanceCount]));
 ResourceLeakageLog(Format('%-60s = %d',['TPolling.CoIntializeFailure.Count', CoIntializeFailureCount]));

end.
