#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

################################################################
## Utility to make UniHeater's configuration.                 ##
################################################################

function fatal(){
 unix tooltip-notifier text "$scriptname: $2" preset stdError delay 60000;
 1>&2 echo -ne "\n$2\n\n";
 exit $1;
};
function call(){
 if [ $# -gt 0 ]; then
  eval "$@";
 fi;
};
function rem(){
 return 0;
};

################################
# Current script identification.
################################
declare startupdir="$(pwd -LP)";
declare scriptfile="${BASH_SOURCE[0]}";
declare scriptname="$(basename $scriptfile)";
declare scriptbase="$(basename $scriptfile .sh)";
declare scripthome="$(dirname  $scriptfile)";
declare scriptFILE="$(realpath $scriptfile)";
declare scriptHOME="$(dirname  $scriptFILE)";

declare -x SelfExt=".sh";
declare -x SelfBase="$scriptbase";
declare -x SelfFile="$scriptFILE";
declare -x SelfName="$scriptname";
declare -x HomeDir="$startupdir";
declare -x SelfDir="$scriptHOME";

declare -i errors=0;


function uniheater_make_cfg(){
 call SetErrors 0;
 call SetErrorLevel 0;
 #ParseOptions
 case $1 in
  -h|--help)  call PrintHelp; exit 0; ;;
  --version)  call PrintVersion; exit 0; ;;
  *) ;;
 esac;
 #ParseArguments
 call RequiredParamStr UH_PrefixName $1                     || exit 1       ; ### Prefix string to identify UniHeater
 call RequiredParamInt UH_NumHeaters $2                     || exit 1       ; ### Number of Unified Heaters, 1..32
 call RequiredParamInt UH_TileNumber $3                     || exit 1       ; ### Tile number per Row, 1..UH_NumHeaters
 call RequiredParamStr UH_CfgFolder  $4                     || exit 1       ; ### Target Directory for CFG files
 call RequiredParamStr UH_CrcFolder  $5                     || exit 1       ; ### Target Directory for CRC files
 call RequiredParamStr UH_BmpFolder  $6                     || exit 1       ; ### Target Directory for BMP files
 #SetAddonVariables
 call GetPrefixFile UH_PrefixFile $UH_PrefixName                            ; ### File name prefix
 call Define UH_PrefixFile      $(printenv UH_PrefixName | tr '.' '_')      ; ### File name prefix
 call Define UH_HomeDir   $CRW_DAQ_SYS_HOME_DIR/resource/daqsite/uniheater  ; ### Source directory of UniHeater
 call Define UH_PrefixTSen      SENSOR.T                                    ; ### Prefix of T sensors to bind
 call Define UH_PrefixPSen      SENSOR.P                                    ; ### Prefix of P sensors to bind
 call Define UH_NumberTSen      $UH_NumHeaters                              ; ### Number of T sensors to bind
 call Define UH_NumberPSen      $UH_NumHeaters                              ; ### Number of P sensors to bind
 call Define UH_NickTSen        T                                           ; ### Prefix of T sensors nicknames to display
 call Define UH_NickPSen        P                                           ; ### Prefix of P sensors nicknames to display
 call Define UH_SmoothingWw     1e-4                                        ; ### Smoothing Window Width, in TimeUnits
 call Define UH_SmoothingPw     0                                           ; ### Smoothing Power, -1..9
 call Define UH_SmoothingK1     2                                           ; ### Smoothing Param K1,  0..9
 call Define UH_SmoothingK2     2                                           ; ### Smoothing Param K2,  0..9
 call Define UH_HistoryLen      1000                                        ; ### History for curves
 call Define UH_EncryptMethod   RC6                                         ; ### DEFAULT,BLOWFISH,GOST,RC2,RC4,RC5,RC6,BASE64,HEX,NONE
 #CheckRequiredEntities
 call RequiredUnixUtils                                      || exit 1;
 call RequiredAdminRights                                    || exit 1;
 call RequiredParentCrwDaq                                   || exit 1;
 call RequiredDefined UH_PrefixName                          || exit 1;
 call RequiredDefined UH_PrefixFile                          || exit 1;
 call RequiredDefined UH_NumHeaters                          || exit 1;
 call RequiredDefined UH_TileNumber                          || exit 1;
 call RequiredIntRange $UH_NumHeaters 1 32                   || exit 1;
 call RequiredIntRange $UH_TileNumber 1 $UH_NumHeaters       || exit 1;
 call RequiredDefined UH_CfgFolder                           || exit 1;
 call RequiredDefined UH_BmpFolder                           || exit 1;
 call RequiredDefined UH_CrcFolder                           || exit 1;
 call RequiredFile "$UH_CfgFolder/"                          || exit 1;
 call RequiredFile "$UH_BmpFolder/"                          || exit 1;
 call RequiredFile "$UH_CrcFolder/"                          || exit 1;
 call RequiredDefined UH_HomeDir                             || exit 1;
 call RequiredFile "$UH_HomeDir/"                            || exit 1;
 call RequiredFile "$UH_HomeDir/uniheater_make_crc.sh"       || exit 1;
 if pushd "$HomeDir" >/dev/null 2>&1; then
  call UniHeaterConfig;
 fi;
 popd >/dev/null 2>&1;
 call HasNoErrors || call ErrorPrintLn "$errors error(s) found in $SelfName.";
 return $?;
};

function PrintVersion(){
  echo "$SelfBase version 1.0";
};

function  PrintHelp(){
 call PrintVersion;
 echo "Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "Utility to make UniHeater's configuration.";
 echo "Usage:";
 echo " $SelfBase [Options] [Arguments]";
 echo "Options:";
 echo " -h,--help  - print help";
 echo " --version  - print version";
 echo "Arguments:";
 echo " UH_PrefixName - Prefix name to identify Uniheater";
 echo " UH_NumHeaters - Number of Unified Heaters, 1..32";
 echo " UH_TileNumber - Tile number = columns per row, 1..UH_NumHeaters";
 echo " UH_CfgFolder  - Target directory for CFG files, like ../config";
 echo " UH_CrcFolder  - Target directory for CRC files, like ../circuits";
 echo " UH_BmpFolder  - Target directory for BMP files, like ../bitmaps";
 echo "Example:";
 echo " call $SelfBase DEMO 16 8 ../config ../circuits ../bitmaps";
};

function UniHeaterConfig(){
 call UniHeaterEnvironment                                                    || call IncErrors 1;
 call UniHeaterBindCfg  > "$UH_CfgFolder/${UH_PrefixFile}_uniheater_bind.cfg" || call IncErrors 1;
 call UniHeaterCtrlCfg  > "$UH_CfgFolder/${UH_PrefixFile}_uniheater_ctrl.cfg" || call IncErrors 1;
 call UniHeaterDisCfg   > "$UH_CfgFolder/${UH_PrefixFile}_uniheater_dis.cfg"  || call IncErrors 1;
 call UniHeaterDicCfg   > "$UH_CfgFolder/${UH_PrefixFile}_uniheater_dic.cfg"  || call IncErrors 1;
 call Exec "$UH_HomeDir/uniheater_make_crc.sh" $UH_PrefixName $UH_NumHeaters $UH_TileNumber "$UH_CrcFolder" "$UH_BmpFolder" || call IncErrors 1;
};

function UniHeaterEnvironment(){
 export UH_HomeRef="~~/resource/daqsite/uniheater"  ; ### UniHeater home directory reference
 export UH_PREF="$UH_PrefixName.UNIHEATER"          ; ### UniHeater name prefix
 export UH_FILE="$(printenv UH_PREF | tr '.' '_')"  ; ### UniHeater file prefix
 export UH_PATH="$(printenv UH_PREF | tr '.' '/')"  ; ### UniHeater path prefix
};

function UniHeaterBindCfg(){
 call UniHeaterSelector;
};

function UniHeaterCtrlCfg(){
     call UniHeaterHeader;
     call UniHeaterBinding;
     call UniHeaterDataStorage      $UH_NumHeaters;
     call UniHeaterTagList          $UH_NumHeaters;
     call UniHeaterWinCtrl          $UH_NumHeaters;
     call UniHeaterWinProp;
     call UniHeaterWinPlot          $UH_NumHeaters;
     call UniHeaterCtrl             $UH_NumHeaters;
     call UniHeaterCustomParam      $UH_NumHeaters;
     call UniHeaterStartupScript    $UH_NumHeaters    $UH_TileNumber;
     call UniHeaterPulser           $UH_NumHeaters;
     call UniHeaterSelect           $UH_NumHeaters;
 rem call UniHeaterDimDis           $UH_NumHeaters;
 rem call UniHeaterDimDic           $UH_NumHeaters;
};

function UniHeaterDisCfg(){
     call UniHeaterHeader
     call UniHeaterBinding
     call UniHeaterDataStorage      $UH_NumHeaters
     call UniHeaterTagList          $UH_NumHeaters
     call UniHeaterWinCtrl          $UH_NumHeaters
     call UniHeaterWinProp
     call UniHeaterWinPlot          $UH_NumHeaters
     call UniHeaterCtrl             $UH_NumHeaters
     call UniHeaterCustomParam      $UH_NumHeaters
     call UniHeaterStartupScript    $UH_NumHeaters    $UH_TileNumber
     call UniHeaterPulser           $UH_NumHeaters
     call UniHeaterSelect           $UH_NumHeaters
     call UniHeaterDimDis           $UH_NumHeaters
 rem call UniHeaterDimDic           $UH_NumHeaters
};

function UniHeaterDicCfg(){
     call UniHeaterHeader;
 rem call UniHeaterBinding;
     call UniHeaterDataStorage      $UH_NumHeaters;
     call UniHeaterTagList          $UH_NumHeaters;
     call UniHeaterWinCtrl          $UH_NumHeaters;
     call UniHeaterWinProp;
     call UniHeaterWinPlot          $UH_NumHeaters;
     call UniHeaterCtrl             $UH_NumHeaters;
 rem call UniHeaterCustomParam      $UH_NumHeaters
     call UniHeaterStartupScript    $UH_NumHeaters    $UH_TileNumber;
 rem call UniHeaterPulser           $UH_NumHeaters;
 rem call UniHeaterSelect           $UH_NumHeaters;
 rem call UniHeaterDimDis           $UH_NumHeaters;
     call UniHeaterDimDic           $UH_NumHeaters;
};

function UniHeaterHeader(){
 echo ";******************************************************************************";
 echo ";*** ПОДСИСТЕМА УПРАВЛЕНИЯ НАГРЕВАТЕЛЯМИ UNIHEATER (Unified Heaters)         **";
 echo ";******************************************************************************";
 echo "";
 echo ";******************************************************************************";
 echo "; Описание подключения UniHeater к DAQ системе: (здесь i-номер нагревателя)";
 echo "; 1) Для исключения конфликтов имен устройств, кривых и тегов в DAQ-системе";
 echo ";    объекты UniHeater имеют имена вида \"XXXX.UniHeater.PPPP\", где:";
 echo ";    XXXX      - общий префикс, обычно связан с именем установки,";
 echo ";    UniHeater - имя подсистемы нагревателей (от Unified Heaters),";
 echo ";    PPPP      - имя объекта (окна, устройства, кривой, тега).";
 echo ";    Например: DEMO.INIHEATER.GATE_1";
 echo "; 2) Скопировать все файлы uniheater_*.BMP -> Bitmaps, UniHeater_*.PAS -> DaqPas,";
 echo ";    UniHeater_*.CFG -> Config, UniHeater_*.CRC -> Circuits.";
 echo "; 3) Отредактировать файл uniheater_.CFG - селектор сигналов.";
 echo ";    При помощи тегов-селекторов TNUMi,PNUMi этот модуль должен";
 echo ";    обеспечить запись каналов давления и температуры в кривые";
 echo ";    TREALi,PREALi. Используется программа UniHeater_Select.pas.";
 echo ";    Задаются также списки имен каналов TNAMES, PNAMES.";
 echo ";    Имя каждого канала не длиннее 4 символов. Список разделяется";
 echo ";    слешем, например, T1/T2/T3...";
 echo "; 4) Выходные кривые GATE_i, BLK_i надо использовать для управления";
 echo ";    нагревателями как сигнал включения и блокировки соответственно.";
 echo "; 5) При необходимости контроля по току надо обеспечить запись";
 echo ";    в тег ICTRL_i: 0=Ok, 1=Обрыв. По умолчанию не используется (всегда 0).";
 echo "; 6) При сложной логике блокировки внешняя программа может выставлять внешнюю";
 echo ";    блокировку EBLK_i, за которую эта программа должна целиком отвечать.";
 echo "; 7) Если часть каналов не используется, их при старте надо запретить";
 echo ";    записью 0 в тег HENABL_i. Запрещенная печь \"умирает\",  то есть";
 echo ";    не реагирует на нажатие кнопки включения и на блокировки.";
 echo ";    Каналы GATE_i,BLK_i можно просто не использовать.";
 echo ";    Каналы TNUM_i,PNUM_i надо завести на выходы UNIHEATER.SELECT, но входы";
 echo ";    можно не подключать.";
 echo "; 8) При необходимости внешнего управления (нагрев по графику)";
 echo ";    надо просто периодически записывать в тег TGOAL_i нужную";
 echo ";    целевую температуру.";
 echo "; 9) Переменная [&XXXX.UniHeater.CTRL] UsesSpeakSrv = 1 включает речевые сообщения о";
 echo ";    блокировке и обрыве через синтезатор &SpeakSrv, вместо звуковых файлов.";
 echo ";******************************************************************************";
 echo "; 20000830 Alexey Kuryakin";
 echo "; 20131013 Alexey Kuryakin";
 echo "; 20140825 Alexey Kuryakin";
 echo "; 20211126 Alexey Kuryakin";
 echo "; 20211215 Alexey Kuryakin";
 echo "; 20240406 Alexey Kuryakin";
 echo ";******************************************************************************";
 echo "";
};

function UniHeaterDataStorage(){
 echo ";*********************************";
 echo ";*** UniHeater: Declare Curves ***";
 echo ";*********************************";
 echo "";
 for n in $(seq 1 $1); do call UniHeaterCurves $n; done;
};
function UniHeaterCurves(){
 echo "[DataStorage] ; Declare Curves for Heater $1";
 echo "$UH_PREF.GATE_$1  = Curve 0 150 Black 15 1 ; Gate (meander) signal";
 echo "$UH_PREF.BLK_$1   = Curve 0 150 Black 15 1 ; Blocking (safety) signal";
 echo "$UH_PREF.TREAL_$1 = Curve 0 150 Black 15 1 ; Real (actual) Temperature of Heater";
 echo "$UH_PREF.PREAL_$1 = Curve 0 150 Black 15 1 ; Real (actual) Pressure of Heater";
 echo "$UH_PREF.QX_$1    = Curve 0 150 Black 15 1 ; Wanted  power [0..100%]";
 echo "$UH_PREF.QMEAN_$1 = Curve 0 150 Black 15 1 ; Average power [0..100%]";
 echo "$UH_PREF.TMEAN_$1 = Curve 0 150 Black 15 1 ; Average temperature T";
 echo "$UH_PREF.TRMSD_$1 = Curve 0 150 Black 15 1 ; Root mean square deviation of T";
 echo "[]";
 echo "";
};

function UniHeaterTagList(){
 echo ";*******************************";
 echo ";*** UniHeater: Declare Tags ***";
 echo ";*******************************";
 echo "";
 call UniHeaterCommonTags;
 for n in $(seq 0 $1); do call UniHeaterTags $n; done;
};
function UniHeaterTags(){
 if [ "$1" = "0" ]; then echo ";********** N.B. Fake Heater 0 uses to edit SETTINGS"; fi;
 echo "[TagList] ; Declare Tags for Heater $1";
 echo "$UH_PREF.HNAME_$1   = string HEAT_$1 ; Name of Heater";
 echo "$UH_PREF.TNAME_$1   = string  ?     ; Temperature sensor name (by selector)";
 echo "$UH_PREF.PNAME_$1   = string  ?     ; Pressure sensor name (by selector)";
 echo "$UH_PREF.TNUM_$1    = integer $1     ; Temperature sensor number";
 echo "$UH_PREF.PNUM_$1    = integer $1     ; Pressure    sensor number";
 echo "$UH_PREF.MODE_$1    = integer 0     ; Program operation mode (heating/cooling)";
 echo "$UH_PREF.CONF_$1    = integer 5     ; Confirmation flags: Bit0:HEAT_ON, Bit1:HEAT_OFF, Bit2:LOADINI/SAVEINI";
 echo "$UH_PREF.HENABL_$1  = integer 1     ; Enable Heater flag";
 echo "$UH_PREF.TENABL_$1  = integer 1     ; Enable safety blocking by temperature T";
 echo "$UH_PREF.PENABL_$1  = integer 1     ; Enable safety blocking by pressure P";
 echo "$UH_PREF.WENABL_$1  = integer 1     ; Enable safety blocking by watchdog timer WDT";
 echo "$UH_PREF.PERIOD_$1  = real    3000  ; Meander cycle period in milliseconds";
 echo "$UH_PREF.TGOAL_$1   = real    0     ; Goal temperature";
 echo "$UH_PREF.TREAL_$1   = real    0     ; Real (actual) Temperature of Heater";
 echo "$UH_PREF.PREAL_$1   = real    0     ; Real (actual) Pressure    of Heater";
 echo "$UH_PREF.GATE_$1    = integer 0     ; Gate (meander) signal";
 echo "$UH_PREF.BLK_$1     = integer 0     ; Blocking (safety) signal";
 echo "$UH_PREF.BT_$1      = integer 0     ; Button Heater ON/OFF state";
 echo "$UH_PREF.ST_$1      = integer 0     ; Status of Heater, 0..6";
 echo "$UH_PREF.SB_$1      = integer 0     ; Status byte (bit mask)";
 echo "$UH_PREF.QX_$1      = real    0     ; Wanted power [0..100%]";
 echo "$UH_PREF.Q1_$1      = real    50    ; Power on stage 1";
 echo "$UH_PREF.Q2_$1      = real    20    ; Power on stage 2";
 echo "$UH_PREF.Q3_$1      = real    0     ; Power on stage 3";
 echo "$UH_PREF.DT1_$1     = real    5     ; Delta (gap) for Q1/Q2";
 echo "$UH_PREF.DT2_$1     = real    2     ; Delta (gap) for Q3/0";
 echo "$UH_PREF.WDT_$1     = real    5     ; WatchDog Time in seconds";
 echo "$UH_PREF.TBLK_$1    = real    1600  ; Temperature level for blocking";
 echo "$UH_PREF.PBLK_$1    = real    1600  ; Pressure    level for blocking";
 echo "$UH_PREF.EBLK_$1    = integer 0     ; External blocking signal";
 echo "$UH_PREF.ICTRL_$1   = integer 0     ; Enable control of Current";
 echo "$UH_PREF.QMEAN_$1   = real    0     ; Average power [0..100%]";
 echo "$UH_PREF.TMEAN_$1   = real    0     ; Average temperature T";
 echo "$UH_PREF.TRMSD_$1   = real    0     ; Root mean square deviation of T";
 echo "$UH_PREF.TSTAT_$1   = real    60    ; Time, sec to calculate statistics";
 echo "[]";
 echo "";
};
function UniHeaterCommonTags(){
 echo "[TagList] ; Declare Tags for UniHeater";
 echo "$UH_PREF.CLOCK    = string  ?     ; Host clock Date-Time";
 echo "$UH_PREF.SERVID   = string  ?     ; ServerId: pid@hostname";
 echo "$UH_PREF.SOUNDS   = integer 1     ; Enable UniHeater sound";
 echo "[]";
 echo "";
};

function UniHeaterWinCtrl(){
 echo ";*********************************";
 echo ";*** UniHeater: Control Window ***";
 echo ";*********************************";
 echo "";
 echo "[Windows]";
 echo "$UH_PREF.CTRL = Circuit_Window";
 echo "[$UH_PREF.CTRL]";
 echo "Name = $UH_PREF.CTRL";
 echo "Circuit = ../circuits/${UH_PrefixFile}_uniheater_ctrl.crc";
 for n in $(seq 1 $1); do call UniHeaterWinCtrlLinkSens $n; done;
};

function UniHeaterWinCtrlLinkSens(){
 echo ";--- Heater $1 ---";
 echo "Link sensor $UH_PREF.HENABL_$1 with tag   $UH_PREF.HENABL_$1 device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.BT_$1     with tag   $UH_PREF.ST_$1     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TNUM_$1   with tag   $UH_PREF.TNUM_$1   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TNAME_$1  with tag   $UH_PREF.TNAME_$1  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TGOAL_$1  with tag   $UH_PREF.TGOAL_$1  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TREAL_$1  with curve $UH_PREF.TREAL_$1  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.GATE_$1   with curve $UH_PREF.GATE_$1   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.BLK_$1    with curve $UH_PREF.BLK_$1    device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.PROP_$1   with tag   $UH_PREF.HNAME_$1  device &$UH_PREF.CTRL";
};

function UniHeaterWinProp(){
 echo "[]";
 echo "";
 echo ";**********************************";
 echo ";*** UniHeater: Settings Window ***";
 echo ";**********************************";
 echo "[Windows]";
 echo "$UH_PREF.SETTINGS = Circuit_Window";
 echo "[$UH_PREF.SETTINGS]";
 echo "Name = $UH_PREF.SETTINGS";
 echo "Circuit = ../circuits/${UH_PrefixFile}_uniheater_settings.crc";
 echo "Link sensor $UH_PREF.HNAME_0   with tag $UH_PREF.HNAME_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.HENABL_0  with tag $UH_PREF.HENABL_0 device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.PBLK_0    with tag $UH_PREF.PBLK_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.PENABL_0  with tag $UH_PREF.PENABL_0 device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.PREAL_0   with tag $UH_PREF.PREAL_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.PNAME_0   with tag $UH_PREF.PNAME_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.PNUM_0    with tag $UH_PREF.PNUM_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.WDT_0     with tag $UH_PREF.WDT_0    device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.WENABL_0  with tag $UH_PREF.WENABL_0 device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TBLK_0    with tag $UH_PREF.TBLK_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TENABL_0  with tag $UH_PREF.TENABL_0 device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TREAL_0   with tag $UH_PREF.TREAL_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TNAME_0   with tag $UH_PREF.TNAME_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TNUM_0    with tag $UH_PREF.TNUM_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.Q1_0      with tag $UH_PREF.Q1_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.Q2_0      with tag $UH_PREF.Q2_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.Q3_0      with tag $UH_PREF.Q3_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.DT1_0     with tag $UH_PREF.DT1_0    device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.DT2_0     with tag $UH_PREF.DT2_0    device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TGOAL_0   with tag $UH_PREF.TGOAL_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.PERIOD_0  with tag $UH_PREF.PERIOD_0 device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SOUNDS    with tag $UH_PREF.SOUNDS   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SERVID    with tag $UH_PREF.SERVID   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.CLOCK     with tag $UH_PREF.CLOCK    device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.ACCEPT    with                        device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.CANCEL    with                        device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SAVEINI   with                        device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.LOADINI   with                        device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.MANUAL    with                        device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.CLOSE     with                        device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.MAKE.LM9  with                        device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.MODE_0    with tag $UH_PREF.MODE_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.MODE_N    with tag $UH_PREF.MODE_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.CONF_0_0  with tag $UH_PREF.CONF_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.CONF_0_1  with tag $UH_PREF.CONF_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.CONF_0_2  with tag $UH_PREF.CONF_0   device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.QX_0      with tag $UH_PREF.QX_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.QMEAN_0   with tag $UH_PREF.QMEAN_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TMEAN_0   with tag $UH_PREF.TMEAN_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TRMSD_0   with tag $UH_PREF.TRMSD_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.TSTAT_0   with tag $UH_PREF.TSTAT_0  device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_0    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_1    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_2    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_3    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_4    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_5    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_6    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_7    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_8    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_9    with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_10   with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "Link sensor $UH_PREF.SB_0_11   with tag $UH_PREF.SB_0     device &$UH_PREF.CTRL";
 echo "[]";
 echo "";
};

function UniHeaterWinPlot(){
 echo ";************************************";
 echo ";*** UniHeater: Data Plot Windows ***";
 echo ";************************************";
 echo "";
 echo "[Windows]";
 echo "$UH_PREF.T.PLOT = Curve_Window";
 echo "[$UH_PREF.T.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CTemperature^N^L___{°C}, 0, 1000";
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.TREAL_$i"; done;
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.TMEAN_$i"; done;
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.TRMSD_$i"; done;
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$UH_PREF.P.PLOT = Curve_Window";
 echo "[$UH_PREF.P.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CPressure^N^L___{Bar}, 0, 2500";
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.PREAL_$i"; done;
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$UH_PREF.Q.PLOT = Curve_Window";
 echo "[$UH_PREF.Q.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CQX_-_Quantity_of_Power^N^L___{%}, 0, 100";
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.QX_$i"; done;
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.QMEAN_$i"; done;
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$UH_PREF.STATE.PLOT = Curve_Window";
 echo "[$UH_PREF.STATE.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CState_of_GATE_and_BLK^N^L___{State}, 0, 1";
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.GATE_$i"; done;
 for i in $(seq 1 $1); do echo "CurveList = $UH_PREF.BLK_$i"; done;
 echo "[]";
 echo "";
};


function UniHeaterCtrl(){
 let "nai=$1+1";
 let "ndo=$1+1";
 let "nao=$nai+$1+$1+$1";
 let "naf=$nao*32+32";
 let "ndf=$ndo*32+32";
 let "nif=$1*4+32";
 let "nof=$1*4+32";
 echo ";***************************************";
 echo ";*** UniHeater: Main control program ***";
 echo ";***************************************";
 echo "[DeviceList]";
 echo "&$UH_PREF.CTRL = device software program";
 echo "[&$UH_PREF.CTRL]";
 echo "Comment         = Стабилизатор температуры нагревателей UniHeater.";
 echo "InquiryPeriod   = 1";
 echo "DevicePolling   = 10, tpTimeCritical";
 echo "ProgramSource   = $UH_HomeRef/uniheater_ctrl.pas";
 echo "UniHeaterManual = $UH_HomeRef/uniheater.htm";
 echo "DebugFlags      = 3";
 echo "OpenConsole     = 2";
 echo "StartupScript   = [&$UH_PREF.CTRL.StartupScript]";
 echo "FinallyScript   = [&$UH_PREF.CTRL.FinallyScript]";
 echo "StdInFifo       = $nif";
 echo "StdOutFifo      = $nof";
 echo "AnalogFifo      = $naf";
 echo "DigitalFifo     = $ndf";
 echo "EncryptMethod   = $UH_EncryptMethod";
 echo "TrustedUsers    = [TrustedUsers]";
 echo "DIM_GuiClickTag = $UH_PREF.DIMGUICLICK";
 echo "AnalogInputs    = $nai  ; Temperatures";
 for i in $(seq 1 $1); do call UniHeaterCtrlAI_TREAL $i; done;
 echo "DigitalInputs   = $nai  ; Pressures";
 for i in $(seq 1 $1); do call UniHeaterCtrlDI_PREAL $i; done;
 echo "AnalogOutputs   = $nao  ; Wanted Power and Average power";
 for i in $(seq 1 $1); do call UniHeaterCtrlAO_QX $i; done;
 for i in $(seq 1 $1); do call UniHeaterCtrlAO_QMEAN $i $1; done;
 for i in $(seq 1 $1); do call UniHeaterCtrlAO_TMEAN $i $1; done;
 for i in $(seq 1 $1); do call UniHeaterCtrlAO_TRMSD $i $1; done;
 echo "DigitalOutputs  = $ndo  ; Blocking signals";
 for i in $(seq 1 $1); do call UniHeaterCtrlDO_BLK $i; done;
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [$UH_PREF.CustomParameters.TagList]";
 echo "CustomIniFileRef  = ../data/$UH_FILE.ini";
 echo "CustomIniSection  = [$UH_PREF.CustomParameters]";
 echo "CustomIniBackups  = ../data/custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "";
 echo "[ConfigFileList]";
 echo "ConfigFile = ../data/$UH_FILE.ini";
 echo "[]";
 echo "";
};
function UniHeaterCtrlAI_TREAL(){
 echo "Link AnalogInput $1 with curve $UH_PREF.TREAL_$1";
};
function UniHeaterCtrlDI_PREAL(){
 echo "Link DigitalInput $1 with curve $UH_PREF.PREAL_$1  bit 0";
};
function UniHeaterCtrlAO_QX(){
 echo "Link AnalogOutput $1 with curve $UH_PREF.QX_$1 history $UH_HistoryLen";
};
function UniHeaterCtrlAO_QMEAN(){
 let "ao=$1+$2";
 echo "Link AnalogOutput $ao with curve $UH_PREF.QMEAN_$1 history $UH_HistoryLen";
};
function UniHeaterCtrlAO_TMEAN(){
 let "ao=$1+$2+$2";
 echo "Link AnalogOutput $ao with curve $UH_PREF.TMEAN_$1 history $UH_HistoryLen";
};
function UniHeaterCtrlAO_TRMSD(){
 let "ao=$1+$2+$2+$2";
 echo "Link AnalogOutput $ao with curve $UH_PREF.TRMSD_$1 history $UH_HistoryLen";
};
function UniHeaterCtrlDO_BLK(){
 echo "Link DigitalOutput $1 with curve $UH_PREF.BLK_$1 history $UH_HistoryLen";
};



function UniHeaterCustomParam(){
 echo "[$UH_PREF.CustomParameters.TagList]";
 for n in $(seq 1 $1); do call UniHeaterCustomParams $n; done;
 echo "[]";
 echo "";
};
function UniHeaterCustomParams(){
 echo ";--- Heater $1, custom parameters ---";
 echo "TagList = $UH_PREF.HNAME_$1";
 echo "TagList = $UH_PREF.MODE_$1";
 echo "TagList = $UH_PREF.CONF_$1";
 echo "TagList = $UH_PREF.HENABL_$1";
 echo "TagList = $UH_PREF.BT_$1";
 echo "TagList = $UH_PREF.TGOAL_$1";
 echo "TagList = $UH_PREF.TNUM_$1";
 echo "TagList = $UH_PREF.TENABL_$1";
 echo "TagList = $UH_PREF.TBLK_$1";
 echo "TagList = $UH_PREF.PENABL_$1";
 echo "TagList = $UH_PREF.PNUM_$1";
 echo "TagList = $UH_PREF.WENABL_$1";
 echo "TagList = $UH_PREF.WDT_$1";
 echo "TagList = $UH_PREF.DT1_$1";
 echo "TagList = $UH_PREF.DT2_$1";
 echo "TagList = $UH_PREF.PBLK_$1";
 echo "TagList = $UH_PREF.PERIOD_$1";
 echo "TagList = $UH_PREF.Q1_$1";
 echo "TagList = $UH_PREF.Q2_$1";
 echo "TagList = $UH_PREF.Q3_$1";
 echo "TagList = $UH_PREF.TSTAT_$1";
};



function UniHeaterStartupScript(){
 let "ncols=$2"                      ; ### Number of columns
 let "nrows= ( $1 + $2 - 1 ) / $2"   ; ### Number of rows
 let "cx=$ncols * 125 + 15"          ; ### Window size on X axis
 let "cy=$nrows * 119 + 57"          ; ### Window size on Y axis
 echo "[&$UH_PREF.CTRL.StartupScript]";
 echo "@WinShow   $UH_PREF.CTRL";
 echo "@WinDraw   $UH_PREF.CTRL|Top=0|Left=0|Width=$cx|Height=$cy";
 echo "@WinDraw   $UH_PREF.CTRL|Options=-Min,-Max,+Close,+Top,+Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo "@WinSelect $UH_PREF.CTRL";
 echo "@WinHide   $UH_PREF.CTRL";
 echo "@WinShow   $UH_PREF.SETTINGS";
 echo "@WinDraw   $UH_PREF.SETTINGS|Top=170|Left=167|Width=950|Height=420";
 echo "@WinDraw   $UH_PREF.SETTINGS|Options=-Min,-Max,-Close,+Top,+Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo "@WinSelect $UH_PREF.SETTINGS";
 echo "@WinHide   $UH_PREF.SETTINGS";
 echo "@WinShow   $UH_PREF.T.PLOT";
 echo "@WinDraw   $UH_PREF.T.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@WinDraw   $UH_PREF.T.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@WinHide   $UH_PREF.T.PLOT";
 echo "@WinShow   $UH_PREF.P.PLOT";
 echo "@WinDraw   $UH_PREF.P.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@WinDraw   $UH_PREF.P.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@WinHide   $UH_PREF.P.PLOT";
 echo "@WinShow   $UH_PREF.Q.PLOT";
 echo "@WinDraw   $UH_PREF.Q.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@WinDraw   $UH_PREF.Q.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@WinHide   $UH_PREF.Q.PLOT";
 echo "@WinShow   $UH_PREF.STATE.PLOT";
 echo "@WinDraw   $UH_PREF.STATE.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@WinDraw   $UH_PREF.STATE.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@WinHide   $UH_PREF.STATE.PLOT";
 echo "[]";
 echo "";
 echo "[&$UH_PREF.CTRL.FinallyScript]";
 echo "@WinHide $UH_PREF.CTRL";
 echo "@WinHide $UH_PREF.SETTINGS";
 echo "@WinHide $UH_PREF.T.PLOT";
 echo "@WinHide $UH_PREF.P.PLOT";
 echo "@WinHide $UH_PREF.Q.PLOT";
 echo "@WinHide $UH_PREF.STATE.PLOT";
 echo "[]";
 echo "";
 echo ";*******************************************************";
 echo ";*** UniHeater: CRON jobs to draw UniHeater windows  ***";
 echo ";*******************************************************";
 echo "[&CronSrv.StartupScript]";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.CTRL.HOME 0 0 0";
 echo "@cron.job $UH_PREF.CTRL.HOME @WinShow $UH_PREF.CTRL";
 echo "@cron.job $UH_PREF.CTRL.HOME @WinDraw $UH_PREF.CTRL|Top=0|Left=0|Width=$cx|Height=$cy";
 echo "@cron.job $UH_PREF.CTRL.HOME @WinDraw $UH_PREF.CTRL|Options=-Min,-Max,+Close,-Top,-Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.CTRL.SHOW 0 0 0";
 echo "@cron.job $UH_PREF.CTRL.SHOW @WinShow $UH_PREF.CTRL";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.CTRL.HIDE 0 0 0";
 echo "@cron.job $UH_PREF.CTRL.HIDE @WinHide $UH_PREF.CTRL";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.CTRL.SELECT 0 0 0";
 echo "@cron.job $UH_PREF.CTRL.SELECT @WinSelect $UH_PREF.CTRL";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.SETTINGS.HOME 0 0 0";
 echo "@cron.job $UH_PREF.SETTINGS.HOME @WinShow $UH_PREF.SETTINGS";
 echo "@cron.job $UH_PREF.SETTINGS.HOME @WinDraw $UH_PREF.SETTINGS|Top=170|Left=167|Width=950|Height=420";
 echo "@cron.job $UH_PREF.SETTINGS.HOME @WinDraw $UH_PREF.SETTINGS|Options=-Min,-Max,-Close,-Top,-Left,-Width,-Height,-HScroll,-VScroll,+StatusBar";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.SETTINGS.SHOW 0 0 0";
 echo "@cron.job $UH_PREF.SETTINGS.SHOW @WinShow $UH_PREF.SETTINGS";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.SETTINGS.HIDE 0 0 0";
 echo "@cron.job $UH_PREF.SETTINGS.HIDE @WinHide $UH_PREF.SETTINGS";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.SETTINGS.SELECT 0 0 0";
 echo "@cron.job $UH_PREF.SETTINGS.SELECT @WinSelect $UH_PREF.SETTINGS";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.T.PLOT.HOME 0 0 0";
 echo "@cron.job $UH_PREF.T.PLOT.HOME @WinShow $UH_PREF.T.PLOT";
 echo "@cron.job $UH_PREF.T.PLOT.HOME @WinDraw $UH_PREF.T.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@cron.job $UH_PREF.T.PLOT.HOME @WinDraw $UH_PREF.T.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@cron.tab $UH_PREF.T.PLOT.SHOW 0 0 0";
 echo "@cron.job $UH_PREF.T.PLOT.SHOW @WinShow $UH_PREF.T.PLOT";
 echo "@cron.tab $UH_PREF.T.PLOT.HIDE 0 0 0";
 echo "@cron.job $UH_PREF.T.PLOT.HIDE @WinHide $UH_PREF.T.PLOT";
 echo "@cron.tab $UH_PREF.T.PLOT.SELECT 0 0 0";
 echo "@cron.job $UH_PREF.T.PLOT.SELECT @WinSelect $UH_PREF.T.PLOT";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.P.PLOT.HOME 0 0 0";
 echo "@cron.job $UH_PREF.P.PLOT.HOME @WinShow $UH_PREF.P.PLOT";
 echo "@cron.job $UH_PREF.P.PLOT.HOME @WinDraw $UH_PREF.P.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@cron.job $UH_PREF.P.PLOT.HOME @WinDraw $UH_PREF.P.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@cron.tab $UH_PREF.P.PLOT.SHOW 0 0 0";
 echo "@cron.job $UH_PREF.P.PLOT.SHOW @WinShow $UH_PREF.P.PLOT";
 echo "@cron.tab $UH_PREF.P.PLOT.HIDE 0 0 0";
 echo "@cron.job $UH_PREF.P.PLOT.HIDE @WinHide $UH_PREF.P.PLOT";
 echo "@cron.tab $UH_PREF.P.PLOT.SELECT 0 0 0";
 echo "@cron.job $UH_PREF.P.PLOT.SELECT @WinSelect $UH_PREF.P.PLOT";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.Q.PLOT.HOME 0 0 0";
 echo "@cron.job $UH_PREF.Q.PLOT.HOME @WinShow $UH_PREF.Q.PLOT";
 echo "@cron.job $UH_PREF.Q.PLOT.HOME @WinDraw $UH_PREF.Q.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@cron.job $UH_PREF.Q.PLOT.HOME @WinDraw $UH_PREF.Q.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@cron.tab $UH_PREF.Q.PLOT.SHOW 0 0 0";
 echo "@cron.job $UH_PREF.Q.PLOT.SHOW @WinShow $UH_PREF.Q.PLOT";
 echo "@cron.tab $UH_PREF.Q.PLOT.HIDE 0 0 0";
 echo "@cron.job $UH_PREF.Q.PLOT.HIDE @WinHide $UH_PREF.Q.PLOT";
 echo "@cron.tab $UH_PREF.Q.PLOT.SELECT 0 0 0";
 echo "@cron.job $UH_PREF.Q.PLOT.SELECT @WinSelect $UH_PREF.Q.PLOT";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.STATE.PLOT.HOME 0 0 0";
 echo "@cron.job $UH_PREF.STATE.PLOT.HOME @WinShow $UH_PREF.STATE.PLOT";
 echo "@cron.job $UH_PREF.STATE.PLOT.HOME @WinDraw $UH_PREF.STATE.PLOT|Top=317|Left=167|Width=1070|Height=483";
 echo "@cron.job $UH_PREF.STATE.PLOT.HOME @WinDraw $UH_PREF.STATE.PLOT|Options=-Min,-Max,+Close,+Top,+Left,+Width,+Height,+VScroll,+StatusBar";
 echo "@cron.tab $UH_PREF.STATE.PLOT.SHOW 0 0 0";
 echo "@cron.job $UH_PREF.STATE.PLOT.SHOW @WinShow $UH_PREF.STATE.PLOT";
 echo "@cron.tab $UH_PREF.STATE.PLOT.HIDE 0 0 0";
 echo "@cron.job $UH_PREF.STATE.PLOT.HIDE @WinHide $UH_PREF.STATE.PLOT";
 echo "@cron.tab $UH_PREF.STATE.PLOT.SELECT 0 0 0";
 echo "@cron.job $UH_PREF.STATE.PLOT.SELECT @WinSelect $UH_PREF.STATE.PLOT";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.ALL.HOME 0 0 0";
 echo "@cron.job $UH_PREF.ALL.HOME @cron.run $UH_PREF.CTRL.HOME";
 echo "@cron.job $UH_PREF.ALL.HOME @cron.run $UH_PREF.SETTINGS.HOME";
 echo "@cron.job $UH_PREF.ALL.HOME @cron.run $UH_PREF.T.PLOT.HOME";
 echo "@cron.job $UH_PREF.ALL.HOME @cron.run $UH_PREF.P.PLOT.HOME";
 echo "@cron.job $UH_PREF.ALL.HOME @cron.run $UH_PREF.Q.PLOT.HOME";
 echo "@cron.job $UH_PREF.ALL.HOME @cron.run $UH_PREF.STATE.PLOT.HOME";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.ALL.SHOW 0 0 0";
 echo "@cron.job $UH_PREF.ALL.SHOW @cron.run $UH_PREF.CTRL.SHOW";
 echo "@cron.job $UH_PREF.ALL.SHOW @cron.run $UH_PREF.SETTINGS.SHOW";
 echo "@cron.job $UH_PREF.ALL.SHOW @cron.run $UH_PREF.T.PLOT.SHOW";
 echo "@cron.job $UH_PREF.ALL.SHOW @cron.run $UH_PREF.P.PLOT.SHOW";
 echo "@cron.job $UH_PREF.ALL.SHOW @cron.run $UH_PREF.Q.PLOT.SHOW";
 echo "@cron.job $UH_PREF.ALL.SHOW @cron.run $UH_PREF.STATE.PLOT.SHOW";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.ALL.HIDE 0 0 0";
 echo "@cron.job $UH_PREF.ALL.HIDE @cron.run $UH_PREF.CTRL.HIDE";
 echo "@cron.job $UH_PREF.ALL.HIDE @cron.run $UH_PREF.SETTINGS.HIDE";
 echo "@cron.job $UH_PREF.ALL.HIDE @cron.run $UH_PREF.T.PLOT.HIDE";
 echo "@cron.job $UH_PREF.ALL.HIDE @cron.run $UH_PREF.P.PLOT.HIDE";
 echo "@cron.job $UH_PREF.ALL.HIDE @cron.run $UH_PREF.Q.PLOT.HIDE";
 echo "@cron.job $UH_PREF.ALL.HIDE @cron.run $UH_PREF.STATE.PLOT.HIDE";
 echo ";---------------------------------------------------------------------";
 echo "@cron.tab $UH_PREF.ALL.SELECT 0 0 0";
 echo "@cron.job $UH_PREF.ALL.SELECT @cron.run $UH_PREF.CTRL.SELECT";
 echo "@cron.job $UH_PREF.ALL.SELECT @cron.run $UH_PREF.SETTINGS.SELECT";
 echo "@cron.job $UH_PREF.ALL.SELECT @cron.run $UH_PREF.T.PLOT.SELECT";
 echo "@cron.job $UH_PREF.ALL.SELECT @cron.run $UH_PREF.P.PLOT.SELECT";
 echo "@cron.job $UH_PREF.ALL.SELECT @cron.run $UH_PREF.Q.PLOT.SELECT";
 echo "@cron.job $UH_PREF.ALL.SELECT @cron.run $UH_PREF.STATE.PLOT.SELECT";
 echo ";---------------------------------------------------------------------";
 echo "@async @cron.run $UH_PREF.ALL.HOME";
 echo "@async @cron.run $UH_PREF.ALL.HIDE";
 echo ";---------------------------------------------------------------------";
 echo "[]";
 echo "";
};



function UniHeaterPulser(){
 let "nai=$1+1";
 let "ndo=$1+1";
 let "ndf=$ndo*32+32";
 echo ";*******************************************************";
 echo ";*** Генератор скважности для нагревателей UniHeater ***";
 echo ";*******************************************************";
 echo "[DeviceList]";
 echo "&$UH_PREF.PULSER = device software program";
 echo "[&$UH_PREF.PULSER]";
 echo "Comment        = Генератор импульсов скважности для нагревателей UniHeater.";
 echo "InquiryPeriod  = 0";
 echo "DevicePolling  = 4, tpTimeCritical";
 echo "ProgramSource  = $UH_HomeRef/uniheater_pulser.pas";
 echo "DigitalFifo    = $ndf";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "AnalogInputs   = $nai ; Duty factor to control Heaters";
 for n in $(seq 1 $1); do call UniHeaterPulserAI $n; done;
 echo "DigitalOutputs = $ndo ; Heaters ON/OFF signals";
 for n in $(seq 1 $1); do call UniHeaterPulserDO $n; done;
 echo "[]";
 echo "";
};
function UniHeaterPulserAI(){
 echo "Link AnalogInput   $1  with curve $UH_PREF.QX_$1";
};
function UniHeaterPulserDO(){
 echo "Link DigitalOutput $1  with curve $UH_PREF.GATE_$1  history $UH_HistoryLen";
};



function UniHeaterSelect(){
 let "nao=$1+1";
 let "naf=$nao*32+32";
 echo ";******************************************************************";
 echo ";*** Селектор каналов температуры для подсистемы UniHeater.     ***";
 echo ";*** Задаются кривые AnalogInputs и TNAMES - список имен для    ***";
 echo ";*** датчиков T, без пробелов, разделенных слэшем: T1/T2/...    ***";
 echo ";******************************************************************";
 echo "[DeviceList]";
 echo "&$UH_PREF.SELECT_T = device software program";
 echo "[&$UH_PREF.SELECT_T]";
 echo "Comment        = Селектор датчиков температуры для UniHeater.";
 echo "InquiryPeriod  = 1";
 echo "DevicePolling  = 10, tpTimeCritical";
 echo "ProgramSource  = $UH_HomeRef/uniheater_select.pas";
 echo "SelectorPrefix = $UH_PREF.TNUM_";
 echo "AnalogFifo     = $naf";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "AnalogOutputs  = $nao";
 for n in $(seq 1 $1); do call UniHeaterSelectTAO $n; done;
 echo "[]";
 echo "";
 echo ";******************************************************************";
 echo ";*** Селектор каналов давлений для подсистемы UniHeater.        ***";
 echo ";*** Задаются кривые AnalogInputs и PNAMES - список имен для    ***";
 echo ";*** датчиков P, без пробелов, разделенных слэшем: P1/P2/...    ***";
 echo ";******************************************************************";
 echo "[DeviceList]";
 echo "&$UH_PREF.SELECT_P = device software program";
 echo "[&$UH_PREF.SELECT_P]";
 echo "Comment        = Селектор датчиков давления для UniHeater.";
 echo "InquiryPeriod  = 1";
 echo "DevicePolling  = 10, tpTimeCritical";
 echo "ProgramSource  = $UH_HomeRef/uniheater_select.pas";
 echo "SelectorPrefix = $UH_PREF.PNUM_";
 echo "AnalogFifo     = $naf";
 echo "DebugFlags     = 3";
 echo "OpenConsole    = 2";
 echo "AnalogOutputs  = $nao";
 for n in $(seq 1 $1); do call UniHeaterSelectPAO $n; done;
 echo "[]";
 echo "";
};
function UniHeaterSelectTAO(){
 echo "Link AnalogOutput  $1  with curve $UH_PREF.TREAL_$1  history $UH_HistoryLen";
};
function UniHeaterSelectPAO(){
 echo "Link AnalogOutput  $1  with curve $UH_PREF.PREAL_$1  history $UH_HistoryLen";
};



function UniHeaterBinding(){
 echo ";******************************************************************";
 echo ";*** Подключение селекторов каналов датчиков T,P для UniHeater. ***";
 echo ";******************************************************************";
 echo "[ConfigFileList] ; Подключение конфигурации селекторов UniHeater";
 echo "ConfigFile = ${UH_PrefixFile}_uniheater_bind.cfg";
 echo "[]";
 echo "";
};

function UniHeaterSelector(){
 let "ntai=$UH_NumberTSen+1";
 let "npai=$UH_NumberPSen+1";
 echo ";******************************************************************";
 echo ";*** Селектор каналов датчиков T,P для подсистемы UniHeater.    ***";
 echo ";*** Этот файл настраивается индивидуально для каждой системы,  ***";
 echo ";*** т.к. кривые источников данных (датчиков T,P) меняются.     ***";
 echo ";******************************************************************";
 echo "";
 echo ";******************************************************************";
 echo ";*** Селектор каналов температуры для подсистемы UniHeater.     ***";
 echo ";*** Задаются кривые AnalogInputs и TNAMES - список имен для    ***";
 echo ";*** датчиков T, без пробелов, разделенных слэшем: T1/T2/...    ***";
 echo ";******************************************************************";
 echo "[&$UH_PREF.SELECT_T]";
 echo "UsesSmoothing  = 1";
 echo "AnalogInputs   = $ntai";
 export tnames="${UH_NickTSen}1";
 for n in $(seq 2 $UH_NumberTSen); do tnames="$tnames/$UH_NickTSen$n"; done;
 for n in $(seq 1 $UH_NumberTSen); do call UniHeaterComSelectTAI $n; done;
 echo "[]";
 echo "[TagList]";
 echo "$UH_PREF.TNAMES = string $tnames";
 echo "[]";
 echo "";
 echo ";******************************************************************";
 echo ";*** Селектор каналов давлений для подсистемы UniHeater.        ***";
 echo ";*** Задаются кривые AnalogInputs и PNAMES - список имен для    ***";
 echo ";*** датчиков P, без пробелов, разделенных слэшем: P1/P2/...    ***";
 echo ";******************************************************************";
 echo "[&$UH_PREF.SELECT_P]";
 echo "UsesSmoothing  = 1";
 echo "AnalogInputs   = $npai";
 export pnames="${UH_NickPSen}1";
 for n in $(seq 2 $UH_NumberPSen); do pnames="$pnames/$UH_NickPSen$n"; done;
 for n in $(seq 1 $UH_NumberPSen); do call UniHeaterComSelectPAI $n; done;
 echo "[]";
 echo "[TagList]";
 echo "$UH_PREF.PNAMES = string $pnames";
 echo "[]";
};
function UniHeaterComSelectTAI(){
 echo "Link AnalogInput $1 with curve $UH_PrefixTSen$1 smoothing $UH_SmoothingWw $UH_SmoothingPw $UH_SmoothingK1 $UH_SmoothingK2";
};
function UniHeaterComSelectPAI(){
 echo "Link AnalogInput $1 with curve $UH_PrefixPSen$1 smoothing $UH_SmoothingWw $UH_SmoothingPw $UH_SmoothingK2 $UH_SmoothingK2";
};

function UniHeaterDimGuiClick(){
 export devmsg="devPostMsg";
 echo "[TagList]";
 echo "$UH_PREF.DIMGUICLICK string ? ; Uses to Handle DIM GUI Clicks";
 echo "[]";
};

function UniHeaterDimDis(){
 call UniHeaterDimGuiClick;
 unix dimcfg \
  -n section "[&$UH_PREF.CTRL]" \
  -n print    DimServerMode = 1 \
  -n end \
  -n dis_cmnd   $UH_PATH/DIMGUICLICK \
  -n tag        $UH_PREF.DIMGUICLICK \
  -n $devmsg "&$UH_PREF.CTRL @DIMGUICLICK=%**" \
  -n end \
  -n dic_cmnd   $UH_PATH/DIMGUICLICK \
  -n tag        $UH_PREF.DIMGUICLICK \
  -n end \
  -n dis_info   $UH_PATH/CLOCK \
  -n tag        $UH_PREF.CLOCK \
  -n end \
  -n dis_info   $UH_PATH/SERVID \
  -n tag        $UH_PREF.SERVID \
  -n end \
  -n dis_info   $UH_PATH/TNAMES \
  -n tag        $UH_PREF.TNAMES \
  -n end \
  -n dis_info   $UH_PATH/PNAMES \
  -n tag        $UH_PREF.PNAMES \
  -n end \
  -n dis_info   $UH_PATH/SOUNDS \
  -n tag        $UH_PREF.SOUNDS \
  -n end \
  -n ;
  for i in $(seq 1 $UH_NumHeaters); do call UniHeaterDimDisLoop $i; done;
};

function UniHeaterDimDisLoop(){
 unix dimcfg \
  -n dis_info   $UH_PATH/HENABL+HNAME_$1 \
  -n tag        $UH_PREF.HENABL_$1 \
  -n tag        $UH_PREF.HNAME_$1 \
  -n end \
  -n dis_info   $UH_PATH/TENABL+TBLK+TNUM+TNAME_$1 \
  -n tag        $UH_PREF.TENABL_$1 \
  -n tag        $UH_PREF.TBLK_$1 \
  -n tag        $UH_PREF.TNUM_$1 \
  -n tag        $UH_PREF.TNAME_$1 \
  -n end \
  -n dis_info   $UH_PATH/PENABL+PBLK+PNUM+PNAME_$1 \
  -n tag        $UH_PREF.PENABL_$1 \
  -n tag        $UH_PREF.PBLK_$1 \
  -n tag        $UH_PREF.PNUM_$1 \
  -n tag        $UH_PREF.PNAME_$1 \
  -n end \
  -n dis_info   $UH_PATH/WENABL+WDT_$1 \
  -n tag        $UH_PREF.WENABL_$1 \
  -n tag        $UH_PREF.WDT_$1 \
  -n end \
  -n dis_info   $UH_PATH/TGOAL+MODE+PERIOD+Q1+Q2+Q3+DT1+DT2_$1 \
  -n tag        $UH_PREF.TGOAL_$1 \
  -n tag        $UH_PREF.MODE_$1 \
  -n tag        $UH_PREF.PERIOD_$1 \
  -n tag        $UH_PREF.Q1_$1 \
  -n tag        $UH_PREF.Q2_$1 \
  -n tag        $UH_PREF.Q3_$1 \
  -n tag        $UH_PREF.DT1_$1 \
  -n tag        $UH_PREF.DT2_$1 \
  -n end \
  -n dis_info   $UH_PATH/CONF_$1 \
  -n tag        $UH_PREF.CONF_$1 \
  -n end \
  -n dis_info   $UH_PATH/TREAL_$1 \
  -n tag        $UH_PREF.TREAL_$1 \
  -n end \
  -n dis_info   $UH_PATH/PREAL_$1 \
  -n tag        $UH_PREF.PREAL_$1 \
  -n end \
  -n dis_info   $UH_PATH/BT+ST+SB+BLK+QX_$1 \
  -n tag        $UH_PREF.BT_$1 \
  -n tag        $UH_PREF.ST_$1 \
  -n tag        $UH_PREF.SB_$1 \
  -n tag        $UH_PREF.BLK_$1 \
  -n tag        $UH_PREF.QX_$1 \
  -n end \
  -n dis_info   $UH_PATH/GATE_$1 \
  -n tag        $UH_PREF.GATE_$1 \
  -n end \
  -n dis_info   $UH_PATH/EBLK+ICTRL_$1 \
  -n tag        $UH_PREF.EBLK_$1 \
  -n tag        $UH_PREF.ICTRL_$1 \
  -n end \
  -n dis_info   $UH_PATH/TSTAT+QMEAN+TMEAN+TRMSD_$1 \
  -n tag        $UH_PREF.TSTAT_$1 \
  -n tag        $UH_PREF.QMEAN_$1 \
  -n tag        $UH_PREF.TMEAN_$1 \
  -n tag        $UH_PREF.TRMSD_$1 \
  -n end \
  -n ;
};

function UniHeaterDimDic(){
 call UniHeaterDimGuiClick;
 unix dimcfg \
  -n section "[&$UH_PREF.CTRL]" \
  -n print    DimClientMode = 1 \
  -n end \
  -n dic_cmnd   $UH_PATH/DIMGUICLICK \
  -n tag        $UH_PREF.DIMGUICLICK \
  -n end \
  -n dic_info   $UH_PATH/CLOCK \
  -n tag        $UH_PREF.CLOCK \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.CLOCK" \
  -n end \
  -n dic_info   $UH_PATH/SERVID \
  -n tag        $UH_PREF.SERVID \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.SERVID" \
  -n end \
  -n dic_info   $UH_PATH/TNAMES \
  -n tag        $UH_PREF.TNAMES \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TNAMES" \
  -n end \
  -n dic_info   $UH_PATH/PNAMES \
  -n tag        $UH_PREF.PNAMES \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.PNAMES" \
  -n end \
  -n dic_info   $UH_PATH/SOUNDS \
  -n tag        $UH_PREF.SOUNDS \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.SOUNDS" \
  -n end \
  -n ;
 for i in $(seq 1 $UH_NumHeaters); do call UniHeaterDimDicLoop $i; done;
 call UniHeaterDimDicRecv;
};

function UniHeaterDimDicLoop(){
 unix dimcfg \
  -n dic_info   $UH_PATH/HENABL+HNAME_$1 \
  -n tag        $UH_PREF.HENABL_$1 \
  -n tag        $UH_PREF.HNAME_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.HENABL_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.HNAME_$1" \
  -n end \
  -n dic_info   $UH_PATH/TENABL+TBLK+TNUM+TNAME_$1 \
  -n tag        $UH_PREF.TENABL_$1 \
  -n tag        $UH_PREF.TBLK_$1 \
  -n tag        $UH_PREF.TNUM_$1 \
  -n tag        $UH_PREF.TNAME_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TENABL_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TBLK_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TNUM_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TNAME_$1" \
  -n end \
  -n dic_info   $UH_PATH/PENABL+PBLK+PNUM+PNAME_$1 \
  -n tag        $UH_PREF.PENABL_$1 \
  -n tag        $UH_PREF.PBLK_$1 \
  -n tag        $UH_PREF.PNUM_$1 \
  -n tag        $UH_PREF.PNAME_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.PENABL_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.PBLK_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.PNUM_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.PNAME_$1" \
  -n end \
  -n dic_info   $UH_PATH/WENABL+WDT_$1 \
  -n tag        $UH_PREF.WENABL_$1 \
  -n tag        $UH_PREF.WDT_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.WENABL_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.WDT_$1" \
  -n end \
  -n dic_info   $UH_PATH/TGOAL+MODE+PERIOD+Q1+Q2+Q3+DT1+DT2_$1 \
  -n tag        $UH_PREF.TGOAL_$1 \
  -n tag        $UH_PREF.MODE_$1 \
  -n tag        $UH_PREF.PERIOD_$1 \
  -n tag        $UH_PREF.Q1_$1 \
  -n tag        $UH_PREF.Q2_$1 \
  -n tag        $UH_PREF.Q3_$1 \
  -n tag        $UH_PREF.DT1_$1 \
  -n tag        $UH_PREF.DT2_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TGOAL_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.MODE_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.PERIOD_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.Q1_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.Q2_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.Q3_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.DT1_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.DT2_$1" \
  -n end \
  -n dic_info   $UH_PATH/CONF_$1 \
  -n tag        $UH_PREF.CONF_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.CONF_$1" \
  -n end \
  -n dic_info   $UH_PATH/TREAL_$1 \
  -n tag        $UH_PREF.TREAL_$1 \
  -n $devmsg "&$UH_PREF.RECV @DimTagUpdate=$UH_PREF.TREAL_$1" \
  -n end \
  -n dic_info   $UH_PATH/PREAL_$1 \
  -n tag        $UH_PREF.PREAL_$1 \
  -n $devmsg "&$UH_PREF.RECV @DimTagUpdate=$UH_PREF.PREAL_$1" \
  -n end \
  -n dic_info   $UH_PATH/BT+ST+SB+BLK+QX_$1 \
  -n tag        $UH_PREF.BT_$1 \
  -n tag        $UH_PREF.ST_$1 \
  -n tag        $UH_PREF.SB_$1 \
  -n tag        $UH_PREF.BLK_$1 \
  -n tag        $UH_PREF.QX_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.BT_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.ST_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.SB_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.BLK_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.QX_$1" \
  -n end \
  -n dic_info   $UH_PATH/GATE_$1 \
  -n tag        $UH_PREF.GATE_$1 \
  -n $devmsg "&$UH_PREF.RECV @DimTagUpdate=$UH_PREF.GATE_$1" \
  -n end \
  -n dic_info   $UH_PATH/EBLK+ICTRL_$1 \
  -n tag        $UH_PREF.EBLK_$1 \
  -n tag        $UH_PREF.ICTRL_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.EBLK_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.ICTRL_$1" \
  -n end \
  -n dic_info   $UH_PATH/TSTAT+QMEAN+TMEAN+TRMSD_$1 \
  -n tag        $UH_PREF.TSTAT_$1 \
  -n tag        $UH_PREF.QMEAN_$1 \
  -n tag        $UH_PREF.TMEAN_$1 \
  -n tag        $UH_PREF.TRMSD_$1 \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TSTAT_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.QMEAN_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TMEAN_$1" \
  -n $devmsg "&$UH_PREF.CTRL @DimTagUpdate=$UH_PREF.TRMSD_$1" \
  -n end \
  -n ;
};

function UniHeaterDimDicRecv(){
 let "nao=$UH_NumHeaters*3";
 let "naf=$nao*32+32";
 echo "[TagList]";
 echo "$UH_PREF.TNAMES = string $tnames";
 echo "$UH_PREF.PNAMES = string $pnames";
 echo "[]";
 echo "";
 echo ";******************************";
 echo "; Receiver for $UH_PREF";
 echo ";******************************";
 echo "[DeviceList]";
 echo "&$UH_PREF.RECV = device software program";
 echo "[&$UH_PREF.RECV]";
 echo "Comment         = DIM RECEIVER FOR $UH_PREF.";
 echo "InquiryPeriod   = 1";
 echo "DevicePolling   = 10, tpNormal";
 echo "ProgramSource   = ~~/resource/daqsite/dimserver/_dimrcv.pas";
 echo "AnalogFifo      = $naf";
 echo "DebugFlags      = 3";
 echo "OpenConsole     = 2";
 echo "EncryptMethod   = $UH_EncryptMethod";
 echo "TrustedUsers    = [TrustedUsers]";
 echo "DIM_GuiClickTag = $UH_PREF.DIMGUICLICK";
 echo "AnalogOutputs   = $nao";
 for i in $(seq 1 $UH_NumHeaters); do call LinkRecvAO $i; done;
 echo "[]";
 echo "";
};

function LinkRecvAO(){
 let "i1=($1-1)*3+0";
 let "i2=($1-1)*3+1";
 let "i3=($1-1)*3+2";
 echo "Link AnalogOutput $i1 with curve $UH_PREF.TREAL_$1 history $UH_HistoryLen";
 echo "Link AnalogOutput $i2 with curve $UH_PREF.PREAL_$1 history $UH_HistoryLen";
 echo "Link AnalogOutput $i3 with curve $UH_PREF.GATE_$1 history $UH_HistoryLen";
};

################################################################################
## Common library routines                                                    ##
################################################################################

function Define(){
 if [ -z "$1" ]; then return 0; fi; ### Environ Variable Name
 if [ -z "$2" ]; then return 0; fi; ### Environ Variable Value
 if [ -z "$(printenv $1)" ]; then
  export $1="$2";
 fi;
};

function GetPrefixFile(){
 if [ -z "$1" ]; then return; fi;
 if [ -z "$2" ]; then return; fi;
 export $1="$(echo "$2" | tr '[:upper:]' '[:lower:]')";
};

function RequiredParamStr(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then export $1="$2"; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName - missed string param $1. Call $SelfName --help for details.";
};

function RequiredParamInt(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then let "$1=$2"; export $1; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName : missed integer param $1. Call $SelfName --help for details.";
};

function RequiredAdminRights(){
 if sudo -n -v >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: Access denied. Admin rights required.";
};

function RequiredUnixUtils(){
 if which unix >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: UnixUtils is not valid. Install UnixUtils first.";
};

function RequiredParentCrwDaq(){
 local pid="$CRW_DAQ_SYS_EXE_PID";
 local cmd="$CRW_DAQ_SYS_EXE_FILE";
 local exe="$(basename $cmd)";
 local lst="";
 if [ -n "$pid" ] && [ -n "$exe" ]; then
  lst="$(pgrep $exe | grep $pid)";
 fi;
 if [ -z "$lst" ]; then
  fatal 1 "Error: $SelfName: parent CRW-DAQ is not found. Run this script from CRW-DAQ.";
 fi;
};

function RequiredFile(){
 if [ -n "$1" ] && [ -e "$1" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: not found $1.";
};

function RequiredDefined(){
 if [ -n "$1" ] && [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: undefined variable $1. Call $SelfName --help for details.";
};

function RequiredIntRange(){
 if [ -n "$1" ] &&  [ -n "$2" ] &&  [ -n "$3" ] && [ $1 -ge $2 ] && [ $1 -le $3 ]; then return 0; fi;
 fatal 1 "Error: $SelfName: invalid value $1 (not in $2..$3 range). Call $SelfName --help for details.";
};

function SetErrors(){
 let "errors=$1";
};

function IncErrors(){
 let "errors+=$1";
};

function HasNoErrors(){
 if [ "$errors" = "0" ]; then return $?; fi;
 return 1;
};

function Exec(){
 echo "$@";
 call "$@";
};

function PrintLn(){
 echo "$@";
};

function ErrorPrint(){
 1>&2 echo -n "$@";
};

function ErrorPrintLn(){
 1>&2 echo "$@";
};

function SetErrorLevel(){
 return $1;
};

function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then
   rm -f "$1";
  fi;
  shift;
 done;
};


uniheater_make_cfg "$@";

##############
## END OF FILE
##############
