 {
 Standard routines for Times:
 function  MsToDaqTime(ms:Real):Real;
 function  DaqTimeToMs(tm:Real):Real;
 function  MsToUnixTime(ms:Real):Real;
 function  UnixTimeToMs(tm:Real):Real;
 function  MsToJavaTime(ms:Real):Real;
 function  JavaTimeToMs(tm:Real):Real;
 function  MsToFileTime(ms:Real):Real;
 function  FileTimeToMs(tm:Real):Real;
 function  MsToWinsTime(ms:Real):Real;
 function  WinsTimeToMs(tm:Real):Real;
 function  MsToOleTime(ms:Real):Real;
 function  OleTimeToMs(tm:Real):Real;
 function  msElapsedSinceMarker(aMarker:Real):Real;
 function  GetDateAsNumber(ms:real):Integer;
 function  GetMidnight(ms:Real):Real;
 function  ms2DayOfWeek(ms:Real):Integer;
 function  DayOfWeekList(Lang:String):String;
 function  DayOfWeekStr(DayOfWeek:Integer; Lang:String):String;
 function  ms2DayOfWeekStr(ms:Real; Lang:String):String;
 function  MonthList(Lang:String):String;
 function  MonthStr(Month:Integer; Lang:String):String;
 function  ms2MonthStr(ms:Real; Lang:String):String;
 function  ms2HttpTimeStr(ms:Real):String;
 procedure SetClockRes(res:Integer);
 function  FibonacciNumber(arg:Integer):Integer;
 procedure CpuProfilerStress(mks:Real);
 procedure CpuProfilerStart(Poll,View,Mode,Stress:Integer;ms:Real);
 procedure CpuProfilerView;
 procedure CpuProfilerPoll;
 procedure CpuProfilerProcess(arg:String);
 procedure SysTimer_Init;
 procedure SysTimer_Poll;
 function  SysTimer_Pulse(ThePeriod:Integer):Real;
 function  SysTimer_Count:Integer;
 procedure ClearStdTimes;
 procedure InitStdTimes;
 procedure FreeStdTimes;
 procedure PollStdTimes;
 }
 {
 Note: 'ms' means 'milliseconds since Xmas'
 Xmas is: 0001.01.0.1-00:00:00 +0000 (UTC).
 It's DaqPascal msecnow function time unit.
 }
 {
 DAQ Pascal Time function defined in [DAQ] section by parameters:
 TimeBase (date and time of Time zero) and TimeUnit (in seconds).
 Convert DAQ Time (tm) to milliseconds since Xmas (ms) and back.
 }
 function MsToDaqTime(ms:Real):Real;
 begin
  MsToDaqTime:=(ms-TimeBase)/TimeUnits;
 end;
 function DaqTimeToMs(tm:Real):Real;
 begin
  DaqTimeToMs:=tm*TimeUnits+TimeBase;
 end;
 {
 UnixTime is number of seconds since Epoch.
 Epoch is: 1970-01-01 00:00:00 +0000 (UTC).
 It's standard Unix/Linux time(.) function.
 }
 function MsToUnixTime(ms:Real):Real;
 begin
  MsToUnixTime:=(ms-UnixTimeBase)/UnixTimeUnit;
 end;
 function UnixTimeToMs(tm:Real):Real;
 begin
  UnixTimeToMs:=tm*UnixTimeUnit+UnixTimeBase;
 end;
 {
 JavaTime is number of milliseconds since Epoch.
 It's time units uses by JavaScript new Data().
 }
 function MsToJavaTime(ms:Real):Real;
 begin
  MsToJavaTime:=(ms-JavaTimeBase)/JavaTimeUnit;
 end;
 function JavaTimeToMs(tm:Real):Real;
 begin
  JavaTimeToMs:=tm*JavaTimeUnit+JavaTimeBase;
 end;
 {
 FileTime is number of 100 ns units since 1601.01.01-00:00:00 UTC.
 It's time units uses by Win32 GetSystemTimeAsFileTime() FILETIME.
 }
 function MsToFileTime(ms:Real):Real;
 begin
  MsToFileTime:=(ms-FileTimeBase)/FileTimeUnit;
 end;
 function FileTimeToMs(tm:Real):Real;
 begin
  FileTimeToMs:=tm*FileTimeUnit+FileTimeBase;
 end;
 {
 WinsTime is number milliseconds since 1601.01.01-00:00:00 UTC.
 It's time units uses by Win32 GetSystemTime.
 }
 function MsToWinsTime(ms:Real):Real;
 begin
  MsToWinsTime:=(ms-WinsTimeBase)/WinsTimeUnit;
 end;
 function WinsTimeToMs(tm:Real):Real;
 begin
  WinsTimeToMs:=tm*WinsTimeUnit+WinsTimeBase;
 end;
 {
 OleTime is number of days since 1899.12.30-00:00:00 UTC.
 It's time unit uses by OLE, OLEDB, ADO, OPC DA & Delphi.
 }
 function MsToOleTime(ms:Real):Real;
 begin
  MsToOleTime:=(ms-OleTimeBase)/OleTimeUnit;
 end;
 function OleTimeToMs(tm:Real):Real;
 begin
  OleTimeToMs:=tm*OleTimeUnit+OleTimeBase;
 end;
 {
 Milliseconds elapsed since event marker aMarker.
 }
 function msElapsedSinceMarker(aMarker:Real):Real;
 begin
  if (aMarker<=0)
  then msElapsedSinceMarker:=0
  else msElapsedSinceMarker:=mSecNow-aMarker;
 end;
 {
 Get date as integer number, like 20050816 - YearMonthDay.
 }
 function GetDateAsNumber(ms:real):Integer;
 begin
  GetDateAsNumber:=(ms2year(ms)*100+ms2month(ms))*100+ms2day(ms);
 end;
 {
 Find midnight time by given time, ms
 }
 function GetMidnight(ms:Real):Real;
 begin
  GetMidnight:=DateTime2Ms(Ms2Year(ms),Ms2Month(ms),Ms2Day(ms),0,0,0,0);
 end;
 {
 Return day of week: 1..7=Mon..Sun.
 }
 function ms2DayOfWeek(ms:Real):Integer;
 var DaysSinceXmas:Integer;
 begin
  DaysSinceXmas:=Trunc(ms/86400000);
  ms2DayOfWeek:=(DaysSinceXmas mod 7)+1;
 end;
 {
 Return list of week days as language string.
  En:      Mo,Tu,We,Th,Fr,Sa,Su
  Ru:      Пн,Вт,Ср,Чт,Пт,Сб,Вс
  Eng:     Mon,Tue,Wed,Thu,Fri,Sat,Sun
  Rus:     Пнд,Втр,Срд,Чтв,Птн,Сбт,Вск
  English: Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday
  Russian: Понедельник,Вторник,Среда,Четверг,Пятница,Суббота,Воскресенье
  Default: 1,2,3,4,5,6,7    
 }
 function DayOfWeekList(Lang:String):String;
 begin
  if IsSameText(Lang,'En')      then Lang:='Mo,Tu,We,Th,Fr,Sa,Su' else
  if IsSameText(Lang,'Ru')      then Lang:='Пн,Вт,Ср,Чт,Пт,Сб,Вс' else
  if IsSameText(Lang,'Eng')     then Lang:='Mon,Tue,Wed,Thu,Fri,Sat,Sun' else
  if IsSameText(Lang,'Rus')     then Lang:='Пнд,Втр,Срд,Чтв,Птн,Сбт,Вск' else
  if IsSameText(Lang,'English') then Lang:='Monday,Tuesday,Wednesday,Thursday,Friday,Saturday,Sunday' else
  if IsSameText(Lang,'Russian') then Lang:='Понедельник,Вторник,Среда,Четверг,Пятница,Суббота,Воскресенье' else
  Lang:='1,2,3,4,5,6,7';
  DayOfWeekList:=Lang;
 end;
 {
 Return day of week as language string.
 }
 function DayOfWeekStr(DayOfWeek:Integer; Lang:String):String;
 begin
  DayOfWeekStr:=ExtractWord(DayOfWeek,DayOfWeekList(Lang));
 end;
 function ms2DayOfWeekStr(ms:Real; Lang:String):String;
 var DayOfWeek:Integer;
 begin
  DayOfWeek:=ms2DayOfWeek(ms);
  ms2DayOfWeekStr:=DayOfWeekStr(DayOfWeek,Lang);
 end;
 {
 Return list of month as language string.
  Eng:     Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec
  Rus:     Янв,Фев,Мар,Апр,Май,Июн,Июл,Авг,Сен,Окт,Ноя,Дек
  English: January,February,March,April,May,June,July,August,September,October,November,December
  Russian: Январь,Февраль,Март,Апрель,Май,Июнь,Июль,Август,Сентябрь,Октябрь,Ноябрь,Декабрь
  Default: 01,02,03,04,05,06,07,08,09,10,11,12    
 }
 function MonthList(Lang:String):String;
 begin
  if IsSameText(Lang,'Eng')     then Lang:='Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec' else
  if IsSameText(Lang,'Rus')     then Lang:='Янв,Фев,Мар,Апр,Май,Июн,Июл,Авг,Сен,Окт,Ноя,Дек' else
  if IsSameText(Lang,'English')
  then Lang:='January,February,March,April,May,June,July,August,September,October,November,December' else
  if IsSameText(Lang,'Russian')
  then Lang:='Январь,Февраль,Март,Апрель,Май,Июнь,Июль,Август,Сентябрь,Октябрь,Ноябрь,Декабрь' else
  Lang:='01,02,03,04,05,06,07,08,09,10,11,12';
  MonthList:=Lang;
 end;
 function MonthStr(Month:Integer; Lang:String):String;
 begin
  MonthStr:=ExtractWord(Month,MonthList(Lang));
 end;
 function ms2MonthStr(ms:Real; Lang:String):String;
 var Month:Integer;
 begin
  Month:=ms2Month(ms);
  ms2MonthStr:=MonthStr(Month,Lang);
 end;
 {
 Return HTTP (RFC 1123) time:
 Sun, 06 Nov 1994 08:49:37 GMT
 DAY, DD MON YYYY hh:mm:ss GMT
 }
 function ms2HttpTimeStr(ms:Real):String;
 begin
  ms2HttpTimeStr:=StrFmt('%s, ',ms2DayOfWeekStr(ms,'Eng'))
                 +StrFmt('%2.2d ',ms2day(ms))
                 +StrFmt('%s ',ms2MonthStr(ms,'Eng'))
                 +StrFmt('%4.4d ',ms2year(ms))
                 +StrFmt('%2.2d:',ms2hour(ms))
                 +StrFmt('%2.2d:',ms2min(ms))
                 +StrFmt('%2.2d GMT',ms2sec(ms));
 end;
 {
 Set system clock resolution, ms.
 Call SetClockRes(-1) to set previous resolution.
 }
 procedure SetClockRes(res:Integer);
 var r:Real;
 begin
  if res>0 then begin
   r:=Eval('@System @Async SetClockRes('+Str(res)+')');
   SetClockResCount:=SetClockResCount+1;
  end else
  while res<0 do begin
   r:=Eval('@system @async SetClockRes(-1)');
   SetClockResCount:=SetClockResCount-1;
   res:=res+1;
  end;
 end;
 {
 Calculate Fibbonaci numbers.
 }
 function FibonacciNumber(arg:Integer):Integer;
 begin
  if arg<=2
  then FibonacciNumber:=1
  else FibonacciNumber:=FibonacciNumber(arg-1)+FibonacciNumber(arg-2);
 end;
 {
 CPU stress for mks microseconds, with full CPU load.
 }
 procedure CpuProfilerStress(mks:Real);
 var ms:Real; q:Integer;
  procedure StressLoop(n,k:Integer);
  var i:Integer;
  begin
   for i:=1 to n do n:=FibonacciNumber(k);
  end;
 begin
  q:=15;
  ms:=mSecNow+Int(mks/1000/q)*q;
  mks:=mks+mkSecNow;
  while mSecNow<ms do StressLoop(1000,5); 
  while mkSecNow<mks do StressLoop(10,5); 
 end;
 {
 Start CPU performance profiler.
 }
 procedure CpuProfilerStart(Poll,View,Mode,Stress:Integer;ms:Real);
 begin
  if Poll>0 then begin
   CpuProfiler.Init.Kern:=rVal(ExtractWord(7,ParamStr('System Process Times')));
   CpuProfiler.Init.User:=rVal(ExtractWord(8,ParamStr('System Process Times')));
   CpuProfiler.Init.Vdpm:=VDPM_OpCount;
   CpuProfiler.Init.Runs:=RunCount;
   CpuProfiler.Init.Errs:=getErrCount(-1);
   CpuProfiler.Init.Line:=StdIn_LineCount;
   CpuProfiler.Init.Cmnd:=StdIn_CmndCount;
   CpuProfiler.Init.Time:=ms;
   CpuProfiler.Last.Kern:=CpuProfiler.Init.Kern;
   CpuProfiler.Last.User:=CpuProfiler.Init.User;
   CpuProfiler.Last.Vdpm:=CpuProfiler.Init.Vdpm;
   CpuProfiler.Last.Runs:=CpuProfiler.Init.Runs;
   CpuProfiler.Last.Errs:=CpuProfiler.Init.Errs;
   CpuProfiler.Last.Line:=CpuProfiler.Init.Line;
   CpuProfiler.Last.Cmnd:=CpuProfiler.Init.Cmnd;
   CpuProfiler.Last.Time:=CpuProfiler.Init.Time;
  end else begin
   CpuProfiler.Init.Kern:=0;
   CpuProfiler.Init.User:=0;
   CpuProfiler.Init.Vdpm:=0;
   CpuProfiler.Init.Runs:=0;
   CpuProfiler.Init.Errs:=0;
   CpuProfiler.Init.Line:=0;
   CpuProfiler.Init.Cmnd:=0;
   CpuProfiler.Init.Time:=0;
   CpuProfiler.Last.Kern:=0;
   CpuProfiler.Last.User:=0;
   CpuProfiler.Last.Vdpm:=0;
   CpuProfiler.Last.Runs:=0;
   CpuProfiler.Last.Errs:=0;
   CpuProfiler.Last.Line:=0;
   CpuProfiler.Last.Cmnd:=0;
   CpuProfiler.Last.Time:=0;
  end;
  CpuProfiler.Curr.Kern:=0;
  CpuProfiler.Curr.User:=0;
  CpuProfiler.Curr.Vdpm:=0;
  CpuProfiler.Curr.Runs:=0;
  CpuProfiler.Curr.Errs:=0;
  CpuProfiler.Curr.Line:=0;
  CpuProfiler.Curr.Cmnd:=0;
  CpuProfiler.Curr.Time:=0;
  CpuProfiler.Peak.Kern:=0;
  CpuProfiler.Peak.User:=0;
  CpuProfiler.Peak.Vdpm:=0;
  CpuProfiler.Peak.Runs:=0;
  CpuProfiler.Peak.Errs:=0;
  CpuProfiler.Peak.Line:=0;
  CpuProfiler.Peak.Cmnd:=0;
  CpuProfiler.Peak.Time:=0;
  CpuProfiler.Rate.Kern:=0;
  CpuProfiler.Rate.User:=0;
  CpuProfiler.Rate.Vdpm:=0;
  CpuProfiler.Rate.Runs:=0;
  CpuProfiler.Rate.Errs:=0;
  CpuProfiler.Rate.Line:=0;
  CpuProfiler.Rate.Cmnd:=0;
  CpuProfiler.Rate.Time:=0;
  CpuProfiler.Poll.Period:=Poll;
  CpuProfiler.View.Period:=View;
  CpuProfiler.View.Mode:=Mode;
  CpuProfiler.Poll.Stress:=Stress;
  CpuProfiler.View.Last:=ms;
 end;
 {
 Print performance profiler report view.
 1:  Header,Date,Time
 2:  Average CPU load
 }
 procedure CpuProfilerView;
 var dt:Real; i,ma,sa:Integer;
 begin
  dt:=1e-3*(CpuProfiler.Last.Time-CpuProfiler.Init.Time);
  if HasFlags(CpuProfiler.View.Mode,1) then begin
   for i:=1 to 2 do Success(Dump(' '));
   Success('Profiler report at '+GetDateTime(mSecNow)+': '+StrFix(dt,8,0)+' sec.work');
  end;
  if HasFlags(CpuProfiler.View.Mode,2*Ord(dt>0)+4) then begin
   Success(' ---------+----------+----------+----------+--------------+----------+----------+'
          +'----------+----------+');
  end;
  if HasFlags(CpuProfiler.View.Mode,2*Ord(dt>0)) then begin
   Success(' CPU Load | Kernel,% |   User,% | Thread,% | VDPM,kOp/sec | Runs/sec | Errs/sec |'
          +' Line/sec | Cmnd/sec |');
   Success(' ---------+----------+----------+----------+--------------+----------+----------+'
          +'----------+----------+');
   Success('  Average | '+StrFix(1e-5*(CpuProfiler.Last.Kern-CpuProfiler.Init.Kern)/dt,8,3)+' | '
                         +StrFix(1e-5*(CpuProfiler.Last.User-CpuProfiler.Init.User)/dt,8,3)+' | '
                         +StrFix(1e-5*(CpuProfiler.Last.Kern+CpuProfiler.Last.User-
                                      (CpuProfiler.Init.Kern+CpuProfiler.Init.User))/dt,8,3)+' | '
                         +StrFix(1e-3*(CpuProfiler.Last.Vdpm-CpuProfiler.Init.Vdpm)/dt,12,3)+' | '
                         +StrFix(     (CpuProfiler.Last.Runs-CpuProfiler.Init.Runs)/dt,8,3)+' | '
                         +StrFix(     (CpuProfiler.Last.Errs-CpuProfiler.Init.Errs)/dt,8,3)+' | '
                         +StrFix(     (CpuProfiler.Last.Line-CpuProfiler.Init.Line)/dt,8,3)+' | '
                         +StrFix(     (CpuProfiler.Last.Cmnd-CpuProfiler.Init.Cmnd)/dt,8,3)+' | '
                         );
   Success('  LastSec | '+StrFix(CpuProfiler.Rate.Kern,8,3)+' | '
                         +StrFix(CpuProfiler.Rate.User,8,3)+' | '
                         +StrFix(CpuProfiler.Rate.Time,8,3)+' | '
                         +StrFix(CpuProfiler.Rate.Vdpm,12,3)+' | '
                         +StrFix(CpuProfiler.Rate.Runs,8,3)+' | '
                         +StrFix(CpuProfiler.Rate.Errs,8,3)+' | '
                         +StrFix(CpuProfiler.Rate.Line,8,3)+' | '
                         +StrFix(CpuProfiler.Rate.Cmnd,8,3)+' | '
                         );
   Success('  Peaking | '+StrFix(CpuProfiler.Peak.Kern,8,3)+' | '
                         +StrFix(CpuProfiler.Peak.User,8,3)+' | '
                         +StrFix(CpuProfiler.Peak.Time,8,3)+' | '
                         +StrFix(CpuProfiler.Peak.Vdpm,12,3)+' | '
                         +StrFix(CpuProfiler.Peak.Runs,8,3)+' | '
                         +StrFix(CpuProfiler.Peak.Errs,8,3)+' | '
                         +StrFix(CpuProfiler.Peak.Line,8,3)+' | '
                         +StrFix(CpuProfiler.Peak.Cmnd,8,3)+' | '
                         );
   Success(' ---------+----------+----------+----------+--------------+----------+----------+'
          +'----------+----------+');
  end;
  if HasFlags(CpuProfiler.View.Mode,4) then begin
   ma:=MaxAvail; sa:=StackAvail;
   Success(' Status & | # Errors | StrTable |    Stack | Program Runs | >> Lines | Commands |'
          +' VDPM,MOp | Time,sec |');
   Success(' ---------+----------+----------+----------+--------------+----------+----------+'
          +'----------+----------+');
   Success(' Counters | ' +StrFix(getErrCount(-1),8,0)+' | '+StrFix(ma,8,0)+ ' | '+StrFix(sa,8,0)+' | '
          +StrFix(RunCount,12,0)+ ' | '+StrFix(StdIn_LineCount,8,0)+ ' | '+StrFix(StdIn_CmndCount,8,0)+' | '
          +StrFix(1e-6*CpuProfiler.Last.Vdpm,8,0)+' | '+StrFix(1e-3*(mSecNow-FixmSecNow),8,0)+' | '
          );
   Success(' ---------+----------+----------+----------+--------------+----------+----------+'
          +'----------+----------+');
  end;
 end;
 {
 Poll performance profiler.
 }
 procedure CpuProfilerPoll;
 var ms,dt:Real; Stress:Integer;
 begin
  if CpuProfiler.Poll.Period>0 then begin
   ms:=mSecNow; Stress:=0;
   if ms>CpuProfiler.Last.Time+CpuProfiler.Poll.Period then begin
    CpuProfiler.Curr.Kern:=rVal(ExtractWord(7,ParamStr('System Process Times')));
    CpuProfiler.Curr.User:=rVal(ExtractWord(8,ParamStr('System Process Times')));
    CpuProfiler.Curr.Vdpm:=VDPM_OpCount;
    CpuProfiler.Curr.Runs:=RunCount;
    CpuProfiler.Curr.Errs:=getErrCount(-1);
    CpuProfiler.Curr.Line:=StdIn_LineCount;
    CpuProfiler.Curr.Cmnd:=StdIn_CmndCount;
    CpuProfiler.Curr.Time:=ms;
    dt:=1e-3*(CpuProfiler.Curr.Time-CpuProfiler.Last.Time);
    if dt>0 then begin
     CpuProfiler.Rate.Kern:=1e-5*(CpuProfiler.Curr.Kern-CpuProfiler.Last.Kern)/dt;
     CpuProfiler.Rate.User:=1e-5*(CpuProfiler.Curr.User-CpuProfiler.Last.User)/dt;
     CpuProfiler.Rate.Vdpm:=1e-3*(CpuProfiler.Curr.Vdpm-CpuProfiler.Last.Vdpm)/dt;
     CpuProfiler.Rate.Runs:=     (CpuProfiler.Curr.Runs-CpuProfiler.Last.Runs)/dt;
     CpuProfiler.Rate.Errs:=     (CpuProfiler.Curr.Errs-CpuProfiler.Last.Errs)/dt;
     CpuProfiler.Rate.Line:=     (CpuProfiler.Curr.Line-CpuProfiler.Last.Line)/dt;
     CpuProfiler.Rate.Cmnd:=     (CpuProfiler.Curr.Cmnd-CpuProfiler.Last.Cmnd)/dt;
     CpuProfiler.Rate.Time:=CpuProfiler.Rate.Kern+CpuProfiler.Rate.User;
     CpuProfiler.Peak.Kern:=Max(CpuProfiler.Peak.Kern,CpuProfiler.Rate.Kern);
     CpuProfiler.Peak.User:=Max(CpuProfiler.Peak.User,CpuProfiler.Rate.User);
     CpuProfiler.Peak.Vdpm:=Max(CpuProfiler.Peak.Vdpm,CpuProfiler.Rate.Vdpm);
     CpuProfiler.Peak.Runs:=Max(CpuProfiler.Peak.Runs,CpuProfiler.Rate.Runs);
     CpuProfiler.Peak.Errs:=Max(CpuProfiler.Peak.Errs,CpuProfiler.Rate.Errs);
     CpuProfiler.Peak.Line:=Max(CpuProfiler.Peak.Line,CpuProfiler.Rate.Line);
     CpuProfiler.Peak.Cmnd:=Max(CpuProfiler.Peak.Cmnd,CpuProfiler.Rate.Cmnd);
     CpuProfiler.Peak.Time:=Max(CpuProfiler.Peak.Time,CpuProfiler.Rate.Time);
    end;
    CpuProfiler.Last.Kern:=CpuProfiler.Curr.Kern;
    CpuProfiler.Last.User:=CpuProfiler.Curr.User;
    CpuProfiler.Last.Vdpm:=CpuProfiler.Curr.Vdpm;
    CpuProfiler.Last.Runs:=CpuProfiler.Curr.Runs;
    CpuProfiler.Last.Errs:=CpuProfiler.Curr.Errs;
    CpuProfiler.Last.Line:=CpuProfiler.Curr.Line;
    CpuProfiler.Last.Cmnd:=CpuProfiler.Curr.Cmnd;
    CpuProfiler.Last.Time:=CpuProfiler.Curr.Time;
    Stress:=CpuProfiler.Poll.Stress;
   end;
   if CpuProfiler.View.Mode<>0 then
   if CpuProfiler.View.Period>0 then
   if ms>CpuProfiler.View.Last+CpuProfiler.View.Period then begin
    CpuProfilerView;
    CpuProfilerStart(CpuProfiler.Poll.Period,CpuProfiler.View.Period,
                     CpuProfiler.View.Mode,CpuProfiler.Poll.Stress,ms);
   end;
   if Stress>0 then CpuProfilerStress(Stress);
  end;
 end;
 {
 Performance profiler command line (arg) processor.
 }
 procedure CpuProfilerProcess(arg:String);
 var Poll,View,Mode,Stress:Integer; Load:Real;
 begin
  if IsEmptyStr(arg) then begin
   Success('Help on @CpuProfiler:');
   Success(' @CpuProfiler View');
   Success(' @CpuProfiler Stop');
   Success(' @CpuProfiler Start');
   Success(' @CpuProfiler Start 1 60 7 0');
   Success('  1  Poll period, sec');
   Success('  60 View period, sec');
   Success('  7  View mode: 1+2+4=Header+%CPU+Status');
   Success('  0  Stress test, %CPU load');
  end else
  if IsSameText(ExtractWord(1,arg),'View')  then CpuProfilerView else
  if IsSameText(ExtractWord(1,arg),'Stop')  then CpuProfilerStart(0,0,0,0,0) else
  if IsSameText(ExtractWord(1,arg),'Start') then begin
   Poll:=iMax(0,1000*iEvalDef(ExtractWord(2,arg),1));
   View:=iMax(0,1000*iEvalDef(ExtractWord(3,arg),60));
   Mode:=iEvalDef(ExtractWord(4,arg),7);
   Load:=Min(100,Max(0,rEvalDef(ExtractWord(5,arg),0)));
   Stress:=Round(Load*Poll*10);
   CpuProfilerStart(Poll,View,Mode,Stress,mSecNow);
   CpuProfilerView;
  end;
 end;
 {
 Initialize SysTimer.
 }
 procedure SysTimer_Init;
 var i:Integer;
 begin
  for i:=1 to SysTimer_MaxNum do SysTimer_DataRec.Period[i]:=0;
  for i:=1 to SysTimer_MaxNum do SysTimer_DataRec.PulseNum[i]:=0;
  for i:=1 to SysTimer_MaxNum do SysTimer_DataRec.TimeBase[i]:=0;
  for i:=1 to SysTimer_MaxNum do SysTimer_DataRec.LastPulse[i]:=0;
 end;
 {
 Poll SysTimer.
 }
 procedure SysTimer_Poll;
 var i,Period:Integer; ms,PulseNum:Real;
 begin
  if SysTimer_DataRec.Period[1]>0 then begin
   i:=1;
   while i<=SysTimer_MaxNum do begin
    Period:=SysTimer_DataRec.Period[i];
    if Period>0 then begin
     if i=1 then ms:=mSecNow;
     SysTimer_DataRec.PulseNum[i]:=0;
     PulseNum:=Int((ms-SysTimer_DataRec.TimeBase[i])/Period);
     if PulseNum<>SysTimer_DataRec.LastPulse[i] then begin
      SysTimer_DataRec.LastPulse[i]:=PulseNum;
      SysTimer_DataRec.PulseNum[i]:=PulseNum;
     end;
    end else i:=SysTimer_MaxNum;
    i:=i+1;
   end;
  end;
 end;
 {
 Return pulse (a number > 0) each ThePeriod ms.
 Example: if SysTimer_Pulse(1000) then DoItEverySecond;
 Example: rNul(SysTimer_Pulse(-1000)); // Release timer
 }
 function SysTimer_Pulse(ThePeriod:Integer):Real;
 var i,Period:Integer; PulseNum:Real;
 begin
  PulseNum:=0;
  if ThePeriod<>0 then begin
   i:=1;
   while i<=SysTimer_MaxNum do begin
    Period:=SysTimer_DataRec.Period[i];
    if Period<1 then begin
     if ThePeriod>0 then begin
      Period:=ThePeriod;
      SysTimer_DataRec.PulseNum[i]:=0;
      SysTimer_DataRec.LastPulse[i]:=0;
      SysTimer_DataRec.Period[i]:=Period;
      SysTimer_DataRec.TimeBase[i]:=Int(mSecNow/Period)*Period;
     end;
     i:=SysTimer_MaxNum;
    end else begin
     if Period=Abs(ThePeriod) then begin
      if ThePeriod>0 then begin
       PulseNum:=SysTimer_DataRec.PulseNum[i];
       i:=SysTimer_MaxNum;
      end else
      while i<=SysTimer_MaxNum do begin
       if i<SysTimer_MaxNum then begin
        SysTimer_DataRec.Period[i]:=SysTimer_DataRec.Period[i+1];
        SysTimer_DataRec.PulseNum[i]:=SysTimer_DataRec.PulseNum[i+1];
        SysTimer_DataRec.TimeBase[i]:=SysTimer_DataRec.TimeBase[i+1];
        SysTimer_DataRec.LastPulse[i]:=SysTimer_DataRec.LastPulse[i+1];
       end else
       if i=SysTimer_MaxNum then begin
        SysTimer_DataRec.Period[i]:=0;
        SysTimer_DataRec.PulseNum[i]:=0;
        SysTimer_DataRec.TimeBase[i]:=0;
        SysTimer_DataRec.LastPulse[i]:=0;
       end;
       i:=i+1;
      end;
     end;
    end;
    i:=i+1;
   end;
  end;
  SysTimer_Pulse:=PulseNum;
 end;
 {
 Return SysTimer counter, i.e. number of active timers.
 }
 function SysTimer_Count:Integer;
 var i,Count:Integer;
 begin
  Count:=0;
  if SysTimer_DataRec.Period[1]>0 then begin
   i:=1;
   while i<=SysTimer_MaxNum do begin
    if SysTimer_DataRec.Period[i]>0
    then Count:=Count+1
    else i:=SysTimer_MaxNum;
    i:=i+1;
   end;
  end;
  SysTimer_Count:=Count;
 end;
 {
 Clear standard Times.
 }
 procedure ClearStdTimes;
 begin
 end;
 {
 Initialize standard Times.
 }
 procedure InitStdTimes;
 begin
  SetClockResCount:=0;
  CpuProfilerStart(0,0,0,0,0);
  SysTimer_Init;
  ShouldPollStdTimes:=true;
 end;
 {
 Finalize standard Times.
 }
 procedure FreeStdTimes;
 begin
  SysTimer_Init;
  CpuProfilerStart(0,0,0,0,0);
  if SetClockResCount>0 then SetClockRes(-SetClockResCount);
  SetClockResCount:=0;
 end;
 {
 Poll standard Times.
 }
 procedure PollStdTimes;
 begin
  CpuProfilerPoll;
  SysTimer_Poll;
 end;
