 {
 Include FsmManager
 }
 {$I _fun_FsmManager.inc}
 {
 Functions for SmiProxy.
 function  smi_proxy_fsm:Integer;
 function  smi_proxy_dom:Integer;
 function  smi_proxy_obj:Integer;
 function  smi_proxy_state:Integer;
 function  smi_proxy_state_name:String;
 function  smi_proxy_state_color:Integer;
 function  smi_proxy_prompt:String;
 function  smi_extract_state(command:String):String;
 function  smi_extract_action(line:String):String;
 function  smi_to_fsm_type(typ:Integer):Integer;
 function  smi_to_tag_type(typ:Integer):Integer;
 function  smi_proxy_task_exefile:String;
 function  smi_proxy_task_smlfile:String;
 function  smi_proxy_task_cmdline:String;
 function  smi_proxy_task_pid:Integer;
 function  smi_proxy_task_running:Boolean;
 procedure smi_proxy_state_name_put(state:String);
 function  smi_proxy_object_name:String;
 function  smi_proxy_domain_name:String;
 procedure smi_proxy_object_name_put(name:String);
 function  smi_proxy_dns_node:String;
 procedure smi_proxy_dns_node_put(dns:String);
 function  smi_proxy_volatile:Integer;
 procedure smi_proxy_volatile_put(volatile:Integer);
 procedure smi_proxy_handler_params_clear(n:Integer);
 function  smi_proxy_handler_stage:Integer;
 procedure smi_proxy_handler_stage_put(stage:Integer);
 function  smi_proxy_handler_dead:Boolean;
 function  smi_proxy_handler_idle:Boolean;
 function  smi_proxy_handler_read:Boolean;
 function  smi_proxy_handler_busy:Boolean;
 function  smi_proxy_handler_term:Boolean;
 function  smi_proxy_handler_count:Integer;
 procedure smi_proxy_handler_count_put(count:Integer);
 function  smi_proxy_handler_uptime:Real;
 function  smi_proxy_handler_action:String;
 procedure smi_proxy_handler_action_put(action:String);
 function  smi_proxy_handler_command:String;
 procedure smi_proxy_handler_command_put(command:String);
 function  smi_proxy_handler_params_count:Integer;
 function  smi_proxy_handler_params_name(index:Integer):String;
 function  smi_proxy_handler_params_type(index:Integer):Integer;
 function  smi_proxy_handler_params_value(index:Integer):String;
 function  smi_proxy_handler_params_index(name:String):Integer;
 function  smi_proxy_handler_params_type_byname(name:String):Integer;
 function  smi_proxy_handler_params_value_byname(name:String):String;
 function  smi_proxy_find_state_color(objectname,state:String; def:Integer):Integer;
 function  smi_translate_state_command(WantedCommand,s:String):String;
 procedure smi_set_state(state:String);
 procedure smi_set_par(par,value:String; typ:Integer);
 procedure smi_terminate_action(state:String);
 procedure smi_move_to(state:String);
 function  smi_test_action(action:String):Boolean;
 function  smi_test_state(state:String):Boolean;
 procedure smi_proxy_task_send(msg:String);
 procedure smi_proxy_task_processing(var Data:String);
 procedure smi_proxy_become_dead;
 procedure smi_proxy_task_stop(exitcode:Integer);
 procedure smi_proxy_task_start;
 procedure smi_proxy_task_poll;
 procedure smi_proxy_clear_all;
 procedure smi_proxy_initialize;
 procedure smi_proxy_finalize;
 function  smi_proxy_default_handler(var Data,cmd,arg:String; cmdid:Integer):Boolean;
 procedure ClearSmiProxy;
 procedure InitSmiProxy;
 procedure FreeSmiProxy;
 procedure PollSmiProxy;
 }
 //
 // SMI Proxy FSM,Domain,Object,State reference.
 //
 function smi_proxy_fsm:Integer;
 begin
  smi_proxy_fsm:=smi_proxy_tab.fsm;
 end;
 function smi_proxy_dom:Integer;
 begin
  smi_proxy_dom:=smi_proxy_tab.dom;
 end;
 function smi_proxy_obj:Integer;
 begin
  smi_proxy_obj:=smi_proxy_tab.obj;
 end;
 function smi_proxy_state:Integer;
 begin
  smi_proxy_state:=fsm_get_state(smi_proxy_tab.obj);
 end;
 function smi_proxy_state_name:String;
 begin
  smi_proxy_state_name:=fsm_name(fsm_get_state(smi_proxy_tab.obj));
 end;
 function smi_proxy_state_color:Integer;
 var color,sta:Integer;
 begin
  color:=smi_proxy_tab.DefStateColor; sta:=fsm_get_state(smi_proxy_tab.obj);
  if (sta<>0) then color:=StringToColorDef(fsm_ctrl(sta,fsm_sym_color),color);
  smi_proxy_state_color:=color;
 end;
 //
 // Console prompt like SMI => ...
 //
 function smi_proxy_prompt:String;
 begin
  smi_proxy_prompt:=smi_proxy_tab.Prompt;
 end;
 //
 // Extract state/action name from line like READY/NUMBER_T(I)=1/NUMBER_P(I)=1.
 //
 function smi_extract_state(line:String):String;
 var p:Integer;
 begin
  p:=Pos('/',line);
  if (p>0) then line:=Copy(line,1,p-1);
  smi_extract_state:=Trim(line);
 end;
 function smi_extract_action(line:String):String;
 var p:Integer;
 begin
  p:=Pos('/',line);
  if (p>0) then line:=Copy(line,1,p-1);
  smi_extract_action:=Trim(line);
 end;
 //
 // SMI has type 0/1/2=STRING/INTEGER/FLOAT.
 // FSM has type 1/2/3=INTEGER/FLOAT/STRING.
 // CRW has type 1/2/3=INTEGER/REAL/STRING.
 //
 function smi_to_fsm_type(typ:Integer):Integer;
 begin
  if (typ=smi_integer) then typ:=fsm_type_int else
  if (typ=smi_float)   then typ:=fsm_type_float else
  if (typ=smi_string)  then typ:=fsm_type_string else typ:=fsm_type_nil;
  smi_to_fsm_type:=typ;
 end;
 function smi_to_tag_type(typ:Integer):Integer;
 begin
  if (typ=smi_integer) then typ:=1 else
  if (typ=smi_float)   then typ:=2 else
  if (typ=smi_string)  then typ:=3 else typ:=0;
  smi_to_tag_type:=typ;
 end;
 //
 // SMI Proxy EXE and SML file and CmdLine.
 // Evample: C:\Crw32exe\Resource\SmiSite\smi\bin\SMISM.EXE DEMO C:\DAQ32\DEMO_SMITEST\CONFIG\run_con.sobj -dns localhost -d 9 -t
 //
 function smi_proxy_task_exefile:String;
 begin
  smi_proxy_task_exefile:=smi_proxy_tab.task.exefile;
 end;
 function smi_proxy_task_smlfile:String;
 begin
  smi_proxy_task_smlfile:=smi_proxy_tab.task.smlfile;
 end;
 function smi_proxy_task_cmdline:String;
 begin
  smi_proxy_task_cmdline:=smi_proxy_tab.task.cmdline;
 end;
 //
 // SMI Proxy task PID or 0 if not running.
 //
 function smi_proxy_task_pid:Integer;
 begin
  if (smi_proxy_tab.task.tid<>0)
  then smi_proxy_task_pid:=task_pid(smi_proxy_tab.task.tid)
  else smi_proxy_task_pid:=0;
 end;
 //
 // SMI Proxy task is running?
 //
 function smi_proxy_task_running:Boolean;
 begin
  if (smi_proxy_tab.task.tid<>0)
  then smi_proxy_task_running:=task_wait(smi_proxy_tab.task.tid,0)
  else smi_proxy_task_running:=false;
 end;
 //
 // Get/set proxy actual object/domain/state/dns/volatile.
 //
 procedure smi_proxy_state_name_put(state:String);
 begin
  state:=Trim(smi_extract_state(state));
  if IsSameText(state,fsm_sym_dead) then begin
   if (fsm_ctrl(smi_proxy_tab.obj,fsm_sym_dead_state)<>'')
   then state:=fsm_ctrl(smi_proxy_tab.obj,fsm_sym_dead_state);
  end;
  state:=fsm_name(fsm_set_state(smi_proxy_tab.obj,state));
  if (smi_proxy_tab.tagObjectState<>0) then bNul(sSetTag(smi_proxy_tab.tagObjectState,state));
 end;
 function smi_proxy_object_name:String;
 begin
  smi_proxy_object_name:=fsm_path(smi_proxy_tab.obj);
 end;
 function smi_proxy_domain_name:String;
 begin
  smi_proxy_domain_name:=fsm_name(smi_proxy_tab.dom);
 end;
 procedure smi_proxy_object_name_put(name:String);
 begin
  name:=Trim(name);
  if (name<>'') and not IsSameText(name,fsm_path(smi_proxy_tab.obj))
  then Trouble('Invalid object name: '+name+'; expected '+fsm_path(smi_proxy_tab.obj));
  if (smi_proxy_tab.tagObjectName<>0) then bNul(sSetTag(smi_proxy_tab.tagObjectName,name));
 end;
 function smi_proxy_dns_node:String;
 begin
  smi_proxy_dns_node:=fsm_get_cookie(smi_proxy_tab.fsm,fsm_sym_dns_node);
 end;
 procedure smi_proxy_dns_node_put(dns:String);
 begin
  dns:=Trim(dns);
  if (dns<>'') and not IsSameText(dns,smi_proxy_dns_node)
  then Trouble('Invalid DNS_NODE: '+dns+'; expected '+smi_proxy_dns_node);
  if (smi_proxy_tab.tagObjectDns<>0) then bNul(sSetTag(smi_proxy_tab.tagObjectDns,dns));
 end;
 function smi_proxy_volatile:Integer;
 begin
  smi_proxy_volatile:=fsm_link(smi_proxy_tab.fsm,smi_sym_volatile);
 end;
 procedure smi_proxy_volatile_put(volatile:Integer);
 begin
  iNul(fsm_link(smi_proxy_tab.fsm,smi_sym_volatile+'='+Str(volatile)));
 end;
 //
 // SMI Proxy handler action params clear.
 //
 procedure smi_proxy_handler_params_clear(n:Integer);
 var i:Integer;
 begin
  if (smi_proxy_tab.Handler.Params<>0) then begin
   ClearText(smi_proxy_tab.Handler.Params);
   for i:=1 to n do bNul(text_addln(smi_proxy_tab.Handler.Params,''));
  end;
 end;
 //
 // Get proxy handler stage: DEAD,IDLE,READ,BUSY,TERM.
 // User should handle actions and commands on stage BUSY.
 //
 function smi_proxy_handler_stage:Integer;
 begin
  smi_proxy_handler_stage:=smi_proxy_tab.Handler.Stage;
 end;
 procedure smi_proxy_handler_stage_put(stage:Integer);
 begin
  if (stage=smi_proxy_stage_dead) then begin
   smi_proxy_tab.Handler.Stage:=smi_proxy_stage_dead;
   smi_proxy_tab.Handler.Uptimer:=mSecNow;
   smi_proxy_tab.Handler.Action:='';
   smi_proxy_tab.Handler.Command:='';
   smi_proxy_tab.Handler.TermTimer:=0;
   smi_proxy_tab.Handler.TermCount:=0;
   smi_proxy_handler_params_clear(0);
   smi_proxy_tab.Handler.Count:=0;
  end else
  if (stage=smi_proxy_stage_idle) then begin
   smi_proxy_tab.Handler.Stage:=smi_proxy_stage_idle;
   smi_proxy_tab.Handler.Uptimer:=mSecNow;
   smi_proxy_tab.Handler.Action:='';
   smi_proxy_tab.Handler.Command:='';
   smi_proxy_tab.Handler.TermTimer:=0;
   smi_proxy_tab.Handler.TermCount:=0;
   smi_proxy_handler_params_clear(0);
  end else
  if (stage=smi_proxy_stage_read) then begin
   smi_proxy_tab.Handler.Stage:=smi_proxy_stage_read;
   smi_proxy_tab.Handler.Uptimer:=mSecNow;
   smi_proxy_handler_params_clear(0);
  end else
  if (stage=smi_proxy_stage_busy) then begin
   smi_proxy_tab.Handler.Stage:=smi_proxy_stage_busy;
   smi_proxy_tab.Handler.Uptimer:=mSecNow;
  end else
  if (stage=smi_proxy_stage_term) then begin
   smi_proxy_tab.Handler.Stage:=smi_proxy_stage_term;
   smi_proxy_tab.Handler.Uptimer:=mSecNow;
   smi_proxy_tab.Handler.TermTimer:=mSecNow;
   smi_proxy_tab.Handler.TermCount:=0;
  end else
  Trouble('Invalid handler stage '+Str(stage));
  if (smi_proxy_tab.tagHandlerStage<>0) then bNul(iSetTag(smi_proxy_tab.tagHandlerStage,smi_proxy_tab.Handler.Stage));
 end;
 //
 // Check proxy handler stage is DEAD, i.e. server stopped.
 //
 function smi_proxy_handler_dead:Boolean;
 begin
  smi_proxy_handler_dead:=(smi_proxy_handler_stage=smi_proxy_stage_dead);
 end;
 //
 // Check proxy handler stage is IDLE, i.e. nothing happened.
 //
 function smi_proxy_handler_idle:Boolean;
 begin
  smi_proxy_handler_idle:=(smi_proxy_handler_stage=smi_proxy_stage_idle);
 end;
 //
 // Check proxy handler stage is READ, i.e. reading action parameters.
 //
 function smi_proxy_handler_read:Boolean;
 begin
  smi_proxy_handler_read:=(smi_proxy_handler_stage=smi_proxy_stage_read);
 end;
 //
 // Check proxy handler stage is busy, i.e. user should handle a command.
 //
 function smi_proxy_handler_busy:Boolean;
 begin
  smi_proxy_handler_busy:=(smi_proxy_handler_stage=smi_proxy_stage_busy);
 end;
 //
 // Check proxy handler stage is term, i.e. terminating command handling.
 //
 function smi_proxy_handler_term:Boolean;
 begin
  smi_proxy_handler_term:=(smi_proxy_handler_stage=smi_proxy_stage_term);
 end;
 //
 // Get/set proxy handler count.
 //
 function smi_proxy_handler_count:Integer;
 begin
  smi_proxy_handler_count:=smi_proxy_tab.Handler.Count;
 end;
 procedure smi_proxy_handler_count_put(count:Integer);
 begin
  smi_proxy_tab.Handler.Count:=count;
  if (smi_proxy_tab.tagHandlerCount<>0) then bNul(iSetTag(smi_proxy_tab.tagHandlerCount,smi_proxy_tab.Handler.Count));
 end;
 //
 // Check proxy handler uptime since last state changed.
 //
 function smi_proxy_handler_uptime:Real;
 begin
  smi_proxy_handler_uptime:=(mSecNow-smi_proxy_tab.Handler.Uptimer);
 end;
 //
 // Get proxy handler action. Applicable on handler stage BUSY.
 //
 function smi_proxy_handler_action:String;
 begin
  smi_proxy_handler_action:=smi_proxy_tab.Handler.Action;
 end;
 procedure smi_proxy_handler_action_put(action:String);
 begin
  action:=Trim(action);
  smi_proxy_tab.Handler.Action:=action;
  if (smi_proxy_tab.tagHandlerAction<>0) then bNul(sSetTag(smi_proxy_tab.tagHandlerAction,smi_proxy_tab.Handler.Action));
 end;
 //
 // Get proxy handler command. Applicable on handler stage BUSY.
 //
 function smi_proxy_handler_command:String;
 begin
  smi_proxy_handler_command:=smi_proxy_tab.Handler.Command;
 end;
 procedure smi_proxy_handler_command_put(command:String);
 begin
  command:=Trim(command);
  smi_proxy_tab.Handler.Command:=command;
  if (smi_proxy_tab.tagHandlerCommand<>0) then bNul(sSetTag(smi_proxy_tab.tagHandlerCommand,smi_proxy_tab.Handler.Command));
 end;
 //
 // SMI Proxy get action parameters count. Applicable on handler stage BUSY.
 //
 function smi_proxy_handler_params_count:Integer;
 begin
  if (smi_proxy_tab.Handler.Params=0)
  then smi_proxy_handler_params_count:=0
  else smi_proxy_handler_params_count:=text_numln(smi_proxy_tab.Handler.Params);
 end;
 //
 // SMI Proxy get action parameter name by index. Applicable on handler stage BUSY.
 //
 function smi_proxy_handler_params_name(index:Integer):String;
 begin
  if (smi_proxy_tab.Handler.Params=0) or (index<0)
  then smi_proxy_handler_params_name:=''
  else smi_proxy_handler_params_name:=ExtractWord(1,text_getln(smi_proxy_tab.Handler.Params,index));
 end;
 //
 // SMI Proxy get action parameters type by index. Applicable on handler stage BUSY.
 // Return 1/2/3=integer/real/string
 //
 function smi_proxy_handler_params_type(index:Integer):Integer;
 begin
  if (smi_proxy_tab.Handler.Params=0) or (index<0)
  then smi_proxy_handler_params_type:=0
  else smi_proxy_handler_params_type:=Val(ExtractWord(2,text_getln(smi_proxy_tab.Handler.Params,index)));
 end;
 //
 // SMI Proxy get action parameters value by index. Aplicable on handler stage BUSY.
 //
 function smi_proxy_handler_params_value(index:Integer):String;
 begin
  if (smi_proxy_tab.Handler.Params=0) or (index<0)
  then smi_proxy_handler_params_value:=''
  else smi_proxy_handler_params_value:=SkipWords(2,text_getln(smi_proxy_tab.Handler.Params,index));
 end;
 //
 // SMI Proxy get action parameters index by name. Aplicable on handler stage BUSY.
 //
 function smi_proxy_handler_params_index(name:String):Integer;
 var i,n,index:Integer;
 begin
  index:=-1;
  if (name<>'') then begin
   i:=0;
   n:=smi_proxy_handler_params_count;
   while (i<n) and (index<0) do begin
    if IsSameText(name,smi_proxy_handler_params_name(i)) then index:=i;
    i:=i+1;
   end;
  end;
  smi_proxy_handler_params_index:=index;
 end;
 //
 // SMI Proxy get action parameters type by name. Applicable on handler stage BUSY.
 // Return 1/2/3=integer/real/string
 //
 function smi_proxy_handler_params_type_byname(name:String):Integer;
 begin
  smi_proxy_handler_params_type_byname:=smi_proxy_handler_params_type(smi_proxy_handler_params_index(name));
 end;
 //
 // SMI Proxy get action parameters value by name. Applicable on handler stage BUSY.
 //
 function smi_proxy_handler_params_value_byname(name:String):String;
 begin
  smi_proxy_handler_params_value_byname:=smi_proxy_handler_params_value(smi_proxy_handler_params_index(name));
 end;
 //
 // Find object state color from SML file.
 //
 function smi_proxy_find_state_color(objectname,state:String; def:Integer):Integer;
 var color,fsm,sta:Integer;
 begin
  color:=def;
  fsm:=smi_proxy_fsm;
  if (fsm<>0) and (objectname<>'') and (state<>'') then begin
   sta:=fsm_find(fsm_find(fsm,fsm_type_object,objectname),fsm_type_state,state);
   if (sta<>0) then color:=StringToColorDef(fsm_ctrl(sta,fsm_sym_color),def);
  end;
  smi_proxy_find_state_color:=color;
 end;
 //
 // smi_translate_state_command('@smi_set_state=',s)
 // Translate s=DEMO/PAR1=ABC/PAR2(S)=DEF/PAR3(I)=123/PAR4(F)=1.2
 // to commands
 // @smi_set_par=PAR1,S,ABC
 // @smi_set_par=PAR2,S,DEF
 // @smi_set_par=PAR3,I,123
 // @smi_set_par=PAR4,F,1.2
 // @smi_set_state=DEMO
 //
 function smi_translate_state_command(WantedCommand,s:String):String;
 var i,t:Integer;
 begin
  s:=Trim(s);
  WantedCommand:=Trim(WantedCommand);
  if (s<>'') and (WantedCommand<>'') then
  if (Pos('/',s)=0) then s:=WantedCommand+s else begin
   s:=Trim(StringReplace(s,'/',EOL,rfReplaceAll));
   if (s<>'') then begin
    t:=TextAssignString(text_new,Trim(s));
    for i:=1 to text_numln(t)-1 do begin
     s:=Trim(text_getln(t,i));
     if (Pos('=',s)>0) then begin
      if (Pos('(',s)>0) and (Pos(')=',s)>0) then begin
       s:=StringReplace(s,')=',',',0);
       s:=StringReplace(s,'(',',',0);
      end else begin
       s:=StringReplace(s,'=',',S,',0);
      end;
      s:='@smi_set_par='+s;
      bNul(text_putln(t,i,s));
     end;
    end;
    s:=Trim(text_getln(t,0));
    s:=WantedCommand+s;
    bNul(text_addln(t,s));
    bNul(text_delln(t,0));
    s:=TextToString(t);
    bNul(text_free(t));
   end;
  end;
  smi_translate_state_command:=s; s:='';
 end;
 //
 // Send message to SMI Proxy task. Wait for some time if transmitter FIFO is over.
 // Commands @smi_set_state,@smi_terminate_action with ACTION/PARAM.../PARAM.../...
 // should be translated to set of @smi_set_par=... commands before send to server.
 //
 procedure smi_proxy_task_send(msg:String);
 var ms:Real; cmd,arg:String; trans:Boolean;
 begin
  msg:=Trim(msg);
  cmd:=''; arg:='';
  if (smi_proxy_tab.task.tid<>0) and (Length(msg)>0) then begin
   if IsLexeme(msg,lex_AtCall) then begin
    cmd:=ExtractWord(1,msg);
    arg:=SkipWords(1,msg);
    trans:=false;
    if (Pos('/',msg)>0) then
    if IsSameText(cmd,'@smi_set_state') then trans:=true else
    if IsSameText(cmd,'@smi_terminate_action') then trans:=true;
    if trans then msg:=smi_translate_state_command(cmd+'=',arg) else msg:=cmd+'='+arg;
   end;
   if task_txspace(smi_proxy_tab.task.tid)<Length(msg) then begin
    ms:=msecnow; while(msecnow-ms<100) and (task_txspace(smi_proxy_tab.task.tid)<Length(msg)) do bNul(Sleep(1));
   end;
   if (task_send(smi_proxy_tab.task.tid,msg+EOL)=0)
   then Trouble('Send error!') else if DebugFlagEnabled(dfViewExp) then ViewExp(msg);
  end;
  cmd:=''; arg:='';
 end;
 //
 // Set state by sending command to proxy.
 // Parameter state may be like READY/NUMBER(I)=1/VALUE(F)=123/NOTE(S)=OK.
 //
 procedure smi_set_state(state:String);
 begin
  state:=Trim(state);
  if (state='') then state:=smi_proxy_state_name;
  if (state<>'') then begin
   if not IsSameText(smi_extract_state(state),smi_proxy_state_name) then begin
    if (fsm_find(smi_proxy_obj,fsm_type_state,smi_extract_state(state))=0) then begin
     Trouble('Error: undefined state smi_set_state('+state+'). Applied smi_set_state('+smi_proxy_state_name+').');
     state:=smi_proxy_state_name;
    end;
   end;
   smi_proxy_task_send('@smi_set_state='+state);
  end;
 end;
 //
 // Applicable on handler stage BUSY.
 // Set parameter value by sending command to proxy.
 // Parameter type may be (SMI_STRING,SMI_INTEGER,SMI_FLOAT) or (fsm_type_int,fsm_type_float,fsm_type_string).
 //
 procedure smi_set_par(par,value:String; typ:Integer);
 var ctyp:Char;
 begin
  par:=Trim(par); value:=Trim(value);
  if (par<>'') and (value<>'') then begin
   if (typ=SMI_STRING)  or (typ=fsm_type_string) then ctyp:='S' else
   if (typ=SMI_INTEGER) or (typ=fsm_type_int)    then ctyp:='I' else
   if (typ=SMI_FLOAT)   or (typ=fsm_type_float)  then ctyp:='F' else ctyp:='?';
   if (ctyp='?')
   then Trouble('Invalid type in smi_set_par: '+Str(typ))
   else smi_proxy_task_send('@smi_set_par='+par+','+ctyp+','+value);
  end;
 end;
 //
 // Applicable on handler stage BUSY.
 // Terminate action by sending command to proxy, then go to stage TERM.
 //
 procedure smi_terminate_action(state:String);
 begin
  state:=Trim(state);
  if (state='') then state:=smi_proxy_state_name;
  if (state<>'') then begin
   if not IsSameText(smi_extract_state(state),smi_proxy_state_name) then begin
    if (fsm_find(smi_proxy_tab.obj,fsm_type_state,smi_extract_state(state))=0) then begin
     Trouble('Error: undefined state smi_terminate_action('+state+'). Applied smi_terminate_action('+smi_proxy_state_name+').');
     state:=smi_proxy_state_name;
    end;
   end;
   smi_proxy_task_send('@smi_terminate_action='+state);
   if smi_proxy_handler_busy then smi_proxy_handler_stage_put(smi_proxy_stage_term);
  end;
 end;
 procedure smi_move_to(state:String);
 begin
  smi_terminate_action(state);
 end;
 //
 // Check if action is to be executed. Applicable on handler stage BUSY.
 //
 function smi_test_action(action:String):Boolean;
 var flag:Boolean;
 begin
  flag:=false;
  if smi_proxy_handler_busy then
  if IsSameText(action,smi_proxy_handler_action) then flag:=true;
  smi_test_action:=flag;
 end;
 //
 // Check if the Proxy has current state with given name.
 //
 function smi_test_state(state:String):Boolean;
 var sta:Integer;
 begin
  sta:=fsm_get_state(smi_proxy_tab.obj);
  if (sta<>0) and (state<>'')
  then smi_test_state:=IsSameText(fsm_name(sta),state)
  else smi_test_state:=false;
 end;
 //
 // SMI Proxy task StdOut processing.
 //
 // DIM sample string:
 //   PID 7344 - Thu Dec 03 20:00:19 2020 - (INFO) Server Connection established to DIM_DNS@localhost
 //   PID 7344 - Thu Dec 03 21:00:12 2020 - (ERROR) Server Connecting to DIM_DNS@localhost: Connection refused
 // Note: DIM output uses to detect DIM server state.
 //
 // SMIRTL sample string:
 //   SMIRTL - Thu Dec 03 20:52:18 2020 - DEMO::EVT_BUILDER terminating in state RUNNING/NUMBER_T(I)=47/NUMBER_P(I)=35
 // Note: SMIRTL output uses to detect handler TERM acknowledge.
 //
 procedure smi_proxy_task_processing(var Data:String);
 const DimMsgSign='PID '; SmiRtlSign='SMIRTL '; SmiTermSign=' terminating in state ';
 var cmd,arg,name:String; cmdid:Integer; i,n,p:Integer;
 begin
  cmd:=''; arg:=''; name:='';
  if Length(Data)>0 then begin
   if DebugFlagEnabled(dfViewImp) then ViewImp(smi_proxy_prompt+Data);
   if (Copy(Data,1,Length(DimMsgSign))=DimMsgSign) then begin       // Message came from DIM:
    Success(smi_proxy_prompt+Data);                                 // Print this to console.
    Data:='';                                                       // Done.
   end else
   if (Copy(Data,1,Length(SmiRtlSign))=SmiRtlSign) then begin       // Message came from SMIRTL:
    if smi_proxy_handler_term then begin                            // TERM stage? Wait for acknowledge?
     p:=Pos(SmiTermSign,Data);                                      // Find termination acknowledge sign
     if (p>0) then                                                  // On TERM acknowledge should go IDLE
     if (Pos(smi_proxy_object_name,Data)>0) then begin              // It's acknowledge for data source
      name:=Copy(Data,p+Length(SmiTermSign));                       // Get tail after acknowledge sign
      name:=smi_extract_action(ExtractWord(1,name));                // Extract name of action
      if name<>'' then smi_proxy_state_name_put(name);            // It's actual state now
      smi_proxy_handler_stage_put(smi_proxy_stage_idle);            // Goto IDLE
      Success('Handler TERM acknowledge received, state '+name);    // Report
     end;                                                           //
    end;                                                            //
    Success(smi_proxy_prompt+Data);                                 // Print this to console.
    Data:='';                                                       // Done.
   end else
   if GotCommandId(Data,cmd,arg,cmdid) then begin
    //
    // @Exit=n
    //
    if (cmdid=smi_proxy_tab.cmd_exit) then begin
     Success(smi_proxy_prompt+'Server request to stop with exit code '+Trim(arg));
     Data:='';
    end else
    //
    // @Errors=n
    //
    if (cmdid=smi_proxy_tab.cmd_errors) then begin
     smi_proxy_tab.Task.WatchdogTimer:=msecnow; // !!! Reset Watchdog !!!
     n:=Val(Trim(arg));
     if n>0 then begin
      for i:=1 to n-1 do bNul(FixError(errorcode));
      Trouble(StrFmt('Server errors detected (%d).',n));
     end;
     Data:='';
    end else
    //
    // @Memory=n
    //
    if (cmdid=smi_proxy_tab.cmd_memory) then begin
     smi_proxy_tab.Task.WatchdogTimer:=msecnow; // !!! Reset Watchdog !!!
     if DebugFlagEnabled(dfDetails) then
     Details(smi_proxy_prompt+Data);
     Data:='';
    end else
    //
    // @ProcessPriority=p
    //
    if (cmdid=smi_proxy_tab.cmd_processpriority) then begin
     Success(smi_proxy_prompt+Data);
     Data:='';
    end else
    //
    // @ThreadPriority=p
    //
    if (cmdid=smi_proxy_tab.cmd_threadpriority) then begin
     Success(smi_proxy_prompt+Data);
     Data:='';
    end else
    //
    // @OnException=message
    //
    if (cmdid=smi_proxy_tab.cmd_onexception) then begin
     DevPostCmdLocal('@SysEval @Silent @OnException '+Trim(arg));
     Trouble('Server task exception detected.');
     Success(smi_proxy_prompt+Data);
     Data:='';
    end else
    //
    // Alternative case
    //
    begin
     DevPostCmdLocal(Data);
     Data:='';
    end;
   end;
  end;
  cmd:=''; arg:=''; name:='';
 end;
 //
 // Mark SMI Proxy become dead, clear task, handler and actual data.
 //
 procedure smi_proxy_become_dead;
 begin
  smi_proxy_tab.task.tid:=0;
  smi_proxy_tab.task.Line:='';
  smi_proxy_tab.task.Buff:='';
  smi_proxy_tab.task.WatchdogTimer:=0;
  smi_proxy_dns_node_put('');
  smi_proxy_object_name_put('');
  smi_proxy_volatile_put(0);
  smi_proxy_state_name_put(UpCaseStr(fsm_sym_dead));
  smi_proxy_handler_stage_put(smi_proxy_stage_dead);
 end;
 //
 // Stop SMI Proxy task if one started.
 //
 procedure smi_proxy_task_stop(exitcode:Integer);
 begin
  if (smi_proxy_tab.task.tid<>0) then begin
   if task_wait(smi_proxy_tab.task.tid,0) then begin
    Success(smi_proxy_prompt+'Terminating Server...');
    smi_proxy_task_send('@exit='+Str(exitcode));
    bNul(task_wait(smi_proxy_tab.task.tid,2000));
    if task_wait(smi_proxy_tab.task.tid,0) then bNul(task_kill(smi_proxy_tab.task.tid,0,1,0));
    if task_rxcount(smi_proxy_tab.task.tid)>0 then
    while task_readln(smi_proxy_tab.task.tid,smi_proxy_tab.task.line,smi_proxy_tab.task.buff) do begin
     smi_proxy_task_processing(smi_proxy_tab.task.line);
     smi_proxy_tab.task.line:='';
    end;
    Success(smi_proxy_prompt+'Server stopped with exit code '+Str(task_result(smi_proxy_tab.task.tid)));
   end;
   bNul(task_free(smi_proxy_tab.task.tid));
  end;
  smi_proxy_become_dead;
 end;
 //
 // Start SMI proxy task.
 //
 procedure smi_proxy_task_start;
 begin
  if smi_proxy_tab.task.tid=0 then begin
   smi_proxy_tab.task.tid:=task_init(smi_proxy_tab.task.CmdLine);
   if pos('?',task_ctrl(smi_proxy_tab.task.tid,'StdOutPipeSize='+Str(smi_proxy_tab.task.StdOutPipeSize))
             +task_ctrl(smi_proxy_tab.task.tid,'StdInPipeSize='+Str(smi_proxy_tab.task.StdInPipeSize))
             +task_ctrl(smi_proxy_tab.task.tid,'HomeDir='+ExtractFilePath(smi_proxy_tab.task.ExeFile))
             +task_ctrl(smi_proxy_tab.task.tid,'Display='+Str(smi_proxy_tab.task.display))
             )>0
   then begin
    Trouble(smi_proxy_prompt+'Task setup error!');
    smi_proxy_task_stop(0);
   end;
   //
   // Run task if one was created...
   //
   if smi_proxy_tab.task.tid>0 then
   if task_run(smi_proxy_tab.task.tid) then begin
    Success('TaskId  = '+str(smi_proxy_tab.task.tid));
    Success('TaskPid = '+str(task_pid(smi_proxy_tab.task.tid)));
    Success('TaskRef = '+str(task_ref(smi_proxy_tab.task.tid)));
    Success('CmdLine = '+task_ctrl(smi_proxy_tab.task.tid,'CmdLine'));
    Success('HomeDir = '+task_ctrl(smi_proxy_tab.task.tid,'HomeDir'));
    Success('PipeIn  = '+task_ctrl(smi_proxy_tab.task.tid,'StdInPipeSize'));
    Success('PipeOut = '+task_ctrl(smi_proxy_tab.task.tid,'StdOutPipeSize'));
    Success('Display = '+task_ctrl(smi_proxy_tab.task.tid,'Display'));
   end else begin
    Trouble(smi_proxy_prompt+'Could not start Server!');
    smi_proxy_task_stop(0);
   end;
   //
   // Is it Ok with user task? Send startup parameters.
   //
   if smi_proxy_tab.task.tid>0 then
   if task_wait(smi_proxy_tab.task.tid,0) then begin
    smi_proxy_tab.Task.WatchdogTimer:=msecnow; // !!!Start watchdog!!!
    smi_proxy_handler_stage_put(smi_proxy_stage_idle);
    smi_proxy_task_send('@memory');
    smi_proxy_task_send('@errors');
    smi_proxy_tab.task.line:='';
    smi_proxy_tab.task.buff:='';
   end else begin
    Trouble(smi_proxy_prompt+'Failed start Server!');
    smi_proxy_task_stop(0);
   end;
  end;
 end;
 //
 // Poll SMI Proxy task.
 //
 procedure smi_proxy_task_poll;
 begin
  //
  // If server is not still running,
  // try to start server periodically.
  //
  if smi_proxy_tab.task.tid=0 then
  if smi_proxy_tab.task.RecoveryPeriod>0 then
  if msecnow>smi_proxy_tab.task.RecoveryTimer+smi_proxy_tab.task.RecoveryPeriod then begin
   smi_proxy_tab.task.RecoveryTimer:=msecnow;
   smi_proxy_task_start;
  end;
  //
  // Communicate with server if one still running...
  //
  if smi_proxy_tab.task.tid>0 then
  if task_wait(smi_proxy_tab.task.tid,0) then begin
   //
   // If got data from Task StdOut, analyse it...
   //
   if task_rxcount(smi_proxy_tab.task.tid)>0 then
   while Task_Readln(smi_proxy_tab.task.tid,smi_proxy_tab.task.line,smi_proxy_tab.task.buff) do begin
    smi_proxy_task_processing(smi_proxy_tab.task.line);
    smi_proxy_tab.task.line:='';
   end;
   //
   // Reset recovery timer...
   //
   smi_proxy_tab.task.RecoveryTimer:=msecnow;
   //
   // Server timer actions...
   //
   if SysTimer_Pulse(1000)>0 then begin
    smi_proxy_task_send('@Errors=0'); // !!! Required to reset watchdog !!!
    smi_proxy_task_send('@Memory');   // !!! Required to reset watchdog !!!
   end;
  end else begin
   Trouble(smi_proxy_prompt+'Server terminated with exit code '+Str(task_result(smi_proxy_tab.task.tid)));
   smi_proxy_task_stop(0);
  end;
 end;
 //
 // SMI Proxy watchdog.
 //
 procedure smi_proxy_watchdog_poll;
 var DEADLINE:Real;
  procedure Restart(msg:String);
  begin
   Trouble(msg+' DEADLINE. Server restart.');
   smi_proxy_task_stop(0);
   smi_proxy_tab.task.RecoveryTimer:=msecnow-smi_proxy_tab.task.RecoveryPeriod;
  end;
 begin
  DEADLINE:=smi_proxy_tab.task.RecoveryPeriod;
  if smi_proxy_handler_read then begin
   if smi_proxy_handler_uptime>DEADLINE
   then Restart('HANDLER READ');
  end else
  if smi_proxy_handler_term then begin
   if smi_proxy_handler_uptime>DEADLINE
   then Restart('HANDLER TERM');
  end;
  if smi_proxy_tab.task.tid>0 then
  if msElapsedSinceMarker(smi_proxy_tab.task.WatchdogTimer)>DEADLINE
  then Restart('TASK WATCHDOG');
 end;
 //
 // Clear all SMI Proxy data.
 //
 procedure smi_proxy_clear_all;
 begin
  smi_proxy_tab.fsm:=0;
  smi_proxy_tab.dom:=0;
  smi_proxy_tab.obj:=0;
  smi_proxy_tab.task.tid:=0;
  smi_proxy_tab.task.ExeFile:='';
  smi_proxy_tab.task.CmdLine:='';
  smi_proxy_tab.task.SmlFile:='';
  smi_proxy_tab.task.Line:='';
  smi_proxy_tab.task.Buff:='';
  smi_proxy_tab.task.RecoveryTimer:=0;
  smi_proxy_tab.Task.WatchdogTimer:=0;
  smi_proxy_tab.optObj:='';
  smi_proxy_tab.optDom:='';
  smi_proxy_tab.optSet:='';
  smi_proxy_tab.optDns:='';
  smi_proxy_tab.optVol:=0;
  smi_proxy_tab.Handler.Stage:=smi_proxy_stage_dead;
  smi_proxy_tab.Handler.Count:=0;
  smi_proxy_tab.Handler.Action:='';
  smi_proxy_tab.Handler.Command:='';
  smi_proxy_tab.Handler.Uptimer:=0;
  smi_proxy_tab.Handler.Params:=0;
  smi_proxy_tab.Handler.TermCount:=0;
  smi_proxy_tab.Handler.TermRepeater:=0;
  smi_proxy_tab.Handler.TermPeriod:=0;
  smi_proxy_tab.Prompt:='';
  smi_proxy_tab.DefStateColor:=0;
  smi_proxy_tab.StateColorStamp:=0;
  smi_proxy_tab.StateColorAuto:=false;
  smi_proxy_tab.tagObjectName:=0;
  smi_proxy_tab.tagObjectState:=0;
  smi_proxy_tab.tagObjectDns:=0;
  smi_proxy_tab.tagHandlerStage:=0;
  smi_proxy_tab.tagHandlerCount:=0;
  smi_proxy_tab.tagHandlerAction:=0;
  smi_proxy_tab.tagHandlerCommand:=0;
  smi_proxy_tab.tagHandlerUptime:=0;
 end;
 //
 // SMI Proxy initialize params from INI file.
 //
 procedure smi_proxy_initialize;
 var stan:String; ps:Integer;
  procedure ClearLocals;
  begin
   stan:='';
  end;
  procedure smi_proxy_fsm_init(dns,dom,obj,sml:String);
  begin
   smi_proxy_tab.fsm:=fsm_init('-dns '+dns+' -dom '+dom+' -sml '+sml);
   smi_proxy_tab.dom:=fsm_find(smi_proxy_fsm,fsm_type_domain,dom);
   smi_proxy_tab.obj:=fsm_find(smi_proxy_fsm,fsm_type_object,obj);
  end;
  procedure smi_proxy_handler_init;
  begin
   smi_proxy_tab.Handler.Params:=text_new;
  end;
  procedure smi_proxy_register_commands;
  begin
   smi_proxy_tab.cmd_exit                 := RegisterStdInCmd('@Exit',                 '@Exit=n - Exit server with code n');
   smi_proxy_tab.cmd_memory               := RegisterStdInCmd('@Memory',               '@Memory=n - Check memory usage');
   smi_proxy_tab.cmd_errors               := RegisterStdInCmd('@Errors',               '@Errors=n - Return server errors');
   smi_proxy_tab.cmd_ProcessPriority      := RegisterStdInCmd('@ProcessPriority',      '@ProcessPriority=p - Process priority');
   smi_proxy_tab.cmd_ThreadPriority       := RegisterStdInCmd('@ThreadPriority',       '@ThreadPriority=p - Thread priority');
   smi_proxy_tab.cmd_OnException          := RegisterStdInCmd('@OnException',          '@OnException=msg - Exception handler test.');
   smi_proxy_tab.cmd_smi_attach           := RegisterStdInCmd('@smi_attach',           '@smi_attach=status,objectname');
   smi_proxy_tab.cmd_smi_volatile         := RegisterStdInCmd('@smi_volatile',         '@smi_volatile=status');
   smi_proxy_tab.cmd_smi_get_state        := RegisterStdInCmd('@smi_get_state',        '@smi_get_state=status,state');
   smi_proxy_tab.cmd_smi_set_state        := RegisterStdInCmd('@smi_set_state',        '@smi_set_state=status,state');
   smi_proxy_tab.cmd_smi_set_par          := RegisterStdInCmd('@smi_set_par',          '@smi_set_par=status,par,type,value');
   smi_proxy_tab.cmd_smi_proxy_handler    := RegisterStdInCmd('@smi_proxy_handler',    '@smi_proxy_handler=BEGIN/END,counter');
   smi_proxy_tab.cmd_smi_get_action       := RegisterStdInCmd('@smi_get_action',       '@smi_get_action=status,action');
   smi_proxy_tab.cmd_smi_get_command      := RegisterStdInCmd('@smi_get_command',      '@smi_get_command=status,command');
   smi_proxy_tab.cmd_smi_get_next_par     := RegisterStdInCmd('@smi_get_next_par',     '@smi_get_next_par=index,param,type,size');
   smi_proxy_tab.cmd_smi_get_par_value    := RegisterStdInCmd('@smi_get_par_value',    '@smi_get_par_value=index,param,type,value');
   smi_proxy_tab.cmd_dim_dns_node         := RegisterStdInCmd('@dim_dns_node',         '@dim_dns_node=Get/set DIM DNS NODE');
   smi_proxy_tab.cmd_smi_terminate_action := RegisterStdInCmd('@smi_terminate_action', '@smi_terminate_action=Terminate action and set new state');
   smi_proxy_tab.cmd_smi_errors           := RegisterStdInCmd('@smi_errors',           '@smi_errors=count SMI Proxy errors');
   smi_proxy_tab.cmd_smi_report           := RegisterStdInCmd('@smi_report',           '@smi_report=ERROR: message');
   smi_proxy_tab.cmd_smi_restart          := RegisterStdInCmd('@smi_restart',          '@smi_restart=exitcode - Restart Proxy server');
   smi_proxy_tab.cmd_smi_repost           := RegisterStdInCmd('@smi_repost',           '@smi_repost - Repost actual state by smi_terminate_action');
   smi_proxy_tab.cmd_smi_proxy            := RegisterStdInCmd('@smi_proxy',            '@smi_proxy command Send command to SMI Proxy server');
  end;
  procedure smi_proxy_init_tags(tagPrefix:String);
  begin
   InitTag(smi_proxy_tab.tagObjectName,     tagPrefix+'.ObjectName',     -3);
   InitTag(smi_proxy_tab.tagObjectState,    tagPrefix+'.ObjectState',    -3);
   InitTag(smi_proxy_tab.tagObjectDns,      tagPrefix+'.ObjectDns',      -3);
   InitTag(smi_proxy_tab.tagHandlerStage,   tagPrefix+'.HandlerStage',   -1);
   InitTag(smi_proxy_tab.tagHandlerCount,   tagPrefix+'.HandlerCount',   -1);
   InitTag(smi_proxy_tab.tagHandlerAction,  tagPrefix+'.HandlerAction',  -3);
   InitTag(smi_proxy_tab.tagHandlerCommand, tagPrefix+'.HandlerCommand', -3);
   InitTag(smi_proxy_tab.tagHandlerUptime,  tagPrefix+'.HandlerUptime',  -2);
  end;
 begin
  ClearLocals;
  smi_proxy_clear_all;
  smi_proxy_handler_init;
  smi_proxy_register_commands;
  smi_proxy_tab.Prompt:='SMI => ';
  smi_proxy_init_tags(ReadIni('tagPrefix'));
  // SMIPROXY_EXE = ~~\Resource\DaqSite\SmiServer\SmiProxy.exe
  smi_proxy_tab.task.ExeFile:=DaqFileRef(AdaptExeFileName(DefaultExtension(ReadIni('SMIPROXY_EXE'),'.exe')),'');
  Assertion(FileExists(smi_proxy_tab.task.ExeFile),'SMIPROXY_EXE = '+smi_proxy_tab.task.ExeFile);
  // SMIPROXY_SML = run_con.sml
  smi_proxy_tab.task.SmlFile:=DaqFileRef(AdaptFileName(ReadIni('SMIPROXY_SML')),'.sml');
  Assertion(FileExists(smi_proxy_tab.task.SmlFile),'SMIPROXY_SML = '+smi_proxy_tab.task.SmlFile);
  // SMIPROXY_DNS = localhost
  smi_proxy_tab.optDns:=LoCaseStr(ExtractWord(1,ReadIniAlter('SMIPROXY_DNS',risModeDefault+risModeAlter)));
  if (smi_proxy_tab.optDns='.') then smi_proxy_tab.optDns:='localhost';
  Assertion((smi_proxy_tab.optDns<>''),'SMIPROXY_DNS = '+smi_proxy_tab.optDns);
  // SMIPROXY_OBJ = DOMAIN::OBJECT/STATE/PARAMETERS
  stan:=ReadIniAlter('SMIPROXY_OBJ',risModeDefault); // SMIPROXY_OBJ = DEMO::EVT_BUILDER/READY/NUMBER_T(I)=1/NUMBER_P(I)=1
  smi_proxy_tab.optDom:=fsm_extract_name(stan,fsm_type_domain); smi_proxy_tab.optDom:=UpCaseStr(Trim(smi_proxy_tab.optDom));
  smi_proxy_tab.optObj:=fsm_extract_name(stan,fsm_type_object); smi_proxy_tab.optObj:=UpCaseStr(Trim(smi_proxy_tab.optObj));
  ps:=NthPos(Dump('/'),stan,2); if (ps>0) then smi_proxy_tab.optSet:=Trim(Copy(stan,ps+1)) else smi_proxy_tab.optSet:='';
  stan:=fsm_extract_name(stan,fsm_type_state); stan:=UpCaseStr(Trim(stan));
  Assertion((stan<>'') and (smi_proxy_tab.optDom<>'') and (smi_proxy_tab.optObj<>''),
           'SMIPROXY_OBJ = '+smi_proxy_tab.optDom+'::'+smi_proxy_tab.optObj+'/'+stan+'/'+smi_proxy_tab.optSet);
  if (smi_proxy_tab.optSet='') then smi_proxy_tab.optSet:=stan else smi_proxy_tab.optSet:=stan+'/'+smi_proxy_tab.optSet;
  smi_proxy_tab.optObj:=smi_proxy_tab.optDom+'::'+smi_proxy_tab.optObj;
  // Display = 0
  smi_proxy_tab.task.display:=iValDef(ReadIni('SmiDisplayMode'),0);
  Assertion(smi_proxy_tab.task.display>=0,StrFmt('SmiDisplayMode = %d',smi_proxy_tab.task.display));
  // StdInPipeSize = 16
  smi_proxy_tab.task.StdInPipeSize:=iValDef(ReadIni('StdInPipeSize'),16)*1024;
  Assertion(smi_proxy_tab.task.StdInPipeSize>=1024,'StdInPipeSize = '+Str(smi_proxy_tab.task.StdInPipeSize div 1024));
  // StdOuPipeSize = 16
  smi_proxy_tab.task.StdOutPipeSize:=iValDef(ReadIni('StdOutPipeSize'),16)*1024;
  Assertion(smi_proxy_tab.task.StdOutPipeSize>=1024,'StdOutPipeSize = '+Str(smi_proxy_tab.task.StdOutPipeSize div 1024));
  // Recovery = 5
  smi_proxy_tab.task.RecoveryPeriod:=iValDef(ReadIni('RecoveryPeriod'),5)*1000;
  Assertion(smi_proxy_tab.task.RecoveryPeriod>=1000,StrFmt('RecoveryPeriod = %d',smi_proxy_tab.task.RecoveryPeriod div 1000));
  // SmiTermRepeater = Repeater, Period
  smi_proxy_tab.Handler.TermRepeater:=iValDef(ExtractWord(1,ReadIniAlter('SmiTermRepeater',risModeDefault)),3);
  smi_proxy_tab.Handler.TermPeriod:=iValDef(ExtractWord(2,ReadIniAlter('SmiTermRepeater',risModeDefault)),50);
  Assertion((smi_proxy_tab.Handler.TermRepeater>0) and (smi_proxy_tab.Handler.TermPeriod>0),
           'SmiTermRepeater = '+Str(smi_proxy_tab.Handler.TermRepeater)+', '+Str(smi_proxy_tab.Handler.TermPeriod));
  // SmiVolatile
  smi_proxy_tab.optVol:=Ord(Val(ReadIni('SmiVolatile'))>0);
  Success('SmiVolatile = '+Str(Ord(smi_proxy_tab.optVol)));
  // CMD
  smi_proxy_tab.task.CmdLine:=smi_proxy_tab.task.ExeFile
                    +' -obj '+smi_proxy_tab.optObj
                    +' -set '+smi_proxy_tab.optSet
                    +' -dns '+smi_proxy_tab.optDns;
  if smi_proxy_tab.optVol>0 then smi_proxy_tab.task.CmdLine:=smi_proxy_tab.task.CmdLine+' -vol';
  Assertion(smi_proxy_tab.task.CmdLine<>'','CmdLine = '+smi_proxy_tab.task.CmdLine);
  // FSM
  smi_proxy_fsm_init(smi_proxy_tab.optDns,
                     smi_proxy_tab.optDom,
                     smi_proxy_tab.optObj,
                     smi_proxy_tab.task.SmlFile);
  Assertion(fsm_type(smi_proxy_tab.dom)=fsm_type_domain,'SmiDomain = '+fsm_path(smi_proxy_tab.dom));
  Assertion(fsm_type(smi_proxy_tab.obj)=fsm_type_object,'SmiObject = '+fsm_path(smi_proxy_tab.obj));
  // DefStateColor
  smi_proxy_tab.StateColorAuto:=true; smi_proxy_tab.StateColorStamp:=pi;
  smi_proxy_tab.DefStateColor:=StringToColorDef(ReadIni('DefStateColor'),clSilver);
  // Task
  smi_proxy_become_dead;
  ClearLocals;
 end;
 //
 // SMI Proxy finalization.
 //
 procedure smi_proxy_finalize;
  procedure smi_proxy_fsm_free;
  begin
   fsm_kill(smi_proxy_tab.fsm);
  end;
  procedure smi_proxy_handler_free;
  begin
   if (smi_proxy_tab.Handler.Params<>0) then bNul(text_free(smi_proxy_tab.Handler.Params));
   smi_proxy_tab.Handler.Params:=0;
  end;
 begin
  smi_proxy_task_stop(0);
  smi_proxy_handler_free;
  smi_proxy_fsm_free;
  smi_proxy_clear_all;
 end;
 //
 // SMI Proxy default console handler.
 //
 function smi_proxy_default_handler(var Data,cmd,arg:String; cmdid:Integer):Boolean;
 var complete:Boolean; i,n,status,n_pars,typ,size:Integer; name,value:String;
 begin
  complete:=false;
  name:=''; value:='';
  if Length(Data)>0 then begin
   if (cmdid>0) then begin
   //
   // Send a command to SMI Proxy server:
   // @smi_proxy @smi_set_state=READY
   //
   if (cmdid = smi_proxy_tab.cmd_smi_proxy) then begin
    smi_proxy_task_send(Trim(arg));
    complete:=true;
    Data:='';
   end else
    //
    // @dim_dns_node=localhost
    //
    if (cmdid=smi_proxy_tab.cmd_dim_dns_node) then begin
     smi_proxy_dns_node_put(arg);
     complete:=true;
    end else
    //
    // @smi_attach=1,DEMO::LOGGER
    //
    if (cmdid=smi_proxy_tab.cmd_smi_attach) then begin
     status:=Val(ExtractWord(1,arg));
     if (status>0) then smi_proxy_object_name_put(ExtractWord(2,arg));
     complete:=true;
    end else
    //
    // @smi_volatile=1
    //
    if (cmdid=smi_proxy_tab.cmd_smi_volatile) then begin
     status:=Val(ExtractWord(1,arg));
     if (status>0) then smi_proxy_volatile_put(1);
     complete:=true;
    end else
    //
    // @smi_get_state=1,READY
    //
    if (cmdid=smi_proxy_tab.cmd_smi_get_state) then begin
     status:=Val(ExtractWord(1,arg));
     if (status>0) then smi_proxy_state_name_put(SkipWords(1,arg));
     complete:=true;
    end else
    //
    // @smi_set_state=1,READY
    // @smi_set_state=1,READY/PAR1(I)=123/PAR2(F)=1.23/PAR3=ABC
    //
    if (cmdid=smi_proxy_tab.cmd_smi_set_state) then begin
     status:=Val(ExtractWord(1,arg));
     if (status>0) then smi_proxy_state_name_put(SkipWords(1,arg));
     complete:=true;
    end else
    //
    // @smi_set_par=1,PAR1,1,123
    //
    if (cmdid=smi_proxy_tab.cmd_smi_set_par) then begin
     status:=Val(ExtractWord(1,arg));
     if (status>0) then Details(smi_proxy_prompt+'Parameter written: '+ExtractWord(2,arg));
     complete:=true;
    end else
    //
    // @smi_proxy_handler=BEGIN,1 => READ params
    // @smi_proxy_handler=END,1   => BUSY to handle commands
    //
    if (cmdid=smi_proxy_tab.cmd_smi_proxy_handler) then begin
     if IsSameText(ExtractWord(1,arg),'BEGIN') then smi_proxy_handler_stage_put(smi_proxy_stage_read) else
     if IsSameText(ExtractWord(1,arg),'END')   then smi_proxy_handler_stage_put(smi_proxy_stage_busy);
     smi_proxy_handler_count_put(Val(ExtractWord(2,arg)));
     complete:=true;
    end else
    //
    // @smi_get_command=1,START/TYPE=1/NR(I)=1
    //
    if (cmdid=smi_proxy_tab.cmd_smi_get_command) then begin
     status:=Val(ExtractWord(1,arg));
     if (status>0) and smi_proxy_handler_read then smi_proxy_handler_command_put(SkipWords(1,arg));
     complete:=true;
    end else
    //
    // @smi_get_action=1,START,2
    //
    if (cmdid=smi_proxy_tab.cmd_smi_get_action) then begin
     status:=Val(ExtractWord(1,arg));
     n_pars:=Val(ExtractWord(3,arg));
     if (status>0) and smi_proxy_handler_read then begin
      smi_proxy_handler_action_put(ExtractWord(2,arg));
      smi_proxy_handler_params_clear(n_pars);
     end;
     complete:=true;
    end else
    //
    // @smi_get_next_par=0,TYPE,0,1
    //
    if (cmdid=smi_proxy_tab.cmd_smi_get_next_par) then begin
     i:=Val(ExtractWord(1,arg));
     name:=ExtractWord(2,arg);
     typ:=Val(ExtractWord(3,arg));
     size:=Val(ExtractWord(4,arg));
     if (name<>'') and (size>0) and smi_proxy_handler_read then begin
      typ:=smi_to_fsm_type(typ);
     end;
     complete:=true;
    end else
    //
    // @smi_get_par_value=1,NR,1,1
    //
    if (cmdid=smi_proxy_tab.cmd_smi_get_par_value) then begin
     i:=Val(ExtractWord(1,arg));
     name:=ExtractWord(2,arg);
     typ:=Val(ExtractWord(3,arg));
     value:=SkipWords(3,arg);
     if (name<>'') and smi_proxy_handler_read then begin
      typ:=smi_to_fsm_type(typ);
      bNul(text_putln(smi_proxy_tab.Handler.Params,i,name+','+Str(typ)+','+value));
     end;
     complete:=true;
    end else
    //
    // @smi_terminate_action=1,RUNNING
    //
    if (cmdid=smi_proxy_tab.cmd_smi_terminate_action) then begin
     status:=Val(ExtractWord(1,arg));
     if (status>0) then smi_proxy_state_name_put(SkipWords(1,arg));
     complete:=true;
    end else
    //
    // @smi_errors=0
    //
    if (cmdid=smi_proxy_tab.cmd_smi_errors) then begin
     n:=Val(ExtractWord(1,arg));
     if n=0
     then Success(smi_proxy_prompt+Trim(Data))
     else Problem(smi_proxy_prompt+Trim(Data));
     Data:=''; // Skip double printing
     complete:=true;
    end else
    //
    // @smi_report=ERROR: message
    //
    if (cmdid=smi_proxy_tab.cmd_smi_report) then begin
     if IsSameText(ExtractWord(1,arg),'ERROR:')
     then Problem(smi_proxy_prompt+Trim(Data))
     else Success(smi_proxy_prompt+Trim(Data));
     Data:=''; // Skip double printing
     complete:=true;
    end else
    //
    // @smi_restart=0
    //
    if (cmdid=smi_proxy_tab.cmd_smi_restart) then begin
     smi_proxy_task_stop(Val(ExtractWord(1,arg)));
     smi_proxy_tab.task.RecoveryTimer:=msecnow-smi_proxy_tab.task.RecoveryPeriod;
     Success(smi_proxy_prompt+Trim(Data));
     Data:=''; // Skip double printing
     complete:=true;
    end else
    //
    // @smi_repost
    //
    if (cmdid=smi_proxy_tab.cmd_smi_repost) then begin
     name:=smi_extract_action(Trim(arg));
     if (name='') and smi_proxy_handler_idle then name:=Trim(smi_proxy_state_name);
     if (name<>'') then DevSendCmdLocal('@smi_proxy @smi_terminate_action '+name);
     if (name<>'') then Success(smi_proxy_prompt+Trim(Data)) else Problem(smi_proxy_prompt+'No data to repost!');
     Data:=''; // Skip double printing
     complete:=true;
    end else
    complete:=false;
   end;
  end;
  name:=''; value:='';
  smi_proxy_default_handler:=complete;
 end;
 {
 Clear SmiProxy.
 }
 procedure ClearSmiProxy;
 begin
  ClearFsmManager;
  smi_proxy_clear_all;
 end;
 {
 Initialize SmiProxy.
 }
 procedure InitSmiProxy;
 begin
  InitFsmManager;
  smi_proxy_initialize;
  ShouldPollSmiProxy:=true;
 end;
 {
 Finalize SmiProxy.
 }
 procedure FreeSmiProxy;
 begin
  smi_proxy_finalize;
  FreeFsmManager;
 end;
 {
 Poll SmiProxy.
 }
 procedure PollSmiProxy;
  procedure smi_proxy_handler_poll;
  begin
   if (smi_proxy_tab.tagHandlerUptime<>0) then bNul(rSetTag(smi_proxy_tab.tagHandlerUptime,smi_proxy_handler_uptime));
   if smi_proxy_handler_term then // Repeater timer on TERM stage
   if msElapsedSinceMarker(smi_proxy_tab.Handler.TermTimer)>smi_proxy_tab.Handler.TermPeriod then begin
    smi_proxy_tab.Handler.TermCount:=smi_proxy_tab.Handler.TermCount+1;
    if (smi_proxy_tab.Handler.TermCount>=smi_proxy_tab.Handler.TermRepeater) then begin
      smi_proxy_handler_stage_put(smi_proxy_stage_idle);
      Trouble('Handler TERM signal failed (deadline).');
    end else smi_proxy_task_send('@smi_terminate_action='+smi_proxy_state_name);
    smi_proxy_tab.Handler.TermTimer:=mSecNow;
   end;
  end;
  procedure smi_proxy_update_state_color;
  var color:Integer;
  begin
   if (smi_proxy_tab.tagObjectState<>0) then
   if ShouldRefresh(smi_proxy_tab.StateColorStamp,HashIndexOf(smi_proxy_state_name,0,0))>0 then begin
    color:=smi_proxy_state_color;
    //color:=smi_proxy_find_state_color(smi_proxy_object_name,smi_proxy_state_name,smi_proxy_tab.DefStateColor);
    if (fsm_verbose_mode>0) then Success('State '+smi_proxy_state_name+' color '+ColorToString(color));
    bNul(SetTagColor(smi_proxy_tab.tagObjectState,color));
   end;
  end;
 begin
  if ShouldPollFsmManager then PollFsmManager;
  smi_proxy_task_poll;
  smi_proxy_handler_poll;
  smi_proxy_watchdog_poll;
  if smi_proxy_tab.StateColorAuto then smi_proxy_update_state_color;
 end;
