////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWKIT.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Guard server for &CronSrv DAQ device.                                      //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20240207 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

program crongrd; // Cron Guard

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 interfaces, lclintf, forms,
 _crw_alloc, _crw_cmdargs, _crw_str, _crw_base64, _crw_fio, _crw_fifo,
 _crw_rtc, _crw_ascio, _crw_az, _crw_polling, _crw_task, _crw_sesman;

//////////////////////////////////
// General variables and constants
//////////////////////////////////
const
 Terminated : Boolean    = False;  // Program should be terminated ?
 LogName    : LongString = '';     // Log file name
 AppName    : LongString = '';     // Application name
 CmdLine    : LongString = '';     // Command line parameters
 HomeDir    : LongString = '';     // Home directory
 Display    : Integer    = 0;      // Display flag
 Period     : Integer    = 15000;  // Guard check period, ms

////////////////////
// Utility functions
////////////////////

//
// Report on exceptions.
//
procedure BugReport(const E:Exception; const ErrorMsg:LongString='');
begin
 if Assigned(E)
 then StdOut.Put:=Format('@OnException=%s,%s',[E.ClassName,E.Message])
 else StdOut.Put:=Format('@OnError=%s',[ErrorMsg]);
 LockedInc(StdIoErrorCount);
end;

//
// Get EXE file version info.
//
function GetVersionInfo(const Name:LongString):LongString;
begin
 Result:=CookieScan(GetFileVersionInfoAsText(ProgName),Name);
end;

//
// Print greetings message.
//
procedure PrintGreetings;
begin
 StdOut.Put:=GetVersionInfo('ProductName')+' version '+GetVersionInfo('ProductVersion')+'.';
 StdOut.Put:=GetVersionInfo('LegalCopyright')+'.';
 StdOut.Put:=GetVersionInfo('ProductName')+' - '+GetVersionInfo('FileDescription');
 StdOut.Put:=GetVersionInfo('Comments');
 StdOut.Put:='Type @help to get Help.';
end;

//
// To be called from polling loop.
//
procedure GuardPolling;
var tid:Integer; Log:Text;
 procedure PutLog(const s:LongString);
 begin
  if (s<>'') then StdOut.Put:=s;
  if (s<>'') and (LogName<>'') then System.Writeln(Log,s);
 end;
begin
if not Terminated then
 if BecameZombie(FILE_TYPE_UNKNOWN,Period) then begin
  StdOut.Put:='Parent process died, PID='+IntToStr(GetParentProcessID)+', EXE='+GetParentProcessExe;
  if Length(AppName+CmdLine)>0 then begin
   if (LogName<>'') then System.Assign(Log,LogName) else AssignNull(Log);
   tid:=Task_Init('');
   try
    Task_Ctrl(tid,'AppName=');
    Task_Ctrl(tid,'HomeDir='+HomeDir);
    Task_Ctrl(tid,'CmdLine='+AppName+' '+CmdLine);
    Task_Ctrl(tid,'Display='+IntToStr(Display));
    if FileExists(LogName)
    then System.Append(Log) else System.Rewrite(Log);
    PutLog(GetDateStr(mSecNow,'.',True)+'-'+GetTimeStr(mSecNow));
    PutLog('Try to run process:');
    PutLog('AppName='+AppName);
    PutLog('CmdLine='+CmdLine);
    PutLog('HomeDir='+HomeDir);
    if Task_Run(tid) then begin
     PutLog('Process '+IntToStr(Task_Pid(tid))+' started.');
     Task_Wait(tid,Period);
     PutLog('Process '+IntToStr(Task_Pid(tid))+' exit code '+IntToStr(Task_Result(tid)));
    end else begin
     PutLog('Could not start process!');
    end;
    PutLog('');
   finally
    Task_Free(tid);
    SmartFileClose(Log);
   end;
  end;
  StdIn.Put:='@Exit';
 end;
end;

///////////////////////////////////////////
// Incoming request handling implementation
///////////////////////////////////////////
//
// Request: @exit
// Request: @exit=n
// Reply:   @exit=n
// Comment: Terminate program with exit code n.
//
procedure DoExit(const cmnd,args:LongString);
begin
 try
  Terminated:=True;
  System.ExitCode:=StrToIntDef(args,0);
  StdOut.Put:=Format('%s=%d',[cmnd,System.ExitCode]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @errors
// Request: @errors=n
// Reply:   @errors=n
// Comment: Return value n of error counter.
//
procedure DoErrors(const cmnd,args:LongString);
var
 n : LongInt;
begin
 try
  if Str2Long(args,n)
  then n:=LockedExchange(StdIoErrorCount,n)
  else n:=LockedGet(StdIoErrorCount);
  StdOut.Put:=Format('%s=%d',[cmnd,n]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @memory
// Reply:   @memory=n
// Comment: Return AllocMemSize, i.e. memory usage counter.
//
procedure DoMemory(const cmnd,args:LongString);
begin
 try
  StdOut.Put:=Format('%s=%d',[cmnd,GetAllocMemSize]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @ProcessPriority
// Request: @ProcessPriority=n
// Reply:   @ProcessPriority=n
// Comment: Set process priority class.
//
procedure DoProcessPriority(const cmnd,args:LongString);
var p:DWORD;
begin
 try
  if not IsEmptyStr(args) then begin
   p:=GetPriorityClassByName(args);
   if p>0 then SetPriorityClass(GetCurrentProcess,p);
  end;
  p:=ProcessPriorityToClass(GetProcessPriority);
  StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityClassName(p)]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @ThreadPriority
// Request: @ThreadPriority=n
// Reply:   @ThreadPriority=n
// Comment: Set thread priority class.
//
procedure DoThreadPriority(const cmnd,args:LongString);
var p:TThreadPriority;
begin
 try
  if not IsEmptyStr(args) then begin
   p:=GetPriorityByName(args);
   StdIn.Priority:=p;
   StdOut.Priority:=p;
   {$IFDEF WINDOWS}
   case p of
    tpIdle         : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_IDLE);
    tpLowest       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_LOWEST);
    tpLower        : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_BELOW_NORMAL);
    tpNormal       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
    tpHigher       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_ABOVE_NORMAL);
    tpHighest      : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_HIGHEST);
    tpTimeCritical : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
   end;
   {$ENDIF ~WINDOWS}
  end;
  StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityName(StdIn.Priority)]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @AppName
// Request: @AppName=n
// Reply:   @AppName=n
// Comment: Set/get application name.
//
procedure DoAppName(const cmnd,args:LongString);
begin
 try
  AppName:=Trim(args);
  StdOut.Put:=Format('%s=%s',[cmnd,AppName]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @CmdLine
// Request: @CmdLine=c
// Reply:   @CmdLine=c
// Comment: Set/get command line parameters.
//
procedure DoCmdLine(const cmnd,args:LongString);
begin
 try
  CmdLine:=Trim(args);
  StdOut.Put:=Format('%s=%s',[cmnd,CmdLine]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @HomeDir
// Request: @HomeDir=c
// Reply:   @HomeDir=c
// Comment: Set/get program home directory.
//
procedure DoHomeDir(const cmnd,args:LongString);
begin
 try
  HomeDir:=Trim(args);
  StdOut.Put:=Format('%s=%s',[cmnd,HomeDir]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @Period
// Request: @Period=n
// Reply:   @Period=n
// Comment: Set/get Period, ms.
//
procedure DoPeriod(const cmnd,args:LongString);
var
 n : LongInt;
begin
 try
  if Str2Long(args,n) then Period:=n;
  StdOut.Put:=Format('%s=%d',[cmnd,Period]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @Display
// Request: @Display=n
// Reply:   @Display=n
// Comment: Set/get Display,0/1.
//
procedure DoDisplay(const cmnd,args:LongString);
var
 n : LongInt;
begin
 try
  if Str2Long(args,n) then Display:=n;
  StdOut.Put:=Format('%s=%d',[cmnd,Display]);
 except
  on E:Exception do BugReport(E);
 end;
end;
//
// Request: @help
// Reply:   help info
// Comment: Show help.
//
procedure DoHelp(const cmnd,args:LongString);
begin
 StdOut.Put:='>> Command list:';
 StdOut.Put:='>> @Help                       This help.';
 StdOut.Put:='>> @Exit=n                     Exit program with code n.';
 StdOut.Put:='>> @Errors                     Return error counter.';
 StdOut.Put:='>> @Memory                     Return memory status.';
 StdOut.Put:='>> @ProcessPriority=n          Set process priority:Idle/Low/Normal/High/RealTime.';
 StdOut.Put:='>> @ThreadPriority=n           Set thread priority:tpIdle/tpLow/tpNormal/tpHigh/tpTimeCritical.';
 StdOut.Put:='>> @AppName=n                  Set/get Application name.';
 StdOut.Put:='>> @CmdLine=n                  Set/get Command Line parameters.';
 StdOut.Put:='>> @HomeDir=n                  Set/get home directory.';
 StdOut.Put:='>> @Period=n                   Set/get Period,ms.';
 StdOut.Put:='>> @Display=n                  Set/get Display mode.';
end;
//
// This callback handles unrecognized commands.
//
procedure DoSpecificCommands(const args:LongString);
begin
 if Length(args)>0 then BugReport(NIL,'Unrecognized command '+args);
end;
//
// Application specific initialization.
//
procedure SpecificInitialization;
var i:Integer;
begin
 Application.Initialize;
 SystemEchoProcedure:=StdOutEcho;
 PrintGreetings; StdOut.Put:='';
 StdOut.Put:='Parent process PID='+IntToStr(GetParentProcessId)+', EXE='+GetParentProcessExe;
 //
 //  Set Log file Name
 //
 if SessionManager.Start('? 100') then begin
  LogName:=SessionManager.VarTmpFile(ForceExtension(ApplicationName,'.log'));
 end;
 //
 // Register user commands coming from StdIn.
 //
 StdIn.SpecHandler:=DoSpecificCommands;
 StdIn.AddCommand('@Help',                          DoHelp);
 StdIn.AddCommand('@Exit',                          DoExit);
 StdIn.AddCommand('@Errors',                        DoErrors);
 StdIn.AddCommand('@Memory',                        DoMemory);
 StdIn.AddCommand('@ProcessPriority',               DoProcessPriority);
 StdIn.AddCommand('@ThreadPriority',                DoThreadPriority);
 StdIn.AddCommand('@AppName',                       DoAppName);
 StdIn.AddCommand('@CmdLine',                       DoCmdLine);
 StdIn.AddCommand('@HomeDir',                       DoHomeDir);
 StdIn.AddCommand('@Period',                        DoPeriod);
 StdIn.AddCommand('@Display',                       DoDisplay);
 //
 // Initialize command line parameters.
 //
 for i:=1 to ParamCount do begin
  if Length(ParamStr(i))=0 then Continue;
  if Length(AppName)=0 then AppName:=ParamStr(i) else
  if Length(CmdLine)=0 then CmdLine:=ParamStr(i) else CmdLine:=CmdLine+' '+ParamStr(i);
 end;
 if (AppName<>'') then StdOut.Put:='AppName='+AppName;
 if (CmdLine<>'') then StdOut.Put:='CmdLine='+CmdLine;
end;
//
// Application specific finalization.
//
procedure SpecificFinalization;
begin
 Application.Terminate;
end;
//
// Application specific polling.
//
procedure SpecificPolling;
const LastTicks:QWord=0;
var CurrTicks:QWord;
begin
 CurrTicks:=GetTickCount64;
 if (CurrTicks>LastTicks+1000) then begin
  TTask.PollDetachedPids;
  LastTicks:=CurrTicks;
 end;
 Application.ProcessMessages;
 GuardPolling;
end;
//
// Main program
//
begin
 try
  try
   SpecificInitialization;
   while not Terminated do begin
    while (StdIn.Count>0) do StdIn.Process(StdIn.Get);
    SpecificPolling;
    Sleep(TPolling.DefPollPeriod);
   end;
  finally
   SpecificFinalization;
  end;
 except
  on E:Exception do begin
   StdOut.Put:=E.Message;
   ExitCode:=2;
  end;
 end;
 Sleep(100);
 if BecameZombie(FILE_TYPE_PIPE,0) then ExitCode:=1;
end.

//////////////
// END OF FILE
//////////////

