#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################
# CRW-DAQ script to compile DaqPascal from external editor
##########################################################

function print_comment(){
 echo "***************************************************";
 echo "CRW-DAQ script to compile DaqPascal program code.  ";
 echo "The script expected location: /opt/crwdaq/resource.";
 echo "Param[1]  - program source file full path name.    ";
 echo "--sid s   - session identifier name (s).           ";
 echo "***************************************************";
};

function fatal(){
 1>&2 echo "$2";
 exit $1;
};

readonly datadir="$HOME/.local/share/daqgroup/daqpaseditor";
declare -i paramcount=0;
declare SourceFileName="";
declare DeviceName="";
declare TimeStamp="";
declare sid="";
declare pid="";

#######################################################################################################
# get/set INI file parameter:
# git_ini -get inifile section key
# git_ini -set inifile section key value
#######################################################################################################
function git_ini(){
 local op="$1"; local ini="$2"; local sec="$3"; local key="$4"; local val="$5";
 case $op in
  -get|--get) if [ $# -eq 4 ] && [ -e "$ini" ];                then git config -f $ini $sec.$key;        else return 1; fi; ;;
  -set|--set) if [ $# -eq 5 ] && mkdir -p "$(dirname "$ini")"; then git config -f $ini $sec.$key "$val"; else return 1; fi; ;;
  *) return 1; ;;
 esac;
};

######################################################################################################
# git config section and key could not contain punctuation chars (except -), so we need to replace it.
######################################################################################################
function name_to_sec(){ if [ -n "$1" ]; then echo "$1" | sed -e 's/[[:punct:]]/-/g' -e 's/^-*//' -e 's/-*$//'; fi; };
function name_to_key(){ if [ -n "$1" ]; then echo "$1" | sed -e 's/[[:punct:]]/-/g' -e 's/^-*//' -e 's/-*$//'; fi; };

##############################
# read_ini inifile section key
# value '-' replace to empty
##############################
function read_ini(){
 if [ $# -ge 3 ]; then
  local ini="$1";
  local sec="$(name_to_sec "$2")";
  local key="$(name_to_key "$3")";
  if [ "x$ini" = "x" ]; then return 1; fi;
  if [ "x$sec" = "x" ]; then return 1; fi;
  if [ "x$key" = "x" ]; then return 1; fi;
  local val="$(git_ini -get "$ini" "$sec" "$key")";
  if [ "x$val" = "x" ]; then return 1; fi;
  if [ "$val" = "-" ]; then return 0; fi;
  echo -n "$val";
  return 0;
 fi;
 return 1;
};

#########################
# Get age of timestamp $1
#########################
function get_age(){
 echo "$(date +%s.%N) - $1" | bc | cut -d '.' -f 1;
};

###############################
# load last state from INI file
###############################
function load_last_state(){
 if [ -z "$sid" ]; then
  1>&2 echo "Error: empty --sid (SessionId).";
  return 1;
 fi;
 if [ -z "$SourceFileName" ]; then
  1>&2 echo "Error: empty SourceFileName (parameter 1).";
  return 1;
 fi;
 local ini="$datadir/$sid.ini";
 local sec="$(realpath $SourceFileName)";
 if [ -n "$sid" ] && [ -e "$ini" ]; then
  TimeStamp="$(read_ini "$ini" "$sec" TimeStamp)";
  if [ -n "$TimeStamp" ]; then echo "TimeStamp=$TimeStamp"; else
   1>&2 echo "Error: could not read TimeStamp";
   return 1;
  fi;
  local old=0; let "old=3600*24";
  local age="$(get_age $TimeStamp)";
  if [ $age -le $old ]; then echo "Age=$age sec"; else
   1>&2 echo "Error: INI file is out of time ($age sec).";
   return 1;
  fi;
  pid="$(read_ini "$ini" "$sec" CRW_DAQ_SYS_EXE_PID)";
  if [ -n "$pid" ]; then echo "CRW_DAQ_SYS_EXE_PID=$pid"; else
   1>&2 echo "Error: could not read CRW_DAQ_SYS_EXE_PID";
   return 1;
  fi;
  DeviceName="$(read_ini "$ini" "$sec" DeviceName)";
  if [ -n "$DeviceName" ]; then echo "DeviceName=$DeviceName"; else
   1>&2 echo "Error: could not read DeviceName.";
   return 1;
  fi;
  return 0;
 fi;
 return 1;
};

function handle_param(){
 if [ -n "$1" ]; then
  let paramcount++;
  case $paramcount in
   1) SourceFileName="$1"; ;;
   *) ;;
  esac;
 fi;
};

function parse_cmdline(){
 local npar=0;
 while [ -n "$1" ]; do
  case $1 in
   -s|-sid|--sid) shift; sid="$1"; ;;
   -*) fatal 1 "Error: invalid option $1."; ;;
   *) handle_param $1; ;;
  esac;
  shift;
 done;
};

function launch_compiler(){
 local msg="@daq compile $DeviceName";
 local cmd="unix send2crwdaq -p $pid";
 echo "$cmd -c $msg";
 echo "$msg" | $cmd;
};

function main(){
 print_comment;
 parse_cmdline "$@";
 if load_last_state; then
  launch_compiler;
 else
  fatal 1 "Error: Could not call DaqPascal compiler.";
 fi;
};

main "$@";

#############
# END OF FILE
#############
