#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## CRW-DAQ script to build/compile Free Pascal projects. ##
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

let fatal_notify_crwdaq=1;

function print_comment(){
 echo "#####################################################";
 echo "CRW-DAQ tool to call Help on Free Pascal projects.   ";
 echo "Param[1]  - project (*.lpr) or help (*.chm;*.html).  ";
 echo "This script designed to use with FpcUpDeluxe.        ";
 echo "Note: put fpcupdeluxe.desktop in script directory.   ";
 echo "#####################################################";
};

#########################################
# Some file constants need for crwlazhelp
#########################################
readonly fpcup_desktop="fpcupdeluxe.desktop"; # put this file in script dir
readonly DefautBinDir="/opt/daqgroup/development/tools/fpcupdeluxe/lazarus";
readonly lhelp_class="lhelp.Lhelp"; # LHelp window class to find
readonly toc_chm="toc.chm";         # Lazarus: Table of Contents

# programs uses to open *.chm and *.html
readonly chm_viewers="lhelp xchm chmsee kchmviewer";
readonly htm_viewers="firefox chromium chrome opera konqueror epiphany tor";

#########################################################
# read Exec=… from .desktop file $1 and extract directory
#########################################################
function read_desktop_exec_dir(){
 if [ ! -e "$1" ]; then return 1; fi;
 cat "$1" | grep -P '^Exec=' | sed -e 's/^Exec=\s*//' | head -n 1 | cut -d ' ' -f 1 | xargs dirname;
};

#################################################################
# find lazarus binary directory by .desktop file or default value
#################################################################
function find_lazbindir(){
 local dir="$(read_desktop_exec_dir "$1")";
 if [ -d "$dir" ]; then echo -n "$dir"; return 0; fi;
 if [ -d "$DefautBinDir" ]; then echo -n "$DefautBinDir"; return 0; fi;
 return 1;
};

##########################################################
# get lazarus binary directory by fpcupdeluxe.desktop file
##########################################################
readonly LazBinDir="$(find_lazbindir "$scriptHOME/$fpcup_desktop")";

########################################
# find lhelp in lazarus binary directory
########################################
function find_lhelp(){
 if [ -d "$LazBinDir" ]; then
  local lhelp="$(find $LazBinDir -type f -executable -name lhelp | head -n 1)";
  if [ -x "$lhelp" ]; then echo -n "$lhelp"; return 0; fi;
 fi;
 return 1;
};

readonly LHelp="$(find_lhelp)";

###################################
# find toc.chm in lazarus directory
###################################
function find_toc_chm(){
 if [ -d "$LazBinDir" ]; then
  local toc="$(find $LazBinDir -type f -name $toc_chm | head -n 1)";
  if [ -f "$toc" ]; then echo -n "$toc"; return 0; fi;
 fi;
 return 1;
};

readonly TocChm="$(find_toc_chm)";

##############################
# detect process name $1 exist
##############################
function process_exist(){
 if [ $(pgrep -fc "$1") -gt 0 ]; then return 0; else return 1; fi;
};

###################################
# activate lhelp window by title $1
###################################
function activate_lhelp(){
 local title="$1"; local host="$(hostname)";
 if unix wmquery ActivateWindow -class "$lhelp_class" -title "$title"; then return 0; fi;
 local wnd="$(unix wmquery listwindows | grep " $lhelp_class " | grep " $host $title" | head -n 1 | cut -d ' ' -f 1)";
 if unix wmquery ActivateWindow -wnd $wnd; then return 0; else return 1; fi;
};

function open_lhelp_package(){
 local package="$1";
 if [ -n "$package" ] && process_exist lhelp; then
  for title in "LHelp -  [$package]" "LHelp - [$package]" "LHelp -  $package" "LHelp - $package"; do
   if activate_lhelp "$title"; then echo "Open Found Window \"$title\""; return 0; fi;
  done;
 fi;
 return 1;
};

#####################################
# open toc.chm from lazarus directory
#####################################
function open_toc(){
 if [ ! -x "$LHelp" ]; then fatal 1 "Could not find lhelp."; fi;
 if [ ! -f "$TocChm" ]; then fatal 1 "Could not find toc.chm.";  fi;
 if open_lhelp_package "Table of Contents"; then return 0; fi;
 echo      $LHelp $TocChm;
 unix grun $LHelp $TocChm;
};

###################
# open .chm file $1
###################
function open_chm(){
 if open_lhelp_package "$(basename "$1" .chm)"; then return 0; fi;
 local exe="$(unix env PATH=$PATH:$scriptHOME/shell $(unix which getapppath) $chm_viewers -a .chm xdg-open)";
 if [ ! -x "$exe" ]; then fatal 1 "Error: could not find CHM viewer."; fi;
 unix grun $exe "$1";
};

###################
# open .htm file $1
###################
function open_htm(){
 local exe="$(unix getapppath $htm_viewers -a .html xdg-open)";
 if [ ! -x "$exe" ]; then fatal 1 "Error: could not find HTML browser."; fi;
 unix grun $exe "$1";
};

######################
# main function to run
######################
function main(){
 print_comment;
 case $1 in
  ''|toc)   open_toc "$@"; ;;
  toc.chm)  open_toc "$@"; ;;
  toc.htm)  open_toc "$@"; ;;
  toc.html) open_toc "$@"; ;;
  *.lpr)    open_toc "$@"; ;;
  *.lpi)    open_toc "$@"; ;;
  *.chm)    open_chm "$@"; ;;
  *.htm)    open_htm "$@"; ;;
  *.html)   open_htm "$@"; ;;
  *)        fatal 1 "Error: invalid argiments."; ;;
 esac;
};

main "$@";

#############
# END OF FILE
#############
