#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## CRW-DAQ script to clone (make from sample) a project. ##
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

let fatal_notify_crwdaq=1;

declare projectTYPE="";    # project type: (lpr)
declare projectNAME="";    # project sample base name
declare projectRENAME="";  # project target base name
declare projectSAMPLE="";  # project sample full file name
declare projectTARGET="";  # project target full file name
declare targetHOMEDIR="";  # target directory for new project

function print_comment(){
 echo "#####################################################";
 echo "CRW-DAQ tool to clone (make from sample) a project.  ";
 echo "Param[1]  - sample project (*.lpr) to copy from.     ";
 echo "Param[2]  - target directory for projects.           ";
 echo "Param[3]  - target project rename (optional).        ";
 echo "#####################################################";
};

########################
# check $1 is valid name
# contains [a-zA-Z_0-9].
########################
function is_valid_name(){
 if [ -z "$1" ]; then return 1; fi;
 local s="$(echo -n "$1" | sed 's/\w//g')";
 if [ -n "$s" ]; then return 1; fi;
 return 0;
};

############################
# check and parse arguments:
# $1 = sample project *.lpr
# $2 = target home directory
# $3 = target project rename
############################
function ck_parse_args(){
 if [ $# -lt 2 ]; then
  fatal 1 "$(langstr ru "Недостаточно аргументов!" en "Missed arguments!")";
 fi;
 if [ -n "$3" ]; then
  projectRENAME="$(echo -n "$3" | tr '[:upper:]' '[:lower:]')";
 else
  projectRENAME="$(basename "$1" .lpr)";
 fi;
 if ! is_valid_name "$projectRENAME"; then
  fatal 1 "$(langstr ru "Недопустимое имя проекта $3." en "Invalid project name $3.")";
 fi;
 if [ -e "$2" ]; then
  targetHOMEDIR="$(realpath "$2")";
 else
  fatal 1 "$(langstr ru "Не найден каталог $2." en "Not found folder $2.")";
 fi;
 if [ -e "$1" ]; then
  case $(basename "$1") in
   *.lpr)
    projectTYPE="lpr";
    projectSAMPLE="$(realpath "$1")";
    projectNAME="$(basename "$1" .lpr)";
    projectTARGET="$targetHOMEDIR/$projectRENAME/$projectRENAME.lpr";
    ;;
   *) 
    fatal 1 "$(langstr ru "Неизвестный тип проекта $(basename "$1")." en "Unknown project type $(basename "$1").")";
    ;;
  esac;
 else
  fatal 1 "$(langstr ru "Не найден проект $(basename "$1")." en "Not found project $(basename "$1").")";
 fi;
};

##########################
# make project of any type
##########################
function make_project(){
 echo "Sample: $projectSAMPLE";
 echo "Locate: $targetHOMEDIR";
 echo "Rename: $projectRENAME";
 echo "P.Type: $projectTYPE";
 echo "Target: $projectTARGET";
 check_target_neq_source;
 case $projectTYPE in
  lpr) make_project_lpr; ;;
  *)   fatal 1 "$(langstr ru "Неизвестный тип проекта $(basename "$1")." en "Unknown project type $(basename "$1").")"; ;;
 esac;
};

##############################################
# Check Source directory is not same as Target
# Could not clone project to the same location
##############################################
function check_target_neq_source(){
 local sdir="$(dirname "$projectSAMPLE")"; sdir="$(realpath "$sdir")";
 local tdir="$(dirname "$projectTARGET")"; tdir="$(realpath "$tdir")";
 if [ "$sdir" = "$tdir" ]; then
  fatal 1 "$(langstr ru "Ошибка: Источник = Приемник …" en "Error: Source = Target …")";
 fi;
};

#######################################################
# Check is project sample relevant to target directory?
# plugin/dan folder - require For DataAnalysis plugins.
#######################################################
function is_project_lpr_relevant(){
 if [ -n "$(echo "$targetHOMEDIR" | grep "\bresource/plugin/dan\b")" ]; then
  if [ -z "$(cat "$projectSAMPLE" | grep -i "Library")" ]; then return 1; fi;
  if [ -z "$(cat "$projectSAMPLE" | grep -i "Exports")" ]; then return 1; fi;
  if [ -z "$(cat "$projectSAMPLE" | grep -i "CRWDAQ_PLUGIN")" ]; then return 1; fi;
  if [ -z "$(cat "$projectSAMPLE" | grep -i "ForDataAnalysis")" ]; then return 1; fi;
 fi;
 return 0;
};

##########################
# make project of type lpr
##########################
function make_project_lpr(){
 if ! is_project_lpr_relevant; then
  fatal 1 "$(langstr ru "Шаблон проекта не подходит для плагина «dan»." en "Project Sample is not relevant for «dan» plugin.")";
 fi;
 local sdir="$(dirname "$projectSAMPLE")";
 local tdir="$(dirname "$projectTARGET")";
 if pushd $sdir > /dev/null 2>&1; then
  if [ -e "$tdir" ]; then
   local title="$(langstr ru "Требуется подтверждение …" en "Confirmation required …")";
   local quest="$(langstr ru "<big>Каталог\n\n<b><span fgcolor='green'>$(basename $tdir)</span></b>\n\nуже существует.\n\n<b><span fgcolor='red'>Удалить и продолжить?</span></b></big>" en "<big>Directory\n\n<b><span fgcolor='green'>$(basename $tdir)</span></b>\n\nis already exists.\n\n<b><span fgcolor='red'>Delete and continue?</span></b></big>")";
   if zenity --timeout 180 --question --title "$title" --text "$quest" > /dev/null 2>&1; then
    rm -rfv "$tdir";
   else
    fatal 1 "$(langstr ru "Пользователь Отменил …" en "User Cancel …")";
   fi;
  fi;
  if ! mkdir -v --mode 777 "$tdir"; then
   fatal 1 "$(langstr ru "Не могу mkdir $tdir." en "Could not mkdir $tdir.")";
  fi;
  for fn in $(find . -type f); do
   local sf="$(realpath "$sdir/$fn")";
   local tn="${fn/$projectNAME/$projectRENAME}";
   local tf="$(realpath "$tdir/$tn")";
   cp -fv "$sf" "$tf";
   case $(basename "$tf") in
    *.lpr)
     sed -i "$tf" -e "s|$projectNAME|$projectRENAME|g";
     sed -i "$tf" -e "s|Copyright (c) [0-9\-]* |Copyright (c) 2001-$(date +%Y) |gi";
     ;;
    *.lpi)
     sed -i "$tf" -e "s|$projectNAME|$projectRENAME|g";
     sed -i "$tf" -e "s|Copyright (c) [0-9\-]* |Copyright (c) 2001-$(date +%Y) |gi";
     ;;
    *.lps)
     sed -i "$tf" -e "s|$sdir|$tdir|g";
     sed -i "$tf" -e "s|$projectNAME|$projectRENAME|g";
     ;;
    *)
     ;;
   esac;
  done;
 fi;
 popd > /dev/null 2>&1;
 ######################################################
 # send command to open target project in crwdaq window
 ######################################################
 if [ -n "$CRW_DAQ_SYS_EXE_PID" ] && [ -e "$tdir/$projectRENAME.lpr" ]; then
  local msg="@open -e $tdir/$projectRENAME.lpr";
  echo "$msg" | unix send2crwdaq;
 fi;
 if [ -e "$tdir/$projectRENAME.lpr" ]; then
  unix tooltip-notifier text "$scriptname: $(langstr ru "Проект $(basename "$projectTARGET") успешно создан." en "Project $(basename "$projectTARGET") created.")" preset stdSuccess delay 30000;
 else
  fatal 1 "$(langstr ru "Не смог создать $projectRENAME.lpr …" en "Could not create $projectRENAME.lpr …")";
 fi;
};

######################
# main function to run
######################
function main(){
 print_comment;
 ck_parse_args "$@";
 make_project;
};

main "$@";

#############
# END OF FILE
#############
