#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## CRW-DAQ script to build/compile Free Pascal projects. ##
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

let fatal_notify_crwdaq=1;

function print_comment(){
 echo "#####################################################";
 echo "CRW-DAQ tool to build/compile Free Pascal projects.  ";
 echo "Param[1]  - program source file of project (*.lpr).  ";
 echo "This script designed to use with FpcUpDeluxe         ";
 echo "Build command: 'lazbuld --pcp=$cfg -B $lpi' or       ";
 echo "lazbuld --primary-config-path=$cfg --build-all $lpi  ";
 echo "where $cfg - primary config path, $lpi - project .lpi";
 echo "#####################################################";
};

#########################################
# Some file constants to compose lazbuild
#########################################
readonly fpcup_desktop="fpcupdeluxe.desktop";
readonly FpcUpDeluxeDir="/opt/daqgroup/development/tools/fpcupdeluxe";
readonly PrimaryConfigPath="$FpcUpDeluxeDir/config_lazarus";
readonly LazarusBinDir="$FpcUpDeluxeDir/lazarus";

##########################
# Default lazbuild command
##########################
function lazbuild_def(){
 echo "$LazarusBinDir/lazbuild --primary-config-path=$PrimaryConfigPath --build-all";
};

####################################
# lazbuild command from file.desktop
####################################
function lazbuild_dsk(){
 local dsk="$1";
 if [ -e "$dsk" ]; then
  local cmd="$(cat "$dsk" | grep -P '^Exec=' | sed -e 's/^Exec=\s*//' -e 's/%f//' -e 's/%F//' -e 's/\/lazarus /\/lazbuild /' | grep '/lazbuild' | xargs)";
  if [ -n "$cmd" ] && [ -x "$(echo -n "$cmd" | cut -d ' ' -f 1)" ]; then echo "$cmd -B"; fi;
 fi;
};

###########################################################
# compose lazbuild command from file.desktop or use default
###########################################################
function lazbuild_command(){
 # lazbuild command extracted from fpcupdeluxe.desktop
 local cmd="$(lazbuild_dsk "$scriptHOME/$fpcup_desktop")";
 if [ -z "$cmd" ]; then cmd="$(lazbuild_def)"; fi;
 echo -n "$cmd";
};

##############################
# calculate source files names
##############################
readonly FileName="$(realpath "$1")";
readonly SourceLpr="$(dirname "$FileName")/$(basename "$FileName" ".${FileName##*.}").lpr";
readonly SourceLpi="$(dirname "$FileName")/$(basename "$FileName" ".${FileName##*.}").lpi";
readonly TargetLog="$(dirname "$FileName")/$(basename "$FileName" ".${FileName##*.}")_build.log";
readonly lazbuild="$(lazbuild_command)";
declare  compile_result="Compile fail.";
declare -i compile_status=1;

##############################
# check file name is not empty
##############################
function ActionOnEmptyArgs(){
 if [ -z "$1" ]; then
  fatal 1 "Error: file name is empty.";
 fi;
};

###################################
# check source files names is valid
###################################
function  SetAndCheckSourceFiles(){
 EnsureCreateLpiByLpr;
 CheckSourceFile "$FileName";
 CheckSourceFile "$SourceLpr";
 CheckSourceFile "$SourceLpi";
};

function CheckSourceFile(){
 echo "Checking file $1";
 if [ -e "$1" ]; then return 0; else fatal 1 "Error: file not found - $(basename "$1")."; fi;
};

function EnsureCreateLpiByLpr(){
 if [ -e "$SourceLpr" ] && [ ! -e "$SourceLpi" ]; then
  echo "Create $SourceLpi by $SourceLpr";
 fi;
};

#############################
# check status of compilation
#############################
function CheckLinesCompiled(){
 local line="$(cat "$TargetLog" | grep "lines compiled" | tail -n 1 | xargs)";
 if [ "$(echo "$line" | wc -l)" -gt 0 ] && [ $compile_status -eq 0 ]; then
  CompiledOk $line;
 else
  fatal 1 "Error compiling $(basename "$SourceLpr")";
 fi;
};

###########################
# report on compile succeed
###########################
function CompiledOk(){
 shift;
 local time="$(date +%Y.%m.%d-%H:%M:%S)";
 compile_result="$time - $scriptname: $(basename $SourceLpr) - $*.";
 unix tooltip-notifier text "$compile_result" preset stdOk delay 60000;
};

#####################################
# backup compiler log file to history
#####################################
function BackupCompilerLog(){
 if [ -e "$CRW_DAQ_VAR_TMP_DIR" ]; then
  cat "$TargetLog" >> "$CRW_DAQ_VAR_TMP_DIR/$scriptbase.log";
  echo ""          >> "$CRW_DAQ_VAR_TMP_DIR/$scriptbase.log";
  echo ""          >> "$CRW_DAQ_VAR_TMP_DIR/$scriptbase.log";
 fi;
};

####################
# Strip linking file
####################
function StripLinking(){
 local linking="$(cat "$TargetLog" | grep ') Linking ' | grep "$(basename "$SourceLpr" .lpr)" | cut -d ' ' -f 3)";
 if [ -n "$linking" ] && [ -e "$linking" ] && [ -x "$linking" ]; then
  local n1="$(stat -c%s $linking)"; local mode="$(stat -c%a $linking)";
  env LANG=en strip -v "$linking" | tee -a "$TargetLog";
  env LANG=en chmod -c $mode "$linking" | tee -a "$TargetLog";
  local n2="$(stat -c%s $linking)"; local pc="100"; let "pc=100-100*$n2/$n1";
  echo "File stripped \"$linking\" from $n1 to $n2 bytes - i.e. $pc % stripped." | tee -a "$TargetLog";
 fi;
};

####################################
# execute lazbuild to compile source
####################################
function ExecuteLazBuildSourceLpr(){
 rm -fv "$TargetLog";
 local time="$(date +%Y.%m.%d-%H:%M:%S)";
 echo -ne "\n$time - Build project: $SourceLpr\n\n" | tee "$TargetLog";
 if [ -n "$lazbuild" ]; then
  echo "$lazbuild $SourceLpr" | tee -a "$TargetLog";
  eval "$lazbuild $SourceLpr" | tee -a "$TargetLog";
  local CompilerExitCode="${PIPESTATUS[0]}";
  let "compile_status=$CompilerExitCode";
  echo "CompilerExitCode: $CompilerExitCode" | tee -a "$TargetLog";
 else
  fatal 1 "Error: Invalid lazbuild command $lazbuld.";
 fi;
 StripLinking;
 BackupCompilerLog;
 CheckLinesCompiled;
 # rm -fv "$TargetLog";
 echo -ne "\n$compile_result\n\n";
};

######################
# main function to run
######################
function main(){
 print_comment;
 ActionOnEmptyArgs "$@";
 SetAndCheckSourceFiles;
 ExecuteLazBuildSourceLpr;
 return $compile_status;
};

main "$@";

#############
# END OF FILE
#############
